"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMatchedPackagesSpec = getMatchedPackagesSpec;
exports.hasProxyTo = hasProxyTo;
exports.normalisePackageAccess = normalisePackageAccess;
exports.normalizeUserList = normalizeUserList;
exports.sanityCheckNames = sanityCheckNames;
exports.sanityCheckUplinksProps = sanityCheckUplinksProps;
exports.uplinkSanityCheck = uplinkSanityCheck;

var _assert = _interopRequireDefault(require("assert"));

var _lodash = _interopRequireDefault(require("lodash"));

var _minimatch = _interopRequireDefault(require("minimatch"));

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const BLACKLIST = {
  all: true,
  anonymous: true,
  undefined: true,
  owner: true,
  none: true
};
/**
 * Normalize user list.
 * @return {Array}
 */

function normalizeUserList(oldFormat, newFormat) {
  const result = [];
  /* eslint prefer-rest-params: "off" */

  for (let i = 0; i < arguments.length; i++) {
    if (arguments[i] == null) {
      continue;
    } // if it's a string, split it to array


    if (_lodash.default.isString(arguments[i])) {
      result.push(arguments[i].split(/\s+/));
    } else if (Array.isArray(arguments[i])) {
      result.push(arguments[i]);
    } else {
      throw _utils.ErrorCode.getInternalError('CONFIG: bad package acl (array or string expected): ' + JSON.stringify(arguments[i]));
    }
  }

  return _lodash.default.flatten(result);
}

function uplinkSanityCheck(uplinks, users = BLACKLIST) {
  const newUplinks = _lodash.default.clone(uplinks);

  let newUsers = _lodash.default.clone(users);

  for (const uplink in newUplinks) {
    if (Object.prototype.hasOwnProperty.call(newUplinks, uplink)) {
      if (_lodash.default.isNil(newUplinks[uplink].cache)) {
        newUplinks[uplink].cache = true;
      }

      newUsers = sanityCheckNames(uplink, newUsers);
    }
  }

  return newUplinks;
}

function sanityCheckNames(item, users) {
  (0, _assert.default)(item !== 'all' && item !== 'owner' && item !== 'anonymous' && item !== 'undefined' && item !== 'none', 'CONFIG: reserved uplink name: ' + item);
  (0, _assert.default)(!item.match(/\s/), 'CONFIG: invalid uplink name: ' + item);
  (0, _assert.default)(_lodash.default.isNil(users[item]), 'CONFIG: duplicate uplink name: ' + item);
  users[item] = true;
  return users;
}

function sanityCheckUplinksProps(configUpLinks) {
  const uplinks = _lodash.default.clone(configUpLinks);

  for (const uplink in uplinks) {
    if (Object.prototype.hasOwnProperty.call(uplinks, uplink)) {
      (0, _assert.default)(uplinks[uplink].url, 'CONFIG: no url for uplink: ' + uplink);
      (0, _assert.default)(_lodash.default.isString(uplinks[uplink].url), 'CONFIG: wrong url format for uplink: ' + uplink);
      uplinks[uplink].url = uplinks[uplink].url.replace(/\/$/, '');
    }
  }

  return uplinks;
}
/**
 * Check whether an uplink can proxy
 */


function hasProxyTo(pkg, upLink, packages) {
  const matchedPkg = getMatchedPackagesSpec(pkg, packages);
  const proxyList = typeof matchedPkg !== 'undefined' ? matchedPkg.proxy : [];

  if (proxyList) {
    return proxyList.some(curr => upLink === curr);
  }

  return false;
}

function getMatchedPackagesSpec(pkgName, packages) {
  for (const i in packages) {
    if (_minimatch.default.makeRe(i).exec(pkgName)) {
      return packages[i];
    }
  }

  return;
}

function normalisePackageAccess(packages) {
  const normalizedPkgs = _objectSpread({}, packages); // add a default rule for all packages to make writing plugins easier


  if (_lodash.default.isNil(normalizedPkgs['**'])) {
    normalizedPkgs['**'] = {
      access: [],
      publish: [],
      proxy: []
    };
  }

  for (const pkg in packages) {
    if (Object.prototype.hasOwnProperty.call(packages, pkg)) {
      (0, _assert.default)(_lodash.default.isObject(packages[pkg]) && _lodash.default.isArray(packages[pkg]) === false, `CONFIG: bad "'${pkg}'" package description (object expected)`);
      normalizedPkgs[pkg].access = normalizeUserList(packages[pkg].allow_access, packages[pkg].access);
      delete normalizedPkgs[pkg].allow_access;
      normalizedPkgs[pkg].publish = normalizeUserList(packages[pkg].allow_publish, packages[pkg].publish);
      delete normalizedPkgs[pkg].allow_publish;
      normalizedPkgs[pkg].proxy = normalizeUserList(packages[pkg].proxy_access, packages[pkg].proxy);
      delete normalizedPkgs[pkg].proxy_access; // if unpublish is not defined, we set to false to fallback in publish access

      normalizedPkgs[pkg].unpublish = _lodash.default.isUndefined(packages[pkg].unpublish) ? false : normalizeUserList([], packages[pkg].unpublish);
    }
  }

  return normalizedPkgs;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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