"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _debug = _interopRequireDefault(require("debug"));

var _fs = _interopRequireDefault(require("fs"));

var _lodash = _interopRequireDefault(require("lodash"));

var _mkdirp = _interopRequireDefault(require("mkdirp"));

var _path = _interopRequireDefault(require("path"));

var _logger = require("./logger");

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const debug = (0, _debug.default)('verdaccio:config');
const CONFIG_FILE = 'config.yaml';
const XDG = 'xdg';
const WIN = 'win';
const WIN32 = 'win32'; // eslint-disable-next-line

const pkgJSON = require('../../package.json');

/**
 * Find and get the first config file that match.
 * @return {String} the config file path
 */
function findConfigFile(configPath) {
  if (typeof configPath !== 'undefined') {
    return _path.default.resolve(configPath);
  }

  const configPaths = getConfigPaths();
  debug('%o posible locations found', configPaths.length);

  if (_lodash.default.isEmpty(configPaths)) {
    throw new Error('no configuration files can be processed');
  }

  const primaryConf = _lodash.default.find(configPaths, configLocation => (0, _utils.fileExists)(configLocation.path));

  if (typeof primaryConf !== 'undefined') {
    debug('previous location exist already %s', primaryConf === null || primaryConf === void 0 ? void 0 : primaryConf.path);
    return primaryConf.path;
  }

  return createConfigFile(_lodash.default.head(configPaths)).path;
}

function createConfigFile(configLocation) {
  createConfigFolder(configLocation);
  const defaultConfig = updateStorageLinks(configLocation, readDefaultConfig());

  _fs.default.writeFileSync(configLocation.path, defaultConfig);

  return configLocation;
}

function readDefaultConfig() {
  return _fs.default.readFileSync(require.resolve('../../conf/default.yaml'), 'utf-8');
}

function createConfigFolder(configLocation) {
  _mkdirp.default.sync(_path.default.dirname(configLocation.path));

  _logger.logger.info({
    file: configLocation.path
  }, 'Creating default config file in @{file}');
}

function updateStorageLinks(configLocation, defaultConfig) {
  if (configLocation.type !== XDG) {
    return defaultConfig;
  } // $XDG_DATA_HOME defines the base directory relative to which user specific data files should be stored,
  // If $XDG_DATA_HOME is either not set or empty, a default equal to $HOME/.local/share should be used.
  // $FlowFixMe


  let dataDir = process.env.XDG_DATA_HOME || _path.default.join(process.env.HOME, '.local', 'share');

  if ((0, _utils.folderExists)(dataDir)) {
    dataDir = _path.default.resolve(_path.default.join(dataDir, pkgJSON.name, 'storage'));
    return defaultConfig.replace(/^storage: .\/storage$/m, `storage: ${dataDir}`);
  }

  return defaultConfig;
}

function getConfigPaths() {
  const listPaths = [getXDGDirectory(), getWindowsDirectory(), getRelativeDefaultDirectory(), getOldDirectory()].reduce(function (acc, currentValue) {
    if (_lodash.default.isUndefined(currentValue) === false) {
      acc.push(currentValue);
    }

    return acc;
  }, []);
  return listPaths;
}

const getXDGDirectory = () => {
  const XDGConfig = getXDGHome() || process.env.HOME && _path.default.join(process.env.HOME, '.config');

  if (XDGConfig && (0, _utils.folderExists)(XDGConfig)) {
    return {
      path: _path.default.join(XDGConfig, pkgJSON.name, CONFIG_FILE),
      type: XDG
    };
  }
};

const getXDGHome = () => process.env.XDG_CONFIG_HOME;

const getWindowsDirectory = () => {
  if (process.platform === WIN32 && process.env.APPDATA && (0, _utils.folderExists)(process.env.APPDATA)) {
    return {
      path: _path.default.resolve(_path.default.join(process.env.APPDATA, pkgJSON.name, CONFIG_FILE)),
      type: WIN
    };
  }
};

const getRelativeDefaultDirectory = () => {
  return {
    path: _path.default.resolve(_path.default.join('.', pkgJSON.name, CONFIG_FILE)),
    type: 'def'
  };
};

const getOldDirectory = () => {
  return {
    path: _path.default.resolve(_path.default.join('.', CONFIG_FILE)),
    type: 'old'
  };
};

var _default = findConfigFile;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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