"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.listenDefaultCallback = listenDefaultCallback;
exports.startVerdaccio = startVerdaccio;

var _constants = _interopRequireDefault(require("constants"));

var _fs = _interopRequireDefault(require("fs"));

var _http = _interopRequireDefault(require("http"));

var _https = _interopRequireDefault(require("https"));

var _lodash = require("lodash");

var _url = _interopRequireDefault(require("url"));

var _index = _interopRequireDefault(require("../api/index"));

var _utils = require("./cli/utils");

var _constants2 = require("./constants");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const logger = require('./logger');

function displayExperimentsInfoBox(experiments) {
  const experimentList = Object.keys(experiments);

  if (experimentList.length >= 1) {
    logger.logger.warn('⚠️  experiments are enabled, we recommend do not use experiments in production, comment out this section to disable it');
    experimentList.forEach(experiment => {
      logger.logger.warn(` - support for ${experiment} ${experiments[experiment] ? 'is enabled' : ' is disabled'}`);
    });
  }
}
/**
 * Trigger the server after configuration has been loaded.
 * @param {Object} config
 * @param {Object} cliArguments
 * @param {String} configPath
 * @param {String} pkgVersion
 * @param {String} pkgName
 * @deprecated use runServer instead
 */


function startVerdaccio(config, cliListen, configPath, pkgVersion, pkgName, callback) {
  if ((0, _lodash.isObject)(config) === false) {
    throw new Error(_constants2.API_ERROR.CONFIG_BAD_FORMAT);
  }

  if ('experiments' in config) {
    displayExperimentsInfoBox(config.experiments);
  }

  (0, _index.default)(config).then(app => {
    const addresses = (0, _utils.getListListenAddresses)(cliListen, config.listen);

    if (addresses.length > 1) {
      process.emitWarning('multiple listen addresses are deprecated, please use only one');
    }

    addresses.forEach(function (addr) {
      let webServer;

      if (addr.proto === 'https') {
        webServer = handleHTTPS(app, configPath, config);
      } else {
        // http
        webServer = _http.default.createServer(app);
      }

      if (config.server && typeof config.server.keepAliveTimeout !== 'undefined' && config.server.keepAliveTimeout !== 'null') {
        // library definition for node is not up to date (doesn't contain recent 8.0 changes)
        webServer.keepAliveTimeout = config.server.keepAliveTimeout * 1000;
      }

      unlinkAddressPath(addr);
      callback(webServer, addr, pkgName, pkgVersion);
    });
  });
}

function unlinkAddressPath(addr) {
  if (addr.path && _fs.default.existsSync(addr.path)) {
    _fs.default.unlinkSync(addr.path);
  }
}

function logHTTPSWarning(storageLocation) {
  logger.logger.fatal(['You have enabled HTTPS and need to specify either ', '    "https.key" and "https.cert" or ', '    "https.pfx" and optionally "https.passphrase" ', 'to run https server', '', // commands are borrowed from node.js docs
  'To quickly create self-signed certificate, use:', ' $ openssl genrsa -out ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.keyPem) + ' 2048', ' $ openssl req -new -sha256 -key ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.keyPem) + ' -out ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.csrPem), ' $ openssl x509 -req -in ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.csrPem) + ' -signkey ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.keyPem) + ' -out ' + (0, _utils.resolveConfigPath)(storageLocation, _constants2.certPem), '', 'And then add to config file (' + storageLocation + '):', '  https:', `    key: ${(0, _utils.resolveConfigPath)(storageLocation, _constants2.keyPem)}`, `    cert: ${(0, _utils.resolveConfigPath)(storageLocation, _constants2.certPem)}`].join('\n'));
  process.exit(2);
}

function handleHTTPS(app, configPath, config) {
  try {
    let httpsOptions = {
      secureOptions: _constants.default.SSL_OP_NO_SSLv2 | _constants.default.SSL_OP_NO_SSLv3 // disable insecure SSLv2 and SSLv3

    };
    const keyCertConfig = config.https;
    const pfxConfig = config.https; // https must either have key and cert or a pfx and (optionally) a passphrase

    if (!(keyCertConfig.key && keyCertConfig.cert || pfxConfig.pfx)) {
      logHTTPSWarning(configPath);
    }

    if (pfxConfig.pfx) {
      const {
        pfx,
        passphrase
      } = pfxConfig;
      httpsOptions = (0, _lodash.assign)(httpsOptions, {
        pfx: _fs.default.readFileSync(pfx),
        passphrase: passphrase || ''
      });
    } else {
      const {
        key,
        cert,
        ca
      } = keyCertConfig;
      httpsOptions = (0, _lodash.assign)(httpsOptions, _objectSpread({
        key: _fs.default.readFileSync(key),
        cert: _fs.default.readFileSync(cert)
      }, ca && {
        ca: _fs.default.readFileSync(ca)
      }));
    }

    return _https.default.createServer(httpsOptions, app);
  } catch (err) {
    // catch errors related to certificate loading
    logger.logger.fatal({
      err: err
    }, 'cannot create server: @{err.message}');
    process.exit(2);
  }
}
/**
 *
 * @param webServer
 * @param addr
 * @param pkgName
 * @param pkgVersion
 * @deprecated use initServer instead
 */


function listenDefaultCallback(webServer, addr, pkgName, pkgVersion) {
  const server = webServer.listen(addr.port || addr.path, addr.host, () => {
    // send a message for tests
    if ((0, _lodash.isFunction)(process.send)) {
      process.send({
        verdaccio_started: true
      });
    }
  }).on('error', function (err) {
    logger.logger.fatal({
      err: err
    }, 'cannot create server: @{err.message}');
    process.exit(2);
  });

  function handleShutdownGracefully() {
    logger.logger.fatal('received shutdown signal - closing server gracefully...');
    server.close(() => {
      logger.logger.info('server closed.');
      process.exit(0);
    });
  } // handle shutdown signals nicely when environment says so


  if (process.env.VERDACCIO_HANDLE_KILL_SIGNALS === 'true') {
    process.on('SIGINT', handleShutdownGracefully);
    process.on('SIGTERM', handleShutdownGracefully);
    process.on('SIGHUP', handleShutdownGracefully);
  }

  logger.logger.warn({
    addr: addr.path ? _url.default.format({
      protocol: 'unix',
      pathname: addr.path
    }) : _url.default.format({
      protocol: addr.proto,
      hostname: addr.host,
      port: addr.port,
      pathname: '/'
    }),
    version: pkgName + '/' + pkgVersion
  }, 'http address - @{addr} - @{version}');
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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