"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _debug = _interopRequireDefault(require("debug"));

var _lodash = _interopRequireDefault(require("lodash"));

var _pluginLoader = _interopRequireDefault(require("../lib/plugin-loader"));

var _authUtils = require("./auth-utils");

var _configUtils = require("./config-utils");

var _constants = require("./constants");

var _cryptoUtils = require("./crypto-utils");

var _logger = require("./logger");

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const debug = (0, _debug.default)('verdaccio:auth');

class Auth {
  // pragma: allowlist secret
  constructor(config) {
    _defineProperty(this, "config", void 0);

    _defineProperty(this, "logger", void 0);

    _defineProperty(this, "secret", void 0);

    _defineProperty(this, "plugins", void 0);

    this.config = config;
    this.logger = _logger.logger;
    this.secret = config.secret;
    this.plugins = this._loadPlugin(config);

    this._applyDefaultPlugins();
  }

  _loadPlugin(config) {
    const pluginOptions = {
      config,
      logger: this.logger
    };
    return (0, _pluginLoader.default)(config, config.auth, pluginOptions, plugin => {
      const {
        authenticate,
        allow_access,
        allow_publish
      } = plugin; // @ts-ignore

      return authenticate || allow_access || allow_publish;
    });
  }

  _applyDefaultPlugins() {
    this.plugins.push((0, _authUtils.getDefaultPlugins)(this.logger));
  }

  changePassword(username, password, // pragma: allowlist secret
  newPassword, // pragma: allowlist secret
  cb) {
    const validPlugins = _lodash.default.filter(this.plugins, plugin => _lodash.default.isFunction(plugin.changePassword));

    if (_lodash.default.isEmpty(validPlugins)) {
      return cb(_utils.ErrorCode.getInternalError(_constants.SUPPORT_ERRORS.PLUGIN_MISSING_INTERFACE));
    }

    for (const plugin of validPlugins) {
      if (_lodash.default.isNil(plugin) || _lodash.default.isFunction(plugin.changePassword) === false) {
        debug('auth plugin does not implement changePassword, trying next one');
        continue;
      } else {
        debug('updating password for %o', username);
        plugin.changePassword(username, password, newPassword, (err, profile) => {
          if (err) {
            this.logger.error({
              username,
              err
            }, `An error has been produced
            updating the password for @{username}. Error: @{err.message}`);
            return cb(err);
          }

          this.logger.info({
            username
          }, 'updated password for @{username} was successful');
          return cb(null, profile);
        });
      }
    }
  }

  authenticate(username, password, cb) {
    const plugins = this.plugins.slice(0);
    const self = this;

    (function next() {
      const plugin = plugins.shift();

      if (_lodash.default.isFunction(plugin.authenticate) === false) {
        return next();
      }

      debug('authenticating %o', username);
      plugin.authenticate(username, password, function (err, groups) {
        if (err) {
          self.logger.error({
            username,
            err
          }, 'authenticating for user @{username} failed. Error: @{err.message}');
          return cb(err);
        } // Expect: SKIP if groups is falsey and not an array
        //         with at least one item (truthy length)
        // Expect: CONTINUE otherwise (will error if groups is not
        //         an array, but this is current behavior)
        // Caveat: STRING (if valid) will pass successfully
        //         bug give unexpected results
        // Info: Cannot use `== false to check falsey values`


        if (!!groups && groups.length !== 0) {
          // TODO: create a better understanding of expectations
          if (_lodash.default.isString(groups)) {
            throw new TypeError('plugin group error: invalid type for function');
          }

          const isGroupValid = _lodash.default.isArray(groups);

          if (!isGroupValid) {
            throw new TypeError(_constants.API_ERROR.BAD_FORMAT_USER_GROUP);
          }

          debug('authentication for user %o was successfully. Groups: %o', username, groups);
          return cb(err, (0, _authUtils.createRemoteUser)(username, groups));
        }

        next();
      });
    })();
  }

  add_user(user, password, cb) {
    const self = this;
    const plugins = this.plugins.slice(0);
    debug('add user %o', user);

    (function next() {
      const plugin = plugins.shift();
      let method = 'adduser';

      if (_lodash.default.isFunction(plugin[method]) === false) {
        method = 'add_user';
        self.logger.warn('the plugin method add_user in the auth plugin is deprecated and will be removed in next major release, notify to the plugin author');
      }

      if (_lodash.default.isFunction(plugin[method]) === false) {
        next();
      } else {
        // p.add_user() execution
        plugin[method](user, password, function (err, ok) {
          if (err) {
            self.logger.error({
              user,
              err: err.message
            }, 'the user @{user} could not being added. Error: @{err}');
            return cb(err);
          }

          if (ok) {
            self.logger.info({
              user
            }, 'the user @{user} has been added');
            return self.authenticate(user, password, cb);
          }

          next();
        });
      }
    })();
  }
  /**
   * Allow user to access a package.
   */


  allow_access({
    packageName,
    packageVersion
  }, user, callback) {
    const plugins = this.plugins.slice(0);
    const self = this;
    const pkgAllowAcces = {
      name: packageName,
      version: packageVersion
    };
    const pkg = Object.assign({}, pkgAllowAcces, (0, _configUtils.getMatchedPackagesSpec)(packageName, this.config.packages));
    debug('allow access for %o', packageName);

    (function next() {
      const plugin = plugins.shift();

      if (_lodash.default.isNil(plugin) || _lodash.default.isFunction(plugin.allow_access) === false) {
        return next();
      }

      plugin.allow_access(user, pkg, function (err, ok) {
        if (err) {
          self.logger.error({
            packageName,
            err
          }, 'forbidden access for @{packageName}. Error: @{err.message}');
          return callback(err);
        }

        if (ok) {
          self.logger.info({
            packageName
          }, 'allowed access for @{packageName}');
          return callback(null, ok);
        }

        next(); // cb(null, false) causes next plugin to roll
      });
    })();
  }

  allow_unpublish({
    packageName,
    packageVersion
  }, user, callback) {
    const pkg = Object.assign({
      name: packageName,
      version: packageVersion
    }, (0, _configUtils.getMatchedPackagesSpec)(packageName, this.config.packages));
    debug('allow unpublish for %o', packageName);

    for (const plugin of this.plugins) {
      if (_lodash.default.isNil(plugin) || _lodash.default.isFunction(plugin.allow_unpublish) === false) {
        debug('allow unpublish for %o plugin does not implement allow_unpublish', packageName);
        continue;
      } else {
        plugin.allow_unpublish(user, pkg, (err, ok) => {
          if (err) {
            this.logger.error({
              packageName,
              user: user === null || user === void 0 ? void 0 : user.name
            }, '@{user} forbidden publish for @{packageName}, it will fallback on unpublish permissions');
            return callback(err);
          }

          if (_lodash.default.isNil(ok) === true) {
            debug('we bypass unpublish for %o, publish will handle the access', packageName); // @ts-ignore
            // eslint-disable-next-line

            return this.allow_publish(...arguments);
          }

          if (ok) {
            this.logger.info({
              packageName,
              user: user === null || user === void 0 ? void 0 : user.name
            }, '@{user} allowed unpublish for @{packageName}');
            return callback(null, ok);
          }
        });
      }
    }
  }
  /**
   * Allow user to publish a package.
   */


  allow_publish({
    packageName,
    packageVersion
  }, user, callback) {
    const plugins = this.plugins.slice(0);
    const self = this;
    const pkg = Object.assign({
      name: packageName,
      version: packageVersion
    }, (0, _configUtils.getMatchedPackagesSpec)(packageName, this.config.packages));
    debug('allow publish for %o init | plugins: %o', packageName, plugins);

    (function next() {
      const plugin = plugins.shift();

      if (_lodash.default.isNil(plugin) || _lodash.default.isFunction(plugin.allow_publish) === false) {
        debug('allow publish for %o plugin does not implement allow_publish', packageName);
        return next();
      } // @ts-ignore


      plugin.allow_publish(user, pkg, (err, ok) => {
        if (_lodash.default.isNil(err) === false && _lodash.default.isError(err)) {
          self.logger.error({
            packageName,
            user: user === null || user === void 0 ? void 0 : user.name
          }, '@{user} is forbidden publish for @{packageName}');
          return callback(err);
        }

        if (ok) {
          self.logger.info({
            packageName,
            user: user === null || user === void 0 ? void 0 : user.name
          }, '@{user} is allowed publish for @{packageName}');
          return callback(null, ok);
        }

        debug('allow publish skip validation for %o', packageName);
        next(); // cb(null, false) causes next plugin to roll
      });
    })();
  }

  apiJWTmiddleware() {
    const plugins = this.plugins.slice(0);
    const helpers = {
      createAnonymousRemoteUser: _authUtils.createAnonymousRemoteUser,
      createRemoteUser: _authUtils.createRemoteUser
    };

    for (const plugin of plugins) {
      if (plugin.apiJWTmiddleware) {
        return plugin.apiJWTmiddleware(helpers);
      }
    }

    return (req, res, _next) => {
      req.pause();

      const next = function (err) {
        req.resume(); // uncomment this to reject users with bad auth headers
        // return _next.apply(null, arguments)
        // swallow error, user remains unauthorized
        // set remoteUserError to indicate that user was attempting authentication

        if (err) {
          req.remote_user.error = err.message;
        }

        return _next();
      };

      if (this._isRemoteUserValid(req.remote_user)) {
        return next();
      } // in case auth header does not exist we return anonymous function


      req.remote_user = (0, _authUtils.createAnonymousRemoteUser)();
      const {
        authorization
      } = req.headers;

      if (_lodash.default.isNil(authorization)) {
        return next();
      }

      if (!(0, _authUtils.isAuthHeaderValid)(authorization)) {
        debug('api middleware auth heather is not valid');
        return next(_utils.ErrorCode.getBadRequest(_constants.API_ERROR.BAD_AUTH_HEADER));
      }

      const security = (0, _authUtils.getSecurity)(this.config);
      const {
        secret
      } = this.config;

      if ((0, _authUtils.isAESLegacy)(security)) {
        debug('api middleware using legacy auth token');

        this._handleAESMiddleware(req, security, secret, authorization, next);
      } else {
        debug('api middleware using JWT auth token');

        this._handleJWTAPIMiddleware(req, security, secret, authorization, next);
      }
    };
  }

  _handleJWTAPIMiddleware(req, security, secret, authorization, next) {
    const {
      scheme,
      token
    } = (0, _authUtils.parseAuthTokenHeader)(authorization);

    if (scheme.toUpperCase() === _constants.TOKEN_BASIC.toUpperCase()) {
      // this should happen when client tries to login with an existing user
      const credentials = (0, _utils.convertPayloadToBase64)(token).toString();
      const {
        user,
        password
      } = (0, _authUtils.parseBasicPayload)(credentials);
      this.authenticate(user, password, (err, user) => {
        if (!err) {
          req.remote_user = user;
          next();
        } else {
          req.remote_user = (0, _authUtils.createAnonymousRemoteUser)();
          next(err);
        }
      });
    } else {
      // jwt handler
      const credentials = (0, _authUtils.getMiddlewareCredentials)(security, secret, authorization);

      if (credentials) {
        // if the signature is valid we rely on it
        req.remote_user = credentials;
        next();
      } else {
        // with JWT throw 401
        next(_utils.ErrorCode.getForbidden(_constants.API_ERROR.BAD_USERNAME_PASSWORD));
      }
    }
  }

  _handleAESMiddleware(req, security, secret, authorization, next) {
    const credentials = (0, _authUtils.getMiddlewareCredentials)(security, secret, authorization);

    if (credentials) {
      const {
        user,
        password
      } = credentials;
      this.authenticate(user, password, (err, user) => {
        if (!err) {
          req.remote_user = user;
          next();
        } else {
          req.remote_user = (0, _authUtils.createAnonymousRemoteUser)();
          next(err);
        }
      });
    } else {
      // we force npm client to ask again with basic authentication
      return next(_utils.ErrorCode.getBadRequest(_constants.API_ERROR.BAD_AUTH_HEADER));
    }
  }

  _isRemoteUserValid(remote_user) {
    return _lodash.default.isUndefined(remote_user) === false && _lodash.default.isUndefined(remote_user.name) === false;
  }
  /**
   * JWT middleware for WebUI
   */


  webUIJWTmiddleware() {
    return (req, res, _next) => {
      if (this._isRemoteUserValid(req.remote_user)) {
        return _next();
      }

      req.pause();

      const next = err => {
        req.resume();

        if (err) {
          // req.remote_user.error = err.message;
          res.status(err.statusCode).send(err.message);
        }

        return _next();
      };

      const {
        authorization
      } = req.headers;

      if (_lodash.default.isNil(authorization)) {
        return next();
      }

      if (!(0, _authUtils.isAuthHeaderValid)(authorization)) {
        return next(_utils.ErrorCode.getBadRequest(_constants.API_ERROR.BAD_AUTH_HEADER));
      }

      const token = (authorization || '').replace(`${_constants.TOKEN_BEARER} `, '');

      if (!token) {
        return next();
      }

      let credentials;

      try {
        credentials = (0, _authUtils.verifyJWTPayload)(token, this.config.secret);
      } catch (err) {// FIXME: intended behaviour, do we want it?
      }

      if (this._isRemoteUserValid(credentials)) {
        const {
          name,
          groups
        } = credentials; // $FlowFixMe

        req.remote_user = (0, _authUtils.createRemoteUser)(name, groups);
      } else {
        req.remote_user = (0, _authUtils.createAnonymousRemoteUser)();
      }

      next();
    };
  }

  async jwtEncrypt(user, signOptions) {
    const {
      real_groups,
      name,
      groups
    } = user;
    const realGroupsValidated = _lodash.default.isNil(real_groups) ? [] : real_groups;
    const groupedGroups = _lodash.default.isNil(groups) ? real_groups : Array.from(new Set([...groups.concat(realGroupsValidated)]));
    const payload = {
      real_groups: realGroupsValidated,
      name,
      groups: groupedGroups
    };
    const token = await (0, _cryptoUtils.signPayload)(payload, this.secret, signOptions);
    return token;
  }
  /**
   * Encrypt a string.
   */


  aesEncrypt(buf) {
    return (0, _cryptoUtils.aesEncrypt)(buf, this.secret);
  }

}

var _default = Auth;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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