"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _express = require("express");

var _lodash = _interopRequireDefault(require("lodash"));

var _constants = require("../../../lib/constants");

var _logger = require("../../../lib/logger");

var _utils = require("../../../lib/utils");

var _user = require("../../../utils/user");

var _middleware = require("../../middleware");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const getOrder = (order = 'asc') => {
  return order === 'asc';
};

function addPackageWebApi(storage, auth, config) {
  const can = (0, _middleware.allow)(auth);
  const pkgRouter = (0, _express.Router)();
  /* eslint new-cap: 0 */

  const checkAllow = (name, remoteUser) => new Promise((resolve, reject) => {
    try {
      auth.allow_access({
        packageName: name
      }, remoteUser, (err, allowed) => {
        if (err) {
          resolve(false);
        }

        resolve(allowed);
      });
    } catch (err) {
      reject(err);
    }
  }); // Get list of all visible package


  pkgRouter.get('/packages', function (req, res, next) {
    storage.getLocalDatabase(async function (err, packages) {
      if (err) {
        throw err;
      }

      async function processPackages(packages = []) {
        const permissions = [];
        const packgesCopy = packages.slice();

        for (const pkg of packgesCopy) {
          const pkgCopy = _objectSpread({}, pkg);

          pkgCopy.author = (0, _utils.formatAuthor)(pkg.author);

          try {
            if (await checkAllow(pkg.name, req.remote_user)) {
              if (config.web) {
                pkgCopy.author.avatar = (0, _user.generateGravatarUrl)(pkgCopy.author.email, config.web.gravatar);
              }

              if (!_lodash.default.isNil(pkgCopy.dist) && !_lodash.default.isNull(pkgCopy.dist.tarball)) {
                pkgCopy.dist.tarball = (0, _utils.getLocalRegistryTarballUri)(pkgCopy.dist.tarball, pkg.name, req, config.url_prefix);
              }

              permissions.push(pkgCopy);
            }
          } catch (err) {
            _logger.logger.error({
              name: pkg.name,
              error: err
            }, 'permission process for @{name} has failed: @{error}');

            throw err;
          }
        }

        return permissions;
      }

      const {
        web
      } = config; // @ts-ignore

      const order = config.web ? getOrder(web.sort_packages) : true;

      try {
        next((0, _utils.sortByName)(await processPackages(packages), order));
      } catch (error) {
        next(_utils.ErrorCode.getInternalError());
      }
    });
  }); // Get package readme

  pkgRouter.get('/package/readme/(@:scope/)?:package/:version?', can('access'), function (req, res, next) {
    const packageName = req.params.scope ? (0, _utils.addScope)(req.params.scope, req.params.package) : req.params.package;
    storage.getPackage({
      name: packageName,
      uplinksLook: true,
      req,
      callback: function (err, info) {
        if (err) {
          return next(err);
        }

        res.set(_constants.HEADER_TYPE.CONTENT_TYPE, _constants.HEADERS.TEXT_PLAIN);
        const referer = req.get('Referer');
        const pathname = referer ? new URL(referer).pathname : undefined;
        next((0, _utils.parseReadme)(info.name, info.readme, {
          pathname
        }));
      }
    });
  });
  pkgRouter.get('/sidebar/(@:scope/)?:package', can('access'), function (req, res, next) {
    const packageName = req.params.scope ? (0, _utils.addScope)(req.params.scope, req.params.package) : req.params.package;
    storage.getPackage({
      name: packageName,
      uplinksLook: true,
      keepUpLinkData: true,
      req,
      callback: function (err, info) {
        if (_lodash.default.isNil(err)) {
          const {
            v
          } = req.query;

          let sideBarInfo = _lodash.default.clone(info);

          sideBarInfo.versions = (0, _utils.convertDistRemoteToLocalTarballUrls)(info, req, config.url_prefix).versions;

          if ((0, _utils.isVersionValid)(info, v)) {
            // @ts-ignore
            sideBarInfo.latest = sideBarInfo.versions[v];
            sideBarInfo.latest.author = (0, _utils.formatAuthor)(sideBarInfo.latest.author);
          } else {
            var _sideBarInfo;

            sideBarInfo.latest = sideBarInfo.versions[info[_constants.DIST_TAGS].latest];

            if ((_sideBarInfo = sideBarInfo) !== null && _sideBarInfo !== void 0 && _sideBarInfo.latest) {
              sideBarInfo.latest.author = (0, _utils.formatAuthor)(sideBarInfo.latest.author);
            } else {
              res.status(_constants.HTTP_STATUS.NOT_FOUND);
              res.end();
              return;
            }
          }

          sideBarInfo = (0, _utils.deleteProperties)(['readme', '_attachments', '_rev', 'name'], sideBarInfo);

          if (config.web) {
            sideBarInfo = (0, _utils.addGravatarSupport)(sideBarInfo, config.web.gravatar);
          } else {
            sideBarInfo = (0, _utils.addGravatarSupport)(sideBarInfo);
          }

          next(sideBarInfo);
        } else {
          res.status(_constants.HTTP_STATUS.NOT_FOUND);
          res.end();
        }
      }
    });
  });
  return pkgRouter;
}

var _default = addPackageWebApi;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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