"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;

var _lodash = _interopRequireDefault(require("lodash"));

var _semver = _interopRequireDefault(require("semver"));

var _constants = require("../../../../lib/constants");

var _logger = require("../../../../lib/logger");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const personMatch = (person, search) => {
  if (typeof person === 'string') {
    return person.includes(search);
  }

  if (typeof person === 'object') {
    for (const field of Object.values(person)) {
      if (typeof field === 'string' && field.includes(search)) {
        return true;
      }
    }
  }

  return false;
};

const matcher = function (query) {
  const match = query.match(/author:(.*)/);

  if (match !== null) {
    return function (pkg) {
      return personMatch(pkg.author, match[1]);
    };
  } // TODO: maintainer, keywords, boost-exact
  // TODO implement some scoring system for freetext


  return pkg => {
    return ['name', 'displayName', 'description'].map(k => {
      return pkg[k];
    }).filter(x => {
      return x !== undefined;
    }).some(txt => {
      return txt.includes(query);
    });
  };
};

function compileTextSearch(textSearch) {
  const textMatchers = (textSearch || '').split(' ').map(matcher);
  return pkg => textMatchers.every(m => m(pkg));
}

function removeDuplicates(results) {
  const pkgNames = [];
  return results.filter(pkg => {
    var _pkg$package, _pkg$package2;

    if (pkgNames.includes(pkg === null || pkg === void 0 ? void 0 : (_pkg$package = pkg.package) === null || _pkg$package === void 0 ? void 0 : _pkg$package.name)) {
      return false;
    }

    pkgNames.push(pkg === null || pkg === void 0 ? void 0 : (_pkg$package2 = pkg.package) === null || _pkg$package2 === void 0 ? void 0 : _pkg$package2.name);
    return true;
  });
}

function checkAccess(pkg, auth, remoteUser) {
  return new Promise((resolve, reject) => {
    var _pkg$package3;

    auth.allow_access({
      packageName: pkg === null || pkg === void 0 ? void 0 : (_pkg$package3 = pkg.package) === null || _pkg$package3 === void 0 ? void 0 : _pkg$package3.name
    }, remoteUser, function (err, allowed) {
      if (err) {
        if (err.status && String(err.status).match(/^4\d\d$/)) {
          // auth plugin returns 4xx user error,
          // that's equivalent of !allowed basically
          allowed = false;
          return resolve(null);
        } else {
          reject(err);
        }
      } else {
        return resolve(allowed ? pkg : null);
      }
    });
  });
}

async function sendResponse(resultBuf, resultStream, auth, req, from, size) {
  resultStream.destroy();
  const resultsCollection = resultBuf.map(pkg => {
    if (pkg !== null && pkg !== void 0 && pkg.name) {
      return {
        package: pkg,
        // not sure if flags is need it
        flags: {
          unstable: Object.keys(pkg.versions).some(v => _semver.default.satisfies(v, '^1.0.0')) ? undefined : true
        },
        local: true,
        score: {
          final: 1,
          detail: {
            quality: 1,
            popularity: 1,
            maintenance: 0
          }
        },
        searchScore: 100000
      };
    } else {
      return pkg;
    }
  });
  const checkAccessPromises = await Promise.all(removeDuplicates(resultsCollection).map(pkgItem => {
    return checkAccess(pkgItem, auth, req.remote_user);
  }));
  const final = checkAccessPromises.filter(i => !_lodash.default.isNull(i)).slice(from, size);

  _logger.logger.debug(`search results ${final === null || final === void 0 ? void 0 : final.length}`);

  const response = {
    objects: final,
    total: final.length,
    time: new Date().toUTCString()
  };

  _logger.logger.debug(`total response ${final.length}`);

  return response;
}
/**
 * Endpoint for npm search v1
 * req: 'GET /-/v1/search?text=react&size=20&from=0&quality=0.65&popularity=0.98&maintenance=0.5'
 */


function _default(route, auth, storage) {
  route.get('/-/v1/search', async (req, res, next) => {
    // TODO: implement proper result scoring weighted by quality, popularity and maintenance query parameters
    let [text, size, from
    /* , quality, popularity, maintenance */
    ] = ['text', 'size', 'from'
    /* , 'quality', 'popularity', 'maintenance' */
    ].map(k => req.query[k]);
    size = parseInt(size) || 20;
    from = parseInt(from) || 0;
    const isInteresting = compileTextSearch(text);
    const resultStream = storage.search(0, {
      req
    });
    let resultBuf = [];
    let completed = false;
    resultStream.on('data', pkg => {
      // packages from the upstreams
      if (_lodash.default.isArray(pkg)) {
        resultBuf = resultBuf.concat(pkg.filter(pkgItem => {
          var _pkgItem$package;

          if (!isInteresting(pkgItem === null || pkgItem === void 0 ? void 0 : pkgItem.package)) {
            return;
          }

          _logger.logger.debug(`[remote] pkg name ${pkgItem === null || pkgItem === void 0 ? void 0 : (_pkgItem$package = pkgItem.package) === null || _pkgItem$package === void 0 ? void 0 : _pkgItem$package.name}`);

          return true;
        }));
      } else {
        // packages from local
        // due compability with `/-/all` we cannot refactor storage.search();
        if (!isInteresting(pkg)) {
          return;
        }

        _logger.logger.debug(`[local] pkg name ${pkg === null || pkg === void 0 ? void 0 : pkg.name}`);

        resultBuf.push(pkg);
      }
    });
    resultStream.on('error', function () {
      _logger.logger.error('search endpoint has failed');

      res.socket.destroy();
    });
    resultStream.on('end', async () => {
      if (!completed) {
        completed = true;

        try {
          const response = await sendResponse(resultBuf, resultStream, auth, req, from, size);

          _logger.logger.info('search endpoint ok results @{total}', {
            total: response.total
          });

          res.status(_constants.HTTP_STATUS.OK).json(response);
        } catch (err) {
          _logger.logger.error('search endpoint has failed @{err}', {
            err
          });

          next(err);
        }
      }
    });
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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