"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;

var _lodash = _interopRequireDefault(require("lodash"));

var _constants = require("../../../lib/constants");

var _utils = require("../../../lib/utils");

var _middleware = require("../../middleware");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const downloadStream = (packageName, filename, storage, req, res) => {
  const stream = storage.getTarball(packageName, filename);
  stream.on('content-length', function (content) {
    res.header('Content-Length', content);
  });
  stream.on('error', function (err) {
    return res.locals.report_error(err);
  });
  res.header(_constants.HEADERS.CONTENT_TYPE, _constants.HEADERS.OCTET_STREAM);
  stream.pipe(res);
};

const redirectOrDownloadStream = (packageName, filename, storage, req, res, config) => {
  const tarballUrlRedirect = _lodash.default.get(config, 'experiments.tarball_url_redirect');

  storage.hasLocalTarball(packageName, filename).then(hasLocalTarball => {
    if (hasLocalTarball) {
      const context = {
        packageName,
        filename
      };
      const tarballUrl = typeof tarballUrlRedirect === 'function' ? tarballUrlRedirect(context) : _lodash.default.template(tarballUrlRedirect)(context);
      res.redirect(tarballUrl);
    } else {
      downloadStream(packageName, filename, storage, req, res);
    }
  }).catch(err => {
    res.locals.report_error(err);
  });
};

function _default(route, auth, storage, config) {
  const can = (0, _middleware.allow)(auth); // TODO: anonymous user?

  route.get('/:package/:version?', can('access'), function (req, res, next) {
    const getPackageMetaCallback = function (err, metadata) {
      if (err) {
        return next(err);
      }

      metadata = (0, _utils.convertDistRemoteToLocalTarballUrls)(metadata, req, config.url_prefix);
      let queryVersion = req.params.version;

      if (_lodash.default.isNil(queryVersion)) {
        return next(metadata);
      }

      let version = (0, _utils.getVersion)(metadata, queryVersion);

      if (_lodash.default.isNil(version) === false) {
        return next(version);
      }

      if (_lodash.default.isNil(metadata[_constants.DIST_TAGS]) === false) {
        if (_lodash.default.isNil(metadata[_constants.DIST_TAGS][queryVersion]) === false) {
          queryVersion = metadata[_constants.DIST_TAGS][queryVersion];
          version = (0, _utils.getVersion)(metadata, queryVersion);

          if (_lodash.default.isNil(version) === false) {
            return next(version);
          }
        }
      }

      return next(_utils.ErrorCode.getNotFound(`${_constants.API_ERROR.VERSION_NOT_EXIST}: ${req.params.version}`));
    };

    storage.getPackage({
      name: req.params.package,
      uplinksLook: true,
      req,
      callback: getPackageMetaCallback
    });
  });
  route.get('/:scopedPackage/-/:scope/:filename', can('access'), function (req, res) {
    const {
      scopedPackage,
      filename
    } = req.params;

    if (_lodash.default.get(config, 'experiments.tarball_url_redirect') === undefined) {
      downloadStream(scopedPackage, filename, storage, req, res);
    } else {
      redirectOrDownloadStream(scopedPackage, filename, storage, req, res, config);
    }
  });
  route.get('/:package/-/:filename', can('access'), function (req, res) {
    if (_lodash.default.get(config, 'experiments.tarball_url_redirect') === undefined) {
      downloadStream(req.params.package, req.params.filename, storage, req, res);
    } else {
      redirectOrDownloadStream(req.params.package, req.params.filename, storage, req, res, config);
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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