"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.DEFAULT_SLOW_VERIFY_MS = void 0;

var _fs = _interopRequireDefault(require("fs"));

var _path = _interopRequireDefault(require("path"));

var _fileLocking = require("@verdaccio/file-locking");

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const DEFAULT_SLOW_VERIFY_MS = 200;
/**
 * HTPasswd - Verdaccio auth class
 */

exports.DEFAULT_SLOW_VERIFY_MS = DEFAULT_SLOW_VERIFY_MS;

class HTPasswd {
  /**
   *
   * @param {*} config htpasswd file
   * @param {object} stuff config.yaml in object from
   */
  // constructor
  constructor(config, options) {
    _defineProperty(this, "users", void 0);

    _defineProperty(this, "maxUsers", void 0);

    _defineProperty(this, "path", void 0);

    _defineProperty(this, "hashConfig", void 0);

    _defineProperty(this, "slowVerifyMs", void 0);

    _defineProperty(this, "logger", void 0);

    _defineProperty(this, "lastTime", void 0);

    this.users = {}; // verdaccio logger

    this.logger = options.logger; // all this "verdaccio_config" stuff is for b/w compatibility only

    this.maxUsers = config.max_users ? config.max_users : Infinity;
    let algorithm;
    let rounds;

    if (config.algorithm === undefined) {
      // to avoid breaking changes we uses crypt, future version
      // of this plugin uses bcrypt by default
      // https://github.com/verdaccio/verdaccio/pull/2072#issuecomment-770235502
      algorithm = _utils.HtpasswdHashAlgorithm.crypt;
      this.logger.warn( // eslint-disable-next-line max-len
      '"crypt" algorithm is deprecated consider switch to "bcrypt". Read more: https://github.com/verdaccio/monorepo/pull/580');
    } else if (_utils.HtpasswdHashAlgorithm[config.algorithm] !== undefined) {
      algorithm = _utils.HtpasswdHashAlgorithm[config.algorithm];
    } else {
      throw new Error(`Invalid algorithm "${config.algorithm}"`);
    }

    this.lastTime = null;
    const {
      file
    } = config;

    if (!file) {
      throw new Error('should specify "file" in config');
    }

    if (algorithm === _utils.HtpasswdHashAlgorithm.bcrypt) {
      rounds = config.rounds || _utils.DEFAULT_BCRYPT_ROUNDS;
    } else if (config.rounds !== undefined) {
      this.logger.warn({
        algo: algorithm
      }, 'Option "rounds" is not valid for "@{algo}" algorithm');
    }

    this.hashConfig = {
      algorithm,
      rounds
    };
    this.path = _path.default.resolve(_path.default.dirname(options.config.self_path), file);
    this.slowVerifyMs = config.slow_verify_ms || DEFAULT_SLOW_VERIFY_MS;
  }
  /**
   * authenticate - Authenticate user.
   * @param {string} user
   * @param {string} password
   * @param {function} cb
   * @returns {function}
   */


  authenticate(user, password, cb) {
    this.reload(async err => {
      if (err) {
        return cb(err.code === 'ENOENT' ? null : err);
      }

      if (!this.users[user]) {
        return cb(null, false);
      }

      let passwordValid = false;

      try {
        const start = new Date();
        passwordValid = await (0, _utils.verifyPassword)(password, this.users[user]);
        const durationMs = new Date().getTime() - start.getTime();

        if (durationMs > this.slowVerifyMs) {
          this.logger.warn({
            user,
            durationMs
          }, 'Password for user "@{user}" took @{durationMs}ms to verify');
        }
      } catch ({
        message
      }) {
        this.logger.error({
          message
        }, 'Unable to verify user password: @{message}');
      }

      if (!passwordValid) {
        return cb(null, false);
      } // authentication succeeded!
      // return all usergroups this user has access to;
      // (this particular package has no concept of usergroups, so just return
      // user herself)


      return cb(null, [user]);
    });
  }
  /**
   * Add user
   * 1. lock file for writing (other processes can still read)
   * 2. reload .htpasswd
   * 3. write new data into .htpasswd.tmp
   * 4. move .htpasswd.tmp to .htpasswd
   * 5. reload .htpasswd
   * 6. unlock file
   *
   * @param {string} user
   * @param {string} password
   * @param {function} realCb
   * @returns {Promise<any>}
   */


  adduser(user, password, realCb) {
    const pathPass = this.path;
    (0, _utils.sanityCheck)(user, password, _utils.verifyPassword, this.users, this.maxUsers).then(sanity => {
      // preliminary checks, just to ensure that file won't be reloaded if it's
      // not needed
      if (sanity) {
        return realCb(sanity, false);
      }

      (0, _utils.lockAndRead)(pathPass, async (err, res) => {
        let locked = false; // callback that cleans up lock first

        const cb = err => {
          if (locked) {
            (0, _fileLocking.unlockFile)(pathPass, () => {
              // ignore any error from the unlock
              realCb(err, !err);
            });
          } else {
            realCb(err, !err);
          }
        };

        if (!err) {
          locked = true;
        } // ignore ENOENT errors, we'll just create .htpasswd in that case


        if (err && err.code !== 'ENOENT') {
          return cb(err);
        }

        const body = (res || '').toString('utf8');
        this.users = (0, _utils.parseHTPasswd)(body); // real checks, to prevent race conditions
        // parsing users after reading file.

        sanity = await (0, _utils.sanityCheck)(user, password, _utils.verifyPassword, this.users, this.maxUsers);

        if (sanity) {
          return cb(sanity);
        }

        try {
          this._writeFile(await (0, _utils.addUserToHTPasswd)(body, user, password, this.hashConfig), cb);
        } catch (err) {
          return cb(err);
        }
      });
    }).catch(err => realCb(err));
  }
  /**
   * Reload users
   * @param {function} callback
   */


  reload(callback) {
    _fs.default.stat(this.path, (err, stats) => {
      if (err) {
        return callback(err);
      }

      if (this.lastTime === stats.mtime) {
        return callback();
      }

      this.lastTime = stats.mtime;

      _fs.default.readFile(this.path, 'utf8', (err, buffer) => {
        if (err) {
          return callback(err);
        }

        Object.assign(this.users, (0, _utils.parseHTPasswd)(buffer));
        callback();
      });
    });
  }

  _stringToUt8(authentication) {
    return (authentication || '').toString();
  }

  _writeFile(body, cb) {
    _fs.default.writeFile(this.path, body, err => {
      if (err) {
        cb(err);
      } else {
        this.reload(() => {
          cb(null);
        });
      }
    });
  }
  /**
   * changePassword - change password for existing user.
   * @param {string} user
   * @param {string} password
   * @param {string} newPassword
   * @param {function} realCb
   * @returns {function}
   */


  changePassword(user, password, newPassword, realCb) {
    (0, _utils.lockAndRead)(this.path, (err, res) => {
      let locked = false;
      const pathPassFile = this.path; // callback that cleans up lock first

      const cb = err => {
        if (locked) {
          (0, _fileLocking.unlockFile)(pathPassFile, () => {
            // ignore any error from the unlock
            realCb(err, !err);
          });
        } else {
          realCb(err, !err);
        }
      };

      if (!err) {
        locked = true;
      }

      if (err && err.code !== 'ENOENT') {
        return cb(err);
      }

      const body = this._stringToUt8(res);

      this.users = (0, _utils.parseHTPasswd)(body);

      try {
        (0, _utils.changePasswordToHTPasswd)(body, user, password, newPassword, this.hashConfig).then(content => {
          this._writeFile(content, cb);
        }).catch(err => {
          cb(err);
        });
      } catch (err) {
        return cb(err);
      }
    });
  }

}

exports.default = HTPasswd;
//# sourceMappingURL=htpasswd.js.map