import { VDomModel, VDomRenderer } from '@jupyterlab/apputils';
import { TextItem } from '@jupyterlab/statusbar';
import { nullTranslator } from '@jupyterlab/translation';
import * as React from 'react';
/**
 * A pure function for rendering a Command/Edit mode component.
 *
 * @param props: the props for rendering the component.
 *
 * @returns a tsx component for command/edit mode.
 */
function CommandEditComponent(props) {
    const trans = (props.translator || nullTranslator).load('jupyterlab');
    return (React.createElement(TextItem, { source: trans.__('Mode: %1', props.modeNames[props.notebookMode]) }));
}
/**
 * StatusBar item to display which notebook mode user is in.
 */
export class CommandEditStatus extends VDomRenderer {
    /**
     * Construct a new CommandEdit status item.
     */
    constructor(translator) {
        super(new CommandEditStatus.Model());
        this.translator = translator || nullTranslator;
        this._trans = this.translator.load('jupyterlab');
        this._modeNames = {
            command: this._trans.__('Command'),
            edit: this._trans.__('Edit')
        };
    }
    /**
     * Render the CommandEdit status item.
     */
    render() {
        if (!this.model) {
            return null;
        }
        this.node.title = this._trans.__('Notebook is in %1 mode', this._modeNames[this.model.notebookMode]);
        return (React.createElement(CommandEditComponent, { notebookMode: this.model.notebookMode, translator: this.translator, modeNames: this._modeNames }));
    }
}
/**
 * A namespace for CommandEdit statics.
 */
(function (CommandEditStatus) {
    /**
     * A VDomModel for the CommandEdit renderer.
     */
    class Model extends VDomModel {
        constructor() {
            super(...arguments);
            /**
             * On a change to the notebook, update the mode.
             */
            this._onChanged = (_notebook) => {
                const oldMode = this._notebookMode;
                if (this._notebook) {
                    this._notebookMode = _notebook.mode;
                }
                else {
                    this._notebookMode = 'command';
                }
                this._triggerChange(oldMode, this._notebookMode);
            };
            this._notebookMode = 'command';
            this._notebook = null;
        }
        /**
         * The current mode of the current notebook.
         */
        get notebookMode() {
            return this._notebookMode;
        }
        /**
         * Set the current notebook for the model.
         */
        set notebook(notebook) {
            const oldNotebook = this._notebook;
            if (oldNotebook !== null) {
                oldNotebook.stateChanged.disconnect(this._onChanged, this);
                oldNotebook.activeCellChanged.disconnect(this._onChanged, this);
                oldNotebook.modelContentChanged.disconnect(this._onChanged, this);
            }
            const oldMode = this._notebookMode;
            this._notebook = notebook;
            if (this._notebook === null) {
                this._notebookMode = 'command';
            }
            else {
                this._notebookMode = this._notebook.mode;
                this._notebook.stateChanged.connect(this._onChanged, this);
                this._notebook.activeCellChanged.connect(this._onChanged, this);
                this._notebook.modelContentChanged.connect(this._onChanged, this);
            }
            this._triggerChange(oldMode, this._notebookMode);
        }
        /**
         * Trigger a state change for the renderer.
         */
        _triggerChange(oldState, newState) {
            if (oldState !== newState) {
                this.stateChanged.emit(void 0);
            }
        }
    }
    CommandEditStatus.Model = Model;
})(CommandEditStatus || (CommandEditStatus = {}));
//# sourceMappingURL=modestatus.js.map