// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { VDomModel, VDomRenderer } from '@jupyterlab/apputils';
import { TextItem } from '@jupyterlab/statusbar';
import { nullTranslator } from '@jupyterlab/translation';
import React from 'react';
/**
 * A pure functional component for a Saving status item.
 *
 * @param props - the props for the component.
 *
 * @returns a tsx component for rendering the saving state.
 */
function SavingStatusComponent(props) {
    return React.createElement(TextItem, { source: props.fileStatus });
}
/**
 * The amount of time (in ms) to retain the saving completed message
 * before hiding the status item.
 */
const SAVING_COMPLETE_MESSAGE_MILLIS = 2000;
/**
 * A VDomRenderer for a saving status item.
 */
export class SavingStatus extends VDomRenderer {
    /**
     * Create a new SavingStatus item.
     */
    constructor(opts) {
        super(new SavingStatus.Model(opts.docManager));
        const translator = opts.translator || nullTranslator;
        const trans = translator.load('jupyterlab');
        this._statusMap = {
            completed: trans.__('Saving completed'),
            started: trans.__('Saving started'),
            failed: trans.__('Saving failed')
        };
    }
    /**
     * Render the SavingStatus item.
     */
    render() {
        if (this.model === null || this.model.status === null) {
            return null;
        }
        else {
            return (React.createElement(SavingStatusComponent, { fileStatus: this._statusMap[this.model.status] }));
        }
    }
}
/**
 * A namespace for SavingStatus statics.
 */
(function (SavingStatus) {
    /**
     * A VDomModel for the SavingStatus item.
     */
    class Model extends VDomModel {
        /**
         * Create a new SavingStatus model.
         */
        constructor(docManager) {
            super();
            /**
             * React to a saving status change from the current document widget.
             */
            this._onStatusChange = (_, newStatus) => {
                this._status = newStatus;
                if (this._status === 'completed') {
                    setTimeout(() => {
                        this._status = null;
                        this.stateChanged.emit(void 0);
                    }, SAVING_COMPLETE_MESSAGE_MILLIS);
                    this.stateChanged.emit(void 0);
                }
                else {
                    this.stateChanged.emit(void 0);
                }
            };
            this._status = null;
            this._widget = null;
            this._status = null;
            this.widget = null;
            this._docManager = docManager;
        }
        /**
         * The current status of the model.
         */
        get status() {
            return this._status;
        }
        /**
         * The current widget for the model. Any widget can be assigned,
         * but it only has any effect if the widget is an IDocument widget
         * known to the application document manager.
         */
        get widget() {
            return this._widget;
        }
        set widget(widget) {
            var _a, _b;
            const oldWidget = this._widget;
            if (oldWidget !== null) {
                const oldContext = this._docManager.contextForWidget(oldWidget);
                if (oldContext) {
                    oldContext.saveState.disconnect(this._onStatusChange);
                }
                else if ((_a = this._widget.content) === null || _a === void 0 ? void 0 : _a.saveStateChanged) {
                    this._widget.content.saveStateChanged.disconnect(this._onStatusChange);
                }
            }
            this._widget = widget;
            if (this._widget === null) {
                this._status = null;
            }
            else {
                const widgetContext = this._docManager.contextForWidget(this._widget);
                if (widgetContext) {
                    widgetContext.saveState.connect(this._onStatusChange);
                }
                else if ((_b = this._widget.content) === null || _b === void 0 ? void 0 : _b.saveStateChanged) {
                    this._widget.content.saveStateChanged.connect(this._onStatusChange);
                }
            }
        }
    }
    SavingStatus.Model = Model;
})(SavingStatus || (SavingStatus = {}));
//# sourceMappingURL=savingstatus.js.map