import paramiko


def connect_ssh_client_with_key(key_filename: str, hostname: str, username: str, password: str = None,
                                propagate_exceptions=False) -> paramiko.client:
    """
    Get list of remote files in a path that match a wildcard string
    :param key_filename: path to openssh-private-key-file
    :param hostname: hostname
    :param username: username
    :param password: password
    :param propagate_exceptions: (optional) backwards compatible flag to raise access permission issues.
    :return: connected SSH client
    """
    from logging import getLogger
    logger = getLogger()
    max_retries = 3
    while max_retries > 0:
        try:
            ssh = paramiko.SSHClient()
            ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
            if not password:
                ssh.connect(key_filename=key_filename, hostname=hostname, username=username)
            else:
                ssh.connect(key_filename=key_filename, hostname=hostname, username=username, password=password)
            return ssh
        except (paramiko.SSHException, paramiko.SFTPError) as exception:
            logger.warning(exception)
            max_retries -= 1
            if max_retries == 0 and propagate_exceptions:
                raise exception
    return None


def download_remote_file(ssh: paramiko.client, remote_file_path: str, local_dir_path: str = None,
                         propagate_exceptions=False):
    """
    Download a remote file to local path. Assumes Read access to the file
    :param ssh: connected SSH client
    :param remote_file_path: path of the remote file to download from
    :param local_dir_path: (optional) path to the directory to download to
    :param propagate_exceptions: (optional) backwards compatible flag to raise access permission issues
    :return:
    """
    from stratus_api.storage import UPLOAD_FOLDER
    from logging import getLogger
    import os
    logger = getLogger()
    try:
        sftp = ssh.open_sftp()
        if local_dir_path is None:
            local_dir_path = UPLOAD_FOLDER
        file_name = remote_file_path.split('/')[-1]
        local_file_path = os.path.join(local_dir_path, file_name)
        sftp.get(remote_file_path, local_file_path)
        sftp.close()
        ssh.close()
    except (paramiko.SSHException, paramiko.SFTPError) as exception:
        logger.warning(exception)


def upload_local_file(ssh: paramiko.client, upload_dir_path: str, local_file_path: str, propagate_exceptions=False,
                      connection_close=True):
    """
    Upload a local file to remote path. Assumes Write access to the remote location
    :param ssh: connected SSH client
    :param upload_dir_path: path of the directory to upload to
    :param local_file_path: path to the file to upload
    :param propagate_exceptions: (optional) backwards compatible flag to raise access permission issues
    :param connection_close:
    :return:
    """
    from logging import getLogger
    import os
    logger = getLogger()
    try:
        sftp = ssh.open_sftp()
        file_name = local_file_path.split('/')[-1]
        remote_file_path = os.path.join(upload_dir_path, file_name)
        sftp.put(local_file_path, remote_file_path)
        sftp.close()
        if connection_close:
            ssh.close()
        return True
    except (paramiko.SSHException, paramiko.SFTPError) as exception:
        logger.warning(exception)
        if propagate_exceptions:
            raise exception
        return False


def get_list_of_files(ssh: paramiko.client, remote_dir_path: str, propagate_exceptions=False) -> list:
    """
    Get list of file names in a remote directory
    :param ssh: connected SSH client
    :param remote_dir_path: path of the remote directory
    :param propagate_exceptions: (optional) backwards compatible flag to raise access permission issues
    :return: list of file names in the remote directory
    """
    from logging import getLogger
    logger = getLogger()
    list_of_files = list()
    try:
        sftp = ssh.open_sftp()
        sftp.chdir(remote_dir_path)
        for i in sftp.listdir():
            lstatout = str(sftp.lstat(i)).split()[0]
            if 'd' not in lstatout:
                list_of_files.append(i)
        sftp.close()
        ssh.close()
    except (paramiko.SSHException, paramiko.SFTPError) as exception:
        logger.warning(exception)
        if propagate_exceptions:
            raise exception
    return list_of_files


def get_list_of_files_by_pattern(ssh: paramiko.client, remote_dir_path: str, file_pattern: str,
                                 propagate_exceptions=False) -> list:
    """
    Get list of file names in a remote directory that match a wildcard string
    :param ssh: connected SSH client
    :param remote_dir_path: path of the remote directory
    :param file_pattern: file pattern (wildcard string)
    :param propagate_exceptions: (optional) backwards compatible flag to raise access permission issues
    :return: list of file names in the remote directory that match the wildcard string
    """
    from logging import getLogger
    import fnmatch
    logger = getLogger()
    list_of_files = list()
    try:
        sftp = ssh.open_sftp()
        sftp.chdir(remote_dir_path)
        for i in sftp.listdir():
            if fnmatch.fnmatch(i, file_pattern):
                list_of_files.append(i)
        sftp.close()
    except (paramiko.SSHException, paramiko.SFTPError) as exception:
        logger.warning(exception)
        if propagate_exceptions:
            raise exception
    return list_of_files


def get_file_content(ssh: paramiko.client, remote_dir_path: str, file_name: str, propagate_exceptions=False):
    """
    Read file content from remote path. Assumes Read access to the remote file
    :param ssh: connected SSH client
    :param remote_dir_path: path of the directory to read
    :param file_name: file to read
    :param propagate_exceptions: (optional) backwards compatible flag to raise access permission issues
    :return:
    """
    from logging import getLogger
    logger = getLogger()
    file_content = ''
    try:
        sftp = ssh.open_sftp()
        sftp.chdir(remote_dir_path)
        remote_file = sftp.open(file_name)
        file_content = remote_file.read()
        remote_file.close()
        sftp.close()
    except (paramiko.SSHException, paramiko.SFTPError) as exception:
        logger.warning(exception)
    return file_content


def is_file_exists(ssh: paramiko.client, remote_dir_path: str, file_name: str):
    """
    check file exists in remote path.
    :param ssh:
    :param remote_dir_path:
    :param file_name:
    :return:
    """
    try:
        sftp = ssh.open_sftp()
        sftp.chdir(remote_dir_path)
        sftp.stat(file_name)
        sftp.close()
        ssh.close()
        return True
    except IOError as exception:
        return False
