/**
 * This file contains all the major utilities necessary for writing and
 * reading code to a notebook, whether that be a Jupyterlab notebook or
 * a classic Jupyter Notebook.
 *
 * In the case of JupyterLab, we need to interact with the notebook through
 * commands defined on window.commands?.execute. This is because Lab forces
 * us to do this to access the cells of the notebook with a nice API.
 *
 * In the case of the classic Notebook, things are easier. We can interact
 * with the cells through window.Jupyter.notebook, and so we do this directly
 * in this case.
 *
 * These functions are wrappers over both of these, so that in Mito.tsx, we
 * only ever need to call one function and can keep things relatively clean.
 */
import { notebookGetArgs, notebookOverwriteAnalysisToReplayToMitosheetCall, notebookWriteAnalysisToReplayToMitosheetCall, notebookWriteGeneratedCodeToCell } from "./notebook/pluginUtils";
/**
 * NOTE: the next two functions are key to the proper functioning of Mito in
 * these two environments. As such, anytime we are in JupyterLab, the
 * isInJupyterLab MUST return true. We check a variety of these conditions
 * to see if this works (including in cases when mito is remote).
 *
 * If you change this code, make sure to test it with remove servers that
 * have non-standard URL schemes.
 */
export const isInJupyterLab = () => {
    return window.location.pathname.startsWith('/lab') ||
        window.commands !== undefined ||
        window._JUPYTERLAB !== undefined;
};
export const isInJupyterNotebook = () => {
    return window.location.pathname.startsWith('/notebooks') ||
        window.Jupyter !== undefined;
};
export const writeAnalysisToReplayToMitosheetCall = (analysisName, mitoAPI) => {
    var _a;
    if (isInJupyterLab()) {
        (_a = window.commands) === null || _a === void 0 ? void 0 : _a.execute('write-analysis-to-replay-to-mitosheet-call', {
            analysisName: analysisName,
            mitoAPI: mitoAPI
        });
    }
    else if (isInJupyterNotebook()) {
        notebookWriteAnalysisToReplayToMitosheetCall(analysisName, mitoAPI);
    }
    else {
        console.error("Not detected as in Jupyter Notebook or JupyterLab");
    }
};
export const overwriteAnalysisToReplayToMitosheetCall = (oldAnalysisName, newAnalysisName, mitoAPI) => {
    var _a;
    if (isInJupyterLab()) {
        (_a = window.commands) === null || _a === void 0 ? void 0 : _a.execute('overwrite-analysis-to-replay-to-mitosheet-call', {
            oldAnalysisName: oldAnalysisName,
            newAnalysisName: newAnalysisName,
            mitoAPI: mitoAPI
        });
    }
    else if (isInJupyterNotebook()) {
        notebookOverwriteAnalysisToReplayToMitosheetCall(oldAnalysisName, newAnalysisName, mitoAPI);
    }
    else {
        console.error("Not detected as in Jupyter Notebook or JupyterLab");
    }
};
export const writeGeneratedCodeToCell = (analysisName, code, telemetryEnabled) => {
    var _a;
    if (isInJupyterLab()) {
        (_a = window.commands) === null || _a === void 0 ? void 0 : _a.execute('write-generated-code-cell', {
            analysisName: analysisName,
            code: code,
            telemetryEnabled: telemetryEnabled,
        });
    }
    else if (isInJupyterNotebook()) {
        notebookWriteGeneratedCodeToCell(analysisName, code, telemetryEnabled);
    }
    else {
        console.error("Not detected as in Jupyter Notebook or JupyterLab");
    }
};
export const getArgs = (analysisToReplayName) => {
    return new Promise((resolve) => {
        var _a;
        if (isInJupyterLab()) {
            (_a = window.commands) === null || _a === void 0 ? void 0 : _a.execute('get-args', { analysisToReplayName: analysisToReplayName }).then(async (args) => {
                return resolve(args);
            });
            return;
        }
        else if (isInJupyterNotebook()) {
            return resolve(notebookGetArgs(analysisToReplayName));
        }
        else {
            console.error("Not detected as in Jupyter Notebook or JupyterLab");
        }
        return resolve([]);
    });
};
//# sourceMappingURL=jupyterUtils.js.map