import { isInJupyterLab } from "../jupyter/jupyterUtils";
import { ActionEnum } from "../types";
import { useDebouncedEffect } from "./useDebouncedEffect";
const KEYBOARD_SHORTCUTS = {
    'c': ActionEnum.Copy,
    'z': ActionEnum.Undo,
    'y': ActionEnum.Redo,
};
// See comment in plugin.tsx. Because of some JupyterLab plugin issues
// we cannot detect some key press events, and thus we ignore
// some keypresses on JLab to be explicit about where things run.
// NOTE: ignoring these is not actually required, as we don't get
// these keydown events anyways, but I want to be explicit here!
const JUPYTER_LAB_SHORTCUTS_DEFINED_ELSEWHERE = ['z', 'y'];
/*
    This effect actually does keyboard shortcuts.
*/
export const useKeyboardShortcuts = (mitoContainerRef, actions, setGridState) => {
    // NOTE: this effect must be debounced so that we're not reregistering these event
    // listeners 100 times during every single scroll. In practice, this works perf!
    useDebouncedEffect(() => {
        const checkKeyboardShortCut = (e) => {
            var _a, _b;
            // First, we check the user is doing a keyboard shortcut
            if (!Object.keys(KEYBOARD_SHORTCUTS).includes(e.key) || (!e.ctrlKey && !e.metaKey)) {
                return;
            }
            // We have a special case here if the user is doing a copy, where we need to clear
            // the previously copied values. This should always run, even if we're not in this
            // specific mito instance, because this clears the copy anyways
            if (e.key === 'c') {
                setGridState(prevGridState => {
                    return Object.assign(Object.assign({}, prevGridState), { copiedSelections: [] });
                });
            }
            // Then, check that this was actually done by a focus on this mitosheet
            if (!((_a = mitoContainerRef.current) === null || _a === void 0 ? void 0 : _a.contains(document.activeElement))) {
                return;
            }
            // We check if the user is doing a shortcut that need not be defined on lab
            if (JUPYTER_LAB_SHORTCUTS_DEFINED_ELSEWHERE.includes(e.key) && isInJupyterLab()) {
                return;
            }
            // Then, we check if the user is actually focused on some input in Mito,
            // as in this case we don't want to overwrite this action
            if (((_b = document.activeElement) === null || _b === void 0 ? void 0 : _b.tagName.toLowerCase()) === 'input') {
                return;
            }
            // Because JupyterLab has some other event listeners that do weird things with
            // key presses, we stop this from going elsewhere
            e.stopImmediatePropagation();
            e.stopPropagation();
            e.preventDefault();
            actions[KEYBOARD_SHORTCUTS[e.key]].actionFunction();
        };
        document.addEventListener('keydown', checkKeyboardShortCut);
        return () => { document.removeEventListener('keydown', checkKeyboardShortCut); };
    }, [actions], 50);
};
//# sourceMappingURL=useKeyboardShortcuts.js.map