#!/usr/bin/env python3

# © Copyright 2021-2022, Scott Gasch

"""
A bunch of color names mapped into RGB tuples and some methods for
setting the text color, background color, style, etc... using ANSI escape
sequences.  See: https://en.wikipedia.org/wiki/ANSI_escape_code.
"""

import contextlib
import difflib
import io
import logging
import re
import sys
from abc import abstractmethod
from typing import Callable, Dict, Iterable, Optional, Tuple

from overrides import overrides

from pyutils import logging_utils, string_utils

logger = logging.getLogger(__name__)


COLOR_NAMES_TO_RGB: Dict[str, Tuple[int, int, int]] = {
    "abbey": (0x4C, 0x4F, 0x56),
    "acadia": (0x1B, 0x14, 0x04),
    "acapulco": (0x7C, 0xB0, 0xA1),
    "aero blue": (0xC9, 0xFF, 0xE5),
    "affair": (0x71, 0x46, 0x93),
    "akaroa": (0xD4, 0xC4, 0xA8),
    "alabaster": (0xFA, 0xFA, 0xFA),
    "albescent white": (0xF5, 0xE9, 0xD3),
    "algae green": (0x93, 0xDF, 0xB8),
    "alice blue": (0xF0, 0xF8, 0xFF),
    "alizarin crimson": (0xE3, 0x26, 0x36),
    "allports": (0x00, 0x76, 0xA3),
    "almond frost": (0x90, 0x7B, 0x71),
    "almond": (0xEE, 0xD9, 0xC4),
    "alpine": (0xAF, 0x8F, 0x2C),
    "alto": (0xDB, 0xDB, 0xDB),
    "aluminium": (0xA9, 0xAC, 0xB6),
    "amaranth": (0xE5, 0x2B, 0x50),
    "amazon": (0x3B, 0x7A, 0x57),
    "amber": (0xFF, 0xBF, 0x00),
    "americano": (0x87, 0x75, 0x6E),
    "amethyst smoke": (0xA3, 0x97, 0xB4),
    "amethyst": (0x99, 0x66, 0xCC),
    "amour": (0xF9, 0xEA, 0xF3),
    "amulet": (0x7B, 0x9F, 0x80),
    "anakiwa": (0x9D, 0xE5, 0xFF),
    "antique brass": (0xC8, 0x8A, 0x65),
    "antique bronze": (0x70, 0x4A, 0x07),
    "antique white": (0xFA, 0xEB, 0xD7),
    "anzac": (0xE0, 0xB6, 0x46),
    "apache": (0xDF, 0xBE, 0x6F),
    "apple blossom": (0xAF, 0x4D, 0x43),
    "apple green": (0xE2, 0xF3, 0xEC),
    "apple": (0x4F, 0xA8, 0x3D),
    "apricot peach": (0xFB, 0xCE, 0xB1),
    "apricot white": (0xFF, 0xFE, 0xEC),
    "apricot": (0xEB, 0x93, 0x73),
    "aqua deep": (0x01, 0x4B, 0x43),
    "aqua forest": (0x5F, 0xA7, 0x77),
    "aqua haze": (0xED, 0xF5, 0xF5),
    "aqua island": (0xA1, 0xDA, 0xD7),
    "aqua spring": (0xEA, 0xF9, 0xF5),
    "aqua squeeze": (0xE8, 0xF5, 0xF2),
    "aqua": (0x00, 0xFF, 0xFF),
    "aquamarine blue": (0x71, 0xD9, 0xE2),
    "aquamarine": (0x7F, 0xFF, 0xD4),
    "arapawa": (0x11, 0x0C, 0x6C),
    "armadillo": (0x43, 0x3E, 0x37),
    "arrowtown": (0x94, 0x87, 0x71),
    "ash": (0xC6, 0xC3, 0xB5),
    "asparagus": (0x7B, 0xA0, 0x5B),
    "asphalt": (0x13, 0x0A, 0x06),
    "astra": (0xFA, 0xEA, 0xB9),
    "astral": (0x32, 0x7D, 0xA0),
    "astronaut blue": (0x01, 0x3E, 0x62),
    "astronaut": (0x28, 0x3A, 0x77),
    "athens gray": (0xEE, 0xF0, 0xF3),
    "aths special": (0xEC, 0xEB, 0xCE),
    "atlantis": (0x97, 0xCD, 0x2D),
    "atoll": (0x0A, 0x6F, 0x75),
    "atomic tangerine": (0xFF, 0x99, 0x66),
    "au chico": (0x97, 0x60, 0x5D),
    "aubergine": (0x3B, 0x09, 0x10),
    "australian mint": (0xF5, 0xFF, 0xBE),
    "avocado": (0x88, 0x8D, 0x65),
    "axolotl": (0x4E, 0x66, 0x49),
    "azalea": (0xF7, 0xC8, 0xDA),
    "aztec": (0x0D, 0x1C, 0x19),
    "azure radiance": (0x00, 0x7F, 0xFF),
    "azure": (0xF0, 0xFF, 0xFF),
    "baby blue": (0xE0, 0xFF, 0xFF),
    "backup.house": (175, 95, 0),
    "bahama blue": (0x02, 0x63, 0x95),
    "bahia": (0xA5, 0xCB, 0x0C),
    "baja white": (0xFF, 0xF8, 0xD1),
    "bali hai": (0x85, 0x9F, 0xAF),
    "baltic sea": (0x2A, 0x26, 0x30),
    "bamboo": (0xDA, 0x63, 0x04),
    "banana mania": (0xFB, 0xE7, 0xB2),
    "bandicoot": (0x85, 0x84, 0x70),
    "barberry": (0xDE, 0xD7, 0x17),
    "barley corn": (0xA6, 0x8B, 0x5B),
    "barley white": (0xFF, 0xF4, 0xCE),
    "barossa": (0x44, 0x01, 0x2D),
    "bastille": (0x29, 0x21, 0x30),
    "battleship gray": (0x82, 0x8F, 0x72),
    "bay leaf": (0x7D, 0xA9, 0x8D),
    "bay of many": (0x27, 0x3A, 0x81),
    "bazaar": (0x98, 0x77, 0x7B),
    "bean  ": (0x3D, 0x0C, 0x02),
    "beauty bush": (0xEE, 0xC1, 0xBE),
    "beaver": (0x92, 0x6F, 0x5B),
    "beeswax": (0xFE, 0xF2, 0xC7),
    "beige": (0xF5, 0xF5, 0xDC),
    "bermuda gray": (0x6B, 0x8B, 0xA2),
    "bermuda": (0x7D, 0xD8, 0xC6),
    "beryl green": (0xDE, 0xE5, 0xC0),
    "bianca": (0xFC, 0xFB, 0xF3),
    "big stone": (0x16, 0x2A, 0x40),
    "bilbao": (0x32, 0x7C, 0x14),
    "biloba flower": (0xB2, 0xA1, 0xEA),
    "birch": (0x37, 0x30, 0x21),
    "bird flower": (0xD4, 0xCD, 0x16),
    "biscay": (0x1B, 0x31, 0x62),
    "bismark": (0x49, 0x71, 0x83),
    "bison hide": (0xC1, 0xB7, 0xA4),
    "bisque": (0xFF, 0xE4, 0xC4),
    "bistre": (0x3D, 0x2B, 0x1F),
    "bitter lemon": (0xCA, 0xE0, 0x0D),
    "bitter": (0x86, 0x89, 0x74),
    "bittersweet": (0xFE, 0x6F, 0x5E),
    "bizarre": (0xEE, 0xDE, 0xDA),
    "black bean": (0x08, 0x19, 0x10),
    "black forest": (0x0B, 0x13, 0x04),
    "black haze": (0xF6, 0xF7, 0xF7),
    "black marlin": (0x3E, 0x2C, 0x1C),
    "black olive": (0x24, 0x2E, 0x16),
    "black pearl": (0x04, 0x13, 0x22),
    "black rock": (0x0D, 0x03, 0x32),
    "black rose": (0x67, 0x03, 0x2D),
    "black russian": (0x0A, 0x00, 0x1C),
    "black squeeze": (0xF2, 0xFA, 0xFA),
    "black white": (0xFF, 0xFE, 0xF6),
    "black": (0x00, 0x00, 0x00),
    "blackberry": (0x4D, 0x01, 0x35),
    "blackcurrant": (0x32, 0x29, 0x3A),
    "blanched almond": (0xFF, 0xEB, 0xCD),
    "blaze orange": (0xFF, 0x66, 0x00),
    "bleach white": (0xFE, 0xF3, 0xD8),
    "bleached cedar": (0x2C, 0x21, 0x33),
    "blizzard blue": (0xA3, 0xE3, 0xED),
    "blossom": (0xDC, 0xB4, 0xBC),
    "blue bayoux": (0x49, 0x66, 0x79),
    "blue bell": (0x99, 0x99, 0xCC),
    "blue chalk": (0xF1, 0xE9, 0xFF),
    "blue charcoal": (0x01, 0x0D, 0x1A),
    "blue chill": (0x0C, 0x89, 0x90),
    "blue diamond": (0x38, 0x04, 0x74),
    "blue dianne": (0x20, 0x48, 0x52),
    "blue gem": (0x2C, 0x0E, 0x8C),
    "blue haze": (0xBF, 0xBE, 0xD8),
    "blue lagoon": (0x01, 0x79, 0x87),
    "blue marguerite": (0x76, 0x66, 0xC6),
    "blue ribbon": (0x00, 0x66, 0xFF),
    "blue romance": (0xD2, 0xF6, 0xDE),
    "blue smoke": (0x74, 0x88, 0x81),
    "blue stone": (0x01, 0x61, 0x62),
    "blue violet": (0x8A, 0x2B, 0xE2),
    "blue whale": (0x04, 0x2E, 0x4C),
    "blue zodiac": (0x13, 0x26, 0x4D),
    "blue": (0x00, 0x00, 0xFF),
    "blumine": (0x18, 0x58, 0x7A),
    "blush pink": (0xFF, 0x6F, 0xFF),
    "blush": (0xB4, 0x46, 0x68),
    "bombay": (0xAF, 0xB1, 0xB8),
    "bon jour": (0xE5, 0xE0, 0xE1),
    "bondi blue": (0x00, 0x95, 0xB6),
    "bone": (0xE4, 0xD1, 0xC0),
    "bordeaux": (0x5C, 0x01, 0x20),
    "bossanova": (0x4E, 0x2A, 0x5A),
    "boston blue": (0x3B, 0x91, 0xB4),
    "botticelli": (0xC7, 0xDD, 0xE5),
    "bottle green": (0x09, 0x36, 0x24),
    "boulder": (0x7A, 0x7A, 0x7A),
    "bouquet": (0xAE, 0x80, 0x9E),
    "bourbon": (0xBA, 0x6F, 0x1E),
    "bracken": (0x4A, 0x2A, 0x04),
    "brandy punch": (0xCD, 0x84, 0x29),
    "brandy rose": (0xBB, 0x89, 0x83),
    "brandy": (0xDE, 0xC1, 0x96),
    "breaker bay": (0x5D, 0xA1, 0x9F),
    "brick red": (0xC6, 0x2D, 0x42),
    "bridal heath": (0xFF, 0xFA, 0xF4),
    "bridesmaid": (0xFE, 0xF0, 0xEC),
    "bright gray": (0x3C, 0x41, 0x51),
    "bright green": (0x66, 0xFF, 0x00),
    "bright red": (0xB1, 0x00, 0x00),
    "bright sun": (0xFE, 0xD3, 0x3C),
    "bright turquoise": (0x08, 0xE8, 0xDE),
    "brilliant rose": (0xF6, 0x53, 0xA6),
    "brink pink": (0xFB, 0x60, 0x7F),
    "bronco": (0xAB, 0xA1, 0x96),
    "bronze olive": (0x4E, 0x42, 0x0C),
    "bronze": (0x3F, 0x21, 0x09),
    "bronzetone": (0x4D, 0x40, 0x0F),
    "broom": (0xFF, 0xEC, 0x13),
    "brown bramble": (0x59, 0x28, 0x04),
    "brown derby": (0x49, 0x26, 0x15),
    "brown pod": (0x40, 0x18, 0x01),
    "brown rust": (0xAF, 0x59, 0x3E),
    "brown tumbleweed": (0x37, 0x29, 0x0E),
    "brown": (0x96, 0x4B, 0x00),
    "bubbles": (0xE7, 0xFE, 0xFF),
    "buccaneer": (0x62, 0x2F, 0x30),
    "bud": (0xA8, 0xAE, 0x9C),
    "buddha gold": (0xC1, 0xA0, 0x04),
    "buff": (0xF0, 0xDC, 0x82),
    "bulgarian rose": (0x48, 0x06, 0x07),
    "bull shot": (0x86, 0x4D, 0x1E),
    "bunker": (0x0D, 0x11, 0x17),
    "bunting": (0x15, 0x1F, 0x4C),
    "burgundy": (0x90, 0x00, 0x20),
    "burlywood": (0xDE, 0xB8, 0x87),
    "burnham": (0x00, 0x2E, 0x20),
    "burning orange": (0xFF, 0x70, 0x34),
    "burning sand": (0xD9, 0x93, 0x76),
    "burnt maroon": (0x42, 0x03, 0x03),
    "burnt orange": (0xCC, 0x55, 0x00),
    "burnt sienna": (0xE9, 0x74, 0x51),
    "burnt umber": (0x8A, 0x33, 0x24),
    "bush": (0x0D, 0x2E, 0x1C),
    "buttercup": (0xF3, 0xAD, 0x16),
    "buttered rum": (0xA1, 0x75, 0x0D),
    "butterfly bush": (0x62, 0x4E, 0x9A),
    "buttermilk": (0xFF, 0xF1, 0xB5),
    "buttery white": (0xFF, 0xFC, 0xEA),
    "cab sav": (0x4D, 0x0A, 0x18),
    "cabaret": (0xD9, 0x49, 0x72),
    "cabbage pont": (0x3F, 0x4C, 0x3A),
    "cactus": (0x58, 0x71, 0x56),
    "cadet blue": (0x5F, 0x9E, 0xA0),
    "cadillac": (0xB0, 0x4C, 0x6A),
    "cafe royale": (0x6F, 0x44, 0x0C),
    "calico": (0xE0, 0xC0, 0x95),
    "california": (0xFE, 0x9D, 0x04),
    "calypso": (0x31, 0x72, 0x8D),
    "camarone": (0x00, 0x58, 0x1A),
    "camelot": (0x89, 0x34, 0x56),
    "cameo": (0xD9, 0xB9, 0x9B),
    "camouflage green": (0x78, 0x86, 0x6B),
    "camouflage": (0x3C, 0x39, 0x10),
    "can can": (0xD5, 0x91, 0xA4),
    "canary": (0xF3, 0xFB, 0x62),
    "candlelight": (0xFC, 0xD9, 0x17),
    "candy corn": (0xFB, 0xEC, 0x5D),
    "cannon black": (0x25, 0x17, 0x06),
    "cannon pink": (0x89, 0x43, 0x67),
    "cape cod": (0x3C, 0x44, 0x43),
    "cape honey": (0xFE, 0xE5, 0xAC),
    "cape palliser": (0xA2, 0x66, 0x45),
    "caper": (0xDC, 0xED, 0xB4),
    "caramel": (0xFF, 0xDD, 0xAF),
    "cararra": (0xEE, 0xEE, 0xE8),
    "cardin green": (0x01, 0x36, 0x1C),
    "cardinal pink": (0x8C, 0x05, 0x5E),
    "cardinal": (0xC4, 0x1E, 0x3A),
    "careys pink": (0xD2, 0x9E, 0xAA),
    "caribbean green": (0x00, 0xCC, 0x99),
    "carissma": (0xEA, 0x88, 0xA8),
    "carla": (0xF3, 0xFF, 0xD8),
    "carmine": (0x96, 0x00, 0x18),
    "carnaby tan": (0x5C, 0x2E, 0x01),
    "carnation pink": (0xFF, 0xA6, 0xC9),
    "carnation": (0xF9, 0x5A, 0x61),
    "carousel pink": (0xF9, 0xE0, 0xED),
    "carrot orange": (0xED, 0x91, 0x21),
    "casablanca": (0xF8, 0xB8, 0x53),
    "casal": (0x2F, 0x61, 0x68),
    "cascade": (0x8B, 0xA9, 0xA5),
    "cashmere": (0xE6, 0xBE, 0xA5),
    "casper": (0xAD, 0xBE, 0xD1),
    "castro": (0x52, 0x00, 0x1F),
    "catalina blue": (0x06, 0x2A, 0x78),
    "catskill white": (0xEE, 0xF6, 0xF7),
    "cavern pink": (0xE3, 0xBE, 0xBE),
    "cedar wood finish": (0x71, 0x1A, 0x00),
    "cedar": (0x3E, 0x1C, 0x14),
    "celadon": (0xAC, 0xE1, 0xAF),
    "celery": (0xB8, 0xC2, 0x5D),
    "celeste": (0xD1, 0xD2, 0xCA),
    "cello": (0x1E, 0x38, 0x5B),
    "celtic": (0x16, 0x32, 0x22),
    "cement": (0x8D, 0x76, 0x62),
    "ceramic": (0xFC, 0xFF, 0xF9),
    "cerise red": (0xDE, 0x31, 0x63),
    "cerise": (0xDA, 0x32, 0x87),
    "cerulean blue": (0x2A, 0x52, 0xBE),
    "cerulean": (0x02, 0xA4, 0xD3),
    "chablis": (0xFF, 0xF4, 0xF3),
    "chalet green": (0x51, 0x6E, 0x3D),
    "chalky": (0xEE, 0xD7, 0x94),
    "chambray": (0x35, 0x4E, 0x8C),
    "chamois": (0xED, 0xDC, 0xB1),
    "champagne": (0xFA, 0xEC, 0xCC),
    "chantilly": (0xF8, 0xC3, 0xDF),
    "charade": (0x29, 0x29, 0x37),
    "chardon": (0xFF, 0xF3, 0xF1),
    "chardonnay": (0xFF, 0xCD, 0x8C),
    "charlotte": (0xBA, 0xEE, 0xF9),
    "charm": (0xD4, 0x74, 0x94),
    "chartreuse yellow": (0xDF, 0xFF, 0x00),
    "chartreuse": (0x7F, 0xFF, 0x00),
    "chateau green": (0x40, 0xA8, 0x60),
    "chatelle": (0xBD, 0xB3, 0xC7),
    "chathams blue": (0x17, 0x55, 0x79),
    "cheetah.house": (95, 0x00, 0x00),
    "chelsea cucumber": (0x83, 0xAA, 0x5D),
    "chelsea gem": (0x9E, 0x53, 0x02),
    "chenin": (0xDF, 0xCD, 0x6F),
    "cherokee": (0xFC, 0xDA, 0x98),
    "cherry pie": (0x2A, 0x03, 0x59),
    "cherrywood": (0x65, 0x1A, 0x14),
    "cherub": (0xF8, 0xD9, 0xE9),
    "chestnut rose": (0xCD, 0x5C, 0x5C),
    "chestnut": (0xB9, 0x4E, 0x48),
    "chetwode blue": (0x85, 0x81, 0xD9),
    "chicago": (0x5D, 0x5C, 0x58),
    "chiffon": (0xF1, 0xFF, 0xC8),
    "chilean fire": (0xF7, 0x77, 0x03),
    "chilean heath": (0xFF, 0xFD, 0xE6),
    "china ivory": (0xFC, 0xFF, 0xE7),
    "chino": (0xCE, 0xC7, 0xA7),
    "chinook": (0xA8, 0xE3, 0xBD),
    "chocolate": (0x37, 0x02, 0x02),
    "christalle": (0x33, 0x03, 0x6B),
    "christi": (0x67, 0xA7, 0x12),
    "christine": (0xE7, 0x73, 0x0A),
    "chrome white": (0xE8, 0xF1, 0xD4),
    "cinder": (0x0E, 0x0E, 0x18),
    "cinderella": (0xFD, 0xE1, 0xDC),
    "cinnabar": (0xE3, 0x42, 0x34),
    "cinnamon": (0x7B, 0x3F, 0x00),
    "cioccolato": (0x55, 0x28, 0x0C),
    "citrine white": (0xFA, 0xF7, 0xD6),
    "citron": (0x9E, 0xA9, 0x1F),
    "citrus": (0xA1, 0xC5, 0x0A),
    "clairvoyant": (0x48, 0x06, 0x56),
    "clam shell": (0xD4, 0xB6, 0xAF),
    "claret": (0x7F, 0x17, 0x34),
    "classic rose": (0xFB, 0xCC, 0xE7),
    "clay ash": (0xBD, 0xC8, 0xB3),
    "clay creek": (0x8A, 0x83, 0x60),
    "clear day": (0xE9, 0xFF, 0xFD),
    "clementine": (0xE9, 0x6E, 0x00),
    "clinker": (0x37, 0x1D, 0x09),
    "cloud burst": (0x20, 0x2E, 0x54),
    "cloud": (0xC7, 0xC4, 0xBF),
    "cloudy": (0xAC, 0xA5, 0x9F),
    "clover": (0x38, 0x49, 0x10),
    "cobalt": (0x00, 0x47, 0xAB),
    "cocoa bean": (0x48, 0x1C, 0x1C),
    "cocoa brown": (0x30, 0x1F, 0x1E),
    "coconut cream": (0xF8, 0xF7, 0xDC),
    "cod gray": (0x0B, 0x0B, 0x0B),
    "coffee bean": (0x2A, 0x14, 0x0E),
    "coffee": (0x70, 0x65, 0x55),
    "cognac": (0x9F, 0x38, 0x1D),
    "cola": (0x3F, 0x25, 0x00),
    "cold purple": (0xAB, 0xA0, 0xD9),
    "cold turkey": (0xCE, 0xBA, 0xBA),
    "colonial white": (0xFF, 0xED, 0xBC),
    "comet": (0x5C, 0x5D, 0x75),
    "como": (0x51, 0x7C, 0x66),
    "conch": (0xC9, 0xD9, 0xD2),
    "concord": (0x7C, 0x7B, 0x7A),
    "concrete": (0xF2, 0xF2, 0xF2),
    "confetti": (0xE9, 0xD7, 0x5A),
    "congo brown": (0x59, 0x37, 0x37),
    "congress blue": (0x02, 0x47, 0x8E),
    "conifer": (0xAC, 0xDD, 0x4D),
    "contessa": (0xC6, 0x72, 0x6B),
    "copper canyon": (0x7E, 0x3A, 0x15),
    "copper rose": (0x99, 0x66, 0x66),
    "copper rust": (0x94, 0x47, 0x47),
    "copper": (0xB8, 0x73, 0x33),
    "copperfield": (0xDA, 0x8A, 0x67),
    "coral red": (0xFF, 0x40, 0x40),
    "coral reef": (0xC7, 0xBC, 0xA2),
    "coral tree": (0xA8, 0x6B, 0x6B),
    "coral": (0xFF, 0x7F, 0x50),
    "corduroy": (0x60, 0x6E, 0x68),
    "coriander": (0xC4, 0xD0, 0xB0),
    "cork": (0x40, 0x29, 0x1D),
    "corn field": (0xF8, 0xFA, 0xCD),
    "corn harvest": (0x8B, 0x6B, 0x0B),
    "corn silk": (0xFF, 0xF8, 0xDC),
    "corn": (0xE7, 0xBF, 0x05),
    "cornflower blue": (0x64, 0x95, 0xED),
    "cornflower lilac": (0xFF, 0xB0, 0xAC),
    "cornflower": (0x93, 0xCC, 0xEA),
    "corvette": (0xFA, 0xD3, 0xA2),
    "cosmic": (0x76, 0x39, 0x5D),
    "cosmos": (0xFF, 0xD8, 0xD9),
    "costa del sol": (0x61, 0x5D, 0x30),
    "cotton candy": (0xFF, 0xB7, 0xD5),
    "cotton seed": (0xC2, 0xBD, 0xB6),
    "county green": (0x01, 0x37, 0x1A),
    "cowboy": (0x4D, 0x28, 0x2D),
    "crail": (0xB9, 0x51, 0x40),
    "cranberry": (0xDB, 0x50, 0x79),
    "crater brown": (0x46, 0x24, 0x25),
    "cream brulee": (0xFF, 0xE5, 0xA0),
    "cream can": (0xF5, 0xC8, 0x5C),
    "cream": (0xFF, 0xFD, 0xD0),
    "creole": (0x1E, 0x0F, 0x04),
    "crete": (0x73, 0x78, 0x29),
    "crimson": (0xDC, 0x14, 0x3C),
    "crocodile": (0x73, 0x6D, 0x58),
    "crown of thorns": (0x77, 0x1F, 0x1F),
    "crowshead": (0x1C, 0x12, 0x08),
    "cruise": (0xB5, 0xEC, 0xDF),
    "crusoe": (0x00, 0x48, 0x16),
    "crusta": (0xFD, 0x7B, 0x33),
    "cumin": (0x92, 0x43, 0x21),
    "cumulus": (0xFD, 0xFF, 0xD5),
    "cupid": (0xFB, 0xBE, 0xDA),
    "curious blue": (0x25, 0x96, 0xD1),
    "cutty sark": (0x50, 0x76, 0x72),
    "cyan": (0x00, 0xFF, 0xFF),
    "cyprus": (0x00, 0x3E, 0x40),
    "daintree": (0x01, 0x27, 0x31),
    "dairy cream": (0xF9, 0xE4, 0xBC),
    "daisy bush": (0x4F, 0x23, 0x98),
    "dallas": (0x6E, 0x4B, 0x26),
    "dandelion": (0xFE, 0xD8, 0x5D),
    "danube": (0x60, 0x93, 0xD1),
    "dark blue": (0x00, 0x00, 0x8B),
    "dark burgundy": (0x77, 0x0F, 0x05),
    "dark cyan": (0x00, 0x8B, 0x8B),
    "dark ebony": (0x3C, 0x20, 0x05),
    "dark fern": (0x0A, 0x48, 0x0D),
    "dark goldenrod": (0xB8, 0x86, 0x0B),
    "dark gray": (0xA9, 0xA9, 0xA9),
    "dark green": (0x18, 0x2D, 0x09),
    "dark magenta": (0xAF, 0x00, 0xAF),
    "dark olive green": (0x55, 0x6B, 0x2F),
    "dark orange": (0xFF, 0x8C, 0x00),
    "dark orchid": (0x99, 0x32, 0xCC),
    "dark purple": (0x36, 0x00, 0x79),
    "dark red": (0x64, 0x00, 0x00),
    "dark salmon": (0xE9, 0x96, 0x7A),
    "dark sea green": (0x8F, 0xBC, 0x8F),
    "dark slate gray": (0x2F, 0x4F, 0x4F),
    "dark tan": (0x66, 0x10, 0x10),
    "dark turquoise": (0x00, 0xCE, 0xD1),
    "dark violet": (0x94, 0x00, 0xD3),
    "dawn pink": (0xF3, 0xE9, 0xE5),
    "dawn": (0xA6, 0xA2, 0x9A),
    "de york": (0x7A, 0xC4, 0x88),
    "deco": (0xD2, 0xDA, 0x97),
    "deep blue": (0x22, 0x08, 0x78),
    "deep blush": (0xE4, 0x76, 0x98),
    "deep bronze": (0x4A, 0x30, 0x04),
    "deep cerulean": (0x00, 0x7B, 0xA7),
    "deep cove": (0x05, 0x10, 0x40),
    "deep fir": (0x00, 0x29, 0x00),
    "deep forest green": (0x18, 0x2D, 0x09),
    "deep koamaru": (0x1B, 0x12, 0x7B),
    "deep oak": (0x41, 0x20, 0x10),
    "deep pink": (0xFF, 0x14, 0x93),
    "deep sapphire": (0x08, 0x25, 0x67),
    "deep sea green": (0x09, 0x58, 0x59),
    "deep sea": (0x01, 0x82, 0x6B),
    "deep sky blue": (0x00, 0xBF, 0xFF),
    "deep teal": (0x00, 0x35, 0x32),
    "del rio": (0xB0, 0x9A, 0x95),
    "dell": (0x39, 0x64, 0x13),
    "delta": (0xA4, 0xA4, 0x9D),
    "deluge": (0x75, 0x63, 0xA8),
    "denim": (0x15, 0x60, 0xBD),
    "derby": (0xFF, 0xEE, 0xD8),
    "desert sand": (0xED, 0xC9, 0xAF),
    "desert storm": (0xF8, 0xF8, 0xF7),
    "desert": (0xAE, 0x60, 0x20),
    "dew": (0xEA, 0xFF, 0xFE),
    "di serria": (0xDB, 0x99, 0x5E),
    "diesel": (0x13, 0x00, 0x00),
    "dim gray": (0x69, 0x69, 0x69),
    "dingley": (0x5D, 0x77, 0x47),
    "disco": (0x87, 0x15, 0x50),
    "dixie": (0xE2, 0x94, 0x18),
    "dodger blue": (0x1E, 0x90, 0xFF),
    "dolly": (0xF9, 0xFF, 0x8B),
    "dolphin": (0x64, 0x60, 0x77),
    "domino": (0x8E, 0x77, 0x5E),
    "don juan": (0x5D, 0x4C, 0x51),
    "donkey brown": (0xA6, 0x92, 0x79),
    "dorado": (0x6B, 0x57, 0x55),
    "double colonial white": (0xEE, 0xE3, 0xAD),
    "double pearl lusta": (0xFC, 0xF4, 0xD0),
    "double spanish white": (0xE6, 0xD7, 0xB9),
    "dove gray": (0x6D, 0x6C, 0x6C),
    "downriver": (0x09, 0x22, 0x56),
    "downy": (0x6F, 0xD0, 0xC5),
    "driftwood": (0xAF, 0x87, 0x51),
    "drover": (0xFD, 0xF7, 0xAD),
    "dull lavender": (0xA8, 0x99, 0xE6),
    "dune": (0x38, 0x35, 0x33),
    "dust storm": (0xE5, 0xCC, 0xC9),
    "dusty gray": (0xA8, 0x98, 0x9B),
    "eagle": (0xB6, 0xBA, 0xA4),
    "earls green": (0xC9, 0xB9, 0x3B),
    "early dawn": (0xFF, 0xF9, 0xE6),
    "east bay": (0x41, 0x4C, 0x7D),
    "east side": (0xAC, 0x91, 0xCE),
    "eastern blue": (0x1E, 0x9A, 0xB0),
    "ebb": (0xE9, 0xE3, 0xE3),
    "ebony clay": (0x26, 0x28, 0x3B),
    "ebony": (0x0C, 0x0B, 0x1D),
    "eclipse": (0x31, 0x1C, 0x17),
    "ecru white": (0xF5, 0xF3, 0xE5),
    "ecstasy": (0xFA, 0x78, 0x14),
    "eden": (0x10, 0x58, 0x52),
    "edgewater": (0xC8, 0xE3, 0xD7),
    "edward": (0xA2, 0xAE, 0xAB),
    "egg sour": (0xFF, 0xF4, 0xDD),
    "egg white": (0xFF, 0xEF, 0xC1),
    "eggplant": (0x61, 0x40, 0x51),
    "el paso": (0x1E, 0x17, 0x08),
    "el salva": (0x8F, 0x3E, 0x33),
    "electric lime": (0xCC, 0xFF, 0x00),
    "electric violet": (0x8B, 0x00, 0xFF),
    "elephant": (0x12, 0x34, 0x47),
    "elf green": (0x1B, 0x8A, 0x6B),
    "elm": (0x1C, 0x7C, 0x7D),
    "emerald": (0x50, 0xC8, 0x78),
    "eminence": (0x6C, 0x30, 0x82),
    "emperor": (0x51, 0x46, 0x49),
    "empress": (0x81, 0x73, 0x77),
    "endeavour": (0x00, 0x56, 0xA7),
    "energy yellow": (0xF8, 0xDD, 0x5C),
    "english holly": (0x02, 0x2D, 0x15),
    "english walnut": (0x3E, 0x2B, 0x23),
    "envy": (0x8B, 0xA6, 0x90),
    "equator": (0xE1, 0xBC, 0x64),
    "espresso": (0x61, 0x27, 0x18),
    "eternity": (0x21, 0x1A, 0x0E),
    "eucalyptus": (0x27, 0x8A, 0x5B),
    "eunry": (0xCF, 0xA3, 0x9D),
    "evening sea": (0x02, 0x4E, 0x46),
    "everglade": (0x1C, 0x40, 0x2E),
    "faded jade": (0x42, 0x79, 0x77),
    "fair pink": (0xFF, 0xEF, 0xEC),
    "falcon": (0x7F, 0x62, 0x6D),
    "fall green": (0xEC, 0xEB, 0xBD),
    "falu red": (0x80, 0x18, 0x18),
    "fantasy": (0xFA, 0xF3, 0xF0),
    "fedora": (0x79, 0x6A, 0x78),
    "feijoa": (0x9F, 0xDD, 0x8C),
    "fern frond": (0x65, 0x72, 0x20),
    "fern green": (0x4F, 0x79, 0x42),
    "fern": (0x63, 0xB7, 0x6C),
    "ferra": (0x70, 0x4F, 0x50),
    "festival": (0xFB, 0xE9, 0x6C),
    "feta": (0xF0, 0xFC, 0xEA),
    "fiery orange": (0xB3, 0x52, 0x13),
    "finch": (0x62, 0x66, 0x49),
    "finlandia": (0x55, 0x6D, 0x56),
    "finn": (0x69, 0x2D, 0x54),
    "fiord": (0x40, 0x51, 0x69),
    "fire brick": (0xB2, 0x22, 0x22),
    "fire bush": (0xE8, 0x99, 0x28),
    "fire": (0xAA, 0x42, 0x03),
    "firefly": (0x0E, 0x2A, 0x30),
    "flame pea": (0xDA, 0x5B, 0x38),
    "flamenco": (0xFF, 0x7D, 0x07),
    "flamingo": (0xF2, 0x55, 0x2A),
    "flax smoke": (0x7B, 0x82, 0x65),
    "flax": (0xEE, 0xDC, 0x82),
    "flesh": (0xFF, 0xCB, 0xA4),
    "flint": (0x6F, 0x6A, 0x61),
    "flirt": (0xA2, 0x00, 0x6D),
    "floral white": (0xFF, 0xFA, 0xF0),
    "flush mahogany": (0xCA, 0x34, 0x35),
    "flush orange": (0xFF, 0x7F, 0x00),
    "foam": (0xD8, 0xFC, 0xFA),
    "fog": (0xD7, 0xD0, 0xFF),
    "foggy gray": (0xCB, 0xCA, 0xB6),
    "forest green": (0x22, 0x8B, 0x22),
    "forget me not": (0xFF, 0xF1, 0xEE),
    "fountain blue": (0x56, 0xB4, 0xBE),
    "frangipani": (0xFF, 0xDE, 0xB3),
    "french gray": (0xBD, 0xBD, 0xC6),
    "french lilac": (0xEC, 0xC7, 0xEE),
    "french pass": (0xBD, 0xED, 0xFD),
    "french rose": (0xF6, 0x4A, 0x8A),
    "fresh eggplant": (0x99, 0x00, 0x66),
    "friar gray": (0x80, 0x7E, 0x79),
    "fringy flower": (0xB1, 0xE2, 0xC1),
    "froly": (0xF5, 0x75, 0x84),
    "frost": (0xED, 0xF5, 0xDD),
    "frosted mint": (0xDB, 0xFF, 0xF8),
    "frostee": (0xE4, 0xF6, 0xE7),
    "fruit salad": (0x4F, 0x9D, 0x5D),
    "fuchsia blue": (0x7A, 0x58, 0xC1),
    "fuchsia pink": (0xC1, 0x54, 0xC1),
    "fuchsia": (0xFF, 0x00, 0xFF),
    "fuego": (0xBE, 0xDE, 0x0D),
    "fuel yellow": (0xEC, 0xA9, 0x27),
    "fun blue": (0x19, 0x59, 0xA8),
    "fun green": (0x01, 0x6D, 0x39),
    "fuscous gray": (0x54, 0x53, 0x4D),
    "fuzzy wuzzy brown": (0xC4, 0x56, 0x55),
    "gable green": (0x16, 0x35, 0x31),
    "gainsboro": (0xDC, 0xDC, 0xDC),
    "gallery": (0xEF, 0xEF, 0xEF),
    "galliano": (0xDC, 0xB2, 0x0C),
    "gamboge": (0xE4, 0x9B, 0x0F),
    "geebung": (0xD1, 0x8F, 0x1B),
    "genoa": (0x15, 0x73, 0x6B),
    "geraldine": (0xFB, 0x89, 0x89),
    "geyser": (0xD4, 0xDF, 0xE2),
    "ghost white": (0xF8, 0xF8, 0xFF),
    "ghost": (0xC7, 0xC9, 0xD5),
    "gigas": (0x52, 0x3C, 0x94),
    "gimblet": (0xB8, 0xB5, 0x6A),
    "gin fizz": (0xFF, 0xF9, 0xE2),
    "gin": (0xE8, 0xF2, 0xEB),
    "givry": (0xF8, 0xE4, 0xBF),
    "glacier": (0x80, 0xB3, 0xC4),
    "glade green": (0x61, 0x84, 0x5F),
    "go ben": (0x72, 0x6D, 0x4E),
    "goblin": (0x3D, 0x7D, 0x52),
    "gold drop": (0xF1, 0x82, 0x00),
    "gold sand": (0xE6, 0xBE, 0x8A),
    "gold tips": (0xDE, 0xBA, 0x13),
    "gold": (0xFF, 0xD7, 0x00),
    "golden bell": (0xE2, 0x89, 0x13),
    "golden dream": (0xF0, 0xD5, 0x2D),
    "golden fizz": (0xF5, 0xFB, 0x3D),
    "golden glow": (0xFD, 0xE2, 0x95),
    "golden grass": (0xDA, 0xA5, 0x20),
    "golden sand": (0xF0, 0xDB, 0x7D),
    "golden tainoi": (0xFF, 0xCC, 0x5C),
    "goldenrod": (0xFC, 0xD6, 0x67),
    "gondola": (0x26, 0x14, 0x14),
    "gordons green": (0x0B, 0x11, 0x07),
    "gorse": (0xFF, 0xF1, 0x4F),
    "gossamer": (0x06, 0x9B, 0x81),
    "gossip": (0xD2, 0xF8, 0xB0),
    "gothic": (0x6D, 0x92, 0xA1),
    "governor bay": (0x2F, 0x3C, 0xB3),
    "grain brown": (0xE4, 0xD5, 0xB7),
    "grandis": (0xFF, 0xD3, 0x8C),
    "granite green": (0x8D, 0x89, 0x74),
    "granny apple": (0xD5, 0xF6, 0xE3),
    "granny smith apple": (0x9D, 0xE0, 0x93),
    "granny smith": (0x84, 0xA0, 0xA0),
    "grape": (0x38, 0x1A, 0x51),
    "graphite": (0x25, 0x16, 0x07),
    "gravel": (0x4A, 0x44, 0x4B),
    "gray asparagus": (0x46, 0x59, 0x45),
    "gray chateau": (0xA2, 0xAA, 0xB3),
    "gray nickel": (0xC3, 0xC3, 0xBD),
    "gray nurse": (0xE7, 0xEC, 0xE6),
    "gray olive": (0xA9, 0xA4, 0x91),
    "gray suit": (0xC1, 0xBE, 0xCD),
    "gray": (0x80, 0x80, 0x80),
    "green haze": (0x01, 0xA3, 0x68),
    "green house": (0x24, 0x50, 0x0F),
    "green kelp": (0x25, 0x31, 0x1C),
    "green leaf": (0x43, 0x6A, 0x0D),
    "green mist": (0xCB, 0xD3, 0xB0),
    "green pea": (0x1D, 0x61, 0x42),
    "green smoke": (0xA4, 0xAF, 0x6E),
    "green spring": (0xB8, 0xC1, 0xB1),
    "green vogue": (0x03, 0x2B, 0x52),
    "green waterloo": (0x10, 0x14, 0x05),
    "green white": (0xE8, 0xEB, 0xE0),
    "green yellow": (0xAD, 0xFF, 0x2F),
    "green": (0x00, 0xFF, 0x00),
    "grenadier": (0xD5, 0x46, 0x00),
    "guardsman red": (0xBA, 0x01, 0x01),
    "gulf blue": (0x05, 0x16, 0x57),
    "gulf stream": (0x80, 0xB3, 0xAE),
    "gull gray": (0x9D, 0xAC, 0xB7),
    "gum leaf": (0xB6, 0xD3, 0xBF),
    "gumbo": (0x7C, 0xA1, 0xA6),
    "gun powder": (0x41, 0x42, 0x57),
    "gunsmoke": (0x82, 0x86, 0x85),
    "gurkha": (0x9A, 0x95, 0x77),
    "hacienda": (0x98, 0x81, 0x1B),
    "hairy heath": (0x6B, 0x2A, 0x14),
    "haiti": (0x1B, 0x10, 0x35),
    "half and half": (0xFF, 0xFE, 0xE1),
    "half baked": (0x85, 0xC4, 0xCC),
    "half colonial white": (0xFD, 0xF6, 0xD3),
    "half dutch white": (0xFE, 0xF7, 0xDE),
    "half spanish white": (0xFE, 0xF4, 0xDB),
    "hampton": (0xE5, 0xD8, 0xAF),
    "harlequin": (0x3F, 0xFF, 0x00),
    "harp": (0xE6, 0xF2, 0xEA),
    "harvest gold": (0xE0, 0xB9, 0x74),
    "havelock blue": (0x55, 0x90, 0xD9),
    "hawaiian tan": (0x9D, 0x56, 0x16),
    "hawkes blue": (0xD4, 0xE2, 0xFC),
    "heath": (0x54, 0x10, 0x12),
    "heather": (0xB7, 0xC3, 0xD0),
    "heathered gray": (0xB6, 0xB0, 0x95),
    "heavy metal": (0x2B, 0x32, 0x28),
    "heliotrope": (0xDF, 0x73, 0xFF),
    "hemlock": (0x5E, 0x5D, 0x3B),
    "hemp": (0x90, 0x78, 0x74),
    "hibiscus": (0xB6, 0x31, 0x6C),
    "highland": (0x6F, 0x8E, 0x63),
    "hillary": (0xAC, 0xA5, 0x86),
    "himalaya": (0x6A, 0x5D, 0x1B),
    "hint of green": (0xE6, 0xFF, 0xE9),
    "hint of red": (0xFB, 0xF9, 0xF9),
    "hint of yellow": (0xFA, 0xFD, 0xE4),
    "hippie blue": (0x58, 0x9A, 0xAF),
    "hippie green": (0x53, 0x82, 0x4B),
    "hippie pink": (0xAE, 0x45, 0x60),
    "hit gray": (0xA1, 0xAD, 0xB5),
    "hit pink": (0xFF, 0xAB, 0x81),
    "hokey pokey": (0xC8, 0xA5, 0x28),
    "hoki": (0x65, 0x86, 0x9F),
    "holly": (0x01, 0x1D, 0x13),
    "hollywood cerise": (0xF4, 0x00, 0xA1),
    "honey flower": (0x4F, 0x1C, 0x70),
    "honeydew": (0xF0, 0xFF, 0xF0),
    "honeysuckle": (0xED, 0xFC, 0x84),
    "hopbush": (0xD0, 0x6D, 0xA1),
    "horizon": (0x5A, 0x87, 0xA0),
    "horses neck": (0x60, 0x49, 0x13),
    "hot cinnamon": (0xD2, 0x69, 0x1E),
    "hot pink": (0xFF, 0x69, 0xB4),
    "hot toddy": (0xB3, 0x80, 0x07),
    "humming bird": (0xCF, 0xF9, 0xF3),
    "hunter green": (0x16, 0x1D, 0x10),
    "hurricane": (0x87, 0x7C, 0x7B),
    "husk": (0xB7, 0xA4, 0x58),
    "ice cold": (0xB1, 0xF4, 0xE7),
    "iceberg": (0xDA, 0xF4, 0xF0),
    "illusion": (0xF6, 0xA4, 0xC9),
    "inch worm": (0xB0, 0xE3, 0x13),
    "indian khaki": (0xC3, 0xB0, 0x91),
    "indian red": (0xCD, 0x5C, 0x5C),
    "indian tan": (0x4D, 0x1E, 0x01),
    "indigo": (0x4F, 0x69, 0xC6),
    "indochine": (0xC2, 0x6B, 0x03),
    "international orange": (0xFF, 0x4F, 0x00),
    "irish coffee": (0x5F, 0x3D, 0x26),
    "iroko": (0x43, 0x31, 0x20),
    "iron": (0xD4, 0xD7, 0xD9),
    "ironside gray": (0x67, 0x66, 0x62),
    "ironstone": (0x86, 0x48, 0x3C),
    "island spice": (0xFF, 0xFC, 0xEE),
    "ivory": (0xFF, 0xFF, 0xF0),
    "jacaranda": (0x2E, 0x03, 0x29),
    "jacarta": (0x3A, 0x2A, 0x6A),
    "jacko bean": (0x2E, 0x19, 0x05),
    "jacksons purple": (0x20, 0x20, 0x8D),
    "jade": (0x00, 0xA8, 0x6B),
    "jaffa": (0xEF, 0x86, 0x3F),
    "jagged ice": (0xC2, 0xE8, 0xE5),
    "jagger": (0x35, 0x0E, 0x57),
    "jaguar": (0x08, 0x01, 0x10),
    "jambalaya": (0x5B, 0x30, 0x13),
    "janna": (0xF4, 0xEB, 0xD3),
    "japanese laurel": (0x0A, 0x69, 0x06),
    "japanese maple": (0x78, 0x01, 0x09),
    "japonica": (0xD8, 0x7C, 0x63),
    "java": (0x1F, 0xC2, 0xC2),
    "jazzberry jam": (0xA5, 0x0B, 0x5E),
    "jelly bean": (0x29, 0x7B, 0x9A),
    "jet stream": (0xB5, 0xD2, 0xCE),
    "jewel": (0x12, 0x6B, 0x40),
    "jon": (0x3B, 0x1F, 0x1F),
    "jonquil": (0xEE, 0xFF, 0x9A),
    "jordy blue": (0x8A, 0xB9, 0xF1),
    "judge gray": (0x54, 0x43, 0x33),
    "jumbo": (0x7C, 0x7B, 0x82),
    "jungle green": (0x29, 0xAB, 0x87),
    "jungle mist": (0xB4, 0xCF, 0xD3),
    "juniper": (0x6D, 0x92, 0x92),
    "just right": (0xEC, 0xCD, 0xB9),
    "kabul": (0x5E, 0x48, 0x3E),
    "kaitoke green": (0x00, 0x46, 0x20),
    "kangaroo": (0xC6, 0xC8, 0xBD),
    "karaka": (0x1E, 0x16, 0x09),
    "karry": (0xFF, 0xEA, 0xD4),
    "kashmir blue": (0x50, 0x70, 0x96),
    "kelp": (0x45, 0x49, 0x36),
    "kenyan copper": (0x7C, 0x1C, 0x05),
    "keppel": (0x3A, 0xB0, 0x9E),
    "key lime pie": (0xBF, 0xC9, 0x21),
    "khaki": (0xF0, 0xE6, 0x8C),
    "kidnapper": (0xE1, 0xEA, 0xD4),
    "kilamanjaro": (0x24, 0x0C, 0x02),
    "killarney": (0x3A, 0x6A, 0x47),
    "kimberly": (0x73, 0x6C, 0x9F),
    "kingfisher daisy": (0x3E, 0x04, 0x80),
    "kiosk.house": (90, 95, 0),
    "klein blue": (0x00, 0x2F, 0xA7),
    "kobi": (0xE7, 0x9F, 0xC4),
    "kokoda": (0x6E, 0x6D, 0x57),
    "korma": (0x8F, 0x4B, 0x0E),
    "koromiko": (0xFF, 0xBD, 0x5F),
    "kournikova": (0xFF, 0xE7, 0x72),
    "kumera": (0x88, 0x62, 0x21),
    "la palma": (0x36, 0x87, 0x16),
    "la rioja": (0xB3, 0xC1, 0x10),
    "las palmas": (0xC6, 0xE6, 0x10),
    "laser lemon": (0xFF, 0xFF, 0x66),
    "laser": (0xC8, 0xB5, 0x68),
    "laurel": (0x74, 0x93, 0x78),
    "lavender blush": (0xFF, 0xF0, 0xF5),
    "lavender gray": (0xBD, 0xBB, 0xD7),
    "lavender magenta": (0xEE, 0x82, 0xEE),
    "lavender pink": (0xFB, 0xAE, 0xD2),
    "lavender purple": (0x96, 0x7B, 0xB6),
    "lavender rose": (0xFB, 0xA0, 0xE3),
    "lavender": (0xB5, 0x7E, 0xDC),
    "lawn green": (0x7C, 0xFC, 0x00),
    "leather": (0x96, 0x70, 0x59),
    "lemon chiffon": (0xFF, 0xFA, 0xCD),
    "lemon ginger": (0xAC, 0x9E, 0x22),
    "lemon grass": (0x9B, 0x9E, 0x8F),
    "lemon": (0xFD, 0xE9, 0x10),
    "light apricot": (0xFD, 0xD5, 0xB1),
    "light blue": (0xAD, 0xD8, 0xE6),
    "light coral": (0xF0, 0x80, 0x80),
    "light cyan": (0xE0, 0xFF, 0xFF),
    "light goldenrod": (0xFA, 0xFA, 0xD2),
    "light gray": (0x26, 0x23, 0x35),
    "light green": (0x90, 0xEE, 0x90),
    "light orchid": (0xE2, 0x9C, 0xD2),
    "light pink": (0xDD, 0xB6, 0xC1),
    "light salmon": (0xDD, 0xA0, 0x7A),
    "light sea green": (0x20, 0xB2, 0xAA),
    "light slate gray": (0x77, 0x88, 0x99),
    "light steel blue": (0xB0, 0xC4, 0xDE),
    "light wisteria": (0xC9, 0xA0, 0xDC),
    "light yellow": (0xFF, 0xFF, 0xE0),
    "lightning yellow": (0xFC, 0xC0, 0x1E),
    "lilac bush": (0x98, 0x74, 0xD3),
    "lilac": (0xC8, 0xA2, 0xC8),
    "lily white": (0xE7, 0xF8, 0xFF),
    "lily": (0xC8, 0xAA, 0xBF),
    "lima": (0x76, 0xBD, 0x17),
    "lime": (0xBF, 0xFF, 0x00),
    "limeade": (0x6F, 0x9D, 0x02),
    "limed ash": (0x74, 0x7D, 0x63),
    "limed oak": (0xAC, 0x8A, 0x56),
    "limed spruce": (0x39, 0x48, 0x51),
    "linen": (0xFA, 0xF0, 0xE6),
    "link water": (0xD9, 0xE4, 0xF5),
    "lipstick": (0xAB, 0x05, 0x63),
    "lisbon brown": (0x42, 0x39, 0x21),
    "livid brown": (0x4D, 0x28, 0x2E),
    "loafer": (0xEE, 0xF4, 0xDE),
    "loblolly": (0xBD, 0xC9, 0xCE),
    "lochinvar": (0x2C, 0x8C, 0x84),
    "lochmara": (0x00, 0x7E, 0xC7),
    "locust": (0xA8, 0xAF, 0x8E),
    "log cabin": (0x24, 0x2A, 0x1D),
    "logan": (0xAA, 0xA9, 0xCD),
    "lola": (0xDF, 0xCF, 0xDB),
    "london hue": (0xBE, 0xA6, 0xC3),
    "lonestar": (0x6D, 0x01, 0x01),
    "lotus": (0x86, 0x3C, 0x3C),
    "loulou": (0x46, 0x0B, 0x41),
    "lucky point": (0x1A, 0x1A, 0x68),
    "lucky": (0xAF, 0x9F, 0x1C),
    "lunar green": (0x3C, 0x49, 0x3A),
    "luxor gold": (0xA7, 0x88, 0x2C),
    "lynch": (0x69, 0x7E, 0x9A),
    "mabel": (0xD9, 0xF7, 0xFF),
    "macaroni and cheese": (0xFF, 0xB9, 0x7B),
    "madang": (0xB7, 0xF0, 0xBE),
    "madison": (0x09, 0x25, 0x5D),
    "madras": (0x3F, 0x30, 0x02),
    "magenta": (0xFF, 0x00, 0xFF),
    "magic mint": (0xAA, 0xF0, 0xD1),
    "magnolia": (0xF8, 0xF4, 0xFF),
    "mahogany": (0x4E, 0x06, 0x06),
    "mai tai": (0xB0, 0x66, 0x08),
    "maize": (0xF5, 0xD5, 0xA0),
    "makara": (0x89, 0x7D, 0x6D),
    "mako": (0x44, 0x49, 0x54),
    "malachite": (0x0B, 0xDA, 0x51),
    "malibu": (0x7D, 0xC8, 0xF7),
    "mallard": (0x23, 0x34, 0x18),
    "malta": (0xBD, 0xB2, 0xA1),
    "mamba": (0x8E, 0x81, 0x90),
    "manatee": (0x8D, 0x90, 0xA1),
    "mandalay": (0xAD, 0x78, 0x1B),
    "mandy": (0xE2, 0x54, 0x65),
    "mandys pink": (0xF2, 0xC3, 0xB2),
    "mango tango": (0xE7, 0x72, 0x00),
    "manhattan": (0xF5, 0xC9, 0x99),
    "mantis": (0x74, 0xC3, 0x65),
    "mantle": (0x8B, 0x9C, 0x90),
    "manz": (0xEE, 0xEF, 0x78),
    "mardi gras": (0x35, 0x00, 0x36),
    "marigold yellow": (0xFB, 0xE8, 0x70),
    "marigold": (0xB9, 0x8D, 0x28),
    "mariner": (0x28, 0x6A, 0xCD),
    "maroon flush": (0xC3, 0x21, 0x48),
    "maroon oak": (0x52, 0x0C, 0x17),
    "maroon": (0x80, 0x00, 0x00),
    "marshland": (0x0B, 0x0F, 0x08),
    "martini": (0xAF, 0xA0, 0x9E),
    "martinique": (0x36, 0x30, 0x50),
    "marzipan": (0xF8, 0xDB, 0x9D),
    "masala": (0x40, 0x3B, 0x38),
    "matisse": (0x1B, 0x65, 0x9D),
    "matrix": (0xB0, 0x5D, 0x54),
    "matterhorn": (0x4E, 0x3B, 0x41),
    "mauve": (0xE0, 0xB0, 0xFF),
    "mauvelous": (0xF0, 0x91, 0xA9),
    "maverick": (0xD8, 0xC2, 0xD5),
    "medium aquamarine": (0x66, 0xCD, 0xAA),
    "medium blue": (0x00, 0x00, 0xCD),
    "medium carmine": (0xAF, 0x40, 0x35),
    "medium orchid": (0xBA, 0x55, 0xD3),
    "medium purple": (0x93, 0x70, 0xDB),
    "medium red violet": (0xBB, 0x33, 0x85),
    "medium sea green": (0x3C, 0xB3, 0x71),
    "medium slate blue": (0x7B, 0x68, 0xEE),
    "medium spring green": (0x00, 0xFA, 0x9A),
    "medium turquoise": (0x48, 0xD1, 0xCC),
    "medium violet red": (0xC7, 0x15, 0x85),
    "meerkat.cabin": (95, 0x00, 95),
    "melanie": (0xE4, 0xC2, 0xD5),
    "melanzane": (0x30, 0x05, 0x29),
    "melon": (0xFE, 0xBA, 0xAD),
    "melrose": (0xC7, 0xC1, 0xFF),
    "mercury": (0xE5, 0xE5, 0xE5),
    "merino": (0xF6, 0xF0, 0xE6),
    "merlin": (0x41, 0x3C, 0x37),
    "merlot": (0x83, 0x19, 0x23),
    "metallic bronze": (0x49, 0x37, 0x1B),
    "metallic copper": (0x71, 0x29, 0x1D),
    "meteor": (0xD0, 0x7D, 0x12),
    "meteorite": (0x3C, 0x1F, 0x76),
    "mexican red": (0xA7, 0x25, 0x25),
    "mid gray": (0x5F, 0x5F, 0x6E),
    "midnight blue": (0x00, 0x33, 0x66),
    "midnight moss": (0x04, 0x10, 0x04),
    "midnight": (0x01, 0x16, 0x35),
    "mikado": (0x2D, 0x25, 0x10),
    "milan": (0xFA, 0xFF, 0xA4),
    "milano red": (0xB8, 0x11, 0x04),
    "milk punch": (0xFF, 0xF6, 0xD4),
    "millbrook": (0x59, 0x44, 0x33),
    "mimosa": (0xF8, 0xFD, 0xD3),
    "mindaro": (0xE3, 0xF9, 0x88),
    "mine shaft": (0x32, 0x32, 0x32),
    "mineral green": (0x3F, 0x5D, 0x53),
    "ming": (0x36, 0x74, 0x7D),
    "minsk": (0x3F, 0x30, 0x7F),
    "mint cream": (0xF5, 0xFF, 0xF1),
    "mint green": (0x98, 0xFF, 0x98),
    "mint julep": (0xF1, 0xEE, 0xC1),
    "mint tulip": (0xC4, 0xF4, 0xEB),
    "mirage": (0x16, 0x19, 0x28),
    "mischka": (0xD1, 0xD2, 0xDD),
    "mist gray": (0xC4, 0xC4, 0xBC),
    "misty rose": (0xFF, 0xE4, 0xE1),
    "mobster": (0x7F, 0x75, 0x89),
    "moccaccino": (0x6E, 0x1D, 0x14),
    "moccasin": (0xFF, 0xE4, 0xB5),
    "mocha": (0x78, 0x2D, 0x19),
    "mojo": (0xC0, 0x47, 0x37),
    "mona lisa": (0xFF, 0xA1, 0x94),
    "monarch": (0x8B, 0x07, 0x23),
    "mondo": (0x4A, 0x3C, 0x30),
    "mongoose": (0xB5, 0xA2, 0x7F),
    "monsoon": (0x8A, 0x83, 0x89),
    "monte carlo": (0x83, 0xD0, 0xC6),
    "monza": (0xC7, 0x03, 0x1E),
    "moody blue": (0x7F, 0x76, 0xD3),
    "moon glow": (0xFC, 0xFE, 0xDA),
    "moon mist": (0xDC, 0xDD, 0xCC),
    "moon raker": (0xD6, 0xCE, 0xF6),
    "morning glory": (0x9E, 0xDE, 0xE0),
    "morocco brown": (0x44, 0x1D, 0x00),
    "mortar": (0x50, 0x43, 0x51),
    "mosque": (0x03, 0x6A, 0x6E),
    "moss green": (0xAD, 0xDF, 0xAD),
    "mountain meadow": (0x1A, 0xB3, 0x85),
    "mountain mist": (0x95, 0x93, 0x96),
    "mountbatten pink": (0x99, 0x7A, 0x8D),
    "muddy waters": (0xB7, 0x8E, 0x5C),
    "muesli": (0xAA, 0x8B, 0x5B),
    "mulberry wood": (0x5C, 0x05, 0x36),
    "mulberry": (0xC5, 0x4B, 0x8C),
    "mule fawn": (0x8C, 0x47, 0x2F),
    "mulled wine": (0x4E, 0x45, 0x62),
    "mustard": (0xFF, 0xDB, 0x58),
    "my pink": (0xD6, 0x91, 0x88),
    "my sin": (0xFF, 0xB3, 0x1F),
    "mystic": (0xE2, 0xEB, 0xED),
    "nandor": (0x4B, 0x5D, 0x52),
    "napa": (0xAC, 0xA4, 0x94),
    "narvik": (0xED, 0xF9, 0xF1),
    "natural gray": (0x8B, 0x86, 0x80),
    "navajo white": (0xFF, 0xDE, 0xAD),
    "navy blue": (0x00, 0x00, 0x80),
    "navy": (0x00, 0x00, 0x80),
    "nebula": (0xCB, 0xDB, 0xD6),
    "negroni": (0xFF, 0xE2, 0xC5),
    "neon carrot": (0xFF, 0x99, 0x33),
    "nepal": (0x8E, 0xAB, 0xC1),
    "neptune": (0x7C, 0xB7, 0xBB),
    "nero": (0x14, 0x06, 0x00),
    "nevada": (0x64, 0x6E, 0x75),
    "new orleans": (0xF3, 0xD6, 0x9D),
    "new york pink": (0xD7, 0x83, 0x7F),
    "niagara": (0x06, 0xA1, 0x89),
    "night rider": (0x1F, 0x12, 0x0F),
    "night shadz": (0xAA, 0x37, 0x5A),
    "nile blue": (0x19, 0x37, 0x51),
    "nobel": (0xB7, 0xB1, 0xB1),
    "nomad": (0xBA, 0xB1, 0xA2),
    "norway": (0xA8, 0xBD, 0x9F),
    "nugget": (0xC5, 0x99, 0x22),
    "nutmeg wood finish": (0x68, 0x36, 0x00),
    "nutmeg": (0x81, 0x42, 0x2C),
    "oasis": (0xFE, 0xEF, 0xCE),
    "observatory": (0x02, 0x86, 0x6F),
    "ocean green": (0x41, 0xAA, 0x78),
    "ochre": (0xCC, 0x77, 0x22),
    "off green": (0xE6, 0xF8, 0xF3),
    "off yellow": (0xFE, 0xF9, 0xE3),
    "oil": (0x28, 0x1E, 0x15),
    "old brick": (0x90, 0x1E, 0x1E),
    "old copper": (0x72, 0x4A, 0x2F),
    "old gold": (0xCF, 0xB5, 0x3B),
    "old lace": (0xFD, 0xF5, 0xE6),
    "old lavender": (0x79, 0x68, 0x78),
    "old rose": (0xC0, 0x80, 0x81),
    "olive drab": (0x6B, 0x8E, 0x23),
    "olive green": (0xB5, 0xB3, 0x5C),
    "olive haze": (0x8B, 0x84, 0x70),
    "olive": (0x80, 0x80, 0x00),
    "olivetone": (0x71, 0x6E, 0x10),
    "olivine": (0x9A, 0xB9, 0x73),
    "onahau": (0xCD, 0xF4, 0xFF),
    "onion": (0x2F, 0x27, 0x0E),
    "opal": (0xA9, 0xC6, 0xC2),
    "opium": (0x8E, 0x6F, 0x70),
    "oracle": (0x37, 0x74, 0x75),
    "orange peel": (0xFF, 0xA0, 0x00),
    "orange red": (0xFF, 0x45, 0x00),
    "orange roughy": (0xC4, 0x57, 0x19),
    "orange white": (0xFE, 0xFC, 0xED),
    "orange": (0xFF, 0x68, 0x1F),
    "orchid white": (0xFF, 0xFD, 0xF3),
    "orchid": (0xDA, 0x70, 0xD6),
    "oregon": (0x9B, 0x47, 0x03),
    "orient": (0x01, 0x5E, 0x85),
    "oriental pink": (0xC6, 0x91, 0x91),
    "orinoco": (0xF3, 0xFB, 0xD4),
    "oslo gray": (0x87, 0x8D, 0x91),
    "ottoman": (0xE9, 0xF8, 0xED),
    "outer space": (0x2D, 0x38, 0x3A),
    "outrageous orange": (0xFF, 0x60, 0x37),
    "oxford blue": (0x38, 0x45, 0x55),
    "oxley": (0x77, 0x9E, 0x86),
    "oyster bay": (0xDA, 0xFA, 0xFF),
    "oyster pink": (0xE9, 0xCE, 0xCD),
    "paarl": (0xA6, 0x55, 0x29),
    "pablo": (0x77, 0x6F, 0x61),
    "pacific blue": (0x00, 0x9D, 0xC4),
    "pacifika": (0x77, 0x81, 0x20),
    "paco": (0x41, 0x1F, 0x10),
    "padua": (0xAD, 0xE6, 0xC4),
    "pale canary": (0xFF, 0xFF, 0x99),
    "pale goldenrod": (0xEE, 0xE8, 0xAA),
    "pale green": (0x98, 0xFB, 0x98),
    "pale leaf": (0xC0, 0xD3, 0xB9),
    "pale oyster": (0x98, 0x8D, 0x77),
    "pale prim": (0xFD, 0xFE, 0xB8),
    "pale rose": (0xFF, 0xE1, 0xF2),
    "pale sky": (0x6E, 0x77, 0x83),
    "pale slate": (0xC3, 0xBF, 0xC1),
    "pale turquoise": (0xAF, 0xEE, 0xEE),
    "pale violet red": (0xDB, 0x70, 0x93),
    "palm green": (0x09, 0x23, 0x0F),
    "palm leaf": (0x19, 0x33, 0x0E),
    "pampas": (0xF4, 0xF2, 0xEE),
    "panache": (0xEA, 0xF6, 0xEE),
    "pancho": (0xED, 0xCD, 0xAB),
    "papaya whip": (0xFF, 0xEF, 0xD5),
    "paprika": (0x8D, 0x02, 0x26),
    "paradiso": (0x31, 0x7D, 0x82),
    "parchment": (0xF1, 0xE9, 0xD2),
    "paris daisy": (0xFF, 0xF4, 0x6E),
    "paris m": (0x26, 0x05, 0x6A),
    "paris white": (0xCA, 0xDC, 0xD4),
    "parsley": (0x13, 0x4F, 0x19),
    "pastel green": (0x77, 0xDD, 0x77),
    "pastel pink": (0xFF, 0xD1, 0xDC),
    "patina": (0x63, 0x9A, 0x8F),
    "pattens blue": (0xDE, 0xF5, 0xFF),
    "paua": (0x26, 0x03, 0x68),
    "pavlova": (0xD7, 0xC4, 0x98),
    "peach cream": (0xFF, 0xF0, 0xDB),
    "peach orange": (0xFF, 0xCC, 0x99),
    "peach puff": (0xFF, 0xDA, 0xB9),
    "peach schnapps": (0xFF, 0xDC, 0xD6),
    "peach yellow": (0xFA, 0xDF, 0xAD),
    "peach": (0xFF, 0xE5, 0xB4),
    "peanut": (0x78, 0x2F, 0x16),
    "pear": (0xD1, 0xE2, 0x31),
    "pearl bush": (0xE8, 0xE0, 0xD5),
    "pearl lusta": (0xFC, 0xF4, 0xDC),
    "peat": (0x71, 0x6B, 0x56),
    "pelorous": (0x3E, 0xAB, 0xBF),
    "peppermint": (0xE3, 0xF5, 0xE1),
    "perano": (0xA9, 0xBE, 0xF2),
    "perfume": (0xD0, 0xBE, 0xF8),
    "periglacial blue": (0xE1, 0xE6, 0xD6),
    "periwinkle gray": (0xC3, 0xCD, 0xE6),
    "periwinkle": (0xCC, 0xCC, 0xFF),
    "persian blue": (0x1C, 0x39, 0xBB),
    "persian green": (0x00, 0xA6, 0x93),
    "persian indigo": (0x32, 0x12, 0x7A),
    "persian pink": (0xF7, 0x7F, 0xBE),
    "persian plum": (0x70, 0x1C, 0x1C),
    "persian red": (0xCC, 0x33, 0x33),
    "persian rose": (0xFE, 0x28, 0xA2),
    "persimmon": (0xFF, 0x6B, 0x53),
    "peru tan": (0x7F, 0x3A, 0x02),
    "peru": (0xCD, 0x85, 0x3F),
    "pesto": (0x7C, 0x76, 0x31),
    "petite orchid": (0xDB, 0x96, 0x90),
    "pewter": (0x96, 0xA8, 0xA1),
    "pharlap": (0xA3, 0x80, 0x7B),
    "picasso": (0xFF, 0xF3, 0x9D),
    "pickled bean": (0x6E, 0x48, 0x26),
    "pickled bluewood": (0x31, 0x44, 0x59),
    "picton blue": (0x45, 0xB1, 0xE8),
    "pig pink": (0xFD, 0xD7, 0xE4),
    "pigeon post": (0xAF, 0xBD, 0xD9),
    "pigment indigo": (0x4B, 0x00, 0x82),
    "pine cone": (0x6D, 0x5E, 0x54),
    "pine glade": (0xC7, 0xCD, 0x90),
    "pine green": (0x01, 0x79, 0x6F),
    "pine tree": (0x17, 0x1F, 0x04),
    "pink flamingo": (0xFF, 0x66, 0xFF),
    "pink flare": (0xE1, 0xC0, 0xC8),
    "pink lace": (0xFF, 0xDD, 0xF4),
    "pink lady": (0xFF, 0xF1, 0xD8),
    "pink salmon": (0xFF, 0x91, 0xA4),
    "pink swan": (0xBE, 0xB5, 0xB7),
    "pink": (0xFF, 0xC0, 0xCB),
    "piper": (0xC9, 0x63, 0x23),
    "pipi": (0xFE, 0xF4, 0xCC),
    "pippin": (0xFF, 0xE1, 0xDF),
    "pirate gold": (0xBA, 0x7F, 0x03),
    "pistachio": (0x9D, 0xC2, 0x09),
    "pixie green": (0xC0, 0xD8, 0xB6),
    "pizazz": (0xFF, 0x90, 0x00),
    "pizza": (0xC9, 0x94, 0x15),
    "plantation": (0x27, 0x50, 0x4B),
    "plum": (0x84, 0x31, 0x79),
    "pohutukawa": (0x8F, 0x02, 0x1C),
    "polar": (0xE5, 0xF9, 0xF6),
    "polo blue": (0x8D, 0xA8, 0xCC),
    "pomegranate": (0xF3, 0x47, 0x23),
    "pompadour": (0x66, 0x00, 0x45),
    "porcelain": (0xEF, 0xF2, 0xF3),
    "porsche": (0xEA, 0xAE, 0x69),
    "port gore": (0x25, 0x1F, 0x4F),
    "portafino": (0xFF, 0xFF, 0xB4),
    "portage": (0x8B, 0x9F, 0xEE),
    "portica": (0xF9, 0xE6, 0x63),
    "pot pourri": (0xF5, 0xE7, 0xE2),
    "potters clay": (0x8C, 0x57, 0x38),
    "powder ash": (0xBC, 0xC9, 0xC2),
    "powder blue": (0xB0, 0xE0, 0xE6),
    "prairie sand": (0x9A, 0x38, 0x20),
    "prelude": (0xD0, 0xC0, 0xE5),
    "prim": (0xF0, 0xE2, 0xEC),
    "primrose": (0xED, 0xEA, 0x99),
    "provincial pink": (0xFE, 0xF5, 0xF1),
    "prussian blue": (0x00, 0x31, 0x53),
    "puce": (0xCC, 0x88, 0x99),
    "pueblo": (0x7D, 0x2C, 0x14),
    "puerto rico": (0x3F, 0xC1, 0xAA),
    "pumice": (0xC2, 0xCA, 0xC4),
    "pumpkin skin": (0xB1, 0x61, 0x0B),
    "pumpkin": (0xFF, 0x75, 0x18),
    "punch": (0xDC, 0x43, 0x33),
    "punga": (0x4D, 0x3D, 0x14),
    "purple heart": (0x65, 0x2D, 0xC1),
    "purple mountain's majesty": (0x96, 0x78, 0xB6),
    "purple pizzazz": (0xFF, 0x00, 0xCC),
    "purple": (0x66, 0x00, 0x99),
    "putty": (0xE7, 0xCD, 0x8C),
    "quarter pearl lusta": (0xFF, 0xFD, 0xF4),
    "quarter spanish white": (0xF7, 0xF2, 0xE1),
    "quicksand": (0xBD, 0x97, 0x8E),
    "quill gray": (0xD6, 0xD6, 0xD1),
    "quincy": (0x62, 0x3F, 0x2D),
    "racing green": (0x0C, 0x19, 0x11),
    "radical red": (0xFF, 0x35, 0x5E),
    "raffia": (0xEA, 0xDA, 0xB8),
    "rainee": (0xB9, 0xC8, 0xAC),
    "rajah": (0xF7, 0xB6, 0x68),
    "rangitoto": (0x2E, 0x32, 0x22),
    "rangoon green": (0x1C, 0x1E, 0x13),
    "raven": (0x72, 0x7B, 0x89),
    "raw sienna": (0xD2, 0x7D, 0x46),
    "raw umber": (0x73, 0x4A, 0x12),
    "razzle dazzle rose": (0xFF, 0x33, 0xCC),
    "razzmatazz": (0xE3, 0x0B, 0x5C),
    "rebecca purple": (0x66, 0x33, 0x99),
    "rebel": (0x3C, 0x12, 0x06),
    "red beech": (0x7B, 0x38, 0x01),
    "red berry": (0x8E, 0x00, 0x00),
    "red damask": (0xDA, 0x6A, 0x41),
    "red devil": (0x86, 0x01, 0x11),
    "red orange": (0xFF, 0x3F, 0x34),
    "red oxide": (0x6E, 0x09, 0x02),
    "red ribbon": (0xED, 0x0A, 0x3F),
    "red robin": (0x80, 0x34, 0x1F),
    "red stage": (0xD0, 0x5F, 0x04),
    "red violet": (0xC7, 0x15, 0x85),
    "red": (0xFF, 0x00, 0x00),
    "redwood": (0x5D, 0x1E, 0x0F),
    "reef gold": (0x9F, 0x82, 0x1C),
    "reef": (0xC9, 0xFF, 0xA2),
    "regal blue": (0x01, 0x3F, 0x6A),
    "regent gray": (0x86, 0x94, 0x9F),
    "regent st blue": (0xAA, 0xD6, 0xE6),
    "remy": (0xFE, 0xEB, 0xF3),
    "reno sand": (0xA8, 0x65, 0x15),
    "resolution blue": (0x00, 0x23, 0x87),
    "revolver": (0x2C, 0x16, 0x32),
    "rhino": (0x2E, 0x3F, 0x62),
    "rice cake": (0xFF, 0xFE, 0xF0),
    "rice flower": (0xEE, 0xFF, 0xE2),
    "rich gold": (0xA8, 0x53, 0x07),
    "rio grande": (0xBB, 0xD0, 0x09),
    "ripe lemon": (0xF4, 0xD8, 0x1C),
    "ripe plum": (0x41, 0x00, 0x56),
    "riptide": (0x8B, 0xE6, 0xD8),
    "river bed": (0x43, 0x4C, 0x59),
    "rob roy": (0xEA, 0xC6, 0x74),
    "robin's egg blue": (0x00, 0xCC, 0xCC),
    "rock blue": (0x9E, 0xB1, 0xCD),
    "rock spray": (0xBA, 0x45, 0x0C),
    "rock": (0x4D, 0x38, 0x33),
    "rodeo dust": (0xC9, 0xB2, 0x9B),
    "rolling stone": (0x74, 0x7D, 0x83),
    "roman coffee": (0x79, 0x5D, 0x4C),
    "roman": (0xDE, 0x63, 0x60),
    "romance": (0xFF, 0xFE, 0xFD),
    "romantic": (0xFF, 0xD2, 0xB7),
    "ronchi": (0xEC, 0xC5, 0x4E),
    "roof terracotta": (0xA6, 0x2F, 0x20),
    "rope": (0x8E, 0x4D, 0x1E),
    "rose bud cherry": (0x80, 0x0B, 0x47),
    "rose bud": (0xFB, 0xB2, 0xA3),
    "rose fog": (0xE7, 0xBC, 0xB4),
    "rose of sharon": (0xBF, 0x55, 0x00),
    "rose white": (0xFF, 0xF6, 0xF5),
    "rose": (0xFF, 0x00, 0x7F),
    "rosewood": (0x65, 0x00, 0x0B),
    "rosy blue": (0xBC, 0x8F, 0x8F),
    "roti": (0xC6, 0xA8, 0x4B),
    "rouge": (0xA2, 0x3B, 0x6C),
    "royal blue": (0x41, 0x69, 0xE1),
    "royal heath": (0xAB, 0x34, 0x72),
    "royal purple": (0x6B, 0x3F, 0xA0),
    "rpi": (208, 95, 0),
    "rum swizzle": (0xF9, 0xF8, 0xE4),
    "rum": (0x79, 0x69, 0x89),
    "russet": (0x80, 0x46, 0x1B),
    "russett": (0x75, 0x5A, 0x57),
    "rust": (0xB7, 0x41, 0x0E),
    "rustic red": (0x48, 0x04, 0x04),
    "rusty nail": (0x86, 0x56, 0x0A),
    "saddle brown": (0x58, 0x34, 0x01),
    "saddle": (0x4C, 0x30, 0x24),
    "saffron mango": (0xF9, 0xBF, 0x58),
    "saffron": (0xF4, 0xC4, 0x30),
    "sage": (0x9E, 0xA5, 0x87),
    "sahara sand": (0xF1, 0xE7, 0x88),
    "sahara": (0xB7, 0xA2, 0x14),
    "sail": (0xB8, 0xE0, 0xF9),
    "salem": (0x09, 0x7F, 0x4B),
    "salmon": (0xFF, 0x8C, 0x69),
    "salomie": (0xFE, 0xDB, 0x8D),
    "salt box": (0x68, 0x5E, 0x6E),
    "saltpan": (0xF1, 0xF7, 0xF2),
    "sambuca": (0x3A, 0x20, 0x10),
    "san felix": (0x0B, 0x62, 0x07),
    "san juan": (0x30, 0x4B, 0x6A),
    "san marino": (0x45, 0x6C, 0xAC),
    "sand dune": (0x82, 0x6F, 0x65),
    "sandal": (0xAA, 0x8D, 0x6F),
    "sandrift": (0xAB, 0x91, 0x7A),
    "sandstone": (0x79, 0x6D, 0x62),
    "sandwisp": (0xF5, 0xE7, 0xA2),
    "sandy beach": (0xFF, 0xEA, 0xC8),
    "sandy brown": (0xF4, 0xA4, 0x60),
    "sangria": (0x92, 0x00, 0x0A),
    "sanguine brown": (0x8D, 0x3D, 0x38),
    "santa fe": (0xB1, 0x6D, 0x52),
    "santas gray": (0x9F, 0xA0, 0xB1),
    "sapling": (0xDE, 0xD4, 0xA4),
    "sapphire": (0x2F, 0x51, 0x9E),
    "saratoga": (0x55, 0x5B, 0x10),
    "satin linen": (0xE6, 0xE4, 0xD4),
    "sauvignon": (0xFF, 0xF5, 0xF3),
    "sazerac": (0xFF, 0xF4, 0xE0),
    "scampi": (0x67, 0x5F, 0xA6),
    "scandal": (0xCF, 0xFA, 0xF4),
    "scarlet gum": (0x43, 0x15, 0x60),
    "scarlet": (0xFF, 0x24, 0x00),
    "scarlett": (0x95, 0x00, 0x15),
    "scarpa flow": (0x58, 0x55, 0x62),
    "schist": (0xA9, 0xB4, 0x97),
    "school bus yellow": (0xFF, 0xD8, 0x00),
    "schooner": (0x8B, 0x84, 0x7E),
    "science blue": (0x00, 0x66, 0xCC),
    "scooter": (0x2E, 0xBF, 0xD4),
    "scorpion": (0x69, 0x5F, 0x62),
    "scotch mist": (0xFF, 0xFB, 0xDC),
    "screamin' green": (0x66, 0xFF, 0x66),
    "screamin green": (0x66, 0xFF, 0x66),
    "screaming green": (0x66, 0xFF, 0x66),
    "sea buckthorn": (0xFB, 0xA1, 0x29),
    "sea green": (0x2E, 0x8B, 0x57),
    "sea mist": (0xC5, 0xDB, 0xCA),
    "sea nymph": (0x78, 0xA3, 0x9C),
    "sea pink": (0xED, 0x98, 0x9E),
    "seagull": (0x80, 0xCC, 0xEA),
    "seance": (0x73, 0x1E, 0x8F),
    "seashell peach": (0xFF, 0xF5, 0xEE),
    "seashell": (0xF1, 0xF1, 0xF1),
    "seaweed": (0x1B, 0x2F, 0x11),
    "selago": (0xF0, 0xEE, 0xFD),
    "selective yellow": (0xFF, 0xBA, 0x00),
    "sepia black": (0x2B, 0x02, 0x02),
    "sepia skin": (0x9E, 0x5B, 0x40),
    "sepia": (0x70, 0x42, 0x14),
    "serenade": (0xFF, 0xF4, 0xE8),
    "shadow green": (0x9A, 0xC2, 0xB8),
    "shadow": (0x83, 0x70, 0x50),
    "shady lady": (0xAA, 0xA5, 0xA9),
    "shakespeare": (0x4E, 0xAB, 0xD1),
    "shalimar": (0xFB, 0xFF, 0xBA),
    "shamrock": (0x33, 0xCC, 0x99),
    "shark": (0x25, 0x27, 0x2C),
    "sherpa blue": (0x00, 0x49, 0x50),
    "sherwood green": (0x02, 0x40, 0x2C),
    "shilo": (0xE8, 0xB9, 0xB3),
    "shingle fawn": (0x6B, 0x4E, 0x31),
    "ship cove": (0x78, 0x8B, 0xBA),
    "ship gray": (0x3E, 0x3A, 0x44),
    "shiraz": (0xB2, 0x09, 0x31),
    "shocking pink": (0xFC, 0x0F, 0xC0),
    "shocking": (0xE2, 0x92, 0xC0),
    "shuttle gray": (0x5F, 0x66, 0x72),
    "siam": (0x64, 0x6A, 0x54),
    "sidecar": (0xF3, 0xE7, 0xBB),
    "sienna": (0xA0, 0x52, 0x2D),
    "silk": (0xBD, 0xB1, 0xA8),
    "silver chalice": (0xAC, 0xAC, 0xAC),
    "silver rust": (0xC9, 0xC0, 0xBB),
    "silver sand": (0xBF, 0xC1, 0xC2),
    "silver tree": (0x66, 0xB5, 0x8F),
    "silver": (0xC0, 0xC0, 0xC0),
    "sinbad": (0x9F, 0xD7, 0xD3),
    "siren": (0x7A, 0x01, 0x3A),
    "sirocco": (0x71, 0x80, 0x80),
    "sisal": (0xD3, 0xCB, 0xBA),
    "skeptic": (0xCA, 0xE6, 0xDA),
    "sky blue": (0x76, 0xD7, 0xEA),
    "slate blue": (0x6A, 0x5A, 0xCD),
    "slate gray": (0x70, 0x80, 0x90),
    "smalt blue": (0x51, 0x80, 0x8F),
    "smalt": (0x00, 0x33, 0x99),
    "smoky": (0x60, 0x5B, 0x73),
    "snow drift": (0xF7, 0xFA, 0xF7),
    "snow flurry": (0xE4, 0xFF, 0xD1),
    "snow": (0xFF, 0xFA, 0xFA),
    "snowy mint": (0xD6, 0xFF, 0xDB),
    "snuff": (0xE2, 0xD8, 0xED),
    "soapstone": (0xFF, 0xFB, 0xF9),
    "soft amber": (0xD1, 0xC6, 0xB4),
    "soft peach": (0xF5, 0xED, 0xEF),
    "solid pink": (0x89, 0x38, 0x43),
    "solitaire": (0xFE, 0xF8, 0xE2),
    "solitude": (0xEA, 0xF6, 0xFF),
    "sorbus": (0xFD, 0x7C, 0x07),
    "sorrell brown": (0xCE, 0xB9, 0x8F),
    "soya bean": (0x6A, 0x60, 0x51),
    "spanish green": (0x81, 0x98, 0x85),
    "spectra": (0x2F, 0x5A, 0x57),
    "spice": (0x6A, 0x44, 0x2E),
    "spicy mix": (0x88, 0x53, 0x42),
    "spicy mustard": (0x74, 0x64, 0x0D),
    "spicy pink": (0x81, 0x6E, 0x71),
    "spindle": (0xB6, 0xD1, 0xEA),
    "spray": (0x79, 0xDE, 0xEC),
    "spring green": (0x00, 0xFF, 0x7F),
    "spring leaves": (0x57, 0x83, 0x63),
    "spring rain": (0xAC, 0xCB, 0xB1),
    "spring sun": (0xF6, 0xFF, 0xDC),
    "spring wood": (0xF8, 0xF6, 0xF1),
    "sprout": (0xC1, 0xD7, 0xB0),
    "spun pearl": (0xAA, 0xAB, 0xB7),
    "squirrel": (0x8F, 0x81, 0x76),
    "st tropaz": (0x2D, 0x56, 0x9B),
    "stack": (0x8A, 0x8F, 0x8A),
    "star dust": (0x9F, 0x9F, 0x9C),
    "stark white": (0xE5, 0xD7, 0xBD),
    "starship": (0xEC, 0xF2, 0x45),
    "steel blue": (0x46, 0x82, 0xB4),
    "steel gray": (0x26, 0x23, 0x35),
    "stiletto": (0x9C, 0x33, 0x36),
    "stonewall": (0x92, 0x85, 0x73),
    "storm dust": (0x64, 0x64, 0x63),
    "storm gray": (0x71, 0x74, 0x86),
    "stratos": (0x00, 0x07, 0x41),
    "straw": (0xD4, 0xBF, 0x8D),
    "strikemaster": (0x95, 0x63, 0x87),
    "stromboli": (0x32, 0x5D, 0x52),
    "studio": (0x71, 0x4A, 0xB2),
    "submarine": (0xBA, 0xC7, 0xC9),
    "sugar cane": (0xF9, 0xFF, 0xF6),
    "sulu": (0xC1, 0xF0, 0x7C),
    "summer green": (0x96, 0xBB, 0xAB),
    "sun": (0xFB, 0xAC, 0x13),
    "sundance": (0xC9, 0xB3, 0x5B),
    "sundown": (0xFF, 0xB1, 0xB3),
    "sunflower": (0xE4, 0xD4, 0x22),
    "sunglo": (0xE1, 0x68, 0x65),
    "sunglow": (0xFF, 0xCC, 0x33),
    "sunset orange": (0xFE, 0x4C, 0x40),
    "sunshade": (0xFF, 0x9E, 0x2C),
    "supernova": (0xFF, 0xC9, 0x01),
    "surf crest": (0xCF, 0xE5, 0xD2),
    "surf": (0xBB, 0xD7, 0xC1),
    "surfie green": (0x0C, 0x7A, 0x79),
    "sushi": (0x87, 0xAB, 0x39),
    "suva gray": (0x88, 0x83, 0x87),
    "swamp green": (0xAC, 0xB7, 0x8E),
    "swamp": (0x00, 0x1B, 0x1C),
    "swans down": (0xDC, 0xF0, 0xEA),
    "sweet corn": (0xFB, 0xEA, 0x8C),
    "sweet pink": (0xFD, 0x9F, 0xA2),
    "swirl": (0xD3, 0xCD, 0xC5),
    "swiss coffee": (0xDD, 0xD6, 0xD5),
    "sycamore": (0x90, 0x8D, 0x39),
    "tabasco": (0xA0, 0x27, 0x12),
    "tacao": (0xED, 0xB3, 0x81),
    "tacha": (0xD6, 0xC5, 0x62),
    "tahiti gold": (0xE9, 0x7C, 0x07),
    "tahuna sands": (0xEE, 0xF0, 0xC8),
    "tall poppy": (0xB3, 0x2D, 0x29),
    "tallow": (0xA8, 0xA5, 0x89),
    "tamarillo": (0x99, 0x16, 0x13),
    "tamarind": (0x34, 0x15, 0x15),
    "tan hide": (0xFA, 0x9D, 0x5A),
    "tan": (0xD2, 0xB4, 0x8C),
    "tana": (0xD9, 0xDC, 0xC1),
    "tangaroa": (0x03, 0x16, 0x3C),
    "tangerine": (0xF2, 0x85, 0x00),
    "tango": (0xED, 0x7A, 0x1C),
    "tapa": (0x7B, 0x78, 0x74),
    "tapestry": (0xB0, 0x5E, 0x81),
    "tara": (0xE1, 0xF6, 0xE8),
    "tarawera": (0x07, 0x3A, 0x50),
    "tasman": (0xCF, 0xDC, 0xCF),
    "taupe gray": (0xB3, 0xAF, 0x95),
    "taupe": (0x48, 0x3C, 0x32),
    "tawny port": (0x69, 0x25, 0x45),
    "te papa green": (0x1E, 0x43, 0x3C),
    "tea green": (0xD0, 0xF0, 0xC0),
    "tea": (0xC1, 0xBA, 0xB0),
    "teak": (0xB1, 0x94, 0x61),
    "teal blue": (0x04, 0x42, 0x59),
    "teal": (0x00, 0x80, 0x80),
    "temptress": (0x3B, 0x00, 0x0B),
    "tenn": (0xCD, 0x57, 0x00),
    "tequila": (0xFF, 0xE6, 0xC7),
    "terracotta": (0xE2, 0x72, 0x5B),
    "texas rose": (0xFF, 0xB5, 0x55),
    "texas": (0xF8, 0xF9, 0x9C),
    "thatch green": (0x40, 0x3D, 0x19),
    "thatch": (0xB6, 0x9D, 0x98),
    "thistle green": (0xCC, 0xCA, 0xA8),
    "thistle": (0xD8, 0xBF, 0xD8),
    "thunder": (0x33, 0x29, 0x2F),
    "thunderbird": (0xC0, 0x2B, 0x18),
    "tia maria": (0xC1, 0x44, 0x0E),
    "tiara": (0xC3, 0xD1, 0xD1),
    "tiber": (0x06, 0x35, 0x37),
    "tickle me pink": (0xFC, 0x80, 0xA5),
    "tidal": (0xF1, 0xFF, 0xAD),
    "tide": (0xBF, 0xB8, 0xB0),
    "timber green": (0x16, 0x32, 0x2C),
    "timberwolf": (0xD9, 0xD6, 0xCF),
    "titan white": (0xF0, 0xEE, 0xFF),
    "toast": (0x9A, 0x6E, 0x61),
    "tobacco brown": (0x71, 0x5D, 0x47),
    "toledo": (0x3A, 0x00, 0x20),
    "tolopea": (0x1B, 0x02, 0x45),
    "tom thumb": (0x3F, 0x58, 0x3B),
    "tomato": (0xFF, 0x63, 0x47),
    "tonys pink": (0xE7, 0x9F, 0x8C),
    "topaz": (0x7C, 0x77, 0x8A),
    "torch red": (0xFD, 0x0E, 0x35),
    "torea bay": (0x0F, 0x2D, 0x9E),
    "tory blue": (0x14, 0x50, 0xAA),
    "tosca": (0x8D, 0x3F, 0x3F),
    "totem pole": (0x99, 0x1B, 0x07),
    "tower gray": (0xA9, 0xBD, 0xBF),
    "tradewind": (0x5F, 0xB3, 0xAC),
    "tranquil": (0xE6, 0xFF, 0xFF),
    "travertine": (0xFF, 0xFD, 0xE8),
    "tree poppy": (0xFC, 0x9C, 0x1D),
    "treehouse": (0x3B, 0x28, 0x20),
    "trendy green": (0x7C, 0x88, 0x1A),
    "trendy pink": (0x8C, 0x64, 0x95),
    "trinidad": (0xE6, 0x4E, 0x03),
    "tropical blue": (0xC3, 0xDD, 0xF9),
    "tropical rain forest": (0x00, 0x75, 0x5E),
    "trout": (0x4A, 0x4E, 0x5A),
    "true v": (0x8A, 0x73, 0xD6),
    "tuatara": (0x36, 0x35, 0x34),
    "tuft bush": (0xFF, 0xDD, 0xCD),
    "tulip tree": (0xEA, 0xB3, 0x3B),
    "tumbleweed": (0xDE, 0xA6, 0x81),
    "tuna": (0x35, 0x35, 0x42),
    "tundora": (0x4A, 0x42, 0x44),
    "turbo": (0xFA, 0xE6, 0x00),
    "turkish rose": (0xB5, 0x72, 0x81),
    "turmeric": (0xCA, 0xBB, 0x48),
    "turquoise blue": (0x6C, 0xDA, 0xE7),
    "turquoise": (0x30, 0xD5, 0xC8),
    "turtle green": (0x2A, 0x38, 0x0B),
    "tuscany": (0xBD, 0x5E, 0x2E),
    "tusk": (0xEE, 0xF3, 0xC3),
    "tussock": (0xC5, 0x99, 0x4B),
    "tutu": (0xFF, 0xF1, 0xF9),
    "twilight blue": (0xEE, 0xFD, 0xFF),
    "twilight": (0xE4, 0xCF, 0xDE),
    "twine": (0xC2, 0x95, 0x5D),
    "tyrian purple": (0x66, 0x02, 0x3C),
    "ultramarine": (0x12, 0x0A, 0x8F),
    "valencia": (0xD8, 0x44, 0x37),
    "valentino": (0x35, 0x0E, 0x42),
    "valhalla": (0x2B, 0x19, 0x4F),
    "van cleef": (0x49, 0x17, 0x0C),
    "vanilla ice": (0xF3, 0xD9, 0xDF),
    "vanilla": (0xD1, 0xBE, 0xA8),
    "varden": (0xFF, 0xF6, 0xDF),
    "venetian red": (0x72, 0x01, 0x0F),
    "venice blue": (0x05, 0x59, 0x89),
    "venus": (0x92, 0x85, 0x90),
    "verdigris": (0x5D, 0x5E, 0x37),
    "verdun green": (0x49, 0x54, 0x00),
    "vermilion": (0xFF, 0x4D, 0x00),
    "vesuvius": (0xB1, 0x4A, 0x0B),
    "victoria": (0x53, 0x44, 0x91),
    "vida loca": (0x54, 0x90, 0x19),
    "viking": (0x64, 0xCC, 0xDB),
    "vin rouge": (0x98, 0x3D, 0x61),
    "viola": (0xCB, 0x8F, 0xA9),
    "violent violet": (0x29, 0x0C, 0x5E),
    "violet eggplant": (0x99, 0x11, 0x99),
    "violet red": (0xF7, 0x46, 0x8A),
    "violet": (0x24, 0x0A, 0x40),
    "viridian green": (0x67, 0x89, 0x75),
    "viridian": (0x40, 0x82, 0x6D),
    "vis vis": (0xFF, 0xEF, 0xA1),
    "vista blue": (0x8F, 0xD6, 0xB4),
    "vista white": (0xFC, 0xF8, 0xF7),
    "vivid tangerine": (0xFF, 0x99, 0x80),
    "vivid violet": (0x80, 0x37, 0x90),
    "voodoo": (0x53, 0x34, 0x55),
    "vulcan": (0x10, 0x12, 0x1D),
    "wafer": (0xDE, 0xCB, 0xC6),
    "waikawa gray": (0x5A, 0x6E, 0x9C),
    "waiouru": (0x36, 0x3C, 0x0D),
    "walnut": (0x77, 0x3F, 0x1A),
    "wannabe.house": (0x00, 0x00, 95),
    "wasabi": (0x78, 0x8A, 0x25),
    "water leaf": (0xA1, 0xE9, 0xDE),
    "watercourse": (0x05, 0x6F, 0x57),
    "waterloo ": (0x7B, 0x7C, 0x94),
    "wattle": (0xDC, 0xD7, 0x47),
    "watusi": (0xFF, 0xDD, 0xCF),
    "wax flower": (0xFF, 0xC0, 0xA8),
    "we peep": (0xF7, 0xDB, 0xE6),
    "web orange": (0xFF, 0xA5, 0x00),
    "wedgewood": (0x4E, 0x7F, 0x9E),
    "well read": (0xB4, 0x33, 0x32),
    "west coast": (0x62, 0x51, 0x19),
    "west side": (0xFF, 0x91, 0x0F),
    "westar": (0xDC, 0xD9, 0xD2),
    "wewak": (0xF1, 0x9B, 0xAB),
    "wheat": (0xF5, 0xDE, 0xB3),
    "wheatfield": (0xF3, 0xED, 0xCF),
    "whiskey": (0xD5, 0x9A, 0x6F),
    "whisper": (0xF7, 0xF5, 0xFA),
    "white ice": (0xDD, 0xF9, 0xF1),
    "white lilac": (0xF8, 0xF7, 0xFC),
    "white linen": (0xF8, 0xF0, 0xE8),
    "white pointer": (0xFE, 0xF8, 0xFF),
    "white rock": (0xEA, 0xE8, 0xD4),
    "white smoke": (0xF5, 0xF5, 0xF5),
    "white": (0xFF, 0xFF, 0xFF),
    "wild blue yonder": (0x7A, 0x89, 0xB8),
    "wild rice": (0xEC, 0xE0, 0x90),
    "wild sand": (0xF4, 0xF4, 0xF4),
    "wild strawberry": (0xFF, 0x33, 0x99),
    "wild watermelon": (0xFD, 0x5B, 0x78),
    "wild willow": (0xB9, 0xC4, 0x6A),
    "william": (0x3A, 0x68, 0x6C),
    "willow brook": (0xDF, 0xEC, 0xDA),
    "willow grove": (0x65, 0x74, 0x5D),
    "windsor": (0x3C, 0x08, 0x78),
    "wine berry": (0x59, 0x1D, 0x35),
    "winter hazel": (0xD5, 0xD1, 0x95),
    "wisp pink": (0xFE, 0xF4, 0xF8),
    "wisteria": (0x97, 0x71, 0xB5),
    "wistful": (0xA4, 0xA6, 0xD3),
    "witch haze": (0xFF, 0xFC, 0x99),
    "wood bark": (0x26, 0x11, 0x05),
    "woodland": (0x4D, 0x53, 0x28),
    "woodrush": (0x30, 0x2A, 0x0F),
    "woodsmoke": (0x0C, 0x0D, 0x0F),
    "woody brown": (0x48, 0x31, 0x31),
    "xanadu": (0x73, 0x86, 0x78),
    "yellow green": (0xC5, 0xE1, 0x7A),
    "yellow metal": (0x71, 0x63, 0x38),
    "yellow orange": (0xFF, 0xAE, 0x42),
    "yellow sea": (0xFE, 0xA9, 0x04),
    "yellow": (0xFF, 0xFF, 0x00),
    "your pink": (0xFF, 0xC3, 0xC0),
    "yukon gold": (0x7B, 0x66, 0x08),
    "yuma": (0xCE, 0xC2, 0x91),
    "zambezi": (0x68, 0x55, 0x58),
    "zanah": (0xDA, 0xEC, 0xD6),
    "zest": (0xE5, 0x84, 0x1B),
    "zeus": (0x29, 0x23, 0x19),
    "ziggurat": (0xBF, 0xDB, 0xE2),
    "zinnwaldite": (0xEB, 0xC2, 0xAF),
    "zircon": (0xF4, 0xF8, 0xFF),
    "zombie": (0xE4, 0xD6, 0x9B),
    "zorba": (0xA5, 0x9B, 0x91),
    "zuccini": (0x04, 0x40, 0x22),
    "zumthor": (0xED, 0xF6, 0xFF),
}


def clear() -> str:
    """Returns:
    An ANSI escape sequence that clears the screen.
    """
    return "\x1B[H\x1B[2J"


def clear_screen() -> str:
    """Returns:
    An ANSI escape sequence that clears the screen.
    """
    return clear()


def clear_line() -> str:
    """Returns:
    An ANSI escape sequence that clears the current line from the cursor
    position to the end of the line.
    """
    return "\x1B[2K\r"


def reset() -> str:
    """Returns:
        An ANSI escape sequence that resets text attributes to 'normal'.  This
        sequence ends any different foreground or background color settings.
        It also ends any special text styling (italics, bold, underline, etc...)
        that have been previously set.

    See also :py:meth:`reset_bg` and :py:meth:`reset_fg`.
    """
    return "\x1B[m"


def normal() -> str:
    """Returns:
        An ANSI escape sequence that resets text attributes to 'normal'.  This
        sequence ends any different foreground or background color settings.
        It also ends any special text styling (italics, bold, underline, etc...)
        that have been previously set.

    See also :py:meth:`reset_bg` and :py:meth:`reset_fg`.
    """
    return reset()


def bold() -> str:
    """Returns:
    The ANSI escape sequence to set text to bold weight.
    """
    return "\x1B[1m"


def italic() -> str:
    """Returns:
    The ANSI escape sequence to set text to italics style.
    """
    return "\x1B[3m"


def italics() -> str:
    """Returns:
    The ANSI escape sequence to set text to italics style.
    """
    return italic()


def underline() -> str:
    """Returns:
    The ANSI escape sequence to set text to underlined style.
    """
    return "\x1B[4m"


def strikethrough() -> str:
    """Returns:
    The ANSI escape sequence to set text to strike-through mode.
    """
    return "\x1B[9m"


def strike_through() -> str:
    """Returns:
    The ANSI escape sequence to set text to strike-through mode.
    """
    return strikethrough()


def _is_16color(num: int) -> bool:
    """One ANSI escape sequences (^[[#m) can be used to change text
    foreground and background color if and only if the R, G and B
    numbers are 128 or 255.  This means this code can be used to
    create 16 colors.
    """
    return num in (255, 128)


def _is_216color(num: int) -> bool:
    """Another ANSI escape sequence (^[[38;5;#m) can be used to change
    text color to 216 separate color values with each of R, G and B
    one of 6 possible values."""
    return num in set([0, 95, 135, 175, 223, 255])


def _simple_color_number(red: int, green: int, blue: int) -> int:
    """Construct a simple color number.  This is a 3 bit number
    used to construct a 16-color escape code."""
    r = red > 0
    g = green > 0
    b = blue > 0
    return b << 2 | g << 1 | r


def fg_16color(red: int, green: int, blue: int) -> str:
    """
    Set text foreground color to a color in 16-color space.

    Args:
        red: the red channel value of the foreground color to set
        green: the green channel value of the foreground color to set
        blue: the blue channel value of the foreground color to set

    Returns:
        An ANSI escape code that sets the foreground color described
        by the red, green and blue from the 16 color space.

    .. note::

        In 16 color mode, the possible color values are limited to
        red, green, yellow, blue, purple, cyan, white and black
        each with or without a "bright" attribute.  This function
        takes R/G/B parameter values that can be used to describe
        colors that can't be represented in 16-color space.  If
        such a color is described by the parameters, it maps the
        color to its closest representation in 16-color space.

    This is used by :py:meth:`fg` internally but can be called
    directly too.  See also :py:meth:`fg_216color`,
    :py:meth:`fg_24bit`, and :py:meth:`bg_16color`.
    """

    code = _simple_color_number(red, green, blue) + 30
    bright_count = 0
    if red > 128:
        bright_count += 1
    if green > 128:
        bright_count += 1
    if blue > 128:
        bright_count += 1
    if bright_count > 1:
        code += 60
    return f"\x1B[{code}m"


def bg_16color(red: int, green: int, blue: int) -> str:
    """
    Set text background color to a color in 16-color space.

    Args:
        red: the red channel value of background color to set
        green: the green channel value of the background color to set
        blue: the blue channel value of the background color to set

    Returns:
        An ANSI escape sequence that sets the background color to the
        color described by the red, green and blue parameters in the
        16 color space.

    .. note::

        In 16 color mode, the possible color values are limited to
        red, green, yellow, blue, purple, cyan, white and black
        each with or without a "bright" attribute.  This function
        takes R/G/B parameter values that can be used to describe
        colors that can't be represented in 16-color space.  If
        such a color is described by the parameters, it maps the
        color to its closest representation in 16-color space.

    This is used by :py:meth:`bg` internally but can be invoked
    directly if needed.  See also :py:meth:`fg_16color`,
    :py:meth:`bg_216color`, and :py:meth:`bg_24bit`.
    """

    code = _simple_color_number(red, green, blue) + 40
    bright_count = 0
    if red > 128:
        bright_count += 1
    if green > 128:
        bright_count += 1
    if blue > 128:
        bright_count += 1
    if bright_count > 1:
        code += 60
    return f"\x1B[{code}m"


def _pixel_to_216color(n: int) -> int:
    """Help convert full RGB color descriptions into valid 216 color space"""
    if n >= 255:
        return 5
    if n >= 233:
        return 4
    if n >= 175:
        return 3
    if n >= 135:
        return 2
    if n >= 95:
        return 1
    return 0


def fg_216color(red: int, green: int, blue: int) -> str:
    """
    Set text foreground color to a color in 216 color space.

    Args:
        red: the red channel value of the foreground color to set
        green: the green channel value of the foreground color to set
        blue: the blue channel value of the foreground color to set

    Returns:
        An ANSI escape code that sets the foreground color described
        by the red, green and blue from the 216 color space.

    .. note::

        In 216 color mode there are 216 total colors available.
        This is less than the 16M (256^3) possibilities that can
        be described by full RGB tuples.  When passed colors that
        are not available in 216 color mode, this code finds the
        closest match in 216 color space and returns that.

    This is used by :py:meth:`fg` internally but can be invoked
    directly if needed.  See also :py:meth:`fg_16color`,
    :py:meth`fg_24bit`, and :py:meth:`bg_216color`.
    """
    r = _pixel_to_216color(red)
    g = _pixel_to_216color(green)
    b = _pixel_to_216color(blue)
    code = 16 + r * 36 + g * 6 + b
    return f"\x1B[38;5;{code}m"


def bg_216color(red: int, green: int, blue: int) -> str:
    """
    Set text background color to a color in 216 color space.

    Args:
        red: the red channel value of the background color to set
        green: the green channel value of the background color to set
        blue: the blue channel value of the foreground color to set

    Returns:
        An ANSI escape code that sets the background color described
        by the red, green and blue from the 216 color space.

    .. note::

        In 216 color mode there are 216 total colors available.
        This is less than the 16M (256^3) possibilities that can
        be described by full RGB tuples.  When passed colors that
        are not available in 216 color mode, this code finds the
        closest match in 216 color space and returns that.

    This is used by :py:meth:`bg` internally but can be invoked
    directly if needed.  See also :py:meth:`bg_16color`,
    :py:meth:`bg_24bit`, and :py:meth:`fg_216color`.
    """
    r = _pixel_to_216color(red)
    g = _pixel_to_216color(green)
    b = _pixel_to_216color(blue)
    code = 16 + r * 36 + g * 6 + b
    return f"\x1B[48;5;{code}m"


def _pixel_to_24bit_color(value: int) -> int:
    """Helper to ensure a color channel value is valid in 24-bit color space."""
    if value < 0:
        return 0
    if value > 255:
        return 255
    return value


def fg_24bit(red: int, green: int, blue: int) -> str:
    """
    Set text foreground color to a color in 24-bit color space.

    Args:
        red: the red channel value of the foreground color to set
        green: the green channel value of the foreground color to set
        blue: the blue channel value of the foreground color to set

    Returns:
        An ANSI escape code that sets the foreground color described
        by the red, green and blue from 24-bit color space.

    .. note::

        In 24-bit color space we can represent any color described
        by red, green or blue values where 0 <= value <= 255.
        Values outside of this range will be mapped into the 24-bit
        color space.

    This is used by :py:meth:`fg` internally but can be invoked directly
    if useful.  See also :py:meth:`fg_216color` and :py:meth:`bg_24bit`.
    """
    return f"\x1B[38;2;{_pixel_to_24bit_color(red)};{_pixel_to_24bit_color(green)};{_pixel_to_24bit_color(blue)}m"


def bg_24bit(red: int, green: int, blue: int) -> str:
    """
    Set text background color to a color in 24-bit color space.

    Args:
        red: the red channel value of the background color to set
        green: the green channel value of the backgrounf color to set
        blue: the blue channel value of the background color to set

    Returns:
        An ANSI escape code that sets the background color described
        by the red, green and blue from 24-bit color space.

    .. note::

        In 24-bit color space we can represent any color described
        by red, green or blue values where 0 <= value <= 255.
        Values outside of this range will be mapped into the 24-bit
        color space.

    This is used by :py:meth:`fg` internally but can be invoked directly
    if useful.  See also :py:meth:`fg_216color` and :py:meth:`bg_24bit`.
    """
    return f"\x1B[48;2;{_pixel_to_24bit_color(red)};{_pixel_to_24bit_color(green)};{_pixel_to_24bit_color(blue)}m"


def _find_color_by_name(name: str) -> Tuple[int, int, int]:
    """Given a color name, look up its RGB channel values from the COLOR_NAMES_TO_RGB
    table."""
    rgb = COLOR_NAMES_TO_RGB.get(name.lower(), None)
    if rgb is None:
        name = _guess_name(name)
        rgb = COLOR_NAMES_TO_RGB.get(name.lower(), None)
        assert rgb is not None
    return rgb


@logging_utils.squelch_repeated_log_messages(1)
def fg(
    name: Optional[str] = "",
    red: Optional[int] = None,
    green: Optional[int] = None,
    blue: Optional[int] = None,
    *,
    force_16color: bool = False,
    force_216color: bool = False,
) -> str:
    """Return the ANSI escape sequence to change the foreground color
    text is printed to the console with.  Target colors may be
    indicated either by name or R/G/B values.  Result will use the 16
    or 216 color scheme if force_16color or force_216color are passed
    (respectively).  Otherwise the code will do what it thinks best.

    Args:
        name: the name of the color to set
        red: the color to set's red component value
        green: the color to set's green component value
        blue: the color to set's blue component value
        force_16color: force fg to use 16 color mode
        force_216color: force fg to use 216 color mode

    Returns:
        String containing the ANSI escape sequence to set desired foreground

    .. note::

        16-color and 216-color spaces can't be used to represent all colors
        describable by 8 bit R, G and B channels (i.e. normal R/G/B hex values)
        If you set the force_16color or force_216color arguments but describe
        a color (by name or R/G/B) that can't be represented in the forced
        color space the code will pick the closest approximation available.

    >>> import string_utils as su
    >>> su.to_base64(fg('blue'))
    b'G1szODs1OzIxbQ==\\n'
    """
    if name is not None and name == 'reset':
        return reset_fg()

    if name is not None and string_utils.is_full_string(name):
        rgb = _find_color_by_name(name)
        return fg(
            None,
            rgb[0],
            rgb[1],
            rgb[2],
            force_16color=force_16color,
            force_216color=force_216color,
        )

    if red is None:
        red = 0
    if green is None:
        green = 0
    if blue is None:
        blue = 0
    if (_is_16color(red) and _is_16color(green) and _is_16color(blue)) or force_16color:
        logger.debug("Using 16-color strategy")
        return fg_16color(red, green, blue)
    if (
        _is_216color(red) and _is_216color(green) and _is_216color(blue)
    ) or force_216color:
        logger.debug("Using 216-color strategy")
        return fg_216color(red, green, blue)
    logger.debug("Using 24-bit color strategy")
    return fg_24bit(red, green, blue)


def reset_fg():
    """Returns: an ANSI escape code to reset just the foreground color
    while preserving the background color and any other formatting
    (bold, italics, etc...)
    """
    return '\033[39m'


def _rgb_to_yiq(rgb: Tuple[int, int, int]) -> int:
    """Helper for contrasting pick_contrasting_color.  Maps an RGB
    color tuple in to YIQ space.  See: https://en.wikipedia.org/wiki/YIQ."""
    return (rgb[0] * 299 + rgb[1] * 587 + rgb[2] * 114) // 1000


def _contrast(rgb: Tuple[int, int, int]) -> Tuple[int, int, int]:
    """Helper for contrasting pick_contrasting_color."""
    if _rgb_to_yiq(rgb) < 128:
        return (0xFF, 0xFF, 0xFF)
    return (0, 0, 0)


def pick_contrasting_color(
    name: Optional[str] = "",
    red: Optional[int] = None,
    green: Optional[int] = None,
    blue: Optional[int] = None,
) -> Tuple[int, int, int]:
    """This method will return a red, green, blue tuple representing a
    contrasting color given the red, green, blue of a background
    color or a color name of the background color.  This is meant to
    help ensure text printed on this background color will be visible.

    Args:
        name: the name of the color to contrast
        red: the color to contrast's red component value
        green: the color to contrast's green component value
        blue: the color to contrast's blue component value

    Returns:
        An RGB tuple containing a contrasting color

    >>> pick_contrasting_color(None, 20, 20, 20)
    (255, 255, 255)

    >>> pick_contrasting_color("white")
    (0, 0, 0)

    """
    if name is not None and string_utils.is_full_string(name):
        rgb = _find_color_by_name(name)
    else:
        r = red if red is not None else 0
        g = green if green is not None else 0
        b = blue if blue is not None else 0
        rgb = (r, g, b)
    assert rgb is not None
    return _contrast(rgb)


def _guess_name(name: str) -> str:
    """Try to guess what color the user is talking about"""
    best_guess = None
    max_ratio = None
    for possibility in COLOR_NAMES_TO_RGB:
        r = difflib.SequenceMatcher(None, name, possibility).ratio()
        if max_ratio is None or r > max_ratio:
            max_ratio = r
            best_guess = possibility
    assert best_guess is not None
    logger.debug("Best guess at color name is %s", best_guess)
    return best_guess


@logging_utils.squelch_repeated_log_messages(1)
def bg(
    name: Optional[str] = "",
    red: Optional[int] = None,
    green: Optional[int] = None,
    blue: Optional[int] = None,
    *,
    force_16color: bool = False,
    force_216color: bool = False,
) -> str:
    """Returns an ANSI color code for changing the current text background
    color.

    Args:
        name: the name of the color to set
        red: the color to set's red component value
        green: the color to set's green component value
        blue: the color to set's blue component value
        force_16color: force bg to use 16 color mode
        force_216color: force bg to use 216 color mode

    Returns:
        A string containing the requested escape sequence

    .. note::

        16-color and 216-color spaces can't be used to represent all colors
        describable by 8 bit R, G and B channels (i.e. normal R/G/B hex values)
        If you set the force_16color or force_216color arguments but describe
        a color (by name or R/G/B) that can't be represented in the forced
        color space the code will pick the closest approximation available.

    >>> import string_utils as su
    >>> su.to_base64(bg("red"))    # b'\x1b[48;5;196m'
    b'G1s0ODs1OzE5Nm0=\\n'
    """
    if name is not None and name == 'reset':
        return reset_bg()

    if name is not None and string_utils.is_full_string(name):
        rgb = _find_color_by_name(name)
        return bg(
            None,
            rgb[0],
            rgb[1],
            rgb[2],
            force_16color=force_16color,
            force_216color=force_216color,
        )
    if red is None:
        red = 0
    if green is None:
        green = 0
    if blue is None:
        blue = 0
    if (_is_16color(red) and _is_16color(green) and _is_16color(blue)) or force_16color:
        logger.debug("Using 16-color strategy")
        return bg_16color(red, green, blue)
    if (
        _is_216color(red) and _is_216color(green) and _is_216color(blue)
    ) or force_216color:
        logger.debug("Using 216-color strategy")
        return bg_216color(red, green, blue)
    logger.debug("Using 24-bit color strategy")
    return bg_24bit(red, green, blue)


def reset_bg():
    """Returns an ANSI escape sequence that resets text background
    color to the default but preserves foreground coloring and text
    attributes like bold, italics, underlines, etc...
    """
    return '\033[49m'


class _StdoutInterceptor(io.TextIOBase, contextlib.AbstractContextManager):
    """An interceptor for data written to stdout.  Use as a context."""

    def __init__(self):
        super().__init__()
        self.saved_stdout: io.TextIO = None
        self.buf = ''

    @abstractmethod
    def write(self, s: str):
        pass

    def __enter__(self):
        self.saved_stdout = sys.stdout
        sys.stdout = self
        return self

    def __exit__(self, *args) -> None:
        sys.stdout = self.saved_stdout
        print(self.buf)


class ProgrammableColorizer(_StdoutInterceptor):
    """A colorizing interceptor; pass it re.Patterns -> methods that
    do something (usually add color to) the match.  This may be useful
    for adding color to non-colorized text in a stream without
    changing the code that emits the text directly.  In the example
    doctest below I'm inserting [RED] and [RESET] strings but you
    could just as easily insert escape sequences returned from
    :py:meth:`fg`, :py:meth:`bg`, and :py:meth:`reset`.

    >>> def red(match: re.Match, s: str) -> str:
    ...     return '[RED]'

    >>> def reset(match: re.Match, s: str) -> str:
    ...     return '[RESET]'

    >>> with ProgrammableColorizer( [ (re.compile('^[^ ]+'), red),
    ...                               (re.compile('$'), reset) ] ) as c:
    ...     c.write("This matches the pattern and will call red()")
    ...     c.write("     ...this won't")
    [RED] matches the pattern and will call red()[RESET]     ...this won't[RESET]
    """

    def __init__(
        self,
        patterns: Iterable[Tuple[re.Pattern, Callable[[re.Match[str], str], str]]],
    ):
        """
        Setup the programmable colorizing context; tell it how to operate.

        Args:
            patterns: an iterable collection of tuples.  Each tuple has an
                re.Pattern that describes the text pattern which
                will trigger the colorization and a method to call when the
                pattern is matched.  These methods receive the `re.MATCH`
                object and usually just emit some ANSI escape sequence to
                colorize the stream.  See the example above.
        """
        super().__init__()
        self.patterns = list(patterns)

    @overrides
    def write(self, s: str):
        """Use this method to feed the stream of text through the colorizer.
        See the example above.

        Args:
            s: A line from the stream to colorize.
        """
        for pattern in self.patterns:
            s = pattern[0].sub(pattern[1], s)
        self.buf += s


if __name__ == '__main__':

    def main() -> None:
        import doctest

        doctest.testmod()

        name = " ".join(sys.argv[1:])
        for possibility in COLOR_NAMES_TO_RGB:
            if name in possibility:
                f = fg(possibility)
                b = bg(possibility)
                _ = pick_contrasting_color(possibility)
                xf = fg(None, _[0], _[1], _[2])
                xb = bg(None, _[0], _[1], _[2])
                print(
                    f'{f}{xb}{possibility}{reset()}\t\t\t'
                    f'{b}{xf}{possibility}{reset()}'
                )

    main()
