#!/usr/bin/env python
from logging import getLogger

import genice2.lattices
import numpy as np
from genice2.cell import cellvectors
from genice2.CIF import fullatoms, operations, waters_and_pairs

desc = {"ref": {"11i": "Hirsch 2004"},
        "usage": "genice2 11i[num]\n\n'num' specifies the structure in Table 1 of the Ref. [Hirsch 2004]",
        "brief": "Sixteen candidates for Ice XI."
        }

Table1 = """
1. Cmc2_1
4.49225, 7.78080, 7.33581, 90.0, 90.0, 90.0
0.0000, 0.3333, 0.5625
0.0000, 0.4510, 0.5244
0.0000, 0.3386, 0.6929
0.5000, 0.1667, 0.4375
0.6685, 0.2282, 0.4836

2. Pna2_1
7.33581, 7.78080, 4.49225, 90.0, 90.0, 90.0
0.5625, 0.9167, 0.5000
0.5164, 0.9702, 0.3235
0.5164, 0.8017, 0.4920
0.5625, 0.4167, 0.0000
0.6929, 0.4193, 0.0080
0.5244, 0.4755, 0.1765

3. Pna2_1
7.33581, 7.78080, 4.49225, 90.0, 90.0, 90.0
0.5625, 0.9167, 0.5000
0.5164, 0.9702, 0.6765
0.5164, 0.8017, 0.5080
0.5625, 0.4167, 0.0000
0.6929, 0.4193, 0.0080
0.5244, 0.4755, 0.1765

4. Pbn2_1
4.49225, 7.78080, 7.33581, 90.0, 90.0, 90.0
0.5000, 0.8333, 0.5625
0.5080, 0.9483, 0.5164
0.6765, 0.7798, 0.5164
0.0000, 0.6667, 0.4375
0.0080, 0.6693, 0.3071
0.1765, 0.7255, 0.4756

5. Pca2_1
7.78080, 4.49225, 7.33581, 90.0, 90.0, 90.0
0.5833, 0.2500, 0.4375
0.5298, 0.0735, 0.4836
0.6983, 0.2420, 0.4836
0.4167, 0.7500, 0.5625
0.4193, 0.7420, 0.6929
0.4755, 0.5735, 0.5244

6. P2_12_12_1
4.49225, 7.78080, 7.33581, 90.0, 90.0, 90.0
0.7500, 0.1667, 0.4375
0.5735, 0.2202, 0.4836
0.7420, 0.0517, 0.4836
0.2500, 0.6667, 0.4375
0.2580, 0.6693, 0.3071
0.4265, 0.7255, 0.4756

7. P2_12_12_1
4.49225, 7.78080, 7.33581, 90.0, 90.0, 90.0
0.7500, 0.1667, 0.4375
0.5735, 0.2202, 0.4836
0.7420, 0.0517, 0.4836
0.2500, 0.6667, 0.4375
0.2420, 0.6693, 0.3071
0.0735, 0.7255, 0.4756

8. C1c1
4.49225, 7.78080, 7.33581, 90.0, 90.0, 90.0
0.0000, 0.3333, 0.5625
0.1765, 0.2798, 0.5164
0.0080, 0.4483, 0.5164
0.5000, 0.1667, 0.4375
0.5080, 0.1693, 0.3071
0.6765, 0.2255, 0.4756

9. P1c1
7.33581, 4.49225, 7.78080, 90.0, 90.0, 90.0
0.0625, 0.2500, 0.1667
0.0164, 0.0735, 0.2202
0.0164, 0.2420, 0.0517
0.5625, 0.2500, 0.8333
0.5164, 0.0815, 0.7719
0.5164, 0.4185, 0.7719
0.9375, 0.7500, 0.3333
0.8071, 0.7420, 0.3307
0.9756, 0.5735, 0.2745
0.4375, 0.7500, 0.6667
0.3071, 0.7500, 0.6614
0.4756, 0.7500, 0.5490

10. P1c1
4.49225, 7.33581, 8.98450, 90.0, 120.0, 90.0
0.6667, 0.6875, 0.1667
0.9020, 0.7256, 0.2255
0.6773, 0.5571, 0.1693
0.3333, 0.1875, 0.8333
0.4563, 0.2336, 0.7798
0.4563, 0.2336, 0.9483
0.6667, 0.6875, 0.6667
0.8966, 0.7336, 0.7282
0.5596, 0.7336, 0.7282
0.3333, 0.1875, 0.3333
0.3386, 0.0571, 0.3386
0.4510, 0.2256, 0.4510

11. P1c1
7.33581, 7.78080, 8.60200, 90.0, 148.52, 90.0
0.5625, 0.9167, 0.5000
0.6929, 0.9114, 0.5000
0.5244, 0.7990, 0.5000
0.0625, 0.5833, 0.5000
0.0084, 0.6983, 0.4920
0.8399, 0.5298, 0.3235
0.9375, 0.0833, 0.0000
0.8151, 0.0219, 0.8315
0.1521, 0.0219, 0.1685
0.4375, 0.4167, 0.0000
0.2992, 0.4193, 0.9920
0.2992, 0.4755, 0.8235

12. P12_11
7.78080, 4.49225, 7.33581, 90.0, 90.0, 90.0
0.5833, 0.5000, 0.0625
0.5298, 0.6765, 0.0164
0.6983, 0.5080, 0.0164
0.9167, 0.5000, 0.5625
0.9702, 0.6765, 0.5164
0.8017, 0.5080, 0.5164
0.0833, 0.0000, 0.0625
0.0807, 0.9920, 0.1929
0.0245, 0.8235, 0.0244
0.4167, 0.0000, 0.5625
0.4193, 0.0080, 0.6929
0.4755, 0.1765, 0.5244

13. P12_11
7.78080, 4.49225, 7.33581, 90.0, 90.0, 90.0
0.5833, 0.5000, 0.0625
0.5298, 0.6765, 0.0164
0.6983, 0.5080, 0.0164
0.9167, 0.5000, 0.5625
0.9702, 0.3235, 0.5164
0.8017, 0.4920, 0.5164
0.0833, 0.0000, 0.0625
0.0807, 0.9920, 0.1929
0.0245, 0.8235, 0.0244
0.4167, 0.0000, 0.5625
0.4193, 0.9920, 0.6929
0.4755, 0.8235, 0.5244

14. P12_11
7.78080, 7.33581, 4.49225, 90.0, 90.0, 90.0
0.9167, 0.0625, 0.2500
0.9114, 0.1929, 0.2500
0.7990, 0.0244, 0.2500
0.5833, 0.5625, 0.2500
0.5298, 0.5164, 0.0735
0.6983, 0.5164, 0.2420
0.0833, 0.9375, 0.7500
0.0218, 0.9836, 0.9185
0.0218, 0.9836, 0.5815
0.4167, 0.4375, 0.7500
0.4193, 0.3071, 0.7420
0.4755, 0.4756, 0.5735

15. P12_11
4.49225, 7.78080, 7.33581, 90.0, 90.0, 90.0
0.2500, 0.8333, 0.6875
0.4265, 0.7745, 0.7256
0.2580, 0.8307, 0.5571
0.2500, 0.1667, 0.1875
0.0735, 0.2202, 0.2336
0.2420, 0.0517, 0.2336
0.7500, 0.3333, 0.6875
0.9185, 0.2718, 0.7336
0.5815, 0.2718, 0.7336
0.7500, 0.6667, 0.1875
0.7500, 0.6614, 0.0571
0.7500, 0.5490, 0.2256

16. P1
4.49225, 7.78080, 7.33581, 90.0, 90.0, 90.0
0.0000, 0.6667, 0.0625
0.1765, 0.7202, 0.0164
0.0080, 0.5517, 0.0164
0.0000, 0.3333, 0.5625
0.1765, 0.2798, 0.5164
0.0080, 0.4483, 0.5164
0.5000, 0.1667, 0.0625
0.3235, 0.2202, 0.0164
0.4920, 0.0517, 0.0164
0.5000, 0.8333, 0.5625
0.5080, 0.9483, 0.5164
0.6765, 0.7798, 0.5164
0.5000, 0.8333, 0.9375
0.6765, 0.7745, 0.9756
0.5080, 0.8307, 0.8071
0.5000, 0.1667, 0.4375
0.5080, 0.1693, 0.3071
0.6765, 0.2255, 0.4756
0.0000, 0.3333, 0.9375
0.9920, 0.3307, 0.8071
0.8235, 0.2745, 0.9756
0.0000, 0.6667, 0.4375
0.0080, 0.6693, 0.3071
0.1765, 0.7255, 0.4756

""".splitlines()


class Lattice(genice2.lattices.Lattice):
    def __init__(self, **kwargs):
        logger = getLogger()

        elevens = dict()
        spaceg = None
        atoms = []
        for line in Table1:
            cols = line.split()
            if len(cols) == 0:
                if spaceg is not None:
                    names = ["OHH"[x%3]+f"{x}" for x in range(len(atoms))]
                    atomd = {name: pos for name, pos in zip(names, atoms)}
                    cur = {
                        "spaceg": spaceg,
                        "atomd": atomd,
                        "cell": cellvectors(a, b, c, A, B, G) / 10
                    }
                    elevens[label] = cur
            elif len(cols) == 2:
                # name of ice
                label, spaceg = cols
                label = int(label.rstrip("."))
            elif len(cols) == 6:
                a,b,c,A,B,G = [float(x) for x in line.split(",")]
                atoms = []
            elif len(cols) == 3:
                atoms.append([float(x) for x in line.split(",")])

        # The default is ferroelectric #1
        typ = 1

        # parse options
        for k, v in kwargs.items():
            if k == "type":
                typ = int(v)
            elif v is True:
                # unlabeled option
                typ = int(k)

        data = elevens[typ]
        spaceg = data["spaceg"]

        # When the cell is too thin, it is doubled.
        rep = [1,1,1]
        for d in range(3):
            if data["cell"][d,d] < 0.5:
                rep[d] = 2
        if typ == 10:
            rep = [2,1,2]

        # molecular positions and the HB network
        self.waters, self.fixed = waters_and_pairs(data["cell"], data["atomd"], operations(spaceg), rep=rep)
        self.pairs = self.fixed

        # define other parameters
        self.cell = data["cell"]
        self.density = 18 * len(self.waters) / 6.022e23 / (np.linalg.det(self.cell) * 1e-21)

        self.coord = "relative"
