"use strict";
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'add stacked property to graphs'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'Test widget',
            stacked: true,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'Test widget',
                    region: { Ref: 'AWS::Region' },
                    stacked: true,
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'add metrics to graphs on either axis'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }),
            ],
            right: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Tast' }),
            ],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                        ['CDK', 'Tast', { yAxis: 'right' }],
                    ],
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'label and color are respected in constructor'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            left: [new lib_1.Metric({ namespace: 'CDK', metricName: 'Test', label: 'MyMetric', color: '000000' })],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { label: 'MyMetric', color: '000000' }],
                    ],
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'singlevalue widget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        // WHEN
        const widget = new lib_1.SingleValueWidget({
            metrics: [metric],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 3,
                properties: {
                    view: 'singleValue',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                },
            }]);
        test.done();
    },
    'query result widget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const logGroup = { logGroupName: 'my-log-group' };
        // WHEN
        const widget = new lib_1.LogQueryWidget({
            logGroupNames: [logGroup.logGroupName],
            queryLines: [
                'fields @message',
                'filter @message like /Error/',
            ],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'log',
                width: 6,
                height: 6,
                properties: {
                    view: 'table',
                    region: { Ref: 'AWS::Region' },
                    query: `SOURCE '${logGroup.logGroupName}' | fields @message\n| filter @message like /Error/`,
                },
            }]);
        test.done();
    },
    'alarm widget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const alarm = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }).createAlarm(stack, 'Alarm', {
            evaluationPeriods: 2,
            threshold: 1000,
        });
        // WHEN
        const widget = new lib_1.AlarmWidget({
            alarm,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    annotations: {
                        alarms: [{ 'Fn::GetAtt': ['Alarm7103F465', 'Arn'] }],
                    },
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'add annotations to graph'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }),
            ],
            leftAnnotations: [{
                    value: 1000,
                    color: '667788',
                    fill: lib_1.Shading.BELOW,
                    label: 'this is the annotation',
                }],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                    annotations: { horizontal: [{
                                yAxis: 'left',
                                value: 1000,
                                color: '667788',
                                fill: 'below',
                                label: 'this is the annotation',
                            }] },
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'convert alarm to annotation'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        const alarm = metric.createAlarm(stack, 'Alarm', {
            evaluationPeriods: 7,
            datapointsToAlarm: 2,
            threshold: 1000,
        });
        // WHEN
        const widget = new lib_1.GraphWidget({
            right: [metric],
            rightAnnotations: [alarm.toAnnotation()],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { yAxis: 'right' }],
                    ],
                    annotations: {
                        horizontal: [{
                                yAxis: 'right',
                                value: 1000,
                                label: 'Test >= 1000 for 2 datapoints within 35 minutes',
                            }],
                    },
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'add yAxis to graph'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }),
            ],
            right: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Tast' }),
            ],
            leftYAxis: ({
                label: 'Left yAxis',
                max: 100,
            }),
            rightYAxis: ({
                label: 'Right yAxis',
                min: 10,
                showUnits: false,
            }),
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                        ['CDK', 'Tast', { yAxis: 'right' }],
                    ],
                    yAxis: {
                        left: { label: 'Left yAxis', max: 100 },
                        right: { label: 'Right yAxis', min: 10, showUnits: false }
                    },
                },
            }]);
        test.done();
    },
    'can use imported alarm with graph'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const alarm = lib_1.Alarm.fromAlarmArn(stack, 'Alarm', 'arn:aws:cloudwatch:region:account-id:alarm:alarm-name');
        // WHEN
        new lib_1.AlarmWidget({
            title: 'My fancy graph',
            alarm,
        });
        // THEN: Compiles
        test.done();
    },
    'add setPeriodToTimeRange to singleValueWidget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        // WHEN
        const widget = new lib_1.SingleValueWidget({
            metrics: [metric],
            setPeriodToTimeRange: true,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 3,
                properties: {
                    view: 'singleValue',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                    setPeriodToTimeRange: true,
                },
            }]);
        test.done();
    },
    'allows overriding custom values of dashboard widgets'(test) {
        class HiddenMetric extends lib_1.Metric {
            toMetricConfig() {
                const ret = super.toMetricConfig();
                // @ts-ignore
                ret.renderingProperties.visible = false;
                return ret;
            }
        }
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            left: [
                new HiddenMetric({ namespace: 'CDK', metricName: 'Test' }),
            ],
        });
        // test.ok(widget.toJson()[0].properties.metrics[0].visible === false);
        test.deepEqual(stack.resolve(widget.toJson())[0].properties.metrics[0], ['CDK', 'Test', { visible: false }]);
        test.done();
    },
    'GraphColor is correctly converted into the correct hexcode'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        // WHEN
        const widget = new lib_1.GraphWidget({
            left: [metric.with({
                    color: lib_1.Color.BLUE,
                })],
            leftAnnotations: [
                { color: lib_1.Color.RED, value: 100 },
            ],
        });
        test.deepEqual(stack.resolve(widget.toJson())[0].properties.metrics[0], ['CDK', 'Test', { color: '#1f77b4' }]);
        test.deepEqual(stack.resolve(widget.toJson())[0].properties.annotations.horizontal[0], { yAxis: 'left', value: 100, color: '#d62728' });
        test.done();
    },
    'legend position is respected in constructor'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            left: [new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' })],
            legendPosition: lib_1.LegendPosition.RIGHT,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                    yAxis: {},
                    legend: {
                        position: 'right',
                    },
                },
            }]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5ncmFwaHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LmdyYXBocy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsd0NBQXNDO0FBRXRDLGdDQUFvSTtBQUVwSSxpQkFBUztJQUNQLGdDQUFnQyxDQUFDLElBQVU7UUFDekMsT0FBTztRQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxpQkFBVyxDQUFDO1lBQzdCLEtBQUssRUFBRSxhQUFhO1lBQ3BCLE9BQU8sRUFBRSxJQUFJO1NBQ2QsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxDQUFDO2dCQUM5QyxJQUFJLEVBQUUsUUFBUTtnQkFDZCxLQUFLLEVBQUUsQ0FBQztnQkFDUixNQUFNLEVBQUUsQ0FBQztnQkFDVCxVQUFVLEVBQUU7b0JBQ1YsSUFBSSxFQUFFLFlBQVk7b0JBQ2xCLEtBQUssRUFBRSxhQUFhO29CQUNwQixNQUFNLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29CQUM5QixPQUFPLEVBQUUsSUFBSTtvQkFDYixLQUFLLEVBQUUsRUFBRTtpQkFDVjthQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNDQUFzQyxDQUFDLElBQVU7UUFDL0MsT0FBTztRQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxpQkFBVyxDQUFDO1lBQzdCLEtBQUssRUFBRSxnQkFBZ0I7WUFDdkIsSUFBSSxFQUFFO2dCQUNKLElBQUksWUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxVQUFVLEVBQUUsTUFBTSxFQUFFLENBQUM7YUFDckQ7WUFDRCxLQUFLLEVBQUU7Z0JBQ0wsSUFBSSxZQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxNQUFNLEVBQUUsQ0FBQzthQUNyRDtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQztnQkFDOUMsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsS0FBSyxFQUFFLENBQUM7Z0JBQ1IsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxZQUFZO29CQUNsQixLQUFLLEVBQUUsZ0JBQWdCO29CQUN2QixNQUFNLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29CQUM5QixPQUFPLEVBQUU7d0JBQ1AsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDO3dCQUNmLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsQ0FBQztxQkFDcEM7b0JBQ0QsS0FBSyxFQUFFLEVBQUU7aUJBQ1Y7YUFDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4Q0FBOEMsQ0FBQyxJQUFVO1FBQ3ZELE9BQU87UUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sTUFBTSxHQUFHLElBQUksaUJBQVcsQ0FBQztZQUM3QixJQUFJLEVBQUUsQ0FBQyxJQUFJLFlBQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFFO1NBQ2xHLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQztnQkFDOUMsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsS0FBSyxFQUFFLENBQUM7Z0JBQ1IsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxZQUFZO29CQUNsQixNQUFNLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29CQUM5QixPQUFPLEVBQUU7d0JBQ1AsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFFLENBQUM7cUJBQ3hEO29CQUNELEtBQUssRUFBRSxFQUFFO2lCQUNWO2FBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsb0JBQW9CLENBQUMsSUFBVTtRQUM3QixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLE1BQU0sR0FBRyxJQUFJLFlBQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUM7UUFFcEUsT0FBTztRQUNQLE1BQU0sTUFBTSxHQUFHLElBQUksdUJBQWlCLENBQUM7WUFDbkMsT0FBTyxFQUFFLENBQUUsTUFBTSxDQUFFO1NBQ3BCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQztnQkFDOUMsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsS0FBSyxFQUFFLENBQUM7Z0JBQ1IsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxhQUFhO29CQUNuQixNQUFNLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29CQUM5QixPQUFPLEVBQUU7d0JBQ1AsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDO3FCQUNoQjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFCQUFxQixDQUFDLElBQVU7UUFDOUIsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxRQUFRLEdBQUcsRUFBQyxZQUFZLEVBQUUsY0FBYyxFQUFDLENBQUM7UUFFaEQsT0FBTztRQUNQLE1BQU0sTUFBTSxHQUFHLElBQUksb0JBQWMsQ0FBQztZQUNoQyxhQUFhLEVBQUUsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDO1lBQ3RDLFVBQVUsRUFBRTtnQkFDVixpQkFBaUI7Z0JBQ2pCLDhCQUE4QjthQUMvQjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQztnQkFDOUMsSUFBSSxFQUFFLEtBQUs7Z0JBQ1gsS0FBSyxFQUFFLENBQUM7Z0JBQ1IsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxPQUFPO29CQUNiLE1BQU0sRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7b0JBQzlCLEtBQUssRUFBRSxXQUFXLFFBQVEsQ0FBQyxZQUFZLHFEQUFxRDtpQkFDN0Y7YUFDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxjQUFjLENBQUMsSUFBVTtRQUN2QixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7WUFDN0YsaUJBQWlCLEVBQUUsQ0FBQztZQUNwQixTQUFTLEVBQUUsSUFBSTtTQUNoQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSxpQkFBVyxDQUFDO1lBQzdCLEtBQUs7U0FDTixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLENBQUM7Z0JBQzlDLElBQUksRUFBRSxRQUFRO2dCQUNkLEtBQUssRUFBRSxDQUFDO2dCQUNSLE1BQU0sRUFBRSxDQUFDO2dCQUNULFVBQVUsRUFBRTtvQkFDVixJQUFJLEVBQUUsWUFBWTtvQkFDbEIsTUFBTSxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRTtvQkFDOUIsV0FBVyxFQUFFO3dCQUNYLE1BQU0sRUFBRSxDQUFDLEVBQUUsWUFBWSxFQUFFLENBQUUsZUFBZSxFQUFFLEtBQUssQ0FBRSxFQUFFLENBQUM7cUJBQ3ZEO29CQUNELEtBQUssRUFBRSxFQUFFO2lCQUNWO2FBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMEJBQTBCLENBQUMsSUFBVTtRQUNuQyxPQUFPO1FBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLE1BQU0sR0FBRyxJQUFJLGlCQUFXLENBQUM7WUFDN0IsS0FBSyxFQUFFLGdCQUFnQjtZQUN2QixJQUFJLEVBQUU7Z0JBQ0osSUFBSSxZQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxNQUFNLEVBQUUsQ0FBQzthQUNyRDtZQUNELGVBQWUsRUFBRSxDQUFDO29CQUNoQixLQUFLLEVBQUUsSUFBSTtvQkFDWCxLQUFLLEVBQUUsUUFBUTtvQkFDZixJQUFJLEVBQUUsYUFBTyxDQUFDLEtBQUs7b0JBQ25CLEtBQUssRUFBRSx3QkFBd0I7aUJBQ2hDLENBQUM7U0FDSCxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLENBQUM7Z0JBQzlDLElBQUksRUFBRSxRQUFRO2dCQUNkLEtBQUssRUFBRSxDQUFDO2dCQUNSLE1BQU0sRUFBRSxDQUFDO2dCQUNULFVBQVUsRUFBRTtvQkFDVixJQUFJLEVBQUUsWUFBWTtvQkFDbEIsS0FBSyxFQUFFLGdCQUFnQjtvQkFDdkIsTUFBTSxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRTtvQkFDOUIsT0FBTyxFQUFFO3dCQUNQLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQztxQkFDaEI7b0JBQ0QsV0FBVyxFQUFFLEVBQUUsVUFBVSxFQUFFLENBQUM7Z0NBQzFCLEtBQUssRUFBRSxNQUFNO2dDQUNiLEtBQUssRUFBRSxJQUFJO2dDQUNYLEtBQUssRUFBRSxRQUFRO2dDQUNmLElBQUksRUFBRSxPQUFPO2dDQUNiLEtBQUssRUFBRSx3QkFBd0I7NkJBQ2hDLENBQUMsRUFBRTtvQkFDSixLQUFLLEVBQUUsRUFBRTtpQkFDVjthQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDZCQUE2QixDQUFDLElBQVU7UUFDdEMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBRXBFLE1BQU0sS0FBSyxHQUFHLE1BQU0sQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtZQUMvQyxpQkFBaUIsRUFBRSxDQUFDO1lBQ3BCLGlCQUFpQixFQUFFLENBQUM7WUFDcEIsU0FBUyxFQUFFLElBQUk7U0FDaEIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sTUFBTSxHQUFHLElBQUksaUJBQVcsQ0FBQztZQUM3QixLQUFLLEVBQUUsQ0FBRSxNQUFNLENBQUU7WUFDakIsZ0JBQWdCLEVBQUUsQ0FBRSxLQUFLLENBQUMsWUFBWSxFQUFFLENBQUU7U0FDM0MsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxDQUFDO2dCQUM5QyxJQUFJLEVBQUUsUUFBUTtnQkFDZCxLQUFLLEVBQUUsQ0FBQztnQkFDUixNQUFNLEVBQUUsQ0FBQztnQkFDVCxVQUFVLEVBQUU7b0JBQ1YsSUFBSSxFQUFFLFlBQVk7b0JBQ2xCLE1BQU0sRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7b0JBQzlCLE9BQU8sRUFBRTt3QkFDUCxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLENBQUM7cUJBQ3BDO29CQUNELFdBQVcsRUFBRTt3QkFDWCxVQUFVLEVBQUUsQ0FBQztnQ0FDWCxLQUFLLEVBQUUsT0FBTztnQ0FDZCxLQUFLLEVBQUUsSUFBSTtnQ0FDWCxLQUFLLEVBQUUsaURBQWlEOzZCQUN6RCxDQUFDO3FCQUNIO29CQUNELEtBQUssRUFBRSxFQUFFO2lCQUNWO2FBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsb0JBQW9CLENBQUMsSUFBVTtRQUM3QixPQUFPO1FBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLE1BQU0sR0FBRyxJQUFJLGlCQUFXLENBQUM7WUFDN0IsS0FBSyxFQUFFLGdCQUFnQjtZQUN2QixJQUFJLEVBQUU7Z0JBQ0osSUFBSSxZQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxNQUFNLEVBQUUsQ0FBQzthQUNyRDtZQUNELEtBQUssRUFBRTtnQkFDTCxJQUFJLFlBQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLE1BQU0sRUFBRSxDQUFDO2FBQ3JEO1lBQ0QsU0FBUyxFQUFFLENBQUM7Z0JBQ1YsS0FBSyxFQUFFLFlBQVk7Z0JBQ25CLEdBQUcsRUFBRSxHQUFHO2FBQ1QsQ0FBQztZQUNGLFVBQVUsRUFBRSxDQUFDO2dCQUNYLEtBQUssRUFBRSxhQUFhO2dCQUNwQixHQUFHLEVBQUUsRUFBRTtnQkFDUCxTQUFTLEVBQUUsS0FBSzthQUNqQixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxDQUFDO2dCQUM5QyxJQUFJLEVBQUUsUUFBUTtnQkFDZCxLQUFLLEVBQUUsQ0FBQztnQkFDUixNQUFNLEVBQUUsQ0FBQztnQkFDVCxVQUFVLEVBQUU7b0JBQ1YsSUFBSSxFQUFFLFlBQVk7b0JBQ2xCLEtBQUssRUFBRSxnQkFBZ0I7b0JBQ3ZCLE1BQU0sRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7b0JBQzlCLE9BQU8sRUFBRTt3QkFDUCxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUM7d0JBQ2YsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxDQUFDO3FCQUNwQztvQkFDRCxLQUFLLEVBQUU7d0JBQ0wsSUFBSSxFQUFFLEVBQUUsS0FBSyxFQUFFLFlBQVksRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFO3dCQUN2QyxLQUFLLEVBQUUsRUFBRSxLQUFLLEVBQUUsYUFBYSxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRTtxQkFBRTtpQkFDL0Q7YUFDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1FBQzVDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sS0FBSyxHQUFHLFdBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSx1REFBdUQsQ0FBQyxDQUFDO1FBRTFHLE9BQU87UUFDUCxJQUFJLGlCQUFXLENBQUM7WUFDZCxLQUFLLEVBQUUsZ0JBQWdCO1lBQ3ZCLEtBQUs7U0FDTixDQUFDLENBQUM7UUFFSCxpQkFBaUI7UUFFakIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELCtDQUErQyxDQUFDLElBQVU7UUFDeEQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBRXBFLE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLHVCQUFpQixDQUFDO1lBQ25DLE9BQU8sRUFBRSxDQUFFLE1BQU0sQ0FBRTtZQUNuQixvQkFBb0IsRUFBRSxJQUFJO1NBQzNCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQztnQkFDOUMsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsS0FBSyxFQUFFLENBQUM7Z0JBQ1IsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxhQUFhO29CQUNuQixNQUFNLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29CQUM5QixPQUFPLEVBQUU7d0JBQ1AsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDO3FCQUNoQjtvQkFDRCxvQkFBb0IsRUFBRSxJQUFJO2lCQUMzQjthQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNEQUFzRCxDQUFDLElBQVU7UUFDL0QsTUFBTSxZQUFhLFNBQVEsWUFBTTtZQUN4QixjQUFjO2dCQUNuQixNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsY0FBYyxFQUFFLENBQUM7Z0JBQ25DLGFBQWE7Z0JBQ2IsR0FBRyxDQUFDLG1CQUFtQixDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUM7Z0JBQ3hDLE9BQU8sR0FBRyxDQUFDO1lBQ2IsQ0FBQztTQUNGO1FBRUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLE1BQU0sR0FBRyxJQUFJLGlCQUFXLENBQUM7WUFDN0IsSUFBSSxFQUFFO2dCQUNKLElBQUksWUFBWSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxVQUFVLEVBQUUsTUFBTSxFQUFFLENBQUM7YUFDM0Q7U0FDRixDQUFDLENBQUM7UUFFSCx1RUFBdUU7UUFDdkUsSUFBSSxDQUFDLFNBQVMsQ0FDWixLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQ3ZELENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUNwQyxDQUFDO1FBRUYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDREQUE0RCxDQUFDLElBQVU7UUFDckUsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBRXBFLE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLGlCQUFXLENBQUM7WUFDN0IsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQztvQkFDakIsS0FBSyxFQUFFLFdBQUssQ0FBQyxJQUFJO2lCQUNsQixDQUFDLENBQUM7WUFDSCxlQUFlLEVBQUU7Z0JBQ2YsRUFBRSxLQUFLLEVBQUUsV0FBSyxDQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFO2FBQ2pDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxDQUFFLENBQUMsQ0FBQztRQUNqSCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDeEksSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDZDQUE2QyxDQUFDLElBQVU7UUFDdEQsT0FBTztRQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxpQkFBVyxDQUFDO1lBQzdCLElBQUksRUFBRSxDQUFDLElBQUksWUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxVQUFVLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBRTtZQUM3RCxjQUFjLEVBQUUsb0JBQWMsQ0FBQyxLQUFLO1NBQ3JDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQztnQkFDOUMsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsS0FBSyxFQUFFLENBQUM7Z0JBQ1IsTUFBTSxFQUFFLENBQUM7Z0JBQ1QsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxZQUFZO29CQUNsQixNQUFNLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29CQUM5QixPQUFPLEVBQUU7d0JBQ1AsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDO3FCQUNoQjtvQkFDRCxLQUFLLEVBQUUsRUFBRTtvQkFDVCxNQUFNLEVBQUU7d0JBQ04sUUFBUSxFQUFFLE9BQU87cUJBQ2xCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgQWxhcm0sIEFsYXJtV2lkZ2V0LCBDb2xvciwgR3JhcGhXaWRnZXQsIExlZ2VuZFBvc2l0aW9uLCBMb2dRdWVyeVdpZGdldCwgTWV0cmljLCBTaGFkaW5nLCBTaW5nbGVWYWx1ZVdpZGdldCB9IGZyb20gJy4uL2xpYic7XG5cbmV4cG9ydCA9IHtcbiAgJ2FkZCBzdGFja2VkIHByb3BlcnR5IHRvIGdyYXBocycodGVzdDogVGVzdCkge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHdpZGdldCA9IG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICB0aXRsZTogJ1Rlc3Qgd2lkZ2V0JyxcbiAgICAgIHN0YWNrZWQ6IHRydWUsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh3aWRnZXQudG9Kc29uKCkpLCBbe1xuICAgICAgdHlwZTogJ21ldHJpYycsXG4gICAgICB3aWR0aDogNixcbiAgICAgIGhlaWdodDogNixcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgdmlldzogJ3RpbWVTZXJpZXMnLFxuICAgICAgICB0aXRsZTogJ1Rlc3Qgd2lkZ2V0JyxcbiAgICAgICAgcmVnaW9uOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICBzdGFja2VkOiB0cnVlLFxuICAgICAgICB5QXhpczoge30sXG4gICAgICB9LFxuICAgIH1dKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdhZGQgbWV0cmljcyB0byBncmFwaHMgb24gZWl0aGVyIGF4aXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB3aWRnZXQgPSBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgdGl0bGU6ICdNeSBmYW5jeSBncmFwaCcsXG4gICAgICBsZWZ0OiBbXG4gICAgICAgIG5ldyBNZXRyaWMoeyBuYW1lc3BhY2U6ICdDREsnLCBtZXRyaWNOYW1lOiAnVGVzdCcgfSksXG4gICAgICBdLFxuICAgICAgcmlnaHQ6IFtcbiAgICAgICAgbmV3IE1ldHJpYyh7IG5hbWVzcGFjZTogJ0NESycsIG1ldHJpY05hbWU6ICdUYXN0JyB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh3aWRnZXQudG9Kc29uKCkpLCBbe1xuICAgICAgdHlwZTogJ21ldHJpYycsXG4gICAgICB3aWR0aDogNixcbiAgICAgIGhlaWdodDogNixcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgdmlldzogJ3RpbWVTZXJpZXMnLFxuICAgICAgICB0aXRsZTogJ015IGZhbmN5IGdyYXBoJyxcbiAgICAgICAgcmVnaW9uOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICBtZXRyaWNzOiBbXG4gICAgICAgICAgWydDREsnLCAnVGVzdCddLFxuICAgICAgICAgIFsnQ0RLJywgJ1Rhc3QnLCB7IHlBeGlzOiAncmlnaHQnIH1dLFxuICAgICAgICBdLFxuICAgICAgICB5QXhpczoge30sXG4gICAgICB9LFxuICAgIH1dKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdsYWJlbCBhbmQgY29sb3IgYXJlIHJlc3BlY3RlZCBpbiBjb25zdHJ1Y3RvcicodGVzdDogVGVzdCkge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHdpZGdldCA9IG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICBsZWZ0OiBbbmV3IE1ldHJpYyh7IG5hbWVzcGFjZTogJ0NESycsIG1ldHJpY05hbWU6ICdUZXN0JywgbGFiZWw6ICdNeU1ldHJpYycsIGNvbG9yOiAnMDAwMDAwJyB9KSBdLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUod2lkZ2V0LnRvSnNvbigpKSwgW3tcbiAgICAgIHR5cGU6ICdtZXRyaWMnLFxuICAgICAgd2lkdGg6IDYsXG4gICAgICBoZWlnaHQ6IDYsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIHZpZXc6ICd0aW1lU2VyaWVzJyxcbiAgICAgICAgcmVnaW9uOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICBtZXRyaWNzOiBbXG4gICAgICAgICAgWydDREsnLCAnVGVzdCcsIHsgbGFiZWw6ICdNeU1ldHJpYycsIGNvbG9yOiAnMDAwMDAwJyB9XSxcbiAgICAgICAgXSxcbiAgICAgICAgeUF4aXM6IHt9LFxuICAgICAgfSxcbiAgICB9XSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc2luZ2xldmFsdWUgd2lkZ2V0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IG1ldHJpYyA9IG5ldyBNZXRyaWMoeyBuYW1lc3BhY2U6ICdDREsnLCBtZXRyaWNOYW1lOiAnVGVzdCcgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgd2lkZ2V0ID0gbmV3IFNpbmdsZVZhbHVlV2lkZ2V0KHtcbiAgICAgIG1ldHJpY3M6IFsgbWV0cmljIF0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh3aWRnZXQudG9Kc29uKCkpLCBbe1xuICAgICAgdHlwZTogJ21ldHJpYycsXG4gICAgICB3aWR0aDogNixcbiAgICAgIGhlaWdodDogMyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgdmlldzogJ3NpbmdsZVZhbHVlJyxcbiAgICAgICAgcmVnaW9uOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICBtZXRyaWNzOiBbXG4gICAgICAgICAgWydDREsnLCAnVGVzdCddLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9XSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAncXVlcnkgcmVzdWx0IHdpZGdldCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBsb2dHcm91cCA9IHtsb2dHcm91cE5hbWU6ICdteS1sb2ctZ3JvdXAnfTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB3aWRnZXQgPSBuZXcgTG9nUXVlcnlXaWRnZXQoe1xuICAgICAgbG9nR3JvdXBOYW1lczogW2xvZ0dyb3VwLmxvZ0dyb3VwTmFtZV0sXG4gICAgICBxdWVyeUxpbmVzOiBbXG4gICAgICAgICdmaWVsZHMgQG1lc3NhZ2UnLFxuICAgICAgICAnZmlsdGVyIEBtZXNzYWdlIGxpa2UgL0Vycm9yLycsXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUod2lkZ2V0LnRvSnNvbigpKSwgW3tcbiAgICAgIHR5cGU6ICdsb2cnLFxuICAgICAgd2lkdGg6IDYsXG4gICAgICBoZWlnaHQ6IDYsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIHZpZXc6ICd0YWJsZScsXG4gICAgICAgIHJlZ2lvbjogeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSxcbiAgICAgICAgcXVlcnk6IGBTT1VSQ0UgJyR7bG9nR3JvdXAubG9nR3JvdXBOYW1lfScgfCBmaWVsZHMgQG1lc3NhZ2VcXG58IGZpbHRlciBAbWVzc2FnZSBsaWtlIC9FcnJvci9gLFxuICAgICAgfSxcbiAgICB9XSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWxhcm0gd2lkZ2V0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgY29uc3QgYWxhcm0gPSBuZXcgTWV0cmljKHsgbmFtZXNwYWNlOiAnQ0RLJywgbWV0cmljTmFtZTogJ1Rlc3QnIH0pLmNyZWF0ZUFsYXJtKHN0YWNrLCAnQWxhcm0nLCB7XG4gICAgICBldmFsdWF0aW9uUGVyaW9kczogMixcbiAgICAgIHRocmVzaG9sZDogMTAwMCxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB3aWRnZXQgPSBuZXcgQWxhcm1XaWRnZXQoe1xuICAgICAgYWxhcm0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh3aWRnZXQudG9Kc29uKCkpLCBbe1xuICAgICAgdHlwZTogJ21ldHJpYycsXG4gICAgICB3aWR0aDogNixcbiAgICAgIGhlaWdodDogNixcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgdmlldzogJ3RpbWVTZXJpZXMnLFxuICAgICAgICByZWdpb246IHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sXG4gICAgICAgIGFubm90YXRpb25zOiB7XG4gICAgICAgICAgYWxhcm1zOiBbeyAnRm46OkdldEF0dCc6IFsgJ0FsYXJtNzEwM0Y0NjUnLCAnQXJuJyBdIH1dLFxuICAgICAgICB9LFxuICAgICAgICB5QXhpczoge30sXG4gICAgICB9LFxuICAgIH1dKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdhZGQgYW5ub3RhdGlvbnMgdG8gZ3JhcGgnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB3aWRnZXQgPSBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgdGl0bGU6ICdNeSBmYW5jeSBncmFwaCcsXG4gICAgICBsZWZ0OiBbXG4gICAgICAgIG5ldyBNZXRyaWMoeyBuYW1lc3BhY2U6ICdDREsnLCBtZXRyaWNOYW1lOiAnVGVzdCcgfSksXG4gICAgICBdLFxuICAgICAgbGVmdEFubm90YXRpb25zOiBbe1xuICAgICAgICB2YWx1ZTogMTAwMCxcbiAgICAgICAgY29sb3I6ICc2Njc3ODgnLFxuICAgICAgICBmaWxsOiBTaGFkaW5nLkJFTE9XLFxuICAgICAgICBsYWJlbDogJ3RoaXMgaXMgdGhlIGFubm90YXRpb24nLFxuICAgICAgfV0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh3aWRnZXQudG9Kc29uKCkpLCBbe1xuICAgICAgdHlwZTogJ21ldHJpYycsXG4gICAgICB3aWR0aDogNixcbiAgICAgIGhlaWdodDogNixcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgdmlldzogJ3RpbWVTZXJpZXMnLFxuICAgICAgICB0aXRsZTogJ015IGZhbmN5IGdyYXBoJyxcbiAgICAgICAgcmVnaW9uOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICBtZXRyaWNzOiBbXG4gICAgICAgICAgWydDREsnLCAnVGVzdCddLFxuICAgICAgICBdLFxuICAgICAgICBhbm5vdGF0aW9uczogeyBob3Jpem9udGFsOiBbe1xuICAgICAgICAgIHlBeGlzOiAnbGVmdCcsXG4gICAgICAgICAgdmFsdWU6IDEwMDAsXG4gICAgICAgICAgY29sb3I6ICc2Njc3ODgnLFxuICAgICAgICAgIGZpbGw6ICdiZWxvdycsXG4gICAgICAgICAgbGFiZWw6ICd0aGlzIGlzIHRoZSBhbm5vdGF0aW9uJyxcbiAgICAgICAgfV0gfSxcbiAgICAgICAgeUF4aXM6IHt9LFxuICAgICAgfSxcbiAgICB9XSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29udmVydCBhbGFybSB0byBhbm5vdGF0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgY29uc3QgbWV0cmljID0gbmV3IE1ldHJpYyh7IG5hbWVzcGFjZTogJ0NESycsIG1ldHJpY05hbWU6ICdUZXN0JyB9KTtcblxuICAgIGNvbnN0IGFsYXJtID0gbWV0cmljLmNyZWF0ZUFsYXJtKHN0YWNrLCAnQWxhcm0nLCB7XG4gICAgICBldmFsdWF0aW9uUGVyaW9kczogNyxcbiAgICAgIGRhdGFwb2ludHNUb0FsYXJtOiAyLFxuICAgICAgdGhyZXNob2xkOiAxMDAwLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHdpZGdldCA9IG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICByaWdodDogWyBtZXRyaWMgXSxcbiAgICAgIHJpZ2h0QW5ub3RhdGlvbnM6IFsgYWxhcm0udG9Bbm5vdGF0aW9uKCkgXSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHdpZGdldC50b0pzb24oKSksIFt7XG4gICAgICB0eXBlOiAnbWV0cmljJyxcbiAgICAgIHdpZHRoOiA2LFxuICAgICAgaGVpZ2h0OiA2LFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICB2aWV3OiAndGltZVNlcmllcycsXG4gICAgICAgIHJlZ2lvbjogeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSxcbiAgICAgICAgbWV0cmljczogW1xuICAgICAgICAgIFsnQ0RLJywgJ1Rlc3QnLCB7IHlBeGlzOiAncmlnaHQnIH1dLFxuICAgICAgICBdLFxuICAgICAgICBhbm5vdGF0aW9uczoge1xuICAgICAgICAgIGhvcml6b250YWw6IFt7XG4gICAgICAgICAgICB5QXhpczogJ3JpZ2h0JyxcbiAgICAgICAgICAgIHZhbHVlOiAxMDAwLFxuICAgICAgICAgICAgbGFiZWw6ICdUZXN0ID49IDEwMDAgZm9yIDIgZGF0YXBvaW50cyB3aXRoaW4gMzUgbWludXRlcycsXG4gICAgICAgICAgfV0sXG4gICAgICAgIH0sXG4gICAgICAgIHlBeGlzOiB7fSxcbiAgICAgIH0sXG4gICAgfV0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2FkZCB5QXhpcyB0byBncmFwaCcodGVzdDogVGVzdCkge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHdpZGdldCA9IG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICB0aXRsZTogJ015IGZhbmN5IGdyYXBoJyxcbiAgICAgIGxlZnQ6IFtcbiAgICAgICAgbmV3IE1ldHJpYyh7IG5hbWVzcGFjZTogJ0NESycsIG1ldHJpY05hbWU6ICdUZXN0JyB9KSxcbiAgICAgIF0sXG4gICAgICByaWdodDogW1xuICAgICAgICBuZXcgTWV0cmljKHsgbmFtZXNwYWNlOiAnQ0RLJywgbWV0cmljTmFtZTogJ1Rhc3QnIH0pLFxuICAgICAgXSxcbiAgICAgIGxlZnRZQXhpczogKHtcbiAgICAgICAgbGFiZWw6ICdMZWZ0IHlBeGlzJyxcbiAgICAgICAgbWF4OiAxMDAsXG4gICAgICB9KSxcbiAgICAgIHJpZ2h0WUF4aXM6ICh7XG4gICAgICAgIGxhYmVsOiAnUmlnaHQgeUF4aXMnLFxuICAgICAgICBtaW46IDEwLFxuICAgICAgICBzaG93VW5pdHM6IGZhbHNlLFxuICAgICAgfSksXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh3aWRnZXQudG9Kc29uKCkpLCBbe1xuICAgICAgdHlwZTogJ21ldHJpYycsXG4gICAgICB3aWR0aDogNixcbiAgICAgIGhlaWdodDogNixcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgdmlldzogJ3RpbWVTZXJpZXMnLFxuICAgICAgICB0aXRsZTogJ015IGZhbmN5IGdyYXBoJyxcbiAgICAgICAgcmVnaW9uOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICBtZXRyaWNzOiBbXG4gICAgICAgICAgWydDREsnLCAnVGVzdCddLFxuICAgICAgICAgIFsnQ0RLJywgJ1Rhc3QnLCB7IHlBeGlzOiAncmlnaHQnIH1dLFxuICAgICAgICBdLFxuICAgICAgICB5QXhpczoge1xuICAgICAgICAgIGxlZnQ6IHsgbGFiZWw6ICdMZWZ0IHlBeGlzJywgbWF4OiAxMDAgfSxcbiAgICAgICAgICByaWdodDogeyBsYWJlbDogJ1JpZ2h0IHlBeGlzJywgbWluOiAxMCwgc2hvd1VuaXRzOiBmYWxzZSB9IH0sXG4gICAgICB9LFxuICAgIH1dKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW4gdXNlIGltcG9ydGVkIGFsYXJtIHdpdGggZ3JhcGgnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgYWxhcm0gPSBBbGFybS5mcm9tQWxhcm1Bcm4oc3RhY2ssICdBbGFybScsICdhcm46YXdzOmNsb3Vkd2F0Y2g6cmVnaW9uOmFjY291bnQtaWQ6YWxhcm06YWxhcm0tbmFtZScpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBBbGFybVdpZGdldCh7XG4gICAgICB0aXRsZTogJ015IGZhbmN5IGdyYXBoJyxcbiAgICAgIGFsYXJtLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTjogQ29tcGlsZXNcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdhZGQgc2V0UGVyaW9kVG9UaW1lUmFuZ2UgdG8gc2luZ2xlVmFsdWVXaWRnZXQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgbWV0cmljID0gbmV3IE1ldHJpYyh7IG5hbWVzcGFjZTogJ0NESycsIG1ldHJpY05hbWU6ICdUZXN0JyB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB3aWRnZXQgPSBuZXcgU2luZ2xlVmFsdWVXaWRnZXQoe1xuICAgICAgbWV0cmljczogWyBtZXRyaWMgXSxcbiAgICAgIHNldFBlcmlvZFRvVGltZVJhbmdlOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUod2lkZ2V0LnRvSnNvbigpKSwgW3tcbiAgICAgIHR5cGU6ICdtZXRyaWMnLFxuICAgICAgd2lkdGg6IDYsXG4gICAgICBoZWlnaHQ6IDMsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIHZpZXc6ICdzaW5nbGVWYWx1ZScsXG4gICAgICAgIHJlZ2lvbjogeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSxcbiAgICAgICAgbWV0cmljczogW1xuICAgICAgICAgIFsnQ0RLJywgJ1Rlc3QnXSxcbiAgICAgICAgXSxcbiAgICAgICAgc2V0UGVyaW9kVG9UaW1lUmFuZ2U6IHRydWUsXG4gICAgICB9LFxuICAgIH1dKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdhbGxvd3Mgb3ZlcnJpZGluZyBjdXN0b20gdmFsdWVzIG9mIGRhc2hib2FyZCB3aWRnZXRzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY2xhc3MgSGlkZGVuTWV0cmljIGV4dGVuZHMgTWV0cmljIHtcbiAgICAgIHB1YmxpYyB0b01ldHJpY0NvbmZpZygpIHtcbiAgICAgICAgY29uc3QgcmV0ID0gc3VwZXIudG9NZXRyaWNDb25maWcoKTtcbiAgICAgICAgLy8gQHRzLWlnbm9yZVxuICAgICAgICByZXQucmVuZGVyaW5nUHJvcGVydGllcy52aXNpYmxlID0gZmFsc2U7XG4gICAgICAgIHJldHVybiByZXQ7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCB3aWRnZXQgPSBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgbGVmdDogW1xuICAgICAgICBuZXcgSGlkZGVuTWV0cmljKHsgbmFtZXNwYWNlOiAnQ0RLJywgbWV0cmljTmFtZTogJ1Rlc3QnIH0pLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIC8vIHRlc3Qub2sod2lkZ2V0LnRvSnNvbigpWzBdLnByb3BlcnRpZXMubWV0cmljc1swXS52aXNpYmxlID09PSBmYWxzZSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoXG4gICAgICBzdGFjay5yZXNvbHZlKHdpZGdldC50b0pzb24oKSlbMF0ucHJvcGVydGllcy5tZXRyaWNzWzBdLFxuICAgICAgWydDREsnLCAnVGVzdCcsIHsgdmlzaWJsZTogZmFsc2UgfV0sXG4gICAgKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdHcmFwaENvbG9yIGlzIGNvcnJlY3RseSBjb252ZXJ0ZWQgaW50byB0aGUgY29ycmVjdCBoZXhjb2RlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IG1ldHJpYyA9IG5ldyBNZXRyaWMoeyBuYW1lc3BhY2U6ICdDREsnLCBtZXRyaWNOYW1lOiAnVGVzdCcgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgd2lkZ2V0ID0gbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgIGxlZnQ6IFttZXRyaWMud2l0aCh7XG4gICAgICAgIGNvbG9yOiBDb2xvci5CTFVFLFxuICAgICAgfSldLFxuICAgICAgbGVmdEFubm90YXRpb25zOiBbXG4gICAgICAgIHsgY29sb3I6IENvbG9yLlJFRCwgdmFsdWU6IDEwMCB9LFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUod2lkZ2V0LnRvSnNvbigpKVswXS5wcm9wZXJ0aWVzLm1ldHJpY3NbMF0sIFsgJ0NESycsICdUZXN0JywgeyBjb2xvcjogJyMxZjc3YjQnIH0gXSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh3aWRnZXQudG9Kc29uKCkpWzBdLnByb3BlcnRpZXMuYW5ub3RhdGlvbnMuaG9yaXpvbnRhbFswXSwgeyB5QXhpczogJ2xlZnQnLCB2YWx1ZTogMTAwLCBjb2xvcjogJyNkNjI3MjgnIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdsZWdlbmQgcG9zaXRpb24gaXMgcmVzcGVjdGVkIGluIGNvbnN0cnVjdG9yJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3Qgd2lkZ2V0ID0gbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgIGxlZnQ6IFtuZXcgTWV0cmljKHsgbmFtZXNwYWNlOiAnQ0RLJywgbWV0cmljTmFtZTogJ1Rlc3QnIH0pIF0sXG4gICAgICBsZWdlbmRQb3NpdGlvbjogTGVnZW5kUG9zaXRpb24uUklHSFQsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZSh3aWRnZXQudG9Kc29uKCkpLCBbe1xuICAgICAgdHlwZTogJ21ldHJpYycsXG4gICAgICB3aWR0aDogNixcbiAgICAgIGhlaWdodDogNixcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgdmlldzogJ3RpbWVTZXJpZXMnLFxuICAgICAgICByZWdpb246IHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sXG4gICAgICAgIG1ldHJpY3M6IFtcbiAgICAgICAgICBbJ0NESycsICdUZXN0J10sXG4gICAgICAgIF0sXG4gICAgICAgIHlBeGlzOiB7fSxcbiAgICAgICAgbGVnZW5kOiB7XG4gICAgICAgICAgcG9zaXRpb246ICdyaWdodCcsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH1dKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxufTsiXX0=