"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const metric_types_1 = require("../metric-types");
/**
 * Parse a statistic, returning the type of metric that was used (simple or percentile)
 */
function parseStatistic(stat) {
    const lowerStat = stat.toLowerCase();
    // Simple statistics
    const statMap = {
        average: metric_types_1.Statistic.AVERAGE,
        avg: metric_types_1.Statistic.AVERAGE,
        minimum: metric_types_1.Statistic.MINIMUM,
        min: metric_types_1.Statistic.MINIMUM,
        maximum: metric_types_1.Statistic.MAXIMUM,
        max: metric_types_1.Statistic.MAXIMUM,
        samplecount: metric_types_1.Statistic.SAMPLE_COUNT,
        n: metric_types_1.Statistic.SAMPLE_COUNT,
        sum: metric_types_1.Statistic.SUM,
    };
    if (lowerStat in statMap) {
        return {
            type: 'simple',
            statistic: statMap[lowerStat],
        };
    }
    // Percentile statistics
    const re = /^p([\d.]+)$/;
    const m = re.exec(lowerStat);
    if (m) {
        return {
            type: 'percentile',
            percentile: parseFloat(m[1]),
        };
    }
    throw new Error(`Not a valid statistic: '${stat}', must be one of Average | Minimum | Maximum | SampleCount | Sum | pNN.NN`);
}
exports.parseStatistic = parseStatistic;
function normalizeStatistic(stat) {
    const parsed = parseStatistic(stat);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    else {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return stat.toLowerCase();
    }
}
exports.normalizeStatistic = normalizeStatistic;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGlzdGljLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic3RhdGlzdGljLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsa0RBQTRDO0FBVzVDOztHQUVHO0FBQ0gsU0FBZ0IsY0FBYyxDQUFDLElBQVk7SUFDekMsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBRXJDLG9CQUFvQjtJQUNwQixNQUFNLE9BQU8sR0FBNkI7UUFDeEMsT0FBTyxFQUFFLHdCQUFTLENBQUMsT0FBTztRQUMxQixHQUFHLEVBQUUsd0JBQVMsQ0FBQyxPQUFPO1FBQ3RCLE9BQU8sRUFBRSx3QkFBUyxDQUFDLE9BQU87UUFDMUIsR0FBRyxFQUFFLHdCQUFTLENBQUMsT0FBTztRQUN0QixPQUFPLEVBQUUsd0JBQVMsQ0FBQyxPQUFPO1FBQzFCLEdBQUcsRUFBRSx3QkFBUyxDQUFDLE9BQU87UUFDdEIsV0FBVyxFQUFFLHdCQUFTLENBQUMsWUFBWTtRQUNuQyxDQUFDLEVBQUUsd0JBQVMsQ0FBQyxZQUFZO1FBQ3pCLEdBQUcsRUFBRSx3QkFBUyxDQUFDLEdBQUc7S0FDbkIsQ0FBQztJQUVGLElBQUksU0FBUyxJQUFJLE9BQU8sRUFBRTtRQUN4QixPQUFPO1lBQ0wsSUFBSSxFQUFFLFFBQVE7WUFDZCxTQUFTLEVBQUUsT0FBTyxDQUFDLFNBQVMsQ0FBQztTQUM5QixDQUFDO0tBQ0g7SUFFRCx3QkFBd0I7SUFDeEIsTUFBTSxFQUFFLEdBQUcsYUFBYSxDQUFDO0lBQ3pCLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDN0IsSUFBSSxDQUFDLEVBQUU7UUFDTCxPQUFPO1lBQ0wsSUFBSSxFQUFFLFlBQVk7WUFDbEIsVUFBVSxFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDN0IsQ0FBQztLQUNIO0lBRUQsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsSUFBSSw0RUFBNEUsQ0FBQyxDQUFDO0FBQy9ILENBQUM7QUFsQ0Qsd0NBa0NDO0FBRUQsU0FBZ0Isa0JBQWtCLENBQUMsSUFBWTtJQUM3QyxNQUFNLE1BQU0sR0FBRyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDcEMsSUFBSSxNQUFNLENBQUMsSUFBSSxLQUFLLFFBQVEsRUFBRTtRQUM1QixPQUFPLE1BQU0sQ0FBQyxTQUFTLENBQUM7S0FDekI7U0FBTTtRQUNMLDhEQUE4RDtRQUM5RCxvRUFBb0U7UUFDcEUsT0FBTyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7S0FDM0I7QUFDSCxDQUFDO0FBVEQsZ0RBU0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBTdGF0aXN0aWMgfSBmcm9tICcuLi9tZXRyaWMtdHlwZXMnO1xuXG5leHBvcnQgaW50ZXJmYWNlIFNpbXBsZVN0YXRpc3RpYyB7XG4gIHR5cGU6ICdzaW1wbGUnO1xuICBzdGF0aXN0aWM6IFN0YXRpc3RpYztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQZXJjZW50aWxlU3RhdGlzdGljIHtcbiAgdHlwZTogJ3BlcmNlbnRpbGUnO1xuICBwZXJjZW50aWxlOiBudW1iZXI7XG59XG4vKipcbiAqIFBhcnNlIGEgc3RhdGlzdGljLCByZXR1cm5pbmcgdGhlIHR5cGUgb2YgbWV0cmljIHRoYXQgd2FzIHVzZWQgKHNpbXBsZSBvciBwZXJjZW50aWxlKVxuICovXG5leHBvcnQgZnVuY3Rpb24gcGFyc2VTdGF0aXN0aWMoc3RhdDogc3RyaW5nKTogU2ltcGxlU3RhdGlzdGljIHwgUGVyY2VudGlsZVN0YXRpc3RpYyB7XG4gIGNvbnN0IGxvd2VyU3RhdCA9IHN0YXQudG9Mb3dlckNhc2UoKTtcblxuICAvLyBTaW1wbGUgc3RhdGlzdGljc1xuICBjb25zdCBzdGF0TWFwOiB7W2s6IHN0cmluZ106IFN0YXRpc3RpY30gPSB7XG4gICAgYXZlcmFnZTogU3RhdGlzdGljLkFWRVJBR0UsXG4gICAgYXZnOiBTdGF0aXN0aWMuQVZFUkFHRSxcbiAgICBtaW5pbXVtOiBTdGF0aXN0aWMuTUlOSU1VTSxcbiAgICBtaW46IFN0YXRpc3RpYy5NSU5JTVVNLFxuICAgIG1heGltdW06IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgIG1heDogU3RhdGlzdGljLk1BWElNVU0sXG4gICAgc2FtcGxlY291bnQ6IFN0YXRpc3RpYy5TQU1QTEVfQ09VTlQsXG4gICAgbjogU3RhdGlzdGljLlNBTVBMRV9DT1VOVCxcbiAgICBzdW06IFN0YXRpc3RpYy5TVU0sXG4gIH07XG5cbiAgaWYgKGxvd2VyU3RhdCBpbiBzdGF0TWFwKSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHR5cGU6ICdzaW1wbGUnLFxuICAgICAgc3RhdGlzdGljOiBzdGF0TWFwW2xvd2VyU3RhdF0sXG4gICAgfTtcbiAgfVxuXG4gIC8vIFBlcmNlbnRpbGUgc3RhdGlzdGljc1xuICBjb25zdCByZSA9IC9ecChbXFxkLl0rKSQvO1xuICBjb25zdCBtID0gcmUuZXhlYyhsb3dlclN0YXQpO1xuICBpZiAobSkge1xuICAgIHJldHVybiB7XG4gICAgICB0eXBlOiAncGVyY2VudGlsZScsXG4gICAgICBwZXJjZW50aWxlOiBwYXJzZUZsb2F0KG1bMV0pLFxuICAgIH07XG4gIH1cblxuICB0aHJvdyBuZXcgRXJyb3IoYE5vdCBhIHZhbGlkIHN0YXRpc3RpYzogJyR7c3RhdH0nLCBtdXN0IGJlIG9uZSBvZiBBdmVyYWdlIHwgTWluaW11bSB8IE1heGltdW0gfCBTYW1wbGVDb3VudCB8IFN1bSB8IHBOTi5OTmApO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gbm9ybWFsaXplU3RhdGlzdGljKHN0YXQ6IHN0cmluZyk6IHN0cmluZyB7XG4gIGNvbnN0IHBhcnNlZCA9IHBhcnNlU3RhdGlzdGljKHN0YXQpO1xuICBpZiAocGFyc2VkLnR5cGUgPT09ICdzaW1wbGUnKSB7XG4gICAgcmV0dXJuIHBhcnNlZC5zdGF0aXN0aWM7XG4gIH0gZWxzZSB7XG4gICAgLy8gQWxyZWFkeSBwZXJjZW50aWxlLiBBdm9pZCBwYXJzaW5nIGJlY2F1c2Ugd2UgbWlnaHQgZ2V0IGludG9cbiAgICAvLyBmbG9hdGluZyBwb2ludCByb3VuZGluZyBpc3N1ZXMsIHJldHVybiBhcy1pcyBidXQgbG93ZXJjYXNlIHRoZSBwLlxuICAgIHJldHVybiBzdGF0LnRvTG93ZXJDYXNlKCk7XG4gIH1cbn0iXX0=