"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const kms = require("aws-cdk-lib/aws-kms");
const sqs = require("aws-cdk-lib/aws-sqs");
const ecs = require("aws-cdk-lib/aws-ecs");
const ec2 = require("aws-cdk-lib/aws-ec2");
test('New service/new queue, dlq, public API, new VPC', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = true;
    const clusterName = "custom-cluster-name";
    const containerName = "custom-container-name";
    const serviceName = "custom-service-name";
    const queueName = "custom-queue-name";
    const familyName = "family-name";
    const deadLetterQueueName = "dlqQueue";
    const construct = new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr('172.0.0.0/16') },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        queueProps: { queueName },
        deadLetterQueueProps: {
            queueName: deadLetterQueueName
        },
        queuePermissions: ['Read', 'Write']
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(construct.vpc !== null);
    expect(construct.service !== null);
    expect(construct.container !== null);
    expect(construct.sqsQueue !== null);
    expect(construct.deadLetterQueue !== null);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: serviceName
    });
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        Family: familyName
    });
    template.hasResourceProperties("AWS::ECS::Cluster", {
        ClusterName: clusterName
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: queueName
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: deadLetterQueueName
    });
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "sqs:ReceiveMessage",
                        "sqs:ChangeMessageVisibility",
                        "sqs:GetQueueUrl",
                        "sqs:DeleteMessage",
                        "sqs:GetQueueAttributes"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testconstructtestconstructqueue6D12C99B",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: [
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:GetQueueUrl"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testconstructtestconstructqueue6D12C99B",
                            "Arn"
                        ]
                    }
                },
            ],
        }
    });
    // Confirm we created a Public/Private VPC
    template.hasResourceProperties('AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::SQS::Queue', 2);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('New service/new queue, private API, new VPC', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = false;
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr('172.0.0.0/16') },
        deployDeadLetterQueue: false,
        queuePermissions: ['Read']
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "sqs:ReceiveMessage",
                        "sqs:ChangeMessageVisibility",
                        "sqs:GetQueueUrl",
                        "sqs:DeleteMessage",
                        "sqs:GetQueueAttributes"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testconstructtestconstructqueue6D12C99B",
                            "Arn"
                        ]
                    }
                }
            ],
        }
    });
    // Confirm we created an Isolated VPC
    defaults.expectNonexistence(stack, 'AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::SQS::Queue', 1);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('New service/existing fifo queue, private API, existing VPC', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = false;
    const queueName = 'custom-queue-name.fifo';
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    const existingQueue = new sqs.Queue(stack, 'MyQueue', {
        queueName,
        fifo: true
    });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi,
        existingVpc,
        existingQueueObj: existingQueue,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: queueName,
        FifoQueue: true
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: '172.168.0.0/16'
    });
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:GetQueueUrl"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "MyQueueE6CA6235",
                            "Arn"
                        ]
                    }
                },
            ],
        }
    });
    // Confirm we created an Isolated VPC
    defaults.expectNonexistence(stack, 'AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::SQS::Queue', 1);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('Existing service/new queue, public API, existing VPC', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = true;
    const serviceName = 'custom-name';
    const customVariableName = 'CUSTOM_NAME';
    const customArnName = 'CUSTOM_ARN_NAME';
    const queueName = 'testQueue';
    const dlqName = 'dlqQueue';
    const existingVpc = defaults.getTestVpc(stack);
    const createFargateServiceResponse = defaults.CreateFargateService(stack, 'test', existingVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { serviceName });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: createFargateServiceResponse.service,
        existingContainerDefinitionObject: createFargateServiceResponse.containerDefinition,
        existingVpc,
        queueUrlEnvironmentVariableName: customVariableName,
        queueArnEnvironmentVariableName: customArnName,
        queueProps: {
            queueName
        },
        deadLetterQueueProps: {
            queueName: dlqName
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: serviceName
    });
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: customArnName,
                        Value: {
                            "Fn::GetAtt": [
                                "testconstructtestconstructqueue6D12C99B",
                                "Arn"
                            ]
                        }
                    },
                    {
                        Name: customVariableName,
                        Value: {
                            Ref: "testconstructtestconstructqueue6D12C99B"
                        }
                    },
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: "test-container",
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: queueName
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: dlqName
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: '172.168.0.0/16'
    });
    // Confirm we created a Public/Private VPC
    template.hasResourceProperties('AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::SQS::Queue', 2);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
// Test existing service/existing queue, private API, new VPC
test('Existing service/existing queue, private API, existing VPC', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = false;
    const serviceName = 'custom-name';
    const queueName = 'custom-queue-name';
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    const createFargateServiceResponse = defaults.CreateFargateService(stack, 'test', existingVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { serviceName });
    const existingQueue = new sqs.Queue(stack, 'MyQueue', {
        queueName
    });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: createFargateServiceResponse.service,
        existingContainerDefinitionObject: createFargateServiceResponse.containerDefinition,
        existingVpc,
        existingQueueObj: existingQueue
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: serviceName,
    });
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: "SQS_QUEUE_ARN",
                        Value: {
                            "Fn::GetAtt": [
                                "MyQueueE6CA6235",
                                "Arn"
                            ]
                        }
                    },
                    {
                        Name: "SQS_QUEUE_URL",
                        Value: {
                            Ref: "MyQueueE6CA6235"
                        }
                    }
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: "test-container",
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: queueName
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: '172.168.0.0/16'
    });
    // Confirm we created an Isolated VPC
    defaults.expectNonexistence(stack, 'AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::SQS::Queue', 1);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('Test bad queuePermissions', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = false;
    const props = {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr('172.0.0.0/16') },
        deployDeadLetterQueue: false,
        queuePermissions: ['Reed'],
    };
    const app = () => {
        new lib_1.FargateToSqs(stack, 'test-construct', props);
    };
    expect(app).toThrowError('Invalid queue permission submitted - Reed');
});
test('Queue is encrypted with imported CMK when set on encryptionKey prop', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const cmk = new kms.Key(stack, 'cmk');
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        encryptionKey: cmk
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmk01DE03DA",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with imported CMK when set on queueProps.encryptionMasterKey prop', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const cmk = new kms.Key(stack, 'cmk');
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        queueProps: {
            encryptionMasterKey: cmk
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmk01DE03DA",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with provided encrytionKeyProps', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        encryptionKeyProps: {
            alias: 'new-key-alias-from-props'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    template.hasResourceProperties('AWS::KMS::Alias', {
        AliasName: 'alias/new-key-alias-from-props',
        TargetKeyId: {
            'Fn::GetAtt': [
                'testconstructEncryptionKey6153B053',
                'Arn'
            ]
        }
    });
});
test('Queue is encrypted with SQS-managed KMS key when no other encryption propreties are set', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
test('Queue is encrypted with customer managed KMS Key when enable encryption flag is true', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        enableEncryptionWithCustomerManagedKey: true
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testconstructEncryptionKey6153B053",
                "Arn"
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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