![](logo.svg)

# Project description

This is an application to analyze base pairing patterns of DNA/RNA 3D
structures to find and classify tetrads and quadruplexes. ElTetrado
assigns tetrads to one of the ONZ classes (O, N, Z) alongside with the
directionality of the tetrad (+/-) determined by the bonds between bases
and their non-canonical interactions. The interactions follow
Leontis/Westhof classification (Leontis *et al.* 2001). Watson-Crick (W)
edge of first base in the tetrad structure exposed to the Hoogsteen (H)
edge of the next nucleobase from the same tetrad sets the tetrad
directionality, clockwise (+) or anticlockwise (-). For more details,
please refer to Zok *et al.* (2020) and Popenda *et al.* (2020)

# Installation

Please run:

    pip install eltetrado

If you have both Python 2 and Python 3 installed, you need to explicitly
call `pip3`:

    pip3 install eltetrado

# Dependencies

The project is written in Python 3.6+ and requires
[mmcif](https://pypi.org/project/mmcif/),
[orjson](https://github.com/ijl/orjson), [NumPy](https://numpy.org/) and
[requests](https://docs.python-requests.org/en/latest/).

Visualization is created by `R` 3.6+ script which uses
[R4RNA](https://www.e-rna.org/r-chie/) (Lai *et al.* 2012) library. The
dependency will be automatically installed if not present.

Base pairs and stacking interactions are identified by
[BPNET](https://github.com/computational-biology/bpnet) (Roy *et al.*
2022) hosted as a service, so there is no need for installation or
configuration of this component.

# Usage

ElTetrado is a command line application, which requires to be provided
with `--input` and a path to a PDB or PDBx/mmCIF file.

By default, ElTetrado outputs textual results on the standard output. A
JSON version of the output can be obtained with `--output` switch
followed by a path where the file is supposed to be created.

ElTetrado prepares visualization of the whole structure and of each
N4-helices, quadruplexes and tetrads. This can be supplemented with
canonical base pairs visualization when `--complete-2d` is set. All
color settings are located in the first several lines of the `quadraw.R`
file, you can easily change them without knowledge of R language. If you
want ElTetrado to not visualize anything, pass `--no-image` switch to
it.

    usage: eltetrado [-h] [-i INPUT] [-o OUTPUT] [-m MODEL]
                     [--stacking-mismatch STACKING_MISMATCH] [--strict]
                     [--no-reorder] [--complete-2d] [--no-image] [--version]

    options:
      -h, --help            show this help message and exit
      -i INPUT, --input INPUT
                            path to input PDB or PDBx/mmCIF file
      -o OUTPUT, --output OUTPUT
                            (optional) path for output JSON file
      -m MODEL, --model MODEL
                            (optional) model number to process
      --stacking-mismatch STACKING_MISMATCH
                            a perfect tetrad stacking covers 4 nucleotides; this
                            option can be used with value 1 or 2 to allow this
                            number of nucleotides to be non-stacked with otherwise
                            well aligned tetrad [default=2]
      --strict              nucleotides in tetrad are found when linked only by
                            cWH pairing
      --no-reorder          chains of bi- and tetramolecular quadruplexes should
                            be reordered to be able to have them classified; when
                            this is set, chains will be processed in original
                            order, which for bi-/tetramolecular means that they
                            will likely be misclassified; use with care!
      --complete-2d         when set, the visualization will also show canonical
                            base pairs to provide context for the quadruplex
      --no-image            when set, the visualization will not be created at all
      --version             show program's version number and exit

# Chains reorder

ElTetrado keeps a global and unique 5’-3’ index for every nucleotide
which is independent from residue numbers. For example, if a structure
has chain M with 60 nucleotides and chain N with 15 nucleotides, then
ElTetrado will keep index between 0 and 74 which uniquely identifies
every nucleotide. Initially, ElTetrado assigns this indices according to
the order of chains in the input file. Therefore, if M preceded N then
nucleotides in M will be indexed from 0 to 59 and in N from 60 to 74.
Otherwise, nucleotides in N will be indexed from 0 to 14 and in M from
15 to 74.

When `--no-reorder` is present, this initial assignment is used.
Otherwise, ElTetrado exhaustively checks all permutations of chains’
orders. Every permutation check induces recalculation of the global and
unique 5’-3’ index and in effect it changes ONZ classification of
tetrads.

ElTetrado keeps a table of tetrad classification scores according to
these rules:

-   Type preference: `O` \> `N` \> `Z`
-   Direction preference: `+` \> `-`

The table keeps low values for preferred classes i.e. `O+` is 0, `O-` is
1 and so on up to `Z-` with score 5. For every permutation of chain
orders, ElTetrado computes sum of scores for tetrads classification
induced by 5’-3’ indexing. We select permutation with the minimum value.

# Examples

## 2HY9: Human telomere DNA quadruplex structure in K+ solution hybrid-1 form

![](2hy9.png)

    $ curl ftp://ftp.wwpdb.org/pub/pdb/data/structures/divided/mmCIF/my/2hy9.cif.gz | gzip -d > 2hy9.cif
    $ ./eltetrado --input 2hy9.cif --output 2hy9.json

    Chain order: 1
    n4-helix with 3 tetrads
      Oh* V,VI 9a -(pll) quadruplex with 3 tetrads
        1.DG4 1.DG22 1.DG18 1.DG10 cWH cWH cWH cWH O- Vb planarity=0.17  
          direction=hybrid rise=3.21 twist=16.23
        1.DG5 1.DG23 1.DG17 1.DG11 cHW cHW cHW cHW O+ Va planarity=0.1  
          direction=hybrid rise=3.11 twist=27.45
        1.DG6 1.DG24 1.DG16 1.DG12 cHW cHW cHW cHW O+ VIa planarity=0.18  

        Tracts:
          1.DG4, 1.DG5, 1.DG6
          1.DG22, 1.DG23, 1.DG24
          1.DG18, 1.DG17, 1.DG16
          1.DG10, 1.DG11, 1.DG12

        Loops:
          propeller- 1.DT7, 1.DT8, 1.DA9
          lateral- 1.DT13, 1.DT14, 1.DA15
          lateral+ 1.DT19, 1.DT20, 1.DA21

    AAAGGGTTAGGGTTAGGGTTAGGGAA
    ...([{...(((...)))...)]}..
    ...([{...)]}...(((...)))..

<details>
<summary>
Click to see the output JSON
</summary>

``` json
{
  "metals": [],
  "nucleotides": [
    {
      "index": 1,
      "chain": "1",
      "number": 1,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DA1",
      "shortName": "A",
      "chi": 22.308282830857802,
      "glycosidicBond": "syn"
    },
    {
      "index": 2,
      "chain": "1",
      "number": 2,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DA2",
      "shortName": "A",
      "chi": -123.05454402191421,
      "glycosidicBond": "anti"
    },
    {
      "index": 3,
      "chain": "1",
      "number": 3,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DA3",
      "shortName": "A",
      "chi": -94.96579955603106,
      "glycosidicBond": "anti"
    },
    {
      "index": 4,
      "chain": "1",
      "number": 4,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG4",
      "shortName": "G",
      "chi": 79.28363721639316,
      "glycosidicBond": "syn"
    },
    {
      "index": 5,
      "chain": "1",
      "number": 5,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG5",
      "shortName": "G",
      "chi": -126.01709201555563,
      "glycosidicBond": "anti"
    },
    {
      "index": 6,
      "chain": "1",
      "number": 6,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG6",
      "shortName": "G",
      "chi": -127.26656202302102,
      "glycosidicBond": "anti"
    },
    {
      "index": 7,
      "chain": "1",
      "number": 7,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DT7",
      "shortName": "T",
      "chi": -63.10830751967371,
      "glycosidicBond": "syn"
    },
    {
      "index": 8,
      "chain": "1",
      "number": 8,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DT8",
      "shortName": "T",
      "chi": -138.79520345559828,
      "glycosidicBond": "anti"
    },
    {
      "index": 9,
      "chain": "1",
      "number": 9,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DA9",
      "shortName": "A",
      "chi": -148.83990757408878,
      "glycosidicBond": "anti"
    },
    {
      "index": 10,
      "chain": "1",
      "number": 10,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG10",
      "shortName": "G",
      "chi": 58.7787525019158,
      "glycosidicBond": "syn"
    },
    {
      "index": 11,
      "chain": "1",
      "number": 11,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG11",
      "shortName": "G",
      "chi": -123.85746807924986,
      "glycosidicBond": "anti"
    },
    {
      "index": 12,
      "chain": "1",
      "number": 12,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG12",
      "shortName": "G",
      "chi": -84.36679807284759,
      "glycosidicBond": "syn"
    },
    {
      "index": 13,
      "chain": "1",
      "number": 13,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DT13",
      "shortName": "T",
      "chi": -30.819029132834157,
      "glycosidicBond": "syn"
    },
    {
      "index": 14,
      "chain": "1",
      "number": 14,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DT14",
      "shortName": "T",
      "chi": -168.51776782812965,
      "glycosidicBond": "anti"
    },
    {
      "index": 15,
      "chain": "1",
      "number": 15,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DA15",
      "shortName": "A",
      "chi": -105.72881577106517,
      "glycosidicBond": "anti"
    },
    {
      "index": 16,
      "chain": "1",
      "number": 16,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG16",
      "shortName": "G",
      "chi": 74.3227942181243,
      "glycosidicBond": "syn"
    },
    {
      "index": 17,
      "chain": "1",
      "number": 17,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG17",
      "shortName": "G",
      "chi": 81.08424926936044,
      "glycosidicBond": "syn"
    },
    {
      "index": 18,
      "chain": "1",
      "number": 18,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG18",
      "shortName": "G",
      "chi": -122.90397217111551,
      "glycosidicBond": "anti"
    },
    {
      "index": 19,
      "chain": "1",
      "number": 19,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DT19",
      "shortName": "T",
      "chi": -102.98239337113938,
      "glycosidicBond": "anti"
    },
    {
      "index": 20,
      "chain": "1",
      "number": 20,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DT20",
      "shortName": "T",
      "chi": -112.1514601849715,
      "glycosidicBond": "anti"
    },
    {
      "index": 21,
      "chain": "1",
      "number": 21,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DA21",
      "shortName": "A",
      "chi": -89.07113063649612,
      "glycosidicBond": "syn"
    },
    {
      "index": 22,
      "chain": "1",
      "number": 22,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG22",
      "shortName": "G",
      "chi": 83.44318693001902,
      "glycosidicBond": "syn"
    },
    {
      "index": 23,
      "chain": "1",
      "number": 23,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG23",
      "shortName": "G",
      "chi": -115.41210237198398,
      "glycosidicBond": "anti"
    },
    {
      "index": 24,
      "chain": "1",
      "number": 24,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DG24",
      "shortName": "G",
      "chi": -111.14845782593531,
      "glycosidicBond": "anti"
    },
    {
      "index": 25,
      "chain": "1",
      "number": 25,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DA25",
      "shortName": "A",
      "chi": -58.323530637551954,
      "glycosidicBond": "syn"
    },
    {
      "index": 26,
      "chain": "1",
      "number": 26,
      "icode": null,
      "molecule": "DNA",
      "fullName": "1.DA26",
      "shortName": "A",
      "chi": -90.84065243137135,
      "glycosidicBond": "anti"
    }
  ],
  "basePairs": [
    {
      "nt1": "1.DA3",
      "nt2": "1.DA9",
      "lw": "tSW"
    },
    {
      "nt1": "1.DG4",
      "nt2": "1.DG10",
      "lw": "cHW"
    },
    {
      "nt1": "1.DG4",
      "nt2": "1.DG22",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG5",
      "nt2": "1.DG11",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG5",
      "nt2": "1.DG23",
      "lw": "cHW"
    },
    {
      "nt1": "1.DG6",
      "nt2": "1.DG12",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG6",
      "nt2": "1.DG24",
      "lw": "cHW"
    },
    {
      "nt1": "1.DA9",
      "nt2": "1.DA3",
      "lw": "tWS"
    },
    {
      "nt1": "1.DG10",
      "nt2": "1.DG4",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG10",
      "nt2": "1.DG18",
      "lw": "cHW"
    },
    {
      "nt1": "1.DG11",
      "nt2": "1.DG5",
      "lw": "cHW"
    },
    {
      "nt1": "1.DG11",
      "nt2": "1.DG17",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG12",
      "nt2": "1.DG6",
      "lw": "cHW"
    },
    {
      "nt1": "1.DG12",
      "nt2": "1.DG16",
      "lw": "cWH"
    },
    {
      "nt1": "1.DT14",
      "nt2": "1.DA25",
      "lw": "tWW"
    },
    {
      "nt1": "1.DG16",
      "nt2": "1.DG12",
      "lw": "cHW"
    },
    {
      "nt1": "1.DG16",
      "nt2": "1.DG24",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG17",
      "nt2": "1.DG11",
      "lw": "cHW"
    },
    {
      "nt1": "1.DG17",
      "nt2": "1.DG23",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG18",
      "nt2": "1.DG10",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG18",
      "nt2": "1.DG22",
      "lw": "cHW"
    },
    {
      "nt1": "1.DG22",
      "nt2": "1.DG4",
      "lw": "cHW"
    },
    {
      "nt1": "1.DG22",
      "nt2": "1.DG18",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG23",
      "nt2": "1.DG5",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG23",
      "nt2": "1.DG17",
      "lw": "cHW"
    },
    {
      "nt1": "1.DG24",
      "nt2": "1.DG6",
      "lw": "cWH"
    },
    {
      "nt1": "1.DG24",
      "nt2": "1.DG16",
      "lw": "cHW"
    },
    {
      "nt1": "1.DA25",
      "nt2": "1.DT14",
      "lw": "tWW"
    }
  ],
  "helices": [
    {
      "quadruplexes": [
        {
          "tetrads": [
            {
              "id": "1.DG4-1.DG22-1.DG18-1.DG10",
              "nt1": "1.DG4",
              "nt2": "1.DG22",
              "nt3": "1.DG18",
              "nt4": "1.DG10",
              "onz": "O-",
              "gbaClassification": "Vb",
              "planarityDeviation": 0.17372283960377805,
              "ionsChannel": [],
              "ionsOutside": []
            },
            {
              "id": "1.DG5-1.DG23-1.DG17-1.DG11",
              "nt1": "1.DG5",
              "nt2": "1.DG23",
              "nt3": "1.DG17",
              "nt4": "1.DG11",
              "onz": "O+",
              "gbaClassification": "Va",
              "planarityDeviation": 0.10474313820007483,
              "ionsChannel": [],
              "ionsOutside": []
            },
            {
              "id": "1.DG6-1.DG24-1.DG16-1.DG12",
              "nt1": "1.DG6",
              "nt2": "1.DG24",
              "nt3": "1.DG16",
              "nt4": "1.DG12",
              "onz": "O+",
              "gbaClassification": "VIa",
              "planarityDeviation": 0.18293509778060615,
              "ionsChannel": [],
              "ionsOutside": []
            }
          ],
          "onzm": "Oh*",
          "loopClassification": {
            "classification": "9a",
            "loop_progression": "-(pll)"
          },
          "gbaClassification": [
            "V",
            "VI"
          ],
          "tracts": [
            [
              "1.DG4",
              "1.DG5",
              "1.DG6"
            ],
            [
              "1.DG22",
              "1.DG23",
              "1.DG24"
            ],
            [
              "1.DG18",
              "1.DG17",
              "1.DG16"
            ],
            [
              "1.DG10",
              "1.DG11",
              "1.DG12"
            ]
          ],
          "loops": [
            {
              "type": "propeller-",
              "nucleotides": [
                "1.DT7",
                "1.DT8",
                "1.DA9"
              ]
            },
            {
              "type": "lateral-",
              "nucleotides": [
                "1.DT13",
                "1.DT14",
                "1.DA15"
              ]
            },
            {
              "type": "lateral+",
              "nucleotides": [
                "1.DT19",
                "1.DT20",
                "1.DA21"
              ]
            }
          ]
        }
      ],
      "tetradPairs": [
        {
          "tetrad1": "1.DG4-1.DG22-1.DG18-1.DG10",
          "tetrad2": "1.DG5-1.DG23-1.DG17-1.DG11",
          "direction": "hybrid",
          "rise": 3.2109650905140654,
          "twist": 16.228973729066034
        },
        {
          "tetrad1": "1.DG5-1.DG23-1.DG17-1.DG11",
          "tetrad2": "1.DG6-1.DG24-1.DG16-1.DG12",
          "direction": "hybrid",
          "rise": 3.1149939255558747,
          "twist": 27.448958336697046
        }
      ]
    }
  ],
  "dotBracket": {
    "sequence": "AAAGGGTTAGGGTTAGGGTTAGGGAA",
    "line1": "...([{...(((...)))...)]}..",
    "line2": "...([{...)]}...(((...))).."
  }
}
```

</details>

## 4RJ1: Structural variations and solvent structure of UGGGGU quadruplexes stabilized by Sr2+ ions

![](4rj1.png)

    $ curl https://www.ebi.ac.uk/pdbe/static/entry/download/4rj1-assembly-1.cif.gz | gzip -d > 4rj1-1.cif
    $ ./eltetrado --input 4rj1-1.cif --output 4rj1-1.json

    Chain order: A AB AA AC B BC BA BB
    n4-helix with 10 tetrads
      Op* VIII n/a quadruplex with 5 tetrads
        A.U1006 AC.U1006 AA.U1006 AB.U1006 cWH cWH cWH cWH O- VIIIa planarity=1.06 ions_channel=NA ions_outside=A.U1006: [SR] AA.U1006: [SR] AB.U1006: [SR] AC.U1006: [SR]
          direction=parallel rise=3.37 twist=39.96
        A.G1005 AC.G1005 AA.G1005 AB.G1005 cHW cHW cHW cHW O+ VIIIa planarity=0.8  
          direction=parallel rise=3.31 twist=25.9
        A.G1004 AC.G1004 AA.G1004 AB.G1004 cHW cHW cHW cHW O+ VIIIa planarity=0.41 ions_channel=SR 
          direction=parallel rise=3.34 twist=35.81
        A.G1003 AC.G1003 AA.G1003 AB.G1003 cHW cHW cHW cHW O+ VIIIa planarity=0.55 ions_channel=SR 
          direction=parallel rise=3.29 twist=27.12
        A.G1002 AC.G1002 AA.G1002 AB.G1002 cHW cHW cHW cHW O+ VIIIa planarity=0.54  ions_outside=AB.G1002: [CA] AC.G1002: [CA] AA.G1002: [CA] A.G1002: [CA]

        Tracts:
          A.U1006, A.G1005, A.G1004, A.G1003, A.G1002
          AC.U1006, AC.G1005, AC.G1004, AC.G1003, AC.G1002
          AA.U1006, AA.G1005, AA.G1004, AA.G1003, AA.G1002
          AB.U1006, AB.G1005, AB.G1004, AB.G1003, AB.G1002

      Op* VIII n/a quadruplex with 5 tetrads
        B.G2002 BC.G2002 BA.G2002 BB.G2002 cWH cWH cWH cWH O+ VIIIa planarity=0.67  
          direction=parallel rise=3.37 twist=27.41
        B.G2003 BC.G2003 BA.G2003 BB.G2003 cWH cWH cWH cWH O+ VIIIa planarity=0.58 ions_channel=SR ions_outside=B.G2003: [CA] BA.G2003: [CA] BB.G2003: [CA] BC.G2003: [CA]
          direction=parallel rise=3.32 twist=35.04
        B.G2004 BC.G2004 BA.G2004 BB.G2004 cWH cWH cWH cWH O+ VIIIa planarity=0.23 ions_channel=SR 
          direction=parallel rise=3.27 twist=25.15
        B.G2005 BC.G2005 BA.G2005 BB.G2005 cWH cWH cWH cWH O+ VIIIa planarity=0.78 ions_channel=NA 
          direction=parallel rise=7.14 twist=43.41
        B.U2006 BC.U2006 BA.U2006 BB.U2006 cHW cHW cHW cHW O- VIIIa planarity=1.58 ions_channel=NA,NA 

        Tracts:
          B.G2002, B.G2003, B.G2004, B.G2005, B.U2006
          BC.G2002, BC.G2003, BC.G2004, BC.G2005, BC.U2006
          BA.G2002, BA.G2003, BA.G2004, BA.G2005, BA.U2006
          BB.G2002, BB.G2003, BB.G2004, BB.G2005, BB.U2006

    UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU
    .([{<A-.([{<A-.)]}>a-.)]}>a-.([{<A-.)]}>a-.([{<A-.)]}>a
    .([{<A-.)]}>a-.([{<A-.)]}>a-.([{<A-.([{<A-.)]}>a-.)]}>a

<details>
<summary>
Click to see the output JSON
</summary>

``` json
{
  "metals": [
    {
      "symbol": "Sr",
      "count": 8
    },
    {
      "symbol": "Na",
      "count": 4
    },
    {
      "symbol": "Ca",
      "count": 12
    }
  ],
  "nucleotides": [
    {
      "index": 1,
      "chain": "A",
      "number": 1001,
      "icode": null,
      "molecule": "RNA",
      "fullName": "A.U1001",
      "shortName": "U",
      "chi": -141.92671313255752,
      "glycosidicBond": "anti"
    },
    {
      "index": 2,
      "chain": "A",
      "number": 1002,
      "icode": null,
      "molecule": "RNA",
      "fullName": "A.G1002",
      "shortName": "G",
      "chi": -165.93034671112116,
      "glycosidicBond": "anti"
    },
    {
      "index": 3,
      "chain": "A",
      "number": 1003,
      "icode": null,
      "molecule": "RNA",
      "fullName": "A.G1003",
      "shortName": "G",
      "chi": -121.5652426033226,
      "glycosidicBond": "anti"
    },
    {
      "index": 4,
      "chain": "A",
      "number": 1004,
      "icode": null,
      "molecule": "RNA",
      "fullName": "A.G1004",
      "shortName": "G",
      "chi": -156.00957673923344,
      "glycosidicBond": "anti"
    },
    {
      "index": 5,
      "chain": "A",
      "number": 1005,
      "icode": null,
      "molecule": "RNA",
      "fullName": "A.G1005",
      "shortName": "G",
      "chi": -148.10051684016415,
      "glycosidicBond": "anti"
    },
    {
      "index": 6,
      "chain": "A",
      "number": 1006,
      "icode": null,
      "molecule": "RNA",
      "fullName": "A.U1006",
      "shortName": "U",
      "chi": -137.28005568139983,
      "glycosidicBond": "anti"
    },
    {
      "index": 13,
      "chain": "AA",
      "number": 1001,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AA.U1001",
      "shortName": "U",
      "chi": -141.9267131325575,
      "glycosidicBond": "anti"
    },
    {
      "index": 14,
      "chain": "AA",
      "number": 1002,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AA.G1002",
      "shortName": "G",
      "chi": -165.93034671112113,
      "glycosidicBond": "anti"
    },
    {
      "index": 15,
      "chain": "AA",
      "number": 1003,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AA.G1003",
      "shortName": "G",
      "chi": -121.56524260332266,
      "glycosidicBond": "anti"
    },
    {
      "index": 16,
      "chain": "AA",
      "number": 1004,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AA.G1004",
      "shortName": "G",
      "chi": -156.0095767392335,
      "glycosidicBond": "anti"
    },
    {
      "index": 17,
      "chain": "AA",
      "number": 1005,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AA.G1005",
      "shortName": "G",
      "chi": -148.10051684016406,
      "glycosidicBond": "anti"
    },
    {
      "index": 18,
      "chain": "AA",
      "number": 1006,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AA.U1006",
      "shortName": "U",
      "chi": -137.2800556813998,
      "glycosidicBond": "anti"
    },
    {
      "index": 7,
      "chain": "AB",
      "number": 1001,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AB.U1001",
      "shortName": "U",
      "chi": -141.9267131325574,
      "glycosidicBond": "anti"
    },
    {
      "index": 8,
      "chain": "AB",
      "number": 1002,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AB.G1002",
      "shortName": "G",
      "chi": -165.93034671112113,
      "glycosidicBond": "anti"
    },
    {
      "index": 9,
      "chain": "AB",
      "number": 1003,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AB.G1003",
      "shortName": "G",
      "chi": -121.56524260332266,
      "glycosidicBond": "anti"
    },
    {
      "index": 10,
      "chain": "AB",
      "number": 1004,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AB.G1004",
      "shortName": "G",
      "chi": -156.00957673923347,
      "glycosidicBond": "anti"
    },
    {
      "index": 11,
      "chain": "AB",
      "number": 1005,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AB.G1005",
      "shortName": "G",
      "chi": -148.10051684016406,
      "glycosidicBond": "anti"
    },
    {
      "index": 12,
      "chain": "AB",
      "number": 1006,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AB.U1006",
      "shortName": "U",
      "chi": -137.28005568139977,
      "glycosidicBond": "anti"
    },
    {
      "index": 19,
      "chain": "AC",
      "number": 1001,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AC.U1001",
      "shortName": "U",
      "chi": -141.92671313255747,
      "glycosidicBond": "anti"
    },
    {
      "index": 20,
      "chain": "AC",
      "number": 1002,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AC.G1002",
      "shortName": "G",
      "chi": -165.93034671112116,
      "glycosidicBond": "anti"
    },
    {
      "index": 21,
      "chain": "AC",
      "number": 1003,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AC.G1003",
      "shortName": "G",
      "chi": -121.56524260332266,
      "glycosidicBond": "anti"
    },
    {
      "index": 22,
      "chain": "AC",
      "number": 1004,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AC.G1004",
      "shortName": "G",
      "chi": -156.00957673923352,
      "glycosidicBond": "anti"
    },
    {
      "index": 23,
      "chain": "AC",
      "number": 1005,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AC.G1005",
      "shortName": "G",
      "chi": -148.1005168401641,
      "glycosidicBond": "anti"
    },
    {
      "index": 24,
      "chain": "AC",
      "number": 1006,
      "icode": null,
      "molecule": "RNA",
      "fullName": "AC.U1006",
      "shortName": "U",
      "chi": -137.28005568139986,
      "glycosidicBond": "anti"
    },
    {
      "index": 25,
      "chain": "B",
      "number": 2001,
      "icode": null,
      "molecule": "RNA",
      "fullName": "B.U2001",
      "shortName": "U",
      "chi": -146.4615316869476,
      "glycosidicBond": "anti"
    },
    {
      "index": 26,
      "chain": "B",
      "number": 2002,
      "icode": null,
      "molecule": "RNA",
      "fullName": "B.G2002",
      "shortName": "G",
      "chi": -170.79660912745996,
      "glycosidicBond": "anti"
    },
    {
      "index": 27,
      "chain": "B",
      "number": 2003,
      "icode": null,
      "molecule": "RNA",
      "fullName": "B.G2003",
      "shortName": "G",
      "chi": -117.68718110874113,
      "glycosidicBond": "anti"
    },
    {
      "index": 28,
      "chain": "B",
      "number": 2004,
      "icode": null,
      "molecule": "RNA",
      "fullName": "B.G2004",
      "shortName": "G",
      "chi": -153.88587375071324,
      "glycosidicBond": "anti"
    },
    {
      "index": 29,
      "chain": "B",
      "number": 2005,
      "icode": null,
      "molecule": "RNA",
      "fullName": "B.G2005",
      "shortName": "G",
      "chi": -148.8519912845669,
      "glycosidicBond": "anti"
    },
    {
      "index": 30,
      "chain": "B",
      "number": 2006,
      "icode": null,
      "molecule": "RNA",
      "fullName": "B.U2006",
      "shortName": "U",
      "chi": -159.43730655241544,
      "glycosidicBond": "anti"
    },
    {
      "index": 37,
      "chain": "BA",
      "number": 2001,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BA.U2001",
      "shortName": "U",
      "chi": -146.46153168694764,
      "glycosidicBond": "anti"
    },
    {
      "index": 38,
      "chain": "BA",
      "number": 2002,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BA.G2002",
      "shortName": "G",
      "chi": -170.79660912745993,
      "glycosidicBond": "anti"
    },
    {
      "index": 39,
      "chain": "BA",
      "number": 2003,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BA.G2003",
      "shortName": "G",
      "chi": -117.68718110874113,
      "glycosidicBond": "anti"
    },
    {
      "index": 40,
      "chain": "BA",
      "number": 2004,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BA.G2004",
      "shortName": "G",
      "chi": -153.88587375071322,
      "glycosidicBond": "anti"
    },
    {
      "index": 41,
      "chain": "BA",
      "number": 2005,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BA.G2005",
      "shortName": "G",
      "chi": -148.851991284567,
      "glycosidicBond": "anti"
    },
    {
      "index": 42,
      "chain": "BA",
      "number": 2006,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BA.U2006",
      "shortName": "U",
      "chi": -159.43730655241544,
      "glycosidicBond": "anti"
    },
    {
      "index": 43,
      "chain": "BB",
      "number": 2001,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BB.U2001",
      "shortName": "U",
      "chi": -146.4615316869476,
      "glycosidicBond": "anti"
    },
    {
      "index": 44,
      "chain": "BB",
      "number": 2002,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BB.G2002",
      "shortName": "G",
      "chi": -170.79660912745993,
      "glycosidicBond": "anti"
    },
    {
      "index": 45,
      "chain": "BB",
      "number": 2003,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BB.G2003",
      "shortName": "G",
      "chi": -117.68718110874106,
      "glycosidicBond": "anti"
    },
    {
      "index": 46,
      "chain": "BB",
      "number": 2004,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BB.G2004",
      "shortName": "G",
      "chi": -153.8858737507132,
      "glycosidicBond": "anti"
    },
    {
      "index": 47,
      "chain": "BB",
      "number": 2005,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BB.G2005",
      "shortName": "G",
      "chi": -148.85199128456696,
      "glycosidicBond": "anti"
    },
    {
      "index": 48,
      "chain": "BB",
      "number": 2006,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BB.U2006",
      "shortName": "U",
      "chi": -159.43730655241544,
      "glycosidicBond": "anti"
    },
    {
      "index": 31,
      "chain": "BC",
      "number": 2001,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BC.U2001",
      "shortName": "U",
      "chi": -146.4615316869476,
      "glycosidicBond": "anti"
    },
    {
      "index": 32,
      "chain": "BC",
      "number": 2002,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BC.G2002",
      "shortName": "G",
      "chi": -170.79660912745993,
      "glycosidicBond": "anti"
    },
    {
      "index": 33,
      "chain": "BC",
      "number": 2003,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BC.G2003",
      "shortName": "G",
      "chi": -117.68718110874121,
      "glycosidicBond": "anti"
    },
    {
      "index": 34,
      "chain": "BC",
      "number": 2004,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BC.G2004",
      "shortName": "G",
      "chi": -153.88587375071322,
      "glycosidicBond": "anti"
    },
    {
      "index": 35,
      "chain": "BC",
      "number": 2005,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BC.G2005",
      "shortName": "G",
      "chi": -148.85199128456694,
      "glycosidicBond": "anti"
    },
    {
      "index": 36,
      "chain": "BC",
      "number": 2006,
      "icode": null,
      "molecule": "RNA",
      "fullName": "BC.U2006",
      "shortName": "U",
      "chi": -159.43730655241544,
      "glycosidicBond": "anti"
    },
    {
      "index": 49,
      "chain": "C",
      "number": 9001,
      "icode": null,
      "molecule": "Other",
      "fullName": "C.SR9001",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 50,
      "chain": "CA",
      "number": 9001,
      "icode": null,
      "molecule": "Other",
      "fullName": "CA.SR9001",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 51,
      "chain": "CB",
      "number": 9001,
      "icode": null,
      "molecule": "Other",
      "fullName": "CB.SR9001",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 52,
      "chain": "CC",
      "number": 9001,
      "icode": null,
      "molecule": "Other",
      "fullName": "CC.SR9001",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 53,
      "chain": "D",
      "number": 9002,
      "icode": null,
      "molecule": "Other",
      "fullName": "D.SR9002",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 54,
      "chain": "DA",
      "number": 9002,
      "icode": null,
      "molecule": "Other",
      "fullName": "DA.SR9002",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 55,
      "chain": "DB",
      "number": 9002,
      "icode": null,
      "molecule": "Other",
      "fullName": "DB.SR9002",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 56,
      "chain": "DC",
      "number": 9002,
      "icode": null,
      "molecule": "Other",
      "fullName": "DC.SR9002",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 57,
      "chain": "E",
      "number": 9003,
      "icode": null,
      "molecule": "Other",
      "fullName": "E.NA9003",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 58,
      "chain": "EA",
      "number": 9003,
      "icode": null,
      "molecule": "Other",
      "fullName": "EA.NA9003",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 59,
      "chain": "EB",
      "number": 9003,
      "icode": null,
      "molecule": "Other",
      "fullName": "EB.NA9003",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 60,
      "chain": "EC",
      "number": 9003,
      "icode": null,
      "molecule": "Other",
      "fullName": "EC.NA9003",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 61,
      "chain": "F",
      "number": 9004,
      "icode": null,
      "molecule": "Other",
      "fullName": "F.SR9004",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 62,
      "chain": "FA",
      "number": 9004,
      "icode": null,
      "molecule": "Other",
      "fullName": "FA.SR9004",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 63,
      "chain": "FB",
      "number": 9004,
      "icode": null,
      "molecule": "Other",
      "fullName": "FB.SR9004",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 64,
      "chain": "FC",
      "number": 9004,
      "icode": null,
      "molecule": "Other",
      "fullName": "FC.SR9004",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 65,
      "chain": "G",
      "number": 2101,
      "icode": null,
      "molecule": "Other",
      "fullName": "G.SR2101",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 66,
      "chain": "GA",
      "number": 2101,
      "icode": null,
      "molecule": "Other",
      "fullName": "GA.SR2101",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 67,
      "chain": "GB",
      "number": 2101,
      "icode": null,
      "molecule": "Other",
      "fullName": "GB.SR2101",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 68,
      "chain": "GC",
      "number": 2101,
      "icode": null,
      "molecule": "Other",
      "fullName": "GC.SR2101",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 69,
      "chain": "H",
      "number": 2102,
      "icode": null,
      "molecule": "Other",
      "fullName": "H.SR2102",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 70,
      "chain": "HA",
      "number": 2102,
      "icode": null,
      "molecule": "Other",
      "fullName": "HA.SR2102",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 71,
      "chain": "HB",
      "number": 2102,
      "icode": null,
      "molecule": "Other",
      "fullName": "HB.SR2102",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 72,
      "chain": "HC",
      "number": 2102,
      "icode": null,
      "molecule": "Other",
      "fullName": "HC.SR2102",
      "shortName": "R",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 73,
      "chain": "I",
      "number": 2103,
      "icode": null,
      "molecule": "Other",
      "fullName": "I.NA2103",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 74,
      "chain": "IA",
      "number": 2103,
      "icode": null,
      "molecule": "Other",
      "fullName": "IA.NA2103",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 75,
      "chain": "IB",
      "number": 2103,
      "icode": null,
      "molecule": "Other",
      "fullName": "IB.NA2103",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 76,
      "chain": "IC",
      "number": 2103,
      "icode": null,
      "molecule": "Other",
      "fullName": "IC.NA2103",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 77,
      "chain": "J",
      "number": 2104,
      "icode": null,
      "molecule": "Other",
      "fullName": "J.CA2104",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 78,
      "chain": "JA",
      "number": 2104,
      "icode": null,
      "molecule": "Other",
      "fullName": "JA.CA2104",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 79,
      "chain": "JB",
      "number": 2104,
      "icode": null,
      "molecule": "Other",
      "fullName": "JB.CA2104",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 80,
      "chain": "JC",
      "number": 2104,
      "icode": null,
      "molecule": "Other",
      "fullName": "JC.CA2104",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 81,
      "chain": "K",
      "number": 2105,
      "icode": null,
      "molecule": "Other",
      "fullName": "K.CA2105",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 82,
      "chain": "KA",
      "number": 2105,
      "icode": null,
      "molecule": "Other",
      "fullName": "KA.CA2105",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 83,
      "chain": "KB",
      "number": 2105,
      "icode": null,
      "molecule": "Other",
      "fullName": "KB.CA2105",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 84,
      "chain": "KC",
      "number": 2105,
      "icode": null,
      "molecule": "Other",
      "fullName": "KC.CA2105",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 85,
      "chain": "L",
      "number": 2106,
      "icode": null,
      "molecule": "Other",
      "fullName": "L.CA2106",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 86,
      "chain": "LA",
      "number": 2106,
      "icode": null,
      "molecule": "Other",
      "fullName": "LA.CA2106",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 87,
      "chain": "LB",
      "number": 2106,
      "icode": null,
      "molecule": "Other",
      "fullName": "LB.CA2106",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 88,
      "chain": "LC",
      "number": 2106,
      "icode": null,
      "molecule": "Other",
      "fullName": "LC.CA2106",
      "shortName": "A",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 89,
      "chain": "M",
      "number": 9101,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9101",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 90,
      "chain": "M",
      "number": 9102,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9102",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 91,
      "chain": "M",
      "number": 9103,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9103",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 92,
      "chain": "M",
      "number": 9104,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9104",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 93,
      "chain": "M",
      "number": 9105,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9105",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 94,
      "chain": "M",
      "number": 9106,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9106",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 95,
      "chain": "M",
      "number": 9107,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9107",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 96,
      "chain": "M",
      "number": 9108,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9108",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 97,
      "chain": "M",
      "number": 9109,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9109",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 98,
      "chain": "M",
      "number": 9110,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9110",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 99,
      "chain": "M",
      "number": 9111,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9111",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 100,
      "chain": "M",
      "number": 9112,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9112",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 101,
      "chain": "M",
      "number": 9113,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9113",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 102,
      "chain": "M",
      "number": 9114,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9114",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 103,
      "chain": "M",
      "number": 9115,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9115",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 104,
      "chain": "M",
      "number": 9116,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9116",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 105,
      "chain": "M",
      "number": 9117,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9117",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 106,
      "chain": "M",
      "number": 9118,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9118",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 107,
      "chain": "M",
      "number": 9119,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9119",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 108,
      "chain": "M",
      "number": 9120,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9120",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 109,
      "chain": "M",
      "number": 9121,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9121",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 110,
      "chain": "M",
      "number": 9122,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9122",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 111,
      "chain": "M",
      "number": 9123,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9123",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 112,
      "chain": "M",
      "number": 9124,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9124",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 113,
      "chain": "M",
      "number": 9125,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9125",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 114,
      "chain": "M",
      "number": 9126,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9126",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 115,
      "chain": "M",
      "number": 9127,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9127",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 116,
      "chain": "M",
      "number": 9128,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9128",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 117,
      "chain": "M",
      "number": 9129,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9129",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 118,
      "chain": "M",
      "number": 9130,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9130",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 119,
      "chain": "M",
      "number": 9131,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9131",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 120,
      "chain": "M",
      "number": 9132,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9132",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 121,
      "chain": "M",
      "number": 9133,
      "icode": null,
      "molecule": "Other",
      "fullName": "M.HOH9133",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 122,
      "chain": "MA",
      "number": 9101,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9101",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 123,
      "chain": "MA",
      "number": 9102,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9102",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 124,
      "chain": "MA",
      "number": 9103,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9103",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 125,
      "chain": "MA",
      "number": 9104,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9104",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 126,
      "chain": "MA",
      "number": 9105,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9105",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 127,
      "chain": "MA",
      "number": 9106,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9106",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 128,
      "chain": "MA",
      "number": 9107,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9107",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 129,
      "chain": "MA",
      "number": 9108,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9108",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 130,
      "chain": "MA",
      "number": 9109,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9109",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 131,
      "chain": "MA",
      "number": 9110,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9110",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 132,
      "chain": "MA",
      "number": 9111,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9111",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 133,
      "chain": "MA",
      "number": 9112,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9112",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 134,
      "chain": "MA",
      "number": 9113,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9113",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 135,
      "chain": "MA",
      "number": 9114,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9114",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 136,
      "chain": "MA",
      "number": 9115,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9115",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 137,
      "chain": "MA",
      "number": 9116,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9116",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 138,
      "chain": "MA",
      "number": 9117,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9117",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 139,
      "chain": "MA",
      "number": 9118,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9118",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 140,
      "chain": "MA",
      "number": 9119,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9119",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 141,
      "chain": "MA",
      "number": 9120,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9120",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 142,
      "chain": "MA",
      "number": 9121,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9121",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 143,
      "chain": "MA",
      "number": 9122,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9122",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 144,
      "chain": "MA",
      "number": 9123,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9123",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 145,
      "chain": "MA",
      "number": 9124,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9124",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 146,
      "chain": "MA",
      "number": 9125,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9125",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 147,
      "chain": "MA",
      "number": 9126,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9126",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 148,
      "chain": "MA",
      "number": 9127,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9127",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 149,
      "chain": "MA",
      "number": 9128,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9128",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 150,
      "chain": "MA",
      "number": 9129,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9129",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 151,
      "chain": "MA",
      "number": 9130,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9130",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 152,
      "chain": "MA",
      "number": 9131,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9131",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 153,
      "chain": "MA",
      "number": 9132,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9132",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 154,
      "chain": "MA",
      "number": 9133,
      "icode": null,
      "molecule": "Other",
      "fullName": "MA.HOH9133",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 155,
      "chain": "MB",
      "number": 9101,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9101",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 156,
      "chain": "MB",
      "number": 9102,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9102",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 157,
      "chain": "MB",
      "number": 9103,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9103",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 158,
      "chain": "MB",
      "number": 9104,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9104",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 159,
      "chain": "MB",
      "number": 9105,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9105",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 160,
      "chain": "MB",
      "number": 9106,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9106",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 161,
      "chain": "MB",
      "number": 9107,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9107",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 162,
      "chain": "MB",
      "number": 9108,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9108",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 163,
      "chain": "MB",
      "number": 9109,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9109",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 164,
      "chain": "MB",
      "number": 9110,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9110",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 165,
      "chain": "MB",
      "number": 9111,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9111",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 166,
      "chain": "MB",
      "number": 9112,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9112",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 167,
      "chain": "MB",
      "number": 9113,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9113",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 168,
      "chain": "MB",
      "number": 9114,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9114",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 169,
      "chain": "MB",
      "number": 9115,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9115",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 170,
      "chain": "MB",
      "number": 9116,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9116",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 171,
      "chain": "MB",
      "number": 9117,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9117",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 172,
      "chain": "MB",
      "number": 9118,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9118",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 173,
      "chain": "MB",
      "number": 9119,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9119",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 174,
      "chain": "MB",
      "number": 9120,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9120",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 175,
      "chain": "MB",
      "number": 9121,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9121",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 176,
      "chain": "MB",
      "number": 9122,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9122",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 177,
      "chain": "MB",
      "number": 9123,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9123",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 178,
      "chain": "MB",
      "number": 9124,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9124",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 179,
      "chain": "MB",
      "number": 9125,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9125",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 180,
      "chain": "MB",
      "number": 9126,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9126",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 181,
      "chain": "MB",
      "number": 9127,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9127",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 182,
      "chain": "MB",
      "number": 9128,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9128",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 183,
      "chain": "MB",
      "number": 9129,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9129",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 184,
      "chain": "MB",
      "number": 9130,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9130",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 185,
      "chain": "MB",
      "number": 9131,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9131",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 186,
      "chain": "MB",
      "number": 9132,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9132",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 187,
      "chain": "MB",
      "number": 9133,
      "icode": null,
      "molecule": "Other",
      "fullName": "MB.HOH9133",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 188,
      "chain": "MC",
      "number": 9101,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9101",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 189,
      "chain": "MC",
      "number": 9102,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9102",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 190,
      "chain": "MC",
      "number": 9103,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9103",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 191,
      "chain": "MC",
      "number": 9104,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9104",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 192,
      "chain": "MC",
      "number": 9105,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9105",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 193,
      "chain": "MC",
      "number": 9106,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9106",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 194,
      "chain": "MC",
      "number": 9107,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9107",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 195,
      "chain": "MC",
      "number": 9108,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9108",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 196,
      "chain": "MC",
      "number": 9109,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9109",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 197,
      "chain": "MC",
      "number": 9110,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9110",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 198,
      "chain": "MC",
      "number": 9111,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9111",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 199,
      "chain": "MC",
      "number": 9112,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9112",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 200,
      "chain": "MC",
      "number": 9113,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9113",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 201,
      "chain": "MC",
      "number": 9114,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9114",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 202,
      "chain": "MC",
      "number": 9115,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9115",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 203,
      "chain": "MC",
      "number": 9116,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9116",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 204,
      "chain": "MC",
      "number": 9117,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9117",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 205,
      "chain": "MC",
      "number": 9118,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9118",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 206,
      "chain": "MC",
      "number": 9119,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9119",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 207,
      "chain": "MC",
      "number": 9120,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9120",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 208,
      "chain": "MC",
      "number": 9121,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9121",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 209,
      "chain": "MC",
      "number": 9122,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9122",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 210,
      "chain": "MC",
      "number": 9123,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9123",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 211,
      "chain": "MC",
      "number": 9124,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9124",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 212,
      "chain": "MC",
      "number": 9125,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9125",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 213,
      "chain": "MC",
      "number": 9126,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9126",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 214,
      "chain": "MC",
      "number": 9127,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9127",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 215,
      "chain": "MC",
      "number": 9128,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9128",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 216,
      "chain": "MC",
      "number": 9129,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9129",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 217,
      "chain": "MC",
      "number": 9130,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9130",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 218,
      "chain": "MC",
      "number": 9131,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9131",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 219,
      "chain": "MC",
      "number": 9132,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9132",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 220,
      "chain": "MC",
      "number": 9133,
      "icode": null,
      "molecule": "Other",
      "fullName": "MC.HOH9133",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 221,
      "chain": "N",
      "number": 2201,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2201",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 222,
      "chain": "N",
      "number": 2202,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2202",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 223,
      "chain": "N",
      "number": 2203,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2203",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 224,
      "chain": "N",
      "number": 2204,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2204",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 225,
      "chain": "N",
      "number": 2205,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2205",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 226,
      "chain": "N",
      "number": 2206,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2206",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 227,
      "chain": "N",
      "number": 2207,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2207",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 228,
      "chain": "N",
      "number": 2208,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2208",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 229,
      "chain": "N",
      "number": 2209,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2209",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 230,
      "chain": "N",
      "number": 2210,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2210",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 231,
      "chain": "N",
      "number": 2211,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2211",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 232,
      "chain": "N",
      "number": 2212,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2212",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 233,
      "chain": "N",
      "number": 2213,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2213",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 234,
      "chain": "N",
      "number": 2214,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2214",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 235,
      "chain": "N",
      "number": 2215,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2215",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 236,
      "chain": "N",
      "number": 2216,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2216",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 237,
      "chain": "N",
      "number": 2217,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2217",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 238,
      "chain": "N",
      "number": 2218,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2218",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 239,
      "chain": "N",
      "number": 2219,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2219",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 240,
      "chain": "N",
      "number": 2220,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2220",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 241,
      "chain": "N",
      "number": 2221,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2221",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 242,
      "chain": "N",
      "number": 2222,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2222",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 243,
      "chain": "N",
      "number": 2223,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2223",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 244,
      "chain": "N",
      "number": 2224,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2224",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 245,
      "chain": "N",
      "number": 2225,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2225",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 246,
      "chain": "N",
      "number": 2226,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2226",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 247,
      "chain": "N",
      "number": 2227,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2227",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 248,
      "chain": "N",
      "number": 2228,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2228",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 249,
      "chain": "N",
      "number": 2229,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2229",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 250,
      "chain": "N",
      "number": 2230,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2230",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 251,
      "chain": "N",
      "number": 2231,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2231",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 252,
      "chain": "N",
      "number": 2232,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2232",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 253,
      "chain": "N",
      "number": 2233,
      "icode": null,
      "molecule": "Other",
      "fullName": "N.HOH2233",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 254,
      "chain": "NA",
      "number": 2201,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2201",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 255,
      "chain": "NA",
      "number": 2202,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2202",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 256,
      "chain": "NA",
      "number": 2203,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2203",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 257,
      "chain": "NA",
      "number": 2204,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2204",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 258,
      "chain": "NA",
      "number": 2205,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2205",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 259,
      "chain": "NA",
      "number": 2206,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2206",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 260,
      "chain": "NA",
      "number": 2207,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2207",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 261,
      "chain": "NA",
      "number": 2208,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2208",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 262,
      "chain": "NA",
      "number": 2209,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2209",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 263,
      "chain": "NA",
      "number": 2210,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2210",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 264,
      "chain": "NA",
      "number": 2211,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2211",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 265,
      "chain": "NA",
      "number": 2212,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2212",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 266,
      "chain": "NA",
      "number": 2213,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2213",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 267,
      "chain": "NA",
      "number": 2214,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2214",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 268,
      "chain": "NA",
      "number": 2215,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2215",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 269,
      "chain": "NA",
      "number": 2216,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2216",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 270,
      "chain": "NA",
      "number": 2217,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2217",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 271,
      "chain": "NA",
      "number": 2218,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2218",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 272,
      "chain": "NA",
      "number": 2219,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2219",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 273,
      "chain": "NA",
      "number": 2220,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2220",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 274,
      "chain": "NA",
      "number": 2221,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2221",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 275,
      "chain": "NA",
      "number": 2222,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2222",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 276,
      "chain": "NA",
      "number": 2223,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2223",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 277,
      "chain": "NA",
      "number": 2224,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2224",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 278,
      "chain": "NA",
      "number": 2225,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2225",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 279,
      "chain": "NA",
      "number": 2226,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2226",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 280,
      "chain": "NA",
      "number": 2227,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2227",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 281,
      "chain": "NA",
      "number": 2228,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2228",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 282,
      "chain": "NA",
      "number": 2229,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2229",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 283,
      "chain": "NA",
      "number": 2230,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2230",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 284,
      "chain": "NA",
      "number": 2231,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2231",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 285,
      "chain": "NA",
      "number": 2232,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2232",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 286,
      "chain": "NA",
      "number": 2233,
      "icode": null,
      "molecule": "Other",
      "fullName": "NA.HOH2233",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 287,
      "chain": "NB",
      "number": 2201,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2201",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 288,
      "chain": "NB",
      "number": 2202,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2202",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 289,
      "chain": "NB",
      "number": 2203,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2203",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 290,
      "chain": "NB",
      "number": 2204,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2204",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 291,
      "chain": "NB",
      "number": 2205,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2205",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 292,
      "chain": "NB",
      "number": 2206,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2206",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 293,
      "chain": "NB",
      "number": 2207,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2207",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 294,
      "chain": "NB",
      "number": 2208,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2208",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 295,
      "chain": "NB",
      "number": 2209,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2209",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 296,
      "chain": "NB",
      "number": 2210,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2210",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 297,
      "chain": "NB",
      "number": 2211,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2211",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 298,
      "chain": "NB",
      "number": 2212,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2212",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 299,
      "chain": "NB",
      "number": 2213,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2213",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 300,
      "chain": "NB",
      "number": 2214,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2214",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 301,
      "chain": "NB",
      "number": 2215,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2215",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 302,
      "chain": "NB",
      "number": 2216,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2216",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 303,
      "chain": "NB",
      "number": 2217,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2217",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 304,
      "chain": "NB",
      "number": 2218,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2218",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 305,
      "chain": "NB",
      "number": 2219,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2219",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 306,
      "chain": "NB",
      "number": 2220,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2220",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 307,
      "chain": "NB",
      "number": 2221,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2221",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 308,
      "chain": "NB",
      "number": 2222,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2222",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 309,
      "chain": "NB",
      "number": 2223,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2223",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 310,
      "chain": "NB",
      "number": 2224,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2224",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 311,
      "chain": "NB",
      "number": 2225,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2225",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 312,
      "chain": "NB",
      "number": 2226,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2226",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 313,
      "chain": "NB",
      "number": 2227,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2227",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 314,
      "chain": "NB",
      "number": 2228,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2228",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 315,
      "chain": "NB",
      "number": 2229,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2229",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 316,
      "chain": "NB",
      "number": 2230,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2230",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 317,
      "chain": "NB",
      "number": 2231,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2231",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 318,
      "chain": "NB",
      "number": 2232,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2232",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 319,
      "chain": "NB",
      "number": 2233,
      "icode": null,
      "molecule": "Other",
      "fullName": "NB.HOH2233",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 320,
      "chain": "NC",
      "number": 2201,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2201",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 321,
      "chain": "NC",
      "number": 2202,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2202",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 322,
      "chain": "NC",
      "number": 2203,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2203",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 323,
      "chain": "NC",
      "number": 2204,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2204",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 324,
      "chain": "NC",
      "number": 2205,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2205",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 325,
      "chain": "NC",
      "number": 2206,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2206",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 326,
      "chain": "NC",
      "number": 2207,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2207",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 327,
      "chain": "NC",
      "number": 2208,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2208",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 328,
      "chain": "NC",
      "number": 2209,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2209",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 329,
      "chain": "NC",
      "number": 2210,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2210",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 330,
      "chain": "NC",
      "number": 2211,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2211",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 331,
      "chain": "NC",
      "number": 2212,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2212",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 332,
      "chain": "NC",
      "number": 2213,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2213",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 333,
      "chain": "NC",
      "number": 2214,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2214",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 334,
      "chain": "NC",
      "number": 2215,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2215",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 335,
      "chain": "NC",
      "number": 2216,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2216",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 336,
      "chain": "NC",
      "number": 2217,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2217",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 337,
      "chain": "NC",
      "number": 2218,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2218",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 338,
      "chain": "NC",
      "number": 2219,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2219",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 339,
      "chain": "NC",
      "number": 2220,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2220",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 340,
      "chain": "NC",
      "number": 2221,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2221",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 341,
      "chain": "NC",
      "number": 2222,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2222",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 342,
      "chain": "NC",
      "number": 2223,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2223",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 343,
      "chain": "NC",
      "number": 2224,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2224",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 344,
      "chain": "NC",
      "number": 2225,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2225",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 345,
      "chain": "NC",
      "number": 2226,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2226",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 346,
      "chain": "NC",
      "number": 2227,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2227",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 347,
      "chain": "NC",
      "number": 2228,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2228",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 348,
      "chain": "NC",
      "number": 2229,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2229",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 349,
      "chain": "NC",
      "number": 2230,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2230",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 350,
      "chain": "NC",
      "number": 2231,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2231",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 351,
      "chain": "NC",
      "number": 2232,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2232",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    },
    {
      "index": 352,
      "chain": "NC",
      "number": 2233,
      "icode": null,
      "molecule": "Other",
      "fullName": "NC.HOH2233",
      "shortName": "H",
      "chi": null,
      "glycosidicBond": null
    }
  ],
  "basePairs": [
    {
      "nt1": "A.G1002",
      "nt2": "AB.G1002",
      "lw": "cWH"
    },
    {
      "nt1": "A.G1002",
      "nt2": "AC.G1002",
      "lw": "cHW"
    },
    {
      "nt1": "A.G1003",
      "nt2": "AB.G1003",
      "lw": "cWH"
    },
    {
      "nt1": "A.G1003",
      "nt2": "AC.G1003",
      "lw": "cHW"
    },
    {
      "nt1": "A.G1004",
      "nt2": "AB.G1004",
      "lw": "cWH"
    },
    {
      "nt1": "A.G1004",
      "nt2": "AC.G1004",
      "lw": "cHW"
    },
    {
      "nt1": "A.G1005",
      "nt2": "AB.G1005",
      "lw": "cWH"
    },
    {
      "nt1": "A.G1005",
      "nt2": "AC.G1005",
      "lw": "cHW"
    },
    {
      "nt1": "A.U1006",
      "nt2": "AB.U1006",
      "lw": "cHW"
    },
    {
      "nt1": "A.U1006",
      "nt2": "AC.U1006",
      "lw": "cWH"
    },
    {
      "nt1": "AA.G1002",
      "nt2": "AC.G1002",
      "lw": "cWH"
    },
    {
      "nt1": "AA.G1002",
      "nt2": "AB.G1002",
      "lw": "cHW"
    },
    {
      "nt1": "AA.G1003",
      "nt2": "AC.G1003",
      "lw": "cWH"
    },
    {
      "nt1": "AA.G1003",
      "nt2": "AB.G1003",
      "lw": "cHW"
    },
    {
      "nt1": "AA.G1004",
      "nt2": "AC.G1004",
      "lw": "cWH"
    },
    {
      "nt1": "AA.G1004",
      "nt2": "AB.G1004",
      "lw": "cHW"
    },
    {
      "nt1": "AA.G1005",
      "nt2": "AC.G1005",
      "lw": "cWH"
    },
    {
      "nt1": "AA.G1005",
      "nt2": "AB.G1005",
      "lw": "cHW"
    },
    {
      "nt1": "AA.U1006",
      "nt2": "AC.U1006",
      "lw": "cHW"
    },
    {
      "nt1": "AA.U1006",
      "nt2": "AB.U1006",
      "lw": "cWH"
    },
    {
      "nt1": "AB.G1002",
      "nt2": "A.G1002",
      "lw": "cHW"
    },
    {
      "nt1": "AB.G1002",
      "nt2": "AA.G1002",
      "lw": "cWH"
    },
    {
      "nt1": "AB.G1003",
      "nt2": "A.G1003",
      "lw": "cHW"
    },
    {
      "nt1": "AB.G1003",
      "nt2": "AA.G1003",
      "lw": "cWH"
    },
    {
      "nt1": "AB.G1004",
      "nt2": "A.G1004",
      "lw": "cHW"
    },
    {
      "nt1": "AB.G1004",
      "nt2": "AA.G1004",
      "lw": "cWH"
    },
    {
      "nt1": "AB.G1005",
      "nt2": "A.G1005",
      "lw": "cHW"
    },
    {
      "nt1": "AB.G1005",
      "nt2": "AA.G1005",
      "lw": "cWH"
    },
    {
      "nt1": "AB.U1006",
      "nt2": "A.U1006",
      "lw": "cWH"
    },
    {
      "nt1": "AB.U1006",
      "nt2": "AA.U1006",
      "lw": "cHW"
    },
    {
      "nt1": "AC.G1002",
      "nt2": "AA.G1002",
      "lw": "cHW"
    },
    {
      "nt1": "AC.G1002",
      "nt2": "A.G1002",
      "lw": "cWH"
    },
    {
      "nt1": "AC.G1003",
      "nt2": "AA.G1003",
      "lw": "cHW"
    },
    {
      "nt1": "AC.G1003",
      "nt2": "A.G1003",
      "lw": "cWH"
    },
    {
      "nt1": "AC.G1004",
      "nt2": "AA.G1004",
      "lw": "cHW"
    },
    {
      "nt1": "AC.G1004",
      "nt2": "A.G1004",
      "lw": "cWH"
    },
    {
      "nt1": "AC.G1005",
      "nt2": "AA.G1005",
      "lw": "cHW"
    },
    {
      "nt1": "AC.G1005",
      "nt2": "A.G1005",
      "lw": "cWH"
    },
    {
      "nt1": "AC.U1006",
      "nt2": "AA.U1006",
      "lw": "cWH"
    },
    {
      "nt1": "AC.U1006",
      "nt2": "A.U1006",
      "lw": "cHW"
    },
    {
      "nt1": "B.G2002",
      "nt2": "BC.G2002",
      "lw": "cWH"
    },
    {
      "nt1": "B.G2002",
      "nt2": "BB.G2002",
      "lw": "cHW"
    },
    {
      "nt1": "B.G2003",
      "nt2": "BC.G2003",
      "lw": "cWH"
    },
    {
      "nt1": "B.G2003",
      "nt2": "BB.G2003",
      "lw": "cHW"
    },
    {
      "nt1": "B.G2004",
      "nt2": "BC.G2004",
      "lw": "cWH"
    },
    {
      "nt1": "B.G2004",
      "nt2": "BB.G2004",
      "lw": "cHW"
    },
    {
      "nt1": "B.G2005",
      "nt2": "BC.G2005",
      "lw": "cWH"
    },
    {
      "nt1": "B.G2005",
      "nt2": "BB.G2005",
      "lw": "cHW"
    },
    {
      "nt1": "B.U2006",
      "nt2": "BC.U2006",
      "lw": "cHW"
    },
    {
      "nt1": "B.U2006",
      "nt2": "BB.U2006",
      "lw": "cWH"
    },
    {
      "nt1": "BA.G2002",
      "nt2": "BB.G2002",
      "lw": "cWH"
    },
    {
      "nt1": "BA.G2002",
      "nt2": "BC.G2002",
      "lw": "cHW"
    },
    {
      "nt1": "BA.G2003",
      "nt2": "BB.G2003",
      "lw": "cWH"
    },
    {
      "nt1": "BA.G2003",
      "nt2": "BC.G2003",
      "lw": "cHW"
    },
    {
      "nt1": "BA.G2004",
      "nt2": "BB.G2004",
      "lw": "cWH"
    },
    {
      "nt1": "BA.G2004",
      "nt2": "BC.G2004",
      "lw": "cHW"
    },
    {
      "nt1": "BA.G2005",
      "nt2": "BB.G2005",
      "lw": "cWH"
    },
    {
      "nt1": "BA.G2005",
      "nt2": "BC.G2005",
      "lw": "cHW"
    },
    {
      "nt1": "BA.U2006",
      "nt2": "BB.U2006",
      "lw": "cHW"
    },
    {
      "nt1": "BA.U2006",
      "nt2": "BC.U2006",
      "lw": "cWH"
    },
    {
      "nt1": "BB.G2002",
      "nt2": "BA.G2002",
      "lw": "cHW"
    },
    {
      "nt1": "BB.G2002",
      "nt2": "B.G2002",
      "lw": "cWH"
    },
    {
      "nt1": "BB.G2003",
      "nt2": "BA.G2003",
      "lw": "cHW"
    },
    {
      "nt1": "BB.G2003",
      "nt2": "B.G2003",
      "lw": "cWH"
    },
    {
      "nt1": "BB.G2004",
      "nt2": "BA.G2004",
      "lw": "cHW"
    },
    {
      "nt1": "BB.G2004",
      "nt2": "B.G2004",
      "lw": "cWH"
    },
    {
      "nt1": "BB.G2005",
      "nt2": "BA.G2005",
      "lw": "cHW"
    },
    {
      "nt1": "BB.G2005",
      "nt2": "B.G2005",
      "lw": "cWH"
    },
    {
      "nt1": "BB.U2006",
      "nt2": "BA.U2006",
      "lw": "cWH"
    },
    {
      "nt1": "BB.U2006",
      "nt2": "B.U2006",
      "lw": "cHW"
    },
    {
      "nt1": "BC.G2002",
      "nt2": "B.G2002",
      "lw": "cHW"
    },
    {
      "nt1": "BC.G2002",
      "nt2": "BA.G2002",
      "lw": "cWH"
    },
    {
      "nt1": "BC.G2003",
      "nt2": "B.G2003",
      "lw": "cHW"
    },
    {
      "nt1": "BC.G2003",
      "nt2": "BA.G2003",
      "lw": "cWH"
    },
    {
      "nt1": "BC.G2004",
      "nt2": "B.G2004",
      "lw": "cHW"
    },
    {
      "nt1": "BC.G2004",
      "nt2": "BA.G2004",
      "lw": "cWH"
    },
    {
      "nt1": "BC.G2005",
      "nt2": "B.G2005",
      "lw": "cHW"
    },
    {
      "nt1": "BC.G2005",
      "nt2": "BA.G2005",
      "lw": "cWH"
    },
    {
      "nt1": "BC.U2006",
      "nt2": "B.U2006",
      "lw": "cWH"
    },
    {
      "nt1": "BC.U2006",
      "nt2": "BA.U2006",
      "lw": "cHW"
    }
  ],
  "helices": [
    {
      "quadruplexes": [
        {
          "tetrads": [
            {
              "id": "A.U1006-AC.U1006-AA.U1006-AB.U1006",
              "nt1": "A.U1006",
              "nt2": "AC.U1006",
              "nt3": "AA.U1006",
              "nt4": "AB.U1006",
              "onz": "O-",
              "gbaClassification": "VIIIa",
              "planarityDeviation": 1.061,
              "ionsChannel": [
                "NA"
              ],
              "ionsOutside": [
                {
                  "nt": "A.U1006",
                  "ion": "SR"
                },
                {
                  "nt": "AA.U1006",
                  "ion": "SR"
                },
                {
                  "nt": "AB.U1006",
                  "ion": "SR"
                },
                {
                  "nt": "AC.U1006",
                  "ion": "SR"
                }
              ]
            },
            {
              "id": "A.G1005-AC.G1005-AA.G1005-AB.G1005",
              "nt1": "A.G1005",
              "nt2": "AC.G1005",
              "nt3": "AA.G1005",
              "nt4": "AB.G1005",
              "onz": "O+",
              "gbaClassification": "VIIIa",
              "planarityDeviation": 0.7999999999999972,
              "ionsChannel": [],
              "ionsOutside": []
            },
            {
              "id": "A.G1004-AC.G1004-AA.G1004-AB.G1004",
              "nt1": "A.G1004",
              "nt2": "AC.G1004",
              "nt3": "AA.G1004",
              "nt4": "AB.G1004",
              "onz": "O+",
              "gbaClassification": "VIIIa",
              "planarityDeviation": 0.4059999999999988,
              "ionsChannel": [
                "SR"
              ],
              "ionsOutside": []
            },
            {
              "id": "A.G1003-AC.G1003-AA.G1003-AB.G1003",
              "nt1": "A.G1003",
              "nt2": "AC.G1003",
              "nt3": "AA.G1003",
              "nt4": "AB.G1003",
              "onz": "O+",
              "gbaClassification": "VIIIa",
              "planarityDeviation": 0.5549999999999997,
              "ionsChannel": [
                "SR"
              ],
              "ionsOutside": []
            },
            {
              "id": "A.G1002-AC.G1002-AA.G1002-AB.G1002",
              "nt1": "A.G1002",
              "nt2": "AC.G1002",
              "nt3": "AA.G1002",
              "nt4": "AB.G1002",
              "onz": "O+",
              "gbaClassification": "VIIIa",
              "planarityDeviation": 0.541999999999998,
              "ionsChannel": [],
              "ionsOutside": [
                {
                  "nt": "AB.G1002",
                  "ion": "CA"
                },
                {
                  "nt": "AC.G1002",
                  "ion": "CA"
                },
                {
                  "nt": "AA.G1002",
                  "ion": "CA"
                },
                {
                  "nt": "A.G1002",
                  "ion": "CA"
                }
              ]
            }
          ],
          "onzm": "Op*",
          "loopClassification": null,
          "gbaClassification": [
            "VIII"
          ],
          "tracts": [
            [
              "A.U1006",
              "A.G1005",
              "A.G1004",
              "A.G1003",
              "A.G1002"
            ],
            [
              "AC.U1006",
              "AC.G1005",
              "AC.G1004",
              "AC.G1003",
              "AC.G1002"
            ],
            [
              "AA.U1006",
              "AA.G1005",
              "AA.G1004",
              "AA.G1003",
              "AA.G1002"
            ],
            [
              "AB.U1006",
              "AB.G1005",
              "AB.G1004",
              "AB.G1003",
              "AB.G1002"
            ]
          ],
          "loops": []
        },
        {
          "tetrads": [
            {
              "id": "B.G2002-BC.G2002-BA.G2002-BB.G2002",
              "nt1": "B.G2002",
              "nt2": "BC.G2002",
              "nt3": "BA.G2002",
              "nt4": "BB.G2002",
              "onz": "O+",
              "gbaClassification": "VIIIa",
              "planarityDeviation": 0.6730000000000018,
              "ionsChannel": [],
              "ionsOutside": []
            },
            {
              "id": "B.G2003-BC.G2003-BA.G2003-BB.G2003",
              "nt1": "B.G2003",
              "nt2": "BC.G2003",
              "nt3": "BA.G2003",
              "nt4": "BB.G2003",
              "onz": "O+",
              "gbaClassification": "VIIIa",
              "planarityDeviation": 0.5769999999999982,
              "ionsChannel": [
                "SR"
              ],
              "ionsOutside": [
                {
                  "nt": "B.G2003",
                  "ion": "CA"
                },
                {
                  "nt": "BA.G2003",
                  "ion": "CA"
                },
                {
                  "nt": "BB.G2003",
                  "ion": "CA"
                },
                {
                  "nt": "BC.G2003",
                  "ion": "CA"
                }
              ]
            },
            {
              "id": "B.G2004-BC.G2004-BA.G2004-BB.G2004",
              "nt1": "B.G2004",
              "nt2": "BC.G2004",
              "nt3": "BA.G2004",
              "nt4": "BB.G2004",
              "onz": "O+",
              "gbaClassification": "VIIIa",
              "planarityDeviation": 0.2289999999999992,
              "ionsChannel": [
                "SR"
              ],
              "ionsOutside": []
            },
            {
              "id": "B.G2005-BC.G2005-BA.G2005-BB.G2005",
              "nt1": "B.G2005",
              "nt2": "BC.G2005",
              "nt3": "BA.G2005",
              "nt4": "BB.G2005",
              "onz": "O+",
              "gbaClassification": "VIIIa",
              "planarityDeviation": 0.7810000000000006,
              "ionsChannel": [
                "NA"
              ],
              "ionsOutside": []
            },
            {
              "id": "B.U2006-BC.U2006-BA.U2006-BB.U2006",
              "nt1": "B.U2006",
              "nt2": "BC.U2006",
              "nt3": "BA.U2006",
              "nt4": "BB.U2006",
              "onz": "O-",
              "gbaClassification": "VIIIa",
              "planarityDeviation": 1.5840000000000005,
              "ionsChannel": [
                "NA",
                "NA"
              ],
              "ionsOutside": []
            }
          ],
          "onzm": "Op*",
          "loopClassification": null,
          "gbaClassification": [
            "VIII"
          ],
          "tracts": [
            [
              "B.G2002",
              "B.G2003",
              "B.G2004",
              "B.G2005",
              "B.U2006"
            ],
            [
              "BC.G2002",
              "BC.G2003",
              "BC.G2004",
              "BC.G2005",
              "BC.U2006"
            ],
            [
              "BA.G2002",
              "BA.G2003",
              "BA.G2004",
              "BA.G2005",
              "BA.U2006"
            ],
            [
              "BB.G2002",
              "BB.G2003",
              "BB.G2004",
              "BB.G2005",
              "BB.U2006"
            ]
          ],
          "loops": []
        }
      ],
      "tetradPairs": [
        {
          "tetrad1": "A.U1006-AC.U1006-AA.U1006-AB.U1006",
          "tetrad2": "A.G1005-AC.G1005-AA.G1005-AB.G1005",
          "direction": "parallel",
          "rise": 3.366499999999995,
          "twist": 39.962531742191736
        },
        {
          "tetrad1": "A.G1005-AC.G1005-AA.G1005-AB.G1005",
          "tetrad2": "A.G1004-AC.G1004-AA.G1004-AB.G1004",
          "direction": "parallel",
          "rise": 3.308000000000007,
          "twist": 25.89614444631925
        },
        {
          "tetrad1": "A.G1004-AC.G1004-AA.G1004-AB.G1004",
          "tetrad2": "A.G1003-AC.G1003-AA.G1003-AB.G1003",
          "direction": "parallel",
          "rise": 3.339499999999994,
          "twist": 35.81115298630443
        },
        {
          "tetrad1": "A.G1003-AC.G1003-AA.G1003-AB.G1003",
          "tetrad2": "A.G1002-AC.G1002-AA.G1002-AB.G1002",
          "direction": "parallel",
          "rise": 3.2865,
          "twist": 27.11515971986803
        },
        {
          "tetrad1": "A.G1002-AC.G1002-AA.G1002-AB.G1002",
          "tetrad2": "B.G2002-BC.G2002-BA.G2002-BB.G2002",
          "direction": "parallel",
          "rise": 3.369500000000002,
          "twist": 28.993180312675573
        },
        {
          "tetrad1": "B.G2002-BC.G2002-BA.G2002-BB.G2002",
          "tetrad2": "B.G2003-BC.G2003-BA.G2003-BB.G2003",
          "direction": "parallel",
          "rise": 3.371000000000002,
          "twist": 27.410084968596852
        },
        {
          "tetrad1": "B.G2003-BC.G2003-BA.G2003-BB.G2003",
          "tetrad2": "B.G2004-BC.G2004-BA.G2004-BB.G2004",
          "direction": "parallel",
          "rise": 3.318000000000005,
          "twist": 35.04072146975963
        },
        {
          "tetrad1": "B.G2004-BC.G2004-BA.G2004-BB.G2004",
          "tetrad2": "B.G2005-BC.G2005-BA.G2005-BB.G2005",
          "direction": "parallel",
          "rise": 3.2689999999999966,
          "twist": 25.149997949938147
        },
        {
          "tetrad1": "B.G2005-BC.G2005-BA.G2005-BB.G2005",
          "tetrad2": "B.U2006-BC.U2006-BA.U2006-BB.U2006",
          "direction": "parallel",
          "rise": 7.140499999999998,
          "twist": 43.40609492262336
        }
      ]
    }
  ],
  "dotBracket": {
    "sequence": "UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU",
    "line1": ".([{<A-.([{<A-.)]}>a-.)]}>a-.([{<A-.)]}>a-.([{<A-.)]}>a",
    "line2": ".([{<A-.)]}>a-.([{<A-.)]}>a-.([{<A-.([{<A-.)]}>a-.)]}>a"
  }
}
```

</details>

# Funding

This research was supported by the National Science Centre, Poland
\[2016/23/B/ST6/03931, 2019/35/B/ST6/03074\] and Mloda Kadra project
\[09/91/SBAD/0684\] from Poznan University of Technology, and carried
out in the European Centre for Bioinformatics and Genomics (Poland). The
authors also acknowledge partial support by the statutory funds of
Poznan University of Technology, Polish Ministry of Science and Higher
Education, and the Institute of Bioorganic Chemistry, PAS within
intramural financing program.

# Bibliography

<div id="refs" class="references csl-bib-body">

1.  Contact Networks in RNA: A Structural Bioinformatics Study with a
    New Tool. P. Roy, D. Bhattacharyya. *Journal of Computer-Aided
    Molecular Design*. 2022. 36(2):131–140.
    doi:[10.1007/s10822-021-00438-x](https://doi.org/10.1007/s10822-021-00438-x)

2.  Topology-Based Classification of Tetrads and Quadruplex
    Structures. M. Popenda, J. Miskiewicz, J. Sarzynska, T. Zok, M.
    Szachniuk. *Bioinformatics*. 2020. 36(4):1129–1134.
    doi:[10.1093/bioinformatics/btz738](https://doi.org/10.1093/bioinformatics/btz738)

3.  ElTetrado: A Tool for Identification and Classification of Tetrads
    and Quadruplexes. T. Zok, M. Popenda, M. Szachniuk. *BMC
    Bioinformatics*. 2020. 21(1):40.
    doi:[10.1186/s12859-020-3385-1](https://doi.org/10.1186/s12859-020-3385-1)

4.  R-Chie : A Web Server and R Package for Visualizing RNA Secondary
    Structures. D. Lai, J.R. Proctor, J.Y.A. Zhu, I.M. Meyer. *Nucleic
    Acids Research*. 2012. 40(12):e95.
    doi:[f99845](https://doi.org/f99845)

5.  Geometric Nomenclature and Classification of RNA Base Pairs. N.B.
    Leontis, E. Westhof. *RNA*. 2001. 7(4):499–512.
    doi:[10.1017/s1355838201002515](https://doi.org/10.1017/s1355838201002515)

</div>
