"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceNamespace = exports.ScalableTarget = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
/**
 * Define a scalable target
 */
class ScalableTarget extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.actions = new Array();
        core_1.withResolved(props.maxCapacity, max => {
            if (max < 0) {
                throw new RangeError(`maxCapacity cannot be negative, got: ${props.maxCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, min => {
            if (min < 0) {
                throw new RangeError(`minCapacity cannot be negative, got: ${props.minCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, props.maxCapacity, (min, max) => {
            if (max < min) {
                throw new RangeError(`minCapacity (${props.minCapacity}) should be lower than maxCapacity (${props.maxCapacity})`);
            }
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('application-autoscaling.amazonaws.com'),
        });
        const resource = new applicationautoscaling_generated_1.CfnScalableTarget(this, 'Resource', {
            maxCapacity: props.maxCapacity,
            minCapacity: props.minCapacity,
            resourceId: props.resourceId,
            roleArn: this.role.roleArn,
            scalableDimension: props.scalableDimension,
            scheduledActions: core_1.Lazy.anyValue({ produce: () => this.actions }, { omitEmptyArray: true }),
            serviceNamespace: props.serviceNamespace,
        });
        this.scalableTargetId = resource.ref;
    }
    static fromScalableTargetId(scope, id, scalableTargetId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.scalableTargetId = scalableTargetId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a policy statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, action) {
        if (action.minCapacity === undefined && action.maxCapacity === undefined) {
            throw new Error(`You must supply at least one of minCapacity or maxCapacity, got ${JSON.stringify(action)}`);
        }
        this.actions.push({
            scheduledActionName: id,
            schedule: action.schedule.expressionString,
            startTime: action.startTime,
            endTime: action.endTime,
            scalableTargetAction: {
                maxCapacity: action.maxCapacity,
                minCapacity: action.minCapacity,
            },
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
}
exports.ScalableTarget = ScalableTarget;
/**
 * The service that supports Application AutoScaling
 */
var ServiceNamespace;
(function (ServiceNamespace) {
    /**
     * Elastic Container Service
     */
    ServiceNamespace["ECS"] = "ecs";
    /**
     * Elastic Map Reduce
     */
    ServiceNamespace["ELASTIC_MAP_REDUCE"] = "elasticmapreduce";
    /**
     * Elastic Compute Cloud
     */
    ServiceNamespace["EC2"] = "ec2";
    /**
     * App Stream
     */
    ServiceNamespace["APPSTREAM"] = "appstream";
    /**
     * Dynamo DB
     */
    ServiceNamespace["DYNAMODB"] = "dynamodb";
    /**
     * Relational Database Service
     */
    ServiceNamespace["RDS"] = "rds";
    /**
     * SageMaker
     */
    ServiceNamespace["SAGEMAKER"] = "sagemaker";
    /**
     * Custom Resource
     */
    ServiceNamespace["CUSTOM_RESOURCE"] = "custom-resource";
    /**
     * Lambda
     */
    ServiceNamespace["LAMBDA"] = "lambda";
    /**
     * Comprehend
     */
    ServiceNamespace["COMPREHEND"] = "comprehend";
})(ServiceNamespace = exports.ServiceNamespace || (exports.ServiceNamespace = {}));
//# sourceMappingURL=data:application/json;base64,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