"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const cdk = require("@aws-cdk/core");
const http_origin_1 = require("./http-origin");
/**
 * (experimental) An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 *
 * @experimental
 */
class S3Origin {
    /**
     * @experimental
     */
    constructor(bucket, props = {}) {
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    /**
     * (experimental) (experimental) The method called when a given Origin is added (for the first time) to a Distribution.
     *
     * @experimental
     */
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, { originAccessIdentity, ...props }) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
        if (originAccessIdentity) {
            this.originAccessIdentity = originAccessIdentity;
        }
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            // Using a bucket from another stack creates a cyclic reference with
            // the bucket taking a dependency on the generated S3CanonicalUserId when `grantRead` is called,
            // and the distribution having a dependency on the bucket's domain name.
            // Fix this by parenting the OAI in the bucket's stack when cross-stack usage is detected.
            const bucketStack = cdk.Stack.of(this.bucket);
            const bucketInDifferentStack = bucketStack !== cdk.Stack.of(scope);
            const oaiScope = bucketInDifferentStack ? bucketStack : scope;
            const oaiId = bucketInDifferentStack ? `${cdk.Names.uniqueId(scope)}S3Origin` : 'S3Origin';
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(oaiScope, oaiId, {
                comment: `Identity for ${options.originId}`,
            });
            this.bucket.grantRead(this.originAccessIdentity);
        }
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiczMtb3JpZ2luLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiczMtb3JpZ2luLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHNEQUFzRDtBQUV0RCxxQ0FBcUM7QUFDckMsK0NBQTJDOzs7Ozs7Ozs7O0FBZ0MzQyxNQUFhLFFBQVE7Ozs7SUFHbkIsWUFBWSxNQUFrQixFQUFFLFFBQXVCLEVBQUU7UUFDdkQsSUFBSSxDQUFDLE1BQU0sR0FBRyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDOUIsSUFBSSx3QkFBVSxDQUFDLE1BQU0sQ0FBQyx1QkFBdUIsRUFBRTtnQkFDN0MsY0FBYyxFQUFFLFVBQVUsQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTO2dCQUN6RCxHQUFHLEtBQUs7YUFDVCxDQUFDLENBQUMsQ0FBQztZQUNKLElBQUksY0FBYyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN0QyxDQUFDOzs7Ozs7SUFFTSxJQUFJLENBQUMsS0FBb0IsRUFBRSxPQUFxQztRQUNyRSxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMxQyxDQUFDO0NBQ0Y7QUFmRCw0QkFlQztBQUVEOzs7O0dBSUc7QUFDSCxNQUFNLGNBQWUsU0FBUSxVQUFVLENBQUMsVUFBVTtJQUdoRCxZQUE2QixNQUFrQixFQUFFLEVBQUUsb0JBQW9CLEVBQUUsR0FBRyxLQUFLLEVBQWlCO1FBQ2hHLEtBQUssQ0FBQyxNQUFNLENBQUMsd0JBQXdCLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFEbkIsV0FBTSxHQUFOLE1BQU0sQ0FBWTtRQUU3QyxJQUFJLG9CQUFvQixFQUFFO1lBQ3hCLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxvQkFBb0IsQ0FBQztTQUNsRDtJQUNILENBQUM7SUFFTSxJQUFJLENBQUMsS0FBb0IsRUFBRSxPQUFxQztRQUNyRSxJQUFJLENBQUMsSUFBSSxDQUFDLG9CQUFvQixFQUFFO1lBQzlCLG9FQUFvRTtZQUNwRSxnR0FBZ0c7WUFDaEcsd0VBQXdFO1lBQ3hFLDBGQUEwRjtZQUMxRixNQUFNLFdBQVcsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDOUMsTUFBTSxzQkFBc0IsR0FBRyxXQUFXLEtBQUssR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDbkUsTUFBTSxRQUFRLEdBQUcsc0JBQXNCLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO1lBQzlELE1BQU0sS0FBSyxHQUFHLHNCQUFzQixDQUFDLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQztZQUUzRixJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxVQUFVLENBQUMsb0JBQW9CLENBQUMsUUFBUSxFQUFFLEtBQUssRUFBRTtnQkFDL0UsT0FBTyxFQUFFLGdCQUFnQixPQUFPLENBQUMsUUFBUSxFQUFFO2FBQzVDLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1NBQ2xEO1FBQ0QsT0FBTyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRVMsb0JBQW9CO1FBQzVCLE9BQU8sRUFBRSxvQkFBb0IsRUFBRSxxQ0FBcUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLHdCQUF3QixFQUFFLEVBQUUsQ0FBQztJQUM3SCxDQUFDO0NBQ0YiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjbG91ZGZyb250IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZGZyb250JztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBIdHRwT3JpZ2luIH0gZnJvbSAnLi9odHRwLW9yaWdpbic7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTM09yaWdpblByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBvcmlnaW5QYXRoPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBvcmlnaW5BY2Nlc3NJZGVudGl0eT86IGNsb3VkZnJvbnQuSU9yaWdpbkFjY2Vzc0lkZW50aXR5O1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFMzT3JpZ2luIGltcGxlbWVudHMgY2xvdWRmcm9udC5JT3JpZ2luIHtcbiAgcHJpdmF0ZSByZWFkb25seSBvcmlnaW46IGNsb3VkZnJvbnQuSU9yaWdpbjtcblxuICBjb25zdHJ1Y3RvcihidWNrZXQ6IHMzLklCdWNrZXQsIHByb3BzOiBTM09yaWdpblByb3BzID0ge30pIHtcbiAgICB0aGlzLm9yaWdpbiA9IGJ1Y2tldC5pc1dlYnNpdGUgP1xuICAgICAgbmV3IEh0dHBPcmlnaW4oYnVja2V0LmJ1Y2tldFdlYnNpdGVEb21haW5OYW1lLCB7XG4gICAgICAgIHByb3RvY29sUG9saWN5OiBjbG91ZGZyb250Lk9yaWdpblByb3RvY29sUG9saWN5LkhUVFBfT05MWSwgLy8gUzMgb25seSBzdXBwb3J0cyBIVFRQIGZvciB3ZWJzaXRlIGJ1Y2tldHNcbiAgICAgICAgLi4ucHJvcHMsXG4gICAgICB9KSA6XG4gICAgICBuZXcgUzNCdWNrZXRPcmlnaW4oYnVja2V0LCBwcm9wcyk7XG4gIH1cblxuICBwdWJsaWMgYmluZChzY29wZTogY2RrLkNvbnN0cnVjdCwgb3B0aW9uczogY2xvdWRmcm9udC5PcmlnaW5CaW5kT3B0aW9ucyk6IGNsb3VkZnJvbnQuT3JpZ2luQmluZENvbmZpZyB7XG4gICAgcmV0dXJuIHRoaXMub3JpZ2luLmJpbmQoc2NvcGUsIG9wdGlvbnMpO1xuICB9XG59XG5cbi8qKlxuICogQW4gT3JpZ2luIHNwZWNpZmljIHRvIGEgUzMgYnVja2V0IChub3QgY29uZmlndXJlZCBmb3Igd2Vic2l0ZSBob3N0aW5nKS5cbiAqXG4gKiBDb250YWlucyBhZGRpdGlvbmFsIGxvZ2ljIGFyb3VuZCBidWNrZXQgcGVybWlzc2lvbnMgYW5kIG9yaWdpbiBhY2Nlc3MgaWRlbnRpdGllcy5cbiAqL1xuY2xhc3MgUzNCdWNrZXRPcmlnaW4gZXh0ZW5kcyBjbG91ZGZyb250Lk9yaWdpbkJhc2Uge1xuICBwcml2YXRlIG9yaWdpbkFjY2Vzc0lkZW50aXR5ITogY2xvdWRmcm9udC5JT3JpZ2luQWNjZXNzSWRlbnRpdHk7XG5cbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBidWNrZXQ6IHMzLklCdWNrZXQsIHsgb3JpZ2luQWNjZXNzSWRlbnRpdHksIC4uLnByb3BzIH06IFMzT3JpZ2luUHJvcHMpIHtcbiAgICBzdXBlcihidWNrZXQuYnVja2V0UmVnaW9uYWxEb21haW5OYW1lLCBwcm9wcyk7XG4gICAgaWYgKG9yaWdpbkFjY2Vzc0lkZW50aXR5KSB7XG4gICAgICB0aGlzLm9yaWdpbkFjY2Vzc0lkZW50aXR5ID0gb3JpZ2luQWNjZXNzSWRlbnRpdHk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIGJpbmQoc2NvcGU6IGNkay5Db25zdHJ1Y3QsIG9wdGlvbnM6IGNsb3VkZnJvbnQuT3JpZ2luQmluZE9wdGlvbnMpOiBjbG91ZGZyb250Lk9yaWdpbkJpbmRDb25maWcge1xuICAgIGlmICghdGhpcy5vcmlnaW5BY2Nlc3NJZGVudGl0eSkge1xuICAgICAgLy8gVXNpbmcgYSBidWNrZXQgZnJvbSBhbm90aGVyIHN0YWNrIGNyZWF0ZXMgYSBjeWNsaWMgcmVmZXJlbmNlIHdpdGhcbiAgICAgIC8vIHRoZSBidWNrZXQgdGFraW5nIGEgZGVwZW5kZW5jeSBvbiB0aGUgZ2VuZXJhdGVkIFMzQ2Fub25pY2FsVXNlcklkIHdoZW4gYGdyYW50UmVhZGAgaXMgY2FsbGVkLFxuICAgICAgLy8gYW5kIHRoZSBkaXN0cmlidXRpb24gaGF2aW5nIGEgZGVwZW5kZW5jeSBvbiB0aGUgYnVja2V0J3MgZG9tYWluIG5hbWUuXG4gICAgICAvLyBGaXggdGhpcyBieSBwYXJlbnRpbmcgdGhlIE9BSSBpbiB0aGUgYnVja2V0J3Mgc3RhY2sgd2hlbiBjcm9zcy1zdGFjayB1c2FnZSBpcyBkZXRlY3RlZC5cbiAgICAgIGNvbnN0IGJ1Y2tldFN0YWNrID0gY2RrLlN0YWNrLm9mKHRoaXMuYnVja2V0KTtcbiAgICAgIGNvbnN0IGJ1Y2tldEluRGlmZmVyZW50U3RhY2sgPSBidWNrZXRTdGFjayAhPT0gY2RrLlN0YWNrLm9mKHNjb3BlKTtcbiAgICAgIGNvbnN0IG9haVNjb3BlID0gYnVja2V0SW5EaWZmZXJlbnRTdGFjayA/IGJ1Y2tldFN0YWNrIDogc2NvcGU7XG4gICAgICBjb25zdCBvYWlJZCA9IGJ1Y2tldEluRGlmZmVyZW50U3RhY2sgPyBgJHtjZGsuTmFtZXMudW5pcXVlSWQoc2NvcGUpfVMzT3JpZ2luYCA6ICdTM09yaWdpbic7XG5cbiAgICAgIHRoaXMub3JpZ2luQWNjZXNzSWRlbnRpdHkgPSBuZXcgY2xvdWRmcm9udC5PcmlnaW5BY2Nlc3NJZGVudGl0eShvYWlTY29wZSwgb2FpSWQsIHtcbiAgICAgICAgY29tbWVudDogYElkZW50aXR5IGZvciAke29wdGlvbnMub3JpZ2luSWR9YCxcbiAgICAgIH0pO1xuICAgICAgdGhpcy5idWNrZXQuZ3JhbnRSZWFkKHRoaXMub3JpZ2luQWNjZXNzSWRlbnRpdHkpO1xuICAgIH1cbiAgICByZXR1cm4gc3VwZXIuYmluZChzY29wZSwgb3B0aW9ucyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgcmVuZGVyUzNPcmlnaW5Db25maWcoKTogY2xvdWRmcm9udC5DZm5EaXN0cmlidXRpb24uUzNPcmlnaW5Db25maWdQcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHsgb3JpZ2luQWNjZXNzSWRlbnRpdHk6IGBvcmlnaW4tYWNjZXNzLWlkZW50aXR5L2Nsb3VkZnJvbnQvJHt0aGlzLm9yaWdpbkFjY2Vzc0lkZW50aXR5Lm9yaWdpbkFjY2Vzc0lkZW50aXR5TmFtZX1gIH07XG4gIH1cbn1cbiJdfQ==