"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_sdk_1 = require("aws-sdk"); // eslint-disable-line import/no-extraneous-dependencies
const got_1 = require("got");
const secretsmanager = new aws_sdk_1.SecretsManager({ apiVersion: '2017-10-17' });
const slackClient = got_1.default.extend({
    prefixUrl: 'https://slack.com/api',
});
async function handler(event) {
    console.log('Event: %j', event);
    const data = await secretsmanager.getSecretValue({
        SecretId: event.ResourceProperties.configurationTokenSecretArn,
    }).promise();
    if (!data.SecretString) {
        throw new Error('No secret string found in configuration token secret');
    }
    const secret = JSON.parse(data.SecretString);
    let accessToken = secret.accessToken;
    if (!accessToken || isExpired(secret.exp ?? 0)) {
        if (!secret.refreshToken) {
            throw new Error('No refresh token found in configuration token secret');
        }
        console.log('Refreshing access token');
        const rotate = await slackClient.get('tooling.tokens.rotate', {
            searchParams: { refresh_token: secret.refreshToken },
        }).json();
        if (!rotate.ok) {
            throw new Error(`Failed to refresh access token: ${rotate.error}`);
        }
        console.log('Access token refreshed');
        accessToken = rotate.token;
        console.log('Saving access token');
        const putSecretValue = await secretsmanager.putSecretValue({
            SecretId: event.ResourceProperties.configurationTokenSecretArn,
            SecretString: JSON.stringify({
                accessToken,
                refreshToken: rotate.refresh_token,
                exp: rotate.exp,
            }),
        }).promise();
        console.log(`Successfully saved access token in secret ${putSecretValue.ARN}`);
    }
    const operation = event.RequestType.toLowerCase();
    const request = getManifestRequest(event);
    console.log(`Calling ${operation} manifest API: %j`, request);
    const response = await slackClient.post(`apps.manifest.${operation}`, {
        headers: { Authorization: `Bearer ${accessToken}` },
        json: request,
    }).json();
    if (!response.ok) {
        const errors = response.errors
            ? response.errors.map((err) => `${err.message} at ${err.pointer}`).join('\n')
            : '';
        throw new Error(`Failed to ${operation} manifest: ${response.error}.${errors ? `\n${errors}}` : ''}`);
    }
    console.log(`Successfully ${operation}d Slack app ${event.PhysicalResourceId ?? response.app_id}`);
    if (event.RequestType === 'Create' && response.credentials) {
        console.log('Saving app credentials');
        const putSecretValue = await secretsmanager.putSecretValue({
            SecretId: event.ResourceProperties.credentialsSecretArn,
            SecretString: JSON.stringify({
                appId: response.app_id,
                clientId: response.credentials.client_id,
                clientSecret: response.credentials.client_secret,
                verificationToken: response.credentials.verification_token,
                signingSecret: response.credentials.signing_secret,
            }),
        }).promise();
        console.log(`Successfully saved app credentials in secret ${putSecretValue.ARN}`);
    }
    return {
        PhysicalResourceId: response.app_id,
        Data: {
            appId: response.app_id,
        },
    };
}
exports.handler = handler;
function isExpired(iat) {
    return (iat - (Date.now() / 1000)) < 0;
}
function getManifestRequest(event) {
    switch (event.RequestType) {
        case 'Create':
            return {
                manifest: event.ResourceProperties.manifest,
            };
        case 'Update':
            return {
                app_id: event.PhysicalResourceId,
                manifest: event.ResourceProperties.manifest,
            };
        case 'Delete':
            return {
                app_id: event.PhysicalResourceId,
            };
    }
}
//# sourceMappingURL=data:application/json;base64,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