# MIT License
#
# Copyright (c) 2018 Evgeny Medvedev, evge.medvedev@gmail.com
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the 'Software'), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.


import pytest

from klaytnetl.service.klaytn_contract_service import KlaytnContractService


@pytest.mark.parametrize('bytecode,expected_sighashes,is_erc20,is_erc721', [
    ( #0x968d93a44b3ef61168ca621a59ddc0e56583e592, Kai Bond Token
            '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',
            ['0x06fdde03', '0x095ea7b3', '0x18160ddd', '0x23b872dd', '0x29605e77', '0x313ce567', '0x39509351', '0x40c10f19', '0x42966c68', '0x4456eda2', '0x570ca735', '0x656f7065', '0x70a08231', '0x715018a6', '0x79cc6790', '0x8da5cb5b', '0x95d89b41', '0xa457c2d7', '0xa9059cbb', '0xdd62ed3e', '0xf2fde38b', '0xffffffff'],
            True, False),
    ( #0xe3656452c8238334efdfc811d6f98e5962fe4461, CARTO
            '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',
            ['0x01ffc9a7', '0x05f5e100', '0x06fdde03', '0x081812fc', '0x095ea7b3', '0x150b7a02', '0x16f02173', '0x18160ddd', '0x23b872dd', '0x2851c719', '0x2f745c59', '0x42842e0e', '0x44900d2b', '0x4f6ccce7', '0x518c616d', '0x5f0bccf4', '0x6352211e', '0x6758737b', '0x69f71fc8', '0x6f6e7472', '0x70a08231', '0x7c471e91', '0x887a551f', '0x95d89b41', '0xa22cb465', '0xb88d4fde', '0xc6c22d59', '0xc87b56dd', '0xe985e9c5', '0xffffffff'],
            False, True),
    ( # 0xaff706fb4f076a0a8ebfc56827a22cedafced05f
            '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',
            ['0x10693fcd', '0x13a6738a', '0x13e7c9d8', '0x1a2ae53e', '0x1ebdca38', '0x2014e5d1', '0x21421707', '0x22604742', '0x23b872dd', '0x26c23b54', '0x2f88396c', '0x3682a450', '0x3a3099d1', '0x3a348533', '0x3e4fe949', '0x407e6bae', '0x40c10f19', '0x42842e0e', '0x42966c68', '0x4739f7e5', '0x488af871', '0x48a18a6a', '0x4b40b826', '0x50bb4e7f', '0x54edad72', '0x5526f76b', '0x5eb7413a', '0x6e176ec2', '0x715018a6', '0x75ebdc09', '0x7c1a0302', '0x87b04c55', '0x8a75eee2', '0x8c0bd916', '0x8da5cb5b', '0x8f32d59b', '0x9832c1d7', '0x989ba0d3', '0x9ef2017b', '0x9f071329', '0xa066a7ed', '0xa9059cbb', '0xac6fff0b', '0xafb60223', '0xb2c01030', '0xb3f00674', '0xbab2af1d', '0xc263b5d6', '0xc877cf37', '0xcb38f407', '0xcf0da290', '0xd8cf98ca', '0xdd9222d6', '0xea21eade', '0xee2aec65', '0xefdcd974', '0xf1656e53', '0xf1719966', '0xf2fde38b', '0xffa1ad74', '0xffffffff'],
            False, False)

])
def test_get_function_sighashes(bytecode, expected_sighashes, is_erc20, is_erc721):
    eth_contract_service = KlaytnContractService()
    sighashes = eth_contract_service.get_function_sighashes(bytecode)
    assert expected_sighashes == sighashes
    assert eth_contract_service.is_erc20_contract(sighashes) == is_erc20
    assert eth_contract_service.is_erc721_contract(sighashes) == is_erc721
