# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['autoclick',
 'autoclick.composites',
 'autoclick.types',
 'autoclick.validations']

package_data = \
{'': ['*']}

install_requires = \
['click==7.0', 'docparse']

entry_points = \
{'autoclick': ['composites = autoclick.composites',
               'validations = autoclick.validations']}

setup_kwargs = {
    'name': 'autoclick',
    'version': '0.8.1',
    'description': 'Auto-generate Click-based CLIs from python3 type annotations.',
    'long_description': '# AutoClick\n\nAutoClick creates Click-based CLIs using type annotations.\n\nThe simplest use of AutoClick requires annotating your main method with `@autoclick.command`:\n\n```python\n# test.py\nimport autoclick\n\n@autoclick.command("greet")\ndef main(greeting: str, name: str):\n    print(f"{greeting} {name}")\n\nif __name__ == "__main__":\n    main()\n```\n\n```bash\n$ python test.py --help\nUsage: test.py [OPTIONS] [GREETING] [NAME]\n\nOptions:\n  --help  Show this message and exit.\n```\n\nFor additional customization, keyword arguments can be passed to the command decorator:\n\n```python\nimport autoclick\n\n@autoclick.command(\n    short_names={\n        "greeting": "G",\n        "name": "n"\n    },\n    show_defaults=True\n)\ndef main(greeting: str = "hello", name: str = "human"):\n    print(f"{greeting} {name}")\n```\n\n```bash\n$ python test.py --help\nUsage: test.py [OPTIONS]\n\nOptions:\n  -G, --greeting TEXT  [default: hello]\n  -n, --name TEXT      [default: human]\n  --help               Show this message and exit.\n```\n\n## Type conversion\n\nIn Click, type conversion can be done either in a callback or by using a callable type (such as a subclass of ParamType) as the type. In AutoClick, type conversions are performed automatically based on type annotations for callable types. However, for more complex type conversion, there are three additional methods:\n\n1. Automatic conversion functions. A conversion function is decorated by `@conversion`. The conversion decorator by default infers the type being converted to from the return annotation. Otherwise, the destination type can be specified as an argument to the decorator. The decorator registers the function as the converter for the specified type. When that type is encountered as an annotation of a parameter to a command function, the converter function is used to convert the string argument to that type.\n\n```python\nimport autoclick\n\nclass Bork:\n    def __init__(self, n: int):\n        self.n = n\n\n    def __str__(self):\n        print(",".join(["bork"] * self.n))\n\n@autoclick.conversion()\ndef bork(n: int) -> Bork:\n    return Bork(n)\n\n@autoclick.command("bork")\ndef main(b: Bork):\n    print(b)\n```\n\nIn the case where there needs to be specialized handling of common types, new types can be created using `typing.NewType`:\n\n```python\nimport autoclick\nimport typing\n\nDoubleInt = typing.NewType("DoubleInt", int)\n\n@autoclick.conversion(DoubleInt)\ndef double_int(i: str):\n    return int(i) * 2\n\n@autoclick.command("double")\ndef main(i1: int, i2: DoubleInt):\n    print(i1, i2)\n```\n\n2. Conversion functions can also be specified explicitly in the command decorator:\n\n```python\nimport autoclick\n\n@autoclick.command(\n    types={\n        "a": double_int\n    }\n)\ndef main(a: int):\n    print(a)\n```\n\nNote that any of the types in the `click.Types` package can also be used in this way.\n\n3. For composite types, the `autoclick.composite_type` and `autoclick.composite_factory` functions can be used. An example of a composite type is a class that requires more than one parameter to its constructor. For composite types, the same annotation-based CLI creation is performed, and the parameters are injected into the CLI in place of the composite parameter.\n\n```python\nimport autoclick\n@autoclick.composite_type()\nclass Foo:\n    def __init__(self, bar: str, baz: int):\n        self.bar = bar\n        self.baz = baz\n\n@autoclick.command()\ndef main(foo: Foo):\n    print(foo.bar, foo.baz)\n```\n\nIn this case, the options to main would be `--foo-bar` and `--foo-baz`. Once the CLI is processed, the values of these options are used to construct the `Foo` instance, which is then passed to the call to `main`. The parameter name in the command function is prepended to the parameter names of the composite type, so that a composite type can be used multiple types in a command function signature.\n\nA `autoclick.composite_factory` function returns a complex type. For example, the code below is equivalent to the code above:\n\n```python\nimport autoclick\n\nclass Foo:\n    def __init__(self, bar: str, baz: int):\n        ...\n\n@autoclick.composite_factory(Foo)\ndef foo_factory(bar: str, baz: int):\n    return Foo(bar, baz)\n```\n\n## Conditionals and Validations\n\nConditionals and Validations are similar - they are both decorators that take `**kwargs` parameter. The keywords are parameter names and values are parameter values. When the function takes multiple parameters, they should specify the order; ordering depends on python 3.5+ behavior that dictionaries are ordered implicitly.\n\nA conditional function is used to modify the values of one or more parameters conditional on the value(s) of other parameters. A conditional function may return a dict with keys being parameter names that should be updated, and values being the new parameter values.\n\nA validation function is intended to check that one or more parameter values conform to certain restrictions. The return value of a validation function is ignored.\n\nBoth conditional and validation functions can throw ValidationError.\n\nThese functions can be associated with parameters in two ways. First, using the `conditionals` and `validations` arguments of the command decorator. These are dicts with a parameter name or tuple of parameter names being the key and the function being the value. Second, validation functions can be associated with parameters when they are decorated with `@autoclick.validation` and the parameter type matches the type argument of the validation decorator. Multi-parameter validations can only be associated via the first method. Since conditionals are expected to be multi-valued, there is no `@autoclick.conditional` annotation, i.e. they must always be explicitly specified.\n\n### Type matching\n\nYou can also use distinct types created by the `typing.NewType` function for type matching validations. For example, if you want to define a parameter that must be positive and even:\n\n```python\nimport autoclick\nfrom typing import NewType\n\nPositiveEven = NewType(\'PositiveEven\', int)\n\n@autoclick.validation(PositiveEven)\ndef validate_positive_even(arg: int):\n  if arg < 0:\n    raise autoclick.ValidationError()\n  if arg % 2 != 0:\n    raise autoclick.ValidationError()\n```\n\nNote that the typing library does not currently provide an intersection type. Thus, Positive, Even, and PositiveEven must all be distinct validations. There are two ways to simplify:\n\n1. Add the parameter to the validation dict of the command decorator with a tuple of mutliple functions as the value:\n\n```python\nimport autoclick\n\n@autoclick.command(\n    validations={\n        "a": positive, even\n    }\n)\n```\n\n2. Create a composite validation:\n\n```python\nimport autoclick\n\n@autoclick.validation(PositiveEven, (positive, even))\ndef validate_positive_even(arg: int):\n  pass\n```\n\nor even\n\n```python\nimport autoclick\nautoclick.validation(PositiveEven, (positive, even))\n```\n\n### Docstring utilization\n\nAutoClick uses the [docparse](https://github.com/jdidion/docparse) library to parse the docstrings of command functions and composites to extract help text. Note that currently docparse only supports Google-style docstrings.\n\n```python\n# test.py\nimport autoclick\n\n@autoclick.command(show_defaults=True)\ndef main(x: str = "hello"):\n    """Print a string\n\n    Args:\n        x: The string to print.\n    """\n    print(x)\n\nif __name__ == "__main__":\n    main()\n```\n\n```bash\n$ python test.py --help\nUsage: test.py [OPTIONS] [X]\n\n  Print a string\n\nOptions:\n  -x, --x TEXT  The string to print.  [default: hello]\n  --help        Show this message and exit.\n```\n\n## Groups\n\nAutoClick CLIs can have multiple subcommands, by creating command groups:\n\n```python\nfrom autoclick import group\n\n@group("myprog")\ndef myprog():\n    pass\n\n@myprog.command("hi")\ndef say_hi(name: str):\n    print(f"hello {name}")\n    \n@myprog.command("bye")\ndef say_bye(name: str):\n    print(f"byebye {name}")\n```\n\nAutoClick provides alternative group types. For example, `DefaultAutoClickGroup` can take a default command name to run when a command is not specified:\n\n```python\nfrom autoclick import group\nfrom autoclick.commands import DefaultAutoClickGroup\n\n@group(\n    "myprog",\n    group_class=DefaultAutoClickGroup,\n    default="hello"\n)\ndef myprog():\n    pass\n\n# This command is run by default if the command name is not specified\ndef hello(name: str):\n    print(f"hi {name}")\n```\n\n## Installation\n\n```bash\n$ pip intall autoclick\n```\n\n## Runtime Dependencies\n\n* Python 3.6+\n* docparse\n\n## Build dependencies\n\n* poetry 0.12+\n* pytest (with pytest-cov plugin)\n\n## Details\n\n### Option attribute inference\n\nThe following sections describe details of how the arguments to click classes/functions are inferred from the type and docstring information:\n\n#### All Parameters\n\n* name (long): parameter name; underscores converted to dashes unless keep_underscores=True in the command decorator.\n* name (short): starting from the left-most character of the parameter name, the first character that is not used by another parameter or by any built-in; can be overridden by specifying the \'parameter_names\' dictionary in the command decorator.\n* type: inferred from the type hint; if type hint is missing, inferred from the default value; if default value is missing, str.\n* required: by default, true for positional arguments (Arguments) and false for keyword arguments (Options); if positionals_as_options=True in the command decorator, positional arguments are instead required Options. Required keyword arguments can be specified in the \'required\' list in the command decorator.\n* default: unset for positional arguments, keyword value for keyword arguments.\n* nargs: 1 unless type is Tuple (in which case nargs is the number of arguments to the Tuple).\n\n#### Option-only\n\n* hide_input: False unless the command \'hidden\' parameter is specified and includes the parameter name.\n* is_flag: True for keyword arguments of type boolean; assumed to be the True option unless the name starts with \'no\'; the other option will always be inferred by adding/removing \'no-\'\n* multiple: True for sequence types\n* help: Parsed from docstring.\n\n## Todo\n\n* Documentation for positional arguments (see [](https://github.com/pallets/click/issues/587))\n* Handle return values (e.g. if a int, treat as a return code; if a dict, convert to JSON and write to stdout, etc)\n* Look at incorporating features from contributed packages: [](https://github.com/click-contrib)',
    'author': 'John Didion',
    'author_email': 'github@didion.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/jdidion/autoclick',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
