# -*- coding: utf-8 -*-
# Copyright 2022 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
from typing import Awaitable, Callable, Dict, Optional, Sequence, Tuple, Union
import warnings

from google.api_core import gapic_v1, grpc_helpers_async
from google.auth import credentials as ga_credentials  # type: ignore
from google.auth.transport.grpc import SslCredentials  # type: ignore
from google.protobuf import empty_pb2  # type: ignore
import grpc  # type: ignore
from grpc.experimental import aio  # type: ignore

from google.cloud.accessapproval_v1.types import accessapproval

from .base import DEFAULT_CLIENT_INFO, AccessApprovalTransport
from .grpc import AccessApprovalGrpcTransport


class AccessApprovalGrpcAsyncIOTransport(AccessApprovalTransport):
    """gRPC AsyncIO backend transport for AccessApproval.

    This API allows a customer to manage accesses to cloud resources by
    Google personnel. It defines the following resource model:

    -  The API has a collection of
       [ApprovalRequest][google.cloud.accessapproval.v1.ApprovalRequest]
       resources, named ``approvalRequests/{approval_request}``
    -  The API has top-level settings per Project/Folder/Organization,
       named ``accessApprovalSettings``

    The service also periodically emails a list of recipients, defined
    at the Project/Folder/Organization level in the
    accessApprovalSettings, when there is a pending ApprovalRequest for
    them to act on. The ApprovalRequests can also optionally be
    published to a Pub/Sub topic owned by the customer (contact support
    if you would like to enable Pub/Sub notifications).

    ApprovalRequests can be approved or dismissed. Google personnel can
    only access the indicated resource or resources if the request is
    approved (subject to some exclusions:
    https://cloud.google.com/access-approval/docs/overview#exclusions).

    Note: Using Access Approval functionality will mean that Google may
    not be able to meet the SLAs for your chosen products, as any
    support response times may be dramatically increased. As such the
    SLAs do not apply to any service disruption to the extent impacted
    by Customer's use of Access Approval. Do not enable Access Approval
    for projects where you may require high service availability and
    rapid response by Google Cloud Support.

    After a request is approved or dismissed, no further action may be
    taken on it. Requests with the requested_expiration in the past or
    with no activity for 14 days are considered dismissed. When an
    approval expires, the request is considered dismissed.

    If a request is not approved or dismissed, we call it pending.

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends protocol buffers over the wire using gRPC (which is built on
    top of HTTP/2); the ``grpcio`` package must be installed.
    """

    _grpc_channel: aio.Channel
    _stubs: Dict[str, Callable] = {}

    @classmethod
    def create_channel(
        cls,
        host: str = "accessapproval.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        quota_project_id: Optional[str] = None,
        **kwargs,
    ) -> aio.Channel:
        """Create and return a gRPC AsyncIO channel object.
        Args:
            host (Optional[str]): The host for the channel to use.
            credentials (Optional[~.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify this application to the service. If
                none are specified, the client will attempt to ascertain
                the credentials from the environment.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            kwargs (Optional[dict]): Keyword arguments, which are passed to the
                channel creation.
        Returns:
            aio.Channel: A gRPC AsyncIO channel object.
        """

        return grpc_helpers_async.create_channel(
            host,
            credentials=credentials,
            credentials_file=credentials_file,
            quota_project_id=quota_project_id,
            default_scopes=cls.AUTH_SCOPES,
            scopes=scopes,
            default_host=cls.DEFAULT_HOST,
            **kwargs,
        )

    def __init__(
        self,
        *,
        host: str = "accessapproval.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        channel: Optional[aio.Channel] = None,
        api_mtls_endpoint: Optional[str] = None,
        client_cert_source: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        ssl_channel_credentials: Optional[grpc.ChannelCredentials] = None,
        client_cert_source_for_mtls: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        quota_project_id: Optional[str] = None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
        api_audience: Optional[str] = None,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to.
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.
                This argument is ignored if ``channel`` is provided.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            channel (Optional[aio.Channel]): A ``Channel`` instance through
                which to make calls.
            api_mtls_endpoint (Optional[str]): Deprecated. The mutual TLS endpoint.
                If provided, it overrides the ``host`` argument and tries to create
                a mutual TLS channel with client SSL credentials from
                ``client_cert_source`` or application default SSL credentials.
            client_cert_source (Optional[Callable[[], Tuple[bytes, bytes]]]):
                Deprecated. A callback to provide client SSL certificate bytes and
                private key bytes, both in PEM format. It is ignored if
                ``api_mtls_endpoint`` is None.
            ssl_channel_credentials (grpc.ChannelCredentials): SSL credentials
                for the grpc channel. It is ignored if ``channel`` is provided.
            client_cert_source_for_mtls (Optional[Callable[[], Tuple[bytes, bytes]]]):
                A callback to provide client certificate bytes and private key bytes,
                both in PEM format. It is used to configure a mutual TLS channel. It is
                ignored if ``channel`` or ``ssl_channel_credentials`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you're developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.

        Raises:
            google.auth.exceptions.MutualTlsChannelError: If mutual TLS transport
              creation failed for any reason.
          google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """
        self._grpc_channel = None
        self._ssl_channel_credentials = ssl_channel_credentials
        self._stubs: Dict[str, Callable] = {}

        if api_mtls_endpoint:
            warnings.warn("api_mtls_endpoint is deprecated", DeprecationWarning)
        if client_cert_source:
            warnings.warn("client_cert_source is deprecated", DeprecationWarning)

        if channel:
            # Ignore credentials if a channel was passed.
            credentials = False
            # If a channel was explicitly provided, set it.
            self._grpc_channel = channel
            self._ssl_channel_credentials = None
        else:
            if api_mtls_endpoint:
                host = api_mtls_endpoint

                # Create SSL credentials with client_cert_source or application
                # default SSL credentials.
                if client_cert_source:
                    cert, key = client_cert_source()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )
                else:
                    self._ssl_channel_credentials = SslCredentials().ssl_credentials

            else:
                if client_cert_source_for_mtls and not ssl_channel_credentials:
                    cert, key = client_cert_source_for_mtls()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )

        # The base transport sets the host, credentials and scopes
        super().__init__(
            host=host,
            credentials=credentials,
            credentials_file=credentials_file,
            scopes=scopes,
            quota_project_id=quota_project_id,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
            api_audience=api_audience,
        )

        if not self._grpc_channel:
            self._grpc_channel = type(self).create_channel(
                self._host,
                # use the credentials which are saved
                credentials=self._credentials,
                # Set ``credentials_file`` to ``None`` here as
                # the credentials that we saved earlier should be used.
                credentials_file=None,
                scopes=self._scopes,
                ssl_credentials=self._ssl_channel_credentials,
                quota_project_id=quota_project_id,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )

        # Wrap messages. This must be done after self._grpc_channel exists
        self._prep_wrapped_messages(client_info)

    @property
    def grpc_channel(self) -> aio.Channel:
        """Create the channel designed to connect to this service.

        This property caches on the instance; repeated calls return
        the same channel.
        """
        # Return the channel from cache.
        return self._grpc_channel

    @property
    def list_approval_requests(
        self,
    ) -> Callable[
        [accessapproval.ListApprovalRequestsMessage],
        Awaitable[accessapproval.ListApprovalRequestsResponse],
    ]:
        r"""Return a callable for the list approval requests method over gRPC.

        Lists approval requests associated with a project,
        folder, or organization. Approval requests can be
        filtered by state (pending, active, dismissed). The
        order is reverse chronological.

        Returns:
            Callable[[~.ListApprovalRequestsMessage],
                    Awaitable[~.ListApprovalRequestsResponse]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_approval_requests" not in self._stubs:
            self._stubs["list_approval_requests"] = self.grpc_channel.unary_unary(
                "/google.cloud.accessapproval.v1.AccessApproval/ListApprovalRequests",
                request_serializer=accessapproval.ListApprovalRequestsMessage.serialize,
                response_deserializer=accessapproval.ListApprovalRequestsResponse.deserialize,
            )
        return self._stubs["list_approval_requests"]

    @property
    def get_approval_request(
        self,
    ) -> Callable[
        [accessapproval.GetApprovalRequestMessage],
        Awaitable[accessapproval.ApprovalRequest],
    ]:
        r"""Return a callable for the get approval request method over gRPC.

        Gets an approval request. Returns NOT_FOUND if the request does
        not exist.

        Returns:
            Callable[[~.GetApprovalRequestMessage],
                    Awaitable[~.ApprovalRequest]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_approval_request" not in self._stubs:
            self._stubs["get_approval_request"] = self.grpc_channel.unary_unary(
                "/google.cloud.accessapproval.v1.AccessApproval/GetApprovalRequest",
                request_serializer=accessapproval.GetApprovalRequestMessage.serialize,
                response_deserializer=accessapproval.ApprovalRequest.deserialize,
            )
        return self._stubs["get_approval_request"]

    @property
    def approve_approval_request(
        self,
    ) -> Callable[
        [accessapproval.ApproveApprovalRequestMessage],
        Awaitable[accessapproval.ApprovalRequest],
    ]:
        r"""Return a callable for the approve approval request method over gRPC.

        Approves a request and returns the updated ApprovalRequest.

        Returns NOT_FOUND if the request does not exist. Returns
        FAILED_PRECONDITION if the request exists but is not in a
        pending state.

        Returns:
            Callable[[~.ApproveApprovalRequestMessage],
                    Awaitable[~.ApprovalRequest]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "approve_approval_request" not in self._stubs:
            self._stubs["approve_approval_request"] = self.grpc_channel.unary_unary(
                "/google.cloud.accessapproval.v1.AccessApproval/ApproveApprovalRequest",
                request_serializer=accessapproval.ApproveApprovalRequestMessage.serialize,
                response_deserializer=accessapproval.ApprovalRequest.deserialize,
            )
        return self._stubs["approve_approval_request"]

    @property
    def dismiss_approval_request(
        self,
    ) -> Callable[
        [accessapproval.DismissApprovalRequestMessage],
        Awaitable[accessapproval.ApprovalRequest],
    ]:
        r"""Return a callable for the dismiss approval request method over gRPC.

        Dismisses a request. Returns the updated ApprovalRequest.

        NOTE: This does not deny access to the resource if another
        request has been made and approved. It is equivalent in effect
        to ignoring the request altogether.

        Returns NOT_FOUND if the request does not exist.

        Returns FAILED_PRECONDITION if the request exists but is not in
        a pending state.

        Returns:
            Callable[[~.DismissApprovalRequestMessage],
                    Awaitable[~.ApprovalRequest]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "dismiss_approval_request" not in self._stubs:
            self._stubs["dismiss_approval_request"] = self.grpc_channel.unary_unary(
                "/google.cloud.accessapproval.v1.AccessApproval/DismissApprovalRequest",
                request_serializer=accessapproval.DismissApprovalRequestMessage.serialize,
                response_deserializer=accessapproval.ApprovalRequest.deserialize,
            )
        return self._stubs["dismiss_approval_request"]

    @property
    def invalidate_approval_request(
        self,
    ) -> Callable[
        [accessapproval.InvalidateApprovalRequestMessage],
        Awaitable[accessapproval.ApprovalRequest],
    ]:
        r"""Return a callable for the invalidate approval request method over gRPC.

        Invalidates an existing ApprovalRequest. Returns the updated
        ApprovalRequest.

        NOTE: This does not deny access to the resource if another
        request has been made and approved. It only invalidates a single
        approval.

        Returns FAILED_PRECONDITION if the request exists but is not in
        an approved state.

        Returns:
            Callable[[~.InvalidateApprovalRequestMessage],
                    Awaitable[~.ApprovalRequest]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "invalidate_approval_request" not in self._stubs:
            self._stubs["invalidate_approval_request"] = self.grpc_channel.unary_unary(
                "/google.cloud.accessapproval.v1.AccessApproval/InvalidateApprovalRequest",
                request_serializer=accessapproval.InvalidateApprovalRequestMessage.serialize,
                response_deserializer=accessapproval.ApprovalRequest.deserialize,
            )
        return self._stubs["invalidate_approval_request"]

    @property
    def get_access_approval_settings(
        self,
    ) -> Callable[
        [accessapproval.GetAccessApprovalSettingsMessage],
        Awaitable[accessapproval.AccessApprovalSettings],
    ]:
        r"""Return a callable for the get access approval settings method over gRPC.

        Gets the settings associated with a project, folder,
        or organization.

        Returns:
            Callable[[~.GetAccessApprovalSettingsMessage],
                    Awaitable[~.AccessApprovalSettings]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_access_approval_settings" not in self._stubs:
            self._stubs["get_access_approval_settings"] = self.grpc_channel.unary_unary(
                "/google.cloud.accessapproval.v1.AccessApproval/GetAccessApprovalSettings",
                request_serializer=accessapproval.GetAccessApprovalSettingsMessage.serialize,
                response_deserializer=accessapproval.AccessApprovalSettings.deserialize,
            )
        return self._stubs["get_access_approval_settings"]

    @property
    def update_access_approval_settings(
        self,
    ) -> Callable[
        [accessapproval.UpdateAccessApprovalSettingsMessage],
        Awaitable[accessapproval.AccessApprovalSettings],
    ]:
        r"""Return a callable for the update access approval
        settings method over gRPC.

        Updates the settings associated with a project, folder, or
        organization. Settings to update are determined by the value of
        field_mask.

        Returns:
            Callable[[~.UpdateAccessApprovalSettingsMessage],
                    Awaitable[~.AccessApprovalSettings]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_access_approval_settings" not in self._stubs:
            self._stubs[
                "update_access_approval_settings"
            ] = self.grpc_channel.unary_unary(
                "/google.cloud.accessapproval.v1.AccessApproval/UpdateAccessApprovalSettings",
                request_serializer=accessapproval.UpdateAccessApprovalSettingsMessage.serialize,
                response_deserializer=accessapproval.AccessApprovalSettings.deserialize,
            )
        return self._stubs["update_access_approval_settings"]

    @property
    def delete_access_approval_settings(
        self,
    ) -> Callable[
        [accessapproval.DeleteAccessApprovalSettingsMessage], Awaitable[empty_pb2.Empty]
    ]:
        r"""Return a callable for the delete access approval
        settings method over gRPC.

        Deletes the settings associated with a project,
        folder, or organization. This will have the effect of
        disabling Access Approval for the project, folder, or
        organization, but only if all ancestors also have Access
        Approval disabled. If Access Approval is enabled at a
        higher level of the hierarchy, then Access Approval will
        still be enabled at this level as the settings are
        inherited.

        Returns:
            Callable[[~.DeleteAccessApprovalSettingsMessage],
                    Awaitable[~.Empty]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_access_approval_settings" not in self._stubs:
            self._stubs[
                "delete_access_approval_settings"
            ] = self.grpc_channel.unary_unary(
                "/google.cloud.accessapproval.v1.AccessApproval/DeleteAccessApprovalSettings",
                request_serializer=accessapproval.DeleteAccessApprovalSettingsMessage.serialize,
                response_deserializer=empty_pb2.Empty.FromString,
            )
        return self._stubs["delete_access_approval_settings"]

    @property
    def get_access_approval_service_account(
        self,
    ) -> Callable[
        [accessapproval.GetAccessApprovalServiceAccountMessage],
        Awaitable[accessapproval.AccessApprovalServiceAccount],
    ]:
        r"""Return a callable for the get access approval service
        account method over gRPC.

        Retrieves the service account that is used by Access
        Approval to access KMS keys for signing approved
        approval requests.

        Returns:
            Callable[[~.GetAccessApprovalServiceAccountMessage],
                    Awaitable[~.AccessApprovalServiceAccount]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_access_approval_service_account" not in self._stubs:
            self._stubs[
                "get_access_approval_service_account"
            ] = self.grpc_channel.unary_unary(
                "/google.cloud.accessapproval.v1.AccessApproval/GetAccessApprovalServiceAccount",
                request_serializer=accessapproval.GetAccessApprovalServiceAccountMessage.serialize,
                response_deserializer=accessapproval.AccessApprovalServiceAccount.deserialize,
            )
        return self._stubs["get_access_approval_service_account"]

    def close(self):
        return self.grpc_channel.close()


__all__ = ("AccessApprovalGrpcAsyncIOTransport",)
