from unittest import TestCase

from hkscs_unicode_converter import converter


class TestValid(TestCase):
    def test_posessive(self):
        res = converter.convert_char("")
        self.assertEqual(res, "嘅")

    def test_correct(self):
        res = converter.convert_char("")
        self.assertEqual(res, "啱")

    def test_blabber(self):
        res = converter.convert_char("")
        self.assertEqual(res, "噏")

    def test_posessive_chr(self):
        res = converter.convert_char(chr(0xECD1))
        self.assertEqual(res, "嘅")

    def test_weird_edge_case(self):
        res = converter.convert_char(chr(0xF327))
        self.assertEqual(res, "Ê̌")

    def test_gccs(self):
        # GCCS was encoded in the PUA of ISO 10646:1993
        # There are too many to test, so just test one
        # Some GCCS codepoints were moved to the BMP/Extensions B in ISO 10646:2000
        res = converter.convert_char(chr(0xE02B))
        self.assertEqual(res, chr(0x3743))

    def test_hkscs1999(self):
        # New characters added to HKSCS-1999
        # Many were also moved to the BMP/Extensions B in ISO 10646:2000, but some stayed in the PUA
        chars = {
            "E0FA": "3D85",
            "E108": "6160",
            "E144": "48AE",
            "E14C": "97BD",
            "E16F": "8424",
            "E194": "9814",
            "E1C6": "3B2B",
            "E1E0": "3D32",
            "E267": "5C9C",
            "E2AC": "3730",
            "E316": "364E",
            "E33C": "3DAD",
            "E347": "667D",
            "E355": "748D",
            "E378": "4503",
            "E3C7": "347E",
            "E411": "63C1",
            "E413": "7777",
            "E47A": "3AF0",
            "E48F": "708F",
            "E526": "6888",
            "E608": "9366",
            "E615": "928F",
            "E64A": "8614",
            "E682": "4190",
            "E6A5": "678F",
            "E6AB": "3863",
            "E776": "39E2",
            "E7C0": "436E",
            "E86B": "492D",
            "E969": "7E65",
            "E96B": "5D2F",
            "E96C": "3DF3",
            "E970": "7DA4",
            "E971": "8426",
            "E972": "5485",
            "E976": "577E",
            "E979": "3FE5",
            "E97C": "7003",
            "E97E": "5D70",
            "E97F": "738F",
            "E983": "4FC8",
            "E984": "7FE7",
            "E985": "72CD",
            "E986": "7310",
            "E988": "7338",
            "E989": "7339",
            "E98B": "7341",
            "E98C": "7348",
            "E98D": "3EA9",
            "E990": "71F5",
            "E992": "73E1",
            "E994": "3ECA",
            "E996": "3ED1",
            "E999": "7419",
            "E99A": "741E",
            "E99B": "741F",
            "E99C": "3EE2",
            "E99D": "3EF0",
            "E99E": "3EF4",
            "E99F": "3EFA",
            "E9A0": "74D3",
            "E9A1": "3F0E",
            "E9A2": "3F53",
            "E9A3": "7542",
            "E9A4": "756D",
            "E9A5": "7572",
            "E9A6": "758D",
            "E9A7": "3F7C",
            "E9A8": "75C8",
            "E9A9": "75DC",
            "E9AA": "3FC0",
            "E9AB": "764D",
            "E9AC": "3FD7",
            "E9AD": "7674",
            "E9AE": "3FDC",
            "E9AF": "767A",
            "E9B1": "7188",
            "E9B2": "5623",
            "E9B3": "8980",
            "E9B5": "401D",
            "E9B6": "7743",
            "E9B7": "4039",
            "E9B9": "4045",
            "E9BA": "35DB",
            "E9BB": "7798",
            "E9BC": "406A",
            "E9BD": "406F",
            "E9BF": "77BE",
            "E9C0": "77CB",
            "E9C2": "7818",
            "E9C4": "781C",
            "E9C5": "40A8",
            "E9C6": "7839",
            "E9C7": "7847",
            "E9C8": "7851",
            "E9C9": "7866",
            "E9CA": "8448",
            "E9CC": "7933",
            "E9CE": "7932",
            "E9CF": "4103",
            "E9D0": "4109",
            "E9D1": "7991",
            "E9D2": "7999",
            "E9D4": "7A06",
            "E9D6": "4167",
            "E9D7": "7A91",
            "E9D8": "41B2",
            "E9D9": "7ABC",
            "E9DA": "8279",
            "E9DB": "41C4",
            "E9DC": "7ACF",
            "E9DD": "7ADB",
            "E9DE": "41CF",
            "E9E0": "7B62",
            "E9E1": "7B6C",
            "E9E2": "7B7B",
            "E9E3": "7C12",
            "E9E4": "7C1B",
            "E9E5": "4260",
            "E9E6": "427A",
            "E9E7": "7C7B",
            "E9E8": "7C9C",
            "E9E9": "428C",
            "E9EA": "7CB8",
            "E9EB": "4294",
            "E9EC": "7CED",
            "E9ED": "8F93",
            "E9EE": "70C0",
            "E9F0": "7DCF",
            "E9F1": "7DD4",
            "E9F2": "7DD0",
            "E9F3": "7DFD",
            "E9F4": "7FAE",
            "E9F5": "7FB4",
            "E9F6": "729F",
            "E9F7": "4397",
            "E9F8": "8020",
            "E9F9": "8025",
            "E9FB": "802E",
            "E9FC": "8031",
            "E9FD": "8054",
            "E9FF": "57B4",
            "EA00": "70A0",
            "EA01": "80B7",
            "EA02": "80E9",
            "EA03": "43ED",
            "EA04": "810C",
            "EA06": "810E",
            "EA07": "8112",
            "EA09": "8114",
            "EA0A": "4401",
            "EA0B": "3B39",
            "EA0C": "8156",
            "EA0D": "8159",
            "EA0E": "815A",
            "EA0F": "4413",
            "EA11": "817C",
            "EA12": "8184",
            "EA13": "4425",
            "EA14": "8193",
            "EA15": "442D",
            "EA16": "81A5",
            "EA17": "57EF",
            "EA18": "81C1",
            "EA19": "81E4",
            "EA1A": "8254",
            "EA1B": "448F",
            "EA1D": "8276",
            "EA1E": "82CA",
            "EA1F": "82D8",
            "EA20": "82FF",
            "EA21": "44B0",
            "EA22": "8357",
            "EA23": "9669",
            "EA25": "8405",
            "EA26": "70F5",
            "EA27": "8464",
            "EA29": "8488",
            "EA2A": "4504",
            "EA2B": "84BE",
            "EA2C": "84E1",
            "EA2D": "84F8",
            "EA2E": "8510",
            "EA2F": "8538",
            "EA30": "8552",
            "EA31": "453B",
            "EA32": "856F",
            "EA33": "8570",
            "EA34": "85E0",
            "EA35": "4577",
            "EA36": "8672",
            "EA37": "8692",
            "EA38": "86B2",
            "EA39": "86EF",
            "EA3A": "9645",
            "EA3B": "878B",
            "EA3C": "4606",
            "EA3D": "4617",
            "EA3E": "88AE",
            "EA3F": "88FF",
            "EA40": "8924",
            "EA41": "8947",
            "EA42": "8991",
            "EA44": "8A29",
            "EA45": "8A38",
            "EA46": "8A94",
            "EA47": "8AB4",
            "EA48": "8C51",
            "EA49": "8CD4",
            "EA4A": "8CF2",
            "EA4B": "8D1C",
            "EA4C": "4798",
            "EA4D": "585F",
            "EA4E": "8DC3",
            "EA4F": "47ED",
            "EA51": "8E3A",
            "EA52": "55D8",
            "EA53": "5754",
            "EA54": "8E71",
            "EA55": "55F5",
            "EA56": "8EB0",
            "EA57": "4837",
            "EA58": "8ECE",
            "EA59": "8EE2",
            "EA5A": "8EE4",
            "EA5B": "8EED",
            "EA5C": "8EF2",
            "EA5D": "8FB7",
            "EA5E": "8FC1",
            "EA5F": "8FCA",
            "EA60": "8FCC",
            "EA61": "9033",
            "EA63": "48AD",
            "EA64": "98E0",
            "EA65": "9213",
            "EA66": "491E",
            "EA67": "9228",
            "EA68": "9258",
            "EA69": "926B",
            "EA6A": "92B1",
            "EA6B": "92AE",
            "EA6C": "92BF",
            "EA6D": "92E3",
            "EA6E": "92EB",
            "EA6F": "92F3",
            "EA70": "92F4",
            "EA71": "92FD",
            "EA72": "9343",
            "EA73": "9384",
            "EA74": "93AD",
            "EA75": "4945",
            "EA76": "4951",
            "EA78": "9417",
            "EA7A": "941D",
            "EA7B": "942D",
            "EA7C": "943E",
            "EA7D": "496A",
            "EA7E": "9454",
            "EA7F": "9479",
            "EA80": "952D",
            "EA81": "95A2",
            "EA82": "49A7",
            "EA83": "95F4",
            "EA84": "9633",
            "EA85": "49E5",
            "EA87": "4A24",
            "EA88": "9740",
            "EA89": "4A35",
            "EA8A": "97B2",
            "EA8B": "97C2",
            "EA8C": "5654",
            "EA8D": "4AE4",
            "EA8F": "98B9",
            "EA90": "4B19",
            "EA91": "98F1",
            "EA92": "5844",
            "EA93": "990E",
            "EA94": "9919",
            "EA96": "991C",
            "EA97": "9937",
            "EA98": "9942",
            "EA99": "995D",
            "EA9A": "9962",
            "EA9B": "4B70",
            "EA9C": "99C5",
            "EA9D": "4B9D",
            "EA9E": "9A3C",
            "EA9F": "9B0F",
            "EAA1": "9B69",
            "EAA2": "9B81",
            "EAA3": "9BDD",
            "EAA4": "9BF1",
            "EAA5": "9BF4",
            "EAA6": "4C6D",
            "EAA7": "9C20",
            "EAA8": "376F",
            "EAAA": "9D49",
            "EAAC": "9EFE",
            "EAAD": "5650",
            "EAAE": "9D93",
            "EAAF": "9DBD",
            "EAB0": "9DC0",
            "EAB1": "9DFC",
            "EAB2": "94F6",
            "EAB3": "8FB6",
            "EAB4": "9E7B",
            "EAB5": "9EAC",
            "EAB6": "9EB1",
            "EAB7": "9EBD",
            "EAB8": "9EC6",
            "EAB9": "94DC",
            "EABA": "9EE2",
            "EABB": "9EF1",
            "EABC": "9EF8",
            "EABE": "9F44",
            "EAC3": "94C3",
            "EAC7": "94C1",
            "EAC8": "37B9",
            "EACD": "5757",
            "EACE": "7173",
            "EAD5": "549E",
            "EAF7": "3647",
            "EAFC": "5364",
            "EB07": "70BB",
            "EB13": "7198",
            "EB17": "36C8",
            "EB47": "58B5",
            "EB64": "5788",
            "EB75": "5965",
            "EB7C": "573F",
            "EB9C": "57A1",
            "EB9D": "7151",
            "EBBB": "56B9",
            "EBC3": "3618",
            "EBF4": "949F",
            "EC00": "36B9",
            "EC0A": "56D6",
            "EC61": "3DF7",
            "EC64": "8FD0",
            "EC65": "728F",
            "EC74": "7E9F",
            "EC77": "9FD0", # The mapping defined in hkscs1999.tsv is 4CA4, but it has been remapped in HKSCS-2016 to 9FD0
            "EC78": "9547",
            "EC7A": "71A2",
            "EC7C": "4D91",
            "EC7F": "4D9C",
            "EC81": "8FBE",
            "EC83": "8FBA",
            "EC85": "8FB9",
            "EC8B": "4EEA",
            "EC9C": "4E9A",
            "ECAE": "3DC6",
            "ECB8": "4E1C",
            "ED08": "5B90",
            "ED31": "4E1A",
            "ED32": "4E04",
            "ED34": "5B0D",
            "ED37": "36AC",
            "ED39": "537D",
            "ED3A": "36A5",
            "ED3C": "589A",
            "ED40": "57AA",
            "ED76": "57A7",
            "EDAE": "3E06",
            "EE00": "717A",
            "EE89": "3635",
            "EEA0": "5220",
            "EEAB": "5CC1",
            "EEB2": "4562",
            "EEB3": "5B1F",
            "EEB5": "9F50",
            "EEB6": "9EA6",
            "F319": "0100",
            "F31A": "00C1",
            "F31B": "01CD",
            "F31C": "00C0",
            "F31D": "0112",
            "F31E": "00C9",
            "F31F": "011A",
            "F320": "00C8",
            "F321": "014C",
            "F322": "00D3",
            "F323": "01D1",
            "F324": "00D2",
            "F326": "1EBE",
            "F328": "1EC0",
            "F329": "00CA",
            "F32A": "0101",
            "F32B": "00E1",
            "F32C": "01CE",
            "F32D": "00E0",
            "F32E": "0251",
            "F32F": "0113",
            "F330": "00E9",
            "F331": "011B",
            "F332": "00E8",
            "F333": "012B",
            "F334": "00ED",
            "F335": "01D0",
            "F336": "00EC",
            "F337": "014D",
            "F338": "00F3",
            "F339": "01D2",
            "F33A": "00F2",
            "F33B": "016B",
            "F33C": "00FA",
            "F33D": "01D4",
            "F33E": "00F9",
            "F33F": "01D6",
            "F340": "01D8",
            "F341": "01DA",
            "F342": "01DC",
            "F343": "00FC",
            "F345": "1EBF",
            "F347": "1EC1",
            "F348": "00EA",
            "F349": "0261",
            "F3A3": "650A",
            "F3A6": "4E3D",
            "F3A7": "6EDD",
            "F3A8": "9D4E",
            "F3A9": "91DF",
            "F3AD": "6491",
            "F3AE": "4F1A",
            "F3AF": "4F28",
            "F3B0": "4FA8",
            "F3B1": "5156",
            "F3B2": "5174",
            "F3B3": "519C",
            "F3B4": "51E4",
            "F3B5": "52A1",
            "F3B6": "52A8",
            "F3B7": "533B",
            "F3B8": "534E",
            "F3B9": "53D1",
            "F3BA": "53D8",
            "F3BB": "56E2",
            "F3BC": "58F0",
            "F3BD": "5904",
            "F3BE": "5907",
            "F3BF": "5932",
            "F3C0": "5934",
            "F3C1": "5B66",
            "F3C2": "5B9E",
            "F3C3": "5B9F",
            "F3C4": "5C9A",
            "F3C5": "5E86",
            "F3C6": "603B",
            "F3C7": "6589",
            "F3C8": "67FE",
            "F3C9": "6804",
            "F3CA": "6865",
            "F3CB": "6D4E",
            "F3CC": "70BC",
            "F3CD": "7535",
            "F3CE": "7EA4",
            "F3CF": "7EAC",
            "F3D0": "7EBA",
            "F3D1": "7EC7",
            "F3D2": "7ECF",
            "F3D3": "7EDF",
            "F3D4": "7F06",
            "F3D5": "7F37",
            "F3D6": "827A",
            "F3D7": "82CF",
            "F3D8": "836F",
            "F3D9": "89C6",
            "F3DA": "8BBE",
            "F3DB": "8BE2",
            "F3DC": "8F66",
            "F3DD": "8F67",
            "F3DE": "8F6E",
            "F3DF": "7411",
            "F3E0": "7CFC",
            "F3E1": "7DCD",
            "F3E2": "6946",
            "F3E3": "7AC9",
            "F3E4": "5227",
            "F3E9": "918C",
            "F3EA": "78B8",
            "F3EB": "915E",
            "F3EC": "80BC",
            "F3EE": "8D0B",
            "F3EF": "80F6",
            "F3F3": "809F",
            "F3F4": "9EC7",
            "F3F5": "4CCD",
            "F3F6": "9DC9",
            "F3F7": "9E0C",
            "F3F8": "4C3E",
            "F3FB": "9E0A",
            "F3FD": "35C1",
            "F3FF": "6E9A",
            "F400": "823E",
            "F401": "7519",
            "F403": "4911",
            "F404": "9A6C",
            "F405": "9A8F",
            "F406": "9F99",
            "F407": "7987",
            "F40C": "4E24",
            "F40D": "4E81",
            "F40E": "4E80",
            "F40F": "4E87",
            "F410": "4EBF",
            "F411": "4EEB",
            "F412": "4F37",
            "F413": "344C",
            "F414": "4FBD",
            "F415": "3E48",
            "F416": "5003",
            "F417": "5088",
            "F418": "347D",
            "F419": "3493",
            "F41A": "34A5",
            "F41B": "5186",
            "F41C": "5905",
            "F41D": "51DB",
            "F41E": "51FC",
            "F41F": "5205",
            "F420": "4E89",
            "F421": "5279",
            "F422": "5290",
            "F423": "5327",
            "F424": "35C7",
            "F425": "53A9",
            "F426": "3551",
            "F427": "53B0",
            "F428": "3553",
            "F429": "53C2",
            "F42A": "5423",
            "F42B": "356D",
            "F42C": "3572",
            "F42D": "3681",
            "F42E": "5493",
            "F42F": "54A3",
            "F430": "54B4",
            "F431": "54B9",
            "F432": "54D0",
            "F433": "54EF",
            "F434": "5518",
            "F435": "5523",
            "F436": "5528",
            "F437": "3598",
            "F438": "553F",
            "F439": "35A5",
            "F43A": "35BF",
            "F43B": "55D7",
            "F43C": "35C5",
            "F43E": "5525",
            "F443": "5590",
            "F445": "39EC",
            "F447": "8E46",
            "F44A": "4053",
            "F44C": "777A",
            "F44E": "3A34",
            "F44F": "47D5",
            "F453": "64DD",
            "F458": "648D",
            "F459": "8E7E",
            "F464": "47F4",
            "F467": "9AB2",
            "F468": "3A67",
            "F46A": "3FED",
            "F46B": "3506",
            "F470": "9D6E",
            "F471": "9815",
            "F473": "43D9",
            "F475": "64B4",
            "F476": "54E3",
            "F47A": "39FB",
            "F47F": "64EA",
            "F482": "8E68",
            "F487": "480B",
            "F489": "3FFA",
            "F48A": "5873",
            "F492": "5579",
            "F493": "40BB",
            "F494": "43BA",
            "F496": "4AB4",
            "F499": "81AA",
            "F49A": "98F5",
            "F49C": "6379",
            "F49D": "39FE",
            "F49F": "8DC0",
            "F4A0": "56A1",
            "F4A1": "647C",
            "F4A2": "3E43",
            "F4AB": "3992",
            "F4AC": "3A06",
            "F4AE": "3578",
            "F4B1": "5652",
            "F4B5": "34BC",
            "F4B6": "6C3D",
            "F4C2": "7F93",
            "F4C6": "35FB",
            "F4CB": "3F93",
            "F4D6": "3FF9",
            "F4D8": "6432",
            "F4E1": "3A18",
            "F4EA": "95AA",
            "F4EB": "54CC",
            "F4EC": "82C4",
            "F4ED": "55B9",
            "F4F0": "9C26",
            "F4F1": "9AB6",
            "F4F4": "7140",
            "F4F5": "816D",
            "F4F6": "80EC",
            "F4F7": "5C1C",
            "F4F9": "8134",
            "F4FA": "3797",
            "F4FB": "535F",
            "F4FD": "91B6",
            "F502": "35DD",
            "F504": "3609",
            "F506": "56AF",
            "F512": "5A54",
            "F517": "579C",
            "F51E": "3703",
            "F527": "5899",
            "F528": "5268",
            "F529": "361A",
            "F52B": "7BB2",
            "F52C": "5B68",
            "F52D": "4800",
            "F52E": "4B2C",
            "F52F": "9F27",
            "F530": "49E7",
            "F531": "9C1F",
            "F532": "9B8D",
            "F535": "55FB",
            "F536": "35F2",
            "F537": "5689",
            "F538": "4E28",
            "F539": "5902",
            "F53C": "9751",
            "F53E": "4E5B",
            "F53F": "4EBB",
            "F540": "353E",
            "F541": "5C23",
            "F542": "5F51",
            "F543": "5FC4",
            "F544": "38FA",
            "F545": "624C",
            "F546": "6535",
            "F547": "6B7A",
            "F548": "6C35",
            "F549": "6C3A",
            "F54A": "706C",
            "F54B": "722B",
            "F54C": "4E2C",
            "F54D": "72AD",
            "F54F": "7F52",
            "F550": "793B",
            "F551": "7CF9",
            "F552": "7F53",
            "F554": "34C1",
            "F557": "8002",
            "F558": "8080",
            "F55B": "535D",
            "F55C": "8864",
            "F55D": "89C1",
            "F55F": "8BA0",
            "F560": "8D1D",
            "F561": "9485",
            "F562": "9578",
            "F563": "957F",
            "F564": "95E8",
            "F566": "97E6",
            "F567": "9875",
            "F568": "98CE",
            "F569": "98DE",
            "F56A": "9963",
            "F56C": "9C7C",
            "F56D": "9E1F",
            "F56E": "9EC4",
            "F56F": "6B6F",
            "F570": "9F9C", # Is this right? HKSCS-1999 maps F570 to 9F9C, but HKSCS-2001 maps it to F907
            "F571": "4E37",
            "F573": "961D",
            "F574": "6237",
            "F575": "94A2",
            "F634": "5D3E",
            "F635": "5D48",
            "F636": "5D56",
            "F637": "3DFC",
            "F638": "380F",
            "F639": "5DA4",
            "F63A": "5DB9",
            "F63B": "3820",
            "F63C": "3838",
            "F63D": "5E42",
            "F63E": "5EBD",
            "F63F": "5F25",
            "F640": "5F83",
            "F641": "3908",
            "F642": "3914",
            "F643": "393F",
            "F644": "394D",
            "F645": "60D7",
            "F646": "613D",
            "F647": "5CE5",
            "F648": "3989",
            "F649": "61B7",
            "F64A": "61B9",
            "F64B": "61CF",
            "F64C": "39B8",
            "F64D": "622C",
            "F64E": "6290",
            "F64F": "62E5",
            "F650": "6318",
            "F651": "39F8",
            "F652": "56B1",
            "F653": "3A03",
            "F654": "63E2",
            "F655": "63FB",
            "F656": "6407",
            "F657": "645A",
            "F658": "3A4B",
            "F659": "64C0",
            "F65A": "5D15",
            "F65B": "5621",
            "F65C": "9F9F",
            "F65D": "3A97",
            "F65E": "6586",
            "F65F": "3ABD",
            "F660": "65FF",
            "F661": "6653",
            "F662": "3AF2",
            "F663": "6692",
            "F664": "3B22",
            "F665": "6716",
            "F666": "3B42",
            "F667": "67A4",
            "F668": "6800",
            "F669": "3B58",
            "F66A": "684A",
            "F66B": "6884",
            "F66C": "3B72",
            "F66D": "3B71",
            "F66E": "3B7B",
            "F66F": "6909",
            "F670": "6943",
            "F671": "725C",
            "F672": "6964",
            "F673": "699F",
            "F674": "6985",
            "F675": "3BBC",
            "F676": "69D6",
            "F677": "3BDD",
            "F678": "6A65",
            "F679": "6A74",
            "F67A": "6A71",
            "F67B": "6A82",
            "F67C": "3BEC",
            "F67D": "6A99",
            "F67E": "3BF2",
            "F67F": "6AAB",
            "F680": "6AB5",
            "F681": "6AD4",
            "F682": "6AF6",
            "F683": "6B81",
            "F684": "6BC1",
            "F685": "6BEA",
            "F686": "6C75",
            "F687": "6CAA",
            "F688": "3CCB",
            "F689": "6D02",
            "F68A": "6D06",
            "F68B": "6D26",
            "F68C": "6D81",
            "F68D": "3CEF",
            "F68E": "6DA4",
            "F68F": "6DB1",
            "F690": "6E15",
            "F691": "6E18",
            "F692": "6E29",
            "F693": "6E86",
            "F695": "6EBB",
            "F696": "6EE2",
            "F697": "6EDA",
            "F698": "9F7F",
            "F699": "6EE8",
            "F69A": "6EE9",
            "F69B": "6F24",
            "F69C": "6F34",
            "F69D": "3D46",
            "F69F": "6F81",
            "F6A0": "6FBE",
            "F6A1": "3D6A",
            "F6A2": "3D75",
            "F6A3": "71B7",
            "F6A4": "5C99",
            "F6A5": "3D8A",
            "F6A6": "702C",
            "F6A7": "3D91",
            "F6A8": "7050",
            "F6A9": "7054",
            "F6AA": "706F",
            "F6AB": "707F",
            "F6AC": "7089",
            "F6AE": "43C1",
            "F6AF": "35F1",
            "F6B1": "2460",
            "F6B2": "2461",
            "F6B3": "2462",
            "F6B4": "2463",
            "F6B5": "2464",
            "F6B6": "2465",
            "F6B7": "2466",
            "F6B8": "2467",
            "F6B9": "2468",
            "F6BA": "2469",
            "F6BB": "2474",
            "F6BC": "2475",
            "F6BD": "2476",
            "F6BE": "2477",
            "F6BF": "2478",
            "F6C0": "2479",
            "F6C1": "247A",
            "F6C2": "247B",
            "F6C3": "247C",
            "F6C4": "247D",
            "F6C5": "2170",
            "F6C6": "2171",
            "F6C7": "2172",
            "F6C8": "2173",
            "F6C9": "2174",
            "F6CA": "2175",
            "F6CB": "2176",
            "F6CC": "2177",
            "F6CD": "2178",
            "F6CE": "2179",
            "F6CF": "4E36",
            "F6D0": "4E3F",
            "F6D1": "4E85",
            "F6D2": "4EA0",
            "F6D3": "5182",
            "F6D4": "5196",
            "F6D5": "51AB",
            "F6D6": "52F9",
            "F6D7": "5338",
            "F6D8": "5369",
            "F6D9": "53B6",
            "F6DA": "590A",
            "F6DB": "5B80",
            "F6DC": "5DDB",
            "F6DD": "2F33",
            "F6DE": "5E7F",
            "F6DF": "2F35",
            "F6E0": "5F50",
            "F6E1": "5F61",
            "F6E2": "6534",
            "F6E3": "2F46",
            "F6E4": "7592",
            "F6E5": "2F68",
            "F6E6": "8FB5",
            "F6E7": "2FAA",
            "F6E8": "00A8",
            "F6E9": "02C6",
            "F6EA": "30FD",
            "F6EB": "30FE",
            "F6EC": "309D",
            "F6ED": "309E",
            "F6EE": "301E",
            "F6EF": "4EDD",
            "F6F0": "3005",
            "F6F1": "3006",
            "F6F2": "3007",
            "F6F3": "30FC",
            "F6F4": "FF3B",
            "F6F5": "FF3D",
            "F6F6": "273D",
            "F6F7": "3041",
            "F6F8": "3042",
            "F6F9": "3043",
            "F6FA": "3044",
            "F6FB": "3045",
            "F6FC": "3046",
            "F6FD": "3047",
            "F6FE": "3048",
            "F6FF": "3049",
            "F700": "304A",
            "F701": "304B",
            "F702": "304C",
            "F703": "304D",
            "F704": "304E",
            "F705": "304F",
            "F706": "3050",
            "F707": "3051",
            "F708": "3052",
            "F709": "3053",
            "F70A": "3054",
            "F70B": "3055",
            "F70C": "3056",
            "F70D": "3057",
            "F70E": "3058",
            "F70F": "3059",
            "F710": "305A",
            "F711": "305B",
            "F712": "305C",
            "F713": "305D",
            "F714": "305E",
            "F715": "305F",
            "F716": "3060",
            "F717": "3061",
            "F718": "3062",
            "F719": "3063",
            "F71A": "3064",
            "F71B": "3065",
            "F71C": "3066",
            "F71D": "3067",
            "F71E": "3068",
            "F71F": "3069",
            "F720": "306A",
            "F721": "306B",
            "F722": "306C",
            "F723": "306D",
            "F724": "306E",
            "F725": "306F",
            "F726": "3070",
            "F727": "3071",
            "F728": "3072",
            "F729": "3073",
            "F72A": "3074",
            "F72B": "3075",
            "F72C": "3076",
            "F72D": "3077",
            "F72E": "3078",
            "F72F": "3079",
            "F730": "307A",
            "F731": "307B",
            "F732": "307C",
            "F733": "307D",
            "F734": "307E",
            "F735": "307F",
            "F736": "3080",
            "F737": "3081",
            "F738": "3082",
            "F739": "3083",
            "F73A": "3084",
            "F73B": "3085",
            "F73C": "3086",
            "F73D": "3087",
            "F73E": "3088",
            "F73F": "3089",
            "F740": "308A",
            "F741": "308B",
            "F742": "308C",
            "F743": "308D",
            "F744": "308E",
            "F745": "308F",
            "F746": "3090",
            "F747": "3091",
            "F748": "3092",
            "F749": "3093",
            "F74A": "30A1",
            "F74B": "30A2",
            "F74C": "30A3",
            "F74D": "30A4",
            "F74E": "30A5",
            "F74F": "30A6",
            "F750": "30A7",
            "F751": "30A8",
            "F752": "30A9",
            "F753": "30AA",
            "F754": "30AB",
            "F755": "30AC",
            "F756": "30AD",
            "F757": "30AE",
            "F758": "30AF",
            "F759": "30B0",
            "F75A": "30B1",
            "F75B": "30B2",
            "F75C": "30B3",
            "F75D": "30B4",
            "F75E": "30B5",
            "F75F": "30B6",
            "F760": "30B7",
            "F761": "30B8",
            "F762": "30B9",
            "F763": "30BA",
            "F764": "30BB",
            "F765": "30BC",
            "F766": "30BD",
            "F767": "30BE",
            "F768": "30BF",
            "F769": "30C0",
            "F76A": "30C1",
            "F76B": "30C2",
            "F76C": "30C3",
            "F76D": "30C4",
            "F76E": "30C5",
            "F76F": "30C6",
            "F770": "30C7",
            "F771": "30C8",
            "F772": "30C9",
            "F773": "30CA",
            "F774": "30CB",
            "F775": "30CC",
            "F776": "30CD",
            "F777": "30CE",
            "F778": "30CF",
            "F779": "30D0",
            "F77A": "30D1",
            "F77B": "30D2",
            "F77C": "30D3",
            "F77D": "30D4",
            "F77E": "30D5",
            "F77F": "30D6",
            "F780": "30D7",
            "F781": "30D8",
            "F782": "30D9",
            "F783": "30DA",
            "F784": "30DB",
            "F785": "30DC",
            "F786": "30DD",
            "F787": "30DE",
            "F788": "30DF",
            "F789": "30E0",
            "F78A": "30E1",
            "F78B": "30E2",
            "F78C": "30E3",
            "F78D": "30E4",
            "F78E": "30E5",
            "F78F": "30E6",
            "F790": "30E7",
            "F791": "30E8",
            "F792": "30E9",
            "F793": "30EA",
            "F794": "30EB",
            "F795": "30EC",
            "F796": "30ED",
            "F797": "30EE",
            "F798": "30EF",
            "F799": "30F0",
            "F79A": "30F1",
            "F79B": "30F2",
            "F79C": "30F3",
            "F79D": "30F4",
            "F79E": "30F5",
            "F79F": "30F6",
            "F7A0": "0410",
            "F7A1": "0411",
            "F7A2": "0412",
            "F7A3": "0413",
            "F7A4": "0414",
            "F7A5": "0415",
            "F7A6": "0401",
            "F7A7": "0416",
            "F7A8": "0417",
            "F7A9": "0418",
            "F7AA": "0419",
            "F7AB": "041A",
            "F7AC": "041B",
            "F7AD": "041C",
            "F7AE": "041D",
            "F7AF": "041E",
            "F7B0": "041F",
            "F7B1": "0420",
            "F7B2": "0421",
            "F7B3": "0422",
            "F7B4": "0423",
            "F7B5": "0424",
            "F7B6": "0425",
            "F7B7": "0426",
            "F7B8": "0427",
            "F7B9": "0428",
            "F7BA": "0429",
            "F7BB": "042A",
            "F7BC": "042B",
            "F7BD": "042C",
            "F7BE": "042D",
            "F7BF": "042E",
            "F7C0": "042F",
            "F7C1": "0430",
            "F7C2": "0431",
            "F7C3": "0432",
            "F7C4": "0433",
            "F7C5": "0434",
            "F7C6": "0435",
            "F7C7": "0451",
            "F7C8": "0436",
            "F7C9": "0437",
            "F7CA": "0438",
            "F7CB": "0439",
            "F7CC": "043A",
            "F7CD": "043B",
            "F7CE": "043C",
            "F7CF": "043D",
            "F7D0": "043E",
            "F7D1": "043F",
            "F7D2": "0440",
            "F7D3": "0441",
            "F7D4": "0442",
            "F7D5": "0443",
            "F7D6": "0444",
            "F7D7": "0445",
            "F7D8": "0446",
            "F7D9": "0447",
            "F7DA": "0448",
            "F7DB": "0449",
            "F7DC": "044A",
            "F7DD": "044B",
            "F7DE": "044C",
            "F7DF": "044D",
            "F7E0": "044E",
            "F7E1": "044F",
            "F7E2": "21E7",
            "F7E3": "21B8",
            "F7E4": "21B9",
            "F7E7": "4E5A",
            "F7E9": "5202",
            "F7EC": "5188",
            "F817": "FFE2",
            "F818": "FFE4",
            "F819": "FF07",
            "F81A": "FF02",
            "F81B": "3231",
            "F81C": "2116",
            "F81D": "2121",
            "F81E": "309B",
            "F81F": "309C",
            "F820": "2E80",
            "F821": "2E84",
            "F822": "2E86",
            "F823": "2E87",
            "F824": "2E88",
            "F825": "2E8A",
            "F826": "2E8C",
            "F827": "2E8D",
            "F828": "2E95",
            "F829": "2E9C",
            "F82A": "2E9D",
            "F82B": "2EA5",
            "F82C": "2EA7",
            "F82D": "2EAA",
            "F82E": "2EAC",
            "F82F": "2EAE",
            "F830": "2EB6",
            "F831": "2EBC",
            "F832": "2EBE",
            "F833": "2EC6",
            "F834": "2ECA",
            "F835": "2ECC",
            "F836": "2ECD",
            "F837": "2ECF",
            "F838": "2ED6",
            "F839": "2ED7",
            "F83A": "2EDE",
            "F83B": "2EE3",
            "F83F": "0283",
            "F840": "0250",
            "F841": "025B",
            "F842": "0254",
            "F843": "0275",
            "F844": "0153",
            "F845": "00F8",
            "F846": "014B",
            "F847": "028A",
            "F848": "026A",
        }

        for char in chars:
            res = converter.convert_char(chr(int(char, 16)))
            self.assertEqual(
                res,
                "".join(
                    [chr(int(codepoint, 16)) for codepoint in chars[char].split(",")]
                ),
            )

    def test_hkscs2001(self):
        # New mappings from HKSCS-2001
        # These were added in between ISO 10464:2000 -> ISO 10646-2:2001
        chars = {
            "E000": "20547",
            "E002": "205DF",
            "E003": "23FC5",
            "E009": "24942",
            "E00A": "289E4",
            "E00C": "219DB",
            "E00E": "23CC8",
            "E012": "24933",
            "E013": "289AA",
            "E014": "202A0",
            "E015": "26BB3",
            "E016": "21305",
            "E018": "224ED",
            "E01A": "26D29",
            "E01B": "27A84",
            "E01C": "23600",
            "E01D": "24AB1",
            "E01E": "22513",
            "E020": "2037E",
            "E022": "20380",
            "E023": "20347",
            "E025": "2041F",
            "E02D": "249A4",
            "E02E": "20487",
            "E030": "233B4",
            "E032": "20BFF",
            "E033": "220FC",
            "E034": "202E5",
            "E035": "22530",
            "E036": "2058E",
            "E037": "23233",
            "E038": "21983",
            "E03B": "205B3",
            "E03C": "23C99",
            "E045": "24AA6",
            "E04C": "2372D",
            "E05A": "26B13",
            "E05C": "28ADE",
            "E05F": "23F80",
            "E060": "20954",
            "E061": "23FEC",
            "E064": "20BE2",
            "E066": "21726",
            "E06C": "216E8",
            "E071": "286AB",
            "E074": "21596",
            "E075": "21613",
            "E078": "28A9B",
            "E07D": "25772",
            "E087": "20B8F",
            "E089": "23FEB",
            "E08A": "22DA3",
            "E08C": "20C77",
            "E093": "26B53",
            "E094": "20D74",
            "E099": "2170D",
            "E09A": "20EDD",
            "E09C": "20D4D",
            "E09D": "289BC",
            "E09E": "22698",
            "E0A6": "218D7",
            "E0A7": "2403A",
            "E0A9": "24435",
            "E0AB": "210B4",
            "E0AE": "2328A",
            "E0BA": "28B66",
            "E0BC": "2124F",
            "E0BE": "241A5",
            "E0C3": "26C7E",
            "E0C5": "21416",
            "E0C7": "21454",
            "E0C8": "24363",
            "E0CA": "24BF5",
            "E0D0": "2123C",
            "E0D4": "2A150",
            "E0D5": "24278",
            "E0DC": "2163E",
            "E0DF": "21692",
            "E0E2": "20D4E",
            "E0E3": "26C81",
            "E0E4": "26D2A",
            "E0E5": "217DC",
            "E0E7": "217FB",
            "E0E8": "217B2",
            "E0E9": "26DA6",
            "E0EB": "21828",
            "E0EC": "216D5",
            "E0EE": "26E45",
            "E0F2": "249A9",
            "E0F8": "26FA1",
            "E0F9": "22554",
            "E0FB": "21911",
            "E0FD": "216B8",
            "E103": "27A0E",
            "E106": "20204",
            "E109": "21A34",
            "E10A": "259CC",
            "E10B": "205A5",
            "E110": "21B44",
            "E114": "21CA5",
            "E115": "26B28",
            "E11C": "21DF9",
            "E11D": "21E37",
            "E121": "21EA4",
            "E126": "24901",
            "E127": "22049",
            "E128": "22173",
            "E130": "244BC",
            "E131": "20CD3",
            "E136": "21771",
            "E139": "28482",
            "E141": "201C1",
            "E142": "2F894",
            "E145": "2133A",
            "E147": "26888",
            "E148": "223D0",
            "E14A": "22471",
            "E14D": "26E6E",
            "E150": "28A36",
            "E154": "25250",
            "E155": "21F6A",
            "E156": "270F8",
            "E157": "22668",
            "E159": "2029E",
            "E15A": "28A29",
            "E15D": "21877",
            "E162": "227B4",
            "E168": "24982",
            "E169": "2498F",
            "E16A": "27A53",
            "E171": "2F8A6",
            "E172": "26ED2",
            "E175": "20656",
            "E178": "23FB7",
            "E179": "2285F",
            "E17B": "28B9D",
            "E17C": "2995D",
            "E17F": "22980",
            "E180": "228C1",
            "E185": "20118",
            "E187": "21770",
            "E189": "22E0D",
            "E18B": "249DF",
            "E18F": "2138E",
            "E190": "217FC",
            "E193": "22E36",
            "E195": "2408C",
            "E196": "2571D",
            "E19F": "24A28",
            "E1A0": "24A23",
            "E1A4": "2307E",
            "E1A6": "24940",
            "E1AA": "21829",
            "E1AE": "23400",
            "E1B0": "231F7",
            "E1B1": "231F8",
            "E1B3": "231A4",
            "E1B4": "231A5",
            "E1B6": "20E75",
            "E1B8": "251E6",
            "E1BB": "21E3D",
            "E1BC": "23231",
            "E1BD": "285F4",
            "E1BE": "231C8",
            "E1BF": "25313",
            "E1C1": "228F7",
            "E1C4": "2439C",
            "E1C5": "24A21",
            "E1C8": "237C2",
            "E1CC": "241CD",
            "E1CD": "290ED",
            "E1D4": "233E6",
            "E1D5": "26DA0",
            "E1D7": "2346F",
            "E1DA": "28ADF",
            "E1DD": "235CD",
            "E1E2": "2363C",
            "E1E6": "28A4A",
            "E1EB": "203C9",
            "E1EF": "23659",
            "E1F0": "2212A",
            "E1F2": "23703",
            "E1F7": "2919C",
            "E1FA": "20923",
            "E1FE": "227CD",
            "E200": "23ADB",
            "E201": "203B5",
            "E202": "21958",
            "E205": "23B5A",
            "E207": "23EFC",
            "E20A": "2248B",
            "E20B": "248F1",
            "E20C": "26B51",
            "E210": "23DBC",
            "E212": "23DBD",
            "E213": "241A4",
            "E214": "2490C",
            "E215": "24900",
            "E216": "23CC9",
            "E219": "20D32",
            "E21B": "231F9",
            "E21C": "22491",
            "E21F": "26D25",
            "E220": "26DA1",
            "E221": "26DEB",
            "E226": "2497F",
            "E227": "24085",
            "E228": "26E72",
            "E22A": "26F74",
            "E22F": "28B21",
            "E231": "23E2F",
            "E233": "23F82",
            "E237": "2304B",
            "E23B": "23E30",
            "E23D": "21497",
            "E23E": "2403D",
            "E245": "29170",
            "E248": "24144",
            "E24A": "24091",
            "E24B": "24155",
            "E24C": "24039",
            "E24D": "23FF0",
            "E24E": "23FB4",
            "E24F": "2413F",
            "E251": "24156",
            "E252": "24157",
            "E253": "24140",
            "E254": "261DD",
            "E261": "24277",
            "E268": "24365",
            "E26B": "242C1",
            "E26D": "2445A",
            "E26E": "24A27",
            "E26F": "24A22",
            "E271": "28BE8",
            "E27A": "25605",
            "E27D": "24974",
            "E282": "23044",
            "E288": "24823",
            "E289": "2882B",
            "E28B": "28804",
            "E290": "20C3A",
            "E291": "26A2E",
            "E294": "241E2",
            "E295": "216E7",
            "E296": "24A24",
            "E299": "249B7",
            "E29A": "2498D",
            "E29B": "249FB",
            "E29F": "24A26",
            "E2A1": "205C3",
            "E2A4": "228AD",
            "E2A6": "28EB2",
            "E2B2": "24A8C",
            "E2B3": "2415F",
            "E2B4": "24A79",
            "E2B5": "28B8F",
            "E2B7": "28C03",
            "E2B8": "2189E",
            "E2BA": "21988",
            "E2BE": "28ED9",
            "E2BF": "21A4B",
            "E2C1": "28EAC",
            "E2C7": "24F82",
            "E2C9": "24D13",
            "E2D4": "263F5",
            "E2D8": "26911",
            "E2D9": "2690E",
            "E2DE": "26F9F",
            "E2E0": "2509D",
            "E2E1": "2517D",
            "E2E2": "21E1C",
            "E2E7": "25220",
            "E2E9": "232AC",
            "E2EB": "28964",
            "E2EC": "28968",
            "E2ED": "216C1",
            "E2F0": "21376",
            "E2F1": "24A12",
            "E2F8": "255E0",
            "E2FC": "2760C",
            "E2FF": "28B2B",
            "E300": "26083",
            "E301": "2261C",
            "E306": "25857",
            "E308": "27B39",
            "E30C": "27126",
            "E30F": "2910D",
            "E311": "23ED7",
            "E313": "26ED3",
            "E315": "257E0",
            "E318": "28BE9",
            "E31B": "258E1",
            "E31C": "294D9",
            "E31F": "259AC",
            "E325": "2648D",
            "E32C": "25C01",
            "E330": "2530E",
            "E331": "25CFE",
            "E335": "25BB4",
            "E336": "26C7F",
            "E33B": "25D20",
            "E33D": "25C65",
            "E342": "25CC1",
            "E345": "24882",
            "E348": "24578",
            "E352": "26E44",
            "E354": "26ED6",
            "E35B": "24057",
            "E35C": "26029",
            "E360": "217F9",
            "E362": "2836D",
            "E364": "26121",
            "E365": "2615A",
            "E36F": "262D0",
            "E372": "26351",
            "E374": "21661",
            "E375": "20068",
            "E377": "23766",
            "E379": "2833A",
            "E37B": "26489",
            "E381": "2A087",
            "E382": "26CC3",
            "E386": "22714",
            "E388": "26626",
            "E389": "23DE3",
            "E38A": "266E8",
            "E38D": "28A48",
            "E391": "226F6",
            "E393": "26498",
            "E394": "24FB8",
            "E396": "2148A",
            "E398": "2185E",
            "E39A": "24A65",
            "E39B": "24A95",
            "E39E": "20B0D",
            "E39F": "26A52",
            "E3A0": "23D7E",
            "E3A2": "214FD",
            "E3A5": "26B0A",
            "E3A6": "249A7",
            "E3A7": "23530",
            "E3A8": "21773",
            "E3A9": "23DF8",
            "E3AC": "2F994",
            "E3B1": "20E16",
            "E3B2": "217B4",
            "E3B4": "2317D",
            "E3B5": "2355A",
            "E3B9": "23E8B",
            "E3BA": "26DA3",
            "E3BB": "26B05",
            "E3BC": "26B97",
            "E3BD": "235CE",
            "E3C3": "26DA5",
            "E3C8": "26ED4",
            "E3CC": "26E42",
            "E3CD": "22EEF",
            "E3CF": "25BE4",
            "E3D5": "23CB5",
            "E3D8": "26B96",
            "E3D9": "26E77",
            "E3DA": "26E43",
            "E3E1": "25C91",
            "E3E3": "25CC0",
            "E3E7": "26E99",
            "E3ED": "28625",
            "E3EE": "2863B",
            "E3F0": "27088",
            "E3F2": "21582",
            "E3F3": "270CD",
            "E3F4": "2F9B2",
            "E3F8": "218A2",
            "E3FA": "2739A",
            "E3FD": "2A0F8",
            "E3FF": "22C27",
            "E409": "275E0",
            "E40A": "23DB9",
            "E40B": "275E4",
            "E414": "2770F",
            "E41B": "28A25",
            "E41D": "27924",
            "E41E": "27ABD",
            "E423": "27A59",
            "E425": "27B3A",
            "E426": "23F8F",
            "E428": "27B38",
            "E42C": "25430",
            "E42D": "25565",
            "E432": "24A7A",
            "E436": "216DF",
            "E43A": "27D54",
            "E43B": "27D8F",
            "E43C": "2F9D4",
            "E43E": "27D53",
            "E440": "27D98",
            "E441": "27DBD",
            "E444": "21910",
            "E448": "24CC9",
            "E44C": "28002",
            "E44D": "21014",
            "E44E": "2498A",
            "E450": "281BC",
            "E451": "2710C",
            "E459": "28365",
            "E45A": "28412",
            "E45D": "2A29F",
            "E45E": "20A50",
            "E461": "289DE",
            "E462": "2853D",
            "E463": "23DBB",
            "E465": "23262",
            "E467": "2A014",
            "E468": "286BC",
            "E469": "28501",
            "E46A": "22325",
            "E46C": "26ED7",
            "E46E": "2853C",
            "E46F": "27ABE",
            "E471": "2856C",
            "E472": "2860B",
            "E474": "28713",
            "E476": "286E6",
            "E47E": "28933",
            "E47F": "21E89",
            "E484": "255B9",
            "E486": "28AC6",
            "E487": "23C9B",
            "E488": "28B0C",
            "E489": "255DB",
            "E48A": "20D31",
            "E48D": "28AE1",
            "E48E": "28BEB",
            "E491": "28AE2",
            "E492": "28AE5",
            "E495": "28BEC",
            "E496": "28C39",
            "E497": "28BFF",
            "E4A1": "2267A",
            "E4A2": "286D8",
            "E4A3": "2127C",
            "E4A4": "23E2E",
            "E4AB": "26ED5",
            "E4AF": "28AE0",
            "E4B3": "26CB8",
            "E4B4": "20274",
            "E4B5": "26410",
            "E4B6": "290AF",
            "E4B7": "290E5",
            "E4B8": "24AD1",
            "E4B9": "21915",
            "E4BA": "2330A",
            "E4C1": "24AE9",
            "E4C6": "291D5",
            "E4C9": "291EB",
            "E4CD": "230B7",
            "E4CE": "230BC",
            "E4D6": "2546C",
            "E4D9": "29433",
            "E4DC": "2941D",
            "E4DD": "2797A",
            "E4E4": "27175",
            "E4E6": "20630",
            "E4E7": "2415C",
            "E4E8": "25706",
            "E4EE": "26D27",
            "E4EF": "216D3",
            "E4F6": "24A29",
            "E4F8": "29857",
            "E4F9": "29905",
            "E4FF": "25725",
            "E501": "290B1",
            "E502": "29BD5",
            "E505": "29B05",
            "E50A": "28600",
            "E50D": "269A8",
            "E50F": "2307D",
            "E512": "29D3E",
            "E514": "21863",
            "E516": "2424B",
            "E51A": "29E68",
            "E51C": "29FB7",
            "E51D": "2A192",
            "E51E": "2A1AB",
            "E51F": "2A0E1",
            "E520": "2A123",
            "E521": "2A1DF",
            "E524": "2A134",
            "E528": "2215B",
            "E529": "2A193",
            "E52A": "2A220",
            "E52B": "2193B",
            "E52C": "2A233",
            "E52E": "2A0B9",
            "E52F": "2A2B4",
            "E537": "24364",
            "E53E": "28C2B",
            "E543": "26DA2",
            "E54B": "2908B",
            "E554": "24975",
            "E555": "249BB",
            "E557": "249F8",
            "E558": "24348",
            "E559": "24A51",
            "E55B": "28BDA",
            "E55C": "218FA",
            "E55E": "2897E",
            "E55F": "28E36",
            "E562": "28A44",
            "E566": "2896C",
            "E567": "244B9",
            "E56D": "24473",
            "E56E": "243F8",
            "E570": "217EF",
            "E572": "218BE",
            "E573": "23599",
            "E575": "21885",
            "E576": "2542F",
            "E577": "217F8",
            "E579": "216FB",
            "E57A": "21839",
            "E57C": "21774",
            "E57D": "218D1",
            "E57E": "25F4B",
            "E580": "216C0",
            "E582": "24A25",
            "E583": "213FE",
            "E584": "212A8",
            "E585": "213C6",
            "E586": "214B6",
            "E588": "236A6",
            "E58B": "24994",
            "E58C": "27165",
            "E58D": "23E31",
            "E58E": "2555C",
            "E58F": "23EFB",
            "E590": "27052",
            "E592": "236EE",
            "E593": "2999D",
            "E594": "26F26",
            "E59A": "21922",
            "E59D": "2373F",
            "E59E": "240E1",
            "E59F": "2408B",
            "E5A0": "2410F",
            "E5A1": "26C21",
            "E5A4": "266B1",
            "E5A6": "20FDF",
            "E5A7": "20BA8",
            "E5A8": "20E0D",
            "E5AA": "28B13",
            "E5AF": "24436",
            "E5B1": "20465",
            "E5B2": "2037F",
            "E5B5": "25651",
            "E5B7": "201AB",
            "E5B8": "203CB",
            "E5BA": "2030A",
            "E5BB": "20414",
            "E5BE": "202C0",
            "E5BF": "28EB3",
            "E5C0": "20275",
            "E5C2": "2020C",
            "E5C8": "24A0E",
            "E5C9": "23E8A",
            "E5D7": "23595",
            "E5D8": "23E39",
            "E5D9": "23EBF",
            "E5DB": "21884",
            "E5DC": "23E89",
            "E5DF": "205E0",
            "E5E1": "204A3",
            "E5E2": "20492",
            "E5E3": "20491",
            "E5E5": "28A9C",
            "E5E6": "2070E",
            "E5E9": "20873",
            "E5EE": "2438C",
            "E5EF": "20C20",
            "E5F0": "249AC",
            "E5F1": "210E4",
            "E5F3": "20E1D",
            "E5FB": "24ABC",
            "E5FC": "2408D",
            "E600": "240C9",
            "E602": "20345",
            "E604": "20BC6",
            "E60F": "28A46",
            "E610": "216FA",
            "E611": "2176F",
            "E612": "21710",
            "E619": "25946",
            "E61A": "219F3",
            "E61B": "21861",
            "E61C": "24295",
            "E621": "25E83",
            "E623": "28BD7",
            "E624": "20413",
            "E627": "21303",
            "E62B": "289FB",
            "E632": "21996",
            "E633": "2197C",
            "E634": "23AEE",
            "E637": "21903",
            "E638": "21904",
            "E63A": "218A0",
            "E63C": "216FE",
            "E63F": "28A47",
            "E640": "21DBA",
            "E641": "23472",
            "E642": "289A8",
            "E645": "21927",
            "E646": "217AB",
            "E648": "2173B",
            "E64B": "275FD",
            "E64E": "22860",
            "E64F": "2262B",
            "E652": "225AF",
            "E653": "225BE",
            "E654": "29088",
            "E655": "26F73",
            "E657": "2003E",
            "E658": "20046",
            "E659": "2261B",
            "E65D": "22C9B",
            "E65E": "22D07",
            "E65F": "246D4",
            "E660": "2914D",
            "E662": "24665",
            "E663": "22B6A",
            "E665": "22B22",
            "E666": "23450",
            "E667": "298EA",
            "E668": "22E78",
            "E66A": "2A45B",
            "E66E": "249E3",
            "E66F": "22D67",
            "E671": "22CA1",
            "E676": "2308E",
            "E677": "232AD",
            "E678": "24989",
            "E679": "232AB",
            "E67B": "232E0",
            "E67C": "218D9",
            "E67D": "2943F",
            "E67F": "23289",
            "E680": "231B3",
            "E683": "25584",
            "E684": "28B22",
            "E685": "2558F",
            "E686": "216FC",
            "E687": "2555B",
            "E688": "25425",
            "E68A": "23103",
            "E68B": "2182A",
            "E68C": "23234",
            "E68E": "2320F",
            "E68F": "23182",
            "E690": "242C9",
            "E692": "26D24",
            "E696": "27870",
            "E697": "21DEB",
            "E699": "232D2",
            "E69A": "232E1",
            "E69C": "25872",
            "E69E": "2383A",
            "E69F": "237BC",
            "E6A1": "237A2",
            "E6A2": "233FE",
            "E6A6": "2462A",
            "E6AC": "237D5",
            "E6AD": "24487",
            "E6B2": "21912",
            "E6B9": "23FC0",
            "E6BA": "23C9A",
            "E6BD": "28BEA",
            "E6BE": "28ACB",
            "E6C0": "2801E",
            "E6C1": "289DC",
            "E6C7": "23F7F",
            "E6CA": "2403C",
            "E6D1": "2431A",
            "E6D2": "24276",
            "E6D7": "2478F",
            "E6D8": "24725",
            "E6DA": "24AA4",
            "E6DB": "205EB",
            "E6DD": "23EF8",
            "E6DE": "2365F",
            "E6DF": "24A4A",
            "E6E0": "24917",
            "E6E1": "25FE1",
            "E6E4": "24ADF",
            "E6E5": "28C23",
            "E6E6": "23F35",
            "E6EE": "26DEA",
            "E6F2": "24CD9",
            "E6F3": "24D06",
            "E6F7": "2A5C6",
            "E6FA": "28ACC",
            "E6FB": "249AB",
            "E6FC": "2498E",
            "E6FF": "24A4E",
            "E701": "249C5",
            "E702": "248F3",
            "E706": "28AE3",
            "E707": "21864",
            "E709": "25221",
            "E70A": "251E7",
            "E70C": "23232",
            "E710": "24697",
            "E711": "23781",
            "E713": "248F0",
            "E717": "24ABA",
            "E718": "24AC7",
            "E71A": "24A96",
            "E71B": "261AE",
            "E71D": "25581",
            "E724": "27741",
            "E725": "256E3",
            "E72B": "23EFA",
            "E72F": "216E6",
            "E733": "20D4C",
            "E734": "2498C",
            "E735": "20299",
            "E736": "23DBA",
            "E737": "2176E",
            "E73C": "201D4",
            "E73D": "20C0D",
            "E740": "226F5",
            "E741": "25AAF",
            "E742": "25A9C",
            "E744": "2025B",
            "E747": "25BC6",
            "E74D": "25BB3",
            "E74E": "25EBC",
            "E74F": "25EA6",
            "E751": "249F9",
            "E752": "217B0",
            "E75B": "26261",
            "E75C": "2615C",
            "E75D": "27B48",
            "E75F": "25E82",
            "E761": "26B75",
            "E762": "20916",
            "E764": "2004E",
            "E765": "235CF",
            "E767": "26412",
            "E768": "263F8",
            "E769": "24962",
            "E76C": "2082C",
            "E76D": "25AE9",
            "E76E": "25D43",
            "E770": "25E0E",
            "E775": "2343F",
            "E777": "249F7",
            "E778": "265AD",
            "E77A": "265A0",
            "E77C": "27127",
            "E77D": "26CD1",
            "E784": "267B4",
            "E786": "26A42",
            "E789": "26A51",
            "E78B": "26DA7",
            "E78D": "2721B",
            "E790": "21840",
            "E792": "218A1",
            "E794": "218D8",
            "E796": "2F9BC",
            "E797": "23D8F",
            "E799": "27422",
            "E7A2": "25683",
            "E7A4": "27785",
            "E7A5": "27784",
            "E7A6": "28BF5",
            "E7A7": "28BD9",
            "E7A8": "28B9C",
            "E7A9": "289F9",
            "E7B1": "29448",
            "E7B4": "24284",
            "E7B8": "21845",
            "E7BC": "27DDC",
            "E7C4": "24C09",
            "E7CA": "22321",
            "E7CE": "217DA",
            "E7D1": "2492F",
            "E7D8": "28A4B",
            "E7DA": "28AFC",
            "E7DC": "28C1D",
            "E7DD": "28C3B",
            "E7E4": "28D34",
            "E7E7": "248FF",
            "E7E8": "24A42",
            "E7E9": "243EA",
            "E7EB": "23225",
            "E7ED": "28EE7",
            "E7EE": "28E66",
            "E7EF": "28E65",
            "E7F1": "249ED",
            "E7F2": "24A78",
            "E7F3": "23FEE",
            "E7F8": "290B0",
            "E7FB": "29093",
            "E7FC": "257DF",
            "E7FF": "28989",
            "E800": "28C26",
            "E801": "28B2F",
            "E802": "263BE",
            "E808": "2421B",
            "E80C": "20F26",
            "E80E": "28BC5",
            "E80F": "24AB2",
            "E811": "294DA",
            "E813": "295D7",
            "E816": "28B50",
            "E817": "24A67",
            "E818": "28B64",
            "E81A": "28A45",
            "E821": "27B06",
            "E827": "28B65",
            "E828": "258C8",
            "E82D": "298F1",
            "E82F": "29948",
            "E835": "21302",
            "E837": "249B8",
            "E838": "214E8",
            "E83A": "2271F",
            "E83B": "23DB8",
            "E841": "22781",
            "E842": "2296B",
            "E84B": "29E2D",
            "E84D": "2A1F5",
            "E84E": "2A0FE",
            "E851": "24104",
            "E855": "2A1B4",
            "E856": "2A0ED",
            "E857": "2A0F3",
            "E858": "2992F",
            "E85B": "26E12",
            "E85C": "26FDF",
            "E85D": "26B82",
            "E860": "26DA4",
            "E861": "26E84",
            "E862": "26DF0",
            "E865": "26E00",
            "E866": "237D7",
            "E867": "26064",
            "E869": "2359C",
            "E86A": "23640",
            "E86C": "249DE",
            "E871": "202BF",
            "E878": "2555D",
            "E87D": "21757",
            "E87E": "231C9",
            "E87F": "24941",
            "E887": "241B5",
            "E88F": "241AC",
            "E891": "26C40",
            "E892": "24F97",
            "E894": "217B5",
            "E895": "28A49",
            "E89A": "24488",
            "E89D": "289FC",
            "E8A0": "218D6",
            "E8A1": "20F1D",
            "E8A3": "26CC0",
            "E8A4": "21413",
            "E8A5": "242FA",
            "E8A6": "22C26",
            "E8A7": "243C1",
            "E8A9": "23DB7",
            "E8AA": "26741",
            "E8AC": "2615B",
            "E8AD": "260A4",
            "E8AE": "249B9",
            "E8AF": "2498B",
            "E8B0": "289FA",
            "E8B5": "28B63",
            "E8B6": "2189F",
            "E8B8": "24AB3",
            "E8BD": "24A3E",
            "E8BE": "24A94",
            "E8BF": "217D9",
            "E8C0": "24A66",
            "E8C1": "203A7",
            "E8C2": "21424",
            "E8C3": "249E5",
            "E8C5": "24916",
            "E8C7": "24976",
            "E8CB": "204FE",
            "E8CD": "28ACE",
            "E8CE": "28A16",
            "E8D0": "28BE7",
            "E8D1": "255D5",
            "E8D3": "28A82",
            "E8D5": "24943",
            "E8D6": "20CFF",
            "E8D8": "2061A",
            "E8D9": "20BEB",
            "E8DA": "20CB8",
            "E8DD": "217FA",
            "E8DF": "216C2",
            "E8E0": "24A50",
            "E8E1": "21852",
            "E8E5": "28AC0",
            "E8E6": "249AD",
            "E8E8": "218BF",
            "E8E9": "21883",
            "E8EA": "27484",
            "E8ED": "23D5B",
            "E8F1": "28A81",
            "E8F2": "21862",
            "E8F5": "20AB4",
            "E8FB": "2139C",
            "E8FE": "28218",
            "E900": "290E4",
            "E903": "27E4F",
            "E907": "23FED",
            "E908": "23E2D",
            "E90C": "203F5",
            "E917": "28C1C",
            "E918": "26BC0",
            "E91C": "21452",
            "E91E": "24362",
            "E91F": "24A71",
            "E920": "22FE3",
            "E921": "212B0",
            "E922": "223BD",
            "E925": "21398",
            "E926": "234E5",
            "E927": "27BF4",
            "E928": "236DF",
            "E929": "28A83",
            "E92A": "237D6",
            "E92B": "233FA",
            "E92C": "24C9F",
            "E92E": "236AD",
            "E92F": "26CB7",
            "E933": "26D26",
            "E934": "26D51",
            "E935": "26C82",
            "E936": "26FDE",
            "E938": "27109",
            "E93A": "2173A",
            "E93C": "26C80",
            "E93D": "27053",
            "E93E": "217DB",
            "E941": "217B3",
            "E944": "21905",
            "E945": "241FC",
            "E948": "2173C",
            "E94D": "242A5",
            "E950": "24293",
            "E952": "23EF9",
            "E953": "27736",
            "E954": "2445B",
            "E955": "242CA",
            "E957": "24259",
            "E958": "289E1",
            "E95A": "26D28",
            "E95C": "244CE",
            "E95D": "27E4D",
            "E95E": "243BD",
            "E960": "24256",
            "E961": "21304",
            "E964": "243E9",
            "E967": "2F825",
            "E968": "24A4F",
            "E96E": "24A5D",
            "E96F": "217DF",
            "E973": "23AFA",
            "E974": "23300",
            "E975": "20214",
            "E977": "208D5",
            "E978": "20619",
            "E97A": "21F9E",
            "E97B": "2A2B6",
            "E97D": "2915B",
            "E981": "28A59",
            "E982": "29420",
            "E987": "27AF4",
            "E98A": "256F6",
            "E98E": "27B18",
            "E991": "248F2",
            "E9B0": "24F5C",
            "E9CB": "25535",
            "E9EF": "20CCF",
            "EA43": "27967",
            "EAA9": "21BC2",
            "EABF": "20094",
            "EAC0": "202B7",
            "EAC1": "203A0",
            "EAC5": "204D7",
            "EAC9": "205D5",
            "EACA": "20615",
            "EACB": "20676",
            "EACC": "216BA",
            "EACF": "20AC2",
            "EAD0": "20ACD",
            "EAD1": "20BBF",
            "EAD3": "2F83B",
            "EAD4": "20BCB",
            "EAD6": "20BFB",
            "EAD7": "20C3B",
            "EAD8": "20C53",
            "EAD9": "20C65",
            "EADA": "20C7C",
            "EADC": "20C8D",
            "EADE": "20CB5",
            "EADF": "20CDD",
            "EAE0": "20CED",
            "EAE1": "20D6F",
            "EAE2": "20DB2",
            "EAE3": "20DC8",
            "EAE7": "20E04",
            "EAE8": "20E0E",
            "EAE9": "20ED7",
            "EAEA": "20F90",
            "EAEB": "20F2D",
            "EAEC": "20E73",
            "EAEE": "20FBC",
            "EAF0": "2105C",
            "EAF1": "2104F",
            "EAF2": "21076",
            "EAF4": "2107B",
            "EAF5": "21088",
            "EAF6": "21096",
            "EAF8": "210BF",
            "EAF9": "210D3",
            "EAFA": "2112F",
            "EAFB": "2113B",
            "EAFE": "212E3",
            "EAFF": "21375",
            "EB00": "21336",
            "EB02": "21577",
            "EB03": "21619",
            "EB04": "217C3",
            "EB05": "217C7",
            "EB08": "2182D",
            "EB09": "2196A",
            "EB0A": "21A2D",
            "EB0B": "21A45",
            "EB0C": "21C2A",
            "EB0D": "21C70",
            "EB0E": "21CAC",
            "EB0F": "21EC8",
            "EB11": "21ED5",
            "EB12": "21F15",
            "EB15": "22045",
            "EB18": "2227C",
            "EB19": "223D7",
            "EB1A": "223FA",
            "EB1B": "2272A",
            "EB1C": "22871",
            "EB1D": "2294F",
            "EB1F": "22967",
            "EB20": "22993",
            "EB21": "22AD5",
            "EB23": "22AE8",
            "EB25": "22B0E",
            "EB27": "22B3F",
            "EB2A": "22C4C",
            "EB2C": "22C88",
            "EB2D": "22CB7",
            "EB2E": "25BE8",
            "EB2F": "22D08",
            "EB30": "22D12",
            "EB31": "22DB7",
            "EB32": "22D95",
            "EB33": "22E42",
            "EB34": "22F74",
            "EB35": "22FCC",
            "EB36": "23033",
            "EB37": "23066",
            "EB38": "2331F",
            "EB39": "233DE",
            "EB3D": "27A79",
            "EB3E": "23567",
            "EB3F": "235F3",
            "EB41": "249BA",
            "EB43": "2361A",
            "EB44": "23716",
            "EB46": "20346",
            "EB4A": "23AA7",
            "EB4B": "27657",
            "EB4C": "25FE2",
            "EB4D": "23E11",
            "EB4E": "23EB9",
            "EB4F": "275FE",
            "EB50": "2209A",
            "EB53": "24119",
            "EB54": "28A9A",
            "EB55": "242EE",
            "EB56": "2430D",
            "EB57": "2403B",
            "EB58": "24334",
            "EB59": "24396",
            "EB5A": "24A45",
            "EB5B": "205CA",
            "EB5D": "20611",
            "EB5F": "21EA8",
            "EB61": "23CFF",
            "EB62": "24404",
            "EB63": "244D6",
            "EB65": "24674",
            "EB67": "2472F",
            "EB68": "285E8",
            "EB69": "299C9",
            "EB6B": "221C3",
            "EB6D": "28B4E",
            "EB6F": "24812",
            "EB70": "248FB",
            "EB71": "24A15",
            "EB73": "24AC0",
            "EB74": "20C78",
            "EB76": "24EA5",
            "EB77": "24F86",
            "EB78": "20779",
            "EB7A": "2502C",
            "EB7E": "25299",
            "EB7F": "25419",
            "EB80": "23F4A",
            "EB81": "24AA7",
            "EB83": "25446",
            "EB84": "2546E",
            "EB85": "26B52",
            "EB88": "2553F",
            "EB89": "27632",
            "EB8A": "2555E",
            "EB8C": "25562",
            "EB8D": "25566",
            "EB8E": "257C7",
            "EB8F": "2493F",
            "EB90": "2585D",
            "EB93": "233CC",
            "EB95": "25903",
            "EB97": "28948",
            "EB98": "25AAE",
            "EB99": "25B89",
            "EB9A": "25C06",
            "EB9B": "21D90",
            "EB9F": "26102",
            "EBA0": "27C12",
            "EBA2": "261B2",
            "EBA3": "24F9A",
            "EBA5": "26402",
            "EBA6": "2644A",
            "EBA8": "26BF7",
            "EBAA": "26484",
            "EBAB": "2191C",
            "EBAD": "249F6",
            "EBAE": "26488",
            "EBAF": "23FEF",
            "EBB0": "26512",
            "EBB2": "265BF",
            "EBB3": "266B5",
            "EBB4": "2271B",
            "EBB6": "257E1",
            "EBB9": "2F8CD",
            "EBBC": "24521",
            "EBBD": "266FC",
            "EBBF": "24934",
            "EBC2": "26CBD",
            "EBC5": "26799",
            "EBC6": "2686E",
            "EBC7": "26411",
            "EBC8": "2685E",
            "EBCA": "268C7",
            "EBCC": "290C0",
            "EBCD": "20A11",
            "EBCE": "26926",
            "EBD0": "26939",
            "EBD3": "269FA",
            "EBD5": "26A2D",
            "EBD7": "26469",
            "EBD8": "20021",
            "EBDA": "26A34",
            "EBDB": "26B5B",
            "EBDD": "23519",
            "EBDF": "26B9D",
            "EBE1": "26CA4",
            "EBE4": "26DAE",
            "EBE7": "2258D",
            "EBE8": "2704B",
            "EBE9": "271CD",
            "EBEB": "27280",
            "EBEC": "27285",
            "EBEE": "2217A",
            "EBEF": "2728B",
            "EBF1": "272E6",
            "EBF2": "249D0",
            "EBF5": "27450",
            "EBF6": "20EF8",
            "EBF9": "22926",
            "EBFA": "28473",
            "EBFB": "217B1",
            "EBFD": "24A2A",
            "EBFE": "21820",
            "EC05": "29CAD",
            "EC06": "298A4",
            "EC08": "277CC",
            "EC09": "27858",
            "EC0C": "2160A",
            "EC0E": "2372F",
            "EC0F": "280E8",
            "EC10": "213C5",
            "EC13": "279DD",
            "EC14": "291A8",
            "EC17": "270AF",
            "EC18": "289AB",
            "EC19": "279FD",
            "EC1A": "27A0A",
            "EC1B": "27B0B",
            "EC1C": "27D66",
            "EC1D": "2417A",
            "EC20": "28009",
            "EC22": "2A2DF",
            "EC24": "28318",
            "EC26": "26E07",
            "EC2A": "2816F",
            "EC2B": "28023",
            "EC2C": "269B5",
            "EC2D": "213ED",
            "EC2E": "2322F",
            "EC2F": "28048",
            "EC31": "28C30",
            "EC32": "28083",
            "EC35": "28949",
            "EC37": "24988",
            "EC3B": "24AA5",
            "EC3C": "23F81",
            "EC3F": "28090",
            "EC40": "280F4",
            "EC41": "2812E",
            "EC42": "21FA1",
            "EC43": "2814F",
            "EC44": "28189",
            "EC45": "281AF",
            "EC46": "2821A",
            "EC47": "28306",
            "EC48": "2832F",
            "EC49": "2838A",
            "EC4B": "28468",
            "EC4C": "286AA",
            "EC4F": "28956",
            "EC52": "289B8",
            "EC54": "289E7",
            "EC56": "289E8",
            "EC57": "28B46",
            "EC58": "28BD4",
            "EC5A": "28C09",
            "EC5C": "28FC5",
            "EC5D": "290EC",
            "EC5F": "29110",
            "EC60": "2913C",
            "EC62": "2915E",
            "EC63": "24ACA",
            "EC67": "294E7",
            "EC68": "295E9",
            "EC69": "295B0",
            "EC6A": "295B8",
            "EC6B": "29732",
            "EC6C": "298D1",
            "EC6D": "29949",
            "EC6E": "2996A",
            "EC6F": "299C3",
            "EC70": "29A28",
            "EC71": "29B0E",
            "EC72": "29D5A",
            "EC73": "29D9B",
            "EC75": "29EF8",
            "EC76": "29F23",
            "EC79": "2A293",
            "EC7B": "2A2FF",
            "EC7E": "2A5CB",
            "EC80": "20C9C",
            "EC84": "224B0",
            "EC86": "24A93",
            "EC8D": "28B2C",
            "EC8E": "2789D",
            "EC91": "217F5",
            "EC92": "28024",
            "EC93": "28B6C",
            "EC94": "28B99",
            "EC95": "27A3E",
            "EC96": "266AF",
            "EC98": "27655",
            "EC99": "23CB7",
            "EC9A": "25635",
            "EC9B": "25956",
            "EC9D": "25E81",
            "EC9E": "26258",
            "ECA0": "20E6D",
            "ECA3": "23E88",
            "ECA4": "24C9E",
            "ECA7": "217F6",
            "ECA8": "2187B",
            "ECAB": "25C4A",
            "ECAD": "25311",
            "ECAF": "29D98",
            "ECB4": "25ED8",
            "ECB6": "23D40",
            "ECB9": "20FEA",
            "ECBA": "20D49",
            "ECBB": "236BA",
            "ECCF": "20E9D",
            "ECD4": "282E2",
            "ECDB": "20C41",
            "ECDC": "20C96",
            "ECDE": "25148",
            "ECDF": "20E76",
            "ECE0": "22C62",
            "ECE1": "20EA2",
            "ECE5": "21075",
            "ECF0": "22B43",
            "ECF2": "22EB3",
            "ECF5": "20DA7",
            "ECFC": "2688A",
            "ED04": "210C1",
            "ED07": "24706",
            "ED0A": "26893",
            "ED0C": "226F4",
            "ED0D": "27D2F",
            "ED0E": "241A3",
            "ED0F": "27D73",
            "ED10": "26ED0",
            "ED11": "272B6",
            "ED13": "211D9",
            "ED15": "23CFC",
            "ED16": "2A6A9",
            "ED17": "20EAC",
            "ED18": "20EF9",
            "ED1A": "21CA2",
            "ED1C": "24FC2",
            "ED1D": "27FF9",
            "ED1E": "20FEB",
            "ED22": "22DA0",
            "ED24": "247E0",
            "ED25": "29D7C",
            "ED26": "20FEC",
            "ED27": "20E0A",
            "ED29": "275A3",
            "ED2A": "20FED",
            "ED2C": "26048",
            "ED2D": "21187",
            "ED3B": "24618",
            "ED3D": "24B6E",
            "ED41": "25A95",
            "ED42": "20979",
            "ED45": "22465",
            "ED47": "29EAC",
            "ED4A": "23CFE",
            "ED4B": "29F30",
            "ED4D": "24FA9",
            "ED4E": "2959E",
            "ED4F": "29FDE",
            "ED51": "23DB6",
            "ED52": "272B2",
            "ED53": "267B3",
            "ED54": "23720",
            "ED57": "23EF7",
            "ED58": "23E2C",
            "ED5F": "2048E",
            "ED62": "20EB6",
            "ED6B": "2A434",
            "ED71": "230DA",
            "ED72": "243F2",
            "ED75": "212A9",
            "ED77": "24963",
            "ED78": "29E06",
            "ED7A": "270AE",
            "ED81": "2176C",
            "ED84": "294D0",
            "ED85": "26335",
            "ED86": "27164",
            "ED88": "20D28",
            "ED89": "26D22",
            "ED8A": "24AE2",
            "ED8B": "20D71",
            "ED8E": "21F0F",
            "ED91": "21DD1",
            "EDA6": "2493E",
            "EDC6": "2176D",
            "EDCF": "28E97",
            "EDD1": "26D23",
            "EDD7": "203FC",
            "EDDA": "25C21",
            "EDDB": "23CFD",
            "EDDD": "24919",
            "EDE1": "20CD4",
            "EDED": "201F2",
            "EE08": "2A64A",
            "EE11": "2837D",
            "EE15": "24A77",
            "EE1B": "28A5A",
            "EE20": "2A2B2",
            "EE22": "2F840",
            "EE2A": "24ABB",
            "EE30": "2183B",
            "EE31": "26E05",
            "EE34": "2251B",
            "EE48": "26159",
            "EE56": "233F5",
            "EE5B": "28BC2",
            "EE5D": "28E39",
            "EE60": "21D46",
            "EE64": "20F3B",
            "EE6A": "25F1A",
            "EE76": "27486",
            "EE7C": "267CC",
            "EE7E": "26ED1",
            "EE91": "24011",
            "EE9C": "28B2D",
            "EEA1": "3C18",
            "EEA2": "23CC7",
            "EEA3": "25ED7",
            "EEA4": "27656",
            "EEA5": "25531",
            "EEA6": "21944",
            "EEA7": "212FE",
            "EEA8": "29903",
            "EEA9": "26DDC",
            "EEAA": "270AD",
            "EEAC": "261AD",
            "EEAD": "28A0F",
            "EEAE": "23677",
            "EEAF": "200EE",
            "EEB0": "26846",
            "EEB1": "24F0E",
            "EEB4": "2634C",
            "EEB7": "2626B",
            "F308": "2010C",
            "F30A": "200D1",
            "F30B": "200CD",
            "F30E": "200CB",
            "F30F": "21FE8",
            "F311": "200CA",
            "F316": "2010E",
            "F3A0": "2A3A9",
            "F3A1": "21145",
            "F3AC": "27735",
            "F3F0": "209E7",
            "F3F9": "29DF6",
            "F3FA": "2700E",
            "F3FC": "2A133",
            "F408": "2846C",
            "F409": "21DCA",
            "F40A": "205D0",
            "F40B": "22AE6",
            "F43D": "27D84",
            "F440": "20C42",
            "F441": "20D15",
            "F442": "2512B",
            "F444": "22CC6",
            "F446": "20341",
            "F448": "24DB8",
            "F449": "294E5",
            "F44B": "280BE",
            "F44D": "22C38",
            "F450": "2815D",
            "F451": "269F2",
            "F452": "24DEA",
            "F454": "20D7C",
            "F455": "20FB4",
            "F456": "20CD5",
            "F457": "210F4",
            "F45A": "20E96",
            "F45B": "20C0B",
            "F45C": "20F64",
            "F45D": "22CA9",
            "F45E": "28256",
            "F45F": "244D3",
            "F461": "20D46",
            "F462": "29A4D",
            "F463": "280E9",
            "F465": "24EA7",
            "F466": "22CC2",
            "F469": "295F4",
            "F46C": "252C7",
            "F46D": "297D4",
            "F46E": "278C8",
            "F46F": "22D44",
            "F474": "260A5",
            "F477": "22D4C",
            "F478": "22BCA",
            "F479": "21077",
            "F47B": "2106F",
            "F47C": "266DA",
            "F47D": "26716",
            "F47E": "279A0",
            "F480": "25052",
            "F481": "20C43",
            "F483": "221A1",
            "F484": "28B4C",
            "F485": "20731",
            "F488": "201A9",
            "F48B": "22D8D",
            "F48D": "245C8",
            "F48E": "204FC",
            "F48F": "26097",
            "F490": "20F4C",
            "F491": "20D96",
            "F497": "22A66",
            "F498": "2109D",
            "F49B": "20D9C",
            "F49E": "22775",
            "F4A4": "2A601",
            "F4A5": "20E09",
            "F4A6": "22ACF",
            "F4A7": "22CC9",
            "F4A9": "210C8",
            "F4AA": "239C2",
            "F4AD": "2829B",
            "F4AF": "25E49",
            "F4B0": "220C7",
            "F4B2": "20F31",
            "F4B3": "22CB2",
            "F4B4": "29720",
            "F4B7": "24E3B",
            "F4BA": "27574",
            "F4BB": "22E8B",
            "F4BC": "22208",
            "F4BD": "2A65B",
            "F4BE": "28CCD",
            "F4BF": "20E7A",
            "F4C0": "20C34",
            "F4C1": "2681C",
            "F4C3": "210CF",
            "F4C4": "22803",
            "F4C5": "22939",
            "F4C7": "251E3",
            "F4C8": "20E8C",
            "F4C9": "20F8D",
            "F4CA": "20EAA",
            "F4CC": "20F30",
            "F4CD": "20D47",
            "F4CE": "2114F",
            "F4CF": "20E4C",
            "F4D1": "20EAB",
            "F4D2": "20BA9",
            "F4D3": "20D48",
            "F4D4": "210C0",
            "F4D5": "2113D",
            "F4D7": "22696",
            "F4D9": "20FAD",
            "F4DA": "233F4",
            "F4DB": "27639",
            "F4DC": "22BCE",
            "F4DD": "20D7E",
            "F4DE": "20D7F",
            "F4DF": "22C51",
            "F4E0": "22C55",
            "F4E2": "20E98",
            "F4E3": "210C7",
            "F4E4": "20F2E",
            "F4E5": "2A632",
            "F4E6": "26B50",
            "F4E7": "28CD2",
            "F4E8": "28D99",
            "F4E9": "28CCA",
            "F4EF": "29EC3",
            "F4F2": "2775E",
            "F4F3": "22DEE",
            "F4F8": "26572",
            "F4FC": "280BD",
            "F4FE": "20EFA",
            "F4FF": "20E0F",
            "F500": "20E77",
            "F501": "20EFB",
            "F503": "24DEB",
            "F505": "20CD6",
            "F507": "227B5",
            "F508": "210C9",
            "F509": "20E10",
            "F50A": "20E78",
            "F50B": "21078",
            "F50C": "21148",
            "F50D": "28207",
            "F50E": "21455",
            "F50F": "20E79",
            "F510": "24E50",
            "F511": "22DA4",
            "F513": "2101D",
            "F514": "2101E",
            "F515": "210F5",
            "F516": "210F6",
            "F518": "20E11",
            "F519": "27694",
            "F51A": "282CD",
            "F51B": "20FB5",
            "F51C": "20E7B",
            "F51D": "2517E",
            "F51F": "20FB6",
            "F520": "21180",
            "F521": "252D8",
            "F522": "2A2BD",
            "F523": "249DA",
            "F524": "2183A",
            "F525": "24177",
            "F526": "2827C",
            "F52A": "2573D",
            "F533": "25B74",
            "F534": "2313D",
            "F53A": "21BC1",
            "F53B": "2F878",
            "F53D": "20086",
            "F54E": "248E9",
            "F553": "2626A",
            "F556": "2634B",
            "F559": "26612",
            "F55A": "26951",
            "F55E": "278B2",
            "F565": "28E0F",
            "F56B": "29810",
            "F572": "20087",
            "F577": "503B",
            "F578": "6DFE",
            "F579": "29C73",
            "F57B": "3DC9",
            "F57C": "888F",
            "F57D": "2414E",
            "F57E": "7077",
            "F57F": "5CF5",
            "F580": "4B20",
            "F581": "251CD",
            "F582": "3559",
            "F583": "25D30",
            "F584": "6122",
            "F585": "28A32",
            "F586": "8FA7",
            "F587": "91F6",
            "F588": "7191",
            "F589": "6719",
            "F58A": "73BA",
            "F58B": "23281",
            "F58C": "2A107",
            "F58D": "3C8B",
            "F58E": "21980",
            "F58F": "4B10",
            "F590": "78E4",
            "F591": "7402",
            "F592": "51AE",
            "F593": "2870F",
            "F594": "4009",
            "F595": "6A63",
            "F596": "2A2BA",
            "F597": "4223",
            "F598": "860F",
            "F59A": "7A2A",
            "F59B": "29947",
            "F59C": "28AEA",
            "F59D": "9755",
            "F59E": "704D",
            "F59F": "5324",
            "F5A0": "2207E",
            "F5A1": "93F4",
            "F5A2": "76D9",
            "F5A3": "289E3",
            "F5A5": "77DD",
            "F5A6": "4EA3",
            "F5A7": "4FF0",
            "F5A8": "50BC",
            "F5A9": "4E2F",
            "F5AA": "4F17",
            "F5AC": "5434",
            "F5AD": "7D8B",
            "F5AE": "5892",
            "F5AF": "58D0",
            "F5B0": "21DB6",
            "F5B1": "5E92",
            "F5B2": "5E99",
            "F5B3": "5FC2",
            "F5B4": "22712",
            "F5B5": "658B",
            "F5B6": "233F9",
            "F5B7": "6919",
            "F5B8": "6A43",
            "F5B9": "23C63",
            "F5BA": "6CFF",
            "F5BC": "7200",
            "F5BD": "24505",
            "F5BE": "738C",
            "F5BF": "3EDB",
            "F5C0": "24A13",
            "F5C1": "5B15",
            "F5C2": "74B9",
            "F5C3": "8B83",
            "F5C4": "25CA4",
            "F5C5": "25695",
            "F5C6": "7A93",
            "F5C7": "7BEC",
            "F5C8": "7CC3",
            "F5C9": "7E6C",
            "F5CA": "82F8",
            "F5CB": "8597",
            "F5CD": "8890",
            "F5CF": "8EB9",
            "F5D1": "8FCF",
            "F5D2": "855F",
            "F5D3": "99E0",
            "F5D4": "9221",
            "F5D6": "28DB9",
            "F5D7": "2143F",
            "F5D8": "4071",
            "F5D9": "42A2",
            "F5DA": "5A1A",
            "F5DE": "9868",
            "F5DF": "676B",
            "F5E0": "4276",
            "F5E1": "573D",
            "F5E3": "85D6",
            "F5E4": "2497B",
            "F5E5": "82BF",
            "F5E6": "2710D",
            "F5E7": "4C81",
            "F5E8": "26D74",
            "F5E9": "5D7B",
            "F5EA": "26B15",
            "F5EB": "26FBE",
            "F5EE": "5B96",
            "F5F1": "7E5B",
            "F694": "289C0",
            "F69E": "23F41",
            "F6AD": "20325",
            "F6B0": "20ED8",
            "F7E6": "200CC",
            "F7E8": "2008A",
            "F7EE": "27607",
        }

        for char in chars:
            res = converter.convert_char(chr(int(char, 16)))
            self.assertEqual(
                res,
                "".join(
                    [chr(int(codepoint, 16)) for codepoint in chars[char].split(",")]
                ),
            )

    def test_hkscs2001_2(self):
        # There don't appear to be any new characters in this file compared to the HKSCS-2001 file
        pass

    def test_hkscs2004(self):
        # New mappings from HKSCS-2004
        # These were added in ISO 10646-1:2000 -> ISO 10646-2:2004

        chars = {
            "F26B": "27267",
            "F26E": "27CB1",
            "F270": "27CC5",
            "F276": "242BF",
            "F277": "23617",
            "F278": "27352",
            "F279": "26E8B",
            "F27A": "270D2",
            "F27C": "2A351",
            "F280": "27C6C",
            "F284": "26B23",
            "F286": "25A54",
            "F287": "21A63",
            "F288": "23E06",
            "F289": "23F61",
            "F28F": "28BB9",
            "F292": "27BEF",
            "F294": "21D5E",
            "F297": "29EB0",
            "F29D": "29945",
            "F303": "31C0",
            "F304": "31C1",
            "F305": "31C2",
            "F306": "31C3",
            "F307": "31C4",
            "F309": "31C5",
            "F30C": "31C6",
            "F30D": "31C7",
            "F310": "31C8",
            "F312": "31C9",
            "F313": "31CA",
            "F314": "31CB",
            "F315": "31CC",
            "F317": "31CD",
            "F318": "31CE",
            "F325": "00CA,0304",
            "F327": "00CA,030C",
            "F344": "00EA,0304",
            "F346": "00EA,030C",
            "F34A": "23DA",
            "F34B": "23DB",
            "F57A": "9FA6",
            "F599": "20A6F",
            "F5A4": "9FA7",
            "F5AB": "9FA8",
            "F5CC": "9FA9",
            "F5CE": "9FAA",
            "F5D0": "9FAB",
            "F5D5": "9FAC",
            "F5EC": "9FAD",
            "F5ED": "9FAE",
            "F5EF": "9FAF",
            "F5F6": "23256",
            "F5F7": "22796",
            "F5FC": "23B1A",
            "F600": "23551",
            "F602": "9FB2",
            "F605": "240EC",
            "F606": "21E23",
            "F60A": "201A4",
            "F60B": "26C41",
            "F60E": "20239",
            "F612": "298FA",
            "F614": "20B9F",
            "F616": "221C1",
            "F617": "2896D",
            "F61A": "29079",
            "F61C": "9FB3",
            "F61D": "2A1B5",
            "F621": "26C46",
            "F623": "286B2",
            "F624": "273FF",
            "F627": "2549A",
            "F62F": "24B0F",
            "F7E5": "31CF",
            "F7EB": "9FB0",
            "F7ED": "9FB1",
        }

        for char in chars:
            res = converter.convert_char(chr(int(char, 16)))
            self.assertEqual(
                res,
                "".join(
                    [chr(int(codepoint, 16)) for codepoint in chars[char].split(",")]
                ),
            )

    def test_hkscs2008(self):
        # New mappings from HKSCS-2008
        # These were added in ISO 10464-2:2004 -> ISO 10464-2:2008
        chars = {
            "F2A1": "21D53",
            "F2A2": "2369E",
            "F2A3": "26021",
            "F2A5": "258DE",
            "F2A9": "24161",
            "F2AA": "2890D",
            "F2AB": "231EA",
            "F2AC": "20A8A",
            "F2AD": "2325E",
            "F2C4": "25DB9",
            "F2C6": "9FC7",
            "F2CB": "2368E",
            "F2CE": "27B65",
            "F2D1": "26E88",
            "F2D3": "25D99",
            "F2D5": "224BC",
            "F2D6": "9FC8",
            "F2D7": "224C1",
            "F2D8": "224C9",
            "F2D9": "224CC",
            "F2DA": "9FC9",
            "F2DC": "235BB",
            "F2DE": "9FCA",
            "F2E0": "2ADFF",
            "F2E3": "9FCB",
        }

        for char in chars:
            res = converter.convert_char(chr(int(char, 16)))
            self.assertEqual(
                res,
                "".join(
                    [chr(int(codepoint, 16)) for codepoint in chars[char].split(",")]
                ),
            )

    def test_hkscs2016(self):
        # On page 2-1 of the HKSCS-2016 standard, there are two characters that have been reassigned
        res = converter.convert_char(chr(0x4CA4))
        self.assertEqual(res, chr(0x9FD0))

        res = converter.convert_char(chr(0x3D1D))
        self.assertEqual(res, chr(0x2A3ED))


class TestInvalid(TestCase):
    def test_no_arg(self):
        with self.assertRaises(TypeError):
            converter.convert_char()

    def test_not_string(self):
        with self.assertRaises(TypeError):
            converter.convert_char(-1)

    def test_arg_too_long(self):
        with self.assertRaises(ValueError):
            converter.convert_char("Hi!")

    def test_multicodepoint_emoji(self):
        with self.assertRaises(ValueError):
            converter.convert_char("👋🏿")


class TestNoChangeNeeded(TestCase):
    def test_latin(self):
        res = converter.convert_char("a")
        self.assertEqual(res, "a")

    def test_chinese(self):
        res = converter.convert_char("亂")
        self.assertEqual(res, "亂")

    def test_number(self):
        res = converter.convert_char("1")
        self.assertEqual(res, "1")

    def test_emoji(self):
        res = converter.convert_char("❓")
        self.assertEqual(res, "❓")

    def test_punctuation(self):
        res = converter.convert_char("'")
        self.assertEqual(res, "'")

    def test_sinhala(self):
        res = converter.convert_string("අ")
        self.assertEqual(res, "අ")
