# Open Traffic Generator Convergence API 0.1.0
# License: MIT

import importlib
import logging
import json
import yaml
import requests
import urllib3
import io
import sys
import time
import grpc
from google.protobuf import json_format

try:
    from snappi_convergence import otgconvergence_pb2_grpc as pb2_grpc
except ImportError:
    import otgconvergence_pb2_grpc as pb2_grpc
try:
    from snappi_convergence import otgconvergence_pb2 as pb2
except ImportError:
    import otgconvergence_pb2 as pb2

try:
    from typing import Union, Dict, List, Any, Literal
except ImportError:
    from typing_extensions import Literal

if sys.version_info[0] == 3:
    unicode = str


class Transport:
    HTTP = "http"
    GRPC = "grpc"


def api(
    location=None,
    transport=None,
    verify=True,
    logger=None,
    loglevel=logging.INFO,
    ext=None,
):
    """Create an instance of an Api class

    generator.Generator outputs a base Api class with the following:
    - an abstract method for each OpenAPI path item object
    - a concrete properties for each unique OpenAPI path item parameter.

    generator.Generator also outputs an HttpApi class that inherits the base
    Api class, implements the abstract methods and uses the common HttpTransport
    class send_recv method to communicate with a REST based server.

    Args
    ----
    - location (str): The location of an Open Traffic Generator server.
    - transport (enum["http", "grpc"]): Transport Type
    - verify (bool): Verify the server's TLS certificate, or a string, in which
      case it must be a path to a CA bundle to use. Defaults to `True`.
      When set to `False`, requests will accept any TLS certificate presented by
      the server, and will ignore hostname mismatches and/or expired
      certificates, which will make your application vulnerable to
      man-in-the-middle (MitM) attacks. Setting verify to `False`
      may be useful during local development or testing.
    - logger (logging.Logger): A user defined logging.logger, if none is provided
      then a default logger with a stdout handler will be provided
    - loglevel (logging.loglevel): The logging package log level.
      The default loglevel is logging.INFO
    - ext (str): Name of an extension package
    """
    params = locals()
    transport_types = ["http", "grpc"]
    if ext is None:
        transport = "http" if transport is None else transport
        if transport not in transport_types:
            raise Exception(
                "{transport} is not within valid transport types {transport_types}".format(
                    transport=transport, transport_types=transport_types
                )
            )
        if transport == "http":
            return HttpApi(**params)
        else:
            return GrpcApi(**params)
    try:
        if transport is not None:
            raise Exception(
                "ext and transport are not mutually exclusive. Please configure one of them."
            )
        lib = importlib.import_module("snappi_{}.snappi_convergence_api".format(ext))
        return lib.Api(**params)
    except ImportError as err:
        msg = "Extension %s is not installed or invalid: %s"
        raise Exception(msg % (ext, err))


class HttpTransport(object):
    def __init__(self, **kwargs):
        """Use args from api() method to instantiate an HTTP transport"""
        self.location = (
            kwargs["location"]
            if "location" in kwargs and kwargs["location"] is not None
            else "https://localhost:443"
        )
        self.verify = kwargs["verify"] if "verify" in kwargs else False
        self.logger = kwargs["logger"] if "logger" in kwargs else None
        self.loglevel = kwargs["loglevel"] if "loglevel" in kwargs else logging.DEBUG
        if self.logger is None:
            stdout_handler = logging.StreamHandler(sys.stdout)
            formatter = logging.Formatter(
                fmt="%(asctime)s [%(name)s] [%(levelname)s] %(message)s",
                datefmt="%Y-%m-%d %H:%M:%S",
            )
            formatter.converter = time.gmtime
            stdout_handler.setFormatter(formatter)
            self.logger = logging.Logger(self.__module__, level=self.loglevel)
            self.logger.addHandler(stdout_handler)
        self.logger.debug(
            "HttpTransport args: {}".format(
                ", ".join(["{}={!r}".format(k, v) for k, v in kwargs.items()])
            )
        )
        self.set_verify(self.verify)
        self._session = requests.Session()

    def set_verify(self, verify):
        self.verify = verify
        if self.verify is False:
            urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
            self.logger.warning("Certificate verification is disabled")

    def send_recv(
        self, method, relative_url, payload=None, return_object=None, headers=None
    ):
        url = "%s%s" % (self.location, relative_url)
        data = None
        headers = headers or {"Content-Type": "application/json"}
        if payload is not None:
            if isinstance(payload, bytes):
                data = payload
                headers["Content-Type"] = "application/octet-stream"
            elif isinstance(payload, (str, unicode)):
                data = payload
            elif isinstance(payload, OpenApiBase):
                data = payload.serialize()
            else:
                raise Exception("Type of payload provided is unknown")
        response = self._session.request(
            method=method,
            url=url,
            data=data,
            verify=False,
            allow_redirects=True,
            # TODO: add a timeout here
            headers=headers,
        )
        if response.ok:
            if "application/json" in response.headers["content-type"]:
                # TODO: we might want to check for utf-8 charset and decode
                # accordingly, but current impl works for now
                response_dict = yaml.safe_load(response.text)
                if return_object is None:
                    # if response type is not provided, return dictionary
                    # instead of python object
                    return response_dict
                else:
                    return return_object.deserialize(response_dict)
            elif "application/octet-stream" in response.headers["content-type"]:
                return io.BytesIO(response.content)
            else:
                # TODO: for now, return bare response object for unknown
                # content types
                return response
        else:
            raise Exception(response.status_code, yaml.safe_load(response.text))


class OpenApiBase(object):
    """Base class for all generated classes"""

    JSON = "json"
    YAML = "yaml"
    DICT = "dict"

    __slots__ = ()

    def __init__(self):
        pass

    def serialize(self, encoding=JSON):
        """Serialize the current object according to a specified encoding.

        Args
        ----
        - encoding (str[json, yaml, dict]): The object will be recursively
            serialized according to the specified encoding.
            The supported encodings are json, yaml and python dict.

        Returns
        -------
        - obj(Union[str, dict]): A str or dict object depending on the specified
            encoding. The json and yaml encodings will return a str object and
            the dict encoding will return a python dict object.
        """
        if encoding == OpenApiBase.JSON:
            return json.dumps(self._encode(), indent=2, sort_keys=True)
        elif encoding == OpenApiBase.YAML:
            return yaml.safe_dump(self._encode())
        elif encoding == OpenApiBase.DICT:
            return self._encode()
        else:
            raise NotImplementedError("Encoding %s not supported" % encoding)

    def _encode(self):
        raise NotImplementedError()

    def deserialize(self, serialized_object):
        """Deserialize a python object into the current object.

        If the input `serialized_object` does not match the current
        openapi object an exception will be raised.

        Args
        ----
        - serialized_object (Union[str, dict]): The object to deserialize.
            If the serialized_object is of type str then the internal encoding
            of the serialized_object must be json or yaml.

        Returns
        -------
        - obj(OpenApiObject): This object with all the
            serialized_object deserialized within.
        """
        if isinstance(serialized_object, (str, unicode)):
            serialized_object = yaml.safe_load(serialized_object)
        self._decode(serialized_object)
        return self

    def _decode(self, dict_object):
        raise NotImplementedError()


class OpenApiValidator(object):

    __slots__ = ()

    def __init__(self):
        pass

    def validate_mac(self, mac):
        if mac is None or not isinstance(mac, (str, unicode)) or mac.count(" ") != 0:
            return False
        try:
            if len(mac) != 17:
                return False
            return all([0 <= int(oct, 16) <= 255 for oct in mac.split(":")])
        except Exception:
            return False

    def validate_ipv4(self, ip):
        if ip is None or not isinstance(ip, (str, unicode)) or ip.count(" ") != 0:
            return False
        if len(ip.split(".")) != 4:
            return False
        try:
            return all([0 <= int(oct) <= 255 for oct in ip.split(".", 3)])
        except Exception:
            return False

    def validate_ipv6(self, ip):
        if ip is None or not isinstance(ip, (str, unicode)):
            return False
        ip = ip.strip()
        if (
            ip.count(" ") > 0
            or ip.count(":") > 7
            or ip.count("::") > 1
            or ip.count(":::") > 0
        ):
            return False
        if (ip[0] == ":" and ip[:2] != "::") or (ip[-1] == ":" and ip[-2:] != "::"):
            return False
        if ip.count("::") == 0 and ip.count(":") != 7:
            return False
        if ip == "::":
            return True
        if ip[:2] == "::":
            ip = ip.replace("::", "0:")
        elif ip[-2:] == "::":
            ip = ip.replace("::", ":0")
        else:
            ip = ip.replace("::", ":0:")
        try:
            return all(
                [
                    True
                    if (0 <= int(oct, 16) <= 65535) and (1 <= len(oct) <= 4)
                    else False
                    for oct in ip.split(":")
                ]
            )
        except Exception:
            return False

    def validate_hex(self, hex):
        if hex is None or not isinstance(hex, (str, unicode)):
            return False
        try:
            int(hex, 16)
            return True
        except Exception:
            return False

    def validate_integer(self, value, min, max):
        if value is None or not isinstance(value, int):
            return False
        if value < 0:
            return False
        if min is not None and value < min:
            return False
        if max is not None and value > max:
            return False
        return True

    def validate_float(self, value):
        return isinstance(value, (int, float))

    def validate_string(self, value, min_length, max_length):
        if value is None or not isinstance(value, (str, unicode)):
            return False
        if min_length is not None and len(value) < min_length:
            return False
        if max_length is not None and len(value) > max_length:
            return False
        return True

    def validate_bool(self, value):
        return isinstance(value, bool)

    def validate_list(self, value, itemtype, min, max, min_length, max_length):
        if value is None or not isinstance(value, list):
            return False
        v_obj = getattr(self, "validate_{}".format(itemtype), None)
        if v_obj is None:
            raise AttributeError("{} is not a valid attribute".format(itemtype))
        v_obj_lst = []
        for item in value:
            if itemtype == "integer":
                v_obj_lst.append(v_obj(item, min, max))
            elif itemtype == "string":
                v_obj_lst.append(v_obj(item, min_length, max_length))
            else:
                v_obj_lst.append(v_obj(item))
        return v_obj_lst

    def validate_binary(self, value):
        if value is None or not isinstance(value, (str, unicode)):
            return False
        return all([True if int(bin) == 0 or int(bin) == 1 else False for bin in value])

    def types_validation(
        self,
        value,
        type_,
        err_msg,
        itemtype=None,
        min=None,
        max=None,
        min_length=None,
        max_length=None,
    ):
        type_map = {
            int: "integer",
            str: "string",
            float: "float",
            bool: "bool",
            list: "list",
            "int64": "integer",
            "int32": "integer",
            "double": "float",
        }
        if type_ in type_map:
            type_ = type_map[type_]
        if itemtype is not None and itemtype in type_map:
            itemtype = type_map[itemtype]
        v_obj = getattr(self, "validate_{}".format(type_), None)
        if v_obj is None:
            msg = "{} is not a valid or unsupported format".format(type_)
            raise TypeError(msg)
        if type_ == "list":
            verdict = v_obj(value, itemtype, min, max, min_length, max_length)
            if all(verdict) is True:
                return
            err_msg = "{} \n {} are not valid".format(
                err_msg,
                [value[index] for index, item in enumerate(verdict) if item is False],
            )
            verdict = False
        elif type_ == "integer":
            verdict = v_obj(value, min, max)
            if verdict is True:
                return
            min_max = ""
            if min is not None:
                min_max = ", expected min {}".format(min)
            if max is not None:
                min_max = min_max + ", expected max {}".format(max)
            err_msg = "{} \n got {} of type {} {}".format(
                err_msg, value, type(value), min_max
            )
        elif type_ == "string":
            verdict = v_obj(value, min_length, max_length)
            if verdict is True:
                return
            msg = ""
            if min_length is not None:
                msg = ", expected min {}".format(min_length)
            if max_length is not None:
                msg = msg + ", expected max {}".format(max_length)
            err_msg = "{} \n got {} of type {} {}".format(
                err_msg, value, type(value), msg
            )
        else:
            verdict = v_obj(value)
        if verdict is False:
            raise TypeError(err_msg)


class OpenApiObject(OpenApiBase, OpenApiValidator):
    """Base class for any /components/schemas object

    Every OpenApiObject is reuseable within the schema so it can
    exist in multiple locations within the hierarchy.
    That means it can exist in multiple locations as a
    leaf, parent/choice or parent.
    """

    __slots__ = ("_properties", "_parent", "_choice")
    _DEFAULTS = {}
    _TYPES = {}
    _REQUIRED = []

    def __init__(self, parent=None, choice=None):
        super(OpenApiObject, self).__init__()
        self._parent = parent
        self._choice = choice
        self._properties = {}

    @property
    def parent(self):
        return self._parent

    def _set_choice(self, name):
        if self._has_choice(name):
            for enum in self._TYPES["choice"]["enum"]:
                if enum in self._properties and name != enum:
                    self._properties.pop(enum)
            self._properties["choice"] = name

    def _has_choice(self, name):
        if (
            "choice" in dir(self)
            and "_TYPES" in dir(self)
            and "choice" in self._TYPES
            and name in self._TYPES["choice"]["enum"]
        ):
            return True
        else:
            return False

    def _get_property(self, name, default_value=None, parent=None, choice=None):
        if name in self._properties and self._properties[name] is not None:
            return self._properties[name]
        if isinstance(default_value, type) is True:
            self._set_choice(name)
            if "_choice" in default_value.__slots__:
                self._properties[name] = default_value(parent=parent, choice=choice)
            else:
                self._properties[name] = default_value(parent=parent)
            if (
                "_DEFAULTS" in dir(self._properties[name])
                and "choice" in self._properties[name]._DEFAULTS
            ):
                getattr(
                    self._properties[name], self._properties[name]._DEFAULTS["choice"]
                )
        else:
            if default_value is None and name in self._DEFAULTS:
                self._set_choice(name)
                self._properties[name] = self._DEFAULTS[name]
            else:
                self._properties[name] = default_value
        return self._properties[name]

    def _set_property(self, name, value, choice=None):
        if name in self._DEFAULTS and value is None:
            self._set_choice(name)
            self._properties[name] = self._DEFAULTS[name]
        else:
            self._set_choice(name)
            self._properties[name] = value
        if self._parent is not None and self._choice is not None and value is not None:
            self._parent._set_property("choice", self._choice)

    def _encode(self):
        """Helper method for serialization"""
        output = {}
        self._validate_required()
        for key, value in self._properties.items():
            self._validate_types(key, value)
            if isinstance(value, (OpenApiObject, OpenApiIter)):
                output[key] = value._encode()
            elif value is not None:
                if (
                    key in self._TYPES
                    and "format" in self._TYPES[key]
                    and self._TYPES[key]["format"] == "int64"
                ):
                    value = str(value)
                output[key] = value
        return output

    def _decode(self, obj):
        dtypes = [list, str, int, float, bool]
        for property_name, property_value in obj.items():
            if property_name in self._TYPES:
                if isinstance(property_value, dict):
                    child = self._get_child_class(property_name)
                    if "choice" in child[1]._TYPES and "_parent" in child[1].__slots__:
                        property_value = child[1](self, property_name)._decode(
                            property_value
                        )
                    elif "_parent" in child[1].__slots__:
                        property_value = child[1](self)._decode(property_value)
                    else:
                        property_value = child[1]()._decode(property_value)
                elif (
                    isinstance(property_value, list)
                    and property_name in self._TYPES
                    and self._TYPES[property_name]["type"] not in dtypes
                ):
                    child = self._get_child_class(property_name, True)
                    openapi_list = child[0]()
                    for item in property_value:
                        item = child[1]()._decode(item)
                        openapi_list._items.append(item)
                    property_value = openapi_list
                elif property_name in self._DEFAULTS and property_value is None:
                    if isinstance(self._DEFAULTS[property_name], tuple(dtypes)):
                        property_value = self._DEFAULTS[property_name]
                self._set_choice(property_name)
                if (
                    "format" in self._TYPES[property_name]
                    and self._TYPES[property_name]["format"] == "int64"
                ):
                    property_value = int(property_value)
                self._properties[property_name] = property_value
            self._validate_types(property_name, property_value)
        self._validate_required()
        return self

    def _get_child_class(self, property_name, is_property_list=False):
        list_class = None
        class_name = self._TYPES[property_name]["type"]
        module = importlib.import_module(self.__module__)
        object_class = getattr(module, class_name)
        if is_property_list is True:
            list_class = object_class
            object_class = getattr(module, class_name[0:-4])
        return (list_class, object_class)

    def __str__(self):
        return self.serialize(encoding=self.YAML)

    def __deepcopy__(self, memo):
        """Creates a deep copy of the current object"""
        return self.__class__().deserialize(self.serialize())

    def __copy__(self):
        """Creates a deep copy of the current object"""
        return self.__deepcopy__(None)

    def __eq__(self, other):
        return self.__str__() == other.__str__()

    def clone(self):
        """Creates a deep copy of the current object"""
        return self.__deepcopy__(None)

    def _validate_required(self):
        """Validates the required properties are set
        Use getattr as it will set any defaults prior to validating
        """
        if getattr(self, "_REQUIRED", None) is None:
            return
        for name in self._REQUIRED:
            if getattr(self, name, None) is None:
                msg = (
                    "{} is a mandatory property of {}"
                    " and should not be set to None".format(
                        name,
                        self.__class__,
                    )
                )
                raise ValueError(msg)

    def _validate_types(self, property_name, property_value):
        common_data_types = [list, str, int, float, bool]
        if property_name not in self._TYPES:
            # raise ValueError("Invalid Property {}".format(property_name))
            return
        details = self._TYPES[property_name]
        if (
            property_value is None
            and property_name not in self._DEFAULTS
            and property_name not in self._REQUIRED
        ):
            return
        if "enum" in details and property_value not in details["enum"]:
            msg = "property {} shall be one of these" " {} enum, but got {} at {}"
            raise TypeError(
                msg.format(
                    property_name, details["enum"], property_value, self.__class__
                )
            )
        if details["type"] in common_data_types and "format" not in details:
            msg = "property {} shall be of type {} at {}".format(
                property_name, details["type"], self.__class__
            )
            self.types_validation(
                property_value,
                details["type"],
                msg,
                details.get("itemtype"),
                details.get("minimum"),
                details.get("maximum"),
                details.get("minLength"),
                details.get("maxLength"),
            )

        if details["type"] not in common_data_types:
            class_name = details["type"]
            # TODO Need to revisit importlib
            module = importlib.import_module(self.__module__)
            object_class = getattr(module, class_name)
            if not isinstance(property_value, object_class):
                msg = "property {} shall be of type {}," " but got {} at {}"
                raise TypeError(
                    msg.format(
                        property_name, class_name, type(property_value), self.__class__
                    )
                )
        if "format" in details:
            msg = "Invalid {} format, expected {} at {}".format(
                property_value, details["format"], self.__class__
            )
            _type = details["type"] if details["type"] is list else details["format"]
            self.types_validation(
                property_value,
                _type,
                msg,
                details["format"],
                details.get("minimum"),
                details.get("maximum"),
                details.get("minLength"),
                details.get("maxLength"),
            )

    def validate(self):
        self._validate_required()
        for key, value in self._properties.items():
            self._validate_types(key, value)

    def get(self, name, with_default=False):
        """
        getattr for openapi object
        """
        if self._properties.get(name) is not None:
            return self._properties[name]
        elif with_default:
            # TODO need to find a way to avoid getattr
            choice = self._properties.get("choice") if "choice" in dir(self) else None
            getattr(self, name)
            if "choice" in dir(self):
                if choice is None and "choice" in self._properties:
                    self._properties.pop("choice")
                else:
                    self._properties["choice"] = choice
            return self._properties.pop(name)
        return None


class OpenApiIter(OpenApiBase):
    """Container class for OpenApiObject

    Inheriting classes contain 0..n instances of an OpenAPI components/schemas
    object.
    - config.flows.flow(name="1").flow(name="2").flow(name="3")

    The __getitem__ method allows getting an instance using ordinal.
    - config.flows[0]
    - config.flows[1:]
    - config.flows[0:1]
    - f1, f2, f3 = config.flows

    The __iter__ method allows for iterating across the encapsulated contents
    - for flow in config.flows:
    """

    __slots__ = ("_index", "_items")
    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self):
        super(OpenApiIter, self).__init__()
        self._index = -1
        self._items = []

    def __len__(self):
        return len(self._items)

    def _getitem(self, key):
        found = None
        if isinstance(key, int):
            found = self._items[key]
        elif isinstance(key, slice) is True:
            start, stop, step = key.indices(len(self))
            sliced = self.__class__()
            for i in range(start, stop, step):
                sliced._items.append(self._items[i])
            return sliced
        elif isinstance(key, str):
            for item in self._items:
                if item.name == key:
                    found = item
        if found is None:
            raise IndexError()
        if (
            self._GETITEM_RETURNS_CHOICE_OBJECT is True
            and found._properties.get("choice") is not None
        ):
            return found._properties[found._properties["choice"]]
        return found

    def _iter(self):
        self._index = -1
        return self

    def _next(self):
        if self._index + 1 >= len(self._items):
            raise StopIteration
        else:
            self._index += 1
        return self.__getitem__(self._index)

    def __getitem__(self, key):
        raise NotImplementedError("This should be overridden by the generator")

    def _add(self, item):
        self._items.append(item)
        self._index = len(self._items) - 1

    def remove(self, index):
        del self._items[index]
        self._index = len(self._items) - 1

    def append(self, item):
        """Append an item to the end of OpenApiIter
        TBD: type check, raise error on mismatch
        """
        if isinstance(item, OpenApiObject) is False:
            raise Exception("Item is not an instance of OpenApiObject")
        self._add(item)
        return self

    def clear(self):
        del self._items[:]
        self._index = -1

    def _encode(self):
        return [item._encode() for item in self._items]

    def _decode(self, encoded_list):
        item_class_name = self.__class__.__name__.replace("Iter", "")
        module = importlib.import_module(self.__module__)
        object_class = getattr(module, item_class_name)
        self.clear()
        for item in encoded_list:
            self._add(object_class()._decode(item))

    def __copy__(self):
        raise NotImplementedError(
            "Shallow copy of OpenApiIter objects is not supported"
        )

    def __deepcopy__(self, memo):
        raise NotImplementedError("Deep copy of OpenApiIter objects is not supported")

    def __str__(self):
        return yaml.safe_dump(self._encode())

    def __eq__(self, other):
        return self.__str__() == other.__str__()


class ConvergenceConfig(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "config": {"type": "Config"},
        "event_location": {"type": str},
        "rx_rate_threshold": {"type": float},
    }  # type: Dict[str, str]

    _REQUIRED = ("config",)  # type: tuple(str)

    _DEFAULTS = {
        "event_location": "",
        "rx_rate_threshold": 95.0,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, event_location="", rx_rate_threshold=95):
        super(ConvergenceConfig, self).__init__()
        self._parent = parent
        self._set_property("event_location", event_location)
        self._set_property("rx_rate_threshold", rx_rate_threshold)

    @property
    def config(self):
        # type: () -> Config
        """config getter

        A container for all models that are part of the configuration.A container for all models that are part of the configuration.The core open traffic generator configuration

        Returns: Config
        """
        return self._get_property("config", Config)

    @property
    def event_location(self):
        # type: () -> str
        """event_location getter

        The location from which events will be generated.

        Returns: str
        """
        return self._get_property("event_location")

    @event_location.setter
    def event_location(self, value):
        """event_location setter

        The location from which events will be generated.

        value: str
        """
        self._set_property("event_location", value)

    @property
    def rx_rate_threshold(self):
        # type: () -> float
        """rx_rate_threshold getter

        Convergence will be calculated when the rx rate of a flow passes . below then above the threshold value.

        Returns: float
        """
        return self._get_property("rx_rate_threshold")

    @rx_rate_threshold.setter
    def rx_rate_threshold(self, value):
        """rx_rate_threshold setter

        Convergence will be calculated when the rx rate of a flow passes . below then above the threshold value.

        value: float
        """
        self._set_property("rx_rate_threshold", value)


class Config(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ports": {"type": "PortIter"},
        "lags": {"type": "LagIter"},
        "layer1": {"type": "Layer1Iter"},
        "captures": {"type": "CaptureIter"},
        "devices": {"type": "DeviceIter"},
        "flows": {"type": "FlowIter"},
        "events": {"type": "Event"},
        "options": {"type": "ConfigOptions"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(Config, self).__init__()
        self._parent = parent

    @property
    def ports(self):
        # type: () -> PortIter
        """ports getter

        The ports that will be configured on the traffic generator.

        Returns: PortIter
        """
        return self._get_property("ports", PortIter, self._parent, self._choice)

    @property
    def lags(self):
        # type: () -> LagIter
        """lags getter

        The LAGs that will be configured on the traffic generator.

        Returns: LagIter
        """
        return self._get_property("lags", LagIter, self._parent, self._choice)

    @property
    def layer1(self):
        # type: () -> Layer1Iter
        """layer1 getter

        The layer1 settings that will be configured on the traffic generator.

        Returns: Layer1Iter
        """
        return self._get_property("layer1", Layer1Iter, self._parent, self._choice)

    @property
    def captures(self):
        # type: () -> CaptureIter
        """captures getter

        The capture settings that will be configured on the traffic generator.

        Returns: CaptureIter
        """
        return self._get_property("captures", CaptureIter, self._parent, self._choice)

    @property
    def devices(self):
        # type: () -> DeviceIter
        """devices getter

        The emulated devices that will be configured on the traffic generator.. Each device contains configurations for network interfaces and . protocols running on top of those interfaces.

        Returns: DeviceIter
        """
        return self._get_property("devices", DeviceIter, self._parent, self._choice)

    @property
    def flows(self):
        # type: () -> FlowIter
        """flows getter

        The flows that will be configured on the traffic generator.

        Returns: FlowIter
        """
        return self._get_property("flows", FlowIter, self._parent, self._choice)

    @property
    def events(self):
        # type: () -> Event
        """events getter

        The optional container for event configuration.The optional container for event configuration.

        Returns: Event
        """
        return self._get_property("events", Event)

    @property
    def options(self):
        # type: () -> ConfigOptions
        """options getter

        Global configuration options.Global configuration options.

        Returns: ConfigOptions
        """
        return self._get_property("options", ConfigOptions)


class Port(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "location": {"type": str},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, location=None, name=None):
        super(Port, self).__init__()
        self._parent = parent
        self._set_property("location", location)
        self._set_property("name", name)

    @property
    def location(self):
        # type: () -> str
        """location getter

        The location of a test port. It is the endpoint where packets will emit from.. Test port locations can be the following:. - physical appliance with multiple ports. - physical chassis with multiple cards and ports. - local interface. - virtual machine, docker container, kubernetes cluster. . The test port location format is implementation specific. Use the /results/capabilities API to determine what formats an implementation supports for the location property.. Get the configured location state by using the /results/port API.

        Returns: str
        """
        return self._get_property("location")

    @location.setter
    def location(self, value):
        """location setter

        The location of a test port. It is the endpoint where packets will emit from.. Test port locations can be the following:. - physical appliance with multiple ports. - physical chassis with multiple cards and ports. - local interface. - virtual machine, docker container, kubernetes cluster. . The test port location format is implementation specific. Use the /results/capabilities API to determine what formats an implementation supports for the location property.. Get the configured location state by using the /results/port API.

        value: str
        """
        self._set_property("location", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class PortIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(PortIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Port]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> PortIter
        return self._iter()

    def __next__(self):
        # type: () -> Port
        return self._next()

    def next(self):
        # type: () -> Port
        return self._next()

    def port(self, location=None, name=None):
        # type: (str,str) -> PortIter
        """Factory method that creates an instance of the Port class

        An abstract test port.

        Returns: PortIter
        """
        item = Port(parent=self._parent, location=location, name=name)
        self._add(item)
        return self

    def add(self, location=None, name=None):
        # type: (str,str) -> Port
        """Add method that creates and returns an instance of the Port class

        An abstract test port.

        Returns: Port
        """
        item = Port(parent=self._parent, location=location, name=name)
        self._add(item)
        return item


class Lag(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ports": {"type": "LagPortIter"},
        "protocol": {"type": "LagProtocol"},
        "min_links": {
            "type": int,
            "format": "int32",
            "minimum": 0,
            "maximum": 32,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "min_links": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, min_links=1, name=None):
        super(Lag, self).__init__()
        self._parent = parent
        self._set_property("min_links", min_links)
        self._set_property("name", name)

    @property
    def ports(self):
        # type: () -> LagPortIter
        """ports getter

        TBD

        Returns: LagPortIter
        """
        return self._get_property("ports", LagPortIter, self._parent, self._choice)

    @property
    def protocol(self):
        # type: () -> LagProtocol
        """protocol getter



        Returns: LagProtocol
        """
        return self._get_property("protocol", LagProtocol)

    @property
    def min_links(self):
        # type: () -> int
        """min_links getter

        Specifies the mininum number of member interfaces that must be active for the aggregate interface to be available

        Returns: int
        """
        return self._get_property("min_links")

    @min_links.setter
    def min_links(self, value):
        """min_links setter

        Specifies the mininum number of member interfaces that must be active for the aggregate interface to be available

        value: int
        """
        self._set_property("min_links", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class LagPort(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "port_name": {"type": str},
        "lacp": {"type": "LagPortLacp"},
        "ethernet": {"type": "DeviceEthernetBase"},
    }  # type: Dict[str, str]

    _REQUIRED = ("port_name", "ethernet")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, port_name=None):
        super(LagPort, self).__init__()
        self._parent = parent
        self._set_property("port_name", port_name)

    @property
    def port_name(self):
        # type: () -> str
        """port_name getter

        The name of a port object that will be part of the LAG. . . x-constraint:. - /components/schemas/Port/properties/name.

        Returns: str
        """
        return self._get_property("port_name")

    @port_name.setter
    def port_name(self, value):
        """port_name setter

        The name of a port object that will be part of the LAG. . . x-constraint:. - /components/schemas/Port/properties/name.

        value: str
        """
        self._set_property("port_name", value)

    @property
    def lacp(self):
        # type: () -> LagPortLacp
        """lacp getter

        The container for link aggregation control protocol settings of a LAG member (port).The container for link aggregation control protocol settings of a LAG member (port).

        Returns: LagPortLacp
        """
        return self._get_property("lacp", LagPortLacp)

    @property
    def ethernet(self):
        # type: () -> DeviceEthernetBase
        """ethernet getter

        Base Ethernet interface.Base Ethernet interface.

        Returns: DeviceEthernetBase
        """
        return self._get_property("ethernet", DeviceEthernetBase)


class LagPortLacp(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "actor_port_number": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "actor_port_priority": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "actor_activity": {
            "type": str,
            "enum": [
                "passive",
                "active",
            ],
        },
        "lacpdu_periodic_time_interval": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "lacpdu_timeout": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "actor_port_number": 0,
        "actor_port_priority": 1,
        "actor_activity": "active",
        "lacpdu_periodic_time_interval": 0,
        "lacpdu_timeout": 0,
    }  # type: Dict[str, Union(type)]

    PASSIVE = "passive"  # type: str
    ACTIVE = "active"  # type: str

    FAST = 1  #
    SLOW = 30  #
    AUTO = 0  #

    SHORT = 3  #
    LONG = 90  #
    AUTO = 0  #

    def __init__(
        self,
        parent=None,
        actor_port_number=0,
        actor_port_priority=1,
        actor_activity="active",
        lacpdu_periodic_time_interval=0,
        lacpdu_timeout=0,
    ):
        super(LagPortLacp, self).__init__()
        self._parent = parent
        self._set_property("actor_port_number", actor_port_number)
        self._set_property("actor_port_priority", actor_port_priority)
        self._set_property("actor_activity", actor_activity)
        self._set_property(
            "lacpdu_periodic_time_interval", lacpdu_periodic_time_interval
        )
        self._set_property("lacpdu_timeout", lacpdu_timeout)

    @property
    def actor_port_number(self):
        # type: () -> int
        """actor_port_number getter

        The actor port number

        Returns: int
        """
        return self._get_property("actor_port_number")

    @actor_port_number.setter
    def actor_port_number(self, value):
        """actor_port_number setter

        The actor port number

        value: int
        """
        self._set_property("actor_port_number", value)

    @property
    def actor_port_priority(self):
        # type: () -> int
        """actor_port_priority getter

        The actor port priority

        Returns: int
        """
        return self._get_property("actor_port_priority")

    @actor_port_priority.setter
    def actor_port_priority(self, value):
        """actor_port_priority setter

        The actor port priority

        value: int
        """
        self._set_property("actor_port_priority", value)

    @property
    def actor_activity(self):
        # type: () -> Union[Literal["active"], Literal["passive"]]
        """actor_activity getter

        Sets the value of LACP actor activity as either passive or active.. Passive indicates the port's preference for not transmitting LACPDUs unless its partner's control is Active.. Active indicates the port's preference to participate in the protocol regardless of the partner's control value.

        Returns: Union[Literal["active"], Literal["passive"]]
        """
        return self._get_property("actor_activity")

    @actor_activity.setter
    def actor_activity(self, value):
        """actor_activity setter

        Sets the value of LACP actor activity as either passive or active.. Passive indicates the port's preference for not transmitting LACPDUs unless its partner's control is Active.. Active indicates the port's preference to participate in the protocol regardless of the partner's control value.

        value: Union[Literal["active"], Literal["passive"]]
        """
        self._set_property("actor_activity", value)

    @property
    def lacpdu_periodic_time_interval(self):
        # type: () -> int
        """lacpdu_periodic_time_interval getter

        This field defines how frequently LACPDUs are sent to the link partner

        Returns: int
        """
        return self._get_property("lacpdu_periodic_time_interval")

    @lacpdu_periodic_time_interval.setter
    def lacpdu_periodic_time_interval(self, value):
        """lacpdu_periodic_time_interval setter

        This field defines how frequently LACPDUs are sent to the link partner

        value: int
        """
        self._set_property("lacpdu_periodic_time_interval", value)

    @property
    def lacpdu_timeout(self):
        # type: () -> int
        """lacpdu_timeout getter

        This timer is used to detect whether received protocol information has expired

        Returns: int
        """
        return self._get_property("lacpdu_timeout")

    @lacpdu_timeout.setter
    def lacpdu_timeout(self, value):
        """lacpdu_timeout setter

        This timer is used to detect whether received protocol information has expired

        value: int
        """
        self._set_property("lacpdu_timeout", value)


class DeviceEthernetBase(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "mac": {
            "type": str,
            "format": "mac",
        },
        "mtu": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "vlans": {"type": "DeviceVlanIter"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("mac", "name")  # type: tuple(str)

    _DEFAULTS = {
        "mtu": 1500,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, mac=None, mtu=1500, name=None):
        super(DeviceEthernetBase, self).__init__()
        self._parent = parent
        self._set_property("mac", mac)
        self._set_property("mtu", mtu)
        self._set_property("name", name)

    @property
    def mac(self):
        # type: () -> str
        """mac getter

        Media Access Control address.

        Returns: str
        """
        return self._get_property("mac")

    @mac.setter
    def mac(self, value):
        """mac setter

        Media Access Control address.

        value: str
        """
        self._set_property("mac", value)

    @property
    def mtu(self):
        # type: () -> int
        """mtu getter

        Maximum Transmission Unit.

        Returns: int
        """
        return self._get_property("mtu")

    @mtu.setter
    def mtu(self, value):
        """mtu setter

        Maximum Transmission Unit.

        value: int
        """
        self._set_property("mtu", value)

    @property
    def vlans(self):
        # type: () -> DeviceVlanIter
        """vlans getter

        List of VLANs

        Returns: DeviceVlanIter
        """
        return self._get_property("vlans", DeviceVlanIter, self._parent, self._choice)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class DeviceVlan(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "tpid": {
            "type": str,
            "enum": [
                "x8100",
                "x88A8",
                "x9100",
                "x9200",
                "x9300",
            ],
        },
        "priority": {
            "type": int,
            "minimum": 0,
            "maximum": 3,
        },
        "id": {
            "type": int,
            "minimum": 0,
            "maximum": 4095,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "tpid": "x8100",
        "priority": 0,
        "id": 1,
    }  # type: Dict[str, Union(type)]

    X8100 = "x8100"  # type: str
    X88A8 = "x88A8"  # type: str
    X9100 = "x9100"  # type: str
    X9200 = "x9200"  # type: str
    X9300 = "x9300"  # type: str

    def __init__(self, parent=None, tpid="x8100", priority=0, id=1, name=None):
        super(DeviceVlan, self).__init__()
        self._parent = parent
        self._set_property("tpid", tpid)
        self._set_property("priority", priority)
        self._set_property("id", id)
        self._set_property("name", name)

    @property
    def tpid(self):
        # type: () -> Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]]
        """tpid getter

        Tag protocol identifier

        Returns: Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]]
        """
        return self._get_property("tpid")

    @tpid.setter
    def tpid(self, value):
        """tpid setter

        Tag protocol identifier

        value: Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]]
        """
        self._set_property("tpid", value)

    @property
    def priority(self):
        # type: () -> int
        """priority getter

        Priority code point

        Returns: int
        """
        return self._get_property("priority")

    @priority.setter
    def priority(self, value):
        """priority setter

        Priority code point

        value: int
        """
        self._set_property("priority", value)

    @property
    def id(self):
        # type: () -> int
        """id getter

        VLAN identifier

        Returns: int
        """
        return self._get_property("id")

    @id.setter
    def id(self, value):
        """id setter

        VLAN identifier

        value: int
        """
        self._set_property("id", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class DeviceVlanIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceVlanIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceVlan]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceVlanIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceVlan
        return self._next()

    def next(self):
        # type: () -> DeviceVlan
        return self._next()

    def vlan(self, tpid="x8100", priority=0, id=1, name=None):
        # type: (Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]],int,int,str) -> DeviceVlanIter
        """Factory method that creates an instance of the DeviceVlan class

        Emulated VLAN protocol.

        Returns: DeviceVlanIter
        """
        item = DeviceVlan(
            parent=self._parent, tpid=tpid, priority=priority, id=id, name=name
        )
        self._add(item)
        return self

    def add(self, tpid="x8100", priority=0, id=1, name=None):
        # type: (Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]],int,int,str) -> DeviceVlan
        """Add method that creates and returns an instance of the DeviceVlan class

        Emulated VLAN protocol.

        Returns: DeviceVlan
        """
        item = DeviceVlan(
            parent=self._parent, tpid=tpid, priority=priority, id=id, name=name
        )
        self._add(item)
        return item


class LagPortIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(LagPortIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[LagPort]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> LagPortIter
        return self._iter()

    def __next__(self):
        # type: () -> LagPort
        return self._next()

    def next(self):
        # type: () -> LagPort
        return self._next()

    def port(self, port_name=None):
        # type: (str) -> LagPortIter
        """Factory method that creates an instance of the LagPort class

        The container for a port's ethernet interface and LAG protocol settings

        Returns: LagPortIter
        """
        item = LagPort(parent=self._parent, port_name=port_name)
        self._add(item)
        return self

    def add(self, port_name=None):
        # type: (str) -> LagPort
        """Add method that creates and returns an instance of the LagPort class

        The container for a port's ethernet interface and LAG protocol settings

        Returns: LagPort
        """
        item = LagPort(parent=self._parent, port_name=port_name)
        self._add(item)
        return item


class LagProtocol(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "lacp",
                "static",
            ],
        },
        "lacp": {"type": "LagProtocolLacp"},
        "static": {"type": "LagProtocolStatic"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "lacp",
    }  # type: Dict[str, Union(type)]

    LACP = "lacp"  # type: str
    STATIC = "static"  # type: str

    def __init__(self, parent=None, choice=None):
        super(LagProtocol, self).__init__()
        self._parent = parent
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def lacp(self):
        # type: () -> LagProtocolLacp
        """Factory property that returns an instance of the LagProtocolLacp class

        The container for link aggregation control protocol settings of a LAG (ports group).

        Returns: LagProtocolLacp
        """
        return self._get_property("lacp", LagProtocolLacp, self, "lacp")

    @property
    def static(self):
        # type: () -> LagProtocolStatic
        """Factory property that returns an instance of the LagProtocolStatic class

        The container for static link aggregation protocol settings.

        Returns: LagProtocolStatic
        """
        return self._get_property("static", LagProtocolStatic, self, "static")

    @property
    def choice(self):
        # type: () -> Union[Literal["lacp"], Literal["static"]]
        """choice getter

        The type of controlling protocol for the LAG (ports group).

        Returns: Union[Literal["lacp"], Literal["static"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of controlling protocol for the LAG (ports group).

        value: Union[Literal["lacp"], Literal["static"]]
        """
        self._set_property("choice", value)


class LagProtocolLacp(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "actor_system_id": {
            "type": str,
            "format": "mac",
        },
        "actor_system_priority": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "actor_key": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "actor_system_id": "00:00:00:00:00:00",
        "actor_system_priority": 0,
        "actor_key": 0,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        actor_system_id="00:00:00:00:00:00",
        actor_system_priority=0,
        actor_key=0,
    ):
        super(LagProtocolLacp, self).__init__()
        self._parent = parent
        self._set_property("actor_system_id", actor_system_id)
        self._set_property("actor_system_priority", actor_system_priority)
        self._set_property("actor_key", actor_key)

    @property
    def actor_system_id(self):
        # type: () -> str
        """actor_system_id getter

        The actor system id

        Returns: str
        """
        return self._get_property("actor_system_id")

    @actor_system_id.setter
    def actor_system_id(self, value):
        """actor_system_id setter

        The actor system id

        value: str
        """
        self._set_property("actor_system_id", value)

    @property
    def actor_system_priority(self):
        # type: () -> int
        """actor_system_priority getter

        The actor system priority

        Returns: int
        """
        return self._get_property("actor_system_priority")

    @actor_system_priority.setter
    def actor_system_priority(self, value):
        """actor_system_priority setter

        The actor system priority

        value: int
        """
        self._set_property("actor_system_priority", value)

    @property
    def actor_key(self):
        # type: () -> int
        """actor_key getter

        The actor key

        Returns: int
        """
        return self._get_property("actor_key")

    @actor_key.setter
    def actor_key(self, value):
        """actor_key setter

        The actor key

        value: int
        """
        self._set_property("actor_key", value)


class LagProtocolStatic(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "lag_id": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "lag_id": 0,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, lag_id=0):
        super(LagProtocolStatic, self).__init__()
        self._parent = parent
        self._set_property("lag_id", lag_id)

    @property
    def lag_id(self):
        # type: () -> int
        """lag_id getter

        The static lag id

        Returns: int
        """
        return self._get_property("lag_id")

    @lag_id.setter
    def lag_id(self, value):
        """lag_id setter

        The static lag id

        value: int
        """
        self._set_property("lag_id", value)


class LagIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(LagIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Lag]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> LagIter
        return self._iter()

    def __next__(self):
        # type: () -> Lag
        return self._next()

    def next(self):
        # type: () -> Lag
        return self._next()

    def lag(self, min_links=1, name=None):
        # type: (int,str) -> LagIter
        """Factory method that creates an instance of the Lag class

        The container for LAG (ports group) - aggregation of multiple LAG members (ports)

        Returns: LagIter
        """
        item = Lag(parent=self._parent, min_links=min_links, name=name)
        self._add(item)
        return self

    def add(self, min_links=1, name=None):
        # type: (int,str) -> Lag
        """Add method that creates and returns an instance of the Lag class

        The container for LAG (ports group) - aggregation of multiple LAG members (ports)

        Returns: Lag
        """
        item = Lag(parent=self._parent, min_links=min_links, name=name)
        self._add(item)
        return item


class Layer1(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "port_names": {
            "type": list,
            "itemtype": str,
        },
        "speed": {
            "type": str,
            "enum": [
                "speed_10_fd_mbps",
                "speed_10_hd_mbps",
                "speed_100_fd_mbps",
                "speed_100_hd_mbps",
                "speed_1_gbps",
                "speed_10_gbps",
                "speed_25_gbps",
                "speed_40_gbps",
                "speed_50_gbps",
                "speed_100_gbps",
                "speed_200_gbps",
                "speed_400_gbps",
            ],
        },
        "media": {
            "type": str,
            "enum": [
                "copper",
                "fiber",
                "sgmii",
            ],
        },
        "promiscuous": {"type": bool},
        "mtu": {
            "type": int,
            "minimum": 64,
            "maximum": 9000,
        },
        "ieee_media_defaults": {"type": bool},
        "auto_negotiate": {"type": bool},
        "auto_negotiation": {"type": "Layer1AutoNegotiation"},
        "flow_control": {"type": "Layer1FlowControl"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("port_names", "name")  # type: tuple(str)

    _DEFAULTS = {
        "speed": "speed_10_gbps",
        "promiscuous": True,
        "mtu": 1500,
    }  # type: Dict[str, Union(type)]

    SPEED_10_FD_MBPS = "speed_10_fd_mbps"  # type: str
    SPEED_10_HD_MBPS = "speed_10_hd_mbps"  # type: str
    SPEED_100_FD_MBPS = "speed_100_fd_mbps"  # type: str
    SPEED_100_HD_MBPS = "speed_100_hd_mbps"  # type: str
    SPEED_1_GBPS = "speed_1_gbps"  # type: str
    SPEED_10_GBPS = "speed_10_gbps"  # type: str
    SPEED_25_GBPS = "speed_25_gbps"  # type: str
    SPEED_40_GBPS = "speed_40_gbps"  # type: str
    SPEED_50_GBPS = "speed_50_gbps"  # type: str
    SPEED_100_GBPS = "speed_100_gbps"  # type: str
    SPEED_200_GBPS = "speed_200_gbps"  # type: str
    SPEED_400_GBPS = "speed_400_gbps"  # type: str

    COPPER = "copper"  # type: str
    FIBER = "fiber"  # type: str
    SGMII = "sgmii"  # type: str

    def __init__(
        self,
        parent=None,
        port_names=None,
        speed="speed_10_gbps",
        media=None,
        promiscuous=True,
        mtu=1500,
        ieee_media_defaults=None,
        auto_negotiate=None,
        name=None,
    ):
        super(Layer1, self).__init__()
        self._parent = parent
        self._set_property("port_names", port_names)
        self._set_property("speed", speed)
        self._set_property("media", media)
        self._set_property("promiscuous", promiscuous)
        self._set_property("mtu", mtu)
        self._set_property("ieee_media_defaults", ieee_media_defaults)
        self._set_property("auto_negotiate", auto_negotiate)
        self._set_property("name", name)

    @property
    def port_names(self):
        # type: () -> List[str]
        """port_names getter

        A list of unique names of port objects that will share the. choice settings. . . x-constraint:. - /components/schemas/Port/properties/name.

        Returns: List[str]
        """
        return self._get_property("port_names")

    @port_names.setter
    def port_names(self, value):
        """port_names setter

        A list of unique names of port objects that will share the. choice settings. . . x-constraint:. - /components/schemas/Port/properties/name.

        value: List[str]
        """
        self._set_property("port_names", value)

    @property
    def speed(self):
        # type: () -> Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"], Literal["speed_50_gbps"]]
        """speed getter

        Set the speed if supported.

        Returns: Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"], Literal["speed_50_gbps"]]
        """
        return self._get_property("speed")

    @speed.setter
    def speed(self, value):
        """speed setter

        Set the speed if supported.

        value: Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"], Literal["speed_50_gbps"]]
        """
        self._set_property("speed", value)

    @property
    def media(self):
        # type: () -> Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]]
        """media getter

        Set the type of media interface if supported.

        Returns: Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]]
        """
        return self._get_property("media")

    @media.setter
    def media(self, value):
        """media setter

        Set the type of media interface if supported.

        value: Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]]
        """
        self._set_property("media", value)

    @property
    def promiscuous(self):
        # type: () -> bool
        """promiscuous getter

        Enable promiscuous mode if supported.

        Returns: bool
        """
        return self._get_property("promiscuous")

    @promiscuous.setter
    def promiscuous(self, value):
        """promiscuous setter

        Enable promiscuous mode if supported.

        value: bool
        """
        self._set_property("promiscuous", value)

    @property
    def mtu(self):
        # type: () -> int
        """mtu getter

        Set the maximum transmission unit size if supported.

        Returns: int
        """
        return self._get_property("mtu")

    @mtu.setter
    def mtu(self, value):
        """mtu setter

        Set the maximum transmission unit size if supported.

        value: int
        """
        self._set_property("mtu", value)

    @property
    def ieee_media_defaults(self):
        # type: () -> bool
        """ieee_media_defaults getter

        Set to true to override the auto_negotiate, link_training. and rs_fec settings for gigabit ethernet interfaces.

        Returns: bool
        """
        return self._get_property("ieee_media_defaults")

    @ieee_media_defaults.setter
    def ieee_media_defaults(self, value):
        """ieee_media_defaults setter

        Set to true to override the auto_negotiate, link_training. and rs_fec settings for gigabit ethernet interfaces.

        value: bool
        """
        self._set_property("ieee_media_defaults", value)

    @property
    def auto_negotiate(self):
        # type: () -> bool
        """auto_negotiate getter

        Enable/disable auto negotiation.

        Returns: bool
        """
        return self._get_property("auto_negotiate")

    @auto_negotiate.setter
    def auto_negotiate(self, value):
        """auto_negotiate setter

        Enable/disable auto negotiation.

        value: bool
        """
        self._set_property("auto_negotiate", value)

    @property
    def auto_negotiation(self):
        # type: () -> Layer1AutoNegotiation
        """auto_negotiation getter

        Configuration for auto negotiation settingsConfiguration for auto negotiation settings

        Returns: Layer1AutoNegotiation
        """
        return self._get_property("auto_negotiation", Layer1AutoNegotiation)

    @property
    def flow_control(self):
        # type: () -> Layer1FlowControl
        """flow_control getter

        A container for layer1 receive flow control settings.. To enable flow control settings on ports this object must be a valid. object not a null value.A container for layer1 receive flow control settings.. To enable flow control settings on ports this object must be a valid. object not a null value.

        Returns: Layer1FlowControl
        """
        return self._get_property("flow_control", Layer1FlowControl)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class Layer1AutoNegotiation(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "advertise_1000_mbps": {"type": bool},
        "advertise_100_fd_mbps": {"type": bool},
        "advertise_100_hd_mbps": {"type": bool},
        "advertise_10_fd_mbps": {"type": bool},
        "advertise_10_hd_mbps": {"type": bool},
        "link_training": {"type": bool},
        "rs_fec": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "advertise_1000_mbps": True,
        "advertise_100_fd_mbps": True,
        "advertise_100_hd_mbps": True,
        "advertise_10_fd_mbps": True,
        "advertise_10_hd_mbps": True,
        "link_training": False,
        "rs_fec": False,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        advertise_1000_mbps=True,
        advertise_100_fd_mbps=True,
        advertise_100_hd_mbps=True,
        advertise_10_fd_mbps=True,
        advertise_10_hd_mbps=True,
        link_training=False,
        rs_fec=False,
    ):
        super(Layer1AutoNegotiation, self).__init__()
        self._parent = parent
        self._set_property("advertise_1000_mbps", advertise_1000_mbps)
        self._set_property("advertise_100_fd_mbps", advertise_100_fd_mbps)
        self._set_property("advertise_100_hd_mbps", advertise_100_hd_mbps)
        self._set_property("advertise_10_fd_mbps", advertise_10_fd_mbps)
        self._set_property("advertise_10_hd_mbps", advertise_10_hd_mbps)
        self._set_property("link_training", link_training)
        self._set_property("rs_fec", rs_fec)

    @property
    def advertise_1000_mbps(self):
        # type: () -> bool
        """advertise_1000_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property("advertise_1000_mbps")

    @advertise_1000_mbps.setter
    def advertise_1000_mbps(self, value):
        """advertise_1000_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property("advertise_1000_mbps", value)

    @property
    def advertise_100_fd_mbps(self):
        # type: () -> bool
        """advertise_100_fd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property("advertise_100_fd_mbps")

    @advertise_100_fd_mbps.setter
    def advertise_100_fd_mbps(self, value):
        """advertise_100_fd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property("advertise_100_fd_mbps", value)

    @property
    def advertise_100_hd_mbps(self):
        # type: () -> bool
        """advertise_100_hd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property("advertise_100_hd_mbps")

    @advertise_100_hd_mbps.setter
    def advertise_100_hd_mbps(self, value):
        """advertise_100_hd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property("advertise_100_hd_mbps", value)

    @property
    def advertise_10_fd_mbps(self):
        # type: () -> bool
        """advertise_10_fd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property("advertise_10_fd_mbps")

    @advertise_10_fd_mbps.setter
    def advertise_10_fd_mbps(self, value):
        """advertise_10_fd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property("advertise_10_fd_mbps", value)

    @property
    def advertise_10_hd_mbps(self):
        # type: () -> bool
        """advertise_10_hd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property("advertise_10_hd_mbps")

    @advertise_10_hd_mbps.setter
    def advertise_10_hd_mbps(self, value):
        """advertise_10_hd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property("advertise_10_hd_mbps", value)

    @property
    def link_training(self):
        # type: () -> bool
        """link_training getter

        Enable/disable gigabit ethernet link training.

        Returns: bool
        """
        return self._get_property("link_training")

    @link_training.setter
    def link_training(self, value):
        """link_training setter

        Enable/disable gigabit ethernet link training.

        value: bool
        """
        self._set_property("link_training", value)

    @property
    def rs_fec(self):
        # type: () -> bool
        """rs_fec getter

        Enable/disable gigabit ethernet reed solomon forward error correction (RS FEC).

        Returns: bool
        """
        return self._get_property("rs_fec")

    @rs_fec.setter
    def rs_fec(self, value):
        """rs_fec setter

        Enable/disable gigabit ethernet reed solomon forward error correction (RS FEC).

        value: bool
        """
        self._set_property("rs_fec", value)


class Layer1FlowControl(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "directed_address": {
            "type": str,
            "format": "mac",
        },
        "choice": {
            "type": str,
            "enum": [
                "ieee_802_1qbb",
                "ieee_802_3x",
            ],
        },
        "ieee_802_1qbb": {"type": "Layer1Ieee8021qbb"},
        "ieee_802_3x": {"type": "Layer1Ieee8023x"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "directed_address": "01:80:C2:00:00:01",
        "choice": "ieee_802_1qbb",
    }  # type: Dict[str, Union(type)]

    IEEE_802_1QBB = "ieee_802_1qbb"  # type: str
    IEEE_802_3X = "ieee_802_3x"  # type: str

    def __init__(self, parent=None, choice=None, directed_address="01:80:C2:00:00:01"):
        super(Layer1FlowControl, self).__init__()
        self._parent = parent
        self._set_property("directed_address", directed_address)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def ieee_802_1qbb(self):
        # type: () -> Layer1Ieee8021qbb
        """Factory property that returns an instance of the Layer1Ieee8021qbb class

        These settings enhance the existing 802.3x pause priority capabilities . to enable flow control based on 802.1p priorities (classes of service).

        Returns: Layer1Ieee8021qbb
        """
        return self._get_property(
            "ieee_802_1qbb", Layer1Ieee8021qbb, self, "ieee_802_1qbb"
        )

    @property
    def ieee_802_3x(self):
        # type: () -> Layer1Ieee8023x
        """Factory property that returns an instance of the Layer1Ieee8023x class

        A container for ieee 802.3x rx pause settings

        Returns: Layer1Ieee8023x
        """
        return self._get_property("ieee_802_3x", Layer1Ieee8023x, self, "ieee_802_3x")

    @property
    def directed_address(self):
        # type: () -> str
        """directed_address getter

        The 48bit mac address that the layer1 port names will listen on. for a directed pause.

        Returns: str
        """
        return self._get_property("directed_address")

    @directed_address.setter
    def directed_address(self, value):
        """directed_address setter

        The 48bit mac address that the layer1 port names will listen on. for a directed pause.

        value: str
        """
        self._set_property("directed_address", value)

    @property
    def choice(self):
        # type: () -> Union[Literal["ieee_802_1qbb"], Literal["ieee_802_3x"]]
        """choice getter

        The type of priority flow control.

        Returns: Union[Literal["ieee_802_1qbb"], Literal["ieee_802_3x"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of priority flow control.

        value: Union[Literal["ieee_802_1qbb"], Literal["ieee_802_3x"]]
        """
        self._set_property("choice", value)


class Layer1Ieee8021qbb(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "pfc_delay": {"type": int},
        "pfc_class_0": {"type": int},
        "pfc_class_1": {"type": int},
        "pfc_class_2": {"type": int},
        "pfc_class_3": {"type": int},
        "pfc_class_4": {"type": int},
        "pfc_class_5": {"type": int},
        "pfc_class_6": {"type": int},
        "pfc_class_7": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "pfc_delay": 0,
        "pfc_class_0": 0,
        "pfc_class_1": 1,
        "pfc_class_2": 2,
        "pfc_class_3": 3,
        "pfc_class_4": 4,
        "pfc_class_5": 5,
        "pfc_class_6": 6,
        "pfc_class_7": 7,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        pfc_delay=0,
        pfc_class_0=0,
        pfc_class_1=1,
        pfc_class_2=2,
        pfc_class_3=3,
        pfc_class_4=4,
        pfc_class_5=5,
        pfc_class_6=6,
        pfc_class_7=7,
    ):
        super(Layer1Ieee8021qbb, self).__init__()
        self._parent = parent
        self._set_property("pfc_delay", pfc_delay)
        self._set_property("pfc_class_0", pfc_class_0)
        self._set_property("pfc_class_1", pfc_class_1)
        self._set_property("pfc_class_2", pfc_class_2)
        self._set_property("pfc_class_3", pfc_class_3)
        self._set_property("pfc_class_4", pfc_class_4)
        self._set_property("pfc_class_5", pfc_class_5)
        self._set_property("pfc_class_6", pfc_class_6)
        self._set_property("pfc_class_7", pfc_class_7)

    @property
    def pfc_delay(self):
        # type: () -> int
        """pfc_delay getter

        The upper limit on the transmit time of a queue after receiving a . message to pause a specified priority.. A value of 0 or null indicates that pfc delay will not be enabled.

        Returns: int
        """
        return self._get_property("pfc_delay")

    @pfc_delay.setter
    def pfc_delay(self, value):
        """pfc_delay setter

        The upper limit on the transmit time of a queue after receiving a . message to pause a specified priority.. A value of 0 or null indicates that pfc delay will not be enabled.

        value: int
        """
        self._set_property("pfc_delay", value)

    @property
    def pfc_class_0(self):
        # type: () -> int
        """pfc_class_0 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_0")

    @pfc_class_0.setter
    def pfc_class_0(self, value):
        """pfc_class_0 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_0", value)

    @property
    def pfc_class_1(self):
        # type: () -> int
        """pfc_class_1 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_1")

    @pfc_class_1.setter
    def pfc_class_1(self, value):
        """pfc_class_1 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_1", value)

    @property
    def pfc_class_2(self):
        # type: () -> int
        """pfc_class_2 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_2")

    @pfc_class_2.setter
    def pfc_class_2(self, value):
        """pfc_class_2 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_2", value)

    @property
    def pfc_class_3(self):
        # type: () -> int
        """pfc_class_3 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_3")

    @pfc_class_3.setter
    def pfc_class_3(self, value):
        """pfc_class_3 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_3", value)

    @property
    def pfc_class_4(self):
        # type: () -> int
        """pfc_class_4 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_4")

    @pfc_class_4.setter
    def pfc_class_4(self, value):
        """pfc_class_4 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_4", value)

    @property
    def pfc_class_5(self):
        # type: () -> int
        """pfc_class_5 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_5")

    @pfc_class_5.setter
    def pfc_class_5(self, value):
        """pfc_class_5 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_5", value)

    @property
    def pfc_class_6(self):
        # type: () -> int
        """pfc_class_6 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_6")

    @pfc_class_6.setter
    def pfc_class_6(self, value):
        """pfc_class_6 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_6", value)

    @property
    def pfc_class_7(self):
        # type: () -> int
        """pfc_class_7 getter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_7")

    @pfc_class_7.setter
    def pfc_class_7(self, value):
        """pfc_class_7 setter

        The valid values are null, 0 - 7.. A null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_7", value)


class Layer1Ieee8023x(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {}  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(Layer1Ieee8023x, self).__init__()
        self._parent = parent


class Layer1Iter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(Layer1Iter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Layer1]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> Layer1Iter
        return self._iter()

    def __next__(self):
        # type: () -> Layer1
        return self._next()

    def next(self):
        # type: () -> Layer1
        return self._next()

    def layer1(
        self,
        port_names=None,
        speed="speed_10_gbps",
        media=None,
        promiscuous=True,
        mtu=1500,
        ieee_media_defaults=None,
        auto_negotiate=None,
        name=None,
    ):
        # type: (List[str],Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"], Literal["speed_50_gbps"]],Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]],bool,int,bool,bool,str) -> Layer1Iter
        """Factory method that creates an instance of the Layer1 class

        A container for layer1 settings.

        Returns: Layer1Iter
        """
        item = Layer1(
            parent=self._parent,
            port_names=port_names,
            speed=speed,
            media=media,
            promiscuous=promiscuous,
            mtu=mtu,
            ieee_media_defaults=ieee_media_defaults,
            auto_negotiate=auto_negotiate,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        port_names=None,
        speed="speed_10_gbps",
        media=None,
        promiscuous=True,
        mtu=1500,
        ieee_media_defaults=None,
        auto_negotiate=None,
        name=None,
    ):
        # type: (List[str],Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"], Literal["speed_50_gbps"]],Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]],bool,int,bool,bool,str) -> Layer1
        """Add method that creates and returns an instance of the Layer1 class

        A container for layer1 settings.

        Returns: Layer1
        """
        item = Layer1(
            parent=self._parent,
            port_names=port_names,
            speed=speed,
            media=media,
            promiscuous=promiscuous,
            mtu=mtu,
            ieee_media_defaults=ieee_media_defaults,
            auto_negotiate=auto_negotiate,
            name=name,
        )
        self._add(item)
        return item


class Capture(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "port_names": {
            "type": list,
            "itemtype": str,
        },
        "filters": {"type": "CaptureFilterIter"},
        "overwrite": {"type": bool},
        "packet_size": {"type": int},
        "format": {
            "type": str,
            "enum": [
                "pcap",
                "pcapng",
            ],
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("port_names", "name")  # type: tuple(str)

    _DEFAULTS = {
        "overwrite": True,
        "format": "pcap",
    }  # type: Dict[str, Union(type)]

    PCAP = "pcap"  # type: str
    PCAPNG = "pcapng"  # type: str

    def __init__(
        self,
        parent=None,
        port_names=None,
        overwrite=True,
        packet_size=None,
        format="pcap",
        name=None,
    ):
        super(Capture, self).__init__()
        self._parent = parent
        self._set_property("port_names", port_names)
        self._set_property("overwrite", overwrite)
        self._set_property("packet_size", packet_size)
        self._set_property("format", format)
        self._set_property("name", name)

    @property
    def port_names(self):
        # type: () -> List[str]
        """port_names getter

        The unique names of ports that the capture settings will apply to. Port_names cannot be duplicated between capture objects.. . x-constraint:. - /components/schemas/Port/properties/name.

        Returns: List[str]
        """
        return self._get_property("port_names")

    @port_names.setter
    def port_names(self, value):
        """port_names setter

        The unique names of ports that the capture settings will apply to. Port_names cannot be duplicated between capture objects.. . x-constraint:. - /components/schemas/Port/properties/name.

        value: List[str]
        """
        self._set_property("port_names", value)

    @property
    def filters(self):
        # type: () -> CaptureFilterIter
        """filters getter

        A list of filters to apply to the capturing ports. If no filters are specified then all packets will be captured. A capture can have multiple filters. The number of filters supported is determined by the implementation which can be retrieved using the capabilities API.. When multiple filters are specified the capture implementation must && (and) all the filters.

        Returns: CaptureFilterIter
        """
        return self._get_property(
            "filters", CaptureFilterIter, self._parent, self._choice
        )

    @property
    def overwrite(self):
        # type: () -> bool
        """overwrite getter

        Overwrite the capture buffer.

        Returns: bool
        """
        return self._get_property("overwrite")

    @overwrite.setter
    def overwrite(self, value):
        """overwrite setter

        Overwrite the capture buffer.

        value: bool
        """
        self._set_property("overwrite", value)

    @property
    def packet_size(self):
        # type: () -> int
        """packet_size getter

        The maximum size of each captured packet. If no value is specified or it is null then the entire packet will be captured.

        Returns: int
        """
        return self._get_property("packet_size")

    @packet_size.setter
    def packet_size(self, value):
        """packet_size setter

        The maximum size of each captured packet. If no value is specified or it is null then the entire packet will be captured.

        value: int
        """
        self._set_property("packet_size", value)

    @property
    def format(self):
        # type: () -> Union[Literal["pcap"], Literal["pcapng"]]
        """format getter

        The format of the capture file.

        Returns: Union[Literal["pcap"], Literal["pcapng"]]
        """
        return self._get_property("format")

    @format.setter
    def format(self, value):
        """format setter

        The format of the capture file.

        value: Union[Literal["pcap"], Literal["pcapng"]]
        """
        self._set_property("format", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class CaptureFilter(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "custom",
                "ethernet",
                "vlan",
                "ipv4",
                "ipv6",
            ],
        },
        "custom": {"type": "CaptureCustom"},
        "ethernet": {"type": "CaptureEthernet"},
        "vlan": {"type": "CaptureVlan"},
        "ipv4": {"type": "CaptureIpv4"},
        "ipv6": {"type": "CaptureIpv6"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "custom",
    }  # type: Dict[str, Union(type)]

    CUSTOM = "custom"  # type: str
    ETHERNET = "ethernet"  # type: str
    VLAN = "vlan"  # type: str
    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    def __init__(self, parent=None, choice=None):
        super(CaptureFilter, self).__init__()
        self._parent = parent
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def custom(self):
        # type: () -> CaptureCustom
        """Factory property that returns an instance of the CaptureCustom class

        TBD

        Returns: CaptureCustom
        """
        return self._get_property("custom", CaptureCustom, self, "custom")

    @property
    def ethernet(self):
        # type: () -> CaptureEthernet
        """Factory property that returns an instance of the CaptureEthernet class

        TBD

        Returns: CaptureEthernet
        """
        return self._get_property("ethernet", CaptureEthernet, self, "ethernet")

    @property
    def vlan(self):
        # type: () -> CaptureVlan
        """Factory property that returns an instance of the CaptureVlan class

        TBD

        Returns: CaptureVlan
        """
        return self._get_property("vlan", CaptureVlan, self, "vlan")

    @property
    def ipv4(self):
        # type: () -> CaptureIpv4
        """Factory property that returns an instance of the CaptureIpv4 class

        TBD

        Returns: CaptureIpv4
        """
        return self._get_property("ipv4", CaptureIpv4, self, "ipv4")

    @property
    def ipv6(self):
        # type: () -> CaptureIpv6
        """Factory property that returns an instance of the CaptureIpv6 class

        TBD

        Returns: CaptureIpv6
        """
        return self._get_property("ipv6", CaptureIpv6, self, "ipv6")

    @property
    def choice(self):
        # type: () -> Union[Literal["custom"], Literal["ethernet"], Literal["ipv4"], Literal["ipv6"], Literal["vlan"]]
        """choice getter

        The type of capture filter.

        Returns: Union[Literal["custom"], Literal["ethernet"], Literal["ipv4"], Literal["ipv6"], Literal["vlan"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of capture filter.

        value: Union[Literal["custom"], Literal["ethernet"], Literal["ipv4"], Literal["ipv6"], Literal["vlan"]]
        """
        self._set_property("choice", value)


class CaptureCustom(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "offset": {"type": int},
        "bit_length": {"type": int},
        "value": {
            "type": str,
            "format": "hex",
        },
        "mask": {
            "type": str,
            "format": "hex",
        },
        "negate": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "offset": 0,
        "bit_length": 8,
        "value": "00",
        "mask": "00",
        "negate": False,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, offset=0, bit_length=8, value="00", mask="00", negate=False
    ):
        super(CaptureCustom, self).__init__()
        self._parent = parent
        self._set_property("offset", offset)
        self._set_property("bit_length", bit_length)
        self._set_property("value", value)
        self._set_property("mask", mask)
        self._set_property("negate", negate)

    @property
    def offset(self):
        # type: () -> int
        """offset getter

        The bit offset of field to filter on

        Returns: int
        """
        return self._get_property("offset")

    @offset.setter
    def offset(self, value):
        """offset setter

        The bit offset of field to filter on

        value: int
        """
        self._set_property("offset", value)

    @property
    def bit_length(self):
        # type: () -> int
        """bit_length getter

        The bit length of field to filter on

        Returns: int
        """
        return self._get_property("bit_length")

    @bit_length.setter
    def bit_length(self, value):
        """bit_length setter

        The bit length of field to filter on

        value: int
        """
        self._set_property("bit_length", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value)

    @property
    def mask(self):
        # type: () -> str
        """mask getter

        TBD

        Returns: str
        """
        return self._get_property("mask")

    @mask.setter
    def mask(self, value):
        """mask setter

        TBD

        value: str
        """
        self._set_property("mask", value)

    @property
    def negate(self):
        # type: () -> bool
        """negate getter

        TBD

        Returns: bool
        """
        return self._get_property("negate")

    @negate.setter
    def negate(self, value):
        """negate setter

        TBD

        value: bool
        """
        self._set_property("negate", value)


class CaptureEthernet(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "src": {"type": "CaptureField"},
        "dst": {"type": "CaptureField"},
        "ether_type": {"type": "CaptureField"},
        "pfc_queue": {"type": "CaptureField"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(CaptureEthernet, self).__init__()
        self._parent = parent

    @property
    def src(self):
        # type: () -> CaptureField
        """src getter



        Returns: CaptureField
        """
        return self._get_property("src", CaptureField)

    @property
    def dst(self):
        # type: () -> CaptureField
        """dst getter



        Returns: CaptureField
        """
        return self._get_property("dst", CaptureField)

    @property
    def ether_type(self):
        # type: () -> CaptureField
        """ether_type getter



        Returns: CaptureField
        """
        return self._get_property("ether_type", CaptureField)

    @property
    def pfc_queue(self):
        # type: () -> CaptureField
        """pfc_queue getter



        Returns: CaptureField
        """
        return self._get_property("pfc_queue", CaptureField)


class CaptureField(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": str,
            "format": "hex",
        },
        "mask": {
            "type": str,
            "format": "hex",
        },
        "negate": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "value": "00",
        "mask": "00",
        "negate": False,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value="00", mask="00", negate=False):
        super(CaptureField, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("mask", mask)
        self._set_property("negate", negate)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value)

    @property
    def mask(self):
        # type: () -> str
        """mask getter

        TBD

        Returns: str
        """
        return self._get_property("mask")

    @mask.setter
    def mask(self, value):
        """mask setter

        TBD

        value: str
        """
        self._set_property("mask", value)

    @property
    def negate(self):
        # type: () -> bool
        """negate getter

        TBD

        Returns: bool
        """
        return self._get_property("negate")

    @negate.setter
    def negate(self, value):
        """negate setter

        TBD

        value: bool
        """
        self._set_property("negate", value)


class CaptureVlan(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "priority": {"type": "CaptureField"},
        "cfi": {"type": "CaptureField"},
        "id": {"type": "CaptureField"},
        "protocol": {"type": "CaptureField"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(CaptureVlan, self).__init__()
        self._parent = parent

    @property
    def priority(self):
        # type: () -> CaptureField
        """priority getter



        Returns: CaptureField
        """
        return self._get_property("priority", CaptureField)

    @property
    def cfi(self):
        # type: () -> CaptureField
        """cfi getter



        Returns: CaptureField
        """
        return self._get_property("cfi", CaptureField)

    @property
    def id(self):
        # type: () -> CaptureField
        """id getter



        Returns: CaptureField
        """
        return self._get_property("id", CaptureField)

    @property
    def protocol(self):
        # type: () -> CaptureField
        """protocol getter



        Returns: CaptureField
        """
        return self._get_property("protocol", CaptureField)


class CaptureIpv4(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "version": {"type": "CaptureField"},
        "header_length": {"type": "CaptureField"},
        "priority": {"type": "CaptureField"},
        "total_length": {"type": "CaptureField"},
        "identification": {"type": "CaptureField"},
        "reserved": {"type": "CaptureField"},
        "dont_fragment": {"type": "CaptureField"},
        "more_fragments": {"type": "CaptureField"},
        "fragment_offset": {"type": "CaptureField"},
        "time_to_live": {"type": "CaptureField"},
        "protocol": {"type": "CaptureField"},
        "header_checksum": {"type": "CaptureField"},
        "src": {"type": "CaptureField"},
        "dst": {"type": "CaptureField"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(CaptureIpv4, self).__init__()
        self._parent = parent

    @property
    def version(self):
        # type: () -> CaptureField
        """version getter



        Returns: CaptureField
        """
        return self._get_property("version", CaptureField)

    @property
    def header_length(self):
        # type: () -> CaptureField
        """header_length getter



        Returns: CaptureField
        """
        return self._get_property("header_length", CaptureField)

    @property
    def priority(self):
        # type: () -> CaptureField
        """priority getter



        Returns: CaptureField
        """
        return self._get_property("priority", CaptureField)

    @property
    def total_length(self):
        # type: () -> CaptureField
        """total_length getter



        Returns: CaptureField
        """
        return self._get_property("total_length", CaptureField)

    @property
    def identification(self):
        # type: () -> CaptureField
        """identification getter



        Returns: CaptureField
        """
        return self._get_property("identification", CaptureField)

    @property
    def reserved(self):
        # type: () -> CaptureField
        """reserved getter



        Returns: CaptureField
        """
        return self._get_property("reserved", CaptureField)

    @property
    def dont_fragment(self):
        # type: () -> CaptureField
        """dont_fragment getter



        Returns: CaptureField
        """
        return self._get_property("dont_fragment", CaptureField)

    @property
    def more_fragments(self):
        # type: () -> CaptureField
        """more_fragments getter



        Returns: CaptureField
        """
        return self._get_property("more_fragments", CaptureField)

    @property
    def fragment_offset(self):
        # type: () -> CaptureField
        """fragment_offset getter



        Returns: CaptureField
        """
        return self._get_property("fragment_offset", CaptureField)

    @property
    def time_to_live(self):
        # type: () -> CaptureField
        """time_to_live getter



        Returns: CaptureField
        """
        return self._get_property("time_to_live", CaptureField)

    @property
    def protocol(self):
        # type: () -> CaptureField
        """protocol getter



        Returns: CaptureField
        """
        return self._get_property("protocol", CaptureField)

    @property
    def header_checksum(self):
        # type: () -> CaptureField
        """header_checksum getter



        Returns: CaptureField
        """
        return self._get_property("header_checksum", CaptureField)

    @property
    def src(self):
        # type: () -> CaptureField
        """src getter



        Returns: CaptureField
        """
        return self._get_property("src", CaptureField)

    @property
    def dst(self):
        # type: () -> CaptureField
        """dst getter



        Returns: CaptureField
        """
        return self._get_property("dst", CaptureField)


class CaptureIpv6(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "version": {"type": "CaptureField"},
        "traffic_class": {"type": "CaptureField"},
        "flow_label": {"type": "CaptureField"},
        "payload_length": {"type": "CaptureField"},
        "next_header": {"type": "CaptureField"},
        "hop_limit": {"type": "CaptureField"},
        "src": {"type": "CaptureField"},
        "dst": {"type": "CaptureField"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(CaptureIpv6, self).__init__()
        self._parent = parent

    @property
    def version(self):
        # type: () -> CaptureField
        """version getter



        Returns: CaptureField
        """
        return self._get_property("version", CaptureField)

    @property
    def traffic_class(self):
        # type: () -> CaptureField
        """traffic_class getter



        Returns: CaptureField
        """
        return self._get_property("traffic_class", CaptureField)

    @property
    def flow_label(self):
        # type: () -> CaptureField
        """flow_label getter



        Returns: CaptureField
        """
        return self._get_property("flow_label", CaptureField)

    @property
    def payload_length(self):
        # type: () -> CaptureField
        """payload_length getter



        Returns: CaptureField
        """
        return self._get_property("payload_length", CaptureField)

    @property
    def next_header(self):
        # type: () -> CaptureField
        """next_header getter



        Returns: CaptureField
        """
        return self._get_property("next_header", CaptureField)

    @property
    def hop_limit(self):
        # type: () -> CaptureField
        """hop_limit getter



        Returns: CaptureField
        """
        return self._get_property("hop_limit", CaptureField)

    @property
    def src(self):
        # type: () -> CaptureField
        """src getter



        Returns: CaptureField
        """
        return self._get_property("src", CaptureField)

    @property
    def dst(self):
        # type: () -> CaptureField
        """dst getter



        Returns: CaptureField
        """
        return self._get_property("dst", CaptureField)


class CaptureFilterIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = True

    def __init__(self, parent=None, choice=None):
        super(CaptureFilterIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[CaptureCustom, CaptureEthernet, CaptureFilter, CaptureIpv4, CaptureIpv6, CaptureVlan]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> CaptureFilterIter
        return self._iter()

    def __next__(self):
        # type: () -> CaptureFilter
        return self._next()

    def next(self):
        # type: () -> CaptureFilter
        return self._next()

    def filter(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureFilter class

        Configuration for capture filters

        Returns: CaptureFilterIter
        """
        item = CaptureFilter(parent=self._parent, choice=self._choice)
        self._add(item)
        return self

    def add(self):
        # type: () -> CaptureFilter
        """Add method that creates and returns an instance of the CaptureFilter class

        Configuration for capture filters

        Returns: CaptureFilter
        """
        item = CaptureFilter(parent=self._parent, choice=self._choice)
        self._add(item)
        return item

    def custom(self, offset=0, bit_length=8, value="00", mask="00", negate=False):
        # type: (int,int,str,str,bool) -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureCustom class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.custom
        item.choice = "custom"
        self._add(item)
        return self

    def ethernet(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureEthernet class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.ethernet
        item.choice = "ethernet"
        self._add(item)
        return self

    def vlan(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureVlan class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.vlan
        item.choice = "vlan"
        self._add(item)
        return self

    def ipv4(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureIpv4 class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.ipv4
        item.choice = "ipv4"
        self._add(item)
        return self

    def ipv6(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureIpv6 class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.ipv6
        item.choice = "ipv6"
        self._add(item)
        return self


class CaptureIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(CaptureIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Capture]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> CaptureIter
        return self._iter()

    def __next__(self):
        # type: () -> Capture
        return self._next()

    def next(self):
        # type: () -> Capture
        return self._next()

    def capture(
        self,
        port_names=None,
        overwrite=True,
        packet_size=None,
        format="pcap",
        name=None,
    ):
        # type: (List[str],bool,int,Union[Literal["pcap"], Literal["pcapng"]],str) -> CaptureIter
        """Factory method that creates an instance of the Capture class

        Status: under-review. Configuration for capture settings.

        Returns: CaptureIter
        """
        item = Capture(
            parent=self._parent,
            port_names=port_names,
            overwrite=overwrite,
            packet_size=packet_size,
            format=format,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        port_names=None,
        overwrite=True,
        packet_size=None,
        format="pcap",
        name=None,
    ):
        # type: (List[str],bool,int,Union[Literal["pcap"], Literal["pcapng"]],str) -> Capture
        """Add method that creates and returns an instance of the Capture class

        Status: under-review. Configuration for capture settings.

        Returns: Capture
        """
        item = Capture(
            parent=self._parent,
            port_names=port_names,
            overwrite=overwrite,
            packet_size=packet_size,
            format=format,
            name=name,
        )
        self._add(item)
        return item


class Device(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ethernets": {"type": "DeviceEthernetIter"},
        "ipv4_loopbacks": {"type": "DeviceIpv4LoopbackIter"},
        "ipv6_loopbacks": {"type": "DeviceIpv6LoopbackIter"},
        "isis": {"type": "DeviceIsisRouter"},
        "bgp": {"type": "DeviceBgpRouter"},
        "vxlan": {"type": "DeviceVxlan"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, name=None):
        super(Device, self).__init__()
        self._parent = parent
        self._set_property("name", name)

    @property
    def ethernets(self):
        # type: () -> DeviceEthernetIter
        """ethernets getter

        Ethernet configuration for one or more emulated network interfaces.

        Returns: DeviceEthernetIter
        """
        return self._get_property(
            "ethernets", DeviceEthernetIter, self._parent, self._choice
        )

    @property
    def ipv4_loopbacks(self):
        # type: () -> DeviceIpv4LoopbackIter
        """ipv4_loopbacks getter

        IPv4 Loopback interface that can be attached to an Ethernet in the same device or to an Ethernet in another device.

        Returns: DeviceIpv4LoopbackIter
        """
        return self._get_property(
            "ipv4_loopbacks", DeviceIpv4LoopbackIter, self._parent, self._choice
        )

    @property
    def ipv6_loopbacks(self):
        # type: () -> DeviceIpv6LoopbackIter
        """ipv6_loopbacks getter

        IPv6 Loopback interface that can be attached to an Ethernet in the same device or to an Ethernet in another device.

        Returns: DeviceIpv6LoopbackIter
        """
        return self._get_property(
            "ipv6_loopbacks", DeviceIpv6LoopbackIter, self._parent, self._choice
        )

    @property
    def isis(self):
        # type: () -> DeviceIsisRouter
        """isis getter

        Status: under-review. A container of properties for an ISIS router and its interfaces.Status: under-review. A container of properties for an ISIS router and its interfaces.The properties of an IS-IS router and its children, such as IS-IS interfaces and route ranges.

        Returns: DeviceIsisRouter
        """
        return self._get_property("isis", DeviceIsisRouter)

    @property
    def bgp(self):
        # type: () -> DeviceBgpRouter
        """bgp getter

        Configuration for one or more IPv4 or IPv6 BGP peers.Configuration for one or more IPv4 or IPv6 BGP peers.The properties of BGP router and its children, such as BGPv4, BGPv6 peers and their route ranges.

        Returns: DeviceBgpRouter
        """
        return self._get_property("bgp", DeviceBgpRouter)

    @property
    def vxlan(self):
        # type: () -> DeviceVxlan
        """vxlan getter

        Configuration of VXLAN tunnel interfaces RFC Ref: https://datatracker.ietf.org/doc/html/rfc7348

        Returns: DeviceVxlan
        """
        return self._get_property("vxlan", DeviceVxlan)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class DeviceEthernet(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "port_name": {"type": str},
        "connection": {"type": "EthernetConnection"},
        "ipv4_addresses": {"type": "DeviceIpv4Iter"},
        "ipv6_addresses": {"type": "DeviceIpv6Iter"},
        "mac": {
            "type": str,
            "format": "mac",
        },
        "mtu": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "vlans": {"type": "DeviceVlanIter"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("mac", "name")  # type: tuple(str)

    _DEFAULTS = {
        "mtu": 1500,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, port_name=None, mac=None, mtu=1500, name=None):
        super(DeviceEthernet, self).__init__()
        self._parent = parent
        self._set_property("port_name", port_name)
        self._set_property("mac", mac)
        self._set_property("mtu", mtu)
        self._set_property("name", name)

    @property
    def port_name(self):
        # type: () -> str
        """port_name getter

        Status: deprecated. The unique name of a Port or a LAG that will emulate this interface.. port_name is deprecated and will be removed in future release.port_name and connection can't be used together, use either port_name or connection.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name.

        Returns: str
        """
        return self._get_property("port_name")

    @port_name.setter
    def port_name(self, value):
        """port_name setter

        Status: deprecated. The unique name of a Port or a LAG that will emulate this interface.. port_name is deprecated and will be removed in future release.port_name and connection can't be used together, use either port_name or connection.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name.

        value: str
        """
        self._set_property("port_name", value)

    @property
    def connection(self):
        # type: () -> EthernetConnection
        """connection getter

        Ethernet interface connection to a port, LAG or VXLAN tunnel.Ethernet interface connection to a port, LAG or VXLAN tunnel.Device connection to physical, LAG or another device.

        Returns: EthernetConnection
        """
        return self._get_property("connection", EthernetConnection)

    @property
    def ipv4_addresses(self):
        # type: () -> DeviceIpv4Iter
        """ipv4_addresses getter

        List of IPv4 addresses and their gateways.

        Returns: DeviceIpv4Iter
        """
        return self._get_property(
            "ipv4_addresses", DeviceIpv4Iter, self._parent, self._choice
        )

    @property
    def ipv6_addresses(self):
        # type: () -> DeviceIpv6Iter
        """ipv6_addresses getter

        List of global IPv6 addresses and their gateways.. The Link Local IPv6 address will be automatically generated.

        Returns: DeviceIpv6Iter
        """
        return self._get_property(
            "ipv6_addresses", DeviceIpv6Iter, self._parent, self._choice
        )

    @property
    def mac(self):
        # type: () -> str
        """mac getter

        Media Access Control address.

        Returns: str
        """
        return self._get_property("mac")

    @mac.setter
    def mac(self, value):
        """mac setter

        Media Access Control address.

        value: str
        """
        self._set_property("mac", value)

    @property
    def mtu(self):
        # type: () -> int
        """mtu getter

        Maximum Transmission Unit.

        Returns: int
        """
        return self._get_property("mtu")

    @mtu.setter
    def mtu(self, value):
        """mtu setter

        Maximum Transmission Unit.

        value: int
        """
        self._set_property("mtu", value)

    @property
    def vlans(self):
        # type: () -> DeviceVlanIter
        """vlans getter

        List of VLANs

        Returns: DeviceVlanIter
        """
        return self._get_property("vlans", DeviceVlanIter, self._parent, self._choice)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class EthernetConnection(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "port_name",
                "lag_name",
                "vxlan_name",
            ],
        },
        "port_name": {"type": str},
        "lag_name": {"type": str},
        "vxlan_name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    PORT_NAME = "port_name"  # type: str
    LAG_NAME = "lag_name"  # type: str
    VXLAN_NAME = "vxlan_name"  # type: str

    def __init__(
        self, parent=None, choice=None, port_name=None, lag_name=None, vxlan_name=None
    ):
        super(EthernetConnection, self).__init__()
        self._parent = parent
        self._set_property("port_name", port_name)
        self._set_property("lag_name", lag_name)
        self._set_property("vxlan_name", vxlan_name)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def choice(self):
        # type: () -> Union[Literal["lag_name"], Literal["port_name"], Literal["vxlan_name"]]
        """choice getter

        port_name, lag_name or vxlan_name

        Returns: Union[Literal["lag_name"], Literal["port_name"], Literal["vxlan_name"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        port_name, lag_name or vxlan_name

        value: Union[Literal["lag_name"], Literal["port_name"], Literal["vxlan_name"]]
        """
        self._set_property("choice", value)

    @property
    def port_name(self):
        # type: () -> str
        """port_name getter

        Name of the port that the Ethernet interface is configured on.. . x-constraint:. - /components/schemas/Port/properties/name.

        Returns: str
        """
        return self._get_property("port_name")

    @port_name.setter
    def port_name(self, value):
        """port_name setter

        Name of the port that the Ethernet interface is configured on.. . x-constraint:. - /components/schemas/Port/properties/name.

        value: str
        """
        self._set_property("port_name", value, "port_name")

    @property
    def lag_name(self):
        # type: () -> str
        """lag_name getter

        Name of the LAG that the Ethernet interface is configured on.. . x-constraint:. - /components/schemas/Lag/properties/name.

        Returns: str
        """
        return self._get_property("lag_name")

    @lag_name.setter
    def lag_name(self, value):
        """lag_name setter

        Name of the LAG that the Ethernet interface is configured on.. . x-constraint:. - /components/schemas/Lag/properties/name.

        value: str
        """
        self._set_property("lag_name", value, "lag_name")

    @property
    def vxlan_name(self):
        # type: () -> str
        """vxlan_name getter

        Name of the VXLAN instance (or VXLAN tunnel) that this Ethernet interface is connected to.. . x-constraint:. - #/components/schemas/VXlan.V4Tunnel/properties/name. - #/components/schemas/VXlan.V6Tunnel/properties/name.

        Returns: str
        """
        return self._get_property("vxlan_name")

    @vxlan_name.setter
    def vxlan_name(self, value):
        """vxlan_name setter

        Name of the VXLAN instance (or VXLAN tunnel) that this Ethernet interface is connected to.. . x-constraint:. - #/components/schemas/VXlan.V4Tunnel/properties/name. - #/components/schemas/VXlan.V6Tunnel/properties/name.

        value: str
        """
        self._set_property("vxlan_name", value, "vxlan_name")


class DeviceIpv4(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "gateway": {
            "type": str,
            "format": "ipv4",
        },
        "address": {
            "type": str,
            "format": "ipv4",
        },
        "prefix": {
            "type": int,
            "minimum": 1,
            "maximum": 32,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("gateway", "address", "name")  # type: tuple(str)

    _DEFAULTS = {
        "prefix": 24,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, gateway=None, address=None, prefix=24, name=None):
        super(DeviceIpv4, self).__init__()
        self._parent = parent
        self._set_property("gateway", gateway)
        self._set_property("address", address)
        self._set_property("prefix", prefix)
        self._set_property("name", name)

    @property
    def gateway(self):
        # type: () -> str
        """gateway getter

        The IPv4 address of the gateway

        Returns: str
        """
        return self._get_property("gateway")

    @gateway.setter
    def gateway(self, value):
        """gateway setter

        The IPv4 address of the gateway

        value: str
        """
        self._set_property("gateway", value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The IPv4 address

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The IPv4 address

        value: str
        """
        self._set_property("address", value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The prefix of the IPv4 address.

        Returns: int
        """
        return self._get_property("prefix")

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The prefix of the IPv4 address.

        value: int
        """
        self._set_property("prefix", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class DeviceIpv4Iter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceIpv4Iter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceIpv4]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceIpv4Iter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceIpv4
        return self._next()

    def next(self):
        # type: () -> DeviceIpv4
        return self._next()

    def ipv4(self, gateway=None, address=None, prefix=24, name=None):
        # type: (str,str,int,str) -> DeviceIpv4Iter
        """Factory method that creates an instance of the DeviceIpv4 class

        An IPv4 interface with gateway. A base IPv4 interface.

        Returns: DeviceIpv4Iter
        """
        item = DeviceIpv4(
            parent=self._parent,
            gateway=gateway,
            address=address,
            prefix=prefix,
            name=name,
        )
        self._add(item)
        return self

    def add(self, gateway=None, address=None, prefix=24, name=None):
        # type: (str,str,int,str) -> DeviceIpv4
        """Add method that creates and returns an instance of the DeviceIpv4 class

        An IPv4 interface with gateway. A base IPv4 interface.

        Returns: DeviceIpv4
        """
        item = DeviceIpv4(
            parent=self._parent,
            gateway=gateway,
            address=address,
            prefix=prefix,
            name=name,
        )
        self._add(item)
        return item


class DeviceIpv6(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "gateway": {
            "type": str,
            "format": "ipv6",
        },
        "address": {
            "type": str,
            "format": "ipv6",
        },
        "prefix": {
            "type": int,
            "minimum": 1,
            "maximum": 128,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("gateway", "address", "name")  # type: tuple(str)

    _DEFAULTS = {
        "prefix": 64,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, gateway=None, address=None, prefix=64, name=None):
        super(DeviceIpv6, self).__init__()
        self._parent = parent
        self._set_property("gateway", gateway)
        self._set_property("address", address)
        self._set_property("prefix", prefix)
        self._set_property("name", name)

    @property
    def gateway(self):
        # type: () -> str
        """gateway getter

        The IPv6 gateway address.

        Returns: str
        """
        return self._get_property("gateway")

    @gateway.setter
    def gateway(self, value):
        """gateway setter

        The IPv6 gateway address.

        value: str
        """
        self._set_property("gateway", value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The IPv6 address.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The IPv6 address.

        value: str
        """
        self._set_property("address", value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The network prefix.

        Returns: int
        """
        return self._get_property("prefix")

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The network prefix.

        value: int
        """
        self._set_property("prefix", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class DeviceIpv6Iter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceIpv6Iter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceIpv6]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceIpv6Iter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceIpv6
        return self._next()

    def next(self):
        # type: () -> DeviceIpv6
        return self._next()

    def ipv6(self, gateway=None, address=None, prefix=64, name=None):
        # type: (str,str,int,str) -> DeviceIpv6Iter
        """Factory method that creates an instance of the DeviceIpv6 class

        Status: under-review. An IPv6 interface with gateway.. A base IPv6 interface.

        Returns: DeviceIpv6Iter
        """
        item = DeviceIpv6(
            parent=self._parent,
            gateway=gateway,
            address=address,
            prefix=prefix,
            name=name,
        )
        self._add(item)
        return self

    def add(self, gateway=None, address=None, prefix=64, name=None):
        # type: (str,str,int,str) -> DeviceIpv6
        """Add method that creates and returns an instance of the DeviceIpv6 class

        Status: under-review. An IPv6 interface with gateway.. A base IPv6 interface.

        Returns: DeviceIpv6
        """
        item = DeviceIpv6(
            parent=self._parent,
            gateway=gateway,
            address=address,
            prefix=prefix,
            name=name,
        )
        self._add(item)
        return item


class DeviceEthernetIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceEthernetIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceEthernet]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceEthernetIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceEthernet
        return self._next()

    def next(self):
        # type: () -> DeviceEthernet
        return self._next()

    def ethernet(self, port_name=None, mac=None, mtu=1500, name=None):
        # type: (str,str,int,str) -> DeviceEthernetIter
        """Factory method that creates an instance of the DeviceEthernet class

        An Ethernet interface with IPv4 and IPv6 addresses.. Base Ethernet interface.

        Returns: DeviceEthernetIter
        """
        item = DeviceEthernet(
            parent=self._parent, port_name=port_name, mac=mac, mtu=mtu, name=name
        )
        self._add(item)
        return self

    def add(self, port_name=None, mac=None, mtu=1500, name=None):
        # type: (str,str,int,str) -> DeviceEthernet
        """Add method that creates and returns an instance of the DeviceEthernet class

        An Ethernet interface with IPv4 and IPv6 addresses.. Base Ethernet interface.

        Returns: DeviceEthernet
        """
        item = DeviceEthernet(
            parent=self._parent, port_name=port_name, mac=mac, mtu=mtu, name=name
        )
        self._add(item)
        return item


class DeviceIpv4Loopback(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "eth_name": {"type": str},
        "address": {
            "type": str,
            "format": "ipv4",
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("eth_name", "name")  # type: tuple(str)

    _DEFAULTS = {
        "address": "0.0.0.0",
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, eth_name=None, address="0.0.0.0", name=None):
        super(DeviceIpv4Loopback, self).__init__()
        self._parent = parent
        self._set_property("eth_name", eth_name)
        self._set_property("address", address)
        self._set_property("name", name)

    @property
    def eth_name(self):
        # type: () -> str
        """eth_name getter

        The unique name of the Ethernet interface behind which this Loopback interface will be created.. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name.

        Returns: str
        """
        return self._get_property("eth_name")

    @eth_name.setter
    def eth_name(self, value):
        """eth_name setter

        The unique name of the Ethernet interface behind which this Loopback interface will be created.. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name.

        value: str
        """
        self._set_property("eth_name", value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The IPv4 Loopback address with prefix length of 32.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The IPv4 Loopback address with prefix length of 32.

        value: str
        """
        self._set_property("address", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class DeviceIpv4LoopbackIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceIpv4LoopbackIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceIpv4Loopback]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceIpv4LoopbackIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceIpv4Loopback
        return self._next()

    def next(self):
        # type: () -> DeviceIpv4Loopback
        return self._next()

    def ipv4loopback(self, eth_name=None, address="0.0.0.0", name=None):
        # type: (str,str,str) -> DeviceIpv4LoopbackIter
        """Factory method that creates an instance of the DeviceIpv4Loopback class

        An IPv4 Loopback interface.

        Returns: DeviceIpv4LoopbackIter
        """
        item = DeviceIpv4Loopback(
            parent=self._parent, eth_name=eth_name, address=address, name=name
        )
        self._add(item)
        return self

    def add(self, eth_name=None, address="0.0.0.0", name=None):
        # type: (str,str,str) -> DeviceIpv4Loopback
        """Add method that creates and returns an instance of the DeviceIpv4Loopback class

        An IPv4 Loopback interface.

        Returns: DeviceIpv4Loopback
        """
        item = DeviceIpv4Loopback(
            parent=self._parent, eth_name=eth_name, address=address, name=name
        )
        self._add(item)
        return item


class DeviceIpv6Loopback(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "eth_name": {"type": str},
        "address": {
            "type": str,
            "format": "ipv6",
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("eth_name", "name")  # type: tuple(str)

    _DEFAULTS = {
        "address": "::0",
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, eth_name=None, address="::0", name=None):
        super(DeviceIpv6Loopback, self).__init__()
        self._parent = parent
        self._set_property("eth_name", eth_name)
        self._set_property("address", address)
        self._set_property("name", name)

    @property
    def eth_name(self):
        # type: () -> str
        """eth_name getter

        The unique name of the Ethernet interface behind which this Loopback . interface will be created.. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name.

        Returns: str
        """
        return self._get_property("eth_name")

    @eth_name.setter
    def eth_name(self, value):
        """eth_name setter

        The unique name of the Ethernet interface behind which this Loopback . interface will be created.. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name.

        value: str
        """
        self._set_property("eth_name", value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The IPv6 Loopback address with prefix length of 128.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The IPv6 Loopback address with prefix length of 128.

        value: str
        """
        self._set_property("address", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class DeviceIpv6LoopbackIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceIpv6LoopbackIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceIpv6Loopback]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceIpv6LoopbackIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceIpv6Loopback
        return self._next()

    def next(self):
        # type: () -> DeviceIpv6Loopback
        return self._next()

    def ipv6loopback(self, eth_name=None, address="::0", name=None):
        # type: (str,str,str) -> DeviceIpv6LoopbackIter
        """Factory method that creates an instance of the DeviceIpv6Loopback class

        An IPv6 Loopback interface

        Returns: DeviceIpv6LoopbackIter
        """
        item = DeviceIpv6Loopback(
            parent=self._parent, eth_name=eth_name, address=address, name=name
        )
        self._add(item)
        return self

    def add(self, eth_name=None, address="::0", name=None):
        # type: (str,str,str) -> DeviceIpv6Loopback
        """Add method that creates and returns an instance of the DeviceIpv6Loopback class

        An IPv6 Loopback interface

        Returns: DeviceIpv6Loopback
        """
        item = DeviceIpv6Loopback(
            parent=self._parent, eth_name=eth_name, address=address, name=name
        )
        self._add(item)
        return item


class DeviceIsisRouter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "instance": {"type": "DeviceIsisMultiInstance"},
        "system_id": {
            "type": str,
            "format": "hex",
        },
        "interfaces": {"type": "IsisInterfaceIter"},
        "basic": {"type": "IsisBasic"},
        "advanced": {"type": "IsisAdvanced"},
        "router_auth": {"type": "IsisAuthentication"},
        "v4_routes": {"type": "IsisV4RouteRangeIter"},
        "v6_routes": {"type": "IsisV6RouteRangeIter"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("system_id", "interfaces", "name")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, system_id=None, name=None):
        super(DeviceIsisRouter, self).__init__()
        self._parent = parent
        self._set_property("system_id", system_id)
        self._set_property("name", name)

    @property
    def instance(self):
        # type: () -> DeviceIsisMultiInstance
        """instance getter

        Status: under-review. This container properties of an Multi-Instance-capable router (MI-RTR). Status: under-review. This container properties of an Multi-Instance-capable router (MI-RTR). This contains the properties of a Multi-Instance-capable routers or MI-RTR. Each router can emulate one ISIS instance at a time.

        Returns: DeviceIsisMultiInstance
        """
        return self._get_property("instance", DeviceIsisMultiInstance)

    @property
    def system_id(self):
        # type: () -> str
        """system_id getter

        The System ID for this emulated ISIS router, e.g. "640100010000".

        Returns: str
        """
        return self._get_property("system_id")

    @system_id.setter
    def system_id(self, value):
        """system_id setter

        The System ID for this emulated ISIS router, e.g. "640100010000".

        value: str
        """
        self._set_property("system_id", value)

    @property
    def interfaces(self):
        # type: () -> IsisInterfaceIter
        """interfaces getter

        List of ISIS interfaces for this router.

        Returns: IsisInterfaceIter
        """
        return self._get_property(
            "interfaces", IsisInterfaceIter, self._parent, self._choice
        )

    @property
    def basic(self):
        # type: () -> IsisBasic
        """basic getter

        This contains ISIS router basic properties.This contains ISIS router basic properties.Contains basic properties of an ISIS Router.

        Returns: IsisBasic
        """
        return self._get_property("basic", IsisBasic)

    @property
    def advanced(self):
        # type: () -> IsisAdvanced
        """advanced getter

        Contains ISIS router advanced properties.Contains ISIS router advanced properties.Contains advance properties of an ISIS Router..

        Returns: IsisAdvanced
        """
        return self._get_property("advanced", IsisAdvanced)

    @property
    def router_auth(self):
        # type: () -> IsisAuthentication
        """router_auth getter

        This contains ISIS Area/Domain authentication properties.This contains ISIS Area/Domain authentication properties.ISIS Router authentication properties.

        Returns: IsisAuthentication
        """
        return self._get_property("router_auth", IsisAuthentication)

    @property
    def v4_routes(self):
        # type: () -> IsisV4RouteRangeIter
        """v4_routes getter

        Emulated ISIS IPv4 routes.

        Returns: IsisV4RouteRangeIter
        """
        return self._get_property(
            "v4_routes", IsisV4RouteRangeIter, self._parent, self._choice
        )

    @property
    def v6_routes(self):
        # type: () -> IsisV6RouteRangeIter
        """v6_routes getter

        Emulated ISIS IPv6 routes.

        Returns: IsisV6RouteRangeIter
        """
        return self._get_property(
            "v6_routes", IsisV6RouteRangeIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class DeviceIsisMultiInstance(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "iid": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "itids": {
            "type": list,
            "itemtype": int,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "iid": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, iid=1, itids=None):
        super(DeviceIsisMultiInstance, self).__init__()
        self._parent = parent
        self._set_property("iid", iid)
        self._set_property("itids", itids)

    @property
    def iid(self):
        # type: () -> int
        """iid getter

        Instance Identifier (IID) TLV will associate a PDU with an ISIS instance by using a unique 16-bit number and including one or more Instance-Specific Topology Identifiers (ITIDs).

        Returns: int
        """
        return self._get_property("iid")

    @iid.setter
    def iid(self, value):
        """iid setter

        Instance Identifier (IID) TLV will associate a PDU with an ISIS instance by using a unique 16-bit number and including one or more Instance-Specific Topology Identifiers (ITIDs).

        value: int
        """
        self._set_property("iid", value)

    @property
    def itids(self):
        # type: () -> List[int]
        """itids getter

        This contains one or more ITIDs that will be advertised in IID TLV.

        Returns: List[int]
        """
        return self._get_property("itids")

    @itids.setter
    def itids(self, value):
        """itids setter

        This contains one or more ITIDs that will be advertised in IID TLV.

        value: List[int]
        """
        self._set_property("itids", value)


class IsisInterface(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "eth_name": {"type": str},
        "metric": {"type": int},
        "network_type": {
            "type": str,
            "enum": [
                "broadcast",
                "point_to_point",
            ],
        },
        "level_type": {
            "type": str,
            "enum": [
                "level_1",
                "level_2",
                "level_1_2",
            ],
        },
        "l1_settings": {"type": "IsisInterfaceLevel"},
        "l2_settings": {"type": "IsisInterfaceLevel"},
        "multi_topology_ids": {"type": "IsisMTIter"},
        "traffic_engineering": {"type": "LinkStateTEIter"},
        "authentication": {"type": "IsisInterfaceAuthentication"},
        "advanced": {"type": "IsisInterfaceAdvanced"},
        "link_protection": {"type": "IsisInterfaceLinkProtection"},
        "srlg_values": {
            "type": list,
            "itemtype": int,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("eth_name", "name")  # type: tuple(str)

    _DEFAULTS = {
        "metric": 10,
        "network_type": "broadcast",
        "level_type": "level_2",
    }  # type: Dict[str, Union(type)]

    BROADCAST = "broadcast"  # type: str
    POINT_TO_POINT = "point_to_point"  # type: str

    LEVEL_1 = "level_1"  # type: str
    LEVEL_2 = "level_2"  # type: str
    LEVEL_1_2 = "level_1_2"  # type: str

    def __init__(
        self,
        parent=None,
        eth_name=None,
        metric=10,
        network_type="broadcast",
        level_type="level_2",
        srlg_values=None,
        name=None,
    ):
        super(IsisInterface, self).__init__()
        self._parent = parent
        self._set_property("eth_name", eth_name)
        self._set_property("metric", metric)
        self._set_property("network_type", network_type)
        self._set_property("level_type", level_type)
        self._set_property("srlg_values", srlg_values)
        self._set_property("name", name)

    @property
    def eth_name(self):
        # type: () -> str
        """eth_name getter

        The unique name of the Ethernet interface on which ISIS is running. Two ISIS interfaces cannot share the same Ethernet. . . x-constraint:. - /components/schemas/Device.Ethernet/properties/name.

        Returns: str
        """
        return self._get_property("eth_name")

    @eth_name.setter
    def eth_name(self, value):
        """eth_name setter

        The unique name of the Ethernet interface on which ISIS is running. Two ISIS interfaces cannot share the same Ethernet. . . x-constraint:. - /components/schemas/Device.Ethernet/properties/name.

        value: str
        """
        self._set_property("eth_name", value)

    @property
    def metric(self):
        # type: () -> int
        """metric getter

        The default metric cost for the interface.

        Returns: int
        """
        return self._get_property("metric")

    @metric.setter
    def metric(self, value):
        """metric setter

        The default metric cost for the interface.

        value: int
        """
        self._set_property("metric", value)

    @property
    def network_type(self):
        # type: () -> Union[Literal["broadcast"], Literal["point_to_point"]]
        """network_type getter

        The type of network link.

        Returns: Union[Literal["broadcast"], Literal["point_to_point"]]
        """
        return self._get_property("network_type")

    @network_type.setter
    def network_type(self, value):
        """network_type setter

        The type of network link.

        value: Union[Literal["broadcast"], Literal["point_to_point"]]
        """
        self._set_property("network_type", value)

    @property
    def level_type(self):
        # type: () -> Union[Literal["level_1"], Literal["level_1_2"], Literal["level_2"]]
        """level_type getter

        This indicates whether this router is participating in Level-1 (L1), . Level-2 (L2) or both L1 and L2 domains on this interface.

        Returns: Union[Literal["level_1"], Literal["level_1_2"], Literal["level_2"]]
        """
        return self._get_property("level_type")

    @level_type.setter
    def level_type(self, value):
        """level_type setter

        This indicates whether this router is participating in Level-1 (L1), . Level-2 (L2) or both L1 and L2 domains on this interface.

        value: Union[Literal["level_1"], Literal["level_1_2"], Literal["level_2"]]
        """
        self._set_property("level_type", value)

    @property
    def l1_settings(self):
        # type: () -> IsisInterfaceLevel
        """l1_settings getter

        Configuration for the properties of Level 1 Hello.Configuration for the properties of Level 1 Hello.Settings of Level 1 Hello.

        Returns: IsisInterfaceLevel
        """
        return self._get_property("l1_settings", IsisInterfaceLevel)

    @property
    def l2_settings(self):
        # type: () -> IsisInterfaceLevel
        """l2_settings getter

        Configuration for the properties of Level 1 Hello.Configuration for the properties of Level 1 Hello.Settings of Level 2 Hello.

        Returns: IsisInterfaceLevel
        """
        return self._get_property("l2_settings", IsisInterfaceLevel)

    @property
    def multi_topology_ids(self):
        # type: () -> IsisMTIter
        """multi_topology_ids getter

        Contains the properties of multiple topologies.

        Returns: IsisMTIter
        """
        return self._get_property(
            "multi_topology_ids", IsisMTIter, self._parent, self._choice
        )

    @property
    def traffic_engineering(self):
        # type: () -> LinkStateTEIter
        """traffic_engineering getter

        Contains a list of Traffic Engineering attributes.

        Returns: LinkStateTEIter
        """
        return self._get_property(
            "traffic_engineering", LinkStateTEIter, self._parent, self._choice
        )

    @property
    def authentication(self):
        # type: () -> IsisInterfaceAuthentication
        """authentication getter

        Optional container for circuit authentication properties.Optional container for circuit authentication properties.The Circuit authentication method used for the interfaces on this emulated ISIS v4/v6 router.

        Returns: IsisInterfaceAuthentication
        """
        return self._get_property("authentication", IsisInterfaceAuthentication)

    @property
    def advanced(self):
        # type: () -> IsisInterfaceAdvanced
        """advanced getter

        Optional container for advanced interface properties.Optional container for advanced interface properties.Optional container for advanced interface properties.

        Returns: IsisInterfaceAdvanced
        """
        return self._get_property("advanced", IsisInterfaceAdvanced)

    @property
    def link_protection(self):
        # type: () -> IsisInterfaceLinkProtection
        """link_protection getter

        Optional container for the link protection sub TLV (type 20).Optional container for the link protection sub TLV (type 20).Link protection on the ISIS link between two interfaces.

        Returns: IsisInterfaceLinkProtection
        """
        return self._get_property("link_protection", IsisInterfaceLinkProtection)

    @property
    def srlg_values(self):
        # type: () -> List[int]
        """srlg_values getter

        This contains list of SRLG values for the link between two interfaces.

        Returns: List[int]
        """
        return self._get_property("srlg_values")

    @srlg_values.setter
    def srlg_values(self, value):
        """srlg_values setter

        This contains list of SRLG values for the link between two interfaces.

        value: List[int]
        """
        self._set_property("srlg_values", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class IsisInterfaceLevel(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "priority": {"type": int},
        "hello_interval": {"type": int},
        "dead_interval": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "priority": 0,
        "hello_interval": 10,
        "dead_interval": 30,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, priority=0, hello_interval=10, dead_interval=30):
        super(IsisInterfaceLevel, self).__init__()
        self._parent = parent
        self._set_property("priority", priority)
        self._set_property("hello_interval", hello_interval)
        self._set_property("dead_interval", dead_interval)

    @property
    def priority(self):
        # type: () -> int
        """priority getter

        The Priority setting in Level 1 LAN Hellos for Designated Router election.

        Returns: int
        """
        return self._get_property("priority")

    @priority.setter
    def priority(self, value):
        """priority setter

        The Priority setting in Level 1 LAN Hellos for Designated Router election.

        value: int
        """
        self._set_property("priority", value)

    @property
    def hello_interval(self):
        # type: () -> int
        """hello_interval getter

        The Hello interval for Level 1 Hello messages, in seconds.

        Returns: int
        """
        return self._get_property("hello_interval")

    @hello_interval.setter
    def hello_interval(self, value):
        """hello_interval setter

        The Hello interval for Level 1 Hello messages, in seconds.

        value: int
        """
        self._set_property("hello_interval", value)

    @property
    def dead_interval(self):
        # type: () -> int
        """dead_interval getter

        The Dead (Holding Time) interval for Level 1 Hello messages, in seconds.

        Returns: int
        """
        return self._get_property("dead_interval")

    @dead_interval.setter
    def dead_interval(self, value):
        """dead_interval setter

        The Dead (Holding Time) interval for Level 1 Hello messages, in seconds.

        value: int
        """
        self._set_property("dead_interval", value)


class IsisMT(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "mt_id": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "link_metric": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "mt_id": 0,
        "link_metric": 10,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, mt_id=0, link_metric=10):
        super(IsisMT, self).__init__()
        self._parent = parent
        self._set_property("mt_id", mt_id)
        self._set_property("link_metric", link_metric)

    @property
    def mt_id(self):
        # type: () -> int
        """mt_id getter

        The Multi Topology ID for one of the topologies supported on the ISIS interface.

        Returns: int
        """
        return self._get_property("mt_id")

    @mt_id.setter
    def mt_id(self, value):
        """mt_id setter

        The Multi Topology ID for one of the topologies supported on the ISIS interface.

        value: int
        """
        self._set_property("mt_id", value)

    @property
    def link_metric(self):
        # type: () -> int
        """link_metric getter

        Specifies the link metric for this topology on the ISIS interface.

        Returns: int
        """
        return self._get_property("link_metric")

    @link_metric.setter
    def link_metric(self, value):
        """link_metric setter

        Specifies the link metric for this topology on the ISIS interface.

        value: int
        """
        self._set_property("link_metric", value)


class IsisMTIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisMTIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisMT]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisMTIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisMT
        return self._next()

    def next(self):
        # type: () -> IsisMT
        return self._next()

    def mt(self, mt_id=0, link_metric=10):
        # type: (int,int) -> IsisMTIter
        """Factory method that creates an instance of the IsisMT class

        Status: under-review. Configuration of properties per interface per topology when multiple topologies are configured in an ISIS router.. in a ISIS router.

        Returns: IsisMTIter
        """
        item = IsisMT(parent=self._parent, mt_id=mt_id, link_metric=link_metric)
        self._add(item)
        return self

    def add(self, mt_id=0, link_metric=10):
        # type: (int,int) -> IsisMT
        """Add method that creates and returns an instance of the IsisMT class

        Status: under-review. Configuration of properties per interface per topology when multiple topologies are configured in an ISIS router.. in a ISIS router.

        Returns: IsisMT
        """
        item = IsisMT(parent=self._parent, mt_id=mt_id, link_metric=link_metric)
        self._add(item)
        return item


class LinkStateTE(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "administrative_group": {
            "type": str,
            "format": "hex",
        },
        "metric_level": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4261412864,
        },
        "max_bandwith": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "max_reservable_bandwidth": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "priority_bandwidths": {"type": "LinkStatepriorityBandwidths"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "administrative_group": "00000000",
        "metric_level": 0,
        "max_bandwith": 125000000,
        "max_reservable_bandwidth": 125000000,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        administrative_group="00000000",
        metric_level=0,
        max_bandwith=125000000,
        max_reservable_bandwidth=125000000,
    ):
        super(LinkStateTE, self).__init__()
        self._parent = parent
        self._set_property("administrative_group", administrative_group)
        self._set_property("metric_level", metric_level)
        self._set_property("max_bandwith", max_bandwith)
        self._set_property("max_reservable_bandwidth", max_reservable_bandwidth)

    @property
    def administrative_group(self):
        # type: () -> str
        """administrative_group getter

        The Administrative group sub-TLV (sub-TLV 3). It is a 4-octet . user-defined bit mask used to assign administrative group numbers . to the interface, for use in assigning colors and resource classes. . Each set bit corresponds to a single administrative group for this . interface. The settings translate into Group numbers, which range . from 0 to 31 (integers).

        Returns: str
        """
        return self._get_property("administrative_group")

    @administrative_group.setter
    def administrative_group(self, value):
        """administrative_group setter

        The Administrative group sub-TLV (sub-TLV 3). It is a 4-octet . user-defined bit mask used to assign administrative group numbers . to the interface, for use in assigning colors and resource classes. . Each set bit corresponds to a single administrative group for this . interface. The settings translate into Group numbers, which range . from 0 to 31 (integers).

        value: str
        """
        self._set_property("administrative_group", value)

    @property
    def metric_level(self):
        # type: () -> int
        """metric_level getter

        The user-assigned link metric for Traffic Engineering.

        Returns: int
        """
        return self._get_property("metric_level")

    @metric_level.setter
    def metric_level(self, value):
        """metric_level setter

        The user-assigned link metric for Traffic Engineering.

        value: int
        """
        self._set_property("metric_level", value)

    @property
    def max_bandwith(self):
        # type: () -> int
        """max_bandwith getter

        The maximum link bandwidth (sub-TLV 9) in bytes/sec allowed for this . link for a direction.

        Returns: int
        """
        return self._get_property("max_bandwith")

    @max_bandwith.setter
    def max_bandwith(self, value):
        """max_bandwith setter

        The maximum link bandwidth (sub-TLV 9) in bytes/sec allowed for this . link for a direction.

        value: int
        """
        self._set_property("max_bandwith", value)

    @property
    def max_reservable_bandwidth(self):
        # type: () -> int
        """max_reservable_bandwidth getter

        The maximum link bandwidth (sub-TLV 10) in bytes/sec allowed for this . link in a direction.

        Returns: int
        """
        return self._get_property("max_reservable_bandwidth")

    @max_reservable_bandwidth.setter
    def max_reservable_bandwidth(self, value):
        """max_reservable_bandwidth setter

        The maximum link bandwidth (sub-TLV 10) in bytes/sec allowed for this . link in a direction.

        value: int
        """
        self._set_property("max_reservable_bandwidth", value)

    @property
    def priority_bandwidths(self):
        # type: () -> LinkStatepriorityBandwidths
        """priority_bandwidths getter

        Specifies the amount of bandwidth that can be reserved with a setup priority of 0 . through 7, arranged in increasing order with priority 0 having highest priority. . In ISIS, this is sent in sub-TLV (11) of Extended IS Reachability TLV. Specifies the amount of bandwidth that can be reserved with a setup priority of 0 . through 7, arranged in increasing order with priority 0 having highest priority. . In ISIS, this is sent in sub-TLV (11) of Extended IS Reachability TLV. Configuration of bandwidths of priority 0 through priority 7.

        Returns: LinkStatepriorityBandwidths
        """
        return self._get_property("priority_bandwidths", LinkStatepriorityBandwidths)


class LinkStatepriorityBandwidths(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "pb0": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "pb1": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "pb2": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "pb3": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "pb4": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "pb5": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "pb6": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "pb7": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "pb0": 125000000,
        "pb1": 125000000,
        "pb2": 125000000,
        "pb3": 125000000,
        "pb4": 125000000,
        "pb5": 125000000,
        "pb6": 125000000,
        "pb7": 125000000,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        pb0=125000000,
        pb1=125000000,
        pb2=125000000,
        pb3=125000000,
        pb4=125000000,
        pb5=125000000,
        pb6=125000000,
        pb7=125000000,
    ):
        super(LinkStatepriorityBandwidths, self).__init__()
        self._parent = parent
        self._set_property("pb0", pb0)
        self._set_property("pb1", pb1)
        self._set_property("pb2", pb2)
        self._set_property("pb3", pb3)
        self._set_property("pb4", pb4)
        self._set_property("pb5", pb5)
        self._set_property("pb6", pb6)
        self._set_property("pb7", pb7)

    @property
    def pb0(self):
        # type: () -> int
        """pb0 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 0.

        Returns: int
        """
        return self._get_property("pb0")

    @pb0.setter
    def pb0(self, value):
        """pb0 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 0.

        value: int
        """
        self._set_property("pb0", value)

    @property
    def pb1(self):
        # type: () -> int
        """pb1 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 1.

        Returns: int
        """
        return self._get_property("pb1")

    @pb1.setter
    def pb1(self, value):
        """pb1 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 1.

        value: int
        """
        self._set_property("pb1", value)

    @property
    def pb2(self):
        # type: () -> int
        """pb2 getter

        Specify the amount of bandwidth that can be reserved for the Priority 2.

        Returns: int
        """
        return self._get_property("pb2")

    @pb2.setter
    def pb2(self, value):
        """pb2 setter

        Specify the amount of bandwidth that can be reserved for the Priority 2.

        value: int
        """
        self._set_property("pb2", value)

    @property
    def pb3(self):
        # type: () -> int
        """pb3 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 3.

        Returns: int
        """
        return self._get_property("pb3")

    @pb3.setter
    def pb3(self, value):
        """pb3 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 3.

        value: int
        """
        self._set_property("pb3", value)

    @property
    def pb4(self):
        # type: () -> int
        """pb4 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 4.

        Returns: int
        """
        return self._get_property("pb4")

    @pb4.setter
    def pb4(self, value):
        """pb4 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 4.

        value: int
        """
        self._set_property("pb4", value)

    @property
    def pb5(self):
        # type: () -> int
        """pb5 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 5.

        Returns: int
        """
        return self._get_property("pb5")

    @pb5.setter
    def pb5(self, value):
        """pb5 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 5.

        value: int
        """
        self._set_property("pb5", value)

    @property
    def pb6(self):
        # type: () -> int
        """pb6 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 6.

        Returns: int
        """
        return self._get_property("pb6")

    @pb6.setter
    def pb6(self, value):
        """pb6 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 6.

        value: int
        """
        self._set_property("pb6", value)

    @property
    def pb7(self):
        # type: () -> int
        """pb7 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 7.

        Returns: int
        """
        return self._get_property("pb7")

    @pb7.setter
    def pb7(self, value):
        """pb7 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 7.

        value: int
        """
        self._set_property("pb7", value)


class LinkStateTEIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(LinkStateTEIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[LinkStateTE]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> LinkStateTEIter
        return self._iter()

    def __next__(self):
        # type: () -> LinkStateTE
        return self._next()

    def next(self):
        # type: () -> LinkStateTE
        return self._next()

    def te(
        self,
        administrative_group="00000000",
        metric_level=0,
        max_bandwith=125000000,
        max_reservable_bandwidth=125000000,
    ):
        # type: (str,int,int,int) -> LinkStateTEIter
        """Factory method that creates an instance of the LinkStateTE class

        Status: under-review. A container for Traffic Engineering properties on a interface.

        Returns: LinkStateTEIter
        """
        item = LinkStateTE(
            parent=self._parent,
            administrative_group=administrative_group,
            metric_level=metric_level,
            max_bandwith=max_bandwith,
            max_reservable_bandwidth=max_reservable_bandwidth,
        )
        self._add(item)
        return self

    def add(
        self,
        administrative_group="00000000",
        metric_level=0,
        max_bandwith=125000000,
        max_reservable_bandwidth=125000000,
    ):
        # type: (str,int,int,int) -> LinkStateTE
        """Add method that creates and returns an instance of the LinkStateTE class

        Status: under-review. A container for Traffic Engineering properties on a interface.

        Returns: LinkStateTE
        """
        item = LinkStateTE(
            parent=self._parent,
            administrative_group=administrative_group,
            metric_level=metric_level,
            max_bandwith=max_bandwith,
            max_reservable_bandwidth=max_reservable_bandwidth,
        )
        self._add(item)
        return item


class IsisInterfaceAuthentication(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "auth_type": {
            "type": str,
            "enum": [
                "md5",
                "password",
            ],
        },
        "md5": {
            "type": str,
            "minLength": 0,
            "maxLength": 255,
        },
        "password": {
            "type": str,
            "minLength": 0,
            "maxLength": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("auth_type",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    MD5 = "md5"  # type: str
    PASSWORD = "password"  # type: str

    def __init__(self, parent=None, auth_type=None, md5=None, password=None):
        super(IsisInterfaceAuthentication, self).__init__()
        self._parent = parent
        self._set_property("auth_type", auth_type)
        self._set_property("md5", md5)
        self._set_property("password", password)

    @property
    def auth_type(self):
        # type: () -> Union[Literal["md5"], Literal["password"]]
        """auth_type getter

        The circuit authentication method.

        Returns: Union[Literal["md5"], Literal["password"]]
        """
        return self._get_property("auth_type")

    @auth_type.setter
    def auth_type(self, value):
        """auth_type setter

        The circuit authentication method.

        value: Union[Literal["md5"], Literal["password"]]
        """
        self._set_property("auth_type", value)

    @property
    def md5(self):
        # type: () -> str
        """md5 getter

        MD5 key to be used for authentication.

        Returns: str
        """
        return self._get_property("md5")

    @md5.setter
    def md5(self, value):
        """md5 setter

        MD5 key to be used for authentication.

        value: str
        """
        self._set_property("md5", value)

    @property
    def password(self):
        # type: () -> str
        """password getter

        The password, in clear text, to be used for Authentication.

        Returns: str
        """
        return self._get_property("password")

    @password.setter
    def password(self, value):
        """password setter

        The password, in clear text, to be used for Authentication.

        value: str
        """
        self._set_property("password", value)


class IsisInterfaceAdvanced(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "auto_adjust_mtu": {"type": bool},
        "auto_adjust_area": {"type": bool},
        "auto_adjust_supported_protocols": {"type": bool},
        "enable_3way_handshake": {"type": bool},
        "p2p_hellos_to_unicast_mac": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "auto_adjust_mtu": True,
        "auto_adjust_area": True,
        "auto_adjust_supported_protocols": False,
        "enable_3way_handshake": True,
        "p2p_hellos_to_unicast_mac": False,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        auto_adjust_mtu=True,
        auto_adjust_area=True,
        auto_adjust_supported_protocols=False,
        enable_3way_handshake=True,
        p2p_hellos_to_unicast_mac=False,
    ):
        super(IsisInterfaceAdvanced, self).__init__()
        self._parent = parent
        self._set_property("auto_adjust_mtu", auto_adjust_mtu)
        self._set_property("auto_adjust_area", auto_adjust_area)
        self._set_property(
            "auto_adjust_supported_protocols", auto_adjust_supported_protocols
        )
        self._set_property("enable_3way_handshake", enable_3way_handshake)
        self._set_property("p2p_hellos_to_unicast_mac", p2p_hellos_to_unicast_mac)

    @property
    def auto_adjust_mtu(self):
        # type: () -> bool
        """auto_adjust_mtu getter

        If a padded Hello message is received on the interface, the length of . the Hello packets sent out on that interface is adjusted to match.

        Returns: bool
        """
        return self._get_property("auto_adjust_mtu")

    @auto_adjust_mtu.setter
    def auto_adjust_mtu(self, value):
        """auto_adjust_mtu setter

        If a padded Hello message is received on the interface, the length of . the Hello packets sent out on that interface is adjusted to match.

        value: bool
        """
        self._set_property("auto_adjust_mtu", value)

    @property
    def auto_adjust_area(self):
        # type: () -> bool
        """auto_adjust_area getter

        If a Level 1 Hello is received on this emulated router for an area . not currently in its area list, an area from the received Hello is . added to that list. This ensures an area match for all future . Level 1 Hellos from the source L1 router.

        Returns: bool
        """
        return self._get_property("auto_adjust_area")

    @auto_adjust_area.setter
    def auto_adjust_area(self, value):
        """auto_adjust_area setter

        If a Level 1 Hello is received on this emulated router for an area . not currently in its area list, an area from the received Hello is . added to that list. This ensures an area match for all future . Level 1 Hellos from the source L1 router.

        value: bool
        """
        self._set_property("auto_adjust_area", value)

    @property
    def auto_adjust_supported_protocols(self):
        # type: () -> bool
        """auto_adjust_supported_protocols getter

        If a Hello message listing supported protocols is received on this . emulated router, the supported protocols advertised by this router . are changed to match exactly.

        Returns: bool
        """
        return self._get_property("auto_adjust_supported_protocols")

    @auto_adjust_supported_protocols.setter
    def auto_adjust_supported_protocols(self, value):
        """auto_adjust_supported_protocols setter

        If a Hello message listing supported protocols is received on this . emulated router, the supported protocols advertised by this router . are changed to match exactly.

        value: bool
        """
        self._set_property("auto_adjust_supported_protocols", value)

    @property
    def enable_3way_handshake(self):
        # type: () -> bool
        """enable_3way_handshake getter

        If it is true, the Point-to-Point circuit will include 3-way TLV in its Point-to-Point IIH and attempt to establish the adjacency as specified in RFC 5303. This field is not applicable if network_type is set to 'broadcast' type in ISIS interface.

        Returns: bool
        """
        return self._get_property("enable_3way_handshake")

    @enable_3way_handshake.setter
    def enable_3way_handshake(self, value):
        """enable_3way_handshake setter

        If it is true, the Point-to-Point circuit will include 3-way TLV in its Point-to-Point IIH and attempt to establish the adjacency as specified in RFC 5303. This field is not applicable if network_type is set to 'broadcast' type in ISIS interface.

        value: bool
        """
        self._set_property("enable_3way_handshake", value)

    @property
    def p2p_hellos_to_unicast_mac(self):
        # type: () -> bool
        """p2p_hellos_to_unicast_mac getter

        If it is true, the Point-to-Point Hello messages will be sent to the unicast MAC address.

        Returns: bool
        """
        return self._get_property("p2p_hellos_to_unicast_mac")

    @p2p_hellos_to_unicast_mac.setter
    def p2p_hellos_to_unicast_mac(self, value):
        """p2p_hellos_to_unicast_mac setter

        If it is true, the Point-to-Point Hello messages will be sent to the unicast MAC address.

        value: bool
        """
        self._set_property("p2p_hellos_to_unicast_mac", value)


class IsisInterfaceLinkProtection(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "extra_traffic": {"type": bool},
        "unprotected": {"type": bool},
        "shared": {"type": bool},
        "dedicated_1_to_1": {"type": bool},
        "dedicated_1_plus_1": {"type": bool},
        "enhanced": {"type": bool},
        "reserved_40": {"type": bool},
        "reserved_80": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "extra_traffic": False,
        "unprotected": False,
        "shared": False,
        "dedicated_1_to_1": False,
        "dedicated_1_plus_1": False,
        "enhanced": False,
        "reserved_40": False,
        "reserved_80": False,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        extra_traffic=False,
        unprotected=False,
        shared=False,
        dedicated_1_to_1=False,
        dedicated_1_plus_1=False,
        enhanced=False,
        reserved_40=False,
        reserved_80=False,
    ):
        super(IsisInterfaceLinkProtection, self).__init__()
        self._parent = parent
        self._set_property("extra_traffic", extra_traffic)
        self._set_property("unprotected", unprotected)
        self._set_property("shared", shared)
        self._set_property("dedicated_1_to_1", dedicated_1_to_1)
        self._set_property("dedicated_1_plus_1", dedicated_1_plus_1)
        self._set_property("enhanced", enhanced)
        self._set_property("reserved_40", reserved_40)
        self._set_property("reserved_80", reserved_80)

    @property
    def extra_traffic(self):
        # type: () -> bool
        """extra_traffic getter

        Enable this to protect other link or links. LSPs on a link of this type are lost. if any of the links fail.

        Returns: bool
        """
        return self._get_property("extra_traffic")

    @extra_traffic.setter
    def extra_traffic(self, value):
        """extra_traffic setter

        Enable this to protect other link or links. LSPs on a link of this type are lost. if any of the links fail.

        value: bool
        """
        self._set_property("extra_traffic", value)

    @property
    def unprotected(self):
        # type: () -> bool
        """unprotected getter

        Enabling this signifies that there is no other link protecting this . link. LSPs on a link of this type are lost if the link fails.

        Returns: bool
        """
        return self._get_property("unprotected")

    @unprotected.setter
    def unprotected(self, value):
        """unprotected setter

        Enabling this signifies that there is no other link protecting this . link. LSPs on a link of this type are lost if the link fails.

        value: bool
        """
        self._set_property("unprotected", value)

    @property
    def shared(self):
        # type: () -> bool
        """shared getter

        Enable this to share the Extra Traffic links between one or more . links of type Shared.There are one or more disjoint links of type . Extra Traffic that are protecting this link.

        Returns: bool
        """
        return self._get_property("shared")

    @shared.setter
    def shared(self, value):
        """shared setter

        Enable this to share the Extra Traffic links between one or more . links of type Shared.There are one or more disjoint links of type . Extra Traffic that are protecting this link.

        value: bool
        """
        self._set_property("shared", value)

    @property
    def dedicated_1_to_1(self):
        # type: () -> bool
        """dedicated_1_to_1 getter

        Enabling this signifies that there is one dedicated disjoint link . of type Extra Traffic that is protecting this link.

        Returns: bool
        """
        return self._get_property("dedicated_1_to_1")

    @dedicated_1_to_1.setter
    def dedicated_1_to_1(self, value):
        """dedicated_1_to_1 setter

        Enabling this signifies that there is one dedicated disjoint link . of type Extra Traffic that is protecting this link.

        value: bool
        """
        self._set_property("dedicated_1_to_1", value)

    @property
    def dedicated_1_plus_1(self):
        # type: () -> bool
        """dedicated_1_plus_1 getter

        Enabling this signifies that a dedicated disjoint link is protecting . this link. However, the protecting link is not advertised in the . link state database and is therefore not available for the routing . of LSPs.

        Returns: bool
        """
        return self._get_property("dedicated_1_plus_1")

    @dedicated_1_plus_1.setter
    def dedicated_1_plus_1(self, value):
        """dedicated_1_plus_1 setter

        Enabling this signifies that a dedicated disjoint link is protecting . this link. However, the protecting link is not advertised in the . link state database and is therefore not available for the routing . of LSPs.

        value: bool
        """
        self._set_property("dedicated_1_plus_1", value)

    @property
    def enhanced(self):
        # type: () -> bool
        """enhanced getter

        Enabling this signifies that a protection scheme that is more . reliable than Dedicated 1+1.

        Returns: bool
        """
        return self._get_property("enhanced")

    @enhanced.setter
    def enhanced(self, value):
        """enhanced setter

        Enabling this signifies that a protection scheme that is more . reliable than Dedicated 1+1.

        value: bool
        """
        self._set_property("enhanced", value)

    @property
    def reserved_40(self):
        # type: () -> bool
        """reserved_40 getter

        This is a Protection Scheme with value 0x40.

        Returns: bool
        """
        return self._get_property("reserved_40")

    @reserved_40.setter
    def reserved_40(self, value):
        """reserved_40 setter

        This is a Protection Scheme with value 0x40.

        value: bool
        """
        self._set_property("reserved_40", value)

    @property
    def reserved_80(self):
        # type: () -> bool
        """reserved_80 getter

        This is a Protection Scheme with value 0x80.

        Returns: bool
        """
        return self._get_property("reserved_80")

    @reserved_80.setter
    def reserved_80(self, value):
        """reserved_80 setter

        This is a Protection Scheme with value 0x80.

        value: bool
        """
        self._set_property("reserved_80", value)


class IsisInterfaceIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisInterfaceIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisInterface]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisInterfaceIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisInterface
        return self._next()

    def next(self):
        # type: () -> IsisInterface
        return self._next()

    def interface(
        self,
        eth_name=None,
        metric=10,
        network_type="broadcast",
        level_type="level_2",
        srlg_values=None,
        name=None,
    ):
        # type: (str,int,Union[Literal["broadcast"], Literal["point_to_point"]],Union[Literal["level_1"], Literal["level_1_2"], Literal["level_2"]],List[int],str) -> IsisInterfaceIter
        """Factory method that creates an instance of the IsisInterface class

        Status: under-review. Configuration for single ISIS interface.

        Returns: IsisInterfaceIter
        """
        item = IsisInterface(
            parent=self._parent,
            eth_name=eth_name,
            metric=metric,
            network_type=network_type,
            level_type=level_type,
            srlg_values=srlg_values,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        eth_name=None,
        metric=10,
        network_type="broadcast",
        level_type="level_2",
        srlg_values=None,
        name=None,
    ):
        # type: (str,int,Union[Literal["broadcast"], Literal["point_to_point"]],Union[Literal["level_1"], Literal["level_1_2"], Literal["level_2"]],List[int],str) -> IsisInterface
        """Add method that creates and returns an instance of the IsisInterface class

        Status: under-review. Configuration for single ISIS interface.

        Returns: IsisInterface
        """
        item = IsisInterface(
            parent=self._parent,
            eth_name=eth_name,
            metric=metric,
            network_type=network_type,
            level_type=level_type,
            srlg_values=srlg_values,
            name=name,
        )
        self._add(item)
        return item


class IsisBasic(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ipv4_te_router_id": {
            "type": str,
            "format": "ipv4",
        },
        "hostname": {"type": str},
        "enable_wide_metric": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "enable_wide_metric": True,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        ipv4_te_router_id=None,
        hostname=None,
        enable_wide_metric=True,
    ):
        super(IsisBasic, self).__init__()
        self._parent = parent
        self._set_property("ipv4_te_router_id", ipv4_te_router_id)
        self._set_property("hostname", hostname)
        self._set_property("enable_wide_metric", enable_wide_metric)

    @property
    def ipv4_te_router_id(self):
        # type: () -> str
        """ipv4_te_router_id getter

        IPv4 Traffic Engineering(TE) router id. This address should be configured as an IPv4 Loopback address in 'ipv4_loopbacks' in the Device.

        Returns: str
        """
        return self._get_property("ipv4_te_router_id")

    @ipv4_te_router_id.setter
    def ipv4_te_router_id(self, value):
        """ipv4_te_router_id setter

        IPv4 Traffic Engineering(TE) router id. This address should be configured as an IPv4 Loopback address in 'ipv4_loopbacks' in the Device.

        value: str
        """
        self._set_property("ipv4_te_router_id", value)

    @property
    def hostname(self):
        # type: () -> str
        """hostname getter

        Host name for the router. The host name is transmitted in all the packets sent from the router.

        Returns: str
        """
        return self._get_property("hostname")

    @hostname.setter
    def hostname(self, value):
        """hostname setter

        Host name for the router. The host name is transmitted in all the packets sent from the router.

        value: str
        """
        self._set_property("hostname", value)

    @property
    def enable_wide_metric(self):
        # type: () -> bool
        """enable_wide_metric getter

        When set to true, it allows sending of more detailed metric information for the routes using 32-bit wide values using TLV 135 IP reachability and more detailed reachability information for IS reachability by using TLV 22. The detailed usage is described in RFC3784.

        Returns: bool
        """
        return self._get_property("enable_wide_metric")

    @enable_wide_metric.setter
    def enable_wide_metric(self, value):
        """enable_wide_metric setter

        When set to true, it allows sending of more detailed metric information for the routes using 32-bit wide values using TLV 135 IP reachability and more detailed reachability information for IS reachability by using TLV 22. The detailed usage is described in RFC3784.

        value: bool
        """
        self._set_property("enable_wide_metric", value)


class IsisAdvanced(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "enable_hello_padding": {"type": bool},
        "max_area_addresses": {
            "type": int,
            "minimum": 0,
            "maximum": 254,
        },
        "area_addresses": {
            "type": list,
            "itemtype": str,
        },
        "lsp_refresh_rate": {
            "type": int,
            "minimum": 1,
            "maximum": 65535,
        },
        "lsp_lifetime": {
            "type": int,
            "minimum": 1,
            "maximum": 65535,
        },
        "psnp_interval": {
            "type": int,
            "minimum": 1,
            "maximum": 60000,
        },
        "csnp_interval": {
            "type": int,
            "minimum": 1,
            "maximum": 65535000,
        },
        "max_lsp_size": {
            "type": int,
            "minimum": 64,
            "maximum": 9216,
        },
        "lsp_mgroup_min_trans_interval": {
            "type": int,
            "minimum": 1,
            "maximum": 60000,
        },
        "enable_attached_bit": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "enable_hello_padding": True,
        "max_area_addresses": 3,
        "lsp_refresh_rate": 600,
        "lsp_lifetime": 1200,
        "psnp_interval": 2000,
        "csnp_interval": 10000,
        "max_lsp_size": 1492,
        "lsp_mgroup_min_trans_interval": 5000,
        "enable_attached_bit": True,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        enable_hello_padding=True,
        max_area_addresses=3,
        area_addresses=None,
        lsp_refresh_rate=600,
        lsp_lifetime=1200,
        psnp_interval=2000,
        csnp_interval=10000,
        max_lsp_size=1492,
        lsp_mgroup_min_trans_interval=5000,
        enable_attached_bit=True,
    ):
        super(IsisAdvanced, self).__init__()
        self._parent = parent
        self._set_property("enable_hello_padding", enable_hello_padding)
        self._set_property("max_area_addresses", max_area_addresses)
        self._set_property("area_addresses", area_addresses)
        self._set_property("lsp_refresh_rate", lsp_refresh_rate)
        self._set_property("lsp_lifetime", lsp_lifetime)
        self._set_property("psnp_interval", psnp_interval)
        self._set_property("csnp_interval", csnp_interval)
        self._set_property("max_lsp_size", max_lsp_size)
        self._set_property(
            "lsp_mgroup_min_trans_interval", lsp_mgroup_min_trans_interval
        )
        self._set_property("enable_attached_bit", enable_attached_bit)

    @property
    def enable_hello_padding(self):
        # type: () -> bool
        """enable_hello_padding getter

        It enables padding of Hello message to MTU size.

        Returns: bool
        """
        return self._get_property("enable_hello_padding")

    @enable_hello_padding.setter
    def enable_hello_padding(self, value):
        """enable_hello_padding setter

        It enables padding of Hello message to MTU size.

        value: bool
        """
        self._set_property("enable_hello_padding", value)

    @property
    def max_area_addresses(self):
        # type: () -> int
        """max_area_addresses getter

        The Number of Area Addresses permitted, with a valid range from 0 to 254. A zero indicates a maximum of 3 addresses.

        Returns: int
        """
        return self._get_property("max_area_addresses")

    @max_area_addresses.setter
    def max_area_addresses(self, value):
        """max_area_addresses setter

        The Number of Area Addresses permitted, with a valid range from 0 to 254. A zero indicates a maximum of 3 addresses.

        value: int
        """
        self._set_property("max_area_addresses", value)

    @property
    def area_addresses(self):
        # type: () -> List[str]
        """area_addresses getter

        Its combination of the ISP and HO-DSP.Usually all nodes within an area have the same area address. If no area addresses are configured, a default area of "490001" will be advertised.

        Returns: List[str]
        """
        return self._get_property("area_addresses")

    @area_addresses.setter
    def area_addresses(self, value):
        """area_addresses setter

        Its combination of the ISP and HO-DSP.Usually all nodes within an area have the same area address. If no area addresses are configured, a default area of "490001" will be advertised.

        value: List[str]
        """
        self._set_property("area_addresses", value)

    @property
    def lsp_refresh_rate(self):
        # type: () -> int
        """lsp_refresh_rate getter

        The rate at which LSPs are re-sent in seconds.

        Returns: int
        """
        return self._get_property("lsp_refresh_rate")

    @lsp_refresh_rate.setter
    def lsp_refresh_rate(self, value):
        """lsp_refresh_rate setter

        The rate at which LSPs are re-sent in seconds.

        value: int
        """
        self._set_property("lsp_refresh_rate", value)

    @property
    def lsp_lifetime(self):
        # type: () -> int
        """lsp_lifetime getter

        The MaxAge for retaining a learned LSP on this router in seconds.

        Returns: int
        """
        return self._get_property("lsp_lifetime")

    @lsp_lifetime.setter
    def lsp_lifetime(self, value):
        """lsp_lifetime setter

        The MaxAge for retaining a learned LSP on this router in seconds.

        value: int
        """
        self._set_property("lsp_lifetime", value)

    @property
    def psnp_interval(self):
        # type: () -> int
        """psnp_interval getter

        The number of milliseconds between transmissions of Partial Sequence Number PDU.

        Returns: int
        """
        return self._get_property("psnp_interval")

    @psnp_interval.setter
    def psnp_interval(self, value):
        """psnp_interval setter

        The number of milliseconds between transmissions of Partial Sequence Number PDU.

        value: int
        """
        self._set_property("psnp_interval", value)

    @property
    def csnp_interval(self):
        # type: () -> int
        """csnp_interval getter

        The number of milliseconds between transmissions of Partial Sequence Number PDU.

        Returns: int
        """
        return self._get_property("csnp_interval")

    @csnp_interval.setter
    def csnp_interval(self, value):
        """csnp_interval setter

        The number of milliseconds between transmissions of Partial Sequence Number PDU.

        value: int
        """
        self._set_property("csnp_interval", value)

    @property
    def max_lsp_size(self):
        # type: () -> int
        """max_lsp_size getter

        The maximum size in bytes of any LSP that can be transmitted over a link of equal or less than maximum MTU size.

        Returns: int
        """
        return self._get_property("max_lsp_size")

    @max_lsp_size.setter
    def max_lsp_size(self, value):
        """max_lsp_size setter

        The maximum size in bytes of any LSP that can be transmitted over a link of equal or less than maximum MTU size.

        value: int
        """
        self._set_property("max_lsp_size", value)

    @property
    def lsp_mgroup_min_trans_interval(self):
        # type: () -> int
        """lsp_mgroup_min_trans_interval getter

        The number of seconds between transmissions of LSPs/MGROUP-PDUs.

        Returns: int
        """
        return self._get_property("lsp_mgroup_min_trans_interval")

    @lsp_mgroup_min_trans_interval.setter
    def lsp_mgroup_min_trans_interval(self, value):
        """lsp_mgroup_min_trans_interval setter

        The number of seconds between transmissions of LSPs/MGROUP-PDUs.

        value: int
        """
        self._set_property("lsp_mgroup_min_trans_interval", value)

    @property
    def enable_attached_bit(self):
        # type: () -> bool
        """enable_attached_bit getter

        If the Attached bit is enabled, it indicates that the ISIS router is attached to another area or the Level 2 backbone. The purpose of an Attached-Bit is to accomplish Inter-Area Routing. When an L1/L2 router is connected to more than one area, it sets the Attached-bit on its L1 LSP. This can cause a default route ( 0.0.0.0/0 ) to be installed by the receiving router.

        Returns: bool
        """
        return self._get_property("enable_attached_bit")

    @enable_attached_bit.setter
    def enable_attached_bit(self, value):
        """enable_attached_bit setter

        If the Attached bit is enabled, it indicates that the ISIS router is attached to another area or the Level 2 backbone. The purpose of an Attached-Bit is to accomplish Inter-Area Routing. When an L1/L2 router is connected to more than one area, it sets the Attached-bit on its L1 LSP. This can cause a default route ( 0.0.0.0/0 ) to be installed by the receiving router.

        value: bool
        """
        self._set_property("enable_attached_bit", value)


class IsisAuthentication(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ignore_receive_md5": {"type": bool},
        "area_auth": {"type": "IsisAuthenticationBase"},
        "domain_auth": {"type": "IsisAuthenticationBase"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "ignore_receive_md5": True,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, ignore_receive_md5=True):
        super(IsisAuthentication, self).__init__()
        self._parent = parent
        self._set_property("ignore_receive_md5", ignore_receive_md5)

    @property
    def ignore_receive_md5(self):
        # type: () -> bool
        """ignore_receive_md5 getter

        Do not verify MD5 checksum in received LSPs.

        Returns: bool
        """
        return self._get_property("ignore_receive_md5")

    @ignore_receive_md5.setter
    def ignore_receive_md5(self, value):
        """ignore_receive_md5 setter

        Do not verify MD5 checksum in received LSPs.

        value: bool
        """
        self._set_property("ignore_receive_md5", value)

    @property
    def area_auth(self):
        # type: () -> IsisAuthenticationBase
        """area_auth getter

        Optional container for ISIS authentication properties.Optional container for ISIS authentication properties.The Area authentication method used for the emulated ISIS router.. This is used for L1 LSPs.

        Returns: IsisAuthenticationBase
        """
        return self._get_property("area_auth", IsisAuthenticationBase)

    @property
    def domain_auth(self):
        # type: () -> IsisAuthenticationBase
        """domain_auth getter

        Optional container for ISIS authentication properties.Optional container for ISIS authentication properties.The Domain authentication method used for the emulated ISIS router.. This is used for L2 LSPs.

        Returns: IsisAuthenticationBase
        """
        return self._get_property("domain_auth", IsisAuthenticationBase)


class IsisAuthenticationBase(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "auth_type": {
            "type": str,
            "enum": [
                "md5",
                "password",
            ],
        },
        "md5": {
            "type": str,
            "minLength": 0,
            "maxLength": 255,
        },
        "password": {
            "type": str,
            "minLength": 0,
            "maxLength": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("auth_type",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    MD5 = "md5"  # type: str
    PASSWORD = "password"  # type: str

    def __init__(self, parent=None, auth_type=None, md5=None, password=None):
        super(IsisAuthenticationBase, self).__init__()
        self._parent = parent
        self._set_property("auth_type", auth_type)
        self._set_property("md5", md5)
        self._set_property("password", password)

    @property
    def auth_type(self):
        # type: () -> Union[Literal["md5"], Literal["password"]]
        """auth_type getter

        The authentication method.

        Returns: Union[Literal["md5"], Literal["password"]]
        """
        return self._get_property("auth_type")

    @auth_type.setter
    def auth_type(self, value):
        """auth_type setter

        The authentication method.

        value: Union[Literal["md5"], Literal["password"]]
        """
        self._set_property("auth_type", value)

    @property
    def md5(self):
        # type: () -> str
        """md5 getter

        Authentication as an MD5 key.

        Returns: str
        """
        return self._get_property("md5")

    @md5.setter
    def md5(self, value):
        """md5 setter

        Authentication as an MD5 key.

        value: str
        """
        self._set_property("md5", value)

    @property
    def password(self):
        # type: () -> str
        """password getter

        Authentication as a clear text password.

        Returns: str
        """
        return self._get_property("password")

    @password.setter
    def password(self, value):
        """password setter

        Authentication as a clear text password.

        value: str
        """
        self._set_property("password", value)


class IsisV4RouteRange(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "addresses": {"type": "V4RouteAddressIter"},
        "link_metric": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "origin_type": {
            "type": str,
            "enum": [
                "internal",
                "external",
            ],
        },
        "redistribution_type": {
            "type": str,
            "enum": [
                "up",
                "down",
            ],
        },
        "name": {"type": str},
        "prefix_attr_enabled": {"type": bool},
        "x_flag": {"type": bool},
        "r_flag": {"type": bool},
        "n_flag": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "link_metric": 0,
        "origin_type": "internal",
        "redistribution_type": "up",
        "prefix_attr_enabled": False,
        "x_flag": False,
        "r_flag": False,
        "n_flag": False,
    }  # type: Dict[str, Union(type)]

    INTERNAL = "internal"  # type: str
    EXTERNAL = "external"  # type: str

    UP = "up"  # type: str
    DOWN = "down"  # type: str

    def __init__(
        self,
        parent=None,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        super(IsisV4RouteRange, self).__init__()
        self._parent = parent
        self._set_property("link_metric", link_metric)
        self._set_property("origin_type", origin_type)
        self._set_property("redistribution_type", redistribution_type)
        self._set_property("name", name)
        self._set_property("prefix_attr_enabled", prefix_attr_enabled)
        self._set_property("x_flag", x_flag)
        self._set_property("r_flag", r_flag)
        self._set_property("n_flag", n_flag)

    @property
    def addresses(self):
        # type: () -> V4RouteAddressIter
        """addresses getter

        A list of group of IPv4 route addresses.

        Returns: V4RouteAddressIter
        """
        return self._get_property(
            "addresses", V4RouteAddressIter, self._parent, self._choice
        )

    @property
    def link_metric(self):
        # type: () -> int
        """link_metric getter

        The user-defined metric associated with this route range.

        Returns: int
        """
        return self._get_property("link_metric")

    @link_metric.setter
    def link_metric(self, value):
        """link_metric setter

        The user-defined metric associated with this route range.

        value: int
        """
        self._set_property("link_metric", value)

    @property
    def origin_type(self):
        # type: () -> Union[Literal["external"], Literal["internal"]]
        """origin_type getter

        The origin of the advertised route-internal or external to the ISIS area. Options include the following: . Internal-for intra-area routes, through Level 1 LSPs. . External-for inter-area routes redistributed within L1, through Level 1 LSPs.

        Returns: Union[Literal["external"], Literal["internal"]]
        """
        return self._get_property("origin_type")

    @origin_type.setter
    def origin_type(self, value):
        """origin_type setter

        The origin of the advertised route-internal or external to the ISIS area. Options include the following: . Internal-for intra-area routes, through Level 1 LSPs. . External-for inter-area routes redistributed within L1, through Level 1 LSPs.

        value: Union[Literal["external"], Literal["internal"]]
        """
        self._set_property("origin_type", value)

    @property
    def redistribution_type(self):
        # type: () -> Union[Literal["down"], Literal["up"]]
        """redistribution_type getter

        Defines the Up/Down (Redistribution) bit defined for TLVs 128 and 130 by RFC 2966. It is used for domain-wide advertisement of prefix information. . Up (0)-used when a prefix is initially advertised within the ISIS L3 hierarchy, . and for all other prefixes in L1 and L2 LSPs. (default) . Down (1)-used when an L1/L2 router advertises L2 prefixes in L1 LSPs. . The prefixes are being advertised from a higher level (L2) down to a lower level (L1).

        Returns: Union[Literal["down"], Literal["up"]]
        """
        return self._get_property("redistribution_type")

    @redistribution_type.setter
    def redistribution_type(self, value):
        """redistribution_type setter

        Defines the Up/Down (Redistribution) bit defined for TLVs 128 and 130 by RFC 2966. It is used for domain-wide advertisement of prefix information. . Up (0)-used when a prefix is initially advertised within the ISIS L3 hierarchy, . and for all other prefixes in L1 and L2 LSPs. (default) . Down (1)-used when an L1/L2 router advertises L2 prefixes in L1 LSPs. . The prefixes are being advertised from a higher level (L2) down to a lower level (L1).

        value: Union[Literal["down"], Literal["up"]]
        """
        self._set_property("redistribution_type", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)

    @property
    def prefix_attr_enabled(self):
        # type: () -> bool
        """prefix_attr_enabled getter

        Specifies whether the sub-TLV for IPv4/IPv6 Extended Reachability Attribute Flags. will be advertised or not.

        Returns: bool
        """
        return self._get_property("prefix_attr_enabled")

    @prefix_attr_enabled.setter
    def prefix_attr_enabled(self, value):
        """prefix_attr_enabled setter

        Specifies whether the sub-TLV for IPv4/IPv6 Extended Reachability Attribute Flags. will be advertised or not.

        value: bool
        """
        self._set_property("prefix_attr_enabled", value)

    @property
    def x_flag(self):
        # type: () -> bool
        """x_flag getter

        External Prefix Flag (Bit 0)

        Returns: bool
        """
        return self._get_property("x_flag")

    @x_flag.setter
    def x_flag(self, value):
        """x_flag setter

        External Prefix Flag (Bit 0)

        value: bool
        """
        self._set_property("x_flag", value)

    @property
    def r_flag(self):
        # type: () -> bool
        """r_flag getter

        Re-advertisement Flag (Bit 1)

        Returns: bool
        """
        return self._get_property("r_flag")

    @r_flag.setter
    def r_flag(self, value):
        """r_flag setter

        Re-advertisement Flag (Bit 1)

        value: bool
        """
        self._set_property("r_flag", value)

    @property
    def n_flag(self):
        # type: () -> bool
        """n_flag getter

        Node Flag (Bit 2)

        Returns: bool
        """
        return self._get_property("n_flag")

    @n_flag.setter
    def n_flag(self, value):
        """n_flag setter

        Node Flag (Bit 2)

        value: bool
        """
        self._set_property("n_flag", value)


class V4RouteAddress(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "address": {
            "type": str,
            "format": "ipv4",
        },
        "prefix": {
            "type": int,
            "minimum": 0,
            "maximum": 32,
        },
        "count": {
            "type": int,
            "minimum": 1,
        },
        "step": {
            "type": int,
            "minimum": 1,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("address",)  # type: tuple(str)

    _DEFAULTS = {
        "prefix": 24,
        "count": 1,
        "step": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, address=None, prefix=24, count=1, step=1):
        super(V4RouteAddress, self).__init__()
        self._parent = parent
        self._set_property("address", address)
        self._set_property("prefix", prefix)
        self._set_property("count", count)
        self._set_property("step", step)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The starting address of the network.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The starting address of the network.

        value: str
        """
        self._set_property("address", value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The IPv4 network prefix length to be applied to the address.

        Returns: int
        """
        return self._get_property("prefix")

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The IPv4 network prefix length to be applied to the address.

        value: int
        """
        self._set_property("prefix", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        The total number of addresses in the range.

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        The total number of addresses in the range.

        value: int
        """
        self._set_property("count", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        Increments the network address prefixes within a route range where multiple routes are present. The value is incremented according to the Prefix Length and Step.

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        Increments the network address prefixes within a route range where multiple routes are present. The value is incremented according to the Prefix Length and Step.

        value: int
        """
        self._set_property("step", value)


class V4RouteAddressIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(V4RouteAddressIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[V4RouteAddress]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> V4RouteAddressIter
        return self._iter()

    def __next__(self):
        # type: () -> V4RouteAddress
        return self._next()

    def next(self):
        # type: () -> V4RouteAddress
        return self._next()

    def v4routeaddress(self, address=None, prefix=24, count=1, step=1):
        # type: (str,int,int,int) -> V4RouteAddressIter
        """Factory method that creates an instance of the V4RouteAddress class

        A container for IPv4 route addresses.

        Returns: V4RouteAddressIter
        """
        item = V4RouteAddress(
            parent=self._parent, address=address, prefix=prefix, count=count, step=step
        )
        self._add(item)
        return self

    def add(self, address=None, prefix=24, count=1, step=1):
        # type: (str,int,int,int) -> V4RouteAddress
        """Add method that creates and returns an instance of the V4RouteAddress class

        A container for IPv4 route addresses.

        Returns: V4RouteAddress
        """
        item = V4RouteAddress(
            parent=self._parent, address=address, prefix=prefix, count=count, step=step
        )
        self._add(item)
        return item


class IsisV4RouteRangeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisV4RouteRangeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisV4RouteRange]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisV4RouteRangeIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisV4RouteRange
        return self._next()

    def next(self):
        # type: () -> IsisV4RouteRange
        return self._next()

    def v4routerange(
        self,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        # type: (int,Union[Literal["external"], Literal["internal"]],Union[Literal["down"], Literal["up"]],str,bool,bool,bool,bool) -> IsisV4RouteRangeIter
        """Factory method that creates an instance of the IsisV4RouteRange class

        Emulated ISIS IPv4 routes.. Configuration for ISIS route ranges.. This contains the properties of ISIS Prefix attributes for the extended IPv4 and IPv6 reachability. https://www.rfc-editor.org/rfc/rfc7794.html

        Returns: IsisV4RouteRangeIter
        """
        item = IsisV4RouteRange(
            parent=self._parent,
            link_metric=link_metric,
            origin_type=origin_type,
            redistribution_type=redistribution_type,
            name=name,
            prefix_attr_enabled=prefix_attr_enabled,
            x_flag=x_flag,
            r_flag=r_flag,
            n_flag=n_flag,
        )
        self._add(item)
        return self

    def add(
        self,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        # type: (int,Union[Literal["external"], Literal["internal"]],Union[Literal["down"], Literal["up"]],str,bool,bool,bool,bool) -> IsisV4RouteRange
        """Add method that creates and returns an instance of the IsisV4RouteRange class

        Emulated ISIS IPv4 routes.. Configuration for ISIS route ranges.. This contains the properties of ISIS Prefix attributes for the extended IPv4 and IPv6 reachability. https://www.rfc-editor.org/rfc/rfc7794.html

        Returns: IsisV4RouteRange
        """
        item = IsisV4RouteRange(
            parent=self._parent,
            link_metric=link_metric,
            origin_type=origin_type,
            redistribution_type=redistribution_type,
            name=name,
            prefix_attr_enabled=prefix_attr_enabled,
            x_flag=x_flag,
            r_flag=r_flag,
            n_flag=n_flag,
        )
        self._add(item)
        return item


class IsisV6RouteRange(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "addresses": {"type": "V6RouteAddressIter"},
        "link_metric": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "origin_type": {
            "type": str,
            "enum": [
                "internal",
                "external",
            ],
        },
        "redistribution_type": {
            "type": str,
            "enum": [
                "up",
                "down",
            ],
        },
        "name": {"type": str},
        "prefix_attr_enabled": {"type": bool},
        "x_flag": {"type": bool},
        "r_flag": {"type": bool},
        "n_flag": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "link_metric": 0,
        "origin_type": "internal",
        "redistribution_type": "up",
        "prefix_attr_enabled": False,
        "x_flag": False,
        "r_flag": False,
        "n_flag": False,
    }  # type: Dict[str, Union(type)]

    INTERNAL = "internal"  # type: str
    EXTERNAL = "external"  # type: str

    UP = "up"  # type: str
    DOWN = "down"  # type: str

    def __init__(
        self,
        parent=None,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        super(IsisV6RouteRange, self).__init__()
        self._parent = parent
        self._set_property("link_metric", link_metric)
        self._set_property("origin_type", origin_type)
        self._set_property("redistribution_type", redistribution_type)
        self._set_property("name", name)
        self._set_property("prefix_attr_enabled", prefix_attr_enabled)
        self._set_property("x_flag", x_flag)
        self._set_property("r_flag", r_flag)
        self._set_property("n_flag", n_flag)

    @property
    def addresses(self):
        # type: () -> V6RouteAddressIter
        """addresses getter

        A list of group of IPv6 route addresses.

        Returns: V6RouteAddressIter
        """
        return self._get_property(
            "addresses", V6RouteAddressIter, self._parent, self._choice
        )

    @property
    def link_metric(self):
        # type: () -> int
        """link_metric getter

        The user-defined metric associated with this route range.

        Returns: int
        """
        return self._get_property("link_metric")

    @link_metric.setter
    def link_metric(self, value):
        """link_metric setter

        The user-defined metric associated with this route range.

        value: int
        """
        self._set_property("link_metric", value)

    @property
    def origin_type(self):
        # type: () -> Union[Literal["external"], Literal["internal"]]
        """origin_type getter

        The origin of the advertised route-internal or external to the ISIS area. Options include the following: . Internal-for intra-area routes, through Level 1 LSPs. . External-for inter-area routes redistributed within L1, through Level 1 LSPs.

        Returns: Union[Literal["external"], Literal["internal"]]
        """
        return self._get_property("origin_type")

    @origin_type.setter
    def origin_type(self, value):
        """origin_type setter

        The origin of the advertised route-internal or external to the ISIS area. Options include the following: . Internal-for intra-area routes, through Level 1 LSPs. . External-for inter-area routes redistributed within L1, through Level 1 LSPs.

        value: Union[Literal["external"], Literal["internal"]]
        """
        self._set_property("origin_type", value)

    @property
    def redistribution_type(self):
        # type: () -> Union[Literal["down"], Literal["up"]]
        """redistribution_type getter

        Defines the Up/Down (Redistribution) bit defined for TLVs 128 and 130 by RFC 2966. It is used for domain-wide advertisement of prefix information. . Up (0)-used when a prefix is initially advertised within the ISIS L3 hierarchy, . and for all other prefixes in L1 and L2 LSPs. (default) . Down (1)-used when an L1/L2 router advertises L2 prefixes in L1 LSPs. . The prefixes are being advertised from a higher level (L2) down to a lower level (L1).

        Returns: Union[Literal["down"], Literal["up"]]
        """
        return self._get_property("redistribution_type")

    @redistribution_type.setter
    def redistribution_type(self, value):
        """redistribution_type setter

        Defines the Up/Down (Redistribution) bit defined for TLVs 128 and 130 by RFC 2966. It is used for domain-wide advertisement of prefix information. . Up (0)-used when a prefix is initially advertised within the ISIS L3 hierarchy, . and for all other prefixes in L1 and L2 LSPs. (default) . Down (1)-used when an L1/L2 router advertises L2 prefixes in L1 LSPs. . The prefixes are being advertised from a higher level (L2) down to a lower level (L1).

        value: Union[Literal["down"], Literal["up"]]
        """
        self._set_property("redistribution_type", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)

    @property
    def prefix_attr_enabled(self):
        # type: () -> bool
        """prefix_attr_enabled getter

        Specifies whether the sub-TLV for IPv4/IPv6 Extended Reachability Attribute Flags. will be advertised or not.

        Returns: bool
        """
        return self._get_property("prefix_attr_enabled")

    @prefix_attr_enabled.setter
    def prefix_attr_enabled(self, value):
        """prefix_attr_enabled setter

        Specifies whether the sub-TLV for IPv4/IPv6 Extended Reachability Attribute Flags. will be advertised or not.

        value: bool
        """
        self._set_property("prefix_attr_enabled", value)

    @property
    def x_flag(self):
        # type: () -> bool
        """x_flag getter

        External Prefix Flag (Bit 0)

        Returns: bool
        """
        return self._get_property("x_flag")

    @x_flag.setter
    def x_flag(self, value):
        """x_flag setter

        External Prefix Flag (Bit 0)

        value: bool
        """
        self._set_property("x_flag", value)

    @property
    def r_flag(self):
        # type: () -> bool
        """r_flag getter

        Re-advertisement Flag (Bit 1)

        Returns: bool
        """
        return self._get_property("r_flag")

    @r_flag.setter
    def r_flag(self, value):
        """r_flag setter

        Re-advertisement Flag (Bit 1)

        value: bool
        """
        self._set_property("r_flag", value)

    @property
    def n_flag(self):
        # type: () -> bool
        """n_flag getter

        Node Flag (Bit 2)

        Returns: bool
        """
        return self._get_property("n_flag")

    @n_flag.setter
    def n_flag(self, value):
        """n_flag setter

        Node Flag (Bit 2)

        value: bool
        """
        self._set_property("n_flag", value)


class V6RouteAddress(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "address": {
            "type": str,
            "format": "ipv6",
        },
        "prefix": {
            "type": int,
            "minimum": 0,
            "maximum": 128,
        },
        "count": {
            "type": int,
            "minimum": 1,
        },
        "step": {
            "type": int,
            "minimum": 1,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("address",)  # type: tuple(str)

    _DEFAULTS = {
        "prefix": 64,
        "count": 1,
        "step": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, address=None, prefix=64, count=1, step=1):
        super(V6RouteAddress, self).__init__()
        self._parent = parent
        self._set_property("address", address)
        self._set_property("prefix", prefix)
        self._set_property("count", count)
        self._set_property("step", step)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The starting address of the network.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The starting address of the network.

        value: str
        """
        self._set_property("address", value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The IPv6 network prefix length to be applied to the address.

        Returns: int
        """
        return self._get_property("prefix")

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The IPv6 network prefix length to be applied to the address.

        value: int
        """
        self._set_property("prefix", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        The total number of addresses in the range.

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        The total number of addresses in the range.

        value: int
        """
        self._set_property("count", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        Increments the network address prefixes within a route range where multiple routes are present. The value is incremented according to the Prefix Length and Step.

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        Increments the network address prefixes within a route range where multiple routes are present. The value is incremented according to the Prefix Length and Step.

        value: int
        """
        self._set_property("step", value)


class V6RouteAddressIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(V6RouteAddressIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[V6RouteAddress]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> V6RouteAddressIter
        return self._iter()

    def __next__(self):
        # type: () -> V6RouteAddress
        return self._next()

    def next(self):
        # type: () -> V6RouteAddress
        return self._next()

    def v6routeaddress(self, address=None, prefix=64, count=1, step=1):
        # type: (str,int,int,int) -> V6RouteAddressIter
        """Factory method that creates an instance of the V6RouteAddress class

        A container for IPv6 route addresses.

        Returns: V6RouteAddressIter
        """
        item = V6RouteAddress(
            parent=self._parent, address=address, prefix=prefix, count=count, step=step
        )
        self._add(item)
        return self

    def add(self, address=None, prefix=64, count=1, step=1):
        # type: (str,int,int,int) -> V6RouteAddress
        """Add method that creates and returns an instance of the V6RouteAddress class

        A container for IPv6 route addresses.

        Returns: V6RouteAddress
        """
        item = V6RouteAddress(
            parent=self._parent, address=address, prefix=prefix, count=count, step=step
        )
        self._add(item)
        return item


class IsisV6RouteRangeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisV6RouteRangeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisV6RouteRange]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisV6RouteRangeIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisV6RouteRange
        return self._next()

    def next(self):
        # type: () -> IsisV6RouteRange
        return self._next()

    def v6routerange(
        self,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        # type: (int,Union[Literal["external"], Literal["internal"]],Union[Literal["down"], Literal["up"]],str,bool,bool,bool,bool) -> IsisV6RouteRangeIter
        """Factory method that creates an instance of the IsisV6RouteRange class

        Emulated ISIS IPv6 routes.. Configuration for ISIS route ranges.. This contains the properties of ISIS Prefix attributes for the extended IPv4 and IPv6 reachability. https://www.rfc-editor.org/rfc/rfc7794.html

        Returns: IsisV6RouteRangeIter
        """
        item = IsisV6RouteRange(
            parent=self._parent,
            link_metric=link_metric,
            origin_type=origin_type,
            redistribution_type=redistribution_type,
            name=name,
            prefix_attr_enabled=prefix_attr_enabled,
            x_flag=x_flag,
            r_flag=r_flag,
            n_flag=n_flag,
        )
        self._add(item)
        return self

    def add(
        self,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        # type: (int,Union[Literal["external"], Literal["internal"]],Union[Literal["down"], Literal["up"]],str,bool,bool,bool,bool) -> IsisV6RouteRange
        """Add method that creates and returns an instance of the IsisV6RouteRange class

        Emulated ISIS IPv6 routes.. Configuration for ISIS route ranges.. This contains the properties of ISIS Prefix attributes for the extended IPv4 and IPv6 reachability. https://www.rfc-editor.org/rfc/rfc7794.html

        Returns: IsisV6RouteRange
        """
        item = IsisV6RouteRange(
            parent=self._parent,
            link_metric=link_metric,
            origin_type=origin_type,
            redistribution_type=redistribution_type,
            name=name,
            prefix_attr_enabled=prefix_attr_enabled,
            x_flag=x_flag,
            r_flag=r_flag,
            n_flag=n_flag,
        )
        self._add(item)
        return item


class DeviceBgpRouter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "router_id": {
            "type": str,
            "format": "ipv4",
        },
        "ipv4_interfaces": {"type": "BgpV4InterfaceIter"},
        "ipv6_interfaces": {"type": "BgpV6InterfaceIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("router_id",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, router_id=None):
        super(DeviceBgpRouter, self).__init__()
        self._parent = parent
        self._set_property("router_id", router_id)

    @property
    def router_id(self):
        # type: () -> str
        """router_id getter

        The BGP router ID is a unique identifier used by BGP. It is a 32-bit value that is often represented by an IPv4 address.

        Returns: str
        """
        return self._get_property("router_id")

    @router_id.setter
    def router_id(self, value):
        """router_id setter

        The BGP router ID is a unique identifier used by BGP. It is a 32-bit value that is often represented by an IPv4 address.

        value: str
        """
        self._set_property("router_id", value)

    @property
    def ipv4_interfaces(self):
        # type: () -> BgpV4InterfaceIter
        """ipv4_interfaces getter

        This contains an array of references to IPv4 interfaces, each of which will have list of peers to different destinations.

        Returns: BgpV4InterfaceIter
        """
        return self._get_property(
            "ipv4_interfaces", BgpV4InterfaceIter, self._parent, self._choice
        )

    @property
    def ipv6_interfaces(self):
        # type: () -> BgpV6InterfaceIter
        """ipv6_interfaces getter

        This contains an array of references to IPv6 interfaces, each of which will have list of peers to different destinations.

        Returns: BgpV6InterfaceIter
        """
        return self._get_property(
            "ipv6_interfaces", BgpV6InterfaceIter, self._parent, self._choice
        )


class BgpV4Interface(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ipv4_name": {"type": str},
        "peers": {"type": "BgpV4PeerIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv4_name",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, ipv4_name=None):
        super(BgpV4Interface, self).__init__()
        self._parent = parent
        self._set_property("ipv4_name", ipv4_name)

    @property
    def ipv4_name(self):
        # type: () -> str
        """ipv4_name getter

        The unique name of the IPv4 or Loopback IPv4 interface used as the source IP for this list of BGP peers.. . x-constraint:. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv4Loopback/properties/name.

        Returns: str
        """
        return self._get_property("ipv4_name")

    @ipv4_name.setter
    def ipv4_name(self, value):
        """ipv4_name setter

        The unique name of the IPv4 or Loopback IPv4 interface used as the source IP for this list of BGP peers.. . x-constraint:. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv4Loopback/properties/name.

        value: str
        """
        self._set_property("ipv4_name", value)

    @property
    def peers(self):
        # type: () -> BgpV4PeerIter
        """peers getter

        This contains the list of BGPv4 peers configured on this interface.

        Returns: BgpV4PeerIter
        """
        return self._get_property("peers", BgpV4PeerIter, self._parent, self._choice)


class BgpV4Peer(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "peer_address": {
            "type": str,
            "format": "ipv4",
        },
        "evpn_ethernet_segments": {"type": "BgpV4EthernetSegmentIter"},
        "as_type": {
            "type": str,
            "enum": [
                "ibgp",
                "ebgp",
            ],
        },
        "as_number": {"type": int},
        "as_number_width": {
            "type": str,
            "enum": [
                "two",
                "four",
            ],
        },
        "advanced": {"type": "BgpAdvanced"},
        "capability": {"type": "BgpCapability"},
        "learned_information_filter": {"type": "BgpLearnedInformationFilter"},
        "v4_routes": {"type": "BgpV4RouteRangeIter"},
        "v6_routes": {"type": "BgpV6RouteRangeIter"},
        "v4_srte_policies": {"type": "BgpSrteV4PolicyIter"},
        "v6_srte_policies": {"type": "BgpSrteV6PolicyIter"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("peer_address", "as_type", "as_number", "name")  # type: tuple(str)

    _DEFAULTS = {
        "as_number_width": "four",
    }  # type: Dict[str, Union(type)]

    IBGP = "ibgp"  # type: str
    EBGP = "ebgp"  # type: str

    TWO = "two"  # type: str
    FOUR = "four"  # type: str

    def __init__(
        self,
        parent=None,
        peer_address=None,
        as_type=None,
        as_number=None,
        as_number_width="four",
        name=None,
    ):
        super(BgpV4Peer, self).__init__()
        self._parent = parent
        self._set_property("peer_address", peer_address)
        self._set_property("as_type", as_type)
        self._set_property("as_number", as_number)
        self._set_property("as_number_width", as_number_width)
        self._set_property("name", name)

    @property
    def peer_address(self):
        # type: () -> str
        """peer_address getter

        IPv4 address of the BGP peer for the session.

        Returns: str
        """
        return self._get_property("peer_address")

    @peer_address.setter
    def peer_address(self, value):
        """peer_address setter

        IPv4 address of the BGP peer for the session.

        value: str
        """
        self._set_property("peer_address", value)

    @property
    def evpn_ethernet_segments(self):
        # type: () -> BgpV4EthernetSegmentIter
        """evpn_ethernet_segments getter

        This contains the list of Ethernet Virtual Private Network (EVPN) Ethernet Segments (ES) Per BGP Peer for IPv4 Address Family Identifier (AFI).. . Each Ethernet Segment contains a list of EVPN Instances (EVIs) . . Each EVI contains a list of Broadcast Domains. . Each Broadcast Domain contains a list of MAC/IP Ranges. . . <Ethernet Segment, EVI, Broadcast Domain> is responsible for advertising Ethernet Auto-discovery Route Per EVI (Type 1).. . <Ethernet Segment, EVI> is responsible for advertising Ethernet Auto-discovery Route Per Ethernet Segment (Type 1).. . <Ethernet Segment, EVI, Broadcast Domain, MAC/IP> is responsible for advertising MAC/IP Advertisement Route (Type 2).. . <Ethernet Segment, EVI, Broadcast Domain> is responsible for advertising Inclusive Multicast Ethernet Tag Route (Type 3).. . Ethernet Segment is responsible for advertising Ethernet Segment Route (Type 4).

        Returns: BgpV4EthernetSegmentIter
        """
        return self._get_property(
            "evpn_ethernet_segments",
            BgpV4EthernetSegmentIter,
            self._parent,
            self._choice,
        )

    @property
    def as_type(self):
        # type: () -> Union[Literal["ebgp"], Literal["ibgp"]]
        """as_type getter

        The type of BGP autonomous system. External BGP is used for BGP links between two or more autonomous systems (ebgp). Internal BGP is used within a single autonomous system (ibgp). BGP property defaults are aligned with this object defined as an internal BGP peer. If the as_type is specified as 'ebgp' then other properties will need to be specified as per an external BGP peer. Specifically, for 'ebgp', 'as_set_mode' attribute in 'as_path' field in any Route Range should be changed from default value 'do_not_include_local_as' to any other value.

        Returns: Union[Literal["ebgp"], Literal["ibgp"]]
        """
        return self._get_property("as_type")

    @as_type.setter
    def as_type(self, value):
        """as_type setter

        The type of BGP autonomous system. External BGP is used for BGP links between two or more autonomous systems (ebgp). Internal BGP is used within a single autonomous system (ibgp). BGP property defaults are aligned with this object defined as an internal BGP peer. If the as_type is specified as 'ebgp' then other properties will need to be specified as per an external BGP peer. Specifically, for 'ebgp', 'as_set_mode' attribute in 'as_path' field in any Route Range should be changed from default value 'do_not_include_local_as' to any other value.

        value: Union[Literal["ebgp"], Literal["ibgp"]]
        """
        self._set_property("as_type", value)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        Autonomous System Number (AS number or ASN)

        Returns: int
        """
        return self._get_property("as_number")

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        Autonomous System Number (AS number or ASN)

        value: int
        """
        self._set_property("as_number", value)

    @property
    def as_number_width(self):
        # type: () -> Union[Literal["four"], Literal["two"]]
        """as_number_width getter

        The width in bytes of the as_number values. Any as_number values that exceeds the width MUST result in an error.

        Returns: Union[Literal["four"], Literal["two"]]
        """
        return self._get_property("as_number_width")

    @as_number_width.setter
    def as_number_width(self, value):
        """as_number_width setter

        The width in bytes of the as_number values. Any as_number values that exceeds the width MUST result in an error.

        value: Union[Literal["four"], Literal["two"]]
        """
        self._set_property("as_number_width", value)

    @property
    def advanced(self):
        # type: () -> BgpAdvanced
        """advanced getter

        Status: under-review. Configuration for BGP advanced settings.Status: under-review. Configuration for BGP advanced settings.

        Returns: BgpAdvanced
        """
        return self._get_property("advanced", BgpAdvanced)

    @property
    def capability(self):
        # type: () -> BgpCapability
        """capability getter

        Status: under-review. Configuration for BGP capability settings.Status: under-review. Configuration for BGP capability settings.

        Returns: BgpCapability
        """
        return self._get_property("capability", BgpCapability)

    @property
    def learned_information_filter(self):
        # type: () -> BgpLearnedInformationFilter
        """learned_information_filter getter

        Status: under-review. Configuration for controlling storage of BGP learned information recieved from the peer.Status: under-review. Configuration for controlling storage of BGP learned information recieved from the peer.

        Returns: BgpLearnedInformationFilter
        """
        return self._get_property(
            "learned_information_filter", BgpLearnedInformationFilter
        )

    @property
    def v4_routes(self):
        # type: () -> BgpV4RouteRangeIter
        """v4_routes getter

        Emulated BGPv4 route ranges.

        Returns: BgpV4RouteRangeIter
        """
        return self._get_property(
            "v4_routes", BgpV4RouteRangeIter, self._parent, self._choice
        )

    @property
    def v6_routes(self):
        # type: () -> BgpV6RouteRangeIter
        """v6_routes getter

        Emulated BGPv6 route ranges.

        Returns: BgpV6RouteRangeIter
        """
        return self._get_property(
            "v6_routes", BgpV6RouteRangeIter, self._parent, self._choice
        )

    @property
    def v4_srte_policies(self):
        # type: () -> BgpSrteV4PolicyIter
        """v4_srte_policies getter

        Segment Routing Traffic Engineering (SR TE) Policies for IPv4 Address Family Identifier (AFI).

        Returns: BgpSrteV4PolicyIter
        """
        return self._get_property(
            "v4_srte_policies", BgpSrteV4PolicyIter, self._parent, self._choice
        )

    @property
    def v6_srte_policies(self):
        # type: () -> BgpSrteV6PolicyIter
        """v6_srte_policies getter

        Segment Routing Traffic Engineering (SR TE) Policies for IPv6 Address Family Identifier (AFI).

        Returns: BgpSrteV6PolicyIter
        """
        return self._get_property(
            "v6_srte_policies", BgpSrteV6PolicyIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class BgpV4EthernetSegment(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "df_election": {"type": "BgpEthernetSegmentDfElection"},
        "evis": {"type": "BgpV4EvpnEvisIter"},
        "esi": {
            "type": str,
            "format": "hex",
        },
        "active_mode": {
            "type": str,
            "enum": [
                "single_active",
                "all_active",
            ],
        },
        "esi_label": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "esi": "00000000000000000000",
        "active_mode": "all_active",
        "esi_label": 0,
    }  # type: Dict[str, Union(type)]

    SINGLE_ACTIVE = "single_active"  # type: str
    ALL_ACTIVE = "all_active"  # type: str

    def __init__(
        self,
        parent=None,
        esi="00000000000000000000",
        active_mode="all_active",
        esi_label=0,
    ):
        super(BgpV4EthernetSegment, self).__init__()
        self._parent = parent
        self._set_property("esi", esi)
        self._set_property("active_mode", active_mode)
        self._set_property("esi_label", esi_label)

    @property
    def df_election(self):
        # type: () -> BgpEthernetSegmentDfElection
        """df_election getter

        Configuration for Designated Forwarder (DF) election among the Provider Edge (PE) routers on the same Ethernet Segment.Configuration for Designated Forwarder (DF) election among the Provider Edge (PE) routers on the same Ethernet Segment.Designated Forwarder (DF) election configuration.

        Returns: BgpEthernetSegmentDfElection
        """
        return self._get_property("df_election", BgpEthernetSegmentDfElection)

    @property
    def evis(self):
        # type: () -> BgpV4EvpnEvisIter
        """evis getter

        This contains the list of EVIs.

        Returns: BgpV4EvpnEvisIter
        """
        return self._get_property("evis", BgpV4EvpnEvisIter, self._parent, self._choice)

    @property
    def esi(self):
        # type: () -> str
        """esi getter

        10-octet Ethernet Segment Identifier (ESI) Example - For multi-home scenario nonZero ESI is '10000000000000000000' .

        Returns: str
        """
        return self._get_property("esi")

    @esi.setter
    def esi(self, value):
        """esi setter

        10-octet Ethernet Segment Identifier (ESI) Example - For multi-home scenario nonZero ESI is '10000000000000000000' .

        value: str
        """
        self._set_property("esi", value)

    @property
    def active_mode(self):
        # type: () -> Union[Literal["all_active"], Literal["single_active"]]
        """active_mode getter

        Single Active or All Active mode Redundancy mode selection for Multi-home.

        Returns: Union[Literal["all_active"], Literal["single_active"]]
        """
        return self._get_property("active_mode")

    @active_mode.setter
    def active_mode(self, value):
        """active_mode setter

        Single Active or All Active mode Redundancy mode selection for Multi-home.

        value: Union[Literal["all_active"], Literal["single_active"]]
        """
        self._set_property("active_mode", value)

    @property
    def esi_label(self):
        # type: () -> int
        """esi_label getter

        The label value to be advertised as ESI Label in ESI Label Extended Community. This is included in Ethernet Auto-discovery per ES Routes advertised by a router.

        Returns: int
        """
        return self._get_property("esi_label")

    @esi_label.setter
    def esi_label(self, value):
        """esi_label setter

        The label value to be advertised as ESI Label in ESI Label Extended Community. This is included in Ethernet Auto-discovery per ES Routes advertised by a router.

        value: int
        """
        self._set_property("esi_label", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Optional Extended Community settings.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.Optional AS PATH settings.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)


class BgpEthernetSegmentDfElection(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "election_timer": {
            "type": int,
            "minimum": 1,
            "maximum": 300,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "election_timer": 3,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, election_timer=3):
        super(BgpEthernetSegmentDfElection, self).__init__()
        self._parent = parent
        self._set_property("election_timer", election_timer)

    @property
    def election_timer(self):
        # type: () -> int
        """election_timer getter

        The DF election timer in seconds.

        Returns: int
        """
        return self._get_property("election_timer")

    @election_timer.setter
    def election_timer(self, value):
        """election_timer setter

        The DF election timer in seconds.

        value: int
        """
        self._set_property("election_timer", value)


class BgpV4EvpnEvis(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "evi_vxlan",
            ],
        },
        "evi_vxlan": {"type": "BgpV4EviVxlan"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "evi_vxlan",
    }  # type: Dict[str, Union(type)]

    EVI_VXLAN = "evi_vxlan"  # type: str

    def __init__(self, parent=None, choice=None):
        super(BgpV4EvpnEvis, self).__init__()
        self._parent = parent
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def evi_vxlan(self):
        # type: () -> BgpV4EviVxlan
        """Factory property that returns an instance of the BgpV4EviVxlan class

        Configuration for BGP EVPN EVI. Advertises following routes - . . Type 3 - Inclusive Multicast Ethernet Tag Route. . Type 1 - Ethernet Auto-discovery Route (Per EVI). . Type 1 - Ethernet Auto-discovery Route (Per ES)

        Returns: BgpV4EviVxlan
        """
        return self._get_property("evi_vxlan", BgpV4EviVxlan, self, "evi_vxlan")

    @property
    def choice(self):
        # type: () -> Union[Literal["evi_vxlan"]]
        """choice getter

        TBD

        Returns: Union[Literal["evi_vxlan"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["evi_vxlan"]]
        """
        self._set_property("choice", value)


class BgpV4EviVxlan(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "broadcast_domains": {"type": "BgpV4EviVxlanBroadcastDomainIter"},
        "replication_type": {
            "type": str,
            "enum": [
                "ingress_replication",
            ],
        },
        "pmsi_label": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "ad_label": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "route_distinguisher": {"type": "BgpRouteDistinguisher"},
        "route_target_export": {"type": "BgpRouteTargetIter"},
        "route_target_import": {"type": "BgpRouteTargetIter"},
        "l3_route_target_export": {"type": "BgpRouteTargetIter"},
        "l3_route_target_import": {"type": "BgpRouteTargetIter"},
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "replication_type": "ingress_replication",
        "pmsi_label": 16,
        "ad_label": 0,
    }  # type: Dict[str, Union(type)]

    INGRESS_REPLICATION = "ingress_replication"  # type: str

    def __init__(
        self,
        parent=None,
        replication_type="ingress_replication",
        pmsi_label=16,
        ad_label=0,
    ):
        super(BgpV4EviVxlan, self).__init__()
        self._parent = parent
        self._set_property("replication_type", replication_type)
        self._set_property("pmsi_label", pmsi_label)
        self._set_property("ad_label", ad_label)

    @property
    def broadcast_domains(self):
        # type: () -> BgpV4EviVxlanBroadcastDomainIter
        """broadcast_domains getter

        This contains the list of Broadcast Domains to be configured per EVI.

        Returns: BgpV4EviVxlanBroadcastDomainIter
        """
        return self._get_property(
            "broadcast_domains",
            BgpV4EviVxlanBroadcastDomainIter,
            self._parent,
            self._choice,
        )

    @property
    def replication_type(self):
        # type: () -> Union[Literal["ingress_replication"]]
        """replication_type getter

        This model only supports Ingress Replication

        Returns: Union[Literal["ingress_replication"]]
        """
        return self._get_property("replication_type")

    @replication_type.setter
    def replication_type(self, value):
        """replication_type setter

        This model only supports Ingress Replication

        value: Union[Literal["ingress_replication"]]
        """
        self._set_property("replication_type", value)

    @property
    def pmsi_label(self):
        # type: () -> int
        """pmsi_label getter

        Downstream assigned VNI to be carried as Part of P-Multicast Service Interface Tunnel attribute (PMSI Tunnel Attribute) in Type 3 Inclusive Multicast Ethernet Tag Route.

        Returns: int
        """
        return self._get_property("pmsi_label")

    @pmsi_label.setter
    def pmsi_label(self, value):
        """pmsi_label setter

        Downstream assigned VNI to be carried as Part of P-Multicast Service Interface Tunnel attribute (PMSI Tunnel Attribute) in Type 3 Inclusive Multicast Ethernet Tag Route.

        value: int
        """
        self._set_property("pmsi_label", value)

    @property
    def ad_label(self):
        # type: () -> int
        """ad_label getter

        The Auto-discovery Route label (AD label) value, which gets advertised in the Ethernet Auto-discovery Route per <EVI, ESI>

        Returns: int
        """
        return self._get_property("ad_label")

    @ad_label.setter
    def ad_label(self, value):
        """ad_label setter

        The Auto-discovery Route label (AD label) value, which gets advertised in the Ethernet Auto-discovery Route per <EVI, ESI>

        value: int
        """
        self._set_property("ad_label", value)

    @property
    def route_distinguisher(self):
        # type: () -> BgpRouteDistinguisher
        """route_distinguisher getter

        BGP Route Distinguisher.BGP Route Distinguisher.BGP Route Distinguisher.Colon separated Extended Community value of 6 Bytes - "AS number: Value" identifying an EVI. Example - for the as_2octet "60005:100".

        Returns: BgpRouteDistinguisher
        """
        return self._get_property("route_distinguisher", BgpRouteDistinguisher)

    @property
    def route_target_export(self):
        # type: () -> BgpRouteTargetIter
        """route_target_export getter

        List of Layer 2 Virtual Network Identifier (L2VNI) export targets associated with this EVI.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "route_target_export", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def route_target_import(self):
        # type: () -> BgpRouteTargetIter
        """route_target_import getter

        List of L2VNI import targets associated with this EVI.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "route_target_import", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def l3_route_target_export(self):
        # type: () -> BgpRouteTargetIter
        """l3_route_target_export getter

        List of Layer 3 Virtual Network Identifier (L3VNI) Export Route Targets.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "l3_route_target_export", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def l3_route_target_import(self):
        # type: () -> BgpRouteTargetIter
        """l3_route_target_import getter

        List of L3VNI Import Route Targets.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "l3_route_target_import", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Optional Extended Community settings.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.Optional AS PATH settings.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)


class BgpV4EviVxlanBroadcastDomain(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "cmac_ip_range": {"type": "BgpCMacIpRangeIter"},
        "ethernet_tag_id": {
            "type": int,
            "minimum": 0,
            "maximum": 4294967295,
        },
        "vlan_aware_service": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "ethernet_tag_id": 0,
        "vlan_aware_service": False,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, ethernet_tag_id=0, vlan_aware_service=False):
        super(BgpV4EviVxlanBroadcastDomain, self).__init__()
        self._parent = parent
        self._set_property("ethernet_tag_id", ethernet_tag_id)
        self._set_property("vlan_aware_service", vlan_aware_service)

    @property
    def cmac_ip_range(self):
        # type: () -> BgpCMacIpRangeIter
        """cmac_ip_range getter

        This contains the list of Customer MAC/IP Ranges to be configured per Broadcast Domain. . . Advertises following route - . Type 2 - MAC/IP Advertisement Route.

        Returns: BgpCMacIpRangeIter
        """
        return self._get_property(
            "cmac_ip_range", BgpCMacIpRangeIter, self._parent, self._choice
        )

    @property
    def ethernet_tag_id(self):
        # type: () -> int
        """ethernet_tag_id getter

        The Ethernet Tag ID of the Broadcast Domain.

        Returns: int
        """
        return self._get_property("ethernet_tag_id")

    @ethernet_tag_id.setter
    def ethernet_tag_id(self, value):
        """ethernet_tag_id setter

        The Ethernet Tag ID of the Broadcast Domain.

        value: int
        """
        self._set_property("ethernet_tag_id", value)

    @property
    def vlan_aware_service(self):
        # type: () -> bool
        """vlan_aware_service getter

        VLAN-Aware service to be enabled or disabled.

        Returns: bool
        """
        return self._get_property("vlan_aware_service")

    @vlan_aware_service.setter
    def vlan_aware_service(self, value):
        """vlan_aware_service setter

        VLAN-Aware service to be enabled or disabled.

        value: bool
        """
        self._set_property("vlan_aware_service", value)


class BgpCMacIpRange(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "mac_addresses": {"type": "MACRouteAddress"},
        "l2vni": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "ipv4_addresses": {"type": "V4RouteAddress"},
        "ipv6_addresses": {"type": "V6RouteAddress"},
        "l3vni": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "include_default_gateway": {"type": bool},
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "l2vni": 0,
        "l3vni": 0,
        "include_default_gateway": False,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, l2vni=0, l3vni=0, include_default_gateway=False, name=None
    ):
        super(BgpCMacIpRange, self).__init__()
        self._parent = parent
        self._set_property("l2vni", l2vni)
        self._set_property("l3vni", l3vni)
        self._set_property("include_default_gateway", include_default_gateway)
        self._set_property("name", name)

    @property
    def mac_addresses(self):
        # type: () -> MACRouteAddress
        """mac_addresses getter

        A container for MAC route addresses.A container for MAC route addresses.Host MAC address range per Broadcast Domain.

        Returns: MACRouteAddress
        """
        return self._get_property("mac_addresses", MACRouteAddress)

    @property
    def l2vni(self):
        # type: () -> int
        """l2vni getter

        Layer 2 Virtual Network Identifier (L2VNI) to be advertised with MAC/IP Advertisement Route (Type 2)

        Returns: int
        """
        return self._get_property("l2vni")

    @l2vni.setter
    def l2vni(self, value):
        """l2vni setter

        Layer 2 Virtual Network Identifier (L2VNI) to be advertised with MAC/IP Advertisement Route (Type 2)

        value: int
        """
        self._set_property("l2vni", value)

    @property
    def ipv4_addresses(self):
        # type: () -> V4RouteAddress
        """ipv4_addresses getter

        A container for IPv4 route addresses.A container for IPv4 route addresses.Host IPv4 address range per Broadcast Domain.

        Returns: V4RouteAddress
        """
        return self._get_property("ipv4_addresses", V4RouteAddress)

    @property
    def ipv6_addresses(self):
        # type: () -> V6RouteAddress
        """ipv6_addresses getter

        A container for IPv6 route addresses.A container for IPv6 route addresses.Host IPv6 address range per Broadcast Domain.

        Returns: V6RouteAddress
        """
        return self._get_property("ipv6_addresses", V6RouteAddress)

    @property
    def l3vni(self):
        # type: () -> int
        """l3vni getter

        Layer 3 Virtual Network Identifier (L3VNI) to be advertised with MAC/IP Advertisement Route (Type 2).

        Returns: int
        """
        return self._get_property("l3vni")

    @l3vni.setter
    def l3vni(self, value):
        """l3vni setter

        Layer 3 Virtual Network Identifier (L3VNI) to be advertised with MAC/IP Advertisement Route (Type 2).

        value: int
        """
        self._set_property("l3vni", value)

    @property
    def include_default_gateway(self):
        # type: () -> bool
        """include_default_gateway getter

        Include default Gateway Extended Community in MAC/IP Advertisement Route (Type 2).

        Returns: bool
        """
        return self._get_property("include_default_gateway")

    @include_default_gateway.setter
    def include_default_gateway(self, value):
        """include_default_gateway setter

        Include default Gateway Extended Community in MAC/IP Advertisement Route (Type 2).

        value: bool
        """
        self._set_property("include_default_gateway", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Optional Extended Community settings.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.Optional AS PATH settings.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class MACRouteAddress(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "address": {
            "type": str,
            "format": "mac",
        },
        "prefix": {
            "type": int,
            "minimum": 0,
            "maximum": 48,
        },
        "count": {
            "type": int,
            "minimum": 1,
        },
        "step": {
            "type": int,
            "minimum": 1,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("address",)  # type: tuple(str)

    _DEFAULTS = {
        "prefix": 48,
        "count": 1,
        "step": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, address=None, prefix=48, count=1, step=1):
        super(MACRouteAddress, self).__init__()
        self._parent = parent
        self._set_property("address", address)
        self._set_property("prefix", prefix)
        self._set_property("count", count)
        self._set_property("step", step)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The starting address of the MAC Range.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The starting address of the MAC Range.

        value: str
        """
        self._set_property("address", value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The MAC prefix length to be applied to the address.

        Returns: int
        """
        return self._get_property("prefix")

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The MAC prefix length to be applied to the address.

        value: int
        """
        self._set_property("prefix", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        The total number of mac addresses in the range.

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        The total number of mac addresses in the range.

        value: int
        """
        self._set_property("count", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        Increments the mac address prefixes within a mac range where multiple routes are present. The value is incremented according to the mac prefix Length and Step.

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        Increments the mac address prefixes within a mac range where multiple routes are present. The value is incremented according to the mac prefix Length and Step.

        value: int
        """
        self._set_property("step", value)


class BgpRouteAdvanced(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "multi_exit_discriminator": {"type": int},
        "origin": {
            "type": str,
            "enum": [
                "igp",
                "egp",
            ],
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "origin": "igp",
    }  # type: Dict[str, Union(type)]

    IGP = "igp"  # type: str
    EGP = "egp"  # type: str

    def __init__(self, parent=None, multi_exit_discriminator=None, origin="igp"):
        super(BgpRouteAdvanced, self).__init__()
        self._parent = parent
        self._set_property("multi_exit_discriminator", multi_exit_discriminator)
        self._set_property("origin", origin)

    @property
    def multi_exit_discriminator(self):
        # type: () -> int
        """multi_exit_discriminator getter

        The multi exit discriminator (MED) value. A null value means the MED feature is not enabled.

        Returns: int
        """
        return self._get_property("multi_exit_discriminator")

    @multi_exit_discriminator.setter
    def multi_exit_discriminator(self, value):
        """multi_exit_discriminator setter

        The multi exit discriminator (MED) value. A null value means the MED feature is not enabled.

        value: int
        """
        self._set_property("multi_exit_discriminator", value)

    @property
    def origin(self):
        # type: () -> Union[Literal["egp"], Literal["igp"]]
        """origin getter

        The origin value. A null value means the origin feature is not enabled.

        Returns: Union[Literal["egp"], Literal["igp"]]
        """
        return self._get_property("origin")

    @origin.setter
    def origin(self, value):
        """origin setter

        The origin value. A null value means the origin feature is not enabled.

        value: Union[Literal["egp"], Literal["igp"]]
        """
        self._set_property("origin", value)


class BgpCommunity(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "type": {
            "type": str,
            "enum": [
                "manual_as_number",
                "no_export",
                "no_advertised",
                "no_export_subconfed",
                "llgr_stale",
                "no_llgr",
            ],
        },
        "as_number": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "as_custom": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "as_number": 0,
        "as_custom": 0,
    }  # type: Dict[str, Union(type)]

    MANUAL_AS_NUMBER = "manual_as_number"  # type: str
    NO_EXPORT = "no_export"  # type: str
    NO_ADVERTISED = "no_advertised"  # type: str
    NO_EXPORT_SUBCONFED = "no_export_subconfed"  # type: str
    LLGR_STALE = "llgr_stale"  # type: str
    NO_LLGR = "no_llgr"  # type: str

    def __init__(self, parent=None, type=None, as_number=0, as_custom=0):
        super(BgpCommunity, self).__init__()
        self._parent = parent
        self._set_property("type", type)
        self._set_property("as_number", as_number)
        self._set_property("as_custom", as_custom)

    @property
    def type(self):
        # type: () -> Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """type getter

        The type of community AS number.

        Returns: Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """
        return self._get_property("type")

    @type.setter
    def type(self, value):
        """type setter

        The type of community AS number.

        value: Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """
        self._set_property("type", value)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        First two octets of 32 bit community AS number.

        Returns: int
        """
        return self._get_property("as_number")

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        First two octets of 32 bit community AS number.

        value: int
        """
        self._set_property("as_number", value)

    @property
    def as_custom(self):
        # type: () -> int
        """as_custom getter

        Last two octets of the community AS number.

        Returns: int
        """
        return self._get_property("as_custom")

    @as_custom.setter
    def as_custom(self, value):
        """as_custom setter

        Last two octets of the community AS number.

        value: int
        """
        self._set_property("as_custom", value)


class BgpCommunityIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpCommunityIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpCommunity]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpCommunityIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpCommunity
        return self._next()

    def next(self):
        # type: () -> BgpCommunity
        return self._next()

    def community(self, type=None, as_number=0, as_custom=0):
        # type: (Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]],int,int) -> BgpCommunityIter
        """Factory method that creates an instance of the BgpCommunity class

        BGP communities provide additional capability for tagging routes and for modifying BGP routing policy on upstream and downstream routers. BGP community is a 32-bit number which is broken into 16-bit AS number and a 16-bit custom value.

        Returns: BgpCommunityIter
        """
        item = BgpCommunity(
            parent=self._parent, type=type, as_number=as_number, as_custom=as_custom
        )
        self._add(item)
        return self

    def add(self, type=None, as_number=0, as_custom=0):
        # type: (Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]],int,int) -> BgpCommunity
        """Add method that creates and returns an instance of the BgpCommunity class

        BGP communities provide additional capability for tagging routes and for modifying BGP routing policy on upstream and downstream routers. BGP community is a 32-bit number which is broken into 16-bit AS number and a 16-bit custom value.

        Returns: BgpCommunity
        """
        item = BgpCommunity(
            parent=self._parent, type=type, as_number=as_number, as_custom=as_custom
        )
        self._add(item)
        return item


class BgpExtCommunity(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "type": {
            "type": str,
            "enum": [
                "administrator_as_2octet",
                "administrator_ipv4_address",
                "administrator_as_4octet",
                "opaque",
                "evpn",
                "administrator_as_2octet_link_bandwidth",
            ],
        },
        "subtype": {
            "type": str,
            "enum": [
                "route_target",
                "origin",
                "extended_bandwidth",
                "color",
                "encapsulation",
                "mac_address",
            ],
        },
        "value": {
            "type": str,
            "format": "hex",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    ADMINISTRATOR_AS_2OCTET = "administrator_as_2octet"  # type: str
    ADMINISTRATOR_IPV4_ADDRESS = "administrator_ipv4_address"  # type: str
    ADMINISTRATOR_AS_4OCTET = "administrator_as_4octet"  # type: str
    OPAQUE = "opaque"  # type: str
    EVPN = "evpn"  # type: str
    ADMINISTRATOR_AS_2OCTET_LINK_BANDWIDTH = (
        "administrator_as_2octet_link_bandwidth"
    )  # type: str

    ROUTE_TARGET = "route_target"  # type: str
    ORIGIN = "origin"  # type: str
    EXTENDED_BANDWIDTH = "extended_bandwidth"  # type: str
    COLOR = "color"  # type: str
    ENCAPSULATION = "encapsulation"  # type: str
    MAC_ADDRESS = "mac_address"  # type: str

    def __init__(self, parent=None, type=None, subtype=None, value=None):
        super(BgpExtCommunity, self).__init__()
        self._parent = parent
        self._set_property("type", type)
        self._set_property("subtype", subtype)
        self._set_property("value", value)

    @property
    def type(self):
        # type: () -> Union[Literal["administrator_as_2octet"], Literal["administrator_as_2octet_link_bandwidth"], Literal["administrator_as_4octet"], Literal["administrator_ipv4_address"], Literal["evpn"], Literal["opaque"]]
        """type getter

        Extended Community Type field of 1 Byte.. - administrator_as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. - administrator_ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. - administrator_as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).. - opaque: Opaque Extended Community (RFC 7432).. - evpn: EVPN Extended Community (RFC 7153). . - administrator_as_2octet_link_bandwidth : Link Bandwidth Extended Community (RFC 7153).

        Returns: Union[Literal["administrator_as_2octet"], Literal["administrator_as_2octet_link_bandwidth"], Literal["administrator_as_4octet"], Literal["administrator_ipv4_address"], Literal["evpn"], Literal["opaque"]]
        """
        return self._get_property("type")

    @type.setter
    def type(self, value):
        """type setter

        Extended Community Type field of 1 Byte.. - administrator_as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. - administrator_ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. - administrator_as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).. - opaque: Opaque Extended Community (RFC 7432).. - evpn: EVPN Extended Community (RFC 7153). . - administrator_as_2octet_link_bandwidth : Link Bandwidth Extended Community (RFC 7153).

        value: Union[Literal["administrator_as_2octet"], Literal["administrator_as_2octet_link_bandwidth"], Literal["administrator_as_4octet"], Literal["administrator_ipv4_address"], Literal["evpn"], Literal["opaque"]]
        """
        self._set_property("type", value)

    @property
    def subtype(self):
        # type: () -> Union[Literal["color"], Literal["encapsulation"], Literal["extended_bandwidth"], Literal["mac_address"], Literal["origin"], Literal["route_target"]]
        """subtype getter

        Extended Community Sub Type field of 1 Byte.. - route_target: Route Target.. - origin: Origin.. - extended_bandwidth: Specifies the link bandwidth.. - color: Specifies the color value.. - encapsulation: Specifies the Encapsulation Extended Community.. - mac_address: Specifies the Extended community MAC address.

        Returns: Union[Literal["color"], Literal["encapsulation"], Literal["extended_bandwidth"], Literal["mac_address"], Literal["origin"], Literal["route_target"]]
        """
        return self._get_property("subtype")

    @subtype.setter
    def subtype(self, value):
        """subtype setter

        Extended Community Sub Type field of 1 Byte.. - route_target: Route Target.. - origin: Origin.. - extended_bandwidth: Specifies the link bandwidth.. - color: Specifies the color value.. - encapsulation: Specifies the Encapsulation Extended Community.. - mac_address: Specifies the Extended community MAC address.

        value: Union[Literal["color"], Literal["encapsulation"], Literal["extended_bandwidth"], Literal["mac_address"], Literal["origin"], Literal["route_target"]]
        """
        self._set_property("subtype", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        Extended Community value of 6 Bytes. Example - for the Opaque type and Color subtype value can be '0000000000c8' for the color value 200.

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        Extended Community value of 6 Bytes. Example - for the Opaque type and Color subtype value can be '0000000000c8' for the color value 200.

        value: str
        """
        self._set_property("value", value)


class BgpExtCommunityIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpExtCommunityIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpExtCommunity]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpExtCommunityIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpExtCommunity
        return self._next()

    def next(self):
        # type: () -> BgpExtCommunity
        return self._next()

    def extcommunity(self, type=None, subtype=None, value=None):
        # type: (Union[Literal["administrator_as_2octet"], Literal["administrator_as_2octet_link_bandwidth"], Literal["administrator_as_4octet"], Literal["administrator_ipv4_address"], Literal["evpn"], Literal["opaque"]],Union[Literal["color"], Literal["encapsulation"], Literal["extended_bandwidth"], Literal["mac_address"], Literal["origin"], Literal["route_target"]],str) -> BgpExtCommunityIter
        """Factory method that creates an instance of the BgpExtCommunity class

        The Extended Communities Attribute is a transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is also divided into two main parts. The first 2 Bytes of the community encode a type and sub-type fields and the last 6 Bytes carry a unique set of data in a format defined by the type and sub-type field. Extended communities provide a larger range for grouping or categorizing communities.

        Returns: BgpExtCommunityIter
        """
        item = BgpExtCommunity(
            parent=self._parent, type=type, subtype=subtype, value=value
        )
        self._add(item)
        return self

    def add(self, type=None, subtype=None, value=None):
        # type: (Union[Literal["administrator_as_2octet"], Literal["administrator_as_2octet_link_bandwidth"], Literal["administrator_as_4octet"], Literal["administrator_ipv4_address"], Literal["evpn"], Literal["opaque"]],Union[Literal["color"], Literal["encapsulation"], Literal["extended_bandwidth"], Literal["mac_address"], Literal["origin"], Literal["route_target"]],str) -> BgpExtCommunity
        """Add method that creates and returns an instance of the BgpExtCommunity class

        The Extended Communities Attribute is a transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is also divided into two main parts. The first 2 Bytes of the community encode a type and sub-type fields and the last 6 Bytes carry a unique set of data in a format defined by the type and sub-type field. Extended communities provide a larger range for grouping or categorizing communities.

        Returns: BgpExtCommunity
        """
        item = BgpExtCommunity(
            parent=self._parent, type=type, subtype=subtype, value=value
        )
        self._add(item)
        return item


class BgpAsPath(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "as_set_mode": {
            "type": str,
            "enum": [
                "do_not_include_local_as",
                "include_as_seq",
                "include_as_set",
                "include_as_confed_seq",
                "include_as_confed_set",
                "prepend_to_first_segment",
            ],
        },
        "segments": {"type": "BgpAsPathSegmentIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "as_set_mode": "do_not_include_local_as",
    }  # type: Dict[str, Union(type)]

    DO_NOT_INCLUDE_LOCAL_AS = "do_not_include_local_as"  # type: str
    INCLUDE_AS_SEQ = "include_as_seq"  # type: str
    INCLUDE_AS_SET = "include_as_set"  # type: str
    INCLUDE_AS_CONFED_SEQ = "include_as_confed_seq"  # type: str
    INCLUDE_AS_CONFED_SET = "include_as_confed_set"  # type: str
    PREPEND_TO_FIRST_SEGMENT = "prepend_to_first_segment"  # type: str

    def __init__(self, parent=None, as_set_mode="do_not_include_local_as"):
        super(BgpAsPath, self).__init__()
        self._parent = parent
        self._set_property("as_set_mode", as_set_mode)

    @property
    def as_set_mode(self):
        # type: () -> Union[Literal["do_not_include_local_as"], Literal["include_as_confed_seq"], Literal["include_as_confed_set"], Literal["include_as_seq"], Literal["include_as_set"], Literal["prepend_to_first_segment"]]
        """as_set_mode getter

        Defines how the Local AS should be included in the MP REACH NLRI. For iBGP sessions, "Do Not Include Local AS" must be chosen. For eBGP sessions, any choice other than "Do Not Include Local AS" can be chosen.

        Returns: Union[Literal["do_not_include_local_as"], Literal["include_as_confed_seq"], Literal["include_as_confed_set"], Literal["include_as_seq"], Literal["include_as_set"], Literal["prepend_to_first_segment"]]
        """
        return self._get_property("as_set_mode")

    @as_set_mode.setter
    def as_set_mode(self, value):
        """as_set_mode setter

        Defines how the Local AS should be included in the MP REACH NLRI. For iBGP sessions, "Do Not Include Local AS" must be chosen. For eBGP sessions, any choice other than "Do Not Include Local AS" can be chosen.

        value: Union[Literal["do_not_include_local_as"], Literal["include_as_confed_seq"], Literal["include_as_confed_set"], Literal["include_as_seq"], Literal["include_as_set"], Literal["prepend_to_first_segment"]]
        """
        self._set_property("as_set_mode", value)

    @property
    def segments(self):
        # type: () -> BgpAsPathSegmentIter
        """segments getter

        The additional AS path segments to be added in the NLRI. By default, an empty AS path is always included and the local AS is added to it as per the value of 'as_set_mode' attribute.

        Returns: BgpAsPathSegmentIter
        """
        return self._get_property(
            "segments", BgpAsPathSegmentIter, self._parent, self._choice
        )


class BgpAsPathSegment(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "type": {
            "type": str,
            "enum": [
                "as_seq",
                "as_set",
                "as_confed_seq",
                "as_confed_set",
            ],
        },
        "as_numbers": {
            "type": list,
            "itemtype": int,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "type": "as_seq",
    }  # type: Dict[str, Union(type)]

    AS_SEQ = "as_seq"  # type: str
    AS_SET = "as_set"  # type: str
    AS_CONFED_SEQ = "as_confed_seq"  # type: str
    AS_CONFED_SET = "as_confed_set"  # type: str

    def __init__(self, parent=None, type="as_seq", as_numbers=None):
        super(BgpAsPathSegment, self).__init__()
        self._parent = parent
        self._set_property("type", type)
        self._set_property("as_numbers", as_numbers)

    @property
    def type(self):
        # type: () -> Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """type getter

        AS sequence is the most common type of AS_PATH, it contains the list of ASNs starting with the most recent ASN being added read from left to right.. The other three AS_PATH types are used for Confederations - AS_SET is the type of AS_PATH attribute that summarizes routes using using the aggregate-address command, allowing AS_PATHs to be summarized in the update as well. - AS_CONFED_SEQ gives the list of ASNs in the path starting with the most recent ASN to be added reading left to right - AS_CONFED_SET will allow summarization of multiple AS PATHs to be sent in BGP Updates.

        Returns: Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """
        return self._get_property("type")

    @type.setter
    def type(self, value):
        """type setter

        AS sequence is the most common type of AS_PATH, it contains the list of ASNs starting with the most recent ASN being added read from left to right.. The other three AS_PATH types are used for Confederations - AS_SET is the type of AS_PATH attribute that summarizes routes using using the aggregate-address command, allowing AS_PATHs to be summarized in the update as well. - AS_CONFED_SEQ gives the list of ASNs in the path starting with the most recent ASN to be added reading left to right - AS_CONFED_SET will allow summarization of multiple AS PATHs to be sent in BGP Updates.

        value: Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """
        self._set_property("type", value)

    @property
    def as_numbers(self):
        # type: () -> List[int]
        """as_numbers getter

        The AS numbers in this AS path segment.

        Returns: List[int]
        """
        return self._get_property("as_numbers")

    @as_numbers.setter
    def as_numbers(self, value):
        """as_numbers setter

        The AS numbers in this AS path segment.

        value: List[int]
        """
        self._set_property("as_numbers", value)


class BgpAsPathSegmentIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpAsPathSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpAsPathSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpAsPathSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpAsPathSegment
        return self._next()

    def next(self):
        # type: () -> BgpAsPathSegment
        return self._next()

    def aspathsegment(self, type="as_seq", as_numbers=None):
        # type: (Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]],List[int]) -> BgpAsPathSegmentIter
        """Factory method that creates an instance of the BgpAsPathSegment class

        Configuration for a single BGP AS path segment

        Returns: BgpAsPathSegmentIter
        """
        item = BgpAsPathSegment(parent=self._parent, type=type, as_numbers=as_numbers)
        self._add(item)
        return self

    def add(self, type="as_seq", as_numbers=None):
        # type: (Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]],List[int]) -> BgpAsPathSegment
        """Add method that creates and returns an instance of the BgpAsPathSegment class

        Configuration for a single BGP AS path segment

        Returns: BgpAsPathSegment
        """
        item = BgpAsPathSegment(parent=self._parent, type=type, as_numbers=as_numbers)
        self._add(item)
        return item


class BgpCMacIpRangeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpCMacIpRangeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpCMacIpRange]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpCMacIpRangeIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpCMacIpRange
        return self._next()

    def next(self):
        # type: () -> BgpCMacIpRange
        return self._next()

    def cmaciprange(self, l2vni=0, l3vni=0, include_default_gateway=False, name=None):
        # type: (int,int,bool,str) -> BgpCMacIpRangeIter
        """Factory method that creates an instance of the BgpCMacIpRange class

        Configuration for MAC/IP Ranges per Broadcast Domain. . . Advertises following route -. . Type 2 - MAC/IP Advertisement Route.

        Returns: BgpCMacIpRangeIter
        """
        item = BgpCMacIpRange(
            parent=self._parent,
            l2vni=l2vni,
            l3vni=l3vni,
            include_default_gateway=include_default_gateway,
            name=name,
        )
        self._add(item)
        return self

    def add(self, l2vni=0, l3vni=0, include_default_gateway=False, name=None):
        # type: (int,int,bool,str) -> BgpCMacIpRange
        """Add method that creates and returns an instance of the BgpCMacIpRange class

        Configuration for MAC/IP Ranges per Broadcast Domain. . . Advertises following route -. . Type 2 - MAC/IP Advertisement Route.

        Returns: BgpCMacIpRange
        """
        item = BgpCMacIpRange(
            parent=self._parent,
            l2vni=l2vni,
            l3vni=l3vni,
            include_default_gateway=include_default_gateway,
            name=name,
        )
        self._add(item)
        return item


class BgpV4EviVxlanBroadcastDomainIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV4EviVxlanBroadcastDomainIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV4EviVxlanBroadcastDomain]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV4EviVxlanBroadcastDomainIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV4EviVxlanBroadcastDomain
        return self._next()

    def next(self):
        # type: () -> BgpV4EviVxlanBroadcastDomain
        return self._next()

    def broadcastdomain(self, ethernet_tag_id=0, vlan_aware_service=False):
        # type: (int,bool) -> BgpV4EviVxlanBroadcastDomainIter
        """Factory method that creates an instance of the BgpV4EviVxlanBroadcastDomain class

        Configuration for Broadcast Domains per EVI.

        Returns: BgpV4EviVxlanBroadcastDomainIter
        """
        item = BgpV4EviVxlanBroadcastDomain(
            parent=self._parent,
            ethernet_tag_id=ethernet_tag_id,
            vlan_aware_service=vlan_aware_service,
        )
        self._add(item)
        return self

    def add(self, ethernet_tag_id=0, vlan_aware_service=False):
        # type: (int,bool) -> BgpV4EviVxlanBroadcastDomain
        """Add method that creates and returns an instance of the BgpV4EviVxlanBroadcastDomain class

        Configuration for Broadcast Domains per EVI.

        Returns: BgpV4EviVxlanBroadcastDomain
        """
        item = BgpV4EviVxlanBroadcastDomain(
            parent=self._parent,
            ethernet_tag_id=ethernet_tag_id,
            vlan_aware_service=vlan_aware_service,
        )
        self._add(item)
        return item


class BgpRouteDistinguisher(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "rd_type": {
            "type": str,
            "enum": [
                "as_2octet",
                "ipv4_address",
                "as_4octet",
            ],
        },
        "auto_config_rd_ip_addr": {"type": bool},
        "rd_value": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "rd_type": "as_2octet",
        "auto_config_rd_ip_addr": False,
    }  # type: Dict[str, Union(type)]

    AS_2OCTET = "as_2octet"  # type: str
    IPV4_ADDRESS = "ipv4_address"  # type: str
    AS_4OCTET = "as_4octet"  # type: str

    def __init__(
        self,
        parent=None,
        rd_type="as_2octet",
        auto_config_rd_ip_addr=False,
        rd_value=None,
    ):
        super(BgpRouteDistinguisher, self).__init__()
        self._parent = parent
        self._set_property("rd_type", rd_type)
        self._set_property("auto_config_rd_ip_addr", auto_config_rd_ip_addr)
        self._set_property("rd_value", rd_value)

    @property
    def rd_type(self):
        # type: () -> Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """rd_type getter

        Route Distinguisher Type field of 2 Byte.. - as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. - ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. - as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).

        Returns: Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """
        return self._get_property("rd_type")

    @rd_type.setter
    def rd_type(self, value):
        """rd_type setter

        Route Distinguisher Type field of 2 Byte.. - as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. - ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. - as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).

        value: Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """
        self._set_property("rd_type", value)

    @property
    def auto_config_rd_ip_addr(self):
        # type: () -> bool
        """auto_config_rd_ip_addr getter

        Allow to automatically configure RD IP address from local ip.

        Returns: bool
        """
        return self._get_property("auto_config_rd_ip_addr")

    @auto_config_rd_ip_addr.setter
    def auto_config_rd_ip_addr(self, value):
        """auto_config_rd_ip_addr setter

        Allow to automatically configure RD IP address from local ip.

        value: bool
        """
        self._set_property("auto_config_rd_ip_addr", value)

    @property
    def rd_value(self):
        # type: () -> str
        """rd_value getter

        Colon separated Extended Community value of 6 Bytes - "AS number: Value". Example - for the as_2octet or as_4octet "60005:100", for ipv4_address "1.1.1.1:100"

        Returns: str
        """
        return self._get_property("rd_value")

    @rd_value.setter
    def rd_value(self, value):
        """rd_value setter

        Colon separated Extended Community value of 6 Bytes - "AS number: Value". Example - for the as_2octet or as_4octet "60005:100", for ipv4_address "1.1.1.1:100"

        value: str
        """
        self._set_property("rd_value", value)


class BgpRouteTarget(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "rt_type": {
            "type": str,
            "enum": [
                "as_2octet",
                "ipv4_address",
                "as_4octet",
            ],
        },
        "rt_value": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    AS_2OCTET = "as_2octet"  # type: str
    IPV4_ADDRESS = "ipv4_address"  # type: str
    AS_4OCTET = "as_4octet"  # type: str

    def __init__(self, parent=None, rt_type=None, rt_value=None):
        super(BgpRouteTarget, self).__init__()
        self._parent = parent
        self._set_property("rt_type", rt_type)
        self._set_property("rt_value", rt_value)

    @property
    def rt_type(self):
        # type: () -> Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """rt_type getter

        Extended Community Type field of 2 Byte.. - as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. - ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. - as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).

        Returns: Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """
        return self._get_property("rt_type")

    @rt_type.setter
    def rt_type(self, value):
        """rt_type setter

        Extended Community Type field of 2 Byte.. - as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. - ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. - as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).

        value: Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """
        self._set_property("rt_type", value)

    @property
    def rt_value(self):
        # type: () -> str
        """rt_value getter

        Colon separated Extended Community value of 6 Bytes - AS number: Assigned Number. Example - for the as_2octet or as_4octet "60005:100", for ipv4_address "1.1.1.1:100"

        Returns: str
        """
        return self._get_property("rt_value")

    @rt_value.setter
    def rt_value(self, value):
        """rt_value setter

        Colon separated Extended Community value of 6 Bytes - AS number: Assigned Number. Example - for the as_2octet or as_4octet "60005:100", for ipv4_address "1.1.1.1:100"

        value: str
        """
        self._set_property("rt_value", value)


class BgpRouteTargetIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpRouteTargetIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpRouteTarget]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpRouteTargetIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpRouteTarget
        return self._next()

    def next(self):
        # type: () -> BgpRouteTarget
        return self._next()

    def routetarget(self, rt_type=None, rt_value=None):
        # type: (Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]],str) -> BgpRouteTargetIter
        """Factory method that creates an instance of the BgpRouteTarget class

        BGP Route Target.

        Returns: BgpRouteTargetIter
        """
        item = BgpRouteTarget(parent=self._parent, rt_type=rt_type, rt_value=rt_value)
        self._add(item)
        return self

    def add(self, rt_type=None, rt_value=None):
        # type: (Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]],str) -> BgpRouteTarget
        """Add method that creates and returns an instance of the BgpRouteTarget class

        BGP Route Target.

        Returns: BgpRouteTarget
        """
        item = BgpRouteTarget(parent=self._parent, rt_type=rt_type, rt_value=rt_value)
        self._add(item)
        return item


class BgpV4EvpnEvisIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = True

    def __init__(self, parent=None, choice=None):
        super(BgpV4EvpnEvisIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV4EviVxlan, BgpV4EvpnEvis]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV4EvpnEvisIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV4EvpnEvis
        return self._next()

    def next(self):
        # type: () -> BgpV4EvpnEvis
        return self._next()

    def evpnevis(self):
        # type: () -> BgpV4EvpnEvisIter
        """Factory method that creates an instance of the BgpV4EvpnEvis class

        This contains a list of different flavors of EVPN. . For example EVPN over VXLAN or EVPN over MPLS etc to be configured per Ethernet segment. . Need to instantiate correct type of EVPN instance as per requirement.

        Returns: BgpV4EvpnEvisIter
        """
        item = BgpV4EvpnEvis(parent=self._parent, choice=self._choice)
        self._add(item)
        return self

    def add(self):
        # type: () -> BgpV4EvpnEvis
        """Add method that creates and returns an instance of the BgpV4EvpnEvis class

        This contains a list of different flavors of EVPN. . For example EVPN over VXLAN or EVPN over MPLS etc to be configured per Ethernet segment. . Need to instantiate correct type of EVPN instance as per requirement.

        Returns: BgpV4EvpnEvis
        """
        item = BgpV4EvpnEvis(parent=self._parent, choice=self._choice)
        self._add(item)
        return item

    def evi_vxlan(
        self, replication_type="ingress_replication", pmsi_label=16, ad_label=0
    ):
        # type: (Union[Literal["ingress_replication"]],int,int) -> BgpV4EvpnEvisIter
        """Factory method that creates an instance of the BgpV4EviVxlan class

        Configuration for BGP EVPN EVI. Advertises following routes - . . Type 3 - Inclusive Multicast Ethernet Tag Route. . Type 1 - Ethernet Auto-discovery Route (Per EVI). . Type 1 - Ethernet Auto-discovery Route (Per ES)

        Returns: BgpV4EvpnEvisIter
        """
        item = BgpV4EvpnEvis()
        item.evi_vxlan
        item.choice = "evi_vxlan"
        self._add(item)
        return self


class BgpV4EthernetSegmentIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV4EthernetSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV4EthernetSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV4EthernetSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV4EthernetSegment
        return self._next()

    def next(self):
        # type: () -> BgpV4EthernetSegment
        return self._next()

    def ethernetsegment(
        self, esi="00000000000000000000", active_mode="all_active", esi_label=0
    ):
        # type: (str,Union[Literal["all_active"], Literal["single_active"]],int) -> BgpV4EthernetSegmentIter
        """Factory method that creates an instance of the BgpV4EthernetSegment class

        Configuration for BGP Ethernet Segment ranges. Advertises following routes - . . Type 4 - Ethernet Segment Route

        Returns: BgpV4EthernetSegmentIter
        """
        item = BgpV4EthernetSegment(
            parent=self._parent, esi=esi, active_mode=active_mode, esi_label=esi_label
        )
        self._add(item)
        return self

    def add(self, esi="00000000000000000000", active_mode="all_active", esi_label=0):
        # type: (str,Union[Literal["all_active"], Literal["single_active"]],int) -> BgpV4EthernetSegment
        """Add method that creates and returns an instance of the BgpV4EthernetSegment class

        Configuration for BGP Ethernet Segment ranges. Advertises following routes - . . Type 4 - Ethernet Segment Route

        Returns: BgpV4EthernetSegment
        """
        item = BgpV4EthernetSegment(
            parent=self._parent, esi=esi, active_mode=active_mode, esi_label=esi_label
        )
        self._add(item)
        return item


class BgpAdvanced(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "hold_time_interval": {"type": int},
        "keep_alive_interval": {"type": int},
        "update_interval": {"type": int},
        "time_to_live": {"type": int},
        "md5_key": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "hold_time_interval": 90,
        "keep_alive_interval": 30,
        "update_interval": 0,
        "time_to_live": 64,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        hold_time_interval=90,
        keep_alive_interval=30,
        update_interval=0,
        time_to_live=64,
        md5_key=None,
    ):
        super(BgpAdvanced, self).__init__()
        self._parent = parent
        self._set_property("hold_time_interval", hold_time_interval)
        self._set_property("keep_alive_interval", keep_alive_interval)
        self._set_property("update_interval", update_interval)
        self._set_property("time_to_live", time_to_live)
        self._set_property("md5_key", md5_key)

    @property
    def hold_time_interval(self):
        # type: () -> int
        """hold_time_interval getter

        Number of seconds the sender proposes for the value of the Hold Timer.

        Returns: int
        """
        return self._get_property("hold_time_interval")

    @hold_time_interval.setter
    def hold_time_interval(self, value):
        """hold_time_interval setter

        Number of seconds the sender proposes for the value of the Hold Timer.

        value: int
        """
        self._set_property("hold_time_interval", value)

    @property
    def keep_alive_interval(self):
        # type: () -> int
        """keep_alive_interval getter

        Number of seconds between transmissions of Keepalive messages by this peer.

        Returns: int
        """
        return self._get_property("keep_alive_interval")

    @keep_alive_interval.setter
    def keep_alive_interval(self, value):
        """keep_alive_interval setter

        Number of seconds between transmissions of Keepalive messages by this peer.

        value: int
        """
        self._set_property("keep_alive_interval", value)

    @property
    def update_interval(self):
        # type: () -> int
        """update_interval getter

        The time interval at which Update messages are sent to the DUT, expressed as the number of milliseconds between Update messages. The update interval 0 implies to send all the updates as fast as possible.

        Returns: int
        """
        return self._get_property("update_interval")

    @update_interval.setter
    def update_interval(self, value):
        """update_interval setter

        The time interval at which Update messages are sent to the DUT, expressed as the number of milliseconds between Update messages. The update interval 0 implies to send all the updates as fast as possible.

        value: int
        """
        self._set_property("update_interval", value)

    @property
    def time_to_live(self):
        # type: () -> int
        """time_to_live getter

        The limited number of iterations that a unit of data can experience before the data is discarded. This is placed in the TTL field in the IP header of the transmitted packets.

        Returns: int
        """
        return self._get_property("time_to_live")

    @time_to_live.setter
    def time_to_live(self, value):
        """time_to_live setter

        The limited number of iterations that a unit of data can experience before the data is discarded. This is placed in the TTL field in the IP header of the transmitted packets.

        value: int
        """
        self._set_property("time_to_live", value)

    @property
    def md5_key(self):
        # type: () -> str
        """md5_key getter

        The value to be used as a secret MD5 key for authentication. If not configured, MD5 authentication will not be enabled.

        Returns: str
        """
        return self._get_property("md5_key")

    @md5_key.setter
    def md5_key(self, value):
        """md5_key setter

        The value to be used as a secret MD5 key for authentication. If not configured, MD5 authentication will not be enabled.

        value: str
        """
        self._set_property("md5_key", value)


class BgpCapability(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ipv4_unicast": {"type": bool},
        "ipv4_multicast": {"type": bool},
        "ipv6_unicast": {"type": bool},
        "ipv6_multicast": {"type": bool},
        "vpls": {"type": bool},
        "route_refresh": {"type": bool},
        "route_constraint": {"type": bool},
        "link_state_non_vpn": {"type": bool},
        "link_state_vpn": {"type": bool},
        "evpn": {"type": bool},
        "extended_next_hop_encoding": {"type": bool},
        "ipv4_multicast_vpn": {"type": bool},
        "ipv4_mpls_vpn": {"type": bool},
        "ipv4_mdt": {"type": bool},
        "ipv4_multicast_mpls_vpn": {"type": bool},
        "ipv4_unicast_flow_spec": {"type": bool},
        "ipv4_sr_te_policy": {"type": bool},
        "ipv4_unicast_add_path": {"type": bool},
        "ipv6_multicast_vpn": {"type": bool},
        "ipv6_mpls_vpn": {"type": bool},
        "ipv6_mdt": {"type": bool},
        "ipv6_multicast_mpls_vpn": {"type": bool},
        "ipv6_unicast_flow_spec": {"type": bool},
        "ipv6_sr_te_policy": {"type": bool},
        "ipv6_unicast_add_path": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "ipv4_unicast": True,
        "ipv4_multicast": False,
        "ipv6_unicast": True,
        "ipv6_multicast": False,
        "vpls": False,
        "route_refresh": True,
        "route_constraint": False,
        "link_state_non_vpn": False,
        "link_state_vpn": False,
        "evpn": False,
        "extended_next_hop_encoding": False,
        "ipv4_multicast_vpn": False,
        "ipv4_mpls_vpn": False,
        "ipv4_mdt": False,
        "ipv4_multicast_mpls_vpn": False,
        "ipv4_unicast_flow_spec": False,
        "ipv4_sr_te_policy": False,
        "ipv4_unicast_add_path": False,
        "ipv6_multicast_vpn": False,
        "ipv6_mpls_vpn": False,
        "ipv6_mdt": False,
        "ipv6_multicast_mpls_vpn": False,
        "ipv6_unicast_flow_spec": False,
        "ipv6_sr_te_policy": False,
        "ipv6_unicast_add_path": False,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        ipv4_unicast=True,
        ipv4_multicast=False,
        ipv6_unicast=True,
        ipv6_multicast=False,
        vpls=False,
        route_refresh=True,
        route_constraint=False,
        link_state_non_vpn=False,
        link_state_vpn=False,
        evpn=False,
        extended_next_hop_encoding=False,
        ipv4_multicast_vpn=False,
        ipv4_mpls_vpn=False,
        ipv4_mdt=False,
        ipv4_multicast_mpls_vpn=False,
        ipv4_unicast_flow_spec=False,
        ipv4_sr_te_policy=False,
        ipv4_unicast_add_path=False,
        ipv6_multicast_vpn=False,
        ipv6_mpls_vpn=False,
        ipv6_mdt=False,
        ipv6_multicast_mpls_vpn=False,
        ipv6_unicast_flow_spec=False,
        ipv6_sr_te_policy=False,
        ipv6_unicast_add_path=False,
    ):
        super(BgpCapability, self).__init__()
        self._parent = parent
        self._set_property("ipv4_unicast", ipv4_unicast)
        self._set_property("ipv4_multicast", ipv4_multicast)
        self._set_property("ipv6_unicast", ipv6_unicast)
        self._set_property("ipv6_multicast", ipv6_multicast)
        self._set_property("vpls", vpls)
        self._set_property("route_refresh", route_refresh)
        self._set_property("route_constraint", route_constraint)
        self._set_property("link_state_non_vpn", link_state_non_vpn)
        self._set_property("link_state_vpn", link_state_vpn)
        self._set_property("evpn", evpn)
        self._set_property("extended_next_hop_encoding", extended_next_hop_encoding)
        self._set_property("ipv4_multicast_vpn", ipv4_multicast_vpn)
        self._set_property("ipv4_mpls_vpn", ipv4_mpls_vpn)
        self._set_property("ipv4_mdt", ipv4_mdt)
        self._set_property("ipv4_multicast_mpls_vpn", ipv4_multicast_mpls_vpn)
        self._set_property("ipv4_unicast_flow_spec", ipv4_unicast_flow_spec)
        self._set_property("ipv4_sr_te_policy", ipv4_sr_te_policy)
        self._set_property("ipv4_unicast_add_path", ipv4_unicast_add_path)
        self._set_property("ipv6_multicast_vpn", ipv6_multicast_vpn)
        self._set_property("ipv6_mpls_vpn", ipv6_mpls_vpn)
        self._set_property("ipv6_mdt", ipv6_mdt)
        self._set_property("ipv6_multicast_mpls_vpn", ipv6_multicast_mpls_vpn)
        self._set_property("ipv6_unicast_flow_spec", ipv6_unicast_flow_spec)
        self._set_property("ipv6_sr_te_policy", ipv6_sr_te_policy)
        self._set_property("ipv6_unicast_add_path", ipv6_unicast_add_path)

    @property
    def ipv4_unicast(self):
        # type: () -> bool
        """ipv4_unicast getter

        Support for the IPv4 Unicast address family.

        Returns: bool
        """
        return self._get_property("ipv4_unicast")

    @ipv4_unicast.setter
    def ipv4_unicast(self, value):
        """ipv4_unicast setter

        Support for the IPv4 Unicast address family.

        value: bool
        """
        self._set_property("ipv4_unicast", value)

    @property
    def ipv4_multicast(self):
        # type: () -> bool
        """ipv4_multicast getter

        Support for the IPv4 Multicast address family.

        Returns: bool
        """
        return self._get_property("ipv4_multicast")

    @ipv4_multicast.setter
    def ipv4_multicast(self, value):
        """ipv4_multicast setter

        Support for the IPv4 Multicast address family.

        value: bool
        """
        self._set_property("ipv4_multicast", value)

    @property
    def ipv6_unicast(self):
        # type: () -> bool
        """ipv6_unicast getter

        Support for the IPv4 Unicast address family.

        Returns: bool
        """
        return self._get_property("ipv6_unicast")

    @ipv6_unicast.setter
    def ipv6_unicast(self, value):
        """ipv6_unicast setter

        Support for the IPv4 Unicast address family.

        value: bool
        """
        self._set_property("ipv6_unicast", value)

    @property
    def ipv6_multicast(self):
        # type: () -> bool
        """ipv6_multicast getter

        Support for the IPv6 Multicast address family.

        Returns: bool
        """
        return self._get_property("ipv6_multicast")

    @ipv6_multicast.setter
    def ipv6_multicast(self, value):
        """ipv6_multicast setter

        Support for the IPv6 Multicast address family.

        value: bool
        """
        self._set_property("ipv6_multicast", value)

    @property
    def vpls(self):
        # type: () -> bool
        """vpls getter

        Support for VPLS as below. . RFC4761 - Virtual Private LAN Service (VPLS) using BGP for Auto-Discovery. and Signaling. . RFC6624 - Layer 2 Virtual Private Networks using BGP for Auto-Discovery . and Signaling.

        Returns: bool
        """
        return self._get_property("vpls")

    @vpls.setter
    def vpls(self, value):
        """vpls setter

        Support for VPLS as below. . RFC4761 - Virtual Private LAN Service (VPLS) using BGP for Auto-Discovery. and Signaling. . RFC6624 - Layer 2 Virtual Private Networks using BGP for Auto-Discovery . and Signaling.

        value: bool
        """
        self._set_property("vpls", value)

    @property
    def route_refresh(self):
        # type: () -> bool
        """route_refresh getter

        Support for the route refresh capabilities. Route Refresh allows the dynamic exchange of route refresh requests and routing information between BGP peers and the subsequent re-advertisement of the outbound or inbound routing table.

        Returns: bool
        """
        return self._get_property("route_refresh")

    @route_refresh.setter
    def route_refresh(self, value):
        """route_refresh setter

        Support for the route refresh capabilities. Route Refresh allows the dynamic exchange of route refresh requests and routing information between BGP peers and the subsequent re-advertisement of the outbound or inbound routing table.

        value: bool
        """
        self._set_property("route_refresh", value)

    @property
    def route_constraint(self):
        # type: () -> bool
        """route_constraint getter

        Supports for the route constraint capabilities. Route Constraint allows the advertisement of Route Target Membership information. The BGP peers exchange Route Target Reachability Information, which is used to build a route distribution graph. This limits the propagation of VPN Network Layer Reachability Information (NLRI) between different autonomous systems or distinct clusters of the same autonomous system. This is supported for Layer 3 Virtual Private Network scenario.

        Returns: bool
        """
        return self._get_property("route_constraint")

    @route_constraint.setter
    def route_constraint(self, value):
        """route_constraint setter

        Supports for the route constraint capabilities. Route Constraint allows the advertisement of Route Target Membership information. The BGP peers exchange Route Target Reachability Information, which is used to build a route distribution graph. This limits the propagation of VPN Network Layer Reachability Information (NLRI) between different autonomous systems or distinct clusters of the same autonomous system. This is supported for Layer 3 Virtual Private Network scenario.

        value: bool
        """
        self._set_property("route_constraint", value)

    @property
    def link_state_non_vpn(self):
        # type: () -> bool
        """link_state_non_vpn getter

        Support for BGP Link State for ISIS and OSPF.

        Returns: bool
        """
        return self._get_property("link_state_non_vpn")

    @link_state_non_vpn.setter
    def link_state_non_vpn(self, value):
        """link_state_non_vpn setter

        Support for BGP Link State for ISIS and OSPF.

        value: bool
        """
        self._set_property("link_state_non_vpn", value)

    @property
    def link_state_vpn(self):
        # type: () -> bool
        """link_state_vpn getter

        Capability advertisement of BGP Link State for VPNs.

        Returns: bool
        """
        return self._get_property("link_state_vpn")

    @link_state_vpn.setter
    def link_state_vpn(self, value):
        """link_state_vpn setter

        Capability advertisement of BGP Link State for VPNs.

        value: bool
        """
        self._set_property("link_state_vpn", value)

    @property
    def evpn(self):
        # type: () -> bool
        """evpn getter

        Support for the EVPN address family.

        Returns: bool
        """
        return self._get_property("evpn")

    @evpn.setter
    def evpn(self, value):
        """evpn setter

        Support for the EVPN address family.

        value: bool
        """
        self._set_property("evpn", value)

    @property
    def extended_next_hop_encoding(self):
        # type: () -> bool
        """extended_next_hop_encoding getter

        Support for extended Next Hop Encoding for Nexthop field in IPv4 routes advertisement. This allows IPv4 routes being advertised by IPv6 peers to include an IPv6 Nexthop.

        Returns: bool
        """
        return self._get_property("extended_next_hop_encoding")

    @extended_next_hop_encoding.setter
    def extended_next_hop_encoding(self, value):
        """extended_next_hop_encoding setter

        Support for extended Next Hop Encoding for Nexthop field in IPv4 routes advertisement. This allows IPv4 routes being advertised by IPv6 peers to include an IPv6 Nexthop.

        value: bool
        """
        self._set_property("extended_next_hop_encoding", value)

    @property
    def ipv4_multicast_vpn(self):
        # type: () -> bool
        """ipv4_multicast_vpn getter

        Support for the IPv4 Multicast VPN address family.

        Returns: bool
        """
        return self._get_property("ipv4_multicast_vpn")

    @ipv4_multicast_vpn.setter
    def ipv4_multicast_vpn(self, value):
        """ipv4_multicast_vpn setter

        Support for the IPv4 Multicast VPN address family.

        value: bool
        """
        self._set_property("ipv4_multicast_vpn", value)

    @property
    def ipv4_mpls_vpn(self):
        # type: () -> bool
        """ipv4_mpls_vpn getter

        Support for the IPv4 MPLS L3VPN address family.

        Returns: bool
        """
        return self._get_property("ipv4_mpls_vpn")

    @ipv4_mpls_vpn.setter
    def ipv4_mpls_vpn(self, value):
        """ipv4_mpls_vpn setter

        Support for the IPv4 MPLS L3VPN address family.

        value: bool
        """
        self._set_property("ipv4_mpls_vpn", value)

    @property
    def ipv4_mdt(self):
        # type: () -> bool
        """ipv4_mdt getter

        Supports for IPv4 MDT address family messages.

        Returns: bool
        """
        return self._get_property("ipv4_mdt")

    @ipv4_mdt.setter
    def ipv4_mdt(self, value):
        """ipv4_mdt setter

        Supports for IPv4 MDT address family messages.

        value: bool
        """
        self._set_property("ipv4_mdt", value)

    @property
    def ipv4_multicast_mpls_vpn(self):
        # type: () -> bool
        """ipv4_multicast_mpls_vpn getter

        Support for the IPv4 Multicast VPN address family.

        Returns: bool
        """
        return self._get_property("ipv4_multicast_mpls_vpn")

    @ipv4_multicast_mpls_vpn.setter
    def ipv4_multicast_mpls_vpn(self, value):
        """ipv4_multicast_mpls_vpn setter

        Support for the IPv4 Multicast VPN address family.

        value: bool
        """
        self._set_property("ipv4_multicast_mpls_vpn", value)

    @property
    def ipv4_unicast_flow_spec(self):
        # type: () -> bool
        """ipv4_unicast_flow_spec getter

        Support for propagation of IPv4 unicast flow specification rules.

        Returns: bool
        """
        return self._get_property("ipv4_unicast_flow_spec")

    @ipv4_unicast_flow_spec.setter
    def ipv4_unicast_flow_spec(self, value):
        """ipv4_unicast_flow_spec setter

        Support for propagation of IPv4 unicast flow specification rules.

        value: bool
        """
        self._set_property("ipv4_unicast_flow_spec", value)

    @property
    def ipv4_sr_te_policy(self):
        # type: () -> bool
        """ipv4_sr_te_policy getter

        Support for IPv4 SRTE policy.

        Returns: bool
        """
        return self._get_property("ipv4_sr_te_policy")

    @ipv4_sr_te_policy.setter
    def ipv4_sr_te_policy(self, value):
        """ipv4_sr_te_policy setter

        Support for IPv4 SRTE policy.

        value: bool
        """
        self._set_property("ipv4_sr_te_policy", value)

    @property
    def ipv4_unicast_add_path(self):
        # type: () -> bool
        """ipv4_unicast_add_path getter

        Support for IPv4 Unicast Add Path Capability.

        Returns: bool
        """
        return self._get_property("ipv4_unicast_add_path")

    @ipv4_unicast_add_path.setter
    def ipv4_unicast_add_path(self, value):
        """ipv4_unicast_add_path setter

        Support for IPv4 Unicast Add Path Capability.

        value: bool
        """
        self._set_property("ipv4_unicast_add_path", value)

    @property
    def ipv6_multicast_vpn(self):
        # type: () -> bool
        """ipv6_multicast_vpn getter

        Support for the IPv6 Multicast VPN address family.

        Returns: bool
        """
        return self._get_property("ipv6_multicast_vpn")

    @ipv6_multicast_vpn.setter
    def ipv6_multicast_vpn(self, value):
        """ipv6_multicast_vpn setter

        Support for the IPv6 Multicast VPN address family.

        value: bool
        """
        self._set_property("ipv6_multicast_vpn", value)

    @property
    def ipv6_mpls_vpn(self):
        # type: () -> bool
        """ipv6_mpls_vpn getter

        Support for the IPv6 MPLS L3VPN address family.

        Returns: bool
        """
        return self._get_property("ipv6_mpls_vpn")

    @ipv6_mpls_vpn.setter
    def ipv6_mpls_vpn(self, value):
        """ipv6_mpls_vpn setter

        Support for the IPv6 MPLS L3VPN address family.

        value: bool
        """
        self._set_property("ipv6_mpls_vpn", value)

    @property
    def ipv6_mdt(self):
        # type: () -> bool
        """ipv6_mdt getter

        Support for IPv6 MDT address family messages.

        Returns: bool
        """
        return self._get_property("ipv6_mdt")

    @ipv6_mdt.setter
    def ipv6_mdt(self, value):
        """ipv6_mdt setter

        Support for IPv6 MDT address family messages.

        value: bool
        """
        self._set_property("ipv6_mdt", value)

    @property
    def ipv6_multicast_mpls_vpn(self):
        # type: () -> bool
        """ipv6_multicast_mpls_vpn getter

        Support for the IPv6 Multicast VPN address family.

        Returns: bool
        """
        return self._get_property("ipv6_multicast_mpls_vpn")

    @ipv6_multicast_mpls_vpn.setter
    def ipv6_multicast_mpls_vpn(self, value):
        """ipv6_multicast_mpls_vpn setter

        Support for the IPv6 Multicast VPN address family.

        value: bool
        """
        self._set_property("ipv6_multicast_mpls_vpn", value)

    @property
    def ipv6_unicast_flow_spec(self):
        # type: () -> bool
        """ipv6_unicast_flow_spec getter

        Support for propagation of IPv6 unicast flow specification rules.

        Returns: bool
        """
        return self._get_property("ipv6_unicast_flow_spec")

    @ipv6_unicast_flow_spec.setter
    def ipv6_unicast_flow_spec(self, value):
        """ipv6_unicast_flow_spec setter

        Support for propagation of IPv6 unicast flow specification rules.

        value: bool
        """
        self._set_property("ipv6_unicast_flow_spec", value)

    @property
    def ipv6_sr_te_policy(self):
        # type: () -> bool
        """ipv6_sr_te_policy getter

        Support for IPv6 SRTE policy.

        Returns: bool
        """
        return self._get_property("ipv6_sr_te_policy")

    @ipv6_sr_te_policy.setter
    def ipv6_sr_te_policy(self, value):
        """ipv6_sr_te_policy setter

        Support for IPv6 SRTE policy.

        value: bool
        """
        self._set_property("ipv6_sr_te_policy", value)

    @property
    def ipv6_unicast_add_path(self):
        # type: () -> bool
        """ipv6_unicast_add_path getter

        Support for IPv6 Unicast Add Path Capability.

        Returns: bool
        """
        return self._get_property("ipv6_unicast_add_path")

    @ipv6_unicast_add_path.setter
    def ipv6_unicast_add_path(self, value):
        """ipv6_unicast_add_path setter

        Support for IPv6 Unicast Add Path Capability.

        value: bool
        """
        self._set_property("ipv6_unicast_add_path", value)


class BgpLearnedInformationFilter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "unicast_ipv4_prefix": {"type": bool},
        "unicast_ipv6_prefix": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "unicast_ipv4_prefix": False,
        "unicast_ipv6_prefix": False,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, unicast_ipv4_prefix=False, unicast_ipv6_prefix=False
    ):
        super(BgpLearnedInformationFilter, self).__init__()
        self._parent = parent
        self._set_property("unicast_ipv4_prefix", unicast_ipv4_prefix)
        self._set_property("unicast_ipv6_prefix", unicast_ipv6_prefix)

    @property
    def unicast_ipv4_prefix(self):
        # type: () -> bool
        """unicast_ipv4_prefix getter

        If enabled, will store the information related to Unicast IPv4 Prefixes recieved from the peer.

        Returns: bool
        """
        return self._get_property("unicast_ipv4_prefix")

    @unicast_ipv4_prefix.setter
    def unicast_ipv4_prefix(self, value):
        """unicast_ipv4_prefix setter

        If enabled, will store the information related to Unicast IPv4 Prefixes recieved from the peer.

        value: bool
        """
        self._set_property("unicast_ipv4_prefix", value)

    @property
    def unicast_ipv6_prefix(self):
        # type: () -> bool
        """unicast_ipv6_prefix getter

        If enabled, will store the information related to Unicast IPv6 Prefixes recieved from the peer.

        Returns: bool
        """
        return self._get_property("unicast_ipv6_prefix")

    @unicast_ipv6_prefix.setter
    def unicast_ipv6_prefix(self, value):
        """unicast_ipv6_prefix setter

        If enabled, will store the information related to Unicast IPv6 Prefixes recieved from the peer.

        value: bool
        """
        self._set_property("unicast_ipv6_prefix", value)


class BgpV4RouteRange(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "addresses": {"type": "V4RouteAddressIter"},
        "next_hop_mode": {
            "type": str,
            "enum": [
                "local_ip",
                "manual",
            ],
        },
        "next_hop_address_type": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
            ],
        },
        "next_hop_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "next_hop_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
        "add_path": {"type": "BgpAddPath"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "next_hop_mode": "local_ip",
        "next_hop_address_type": "ipv4",
        "next_hop_ipv4_address": "0.0.0.0",
        "next_hop_ipv6_address": "::0",
    }  # type: Dict[str, Union(type)]

    LOCAL_IP = "local_ip"  # type: str
    MANUAL = "manual"  # type: str

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    def __init__(
        self,
        parent=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        super(BgpV4RouteRange, self).__init__()
        self._parent = parent
        self._set_property("next_hop_mode", next_hop_mode)
        self._set_property("next_hop_address_type", next_hop_address_type)
        self._set_property("next_hop_ipv4_address", next_hop_ipv4_address)
        self._set_property("next_hop_ipv6_address", next_hop_ipv6_address)
        self._set_property("name", name)

    @property
    def addresses(self):
        # type: () -> V4RouteAddressIter
        """addresses getter

        A list of group of IPv4 route addresses.

        Returns: V4RouteAddressIter
        """
        return self._get_property(
            "addresses", V4RouteAddressIter, self._parent, self._choice
        )

    @property
    def next_hop_mode(self):
        # type: () -> Union[Literal["local_ip"], Literal["manual"]]
        """next_hop_mode getter

        Specify the NextHop in MP REACH NLRI. The mode for setting the IP address of the NextHop in the MP REACH NLRI can be one of the following:. Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer.. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        Returns: Union[Literal["local_ip"], Literal["manual"]]
        """
        return self._get_property("next_hop_mode")

    @next_hop_mode.setter
    def next_hop_mode(self, value):
        """next_hop_mode setter

        Specify the NextHop in MP REACH NLRI. The mode for setting the IP address of the NextHop in the MP REACH NLRI can be one of the following:. Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer.. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        value: Union[Literal["local_ip"], Literal["manual"]]
        """
        self._set_property("next_hop_mode", value)

    @property
    def next_hop_address_type(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """next_hop_address_type getter

        If the Nexthop Mode is Manual, it sets the type of the NextHop IP address.

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property("next_hop_address_type")

    @next_hop_address_type.setter
    def next_hop_address_type(self, value):
        """next_hop_address_type setter

        If the Nexthop Mode is Manual, it sets the type of the NextHop IP address.

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property("next_hop_address_type", value)

    @property
    def next_hop_ipv4_address(self):
        # type: () -> str
        """next_hop_ipv4_address getter

        The IPv4 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.

        Returns: str
        """
        return self._get_property("next_hop_ipv4_address")

    @next_hop_ipv4_address.setter
    def next_hop_ipv4_address(self, value):
        """next_hop_ipv4_address setter

        The IPv4 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.

        value: str
        """
        self._set_property("next_hop_ipv4_address", value)

    @property
    def next_hop_ipv6_address(self):
        # type: () -> str
        """next_hop_ipv6_address getter

        The IPv6 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv6.

        Returns: str
        """
        return self._get_property("next_hop_ipv6_address")

    @next_hop_ipv6_address.setter
    def next_hop_ipv6_address(self, value):
        """next_hop_ipv6_address setter

        The IPv6 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv6.

        value: str
        """
        self._set_property("next_hop_ipv6_address", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)

    @property
    def add_path(self):
        # type: () -> BgpAddPath
        """add_path getter

        The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: BgpAddPath
        """
        return self._get_property("add_path", BgpAddPath)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class BgpAddPath(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "path_id": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "path_id": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, path_id=1):
        super(BgpAddPath, self).__init__()
        self._parent = parent
        self._set_property("path_id", path_id)

    @property
    def path_id(self):
        # type: () -> int
        """path_id getter

        The id of the additional path.

        Returns: int
        """
        return self._get_property("path_id")

    @path_id.setter
    def path_id(self, value):
        """path_id setter

        The id of the additional path.

        value: int
        """
        self._set_property("path_id", value)


class BgpV4RouteRangeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV4RouteRangeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV4RouteRange]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV4RouteRangeIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV4RouteRange
        return self._next()

    def next(self):
        # type: () -> BgpV4RouteRange
        return self._next()

    def v4routerange(
        self,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        # type: (Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str) -> BgpV4RouteRangeIter
        """Factory method that creates an instance of the BgpV4RouteRange class

        Emulated BGPv4 route range.. Configuration for BGP route ranges.

        Returns: BgpV4RouteRangeIter
        """
        item = BgpV4RouteRange(
            parent=self._parent,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        # type: (Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str) -> BgpV4RouteRange
        """Add method that creates and returns an instance of the BgpV4RouteRange class

        Emulated BGPv4 route range.. Configuration for BGP route ranges.

        Returns: BgpV4RouteRange
        """
        item = BgpV4RouteRange(
            parent=self._parent,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
        )
        self._add(item)
        return item


class BgpV6RouteRange(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "addresses": {"type": "V6RouteAddressIter"},
        "next_hop_mode": {
            "type": str,
            "enum": [
                "local_ip",
                "manual",
            ],
        },
        "next_hop_address_type": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
            ],
        },
        "next_hop_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "next_hop_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
        "add_path": {"type": "BgpAddPath"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "next_hop_mode": "local_ip",
        "next_hop_address_type": "ipv6",
        "next_hop_ipv4_address": "0.0.0.0",
        "next_hop_ipv6_address": "::0",
    }  # type: Dict[str, Union(type)]

    LOCAL_IP = "local_ip"  # type: str
    MANUAL = "manual"  # type: str

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    def __init__(
        self,
        parent=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        super(BgpV6RouteRange, self).__init__()
        self._parent = parent
        self._set_property("next_hop_mode", next_hop_mode)
        self._set_property("next_hop_address_type", next_hop_address_type)
        self._set_property("next_hop_ipv4_address", next_hop_ipv4_address)
        self._set_property("next_hop_ipv6_address", next_hop_ipv6_address)
        self._set_property("name", name)

    @property
    def addresses(self):
        # type: () -> V6RouteAddressIter
        """addresses getter

        A list of group of IPv6 route addresses.

        Returns: V6RouteAddressIter
        """
        return self._get_property(
            "addresses", V6RouteAddressIter, self._parent, self._choice
        )

    @property
    def next_hop_mode(self):
        # type: () -> Union[Literal["local_ip"], Literal["manual"]]
        """next_hop_mode getter

        Specify the NextHop in MP REACH NLRI. The mode for setting the IP address of the NextHop in the MP REACH NLRI can be one of the following:. Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer.. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        Returns: Union[Literal["local_ip"], Literal["manual"]]
        """
        return self._get_property("next_hop_mode")

    @next_hop_mode.setter
    def next_hop_mode(self, value):
        """next_hop_mode setter

        Specify the NextHop in MP REACH NLRI. The mode for setting the IP address of the NextHop in the MP REACH NLRI can be one of the following:. Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer.. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        value: Union[Literal["local_ip"], Literal["manual"]]
        """
        self._set_property("next_hop_mode", value)

    @property
    def next_hop_address_type(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """next_hop_address_type getter

        If the Nexthop Mode is Manual, it sets the type of the NextHop IP address.

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property("next_hop_address_type")

    @next_hop_address_type.setter
    def next_hop_address_type(self, value):
        """next_hop_address_type setter

        If the Nexthop Mode is Manual, it sets the type of the NextHop IP address.

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property("next_hop_address_type", value)

    @property
    def next_hop_ipv4_address(self):
        # type: () -> str
        """next_hop_ipv4_address getter

        The IPv4 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.

        Returns: str
        """
        return self._get_property("next_hop_ipv4_address")

    @next_hop_ipv4_address.setter
    def next_hop_ipv4_address(self, value):
        """next_hop_ipv4_address setter

        The IPv4 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.

        value: str
        """
        self._set_property("next_hop_ipv4_address", value)

    @property
    def next_hop_ipv6_address(self):
        # type: () -> str
        """next_hop_ipv6_address getter

        The IPv6 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv6.

        Returns: str
        """
        return self._get_property("next_hop_ipv6_address")

    @next_hop_ipv6_address.setter
    def next_hop_ipv6_address(self, value):
        """next_hop_ipv6_address setter

        The IPv6 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv6.

        value: str
        """
        self._set_property("next_hop_ipv6_address", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)

    @property
    def add_path(self):
        # type: () -> BgpAddPath
        """add_path getter

        The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: BgpAddPath
        """
        return self._get_property("add_path", BgpAddPath)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class BgpV6RouteRangeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV6RouteRangeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV6RouteRange]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV6RouteRangeIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV6RouteRange
        return self._next()

    def next(self):
        # type: () -> BgpV6RouteRange
        return self._next()

    def v6routerange(
        self,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        # type: (Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str) -> BgpV6RouteRangeIter
        """Factory method that creates an instance of the BgpV6RouteRange class

        Emulated BGPv6 route range.. Configuration for BGP route ranges.

        Returns: BgpV6RouteRangeIter
        """
        item = BgpV6RouteRange(
            parent=self._parent,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        # type: (Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str) -> BgpV6RouteRange
        """Add method that creates and returns an instance of the BgpV6RouteRange class

        Emulated BGPv6 route range.. Configuration for BGP route ranges.

        Returns: BgpV6RouteRange
        """
        item = BgpV6RouteRange(
            parent=self._parent,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
        )
        self._add(item)
        return item


class BgpSrteV4Policy(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "distinguisher": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "color": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "ipv4_endpoint": {
            "type": str,
            "format": "ipv4",
        },
        "next_hop_mode": {
            "type": str,
            "enum": [
                "local_ip",
                "manual",
            ],
        },
        "next_hop_address_type": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
            ],
        },
        "next_hop_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "next_hop_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "add_path": {"type": "BgpAddPath"},
        "as_path": {"type": "BgpAsPath"},
        "communities": {"type": "BgpCommunityIter"},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "tunnel_tlvs": {"type": "BgpSrteV4TunnelTlvIter"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv4_endpoint", "name")  # type: tuple(str)

    _DEFAULTS = {
        "distinguisher": 1,
        "color": 100,
        "next_hop_mode": "local_ip",
        "next_hop_address_type": "ipv4",
        "active": True,
    }  # type: Dict[str, Union(type)]

    LOCAL_IP = "local_ip"  # type: str
    MANUAL = "manual"  # type: str

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    def __init__(
        self,
        parent=None,
        distinguisher=1,
        color=100,
        ipv4_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address=None,
        next_hop_ipv6_address=None,
        name=None,
        active=True,
    ):
        super(BgpSrteV4Policy, self).__init__()
        self._parent = parent
        self._set_property("distinguisher", distinguisher)
        self._set_property("color", color)
        self._set_property("ipv4_endpoint", ipv4_endpoint)
        self._set_property("next_hop_mode", next_hop_mode)
        self._set_property("next_hop_address_type", next_hop_address_type)
        self._set_property("next_hop_ipv4_address", next_hop_ipv4_address)
        self._set_property("next_hop_ipv6_address", next_hop_ipv6_address)
        self._set_property("name", name)
        self._set_property("active", active)

    @property
    def distinguisher(self):
        # type: () -> int
        """distinguisher getter

        4-octet value uniquely identifying the policy in the context of (color, endpoint) tuple. It is used by the SR Policy originator to make unique (from an NLRI perspective) both for multiple candidate paths of the same SR Policy as well as candidate paths of different SR Policies (i.e. with different segment list) with the same Color and Endpoint but meant for different head-ends.

        Returns: int
        """
        return self._get_property("distinguisher")

    @distinguisher.setter
    def distinguisher(self, value):
        """distinguisher setter

        4-octet value uniquely identifying the policy in the context of (color, endpoint) tuple. It is used by the SR Policy originator to make unique (from an NLRI perspective) both for multiple candidate paths of the same SR Policy as well as candidate paths of different SR Policies (i.e. with different segment list) with the same Color and Endpoint but meant for different head-ends.

        value: int
        """
        self._set_property("distinguisher", value)

    @property
    def color(self):
        # type: () -> int
        """color getter

        Policy color is used to match the color of the destination prefixes to steer traffic into the SR Policy.

        Returns: int
        """
        return self._get_property("color")

    @color.setter
    def color(self, value):
        """color setter

        Policy color is used to match the color of the destination prefixes to steer traffic into the SR Policy.

        value: int
        """
        self._set_property("color", value)

    @property
    def ipv4_endpoint(self):
        # type: () -> str
        """ipv4_endpoint getter

        Specifies a single node or a set of nodes (e.g. an anycast address). It is selected on the basis of the SR Policy type (AFI).

        Returns: str
        """
        return self._get_property("ipv4_endpoint")

    @ipv4_endpoint.setter
    def ipv4_endpoint(self, value):
        """ipv4_endpoint setter

        Specifies a single node or a set of nodes (e.g. an anycast address). It is selected on the basis of the SR Policy type (AFI).

        value: str
        """
        self._set_property("ipv4_endpoint", value)

    @property
    def next_hop_mode(self):
        # type: () -> Union[Literal["local_ip"], Literal["manual"]]
        """next_hop_mode getter

        Mode for choosing the NextHop in MP REACH NLRI. Available modes are : Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer. For IPv6 BGP peer the Nexthop Encoding capability should be enabled. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        Returns: Union[Literal["local_ip"], Literal["manual"]]
        """
        return self._get_property("next_hop_mode")

    @next_hop_mode.setter
    def next_hop_mode(self, value):
        """next_hop_mode setter

        Mode for choosing the NextHop in MP REACH NLRI. Available modes are : Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer. For IPv6 BGP peer the Nexthop Encoding capability should be enabled. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        value: Union[Literal["local_ip"], Literal["manual"]]
        """
        self._set_property("next_hop_mode", value)

    @property
    def next_hop_address_type(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """next_hop_address_type getter

        Type of next hop IP address to be used when 'next_hop_mode' is set to 'manual'.

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property("next_hop_address_type")

    @next_hop_address_type.setter
    def next_hop_address_type(self, value):
        """next_hop_address_type setter

        Type of next hop IP address to be used when 'next_hop_mode' is set to 'manual'.

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property("next_hop_address_type", value)

    @property
    def next_hop_ipv4_address(self):
        # type: () -> str
        """next_hop_ipv4_address getter

        The IPv4 address of the next hop if the Nexthop type 'next_hop_mode' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability extended_next_hop_encoding should be enabled.

        Returns: str
        """
        return self._get_property("next_hop_ipv4_address")

    @next_hop_ipv4_address.setter
    def next_hop_ipv4_address(self, value):
        """next_hop_ipv4_address setter

        The IPv4 address of the next hop if the Nexthop type 'next_hop_mode' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability extended_next_hop_encoding should be enabled.

        value: str
        """
        self._set_property("next_hop_ipv4_address", value)

    @property
    def next_hop_ipv6_address(self):
        # type: () -> str
        """next_hop_ipv6_address getter

        The IPv6 address of the next hop if the Nexthop Mode 'next_hop_address_type' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv6.

        Returns: str
        """
        return self._get_property("next_hop_ipv6_address")

    @next_hop_ipv6_address.setter
    def next_hop_ipv6_address(self, value):
        """next_hop_ipv6_address setter

        The IPv6 address of the next hop if the Nexthop Mode 'next_hop_address_type' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv6.

        value: str
        """
        self._set_property("next_hop_ipv6_address", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def add_path(self):
        # type: () -> BgpAddPath
        """add_path getter

        The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: BgpAddPath
        """
        return self._get_property("add_path", BgpAddPath)

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional Community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Optional Extended Community settings.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def tunnel_tlvs(self):
        # type: () -> BgpSrteV4TunnelTlvIter
        """tunnel_tlvs getter

        List Tunnel Encapsulation Attributes.

        Returns: BgpSrteV4TunnelTlvIter
        """
        return self._get_property(
            "tunnel_tlvs", BgpSrteV4TunnelTlvIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteV4TunnelTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "remote_endpoint_sub_tlv": {"type": "BgpSrteRemoteEndpointSubTlv"},
        "color_sub_tlv": {"type": "BgpSrteColorSubTlv"},
        "binding_sub_tlv": {"type": "BgpSrteBindingSubTlv"},
        "preference_sub_tlv": {"type": "BgpSrtePreferenceSubTlv"},
        "policy_priority_sub_tlv": {"type": "BgpSrtePolicyPrioritySubTlv"},
        "policy_name_sub_tlv": {"type": "BgpSrtePolicyNameSubTlv"},
        "explicit_null_label_policy_sub_tlv": {
            "type": "BgpSrteExplicitNullLabelPolicySubTlv"
        },
        "segment_lists": {"type": "BgpSrteSegmentListIter"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "active": True,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, name=None, active=True):
        super(BgpSrteV4TunnelTlv, self).__init__()
        self._parent = parent
        self._set_property("name", name)
        self._set_property("active", active)

    @property
    def remote_endpoint_sub_tlv(self):
        # type: () -> BgpSrteRemoteEndpointSubTlv
        """remote_endpoint_sub_tlv getter

        Configuration for the BGP remote endpoint sub TLV.Configuration for the BGP remote endpoint sub TLV.

        Returns: BgpSrteRemoteEndpointSubTlv
        """
        return self._get_property(
            "remote_endpoint_sub_tlv", BgpSrteRemoteEndpointSubTlv
        )

    @property
    def color_sub_tlv(self):
        # type: () -> BgpSrteColorSubTlv
        """color_sub_tlv getter

        Configuration for the Policy Color attribute sub-TLV. The Color sub-TLV MAY be used as a way to "color" the corresponding Tunnel TLV. The Value field of the sub-TLV is eight octets long and consists of a Color Extended Community. First two octets of its Value field are 0x030b as type and subtype of extended community. Remaining six octets are are exposed to configure. Configuration for the Policy Color attribute sub-TLV. The Color sub-TLV MAY be used as a way to "color" the corresponding Tunnel TLV. The Value field of the sub-TLV is eight octets long and consists of a Color Extended Community. First two octets of its Value field are 0x030b as type and subtype of extended community. Remaining six octets are are exposed to configure.

        Returns: BgpSrteColorSubTlv
        """
        return self._get_property("color_sub_tlv", BgpSrteColorSubTlv)

    @property
    def binding_sub_tlv(self):
        # type: () -> BgpSrteBindingSubTlv
        """binding_sub_tlv getter

        Configuration for the binding SID sub-TLV. This is used to signal the binding SID related information of the SR Policy candidate path.Configuration for the binding SID sub-TLV. This is used to signal the binding SID related information of the SR Policy candidate path.

        Returns: BgpSrteBindingSubTlv
        """
        return self._get_property("binding_sub_tlv", BgpSrteBindingSubTlv)

    @property
    def preference_sub_tlv(self):
        # type: () -> BgpSrtePreferenceSubTlv
        """preference_sub_tlv getter

        Configuration for BGP preference sub TLV of the SR Policy candidate path.Configuration for BGP preference sub TLV of the SR Policy candidate path.

        Returns: BgpSrtePreferenceSubTlv
        """
        return self._get_property("preference_sub_tlv", BgpSrtePreferenceSubTlv)

    @property
    def policy_priority_sub_tlv(self):
        # type: () -> BgpSrtePolicyPrioritySubTlv
        """policy_priority_sub_tlv getter

        Configuration for the Policy Priority sub-TLV. The Policy Priority to indicate the order in which the SR policies are re-computed upon topological change.Configuration for the Policy Priority sub-TLV. The Policy Priority to indicate the order in which the SR policies are re-computed upon topological change.

        Returns: BgpSrtePolicyPrioritySubTlv
        """
        return self._get_property(
            "policy_priority_sub_tlv", BgpSrtePolicyPrioritySubTlv
        )

    @property
    def policy_name_sub_tlv(self):
        # type: () -> BgpSrtePolicyNameSubTlv
        """policy_name_sub_tlv getter

        Configuration for the Policy Name sub-TLV. The Policy Name sub-TLV is used to attach a symbolic name to the SR Policy candidate path.Configuration for the Policy Name sub-TLV. The Policy Name sub-TLV is used to attach a symbolic name to the SR Policy candidate path.

        Returns: BgpSrtePolicyNameSubTlv
        """
        return self._get_property("policy_name_sub_tlv", BgpSrtePolicyNameSubTlv)

    @property
    def explicit_null_label_policy_sub_tlv(self):
        # type: () -> BgpSrteExplicitNullLabelPolicySubTlv
        """explicit_null_label_policy_sub_tlv getter

        Configuration for BGP explicit null label policy sub TLV settings.Configuration for BGP explicit null label policy sub TLV settings.

        Returns: BgpSrteExplicitNullLabelPolicySubTlv
        """
        return self._get_property(
            "explicit_null_label_policy_sub_tlv", BgpSrteExplicitNullLabelPolicySubTlv
        )

    @property
    def segment_lists(self):
        # type: () -> BgpSrteSegmentListIter
        """segment_lists getter

        TBD

        Returns: BgpSrteSegmentListIter
        """
        return self._get_property(
            "segment_lists", BgpSrteSegmentListIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteRemoteEndpointSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "as_number": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "address_family": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
            ],
        },
        "ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "as_number": 0,
        "address_family": "ipv4",
        "ipv4_address": "0.0.0.0",
        "ipv6_address": "::0",
    }  # type: Dict[str, Union(type)]

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    def __init__(
        self,
        parent=None,
        as_number=0,
        address_family="ipv4",
        ipv4_address="0.0.0.0",
        ipv6_address="::0",
    ):
        super(BgpSrteRemoteEndpointSubTlv, self).__init__()
        self._parent = parent
        self._set_property("as_number", as_number)
        self._set_property("address_family", address_family)
        self._set_property("ipv4_address", ipv4_address)
        self._set_property("ipv6_address", ipv6_address)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        Autonomous system (AS) number

        Returns: int
        """
        return self._get_property("as_number")

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        Autonomous system (AS) number

        value: int
        """
        self._set_property("as_number", value)

    @property
    def address_family(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """address_family getter

        Determines the address type

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property("address_family")

    @address_family.setter
    def address_family(self, value):
        """address_family setter

        Determines the address type

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property("address_family", value)

    @property
    def ipv4_address(self):
        # type: () -> str
        """ipv4_address getter

        The IPv4 address of the Remote Endpoint.

        Returns: str
        """
        return self._get_property("ipv4_address")

    @ipv4_address.setter
    def ipv4_address(self, value):
        """ipv4_address setter

        The IPv4 address of the Remote Endpoint.

        value: str
        """
        self._set_property("ipv4_address", value)

    @property
    def ipv6_address(self):
        # type: () -> str
        """ipv6_address getter

        The IPv6 address of the Remote Endpoint.

        Returns: str
        """
        return self._get_property("ipv6_address")

    @ipv6_address.setter
    def ipv6_address(self, value):
        """ipv6_address setter

        The IPv6 address of the Remote Endpoint.

        value: str
        """
        self._set_property("ipv6_address", value)


class BgpSrteColorSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "color": {
            "type": str,
            "format": "hex",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, color=None):
        super(BgpSrteColorSubTlv, self).__init__()
        self._parent = parent
        self._set_property("color", color)

    @property
    def color(self):
        # type: () -> str
        """color getter

        Six octet values. Example: 000000000064 for color value 100.

        Returns: str
        """
        return self._get_property("color")

    @color.setter
    def color(self, value):
        """color setter

        Six octet values. Example: 000000000064 for color value 100.

        value: str
        """
        self._set_property("color", value)


class BgpSrteBindingSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "binding_sid_type": {
            "type": str,
            "enum": [
                "no_binding",
                "four_octet_sid",
                "ipv6_sid",
            ],
        },
        "four_octet_sid": {"type": int},
        "ipv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "s_flag": {"type": bool},
        "i_flag": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "binding_sid_type": "no_binding",
        "s_flag": False,
        "i_flag": False,
    }  # type: Dict[str, Union(type)]

    NO_BINDING = "no_binding"  # type: str
    FOUR_OCTET_SID = "four_octet_sid"  # type: str
    IPV6_SID = "ipv6_sid"  # type: str

    def __init__(
        self,
        parent=None,
        binding_sid_type="no_binding",
        four_octet_sid=None,
        ipv6_sid=None,
        s_flag=False,
        i_flag=False,
    ):
        super(BgpSrteBindingSubTlv, self).__init__()
        self._parent = parent
        self._set_property("binding_sid_type", binding_sid_type)
        self._set_property("four_octet_sid", four_octet_sid)
        self._set_property("ipv6_sid", ipv6_sid)
        self._set_property("s_flag", s_flag)
        self._set_property("i_flag", i_flag)

    @property
    def binding_sid_type(self):
        # type: () -> Union[Literal["four_octet_sid"], Literal["ipv6_sid"], Literal["no_binding"]]
        """binding_sid_type getter

        Type of the binding SID. Supported types are "No Binding SID" or "Four Octets Sid" or "IPv6 SID".

        Returns: Union[Literal["four_octet_sid"], Literal["ipv6_sid"], Literal["no_binding"]]
        """
        return self._get_property("binding_sid_type")

    @binding_sid_type.setter
    def binding_sid_type(self, value):
        """binding_sid_type setter

        Type of the binding SID. Supported types are "No Binding SID" or "Four Octets Sid" or "IPv6 SID".

        value: Union[Literal["four_octet_sid"], Literal["ipv6_sid"], Literal["no_binding"]]
        """
        self._set_property("binding_sid_type", value)

    @property
    def four_octet_sid(self):
        # type: () -> int
        """four_octet_sid getter

        Binding SID is encoded in 4 octets.

        Returns: int
        """
        return self._get_property("four_octet_sid")

    @four_octet_sid.setter
    def four_octet_sid(self, value):
        """four_octet_sid setter

        Binding SID is encoded in 4 octets.

        value: int
        """
        self._set_property("four_octet_sid", value)

    @property
    def ipv6_sid(self):
        # type: () -> str
        """ipv6_sid getter

        IPv6 SID value.

        Returns: str
        """
        return self._get_property("ipv6_sid")

    @ipv6_sid.setter
    def ipv6_sid(self, value):
        """ipv6_sid setter

        IPv6 SID value.

        value: str
        """
        self._set_property("ipv6_sid", value)

    @property
    def s_flag(self):
        # type: () -> bool
        """s_flag getter

        S-Flag encodes the "Specified-BSID-only" behavior.

        Returns: bool
        """
        return self._get_property("s_flag")

    @s_flag.setter
    def s_flag(self, value):
        """s_flag setter

        S-Flag encodes the "Specified-BSID-only" behavior.

        value: bool
        """
        self._set_property("s_flag", value)

    @property
    def i_flag(self):
        # type: () -> bool
        """i_flag getter

        I-Flag encodes the "Drop Upon Invalid" behavior.

        Returns: bool
        """
        return self._get_property("i_flag")

    @i_flag.setter
    def i_flag(self, value):
        """i_flag setter

        I-Flag encodes the "Drop Upon Invalid" behavior.

        value: bool
        """
        self._set_property("i_flag", value)


class BgpSrtePreferenceSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "preference": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "preference": 0,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, preference=0):
        super(BgpSrtePreferenceSubTlv, self).__init__()
        self._parent = parent
        self._set_property("preference", preference)

    @property
    def preference(self):
        # type: () -> int
        """preference getter

        The preference value of the SR Policy candidate path.

        Returns: int
        """
        return self._get_property("preference")

    @preference.setter
    def preference(self, value):
        """preference setter

        The preference value of the SR Policy candidate path.

        value: int
        """
        self._set_property("preference", value)


class BgpSrtePolicyPrioritySubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "policy_priority": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, policy_priority=None):
        super(BgpSrtePolicyPrioritySubTlv, self).__init__()
        self._parent = parent
        self._set_property("policy_priority", policy_priority)

    @property
    def policy_priority(self):
        # type: () -> int
        """policy_priority getter

        One-octet Priority value.

        Returns: int
        """
        return self._get_property("policy_priority")

    @policy_priority.setter
    def policy_priority(self, value):
        """policy_priority setter

        One-octet Priority value.

        value: int
        """
        self._set_property("policy_priority", value)


class BgpSrtePolicyNameSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "policy_name": {
            "type": str,
            "minLength": 1,
            "maxLength": 32,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, policy_name=None):
        super(BgpSrtePolicyNameSubTlv, self).__init__()
        self._parent = parent
        self._set_property("policy_name", policy_name)

    @property
    def policy_name(self):
        # type: () -> str
        """policy_name getter

        Symbolic name for the policy that should be a string of printable ASCII characters, without a NULL terminator.

        Returns: str
        """
        return self._get_property("policy_name")

    @policy_name.setter
    def policy_name(self, value):
        """policy_name setter

        Symbolic name for the policy that should be a string of printable ASCII characters, without a NULL terminator.

        value: str
        """
        self._set_property("policy_name", value)


class BgpSrteExplicitNullLabelPolicySubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "explicit_null_label_policy": {
            "type": str,
            "enum": [
                "reserved_enlp",
                "push_ipv4_enlp",
                "push_ipv6_enlp",
                "push_ipv4_ipv6_enlp",
                "do_not_push_enlp",
            ],
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "explicit_null_label_policy": "do_not_push_enlp",
    }  # type: Dict[str, Union(type)]

    RESERVED_ENLP = "reserved_enlp"  # type: str
    PUSH_IPV4_ENLP = "push_ipv4_enlp"  # type: str
    PUSH_IPV6_ENLP = "push_ipv6_enlp"  # type: str
    PUSH_IPV4_IPV6_ENLP = "push_ipv4_ipv6_enlp"  # type: str
    DO_NOT_PUSH_ENLP = "do_not_push_enlp"  # type: str

    def __init__(self, parent=None, explicit_null_label_policy="do_not_push_enlp"):
        super(BgpSrteExplicitNullLabelPolicySubTlv, self).__init__()
        self._parent = parent
        self._set_property("explicit_null_label_policy", explicit_null_label_policy)

    @property
    def explicit_null_label_policy(self):
        # type: () -> Union[Literal["do_not_push_enlp"], Literal["push_ipv4_enlp"], Literal["push_ipv4_ipv6_enlp"], Literal["push_ipv6_enlp"], Literal["reserved_enlp"]]
        """explicit_null_label_policy getter

        The value of the explicit null label policy

        Returns: Union[Literal["do_not_push_enlp"], Literal["push_ipv4_enlp"], Literal["push_ipv4_ipv6_enlp"], Literal["push_ipv6_enlp"], Literal["reserved_enlp"]]
        """
        return self._get_property("explicit_null_label_policy")

    @explicit_null_label_policy.setter
    def explicit_null_label_policy(self, value):
        """explicit_null_label_policy setter

        The value of the explicit null label policy

        value: Union[Literal["do_not_push_enlp"], Literal["push_ipv4_enlp"], Literal["push_ipv4_ipv6_enlp"], Literal["push_ipv6_enlp"], Literal["reserved_enlp"]]
        """
        self._set_property("explicit_null_label_policy", value)


class BgpSrteSegmentList(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "weight": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "segments": {"type": "BgpSrteSegmentIter"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "weight": 0,
        "active": True,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, weight=0, name=None, active=True):
        super(BgpSrteSegmentList, self).__init__()
        self._parent = parent
        self._set_property("weight", weight)
        self._set_property("name", name)
        self._set_property("active", active)

    @property
    def weight(self):
        # type: () -> int
        """weight getter

        The Weight associated with a given path and the sub-TLV is optional.

        Returns: int
        """
        return self._get_property("weight")

    @weight.setter
    def weight(self, value):
        """weight setter

        The Weight associated with a given path and the sub-TLV is optional.

        value: int
        """
        self._set_property("weight", value)

    @property
    def segments(self):
        # type: () -> BgpSrteSegmentIter
        """segments getter

        TBD

        Returns: BgpSrteSegmentIter
        """
        return self._get_property(
            "segments", BgpSrteSegmentIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteSegment(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "segment_type": {
            "type": str,
            "enum": [
                "type_a",
                "type_b",
                "type_c",
                "type_d",
                "type_e",
                "type_f",
                "type_g",
                "type_h",
                "type_i",
                "type_j",
                "type_k",
            ],
        },
        "type_a": {"type": "BgpSrteSegmentATypeSubTlv"},
        "type_b": {"type": "BgpSrteSegmentBTypeSubTlv"},
        "type_c": {"type": "BgpSrteSegmentCTypeSubTlv"},
        "type_d": {"type": "BgpSrteSegmentDTypeSubTlv"},
        "type_e": {"type": "BgpSrteSegmentETypeSubTlv"},
        "type_f": {"type": "BgpSrteSegmentFTypeSubTlv"},
        "type_g": {"type": "BgpSrteSegmentGTypeSubTlv"},
        "type_h": {"type": "BgpSrteSegmentHTypeSubTlv"},
        "type_i": {"type": "BgpSrteSegmentITypeSubTlv"},
        "type_j": {"type": "BgpSrteSegmentJTypeSubTlv"},
        "type_k": {"type": "BgpSrteSegmentKTypeSubTlv"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("segment_type", "name")  # type: tuple(str)

    _DEFAULTS = {
        "active": True,
    }  # type: Dict[str, Union(type)]

    TYPE_A = "type_a"  # type: str
    TYPE_B = "type_b"  # type: str
    TYPE_C = "type_c"  # type: str
    TYPE_D = "type_d"  # type: str
    TYPE_E = "type_e"  # type: str
    TYPE_F = "type_f"  # type: str
    TYPE_G = "type_g"  # type: str
    TYPE_H = "type_h"  # type: str
    TYPE_I = "type_i"  # type: str
    TYPE_J = "type_j"  # type: str
    TYPE_K = "type_k"  # type: str

    def __init__(self, parent=None, segment_type=None, name=None, active=True):
        super(BgpSrteSegment, self).__init__()
        self._parent = parent
        self._set_property("segment_type", segment_type)
        self._set_property("name", name)
        self._set_property("active", active)

    @property
    def segment_type(self):
        # type: () -> Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]]
        """segment_type getter

        Specify one of the segment type.. https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13. Type A: SID only, in the form of MPLS Label.. Type B: SID only, in the form of IPv6 Address.. Type C: IPv4 Node Address with optional SID.. Type D: IPv6 Node Address with optional SID for SR MPLS.. Type E: IPv4 Address and index with optional SID.. Type F: IPv4 Local and Remote addresses with optional SID.. Type G: IPv6 Address and index for local and remote pair with optional. SID for SR MPLS.. Type H: IPv6 Local and Remote addresses with optional SID for SR MPLS.. Type I: IPv6 Node Address with optional SID for SRv6.. Type J: IPv6 Address and index for local and remote pair with optional. SID for SRv6.. Type K: IPv6 Local and Remote addresses for SRv6.

        Returns: Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]]
        """
        return self._get_property("segment_type")

    @segment_type.setter
    def segment_type(self, value):
        """segment_type setter

        Specify one of the segment type.. https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13. Type A: SID only, in the form of MPLS Label.. Type B: SID only, in the form of IPv6 Address.. Type C: IPv4 Node Address with optional SID.. Type D: IPv6 Node Address with optional SID for SR MPLS.. Type E: IPv4 Address and index with optional SID.. Type F: IPv4 Local and Remote addresses with optional SID.. Type G: IPv6 Address and index for local and remote pair with optional. SID for SR MPLS.. Type H: IPv6 Local and Remote addresses with optional SID for SR MPLS.. Type I: IPv6 Node Address with optional SID for SRv6.. Type J: IPv6 Address and index for local and remote pair with optional. SID for SRv6.. Type K: IPv6 Local and Remote addresses for SRv6.

        value: Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]]
        """
        self._set_property("segment_type", value)

    @property
    def type_a(self):
        # type: () -> BgpSrteSegmentATypeSubTlv
        """type_a getter

        Type A: SID only, in the form of MPLS Label.. Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Type A: SID only, in the form of MPLS Label.. Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.

        Returns: BgpSrteSegmentATypeSubTlv
        """
        return self._get_property("type_a", BgpSrteSegmentATypeSubTlv)

    @property
    def type_b(self):
        # type: () -> BgpSrteSegmentBTypeSubTlv
        """type_b getter

        Type B: SID only, in the form of IPv6 address.Type B: SID only, in the form of IPv6 address.

        Returns: BgpSrteSegmentBTypeSubTlv
        """
        return self._get_property("type_b", BgpSrteSegmentBTypeSubTlv)

    @property
    def type_c(self):
        # type: () -> BgpSrteSegmentCTypeSubTlv
        """type_c getter

        Status: under-review. Type C: IPv4 Node Address with optional SID.Status: under-review. Type C: IPv4 Node Address with optional SID.

        Returns: BgpSrteSegmentCTypeSubTlv
        """
        return self._get_property("type_c", BgpSrteSegmentCTypeSubTlv)

    @property
    def type_d(self):
        # type: () -> BgpSrteSegmentDTypeSubTlv
        """type_d getter

        Type D: IPv6 Node Address with optional SID for SR MPLS.Type D: IPv6 Node Address with optional SID for SR MPLS.

        Returns: BgpSrteSegmentDTypeSubTlv
        """
        return self._get_property("type_d", BgpSrteSegmentDTypeSubTlv)

    @property
    def type_e(self):
        # type: () -> BgpSrteSegmentETypeSubTlv
        """type_e getter

        Type E: IPv4 Address and Local Interface ID with optional SIDType E: IPv4 Address and Local Interface ID with optional SID

        Returns: BgpSrteSegmentETypeSubTlv
        """
        return self._get_property("type_e", BgpSrteSegmentETypeSubTlv)

    @property
    def type_f(self):
        # type: () -> BgpSrteSegmentFTypeSubTlv
        """type_f getter

        Type F: IPv4 Local and Remote addresses with optional SID.Type F: IPv4 Local and Remote addresses with optional SID.

        Returns: BgpSrteSegmentFTypeSubTlv
        """
        return self._get_property("type_f", BgpSrteSegmentFTypeSubTlv)

    @property
    def type_g(self):
        # type: () -> BgpSrteSegmentGTypeSubTlv
        """type_g getter

        Type G: IPv6 Address, Interface ID for local and remote pair with optional SID for SR MPLS.Type G: IPv6 Address, Interface ID for local and remote pair with optional SID for SR MPLS.

        Returns: BgpSrteSegmentGTypeSubTlv
        """
        return self._get_property("type_g", BgpSrteSegmentGTypeSubTlv)

    @property
    def type_h(self):
        # type: () -> BgpSrteSegmentHTypeSubTlv
        """type_h getter

        Type H: IPv6 Local and Remote addresses with optional SID for SR MPLS.Type H: IPv6 Local and Remote addresses with optional SID for SR MPLS.

        Returns: BgpSrteSegmentHTypeSubTlv
        """
        return self._get_property("type_h", BgpSrteSegmentHTypeSubTlv)

    @property
    def type_i(self):
        # type: () -> BgpSrteSegmentITypeSubTlv
        """type_i getter

        Type I: IPv6 Node Address with optional SRv6 SID.Type I: IPv6 Node Address with optional SRv6 SID.

        Returns: BgpSrteSegmentITypeSubTlv
        """
        return self._get_property("type_i", BgpSrteSegmentITypeSubTlv)

    @property
    def type_j(self):
        # type: () -> BgpSrteSegmentJTypeSubTlv
        """type_j getter

        Type J: IPv6 Address, Interface ID for local and remote pair for SRv6 with optional SID.Type J: IPv6 Address, Interface ID for local and remote pair for SRv6 with optional SID.

        Returns: BgpSrteSegmentJTypeSubTlv
        """
        return self._get_property("type_j", BgpSrteSegmentJTypeSubTlv)

    @property
    def type_k(self):
        # type: () -> BgpSrteSegmentKTypeSubTlv
        """type_k getter

        Type K: IPv6 Local and Remote addresses for SRv6 with optional SID.Type K: IPv6 Local and Remote addresses for SRv6 with optional SID.

        Returns: BgpSrteSegmentKTypeSubTlv
        """
        return self._get_property("type_k", BgpSrteSegmentKTypeSubTlv)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteSegmentATypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "label": {
            "type": int,
            "minimum": 0,
            "maximum": 1048575,
        },
        "tc": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "s_bit": {"type": bool},
        "ttl": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, flags=None, label=None, tc=None, s_bit=None, ttl=None
    ):
        super(BgpSrteSegmentATypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("label", label)
        self._set_property("tc", tc)
        self._set_property("s_bit", s_bit)
        self._set_property("ttl", ttl)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def label(self):
        # type: () -> int
        """label getter

        Label value in [0, 2^20 -1].

        Returns: int
        """
        return self._get_property("label")

    @label.setter
    def label(self, value):
        """label setter

        Label value in [0, 2^20 -1].

        value: int
        """
        self._set_property("label", value)

    @property
    def tc(self):
        # type: () -> int
        """tc getter

        Traffic class in bits.

        Returns: int
        """
        return self._get_property("tc")

    @tc.setter
    def tc(self, value):
        """tc setter

        Traffic class in bits.

        value: int
        """
        self._set_property("tc", value)

    @property
    def s_bit(self):
        # type: () -> bool
        """s_bit getter

        Bottom-of-Stack bit.

        Returns: bool
        """
        return self._get_property("s_bit")

    @s_bit.setter
    def s_bit(self, value):
        """s_bit setter

        Bottom-of-Stack bit.

        value: bool
        """
        self._set_property("s_bit", value)

    @property
    def ttl(self):
        # type: () -> int
        """ttl getter

        Time To Live.

        Returns: int
        """
        return self._get_property("ttl")

    @ttl.setter
    def ttl(self, value):
        """ttl setter

        Time To Live.

        value: int
        """
        self._set_property("ttl", value)


class BgpSrteSegmentBTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "srv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid_endpoint_behavior": {
            "type": "BgpSrteSRv6SIDEndpointBehaviorAndStructure"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("srv6_sid",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, flags=None, srv6_sid=None):
        super(BgpSrteSegmentBTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("srv6_sid", srv6_sid)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def srv6_sid(self):
        # type: () -> str
        """srv6_sid getter

        SRv6 SID.

        Returns: str
        """
        return self._get_property("srv6_sid")

    @srv6_sid.setter
    def srv6_sid(self, value):
        """srv6_sid setter

        SRv6 SID.

        value: str
        """
        self._set_property("srv6_sid", value)

    @property
    def srv6_sid_endpoint_behavior(self):
        # type: () -> BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """srv6_sid_endpoint_behavior getter

        Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Optional SRv6 Endpoint Behavior and SID Structure.

        Returns: BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """
        return self._get_property(
            "srv6_sid_endpoint_behavior", BgpSrteSRv6SIDEndpointBehaviorAndStructure
        )


class BgpSrteSRv6SIDEndpointBehaviorAndStructure(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "lb_length": {
            "type": int,
            "minimum": 0,
            "maximum": 128,
        },
        "ln_length": {
            "type": int,
            "minimum": 0,
            "maximum": 128,
        },
        "func_length": {
            "type": int,
            "minimum": 0,
            "maximum": 128,
        },
        "arg_length": {
            "type": int,
            "minimum": 0,
            "maximum": 128,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "lb_length": 0,
        "ln_length": 0,
        "func_length": 0,
        "arg_length": 0,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, lb_length=0, ln_length=0, func_length=0, arg_length=0
    ):
        super(BgpSrteSRv6SIDEndpointBehaviorAndStructure, self).__init__()
        self._parent = parent
        self._set_property("lb_length", lb_length)
        self._set_property("ln_length", ln_length)
        self._set_property("func_length", func_length)
        self._set_property("arg_length", arg_length)

    @property
    def lb_length(self):
        # type: () -> int
        """lb_length getter

        SRv6 SID Locator Block length in bits.

        Returns: int
        """
        return self._get_property("lb_length")

    @lb_length.setter
    def lb_length(self, value):
        """lb_length setter

        SRv6 SID Locator Block length in bits.

        value: int
        """
        self._set_property("lb_length", value)

    @property
    def ln_length(self):
        # type: () -> int
        """ln_length getter

        SRv6 SID Locator Node length in bits.

        Returns: int
        """
        return self._get_property("ln_length")

    @ln_length.setter
    def ln_length(self, value):
        """ln_length setter

        SRv6 SID Locator Node length in bits.

        value: int
        """
        self._set_property("ln_length", value)

    @property
    def func_length(self):
        # type: () -> int
        """func_length getter

        SRv6 SID Function length in bits.

        Returns: int
        """
        return self._get_property("func_length")

    @func_length.setter
    def func_length(self, value):
        """func_length setter

        SRv6 SID Function length in bits.

        value: int
        """
        self._set_property("func_length", value)

    @property
    def arg_length(self):
        # type: () -> int
        """arg_length getter

        SRv6 SID Arguments length in bits.

        Returns: int
        """
        return self._get_property("arg_length")

    @arg_length.setter
    def arg_length(self, value):
        """arg_length setter

        SRv6 SID Arguments length in bits.

        value: int
        """
        self._set_property("arg_length", value)


class BgpSrteSegmentCTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "sr_algorithm": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "ipv4_node_address": {
            "type": str,
            "format": "ipv4",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv4_node_address",)  # type: tuple(str)

    _DEFAULTS = {
        "sr_algorithm": 0,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, flags=None, sr_algorithm=0, ipv4_node_address=None):
        super(BgpSrteSegmentCTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("sr_algorithm", sr_algorithm)
        self._set_property("ipv4_node_address", ipv4_node_address)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def sr_algorithm(self):
        # type: () -> int
        """sr_algorithm getter

        SR Algorithm identifier when A-Flag in on.

        Returns: int
        """
        return self._get_property("sr_algorithm")

    @sr_algorithm.setter
    def sr_algorithm(self, value):
        """sr_algorithm setter

        SR Algorithm identifier when A-Flag in on.

        value: int
        """
        self._set_property("sr_algorithm", value)

    @property
    def ipv4_node_address(self):
        # type: () -> str
        """ipv4_node_address getter

        IPv4 address representing a node.

        Returns: str
        """
        return self._get_property("ipv4_node_address")

    @ipv4_node_address.setter
    def ipv4_node_address(self, value):
        """ipv4_node_address setter

        IPv4 address representing a node.

        value: str
        """
        self._set_property("ipv4_node_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSrMplsSid(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "label": {
            "type": int,
            "minimum": 0,
            "maximum": 1048575,
        },
        "tc": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "s_bit": {"type": bool},
        "ttl": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, label=None, tc=None, s_bit=None, ttl=None):
        super(BgpSrteSrMplsSid, self).__init__()
        self._parent = parent
        self._set_property("label", label)
        self._set_property("tc", tc)
        self._set_property("s_bit", s_bit)
        self._set_property("ttl", ttl)

    @property
    def label(self):
        # type: () -> int
        """label getter

        Label value in [0, 2^20 -1].

        Returns: int
        """
        return self._get_property("label")

    @label.setter
    def label(self, value):
        """label setter

        Label value in [0, 2^20 -1].

        value: int
        """
        self._set_property("label", value)

    @property
    def tc(self):
        # type: () -> int
        """tc getter

        Traffic class in bits.

        Returns: int
        """
        return self._get_property("tc")

    @tc.setter
    def tc(self, value):
        """tc setter

        Traffic class in bits.

        value: int
        """
        self._set_property("tc", value)

    @property
    def s_bit(self):
        # type: () -> bool
        """s_bit getter

        Bottom-of-Stack bit.

        Returns: bool
        """
        return self._get_property("s_bit")

    @s_bit.setter
    def s_bit(self, value):
        """s_bit setter

        Bottom-of-Stack bit.

        value: bool
        """
        self._set_property("s_bit", value)

    @property
    def ttl(self):
        # type: () -> int
        """ttl getter

        Time To Live.

        Returns: int
        """
        return self._get_property("ttl")

    @ttl.setter
    def ttl(self, value):
        """ttl setter

        Time To Live.

        value: int
        """
        self._set_property("ttl", value)


class BgpSrteSegmentDTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "sr_algorithm": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv6_node_address",)  # type: tuple(str)

    _DEFAULTS = {
        "sr_algorithm": 0,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, flags=None, sr_algorithm=0, ipv6_node_address=None):
        super(BgpSrteSegmentDTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("sr_algorithm", sr_algorithm)
        self._set_property("ipv6_node_address", ipv6_node_address)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def sr_algorithm(self):
        # type: () -> int
        """sr_algorithm getter

        specifying SR Algorithm when when A-Flag as defined in above flags.

        Returns: int
        """
        return self._get_property("sr_algorithm")

    @sr_algorithm.setter
    def sr_algorithm(self, value):
        """sr_algorithm setter

        specifying SR Algorithm when when A-Flag as defined in above flags.

        value: int
        """
        self._set_property("sr_algorithm", value)

    @property
    def ipv6_node_address(self):
        # type: () -> str
        """ipv6_node_address getter

        IPv6 address representing a node.

        Returns: str
        """
        return self._get_property("ipv6_node_address")

    @ipv6_node_address.setter
    def ipv6_node_address(self, value):
        """ipv6_node_address setter

        IPv6 address representing a node.

        value: str
        """
        self._set_property("ipv6_node_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSegmentETypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "local_interface_id": {
            "type": int,
            "minimum": 0,
            "maximum": 2147483647,
        },
        "ipv4_node_address": {
            "type": str,
            "format": "ipv4",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv4_node_address",)  # type: tuple(str)

    _DEFAULTS = {
        "local_interface_id": 0,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, flags=None, local_interface_id=0, ipv4_node_address=None
    ):
        super(BgpSrteSegmentETypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("local_interface_id", local_interface_id)
        self._set_property("ipv4_node_address", ipv4_node_address)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def local_interface_id(self):
        # type: () -> int
        """local_interface_id getter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        Returns: int
        """
        return self._get_property("local_interface_id")

    @local_interface_id.setter
    def local_interface_id(self, value):
        """local_interface_id setter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        value: int
        """
        self._set_property("local_interface_id", value)

    @property
    def ipv4_node_address(self):
        # type: () -> str
        """ipv4_node_address getter

        IPv4 address representing a node.

        Returns: str
        """
        return self._get_property("ipv4_node_address")

    @ipv4_node_address.setter
    def ipv4_node_address(self, value):
        """ipv4_node_address setter

        IPv4 address representing a node.

        value: str
        """
        self._set_property("ipv4_node_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSegmentFTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "local_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "remote_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = ("local_ipv4_address", "remote_ipv4_address")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, flags=None, local_ipv4_address=None, remote_ipv4_address=None
    ):
        super(BgpSrteSegmentFTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("local_ipv4_address", local_ipv4_address)
        self._set_property("remote_ipv4_address", remote_ipv4_address)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def local_ipv4_address(self):
        # type: () -> str
        """local_ipv4_address getter

        Local IPv4 Address.

        Returns: str
        """
        return self._get_property("local_ipv4_address")

    @local_ipv4_address.setter
    def local_ipv4_address(self, value):
        """local_ipv4_address setter

        Local IPv4 Address.

        value: str
        """
        self._set_property("local_ipv4_address", value)

    @property
    def remote_ipv4_address(self):
        # type: () -> str
        """remote_ipv4_address getter

        Remote IPv4 Address.

        Returns: str
        """
        return self._get_property("remote_ipv4_address")

    @remote_ipv4_address.setter
    def remote_ipv4_address(self, value):
        """remote_ipv4_address setter

        Remote IPv4 Address.

        value: str
        """
        self._set_property("remote_ipv4_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSegmentGTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "local_interface_id": {
            "type": int,
            "minimum": 0,
            "maximum": 2147483647,
        },
        "local_ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "remote_interface_id": {
            "type": int,
            "minimum": 0,
            "maximum": 2147483647,
        },
        "remote_ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = (
        "local_ipv6_node_address",
        "remote_ipv6_node_address",
    )  # type: tuple(str)

    _DEFAULTS = {
        "local_interface_id": 0,
        "remote_interface_id": 0,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        flags=None,
        local_interface_id=0,
        local_ipv6_node_address=None,
        remote_interface_id=0,
        remote_ipv6_node_address=None,
    ):
        super(BgpSrteSegmentGTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("local_interface_id", local_interface_id)
        self._set_property("local_ipv6_node_address", local_ipv6_node_address)
        self._set_property("remote_interface_id", remote_interface_id)
        self._set_property("remote_ipv6_node_address", remote_ipv6_node_address)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def local_interface_id(self):
        # type: () -> int
        """local_interface_id getter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        Returns: int
        """
        return self._get_property("local_interface_id")

    @local_interface_id.setter
    def local_interface_id(self, value):
        """local_interface_id setter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        value: int
        """
        self._set_property("local_interface_id", value)

    @property
    def local_ipv6_node_address(self):
        # type: () -> str
        """local_ipv6_node_address getter

        IPv6 address representing a node.

        Returns: str
        """
        return self._get_property("local_ipv6_node_address")

    @local_ipv6_node_address.setter
    def local_ipv6_node_address(self, value):
        """local_ipv6_node_address setter

        IPv6 address representing a node.

        value: str
        """
        self._set_property("local_ipv6_node_address", value)

    @property
    def remote_interface_id(self):
        # type: () -> int
        """remote_interface_id getter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        Returns: int
        """
        return self._get_property("remote_interface_id")

    @remote_interface_id.setter
    def remote_interface_id(self, value):
        """remote_interface_id setter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        value: int
        """
        self._set_property("remote_interface_id", value)

    @property
    def remote_ipv6_node_address(self):
        # type: () -> str
        """remote_ipv6_node_address getter

        IPv6 address representing a node.

        Returns: str
        """
        return self._get_property("remote_ipv6_node_address")

    @remote_ipv6_node_address.setter
    def remote_ipv6_node_address(self, value):
        """remote_ipv6_node_address setter

        IPv6 address representing a node.

        value: str
        """
        self._set_property("remote_ipv6_node_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSegmentHTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "local_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "remote_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = ("local_ipv6_address", "remote_ipv6_address")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, flags=None, local_ipv6_address=None, remote_ipv6_address=None
    ):
        super(BgpSrteSegmentHTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("local_ipv6_address", local_ipv6_address)
        self._set_property("remote_ipv6_address", remote_ipv6_address)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def local_ipv6_address(self):
        # type: () -> str
        """local_ipv6_address getter

        Local IPv6 Address.

        Returns: str
        """
        return self._get_property("local_ipv6_address")

    @local_ipv6_address.setter
    def local_ipv6_address(self, value):
        """local_ipv6_address setter

        Local IPv6 Address.

        value: str
        """
        self._set_property("local_ipv6_address", value)

    @property
    def remote_ipv6_address(self):
        # type: () -> str
        """remote_ipv6_address getter

        Remote IPv6 Address.

        Returns: str
        """
        return self._get_property("remote_ipv6_address")

    @remote_ipv6_address.setter
    def remote_ipv6_address(self, value):
        """remote_ipv6_address setter

        Remote IPv6 Address.

        value: str
        """
        self._set_property("remote_ipv6_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSegmentITypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid_endpoint_behavior": {
            "type": "BgpSrteSRv6SIDEndpointBehaviorAndStructure"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv6_node_address",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, flags=None, ipv6_node_address=None, srv6_sid=None):
        super(BgpSrteSegmentITypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("ipv6_node_address", ipv6_node_address)
        self._set_property("srv6_sid", srv6_sid)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def ipv6_node_address(self):
        # type: () -> str
        """ipv6_node_address getter

        IPv6 address representing a node.

        Returns: str
        """
        return self._get_property("ipv6_node_address")

    @ipv6_node_address.setter
    def ipv6_node_address(self, value):
        """ipv6_node_address setter

        IPv6 address representing a node.

        value: str
        """
        self._set_property("ipv6_node_address", value)

    @property
    def srv6_sid(self):
        # type: () -> str
        """srv6_sid getter

        Optional SRv6 SID.

        Returns: str
        """
        return self._get_property("srv6_sid")

    @srv6_sid.setter
    def srv6_sid(self, value):
        """srv6_sid setter

        Optional SRv6 SID.

        value: str
        """
        self._set_property("srv6_sid", value)

    @property
    def srv6_sid_endpoint_behavior(self):
        # type: () -> BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """srv6_sid_endpoint_behavior getter

        Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Optional SRv6 Endpoint Behavior and SID Structure.

        Returns: BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """
        return self._get_property(
            "srv6_sid_endpoint_behavior", BgpSrteSRv6SIDEndpointBehaviorAndStructure
        )


class BgpSrteSegmentJTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "sr_algorithm": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "local_interface_id": {
            "type": int,
            "minimum": 0,
            "maximum": 2147483647,
        },
        "local_ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "remote_interface_id": {
            "type": int,
            "minimum": 0,
            "maximum": 2147483647,
        },
        "remote_ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid_endpoint_behavior": {
            "type": "BgpSrteSRv6SIDEndpointBehaviorAndStructure"
        },
    }  # type: Dict[str, str]

    _REQUIRED = (
        "local_ipv6_node_address",
        "remote_ipv6_node_address",
    )  # type: tuple(str)

    _DEFAULTS = {
        "sr_algorithm": 0,
        "local_interface_id": 0,
        "remote_interface_id": 0,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        flags=None,
        sr_algorithm=0,
        local_interface_id=0,
        local_ipv6_node_address=None,
        remote_interface_id=0,
        remote_ipv6_node_address=None,
        srv6_sid=None,
    ):
        super(BgpSrteSegmentJTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("sr_algorithm", sr_algorithm)
        self._set_property("local_interface_id", local_interface_id)
        self._set_property("local_ipv6_node_address", local_ipv6_node_address)
        self._set_property("remote_interface_id", remote_interface_id)
        self._set_property("remote_ipv6_node_address", remote_ipv6_node_address)
        self._set_property("srv6_sid", srv6_sid)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def sr_algorithm(self):
        # type: () -> int
        """sr_algorithm getter

        SR Algorithm identifier when A-Flag in on.

        Returns: int
        """
        return self._get_property("sr_algorithm")

    @sr_algorithm.setter
    def sr_algorithm(self, value):
        """sr_algorithm setter

        SR Algorithm identifier when A-Flag in on.

        value: int
        """
        self._set_property("sr_algorithm", value)

    @property
    def local_interface_id(self):
        # type: () -> int
        """local_interface_id getter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        Returns: int
        """
        return self._get_property("local_interface_id")

    @local_interface_id.setter
    def local_interface_id(self, value):
        """local_interface_id setter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        value: int
        """
        self._set_property("local_interface_id", value)

    @property
    def local_ipv6_node_address(self):
        # type: () -> str
        """local_ipv6_node_address getter

        IPv6 address representing a node.

        Returns: str
        """
        return self._get_property("local_ipv6_node_address")

    @local_ipv6_node_address.setter
    def local_ipv6_node_address(self, value):
        """local_ipv6_node_address setter

        IPv6 address representing a node.

        value: str
        """
        self._set_property("local_ipv6_node_address", value)

    @property
    def remote_interface_id(self):
        # type: () -> int
        """remote_interface_id getter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        Returns: int
        """
        return self._get_property("remote_interface_id")

    @remote_interface_id.setter
    def remote_interface_id(self, value):
        """remote_interface_id setter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        value: int
        """
        self._set_property("remote_interface_id", value)

    @property
    def remote_ipv6_node_address(self):
        # type: () -> str
        """remote_ipv6_node_address getter

        IPv6 address representing a node.

        Returns: str
        """
        return self._get_property("remote_ipv6_node_address")

    @remote_ipv6_node_address.setter
    def remote_ipv6_node_address(self, value):
        """remote_ipv6_node_address setter

        IPv6 address representing a node.

        value: str
        """
        self._set_property("remote_ipv6_node_address", value)

    @property
    def srv6_sid(self):
        # type: () -> str
        """srv6_sid getter

        Optional SRv6 SID.

        Returns: str
        """
        return self._get_property("srv6_sid")

    @srv6_sid.setter
    def srv6_sid(self, value):
        """srv6_sid setter

        Optional SRv6 SID.

        value: str
        """
        self._set_property("srv6_sid", value)

    @property
    def srv6_sid_endpoint_behavior(self):
        # type: () -> BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """srv6_sid_endpoint_behavior getter

        Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Optional SRv6 Endpoint Behavior and SID Structure.

        Returns: BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """
        return self._get_property(
            "srv6_sid_endpoint_behavior", BgpSrteSRv6SIDEndpointBehaviorAndStructure
        )


class BgpSrteSegmentKTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "sr_algorithm": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "local_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "remote_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid_endpoint_behavior": {
            "type": "BgpSrteSRv6SIDEndpointBehaviorAndStructure"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("local_ipv6_address", "remote_ipv6_address")  # type: tuple(str)

    _DEFAULTS = {
        "sr_algorithm": 0,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        flags=None,
        sr_algorithm=0,
        local_ipv6_address=None,
        remote_ipv6_address=None,
        srv6_sid=None,
    ):
        super(BgpSrteSegmentKTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("sr_algorithm", sr_algorithm)
        self._set_property("local_ipv6_address", local_ipv6_address)
        self._set_property("remote_ipv6_address", remote_ipv6_address)
        self._set_property("srv6_sid", srv6_sid)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def sr_algorithm(self):
        # type: () -> int
        """sr_algorithm getter

        SR Algorithm identifier when A-Flag in on.

        Returns: int
        """
        return self._get_property("sr_algorithm")

    @sr_algorithm.setter
    def sr_algorithm(self, value):
        """sr_algorithm setter

        SR Algorithm identifier when A-Flag in on.

        value: int
        """
        self._set_property("sr_algorithm", value)

    @property
    def local_ipv6_address(self):
        # type: () -> str
        """local_ipv6_address getter

        IPv6 address representing a node.

        Returns: str
        """
        return self._get_property("local_ipv6_address")

    @local_ipv6_address.setter
    def local_ipv6_address(self, value):
        """local_ipv6_address setter

        IPv6 address representing a node.

        value: str
        """
        self._set_property("local_ipv6_address", value)

    @property
    def remote_ipv6_address(self):
        # type: () -> str
        """remote_ipv6_address getter

        IPv6 address representing a node.

        Returns: str
        """
        return self._get_property("remote_ipv6_address")

    @remote_ipv6_address.setter
    def remote_ipv6_address(self, value):
        """remote_ipv6_address setter

        IPv6 address representing a node.

        value: str
        """
        self._set_property("remote_ipv6_address", value)

    @property
    def srv6_sid(self):
        # type: () -> str
        """srv6_sid getter

        Optional SRv6 SID.

        Returns: str
        """
        return self._get_property("srv6_sid")

    @srv6_sid.setter
    def srv6_sid(self, value):
        """srv6_sid setter

        Optional SRv6 SID.

        value: str
        """
        self._set_property("srv6_sid", value)

    @property
    def srv6_sid_endpoint_behavior(self):
        # type: () -> BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """srv6_sid_endpoint_behavior getter

        Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Optional SRv6 Endpoint Behavior and SID Structure.

        Returns: BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """
        return self._get_property(
            "srv6_sid_endpoint_behavior", BgpSrteSRv6SIDEndpointBehaviorAndStructure
        )


class BgpSrteSegmentIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteSegment
        return self._next()

    def next(self):
        # type: () -> BgpSrteSegment
        return self._next()

    def segment(self, segment_type=None, name=None, active=True):
        # type: (Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]],str,bool) -> BgpSrteSegmentIter
        """Factory method that creates an instance of the BgpSrteSegment class

        Status: under-review. A Segment sub-TLV describes a single segment in a segment list i.e., a single element of the explicit path. The Segment sub-TLVs are optional.

        Returns: BgpSrteSegmentIter
        """
        item = BgpSrteSegment(
            parent=self._parent, segment_type=segment_type, name=name, active=active
        )
        self._add(item)
        return self

    def add(self, segment_type=None, name=None, active=True):
        # type: (Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]],str,bool) -> BgpSrteSegment
        """Add method that creates and returns an instance of the BgpSrteSegment class

        Status: under-review. A Segment sub-TLV describes a single segment in a segment list i.e., a single element of the explicit path. The Segment sub-TLVs are optional.

        Returns: BgpSrteSegment
        """
        item = BgpSrteSegment(
            parent=self._parent, segment_type=segment_type, name=name, active=active
        )
        self._add(item)
        return item


class BgpSrteSegmentListIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteSegmentListIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteSegmentList]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteSegmentListIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteSegmentList
        return self._next()

    def next(self):
        # type: () -> BgpSrteSegmentList
        return self._next()

    def segmentlist(self, weight=0, name=None, active=True):
        # type: (int,str,bool) -> BgpSrteSegmentListIter
        """Factory method that creates an instance of the BgpSrteSegmentList class

        Status: under-review. Optional configuration for BGP SR TE Policy segment list. The Segment List sub-TLV encodes a single explicit path towards the Endpoint.

        Returns: BgpSrteSegmentListIter
        """
        item = BgpSrteSegmentList(
            parent=self._parent, weight=weight, name=name, active=active
        )
        self._add(item)
        return self

    def add(self, weight=0, name=None, active=True):
        # type: (int,str,bool) -> BgpSrteSegmentList
        """Add method that creates and returns an instance of the BgpSrteSegmentList class

        Status: under-review. Optional configuration for BGP SR TE Policy segment list. The Segment List sub-TLV encodes a single explicit path towards the Endpoint.

        Returns: BgpSrteSegmentList
        """
        item = BgpSrteSegmentList(
            parent=self._parent, weight=weight, name=name, active=active
        )
        self._add(item)
        return item


class BgpSrteV4TunnelTlvIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteV4TunnelTlvIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteV4TunnelTlv]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteV4TunnelTlvIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteV4TunnelTlv
        return self._next()

    def next(self):
        # type: () -> BgpSrteV4TunnelTlv
        return self._next()

    def v4tunneltlv(self, name=None, active=True):
        # type: (str,bool) -> BgpSrteV4TunnelTlvIter
        """Factory method that creates an instance of the BgpSrteV4TunnelTlv class

        Configuration for BGP SRTE Tunnel TLV.

        Returns: BgpSrteV4TunnelTlvIter
        """
        item = BgpSrteV4TunnelTlv(parent=self._parent, name=name, active=active)
        self._add(item)
        return self

    def add(self, name=None, active=True):
        # type: (str,bool) -> BgpSrteV4TunnelTlv
        """Add method that creates and returns an instance of the BgpSrteV4TunnelTlv class

        Configuration for BGP SRTE Tunnel TLV.

        Returns: BgpSrteV4TunnelTlv
        """
        item = BgpSrteV4TunnelTlv(parent=self._parent, name=name, active=active)
        self._add(item)
        return item


class BgpSrteV4PolicyIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteV4PolicyIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteV4Policy]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteV4PolicyIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteV4Policy
        return self._next()

    def next(self):
        # type: () -> BgpSrteV4Policy
        return self._next()

    def v4policy(
        self,
        distinguisher=1,
        color=100,
        ipv4_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address=None,
        next_hop_ipv6_address=None,
        name=None,
        active=True,
    ):
        # type: (int,int,str,Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str,bool) -> BgpSrteV4PolicyIter
        """Factory method that creates an instance of the BgpSrteV4Policy class

        Configuration for BGP Segment Routing Traffic Engineering(SRTE) . policy..

        Returns: BgpSrteV4PolicyIter
        """
        item = BgpSrteV4Policy(
            parent=self._parent,
            distinguisher=distinguisher,
            color=color,
            ipv4_endpoint=ipv4_endpoint,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
            active=active,
        )
        self._add(item)
        return self

    def add(
        self,
        distinguisher=1,
        color=100,
        ipv4_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address=None,
        next_hop_ipv6_address=None,
        name=None,
        active=True,
    ):
        # type: (int,int,str,Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str,bool) -> BgpSrteV4Policy
        """Add method that creates and returns an instance of the BgpSrteV4Policy class

        Configuration for BGP Segment Routing Traffic Engineering(SRTE) . policy..

        Returns: BgpSrteV4Policy
        """
        item = BgpSrteV4Policy(
            parent=self._parent,
            distinguisher=distinguisher,
            color=color,
            ipv4_endpoint=ipv4_endpoint,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
            active=active,
        )
        self._add(item)
        return item


class BgpSrteV6Policy(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "distinguisher": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "color": {
            "type": int,
            "format": "int64",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "ipv6_endpoint": {
            "type": str,
            "format": "ipv6",
        },
        "next_hop_mode": {
            "type": str,
            "enum": [
                "local_ip",
                "manual",
            ],
        },
        "next_hop_address_type": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
            ],
        },
        "next_hop_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "next_hop_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "add_path": {"type": "BgpAddPath"},
        "as_path": {"type": "BgpAsPath"},
        "communities": {"type": "BgpCommunityIter"},
        "extcommunities": {"type": "BgpExtCommunityIter"},
        "tunnel_tlvs": {"type": "BgpSrteV6TunnelTlvIter"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv6_endpoint", "name")  # type: tuple(str)

    _DEFAULTS = {
        "distinguisher": 1,
        "color": 100,
        "next_hop_mode": "local_ip",
        "next_hop_address_type": "ipv6",
        "next_hop_ipv4_address": "0.0.0.0",
        "next_hop_ipv6_address": "::0",
        "active": True,
    }  # type: Dict[str, Union(type)]

    LOCAL_IP = "local_ip"  # type: str
    MANUAL = "manual"  # type: str

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    def __init__(
        self,
        parent=None,
        distinguisher=1,
        color=100,
        ipv6_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
        active=True,
    ):
        super(BgpSrteV6Policy, self).__init__()
        self._parent = parent
        self._set_property("distinguisher", distinguisher)
        self._set_property("color", color)
        self._set_property("ipv6_endpoint", ipv6_endpoint)
        self._set_property("next_hop_mode", next_hop_mode)
        self._set_property("next_hop_address_type", next_hop_address_type)
        self._set_property("next_hop_ipv4_address", next_hop_ipv4_address)
        self._set_property("next_hop_ipv6_address", next_hop_ipv6_address)
        self._set_property("name", name)
        self._set_property("active", active)

    @property
    def distinguisher(self):
        # type: () -> int
        """distinguisher getter

        Identifies the policy in the context of (color and endpoint) tuple. It is used by the SR Policy originator to make unique multiple occurrences of the same SR Policy.

        Returns: int
        """
        return self._get_property("distinguisher")

    @distinguisher.setter
    def distinguisher(self, value):
        """distinguisher setter

        Identifies the policy in the context of (color and endpoint) tuple. It is used by the SR Policy originator to make unique multiple occurrences of the same SR Policy.

        value: int
        """
        self._set_property("distinguisher", value)

    @property
    def color(self):
        # type: () -> int
        """color getter

        Identifies the policy. It is used to match the color of the destination prefixes to steer traffic into the SR Policy.

        Returns: int
        """
        return self._get_property("color")

    @color.setter
    def color(self, value):
        """color setter

        Identifies the policy. It is used to match the color of the destination prefixes to steer traffic into the SR Policy.

        value: int
        """
        self._set_property("color", value)

    @property
    def ipv6_endpoint(self):
        # type: () -> str
        """ipv6_endpoint getter

        Specifies a single node or a set of nodes (e.g., an anycast address). It is selected on the basis of the SR Policy type (AFI).

        Returns: str
        """
        return self._get_property("ipv6_endpoint")

    @ipv6_endpoint.setter
    def ipv6_endpoint(self, value):
        """ipv6_endpoint setter

        Specifies a single node or a set of nodes (e.g., an anycast address). It is selected on the basis of the SR Policy type (AFI).

        value: str
        """
        self._set_property("ipv6_endpoint", value)

    @property
    def next_hop_mode(self):
        # type: () -> Union[Literal["local_ip"], Literal["manual"]]
        """next_hop_mode getter

        Mode for choosing the NextHop in MP REACH NLRI. Available modes are : Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer. For IPv6 BGP peer the Nexthop Encoding capability should be enabled. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        Returns: Union[Literal["local_ip"], Literal["manual"]]
        """
        return self._get_property("next_hop_mode")

    @next_hop_mode.setter
    def next_hop_mode(self, value):
        """next_hop_mode setter

        Mode for choosing the NextHop in MP REACH NLRI. Available modes are : Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer. For IPv6 BGP peer the Nexthop Encoding capability should be enabled. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        value: Union[Literal["local_ip"], Literal["manual"]]
        """
        self._set_property("next_hop_mode", value)

    @property
    def next_hop_address_type(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """next_hop_address_type getter

        Type of next hop IP address to be used when 'next_hop_mode' is set to 'manual'.

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property("next_hop_address_type")

    @next_hop_address_type.setter
    def next_hop_address_type(self, value):
        """next_hop_address_type setter

        Type of next hop IP address to be used when 'next_hop_mode' is set to 'manual'.

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property("next_hop_address_type", value)

    @property
    def next_hop_ipv4_address(self):
        # type: () -> str
        """next_hop_ipv4_address getter

        The IPv4 address of the Nexthop if the 'next_hop_mode' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability extended_next_hop_encoding should be enabled.

        Returns: str
        """
        return self._get_property("next_hop_ipv4_address")

    @next_hop_ipv4_address.setter
    def next_hop_ipv4_address(self, value):
        """next_hop_ipv4_address setter

        The IPv4 address of the Nexthop if the 'next_hop_mode' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability extended_next_hop_encoding should be enabled.

        value: str
        """
        self._set_property("next_hop_ipv4_address", value)

    @property
    def next_hop_ipv6_address(self):
        # type: () -> str
        """next_hop_ipv6_address getter

        The IPv6 address of the next hop if the Nexthop Mode 'next_hop_address_type' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv6.

        Returns: str
        """
        return self._get_property("next_hop_ipv6_address")

    @next_hop_ipv6_address.setter
    def next_hop_ipv6_address(self, value):
        """next_hop_ipv6_address setter

        The IPv6 address of the next hop if the Nexthop Mode 'next_hop_address_type' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv6.

        value: str
        """
        self._set_property("next_hop_ipv6_address", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def add_path(self):
        # type: () -> BgpAddPath
        """add_path getter

        The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is a BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: BgpAddPath
        """
        return self._get_property("add_path", BgpAddPath)

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def extcommunities(self):
        # type: () -> BgpExtCommunityIter
        """extcommunities getter

        Optional Extended Community settings.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "extcommunities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def tunnel_tlvs(self):
        # type: () -> BgpSrteV6TunnelTlvIter
        """tunnel_tlvs getter

        List of optional tunnel TLV settings.

        Returns: BgpSrteV6TunnelTlvIter
        """
        return self._get_property(
            "tunnel_tlvs", BgpSrteV6TunnelTlvIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteV6TunnelTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "remote_endpoint_sub_tlv": {"type": "BgpSrteRemoteEndpointSubTlv"},
        "color_sub_tlv": {"type": "BgpSrteColorSubTlv"},
        "binding_sub_tlv": {"type": "BgpSrteBindingSubTlv"},
        "preference_sub_tlv": {"type": "BgpSrtePreferenceSubTlv"},
        "policy_priority_sub_tlv": {"type": "BgpSrtePolicyPrioritySubTlv"},
        "policy_name_sub_tlv": {"type": "BgpSrtePolicyNameSubTlv"},
        "explicit_null_label_policy_sub_tlv": {
            "type": "BgpSrteExplicitNullLabelPolicySubTlv"
        },
        "segment_lists": {"type": "BgpSrteSegmentListIter"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "active": True,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, name=None, active=True):
        super(BgpSrteV6TunnelTlv, self).__init__()
        self._parent = parent
        self._set_property("name", name)
        self._set_property("active", active)

    @property
    def remote_endpoint_sub_tlv(self):
        # type: () -> BgpSrteRemoteEndpointSubTlv
        """remote_endpoint_sub_tlv getter

        Configuration for the BGP remote endpoint sub TLV.Configuration for the BGP remote endpoint sub TLV.

        Returns: BgpSrteRemoteEndpointSubTlv
        """
        return self._get_property(
            "remote_endpoint_sub_tlv", BgpSrteRemoteEndpointSubTlv
        )

    @property
    def color_sub_tlv(self):
        # type: () -> BgpSrteColorSubTlv
        """color_sub_tlv getter

        Configuration for the Policy Color attribute sub-TLV. The Color sub-TLV MAY be used as a way to "color" the corresponding Tunnel TLV. The Value field of the sub-TLV is eight octets long and consists of a Color Extended Community. First two octets of its Value field are 0x030b as type and subtype of extended community. Remaining six octets are are exposed to configure. Configuration for the Policy Color attribute sub-TLV. The Color sub-TLV MAY be used as a way to "color" the corresponding Tunnel TLV. The Value field of the sub-TLV is eight octets long and consists of a Color Extended Community. First two octets of its Value field are 0x030b as type and subtype of extended community. Remaining six octets are are exposed to configure.

        Returns: BgpSrteColorSubTlv
        """
        return self._get_property("color_sub_tlv", BgpSrteColorSubTlv)

    @property
    def binding_sub_tlv(self):
        # type: () -> BgpSrteBindingSubTlv
        """binding_sub_tlv getter

        Configuration for the binding SID sub-TLV. This is used to signal the binding SID related information of the SR Policy candidate path.Configuration for the binding SID sub-TLV. This is used to signal the binding SID related information of the SR Policy candidate path.

        Returns: BgpSrteBindingSubTlv
        """
        return self._get_property("binding_sub_tlv", BgpSrteBindingSubTlv)

    @property
    def preference_sub_tlv(self):
        # type: () -> BgpSrtePreferenceSubTlv
        """preference_sub_tlv getter

        Configuration for BGP preference sub TLV of the SR Policy candidate path.Configuration for BGP preference sub TLV of the SR Policy candidate path.

        Returns: BgpSrtePreferenceSubTlv
        """
        return self._get_property("preference_sub_tlv", BgpSrtePreferenceSubTlv)

    @property
    def policy_priority_sub_tlv(self):
        # type: () -> BgpSrtePolicyPrioritySubTlv
        """policy_priority_sub_tlv getter

        Configuration for the Policy Priority sub-TLV. The Policy Priority to indicate the order in which the SR policies are re-computed upon topological change.Configuration for the Policy Priority sub-TLV. The Policy Priority to indicate the order in which the SR policies are re-computed upon topological change.

        Returns: BgpSrtePolicyPrioritySubTlv
        """
        return self._get_property(
            "policy_priority_sub_tlv", BgpSrtePolicyPrioritySubTlv
        )

    @property
    def policy_name_sub_tlv(self):
        # type: () -> BgpSrtePolicyNameSubTlv
        """policy_name_sub_tlv getter

        Configuration for the Policy Name sub-TLV. The Policy Name sub-TLV is used to attach a symbolic name to the SR Policy candidate path.Configuration for the Policy Name sub-TLV. The Policy Name sub-TLV is used to attach a symbolic name to the SR Policy candidate path.

        Returns: BgpSrtePolicyNameSubTlv
        """
        return self._get_property("policy_name_sub_tlv", BgpSrtePolicyNameSubTlv)

    @property
    def explicit_null_label_policy_sub_tlv(self):
        # type: () -> BgpSrteExplicitNullLabelPolicySubTlv
        """explicit_null_label_policy_sub_tlv getter

        Configuration for BGP explicit null label policy sub TLV settings.Configuration for BGP explicit null label policy sub TLV settings.

        Returns: BgpSrteExplicitNullLabelPolicySubTlv
        """
        return self._get_property(
            "explicit_null_label_policy_sub_tlv", BgpSrteExplicitNullLabelPolicySubTlv
        )

    @property
    def segment_lists(self):
        # type: () -> BgpSrteSegmentListIter
        """segment_lists getter

        TBD

        Returns: BgpSrteSegmentListIter
        """
        return self._get_property(
            "segment_lists", BgpSrteSegmentListIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteV6TunnelTlvIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteV6TunnelTlvIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteV6TunnelTlv]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteV6TunnelTlvIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteV6TunnelTlv
        return self._next()

    def next(self):
        # type: () -> BgpSrteV6TunnelTlv
        return self._next()

    def v6tunneltlv(self, name=None, active=True):
        # type: (str,bool) -> BgpSrteV6TunnelTlvIter
        """Factory method that creates an instance of the BgpSrteV6TunnelTlv class

        Configuration for BGP SRTE Tunnel TLV.

        Returns: BgpSrteV6TunnelTlvIter
        """
        item = BgpSrteV6TunnelTlv(parent=self._parent, name=name, active=active)
        self._add(item)
        return self

    def add(self, name=None, active=True):
        # type: (str,bool) -> BgpSrteV6TunnelTlv
        """Add method that creates and returns an instance of the BgpSrteV6TunnelTlv class

        Configuration for BGP SRTE Tunnel TLV.

        Returns: BgpSrteV6TunnelTlv
        """
        item = BgpSrteV6TunnelTlv(parent=self._parent, name=name, active=active)
        self._add(item)
        return item


class BgpSrteV6PolicyIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteV6PolicyIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteV6Policy]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteV6PolicyIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteV6Policy
        return self._next()

    def next(self):
        # type: () -> BgpSrteV6Policy
        return self._next()

    def v6policy(
        self,
        distinguisher=1,
        color=100,
        ipv6_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
        active=True,
    ):
        # type: (int,int,str,Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str,bool) -> BgpSrteV6PolicyIter
        """Factory method that creates an instance of the BgpSrteV6Policy class

        Configuration for BGP Segment Routing Traffic Engineering policy..

        Returns: BgpSrteV6PolicyIter
        """
        item = BgpSrteV6Policy(
            parent=self._parent,
            distinguisher=distinguisher,
            color=color,
            ipv6_endpoint=ipv6_endpoint,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
            active=active,
        )
        self._add(item)
        return self

    def add(
        self,
        distinguisher=1,
        color=100,
        ipv6_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
        active=True,
    ):
        # type: (int,int,str,Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str,bool) -> BgpSrteV6Policy
        """Add method that creates and returns an instance of the BgpSrteV6Policy class

        Configuration for BGP Segment Routing Traffic Engineering policy..

        Returns: BgpSrteV6Policy
        """
        item = BgpSrteV6Policy(
            parent=self._parent,
            distinguisher=distinguisher,
            color=color,
            ipv6_endpoint=ipv6_endpoint,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
            active=active,
        )
        self._add(item)
        return item


class BgpV4PeerIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV4PeerIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV4Peer]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV4PeerIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV4Peer
        return self._next()

    def next(self):
        # type: () -> BgpV4Peer
        return self._next()

    def v4peer(
        self,
        peer_address=None,
        as_type=None,
        as_number=None,
        as_number_width="four",
        name=None,
    ):
        # type: (str,Union[Literal["ebgp"], Literal["ibgp"]],int,Union[Literal["four"], Literal["two"]],str) -> BgpV4PeerIter
        """Factory method that creates an instance of the BgpV4Peer class

        Status: under-review. Configuration for emulated BGPv4 peers and routes.. Configuration for basic emulated BGP peer settings.

        Returns: BgpV4PeerIter
        """
        item = BgpV4Peer(
            parent=self._parent,
            peer_address=peer_address,
            as_type=as_type,
            as_number=as_number,
            as_number_width=as_number_width,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        peer_address=None,
        as_type=None,
        as_number=None,
        as_number_width="four",
        name=None,
    ):
        # type: (str,Union[Literal["ebgp"], Literal["ibgp"]],int,Union[Literal["four"], Literal["two"]],str) -> BgpV4Peer
        """Add method that creates and returns an instance of the BgpV4Peer class

        Status: under-review. Configuration for emulated BGPv4 peers and routes.. Configuration for basic emulated BGP peer settings.

        Returns: BgpV4Peer
        """
        item = BgpV4Peer(
            parent=self._parent,
            peer_address=peer_address,
            as_type=as_type,
            as_number=as_number,
            as_number_width=as_number_width,
            name=name,
        )
        self._add(item)
        return item


class BgpV4InterfaceIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV4InterfaceIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV4Interface]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV4InterfaceIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV4Interface
        return self._next()

    def next(self):
        # type: () -> BgpV4Interface
        return self._next()

    def v4interface(self, ipv4_name=None):
        # type: (str) -> BgpV4InterfaceIter
        """Factory method that creates an instance of the BgpV4Interface class

        Configuration for emulated BGPv4 peers and routes on a single IPv4 interface.

        Returns: BgpV4InterfaceIter
        """
        item = BgpV4Interface(parent=self._parent, ipv4_name=ipv4_name)
        self._add(item)
        return self

    def add(self, ipv4_name=None):
        # type: (str) -> BgpV4Interface
        """Add method that creates and returns an instance of the BgpV4Interface class

        Configuration for emulated BGPv4 peers and routes on a single IPv4 interface.

        Returns: BgpV4Interface
        """
        item = BgpV4Interface(parent=self._parent, ipv4_name=ipv4_name)
        self._add(item)
        return item


class BgpV6Interface(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ipv6_name": {"type": str},
        "peers": {"type": "BgpV6PeerIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv6_name",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, ipv6_name=None):
        super(BgpV6Interface, self).__init__()
        self._parent = parent
        self._set_property("ipv6_name", ipv6_name)

    @property
    def ipv6_name(self):
        # type: () -> str
        """ipv6_name getter

        The unique name of IPv6 or Loopback IPv6 interface used as the source IP for this list of BGP peers.. . x-constraint:. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Device.Ipv6Loopback/properties/name.

        Returns: str
        """
        return self._get_property("ipv6_name")

    @ipv6_name.setter
    def ipv6_name(self, value):
        """ipv6_name setter

        The unique name of IPv6 or Loopback IPv6 interface used as the source IP for this list of BGP peers.. . x-constraint:. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Device.Ipv6Loopback/properties/name.

        value: str
        """
        self._set_property("ipv6_name", value)

    @property
    def peers(self):
        # type: () -> BgpV6PeerIter
        """peers getter

        This contains the list of BGPv6 peers configured on this interface.

        Returns: BgpV6PeerIter
        """
        return self._get_property("peers", BgpV6PeerIter, self._parent, self._choice)


class BgpV6Peer(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "peer_address": {
            "type": str,
            "format": "ipv6",
        },
        "segment_routing": {"type": "BgpV6SegmentRouting"},
        "evpn_ethernet_segments": {"type": "BgpV6EthernetSegmentIter"},
        "as_type": {
            "type": str,
            "enum": [
                "ibgp",
                "ebgp",
            ],
        },
        "as_number": {"type": int},
        "as_number_width": {
            "type": str,
            "enum": [
                "two",
                "four",
            ],
        },
        "advanced": {"type": "BgpAdvanced"},
        "capability": {"type": "BgpCapability"},
        "learned_information_filter": {"type": "BgpLearnedInformationFilter"},
        "v4_routes": {"type": "BgpV4RouteRangeIter"},
        "v6_routes": {"type": "BgpV6RouteRangeIter"},
        "v4_srte_policies": {"type": "BgpSrteV4PolicyIter"},
        "v6_srte_policies": {"type": "BgpSrteV6PolicyIter"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("peer_address", "as_type", "as_number", "name")  # type: tuple(str)

    _DEFAULTS = {
        "as_number_width": "four",
    }  # type: Dict[str, Union(type)]

    IBGP = "ibgp"  # type: str
    EBGP = "ebgp"  # type: str

    TWO = "two"  # type: str
    FOUR = "four"  # type: str

    def __init__(
        self,
        parent=None,
        peer_address=None,
        as_type=None,
        as_number=None,
        as_number_width="four",
        name=None,
    ):
        super(BgpV6Peer, self).__init__()
        self._parent = parent
        self._set_property("peer_address", peer_address)
        self._set_property("as_type", as_type)
        self._set_property("as_number", as_number)
        self._set_property("as_number_width", as_number_width)
        self._set_property("name", name)

    @property
    def peer_address(self):
        # type: () -> str
        """peer_address getter

        IPv6 address of the BGP peer for the session

        Returns: str
        """
        return self._get_property("peer_address")

    @peer_address.setter
    def peer_address(self, value):
        """peer_address setter

        IPv6 address of the BGP peer for the session

        value: str
        """
        self._set_property("peer_address", value)

    @property
    def segment_routing(self):
        # type: () -> BgpV6SegmentRouting
        """segment_routing getter

        Status: under-review. Configuration for BGPv6 segment routing settings.Status: under-review. Configuration for BGPv6 segment routing settings.

        Returns: BgpV6SegmentRouting
        """
        return self._get_property("segment_routing", BgpV6SegmentRouting)

    @property
    def evpn_ethernet_segments(self):
        # type: () -> BgpV6EthernetSegmentIter
        """evpn_ethernet_segments getter

        This contains the list of Ethernet Virtual Private Network (EVPN) Ethernet Segments (ES) Per BGP Peer for IPv6 Address Family Identifier (AFI).. . Each Ethernet Segment contains a list of EVPN Instances (EVIs) . . Each EVI contains a list of Broadcast Domains. . Each Broadcast Domain contains a list of MAC/IP Ranges. . . <Ethernet Segment, EVI, Broadcast Domain> is responsible for advertising Ethernet Auto-discovery Route Per EVI (Type 1).. . <Ethernet Segment, EVI> is responsible for advertising Ethernet Auto-discovery Route Per Ethernet Segment (Type 1).. . <Ethernet Segment, EVI, Broadcast Domain, MAC/IP> is responsible for advertising MAC/IP Advertisement Route (Type 2).. . <Ethernet Segment, EVI, Broadcast Domain> is responsible for advertising Inclusive Multicast Ethernet Tag Route (Type 3).. . Ethernet Segment is responsible for advertising Ethernet Segment Route (Type 4).

        Returns: BgpV6EthernetSegmentIter
        """
        return self._get_property(
            "evpn_ethernet_segments",
            BgpV6EthernetSegmentIter,
            self._parent,
            self._choice,
        )

    @property
    def as_type(self):
        # type: () -> Union[Literal["ebgp"], Literal["ibgp"]]
        """as_type getter

        The type of BGP autonomous system. External BGP is used for BGP links between two or more autonomous systems (ebgp). Internal BGP is used within a single autonomous system (ibgp). BGP property defaults are aligned with this object defined as an internal BGP peer. If the as_type is specified as 'ebgp' then other properties will need to be specified as per an external BGP peer. Specifically, for 'ebgp', 'as_set_mode' attribute in 'as_path' field in any Route Range should be changed from default value 'do_not_include_local_as' to any other value.

        Returns: Union[Literal["ebgp"], Literal["ibgp"]]
        """
        return self._get_property("as_type")

    @as_type.setter
    def as_type(self, value):
        """as_type setter

        The type of BGP autonomous system. External BGP is used for BGP links between two or more autonomous systems (ebgp). Internal BGP is used within a single autonomous system (ibgp). BGP property defaults are aligned with this object defined as an internal BGP peer. If the as_type is specified as 'ebgp' then other properties will need to be specified as per an external BGP peer. Specifically, for 'ebgp', 'as_set_mode' attribute in 'as_path' field in any Route Range should be changed from default value 'do_not_include_local_as' to any other value.

        value: Union[Literal["ebgp"], Literal["ibgp"]]
        """
        self._set_property("as_type", value)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        Autonomous System Number (AS number or ASN)

        Returns: int
        """
        return self._get_property("as_number")

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        Autonomous System Number (AS number or ASN)

        value: int
        """
        self._set_property("as_number", value)

    @property
    def as_number_width(self):
        # type: () -> Union[Literal["four"], Literal["two"]]
        """as_number_width getter

        The width in bytes of the as_number values. Any as_number values that exceeds the width MUST result in an error.

        Returns: Union[Literal["four"], Literal["two"]]
        """
        return self._get_property("as_number_width")

    @as_number_width.setter
    def as_number_width(self, value):
        """as_number_width setter

        The width in bytes of the as_number values. Any as_number values that exceeds the width MUST result in an error.

        value: Union[Literal["four"], Literal["two"]]
        """
        self._set_property("as_number_width", value)

    @property
    def advanced(self):
        # type: () -> BgpAdvanced
        """advanced getter

        Status: under-review. Configuration for BGP advanced settings.Status: under-review. Configuration for BGP advanced settings.

        Returns: BgpAdvanced
        """
        return self._get_property("advanced", BgpAdvanced)

    @property
    def capability(self):
        # type: () -> BgpCapability
        """capability getter

        Status: under-review. Configuration for BGP capability settings.Status: under-review. Configuration for BGP capability settings.

        Returns: BgpCapability
        """
        return self._get_property("capability", BgpCapability)

    @property
    def learned_information_filter(self):
        # type: () -> BgpLearnedInformationFilter
        """learned_information_filter getter

        Status: under-review. Configuration for controlling storage of BGP learned information recieved from the peer.Status: under-review. Configuration for controlling storage of BGP learned information recieved from the peer.

        Returns: BgpLearnedInformationFilter
        """
        return self._get_property(
            "learned_information_filter", BgpLearnedInformationFilter
        )

    @property
    def v4_routes(self):
        # type: () -> BgpV4RouteRangeIter
        """v4_routes getter

        Emulated BGPv4 route ranges.

        Returns: BgpV4RouteRangeIter
        """
        return self._get_property(
            "v4_routes", BgpV4RouteRangeIter, self._parent, self._choice
        )

    @property
    def v6_routes(self):
        # type: () -> BgpV6RouteRangeIter
        """v6_routes getter

        Emulated BGPv6 route ranges.

        Returns: BgpV6RouteRangeIter
        """
        return self._get_property(
            "v6_routes", BgpV6RouteRangeIter, self._parent, self._choice
        )

    @property
    def v4_srte_policies(self):
        # type: () -> BgpSrteV4PolicyIter
        """v4_srte_policies getter

        Segment Routing Traffic Engineering (SR TE) Policies for IPv4 Address Family Identifier (AFI).

        Returns: BgpSrteV4PolicyIter
        """
        return self._get_property(
            "v4_srte_policies", BgpSrteV4PolicyIter, self._parent, self._choice
        )

    @property
    def v6_srte_policies(self):
        # type: () -> BgpSrteV6PolicyIter
        """v6_srte_policies getter

        Segment Routing Traffic Engineering (SR TE) Policies for IPv6 Address Family Identifier (AFI).

        Returns: BgpSrteV6PolicyIter
        """
        return self._get_property(
            "v6_srte_policies", BgpSrteV6PolicyIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class BgpV6SegmentRouting(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ingress_supports_vpn": {"type": bool},
        "reduced_encapsulation": {"type": bool},
        "copy_time_to_live": {"type": bool},
        "time_to_live": {"type": int},
        "max_sids_per_srh": {"type": int},
        "auto_generate_segment_left_value": {"type": bool},
        "segment_left_value": {"type": int},
        "advertise_sr_te_policy": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "ingress_supports_vpn": False,
        "reduced_encapsulation": False,
        "copy_time_to_live": False,
        "time_to_live": 0,
        "max_sids_per_srh": 0,
        "auto_generate_segment_left_value": False,
        "segment_left_value": 0,
        "advertise_sr_te_policy": False,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        ingress_supports_vpn=False,
        reduced_encapsulation=False,
        copy_time_to_live=False,
        time_to_live=0,
        max_sids_per_srh=0,
        auto_generate_segment_left_value=False,
        segment_left_value=0,
        advertise_sr_te_policy=False,
    ):
        super(BgpV6SegmentRouting, self).__init__()
        self._parent = parent
        self._set_property("ingress_supports_vpn", ingress_supports_vpn)
        self._set_property("reduced_encapsulation", reduced_encapsulation)
        self._set_property("copy_time_to_live", copy_time_to_live)
        self._set_property("time_to_live", time_to_live)
        self._set_property("max_sids_per_srh", max_sids_per_srh)
        self._set_property(
            "auto_generate_segment_left_value", auto_generate_segment_left_value
        )
        self._set_property("segment_left_value", segment_left_value)
        self._set_property("advertise_sr_te_policy", advertise_sr_te_policy)

    @property
    def ingress_supports_vpn(self):
        # type: () -> bool
        """ingress_supports_vpn getter

        TBD

        Returns: bool
        """
        return self._get_property("ingress_supports_vpn")

    @ingress_supports_vpn.setter
    def ingress_supports_vpn(self, value):
        """ingress_supports_vpn setter

        TBD

        value: bool
        """
        self._set_property("ingress_supports_vpn", value)

    @property
    def reduced_encapsulation(self):
        # type: () -> bool
        """reduced_encapsulation getter

        TBD

        Returns: bool
        """
        return self._get_property("reduced_encapsulation")

    @reduced_encapsulation.setter
    def reduced_encapsulation(self, value):
        """reduced_encapsulation setter

        TBD

        value: bool
        """
        self._set_property("reduced_encapsulation", value)

    @property
    def copy_time_to_live(self):
        # type: () -> bool
        """copy_time_to_live getter

        TBD

        Returns: bool
        """
        return self._get_property("copy_time_to_live")

    @copy_time_to_live.setter
    def copy_time_to_live(self, value):
        """copy_time_to_live setter

        TBD

        value: bool
        """
        self._set_property("copy_time_to_live", value)

    @property
    def time_to_live(self):
        # type: () -> int
        """time_to_live getter

        TBD

        Returns: int
        """
        return self._get_property("time_to_live")

    @time_to_live.setter
    def time_to_live(self, value):
        """time_to_live setter

        TBD

        value: int
        """
        self._set_property("time_to_live", value)

    @property
    def max_sids_per_srh(self):
        # type: () -> int
        """max_sids_per_srh getter

        TBD

        Returns: int
        """
        return self._get_property("max_sids_per_srh")

    @max_sids_per_srh.setter
    def max_sids_per_srh(self, value):
        """max_sids_per_srh setter

        TBD

        value: int
        """
        self._set_property("max_sids_per_srh", value)

    @property
    def auto_generate_segment_left_value(self):
        # type: () -> bool
        """auto_generate_segment_left_value getter

        TBD

        Returns: bool
        """
        return self._get_property("auto_generate_segment_left_value")

    @auto_generate_segment_left_value.setter
    def auto_generate_segment_left_value(self, value):
        """auto_generate_segment_left_value setter

        TBD

        value: bool
        """
        self._set_property("auto_generate_segment_left_value", value)

    @property
    def segment_left_value(self):
        # type: () -> int
        """segment_left_value getter

        TBD

        Returns: int
        """
        return self._get_property("segment_left_value")

    @segment_left_value.setter
    def segment_left_value(self, value):
        """segment_left_value setter

        TBD

        value: int
        """
        self._set_property("segment_left_value", value)

    @property
    def advertise_sr_te_policy(self):
        # type: () -> bool
        """advertise_sr_te_policy getter

        TBD

        Returns: bool
        """
        return self._get_property("advertise_sr_te_policy")

    @advertise_sr_te_policy.setter
    def advertise_sr_te_policy(self, value):
        """advertise_sr_te_policy setter

        TBD

        value: bool
        """
        self._set_property("advertise_sr_te_policy", value)


class BgpV6EthernetSegment(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "df_election": {"type": "BgpEthernetSegmentDfElection"},
        "evis": {"type": "BgpV6EvpnEvisIter"},
        "esi": {
            "type": str,
            "format": "hex",
        },
        "active_mode": {
            "type": str,
            "enum": [
                "single_active",
                "all_active",
            ],
        },
        "esi_label": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "esi": "00000000000000000000",
        "active_mode": "all_active",
        "esi_label": 0,
    }  # type: Dict[str, Union(type)]

    SINGLE_ACTIVE = "single_active"  # type: str
    ALL_ACTIVE = "all_active"  # type: str

    def __init__(
        self,
        parent=None,
        esi="00000000000000000000",
        active_mode="all_active",
        esi_label=0,
    ):
        super(BgpV6EthernetSegment, self).__init__()
        self._parent = parent
        self._set_property("esi", esi)
        self._set_property("active_mode", active_mode)
        self._set_property("esi_label", esi_label)

    @property
    def df_election(self):
        # type: () -> BgpEthernetSegmentDfElection
        """df_election getter

        Configuration for Designated Forwarder (DF) election among the Provider Edge (PE) routers on the same Ethernet Segment.Configuration for Designated Forwarder (DF) election among the Provider Edge (PE) routers on the same Ethernet Segment.Designated Forwarder (DF) election configuration.

        Returns: BgpEthernetSegmentDfElection
        """
        return self._get_property("df_election", BgpEthernetSegmentDfElection)

    @property
    def evis(self):
        # type: () -> BgpV6EvpnEvisIter
        """evis getter

        This contains the list of EVIs.

        Returns: BgpV6EvpnEvisIter
        """
        return self._get_property("evis", BgpV6EvpnEvisIter, self._parent, self._choice)

    @property
    def esi(self):
        # type: () -> str
        """esi getter

        10-octet Ethernet Segment Identifier (ESI) Example - For multi-home scenario nonZero ESI is '10000000000000000000' .

        Returns: str
        """
        return self._get_property("esi")

    @esi.setter
    def esi(self, value):
        """esi setter

        10-octet Ethernet Segment Identifier (ESI) Example - For multi-home scenario nonZero ESI is '10000000000000000000' .

        value: str
        """
        self._set_property("esi", value)

    @property
    def active_mode(self):
        # type: () -> Union[Literal["all_active"], Literal["single_active"]]
        """active_mode getter

        Single Active or All Active mode Redundancy mode selection for Multi-home.

        Returns: Union[Literal["all_active"], Literal["single_active"]]
        """
        return self._get_property("active_mode")

    @active_mode.setter
    def active_mode(self, value):
        """active_mode setter

        Single Active or All Active mode Redundancy mode selection for Multi-home.

        value: Union[Literal["all_active"], Literal["single_active"]]
        """
        self._set_property("active_mode", value)

    @property
    def esi_label(self):
        # type: () -> int
        """esi_label getter

        The label value to be advertised as ESI Label in ESI Label Extended Community. This is included in Ethernet Auto-discovery per ES Routes advertised by a router.

        Returns: int
        """
        return self._get_property("esi_label")

    @esi_label.setter
    def esi_label(self, value):
        """esi_label setter

        The label value to be advertised as ESI Label in ESI Label Extended Community. This is included in Ethernet Auto-discovery per ES Routes advertised by a router.

        value: int
        """
        self._set_property("esi_label", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Optional Extended Community settings.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.Optional AS PATH settings.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)


class BgpV6EvpnEvis(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "evi_vxlan",
            ],
        },
        "evi_vxlan": {"type": "BgpV6EviVxlan"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "evi_vxlan",
    }  # type: Dict[str, Union(type)]

    EVI_VXLAN = "evi_vxlan"  # type: str

    def __init__(self, parent=None, choice=None):
        super(BgpV6EvpnEvis, self).__init__()
        self._parent = parent
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def evi_vxlan(self):
        # type: () -> BgpV6EviVxlan
        """Factory property that returns an instance of the BgpV6EviVxlan class

        Configuration for BGP EVPN EVI. Advertises following routes -. . Type 3 - Inclusive Multicast Ethernet Tag Route. . Type 1 - Ethernet Auto-discovery Route (Per EVI). . Type 1 - Ethernet Auto-discovery Route (Per ES)

        Returns: BgpV6EviVxlan
        """
        return self._get_property("evi_vxlan", BgpV6EviVxlan, self, "evi_vxlan")

    @property
    def choice(self):
        # type: () -> Union[Literal["evi_vxlan"]]
        """choice getter

        TBD

        Returns: Union[Literal["evi_vxlan"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["evi_vxlan"]]
        """
        self._set_property("choice", value)


class BgpV6EviVxlan(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "broadcast_domains": {"type": "BgpV6EviVxlanBroadcastDomainIter"},
        "replication_type": {
            "type": str,
            "enum": [
                "ingress_replication",
            ],
        },
        "pmsi_label": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "ad_label": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "route_distinguisher": {"type": "BgpRouteDistinguisher"},
        "route_target_export": {"type": "BgpRouteTargetIter"},
        "route_target_import": {"type": "BgpRouteTargetIter"},
        "l3_route_target_export": {"type": "BgpRouteTargetIter"},
        "l3_route_target_import": {"type": "BgpRouteTargetIter"},
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "replication_type": "ingress_replication",
        "pmsi_label": 16,
        "ad_label": 0,
    }  # type: Dict[str, Union(type)]

    INGRESS_REPLICATION = "ingress_replication"  # type: str

    def __init__(
        self,
        parent=None,
        replication_type="ingress_replication",
        pmsi_label=16,
        ad_label=0,
    ):
        super(BgpV6EviVxlan, self).__init__()
        self._parent = parent
        self._set_property("replication_type", replication_type)
        self._set_property("pmsi_label", pmsi_label)
        self._set_property("ad_label", ad_label)

    @property
    def broadcast_domains(self):
        # type: () -> BgpV6EviVxlanBroadcastDomainIter
        """broadcast_domains getter

        This contains the list of Broadcast Domains to be configured per EVI.

        Returns: BgpV6EviVxlanBroadcastDomainIter
        """
        return self._get_property(
            "broadcast_domains",
            BgpV6EviVxlanBroadcastDomainIter,
            self._parent,
            self._choice,
        )

    @property
    def replication_type(self):
        # type: () -> Union[Literal["ingress_replication"]]
        """replication_type getter

        This model only supports Ingress Replication

        Returns: Union[Literal["ingress_replication"]]
        """
        return self._get_property("replication_type")

    @replication_type.setter
    def replication_type(self, value):
        """replication_type setter

        This model only supports Ingress Replication

        value: Union[Literal["ingress_replication"]]
        """
        self._set_property("replication_type", value)

    @property
    def pmsi_label(self):
        # type: () -> int
        """pmsi_label getter

        Downstream assigned VNI to be carried as Part of P-Multicast Service Interface Tunnel attribute (PMSI Tunnel Attribute) in Type 3 Inclusive Multicast Ethernet Tag Route.

        Returns: int
        """
        return self._get_property("pmsi_label")

    @pmsi_label.setter
    def pmsi_label(self, value):
        """pmsi_label setter

        Downstream assigned VNI to be carried as Part of P-Multicast Service Interface Tunnel attribute (PMSI Tunnel Attribute) in Type 3 Inclusive Multicast Ethernet Tag Route.

        value: int
        """
        self._set_property("pmsi_label", value)

    @property
    def ad_label(self):
        # type: () -> int
        """ad_label getter

        The Auto-discovery Route label (AD label) value, which gets advertised in the Ethernet Auto-discovery Route per <EVI, ESI>

        Returns: int
        """
        return self._get_property("ad_label")

    @ad_label.setter
    def ad_label(self, value):
        """ad_label setter

        The Auto-discovery Route label (AD label) value, which gets advertised in the Ethernet Auto-discovery Route per <EVI, ESI>

        value: int
        """
        self._set_property("ad_label", value)

    @property
    def route_distinguisher(self):
        # type: () -> BgpRouteDistinguisher
        """route_distinguisher getter

        BGP Route Distinguisher.BGP Route Distinguisher.BGP Route Distinguisher.Colon separated Extended Community value of 6 Bytes - "AS number: Value" identifying an EVI. Example - for the as_2octet "60005:100".

        Returns: BgpRouteDistinguisher
        """
        return self._get_property("route_distinguisher", BgpRouteDistinguisher)

    @property
    def route_target_export(self):
        # type: () -> BgpRouteTargetIter
        """route_target_export getter

        List of Layer 2 Virtual Network Identifier (L2VNI) export targets associated with this EVI.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "route_target_export", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def route_target_import(self):
        # type: () -> BgpRouteTargetIter
        """route_target_import getter

        List of L2VNI import targets associated with this EVI.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "route_target_import", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def l3_route_target_export(self):
        # type: () -> BgpRouteTargetIter
        """l3_route_target_export getter

        List of Layer 3 Virtual Network Identifier (L3VNI) Export Route Targets.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "l3_route_target_export", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def l3_route_target_import(self):
        # type: () -> BgpRouteTargetIter
        """l3_route_target_import getter

        List of L3VNI Import Route Targets.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "l3_route_target_import", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Optional Extended Community settings.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of a Set or Sequence of Autonomous Systems (AS) numbers that a routing information passes through to reach the destination.Optional AS PATH settings.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)


class BgpV6EviVxlanBroadcastDomain(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "cmac_ip_range": {"type": "BgpCMacIpRangeIter"},
        "ethernet_tag_id": {
            "type": int,
            "minimum": 0,
            "maximum": 4294967295,
        },
        "vlan_aware_service": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "ethernet_tag_id": 0,
        "vlan_aware_service": False,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, ethernet_tag_id=0, vlan_aware_service=False):
        super(BgpV6EviVxlanBroadcastDomain, self).__init__()
        self._parent = parent
        self._set_property("ethernet_tag_id", ethernet_tag_id)
        self._set_property("vlan_aware_service", vlan_aware_service)

    @property
    def cmac_ip_range(self):
        # type: () -> BgpCMacIpRangeIter
        """cmac_ip_range getter

        This contains the list of Customer MAC/IP Ranges to be configured per Broadcast Domain. . . Advertises following route - . Type 2 - MAC/IP Advertisement Route.

        Returns: BgpCMacIpRangeIter
        """
        return self._get_property(
            "cmac_ip_range", BgpCMacIpRangeIter, self._parent, self._choice
        )

    @property
    def ethernet_tag_id(self):
        # type: () -> int
        """ethernet_tag_id getter

        The Ethernet Tag ID of the Broadcast Domain.

        Returns: int
        """
        return self._get_property("ethernet_tag_id")

    @ethernet_tag_id.setter
    def ethernet_tag_id(self, value):
        """ethernet_tag_id setter

        The Ethernet Tag ID of the Broadcast Domain.

        value: int
        """
        self._set_property("ethernet_tag_id", value)

    @property
    def vlan_aware_service(self):
        # type: () -> bool
        """vlan_aware_service getter

        VLAN-Aware service to be enabled or disabled.

        Returns: bool
        """
        return self._get_property("vlan_aware_service")

    @vlan_aware_service.setter
    def vlan_aware_service(self, value):
        """vlan_aware_service setter

        VLAN-Aware service to be enabled or disabled.

        value: bool
        """
        self._set_property("vlan_aware_service", value)


class BgpV6EviVxlanBroadcastDomainIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV6EviVxlanBroadcastDomainIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV6EviVxlanBroadcastDomain]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV6EviVxlanBroadcastDomainIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV6EviVxlanBroadcastDomain
        return self._next()

    def next(self):
        # type: () -> BgpV6EviVxlanBroadcastDomain
        return self._next()

    def broadcastdomain(self, ethernet_tag_id=0, vlan_aware_service=False):
        # type: (int,bool) -> BgpV6EviVxlanBroadcastDomainIter
        """Factory method that creates an instance of the BgpV6EviVxlanBroadcastDomain class

        Configuration for Broadcast Domains per EVI.

        Returns: BgpV6EviVxlanBroadcastDomainIter
        """
        item = BgpV6EviVxlanBroadcastDomain(
            parent=self._parent,
            ethernet_tag_id=ethernet_tag_id,
            vlan_aware_service=vlan_aware_service,
        )
        self._add(item)
        return self

    def add(self, ethernet_tag_id=0, vlan_aware_service=False):
        # type: (int,bool) -> BgpV6EviVxlanBroadcastDomain
        """Add method that creates and returns an instance of the BgpV6EviVxlanBroadcastDomain class

        Configuration for Broadcast Domains per EVI.

        Returns: BgpV6EviVxlanBroadcastDomain
        """
        item = BgpV6EviVxlanBroadcastDomain(
            parent=self._parent,
            ethernet_tag_id=ethernet_tag_id,
            vlan_aware_service=vlan_aware_service,
        )
        self._add(item)
        return item


class BgpV6EvpnEvisIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = True

    def __init__(self, parent=None, choice=None):
        super(BgpV6EvpnEvisIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV6EviVxlan, BgpV6EvpnEvis]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV6EvpnEvisIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV6EvpnEvis
        return self._next()

    def next(self):
        # type: () -> BgpV6EvpnEvis
        return self._next()

    def evpnevis(self):
        # type: () -> BgpV6EvpnEvisIter
        """Factory method that creates an instance of the BgpV6EvpnEvis class

        This contains a list of different flavors of EVPN. . For example EVPN over VXLAN or EVPN over MPLS etc to be configured per Ethernet segment. . Need to instantiate correct type of EVPN instance as per requirement.

        Returns: BgpV6EvpnEvisIter
        """
        item = BgpV6EvpnEvis(parent=self._parent, choice=self._choice)
        self._add(item)
        return self

    def add(self):
        # type: () -> BgpV6EvpnEvis
        """Add method that creates and returns an instance of the BgpV6EvpnEvis class

        This contains a list of different flavors of EVPN. . For example EVPN over VXLAN or EVPN over MPLS etc to be configured per Ethernet segment. . Need to instantiate correct type of EVPN instance as per requirement.

        Returns: BgpV6EvpnEvis
        """
        item = BgpV6EvpnEvis(parent=self._parent, choice=self._choice)
        self._add(item)
        return item

    def evi_vxlan(
        self, replication_type="ingress_replication", pmsi_label=16, ad_label=0
    ):
        # type: (Union[Literal["ingress_replication"]],int,int) -> BgpV6EvpnEvisIter
        """Factory method that creates an instance of the BgpV6EviVxlan class

        Configuration for BGP EVPN EVI. Advertises following routes -. . Type 3 - Inclusive Multicast Ethernet Tag Route. . Type 1 - Ethernet Auto-discovery Route (Per EVI). . Type 1 - Ethernet Auto-discovery Route (Per ES)

        Returns: BgpV6EvpnEvisIter
        """
        item = BgpV6EvpnEvis()
        item.evi_vxlan
        item.choice = "evi_vxlan"
        self._add(item)
        return self


class BgpV6EthernetSegmentIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV6EthernetSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV6EthernetSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV6EthernetSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV6EthernetSegment
        return self._next()

    def next(self):
        # type: () -> BgpV6EthernetSegment
        return self._next()

    def ethernetsegment(
        self, esi="00000000000000000000", active_mode="all_active", esi_label=0
    ):
        # type: (str,Union[Literal["all_active"], Literal["single_active"]],int) -> BgpV6EthernetSegmentIter
        """Factory method that creates an instance of the BgpV6EthernetSegment class

        Configuration for BGP Ethernet Segment ranges. Advertises following routes -. . Type 4 - Ethernet Segment Route

        Returns: BgpV6EthernetSegmentIter
        """
        item = BgpV6EthernetSegment(
            parent=self._parent, esi=esi, active_mode=active_mode, esi_label=esi_label
        )
        self._add(item)
        return self

    def add(self, esi="00000000000000000000", active_mode="all_active", esi_label=0):
        # type: (str,Union[Literal["all_active"], Literal["single_active"]],int) -> BgpV6EthernetSegment
        """Add method that creates and returns an instance of the BgpV6EthernetSegment class

        Configuration for BGP Ethernet Segment ranges. Advertises following routes -. . Type 4 - Ethernet Segment Route

        Returns: BgpV6EthernetSegment
        """
        item = BgpV6EthernetSegment(
            parent=self._parent, esi=esi, active_mode=active_mode, esi_label=esi_label
        )
        self._add(item)
        return item


class BgpV6PeerIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV6PeerIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV6Peer]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV6PeerIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV6Peer
        return self._next()

    def next(self):
        # type: () -> BgpV6Peer
        return self._next()

    def v6peer(
        self,
        peer_address=None,
        as_type=None,
        as_number=None,
        as_number_width="four",
        name=None,
    ):
        # type: (str,Union[Literal["ebgp"], Literal["ibgp"]],int,Union[Literal["four"], Literal["two"]],str) -> BgpV6PeerIter
        """Factory method that creates an instance of the BgpV6Peer class

        Status: under-review. Configuration for BGPv6 peer settings and routes.. Configuration for basic emulated BGP peer settings.

        Returns: BgpV6PeerIter
        """
        item = BgpV6Peer(
            parent=self._parent,
            peer_address=peer_address,
            as_type=as_type,
            as_number=as_number,
            as_number_width=as_number_width,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        peer_address=None,
        as_type=None,
        as_number=None,
        as_number_width="four",
        name=None,
    ):
        # type: (str,Union[Literal["ebgp"], Literal["ibgp"]],int,Union[Literal["four"], Literal["two"]],str) -> BgpV6Peer
        """Add method that creates and returns an instance of the BgpV6Peer class

        Status: under-review. Configuration for BGPv6 peer settings and routes.. Configuration for basic emulated BGP peer settings.

        Returns: BgpV6Peer
        """
        item = BgpV6Peer(
            parent=self._parent,
            peer_address=peer_address,
            as_type=as_type,
            as_number=as_number,
            as_number_width=as_number_width,
            name=name,
        )
        self._add(item)
        return item


class BgpV6InterfaceIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV6InterfaceIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV6Interface]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV6InterfaceIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV6Interface
        return self._next()

    def next(self):
        # type: () -> BgpV6Interface
        return self._next()

    def v6interface(self, ipv6_name=None):
        # type: (str) -> BgpV6InterfaceIter
        """Factory method that creates an instance of the BgpV6Interface class

        Configuration for emulated BGPv6 peers and routes on a single IPv6 interface.

        Returns: BgpV6InterfaceIter
        """
        item = BgpV6Interface(parent=self._parent, ipv6_name=ipv6_name)
        self._add(item)
        return self

    def add(self, ipv6_name=None):
        # type: (str) -> BgpV6Interface
        """Add method that creates and returns an instance of the BgpV6Interface class

        Configuration for emulated BGPv6 peers and routes on a single IPv6 interface.

        Returns: BgpV6Interface
        """
        item = BgpV6Interface(parent=self._parent, ipv6_name=ipv6_name)
        self._add(item)
        return item


class DeviceVxlan(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "v4_tunnels": {"type": "VxlanV4TunnelIter"},
        "v6_tunnels": {"type": "VxlanV6TunnelIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(DeviceVxlan, self).__init__()
        self._parent = parent

    @property
    def v4_tunnels(self):
        # type: () -> VxlanV4TunnelIter
        """v4_tunnels getter

        IPv4 VXLAN Tunnels

        Returns: VxlanV4TunnelIter
        """
        return self._get_property(
            "v4_tunnels", VxlanV4TunnelIter, self._parent, self._choice
        )

    @property
    def v6_tunnels(self):
        # type: () -> VxlanV6TunnelIter
        """v6_tunnels getter

        IPv6 VXLAN Tunnels

        Returns: VxlanV6TunnelIter
        """
        return self._get_property(
            "v6_tunnels", VxlanV6TunnelIter, self._parent, self._choice
        )


class VxlanV4Tunnel(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "source_interface": {"type": str},
        "destination_ip_mode": {"type": "VxlanV4TunnelDestinationIPMode"},
        "vni": {
            "type": int,
            "minimum": 1,
            "maximum": 16777215,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("source_interface", "vni", "name")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, source_interface=None, vni=None, name=None):
        super(VxlanV4Tunnel, self).__init__()
        self._parent = parent
        self._set_property("source_interface", source_interface)
        self._set_property("vni", vni)
        self._set_property("name", name)

    @property
    def source_interface(self):
        # type: () -> str
        """source_interface getter

        Determines the source interface.. . x-constraint:. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv4Loopback/properties/name.

        Returns: str
        """
        return self._get_property("source_interface")

    @source_interface.setter
    def source_interface(self, value):
        """source_interface setter

        Determines the source interface.. . x-constraint:. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv4Loopback/properties/name.

        value: str
        """
        self._set_property("source_interface", value)

    @property
    def destination_ip_mode(self):
        # type: () -> VxlanV4TunnelDestinationIPMode
        """destination_ip_mode getter

        Communication mode between the VTEPs, either unicast or multicast.Communication mode between the VTEPs, either unicast or multicast.

        Returns: VxlanV4TunnelDestinationIPMode
        """
        return self._get_property("destination_ip_mode", VxlanV4TunnelDestinationIPMode)

    @property
    def vni(self):
        # type: () -> int
        """vni getter

        VXLAN Network Identifier (VNI) to distinguish network instances on the wire

        Returns: int
        """
        return self._get_property("vni")

    @vni.setter
    def vni(self, value):
        """vni setter

        VXLAN Network Identifier (VNI) to distinguish network instances on the wire

        value: int
        """
        self._set_property("vni", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class VxlanV4TunnelDestinationIPMode(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "unicast",
                "multicast",
            ],
        },
        "unicast": {"type": "VxlanV4TunnelDestinationIPModeUnicast"},
        "multicast": {"type": "VxlanV4TunnelDestinationIPModeMulticast"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "multicast",
    }  # type: Dict[str, Union(type)]

    UNICAST = "unicast"  # type: str
    MULTICAST = "multicast"  # type: str

    def __init__(self, parent=None, choice=None):
        super(VxlanV4TunnelDestinationIPMode, self).__init__()
        self._parent = parent
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def unicast(self):
        # type: () -> VxlanV4TunnelDestinationIPModeUnicast
        """Factory property that returns an instance of the VxlanV4TunnelDestinationIPModeUnicast class

        TBD

        Returns: VxlanV4TunnelDestinationIPModeUnicast
        """
        return self._get_property(
            "unicast", VxlanV4TunnelDestinationIPModeUnicast, self, "unicast"
        )

    @property
    def multicast(self):
        # type: () -> VxlanV4TunnelDestinationIPModeMulticast
        """Factory property that returns an instance of the VxlanV4TunnelDestinationIPModeMulticast class

        Multicast Group address for member VNI(VXLAN Network Identifier)

        Returns: VxlanV4TunnelDestinationIPModeMulticast
        """
        return self._get_property(
            "multicast", VxlanV4TunnelDestinationIPModeMulticast, self, "multicast"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["multicast"], Literal["unicast"]]
        """choice getter

        unicast or multicast

        Returns: Union[Literal["multicast"], Literal["unicast"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        unicast or multicast

        value: Union[Literal["multicast"], Literal["unicast"]]
        """
        self._set_property("choice", value)


class VxlanV4TunnelDestinationIPModeUnicast(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "vteps": {"type": "VxlanV4TunnelDestinationIPModeUnicastVtepIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(VxlanV4TunnelDestinationIPModeUnicast, self).__init__()
        self._parent = parent

    @property
    def vteps(self):
        # type: () -> VxlanV4TunnelDestinationIPModeUnicastVtepIter
        """vteps getter

        List of VTEPs for member VNI(VXLAN Network Identifier)

        Returns: VxlanV4TunnelDestinationIPModeUnicastVtepIter
        """
        return self._get_property(
            "vteps",
            VxlanV4TunnelDestinationIPModeUnicastVtepIter,
            self._parent,
            self._choice,
        )


class VxlanV4TunnelDestinationIPModeUnicastVtep(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "remote_vtep_address": {
            "type": str,
            "format": "ipv4",
        },
        "arp_suppression_cache": {
            "type": "VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, remote_vtep_address=None):
        super(VxlanV4TunnelDestinationIPModeUnicastVtep, self).__init__()
        self._parent = parent
        self._set_property("remote_vtep_address", remote_vtep_address)

    @property
    def remote_vtep_address(self):
        # type: () -> str
        """remote_vtep_address getter

        Remote VXLAN Tunnel End Point address

        Returns: str
        """
        return self._get_property("remote_vtep_address")

    @remote_vtep_address.setter
    def remote_vtep_address(self, value):
        """remote_vtep_address setter

        Remote VXLAN Tunnel End Point address

        value: str
        """
        self._set_property("remote_vtep_address", value)

    @property
    def arp_suppression_cache(self):
        # type: () -> VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter
        """arp_suppression_cache getter

        Each VTEP maintains an ARP suppression cache table for known IP hosts and their associated MAC addresses in the VNI segment. When an end host in the VNI sends an ARP request for another end-host IP address, its local VTEP intercepts the ARP request and checks for the ARP-resolved IP address in its ARP suppression cache table. If it finds a match, the local VTEP sends an ARP response on behalf of the remote end host.

        Returns: VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter
        """
        return self._get_property(
            "arp_suppression_cache",
            VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter,
            self._parent,
            self._choice,
        )


class VxlanTunnelDestinationIPModeUnicastArpSuppressionCache(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "remote_vm_mac": {
            "type": str,
            "format": "mac",
        },
        "remote_vm_ipv4": {
            "type": str,
            "format": "ipv4",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, remote_vm_mac=None, remote_vm_ipv4=None):
        super(VxlanTunnelDestinationIPModeUnicastArpSuppressionCache, self).__init__()
        self._parent = parent
        self._set_property("remote_vm_mac", remote_vm_mac)
        self._set_property("remote_vm_ipv4", remote_vm_ipv4)

    @property
    def remote_vm_mac(self):
        # type: () -> str
        """remote_vm_mac getter

        Remote VM MAC address bound to Remote VM IPv4 address

        Returns: str
        """
        return self._get_property("remote_vm_mac")

    @remote_vm_mac.setter
    def remote_vm_mac(self, value):
        """remote_vm_mac setter

        Remote VM MAC address bound to Remote VM IPv4 address

        value: str
        """
        self._set_property("remote_vm_mac", value)

    @property
    def remote_vm_ipv4(self):
        # type: () -> str
        """remote_vm_ipv4 getter

        Remote VM IPv4 address

        Returns: str
        """
        return self._get_property("remote_vm_ipv4")

    @remote_vm_ipv4.setter
    def remote_vm_ipv4(self, value):
        """remote_vm_ipv4 setter

        Remote VM IPv4 address

        value: str
        """
        self._set_property("remote_vm_ipv4", value)


class VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(
            VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter, self
        ).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[VxlanTunnelDestinationIPModeUnicastArpSuppressionCache]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter
        return self._iter()

    def __next__(self):
        # type: () -> VxlanTunnelDestinationIPModeUnicastArpSuppressionCache
        return self._next()

    def next(self):
        # type: () -> VxlanTunnelDestinationIPModeUnicastArpSuppressionCache
        return self._next()

    def arpsuppressioncache(self, remote_vm_mac=None, remote_vm_ipv4=None):
        # type: (str,str) -> VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter
        """Factory method that creates an instance of the VxlanTunnelDestinationIPModeUnicastArpSuppressionCache class

        Each VTEP maintains an ARP suppression cache table for known IP hosts and their associated MAC addresses in the VNI segment. When an end host in the VNI sends an ARP request for another end-host IP address, its local VTEP intercepts the ARP request and checks for the ARP-resolved IP address in its ARP suppression cache table. If it finds a match, the local VTEP sends an ARP response on behalf of the remote end host.

        Returns: VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter
        """
        item = VxlanTunnelDestinationIPModeUnicastArpSuppressionCache(
            parent=self._parent,
            remote_vm_mac=remote_vm_mac,
            remote_vm_ipv4=remote_vm_ipv4,
        )
        self._add(item)
        return self

    def add(self, remote_vm_mac=None, remote_vm_ipv4=None):
        # type: (str,str) -> VxlanTunnelDestinationIPModeUnicastArpSuppressionCache
        """Add method that creates and returns an instance of the VxlanTunnelDestinationIPModeUnicastArpSuppressionCache class

        Each VTEP maintains an ARP suppression cache table for known IP hosts and their associated MAC addresses in the VNI segment. When an end host in the VNI sends an ARP request for another end-host IP address, its local VTEP intercepts the ARP request and checks for the ARP-resolved IP address in its ARP suppression cache table. If it finds a match, the local VTEP sends an ARP response on behalf of the remote end host.

        Returns: VxlanTunnelDestinationIPModeUnicastArpSuppressionCache
        """
        item = VxlanTunnelDestinationIPModeUnicastArpSuppressionCache(
            parent=self._parent,
            remote_vm_mac=remote_vm_mac,
            remote_vm_ipv4=remote_vm_ipv4,
        )
        self._add(item)
        return item


class VxlanV4TunnelDestinationIPModeUnicastVtepIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(VxlanV4TunnelDestinationIPModeUnicastVtepIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[VxlanV4TunnelDestinationIPModeUnicastVtep]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> VxlanV4TunnelDestinationIPModeUnicastVtepIter
        return self._iter()

    def __next__(self):
        # type: () -> VxlanV4TunnelDestinationIPModeUnicastVtep
        return self._next()

    def next(self):
        # type: () -> VxlanV4TunnelDestinationIPModeUnicastVtep
        return self._next()

    def vtep(self, remote_vtep_address=None):
        # type: (str) -> VxlanV4TunnelDestinationIPModeUnicastVtepIter
        """Factory method that creates an instance of the VxlanV4TunnelDestinationIPModeUnicastVtep class

        VTEP (VXLAN Tunnel End Point (VTEP)) parameters. DestinationIPMode Unicast VTEP Base

        Returns: VxlanV4TunnelDestinationIPModeUnicastVtepIter
        """
        item = VxlanV4TunnelDestinationIPModeUnicastVtep(
            parent=self._parent, remote_vtep_address=remote_vtep_address
        )
        self._add(item)
        return self

    def add(self, remote_vtep_address=None):
        # type: (str) -> VxlanV4TunnelDestinationIPModeUnicastVtep
        """Add method that creates and returns an instance of the VxlanV4TunnelDestinationIPModeUnicastVtep class

        VTEP (VXLAN Tunnel End Point (VTEP)) parameters. DestinationIPMode Unicast VTEP Base

        Returns: VxlanV4TunnelDestinationIPModeUnicastVtep
        """
        item = VxlanV4TunnelDestinationIPModeUnicastVtep(
            parent=self._parent, remote_vtep_address=remote_vtep_address
        )
        self._add(item)
        return item


class VxlanV4TunnelDestinationIPModeMulticast(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "address": {
            "type": str,
            "format": "ipv4",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, address=None):
        super(VxlanV4TunnelDestinationIPModeMulticast, self).__init__()
        self._parent = parent
        self._set_property("address", address)

    @property
    def address(self):
        # type: () -> str
        """address getter

        IPv4 Multicast address

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        IPv4 Multicast address

        value: str
        """
        self._set_property("address", value)


class VxlanV4TunnelIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(VxlanV4TunnelIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[VxlanV4Tunnel]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> VxlanV4TunnelIter
        return self._iter()

    def __next__(self):
        # type: () -> VxlanV4Tunnel
        return self._next()

    def next(self):
        # type: () -> VxlanV4Tunnel
        return self._next()

    def v4tunnel(self, source_interface=None, vni=None, name=None):
        # type: (str,int,str) -> VxlanV4TunnelIter
        """Factory method that creates an instance of the VxlanV4Tunnel class

        Configuration and operational state parameters relating to IPv4 VXLAN tunnel end-point interface.

        Returns: VxlanV4TunnelIter
        """
        item = VxlanV4Tunnel(
            parent=self._parent, source_interface=source_interface, vni=vni, name=name
        )
        self._add(item)
        return self

    def add(self, source_interface=None, vni=None, name=None):
        # type: (str,int,str) -> VxlanV4Tunnel
        """Add method that creates and returns an instance of the VxlanV4Tunnel class

        Configuration and operational state parameters relating to IPv4 VXLAN tunnel end-point interface.

        Returns: VxlanV4Tunnel
        """
        item = VxlanV4Tunnel(
            parent=self._parent, source_interface=source_interface, vni=vni, name=name
        )
        self._add(item)
        return item


class VxlanV6Tunnel(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "source_interface": {"type": str},
        "destination_ip_mode": {"type": "VxlanV6TunnelDestinationIPMode"},
        "vni": {
            "type": int,
            "minimum": 1,
            "maximum": 16777215,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("source_interface", "vni", "name")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, source_interface=None, vni=None, name=None):
        super(VxlanV6Tunnel, self).__init__()
        self._parent = parent
        self._set_property("source_interface", source_interface)
        self._set_property("vni", vni)
        self._set_property("name", name)

    @property
    def source_interface(self):
        # type: () -> str
        """source_interface getter

        Determines the source interface.. . x-constraint:. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Device.Ipv6Loopback/properties/name.

        Returns: str
        """
        return self._get_property("source_interface")

    @source_interface.setter
    def source_interface(self, value):
        """source_interface setter

        Determines the source interface.. . x-constraint:. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Device.Ipv6Loopback/properties/name.

        value: str
        """
        self._set_property("source_interface", value)

    @property
    def destination_ip_mode(self):
        # type: () -> VxlanV6TunnelDestinationIPMode
        """destination_ip_mode getter

        Communication mode between the VTEPs, either unicast or multicast.Communication mode between the VTEPs, either unicast or multicast.

        Returns: VxlanV6TunnelDestinationIPMode
        """
        return self._get_property("destination_ip_mode", VxlanV6TunnelDestinationIPMode)

    @property
    def vni(self):
        # type: () -> int
        """vni getter

        VXLAN Network Identifier (VNI) to distinguish network instances on the wire

        Returns: int
        """
        return self._get_property("vni")

    @vni.setter
    def vni(self, value):
        """vni setter

        VXLAN Network Identifier (VNI) to distinguish network instances on the wire

        value: int
        """
        self._set_property("vni", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class VxlanV6TunnelDestinationIPMode(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "unicast",
                "multicast",
            ],
        },
        "unicast": {"type": "VxlanV6TunnelDestinationIPModeUnicast"},
        "multicast": {"type": "VxlanV6TunnelDestinationIPModeMulticast"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "multicast",
    }  # type: Dict[str, Union(type)]

    UNICAST = "unicast"  # type: str
    MULTICAST = "multicast"  # type: str

    def __init__(self, parent=None, choice=None):
        super(VxlanV6TunnelDestinationIPMode, self).__init__()
        self._parent = parent
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def unicast(self):
        # type: () -> VxlanV6TunnelDestinationIPModeUnicast
        """Factory property that returns an instance of the VxlanV6TunnelDestinationIPModeUnicast class

        TBD

        Returns: VxlanV6TunnelDestinationIPModeUnicast
        """
        return self._get_property(
            "unicast", VxlanV6TunnelDestinationIPModeUnicast, self, "unicast"
        )

    @property
    def multicast(self):
        # type: () -> VxlanV6TunnelDestinationIPModeMulticast
        """Factory property that returns an instance of the VxlanV6TunnelDestinationIPModeMulticast class

        Multicast Group address for member VNI(VXLAN Network Identifier)

        Returns: VxlanV6TunnelDestinationIPModeMulticast
        """
        return self._get_property(
            "multicast", VxlanV6TunnelDestinationIPModeMulticast, self, "multicast"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["multicast"], Literal["unicast"]]
        """choice getter

        unicast or multicast

        Returns: Union[Literal["multicast"], Literal["unicast"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        unicast or multicast

        value: Union[Literal["multicast"], Literal["unicast"]]
        """
        self._set_property("choice", value)


class VxlanV6TunnelDestinationIPModeUnicast(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "vteps": {"type": "VxlanV6TunnelDestinationIPModeUnicastVtepIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(VxlanV6TunnelDestinationIPModeUnicast, self).__init__()
        self._parent = parent

    @property
    def vteps(self):
        # type: () -> VxlanV6TunnelDestinationIPModeUnicastVtepIter
        """vteps getter

        List of VTEPs for member VNI(VXLAN Network Identifier)

        Returns: VxlanV6TunnelDestinationIPModeUnicastVtepIter
        """
        return self._get_property(
            "vteps",
            VxlanV6TunnelDestinationIPModeUnicastVtepIter,
            self._parent,
            self._choice,
        )


class VxlanV6TunnelDestinationIPModeUnicastVtep(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "remote_vtep_address": {
            "type": str,
            "format": "ipv6",
        },
        "arp_suppression_cache": {
            "type": "VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, remote_vtep_address=None):
        super(VxlanV6TunnelDestinationIPModeUnicastVtep, self).__init__()
        self._parent = parent
        self._set_property("remote_vtep_address", remote_vtep_address)

    @property
    def remote_vtep_address(self):
        # type: () -> str
        """remote_vtep_address getter

        Remote VXLAN Tunnel End Point address

        Returns: str
        """
        return self._get_property("remote_vtep_address")

    @remote_vtep_address.setter
    def remote_vtep_address(self, value):
        """remote_vtep_address setter

        Remote VXLAN Tunnel End Point address

        value: str
        """
        self._set_property("remote_vtep_address", value)

    @property
    def arp_suppression_cache(self):
        # type: () -> VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter
        """arp_suppression_cache getter

        Each VTEP maintains an ARP suppression cache table for known IP hosts and their associated MAC addresses in the VNI segment. When an end host in the VNI sends an ARP request for another end-host IP address, its local VTEP intercepts the ARP request and checks for the ARP-resolved IP address in its ARP suppression cache table. If it finds a match, the local VTEP sends an ARP response on behalf of the remote end host.

        Returns: VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter
        """
        return self._get_property(
            "arp_suppression_cache",
            VxlanTunnelDestinationIPModeUnicastArpSuppressionCacheIter,
            self._parent,
            self._choice,
        )


class VxlanV6TunnelDestinationIPModeUnicastVtepIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(VxlanV6TunnelDestinationIPModeUnicastVtepIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[VxlanV6TunnelDestinationIPModeUnicastVtep]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> VxlanV6TunnelDestinationIPModeUnicastVtepIter
        return self._iter()

    def __next__(self):
        # type: () -> VxlanV6TunnelDestinationIPModeUnicastVtep
        return self._next()

    def next(self):
        # type: () -> VxlanV6TunnelDestinationIPModeUnicastVtep
        return self._next()

    def vtep(self, remote_vtep_address=None):
        # type: (str) -> VxlanV6TunnelDestinationIPModeUnicastVtepIter
        """Factory method that creates an instance of the VxlanV6TunnelDestinationIPModeUnicastVtep class

        VTEP (VXLAN Tunnel End Point (VTEP)) parameters. DestinationIPMode Unicast VTEP Base

        Returns: VxlanV6TunnelDestinationIPModeUnicastVtepIter
        """
        item = VxlanV6TunnelDestinationIPModeUnicastVtep(
            parent=self._parent, remote_vtep_address=remote_vtep_address
        )
        self._add(item)
        return self

    def add(self, remote_vtep_address=None):
        # type: (str) -> VxlanV6TunnelDestinationIPModeUnicastVtep
        """Add method that creates and returns an instance of the VxlanV6TunnelDestinationIPModeUnicastVtep class

        VTEP (VXLAN Tunnel End Point (VTEP)) parameters. DestinationIPMode Unicast VTEP Base

        Returns: VxlanV6TunnelDestinationIPModeUnicastVtep
        """
        item = VxlanV6TunnelDestinationIPModeUnicastVtep(
            parent=self._parent, remote_vtep_address=remote_vtep_address
        )
        self._add(item)
        return item


class VxlanV6TunnelDestinationIPModeMulticast(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "address": {
            "type": str,
            "format": "ipv6",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, address=None):
        super(VxlanV6TunnelDestinationIPModeMulticast, self).__init__()
        self._parent = parent
        self._set_property("address", address)

    @property
    def address(self):
        # type: () -> str
        """address getter

        IPv6 Multicast address

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        IPv6 Multicast address

        value: str
        """
        self._set_property("address", value)


class VxlanV6TunnelIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(VxlanV6TunnelIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[VxlanV6Tunnel]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> VxlanV6TunnelIter
        return self._iter()

    def __next__(self):
        # type: () -> VxlanV6Tunnel
        return self._next()

    def next(self):
        # type: () -> VxlanV6Tunnel
        return self._next()

    def v6tunnel(self, source_interface=None, vni=None, name=None):
        # type: (str,int,str) -> VxlanV6TunnelIter
        """Factory method that creates an instance of the VxlanV6Tunnel class

        Configuration and operational state parameters relating to IPv6 VXLAN tunnel end-point interface.

        Returns: VxlanV6TunnelIter
        """
        item = VxlanV6Tunnel(
            parent=self._parent, source_interface=source_interface, vni=vni, name=name
        )
        self._add(item)
        return self

    def add(self, source_interface=None, vni=None, name=None):
        # type: (str,int,str) -> VxlanV6Tunnel
        """Add method that creates and returns an instance of the VxlanV6Tunnel class

        Configuration and operational state parameters relating to IPv6 VXLAN tunnel end-point interface.

        Returns: VxlanV6Tunnel
        """
        item = VxlanV6Tunnel(
            parent=self._parent, source_interface=source_interface, vni=vni, name=name
        )
        self._add(item)
        return item


class DeviceIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Device]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceIter
        return self._iter()

    def __next__(self):
        # type: () -> Device
        return self._next()

    def next(self):
        # type: () -> Device
        return self._next()

    def device(self, name=None):
        # type: (str) -> DeviceIter
        """Factory method that creates an instance of the Device class

        A container for emulated interfaces, loopback interfaces and protocol configurations.

        Returns: DeviceIter
        """
        item = Device(parent=self._parent, name=name)
        self._add(item)
        return self

    def add(self, name=None):
        # type: (str) -> Device
        """Add method that creates and returns an instance of the Device class

        A container for emulated interfaces, loopback interfaces and protocol configurations.

        Returns: Device
        """
        item = Device(parent=self._parent, name=name)
        self._add(item)
        return item


class Flow(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "tx_rx": {"type": "FlowTxRx"},
        "packet": {"type": "FlowHeaderIter"},
        "size": {"type": "FlowSize"},
        "rate": {"type": "FlowRate"},
        "duration": {"type": "FlowDuration"},
        "metrics": {"type": "FlowMetrics"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("tx_rx", "name")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, name=None):
        super(Flow, self).__init__()
        self._parent = parent
        self._set_property("name", name)

    @property
    def tx_rx(self):
        # type: () -> FlowTxRx
        """tx_rx getter

        A container for different types of transmit and receive . endpoint containers.A container for different types of transmit and receive . endpoint containers.The transmit and receive endpoints.

        Returns: FlowTxRx
        """
        return self._get_property("tx_rx", FlowTxRx)

    @property
    def packet(self):
        # type: () -> FlowHeaderIter
        """packet getter

        The header is a list of traffic protocol headers.. . The order of traffic protocol headers assigned to the list is the. order they will appear on the wire.. . In the case of an empty list the keyword/value of minItems: 1 . indicates that an implementation MUST provide at least one . Flow.Header object.. . The default value for the Flow.Header choice property is ethernet . which will result in an implementation by default providing at least . one ethernet packet header.

        Returns: FlowHeaderIter
        """
        return self._get_property("packet", FlowHeaderIter, self._parent, self._choice)

    @property
    def size(self):
        # type: () -> FlowSize
        """size getter

        The frame size which overrides the total length of the packetThe frame size which overrides the total length of the packetThe size of the packets.

        Returns: FlowSize
        """
        return self._get_property("size", FlowSize)

    @property
    def rate(self):
        # type: () -> FlowRate
        """rate getter

        The rate of packet transmissionThe rate of packet transmissionThe transmit rate of the packets.

        Returns: FlowRate
        """
        return self._get_property("rate", FlowRate)

    @property
    def duration(self):
        # type: () -> FlowDuration
        """duration getter

        A container for different transmit durations. A container for different transmit durations. The transmit duration of the packets.

        Returns: FlowDuration
        """
        return self._get_property("duration", FlowDuration)

    @property
    def metrics(self):
        # type: () -> FlowMetrics
        """metrics getter

        The optional container for configuring flow metrics.The optional container for configuring flow metrics.Flow metrics.

        Returns: FlowMetrics
        """
        return self._get_property("metrics", FlowMetrics)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        self._set_property("name", value)


class FlowTxRx(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "port",
                "device",
            ],
        },
        "port": {"type": "FlowPort"},
        "device": {"type": "FlowRouter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "port",
    }  # type: Dict[str, Union(type)]

    PORT = "port"  # type: str
    DEVICE = "device"  # type: str

    def __init__(self, parent=None, choice=None):
        super(FlowTxRx, self).__init__()
        self._parent = parent
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def port(self):
        # type: () -> FlowPort
        """Factory property that returns an instance of the FlowPort class

        A container for a transmit port and 0..n intended receive ports.. When assigning this container to a flow the flows's . packet headers will not be populated with any address resolution . information such as source and/or destination addresses. . For example Flow.Ethernet dst mac address values will be defaulted to 0. . For full control over the Flow.properties.packet header contents use this . container.

        Returns: FlowPort
        """
        return self._get_property("port", FlowPort, self, "port")

    @property
    def device(self):
        # type: () -> FlowRouter
        """Factory property that returns an instance of the FlowRouter class

        A container for declaring a map of 1..n transmit devices to 1..n receive devices. This allows for a single flow to have different tx to rx device flows such as a single one to one map or a many to many map.

        Returns: FlowRouter
        """
        return self._get_property("device", FlowRouter, self, "device")

    @property
    def choice(self):
        # type: () -> Union[Literal["device"], Literal["port"]]
        """choice getter

        The type of transmit and receive container used by the flow.

        Returns: Union[Literal["device"], Literal["port"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of transmit and receive container used by the flow.

        value: Union[Literal["device"], Literal["port"]]
        """
        self._set_property("choice", value)


class FlowPort(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "tx_name": {"type": str},
        "rx_name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("tx_name",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, tx_name=None, rx_name=None):
        super(FlowPort, self).__init__()
        self._parent = parent
        self._set_property("tx_name", tx_name)
        self._set_property("rx_name", rx_name)

    @property
    def tx_name(self):
        # type: () -> str
        """tx_name getter

        The unique name of a port that is the transmit port.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name.

        Returns: str
        """
        return self._get_property("tx_name")

    @tx_name.setter
    def tx_name(self, value):
        """tx_name setter

        The unique name of a port that is the transmit port.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name.

        value: str
        """
        self._set_property("tx_name", value)

    @property
    def rx_name(self):
        # type: () -> str
        """rx_name getter

        The unique name of a port that is the intended receive port.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name.

        Returns: str
        """
        return self._get_property("rx_name")

    @rx_name.setter
    def rx_name(self, value):
        """rx_name setter

        The unique name of a port that is the intended receive port.. . x-constraint:. - /components/schemas/Port/properties/name. - /components/schemas/Lag/properties/name.

        value: str
        """
        self._set_property("rx_name", value)


class FlowRouter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "mode": {
            "type": str,
            "enum": [
                "mesh",
                "one_to_one",
            ],
        },
        "tx_names": {
            "type": list,
            "itemtype": str,
        },
        "rx_names": {
            "type": list,
            "itemtype": str,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("tx_names", "rx_names")  # type: tuple(str)

    _DEFAULTS = {
        "mode": "mesh",
    }  # type: Dict[str, Union(type)]

    MESH = "mesh"  # type: str
    ONE_TO_ONE = "one_to_one"  # type: str

    def __init__(self, parent=None, mode="mesh", tx_names=None, rx_names=None):
        super(FlowRouter, self).__init__()
        self._parent = parent
        self._set_property("mode", mode)
        self._set_property("tx_names", tx_names)
        self._set_property("rx_names", rx_names)

    @property
    def mode(self):
        # type: () -> Union[Literal["mesh"], Literal["one_to_one"]]
        """mode getter

        Determines the mapping of tx_names to rx_names.. . The mesh mode creates traffic between each value in tx_names to. every value in rx_names except in the case where the individual . tx_names value is equal to rx_names value which will be ignored.. Identical values in tx_names or rx_names MUST raise an error.. . The one_to_one mode creates traffic between each value pair in . of tx_names and rx_names by index.. The length of tx_names and rx_name MUST be the same.. Identical values in tx_names or rx_names MUST raise an error.

        Returns: Union[Literal["mesh"], Literal["one_to_one"]]
        """
        return self._get_property("mode")

    @mode.setter
    def mode(self, value):
        """mode setter

        Determines the mapping of tx_names to rx_names.. . The mesh mode creates traffic between each value in tx_names to. every value in rx_names except in the case where the individual . tx_names value is equal to rx_names value which will be ignored.. Identical values in tx_names or rx_names MUST raise an error.. . The one_to_one mode creates traffic between each value pair in . of tx_names and rx_names by index.. The length of tx_names and rx_name MUST be the same.. Identical values in tx_names or rx_names MUST raise an error.

        value: Union[Literal["mesh"], Literal["one_to_one"]]
        """
        self._set_property("mode", value)

    @property
    def tx_names(self):
        # type: () -> List[str]
        """tx_names getter

        TBD. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Bgp.V4RouteRange/properties/name. - /components/schemas/Bgp.V6RouteRange/properties/name. - /components/schemas/Bgp.CMacIpRange/properties/name.

        Returns: List[str]
        """
        return self._get_property("tx_names")

    @tx_names.setter
    def tx_names(self, value):
        """tx_names setter

        TBD. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Bgp.V4RouteRange/properties/name. - /components/schemas/Bgp.V6RouteRange/properties/name. - /components/schemas/Bgp.CMacIpRange/properties/name.

        value: List[str]
        """
        self._set_property("tx_names", value)

    @property
    def rx_names(self):
        # type: () -> List[str]
        """rx_names getter

        TBD. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Bgp.V4RouteRange/properties/name. - /components/schemas/Bgp.V6RouteRange/properties/name. - /components/schemas/Bgp.CMacIpRange/properties/name.

        Returns: List[str]
        """
        return self._get_property("rx_names")

    @rx_names.setter
    def rx_names(self, value):
        """rx_names setter

        TBD. . x-constraint:. - /components/schemas/Device.Ethernet/properties/name. - /components/schemas/Device.Ipv4/properties/name. - /components/schemas/Device.Ipv6/properties/name. - /components/schemas/Bgp.V4RouteRange/properties/name. - /components/schemas/Bgp.V6RouteRange/properties/name. - /components/schemas/Bgp.CMacIpRange/properties/name.

        value: List[str]
        """
        self._set_property("rx_names", value)


class FlowHeader(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "custom",
                "ethernet",
                "vlan",
                "vxlan",
                "ipv4",
                "ipv6",
                "pfcpause",
                "ethernetpause",
                "tcp",
                "udp",
                "gre",
                "gtpv1",
                "gtpv2",
                "arp",
                "icmp",
                "icmpv6",
                "ppp",
                "igmpv1",
                "mpls",
            ],
        },
        "custom": {"type": "FlowCustom"},
        "ethernet": {"type": "FlowEthernet"},
        "vlan": {"type": "FlowVlan"},
        "vxlan": {"type": "FlowVxlan"},
        "ipv4": {"type": "FlowIpv4"},
        "ipv6": {"type": "FlowIpv6"},
        "pfcpause": {"type": "FlowPfcPause"},
        "ethernetpause": {"type": "FlowEthernetPause"},
        "tcp": {"type": "FlowTcp"},
        "udp": {"type": "FlowUdp"},
        "gre": {"type": "FlowGre"},
        "gtpv1": {"type": "FlowGtpv1"},
        "gtpv2": {"type": "FlowGtpv2"},
        "arp": {"type": "FlowArp"},
        "icmp": {"type": "FlowIcmp"},
        "icmpv6": {"type": "FlowIcmpv6"},
        "ppp": {"type": "FlowPpp"},
        "igmpv1": {"type": "FlowIgmpv1"},
        "mpls": {"type": "FlowMpls"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "ethernet",
    }  # type: Dict[str, Union(type)]

    CUSTOM = "custom"  # type: str
    ETHERNET = "ethernet"  # type: str
    VLAN = "vlan"  # type: str
    VXLAN = "vxlan"  # type: str
    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str
    PFCPAUSE = "pfcpause"  # type: str
    ETHERNETPAUSE = "ethernetpause"  # type: str
    TCP = "tcp"  # type: str
    UDP = "udp"  # type: str
    GRE = "gre"  # type: str
    GTPV1 = "gtpv1"  # type: str
    GTPV2 = "gtpv2"  # type: str
    ARP = "arp"  # type: str
    ICMP = "icmp"  # type: str
    ICMPV6 = "icmpv6"  # type: str
    PPP = "ppp"  # type: str
    IGMPV1 = "igmpv1"  # type: str
    MPLS = "mpls"  # type: str

    def __init__(self, parent=None, choice=None):
        super(FlowHeader, self).__init__()
        self._parent = parent
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def custom(self):
        # type: () -> FlowCustom
        """Factory property that returns an instance of the FlowCustom class

        Custom packet header

        Returns: FlowCustom
        """
        return self._get_property("custom", FlowCustom, self, "custom")

    @property
    def ethernet(self):
        # type: () -> FlowEthernet
        """Factory property that returns an instance of the FlowEthernet class

        Ethernet packet header

        Returns: FlowEthernet
        """
        return self._get_property("ethernet", FlowEthernet, self, "ethernet")

    @property
    def vlan(self):
        # type: () -> FlowVlan
        """Factory property that returns an instance of the FlowVlan class

        VLAN packet header

        Returns: FlowVlan
        """
        return self._get_property("vlan", FlowVlan, self, "vlan")

    @property
    def vxlan(self):
        # type: () -> FlowVxlan
        """Factory property that returns an instance of the FlowVxlan class

        VXLAN packet header

        Returns: FlowVxlan
        """
        return self._get_property("vxlan", FlowVxlan, self, "vxlan")

    @property
    def ipv4(self):
        # type: () -> FlowIpv4
        """Factory property that returns an instance of the FlowIpv4 class

        IPv4 packet header

        Returns: FlowIpv4
        """
        return self._get_property("ipv4", FlowIpv4, self, "ipv4")

    @property
    def ipv6(self):
        # type: () -> FlowIpv6
        """Factory property that returns an instance of the FlowIpv6 class

        IPv6 packet header

        Returns: FlowIpv6
        """
        return self._get_property("ipv6", FlowIpv6, self, "ipv6")

    @property
    def pfcpause(self):
        # type: () -> FlowPfcPause
        """Factory property that returns an instance of the FlowPfcPause class

        IEEE 802.1Qbb PFC Pause packet header.

        Returns: FlowPfcPause
        """
        return self._get_property("pfcpause", FlowPfcPause, self, "pfcpause")

    @property
    def ethernetpause(self):
        # type: () -> FlowEthernetPause
        """Factory property that returns an instance of the FlowEthernetPause class

        IEEE 802.3x global ethernet pause packet header

        Returns: FlowEthernetPause
        """
        return self._get_property(
            "ethernetpause", FlowEthernetPause, self, "ethernetpause"
        )

    @property
    def tcp(self):
        # type: () -> FlowTcp
        """Factory property that returns an instance of the FlowTcp class

        TCP packet header

        Returns: FlowTcp
        """
        return self._get_property("tcp", FlowTcp, self, "tcp")

    @property
    def udp(self):
        # type: () -> FlowUdp
        """Factory property that returns an instance of the FlowUdp class

        UDP packet header

        Returns: FlowUdp
        """
        return self._get_property("udp", FlowUdp, self, "udp")

    @property
    def gre(self):
        # type: () -> FlowGre
        """Factory property that returns an instance of the FlowGre class

        Standard GRE packet header (RFC2784)

        Returns: FlowGre
        """
        return self._get_property("gre", FlowGre, self, "gre")

    @property
    def gtpv1(self):
        # type: () -> FlowGtpv1
        """Factory property that returns an instance of the FlowGtpv1 class

        GTPv1 packet header

        Returns: FlowGtpv1
        """
        return self._get_property("gtpv1", FlowGtpv1, self, "gtpv1")

    @property
    def gtpv2(self):
        # type: () -> FlowGtpv2
        """Factory property that returns an instance of the FlowGtpv2 class

        GTPv2 packet header

        Returns: FlowGtpv2
        """
        return self._get_property("gtpv2", FlowGtpv2, self, "gtpv2")

    @property
    def arp(self):
        # type: () -> FlowArp
        """Factory property that returns an instance of the FlowArp class

        ARP packet header

        Returns: FlowArp
        """
        return self._get_property("arp", FlowArp, self, "arp")

    @property
    def icmp(self):
        # type: () -> FlowIcmp
        """Factory property that returns an instance of the FlowIcmp class

        ICMP packet header

        Returns: FlowIcmp
        """
        return self._get_property("icmp", FlowIcmp, self, "icmp")

    @property
    def icmpv6(self):
        # type: () -> FlowIcmpv6
        """Factory property that returns an instance of the FlowIcmpv6 class

        ICMPv6 packet header

        Returns: FlowIcmpv6
        """
        return self._get_property("icmpv6", FlowIcmpv6, self, "icmpv6")

    @property
    def ppp(self):
        # type: () -> FlowPpp
        """Factory property that returns an instance of the FlowPpp class

        PPP packet header

        Returns: FlowPpp
        """
        return self._get_property("ppp", FlowPpp, self, "ppp")

    @property
    def igmpv1(self):
        # type: () -> FlowIgmpv1
        """Factory property that returns an instance of the FlowIgmpv1 class

        IGMPv1 packet header

        Returns: FlowIgmpv1
        """
        return self._get_property("igmpv1", FlowIgmpv1, self, "igmpv1")

    @property
    def mpls(self):
        # type: () -> FlowMpls
        """Factory property that returns an instance of the FlowMpls class

        Mpls packet header

        Returns: FlowMpls
        """
        return self._get_property("mpls", FlowMpls, self, "mpls")

    @property
    def choice(self):
        # type: () -> Union[Literal["arp"], Literal["custom"], Literal["ethernet"], Literal["ethernetpause"], Literal["gre"], Literal["gtpv1"], Literal["gtpv2"], Literal["icmp"], Literal["icmpv6"], Literal["igmpv1"], Literal["ipv4"], Literal["ipv6"], Literal["mpls"], Literal["pfcpause"], Literal["ppp"], Literal["tcp"], Literal["udp"], Literal["vlan"], Literal["vxlan"]]
        """choice getter

        The available types of flow headers. If one is not provided the . default ethernet packet header MUST be provided.

        Returns: Union[Literal["arp"], Literal["custom"], Literal["ethernet"], Literal["ethernetpause"], Literal["gre"], Literal["gtpv1"], Literal["gtpv2"], Literal["icmp"], Literal["icmpv6"], Literal["igmpv1"], Literal["ipv4"], Literal["ipv6"], Literal["mpls"], Literal["pfcpause"], Literal["ppp"], Literal["tcp"], Literal["udp"], Literal["vlan"], Literal["vxlan"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The available types of flow headers. If one is not provided the . default ethernet packet header MUST be provided.

        value: Union[Literal["arp"], Literal["custom"], Literal["ethernet"], Literal["ethernetpause"], Literal["gre"], Literal["gtpv1"], Literal["gtpv2"], Literal["icmp"], Literal["icmpv6"], Literal["igmpv1"], Literal["ipv4"], Literal["ipv6"], Literal["mpls"], Literal["pfcpause"], Literal["ppp"], Literal["tcp"], Literal["udp"], Literal["vlan"], Literal["vxlan"]]
        """
        self._set_property("choice", value)


class FlowCustom(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "bytes": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("bytes",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, bytes=None):
        super(FlowCustom, self).__init__()
        self._parent = parent
        self._set_property("bytes", bytes)

    @property
    def bytes(self):
        # type: () -> str
        """bytes getter

        A custom packet header defined as a string of hex bytes. The string MUST contain sequence of valid hex bytes. Spaces or colons can be part of the bytes but will be discarded. This packet header can be used in multiple places in the packet.

        Returns: str
        """
        return self._get_property("bytes")

    @bytes.setter
    def bytes(self, value):
        """bytes setter

        A custom packet header defined as a string of hex bytes. The string MUST contain sequence of valid hex bytes. Spaces or colons can be part of the bytes but will be discarded. This packet header can be used in multiple places in the packet.

        value: str
        """
        self._set_property("bytes", value)


class FlowEthernet(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "dst": {"type": "PatternFlowEthernetDst"},
        "src": {"type": "PatternFlowEthernetSrc"},
        "ether_type": {"type": "PatternFlowEthernetEtherType"},
        "pfc_queue": {"type": "PatternFlowEthernetPfcQueue"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(FlowEthernet, self).__init__()
        self._parent = parent

    @property
    def dst(self):
        # type: () -> PatternFlowEthernetDst
        """dst getter

        Destination MAC addressDestination MAC addressDestination MAC address

        Returns: PatternFlowEthernetDst
        """
        return self._get_property("dst", PatternFlowEthernetDst)

    @property
    def src(self):
        # type: () -> PatternFlowEthernetSrc
        """src getter

        Source MAC addressSource MAC addressSource MAC address

        Returns: PatternFlowEthernetSrc
        """
        return self._get_property("src", PatternFlowEthernetSrc)

    @property
    def ether_type(self):
        # type: () -> PatternFlowEthernetEtherType
        """ether_type getter

        Ethernet typeEthernet typeEthernet type

        Returns: PatternFlowEthernetEtherType
        """
        return self._get_property("ether_type", PatternFlowEthernetEtherType)

    @property
    def pfc_queue(self):
        # type: () -> PatternFlowEthernetPfcQueue
        """pfc_queue getter

        Priority flow control queuePriority flow control queuePriority flow control queue

        Returns: PatternFlowEthernetPfcQueue
        """
        return self._get_property("pfc_queue", PatternFlowEthernetPfcQueue)


class PatternFlowEthernetDst(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "auto",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": str,
            "format": "mac",
        },
        "values": {
            "type": list,
            "format": "mac",
            "itemtype": str,
        },
        "auto": {
            "type": str,
            "format": "mac",
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowEthernetDstCounter"},
        "decrement": {"type": "PatternFlowEthernetDstCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "auto",
        "value": "00:00:00:00:00:00",
        "values": ["00:00:00:00:00:00"],
        "auto": "00:00:00:00:00:00",
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    AUTO = "auto"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value="00:00:00:00:00:00",
        values=["00:00:00:00:00:00"],
        auto="00:00:00:00:00:00",
        metric_group=None,
    ):
        super(PatternFlowEthernetDst, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("auto", auto)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetDstCounter
        """Factory property that returns an instance of the PatternFlowEthernetDstCounter class

        mac counter pattern

        Returns: PatternFlowEthernetDstCounter
        """
        return self._get_property(
            "increment", PatternFlowEthernetDstCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetDstCounter
        """Factory property that returns an instance of the PatternFlowEthernetDstCounter class

        mac counter pattern

        Returns: PatternFlowEthernetDstCounter
        """
        return self._get_property(
            "decrement", PatternFlowEthernetDstCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property("values", value, "values")

    @property
    def auto(self):
        # type: () -> str
        """auto getter

        The OTG implementation can provide a system generated. value for this property. If the OTG is unable to generate a value. the default value must be used.

        Returns: str
        """
        return self._get_property("auto")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowEthernetDstCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": str,
            "format": "mac",
        },
        "step": {
            "type": str,
            "format": "mac",
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": "00:00:00:00:00:00",
        "step": "00:00:00:00:00:01",
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, start="00:00:00:00:00:00", step="00:00:00:00:00:01", count=1
    ):
        super(PatternFlowEthernetDstCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowEthernetSrc(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": str,
            "format": "mac",
        },
        "values": {
            "type": list,
            "format": "mac",
            "itemtype": str,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowEthernetSrcCounter"},
        "decrement": {"type": "PatternFlowEthernetSrcCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": "00:00:00:00:00:00",
        "values": ["00:00:00:00:00:00"],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value="00:00:00:00:00:00",
        values=["00:00:00:00:00:00"],
        metric_group=None,
    ):
        super(PatternFlowEthernetSrc, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetSrcCounter
        """Factory property that returns an instance of the PatternFlowEthernetSrcCounter class

        mac counter pattern

        Returns: PatternFlowEthernetSrcCounter
        """
        return self._get_property(
            "increment", PatternFlowEthernetSrcCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetSrcCounter
        """Factory property that returns an instance of the PatternFlowEthernetSrcCounter class

        mac counter pattern

        Returns: PatternFlowEthernetSrcCounter
        """
        return self._get_property(
            "decrement", PatternFlowEthernetSrcCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowEthernetSrcCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": str,
            "format": "mac",
        },
        "step": {
            "type": str,
            "format": "mac",
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": "00:00:00:00:00:00",
        "step": "00:00:00:00:00:01",
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, start="00:00:00:00:00:00", step="00:00:00:00:00:01", count=1
    ):
        super(PatternFlowEthernetSrcCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowEthernetEtherType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "auto",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "auto": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowEthernetEtherTypeCounter"},
        "decrement": {"type": "PatternFlowEthernetEtherTypeCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "auto",
        "value": 65535,
        "values": [65535],
        "auto": 65535,
    }  # type: Dict[str, Union(type)]

    IPV4 = 2048  #
    IPV6 = 34525  #
    ARP = 2054  #
    VLAN_802_1_Q = 33024  #
    RESERVED = 65535  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    AUTO = "auto"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value=65535,
        values=[65535],
        auto=65535,
        metric_group=None,
    ):
        super(PatternFlowEthernetEtherType, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("auto", auto)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetEtherTypeCounter
        """Factory property that returns an instance of the PatternFlowEthernetEtherTypeCounter class

        integer counter pattern

        Returns: PatternFlowEthernetEtherTypeCounter
        """
        return self._get_property(
            "increment", PatternFlowEthernetEtherTypeCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetEtherTypeCounter
        """Factory property that returns an instance of the PatternFlowEthernetEtherTypeCounter class

        integer counter pattern

        Returns: PatternFlowEthernetEtherTypeCounter
        """
        return self._get_property(
            "decrement", PatternFlowEthernetEtherTypeCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def auto(self):
        # type: () -> int
        """auto getter

        The OTG implementation can provide a system generated. value for this property. If the OTG is unable to generate a value. the default value must be used.

        Returns: int
        """
        return self._get_property("auto")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowEthernetEtherTypeCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 65535,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    IPV4 = 2048  #
    IPV6 = 34525  #
    ARP = 2054  #
    VLAN_802_1_Q = 33024  #
    RESERVED = 65535  #

    def __init__(self, parent=None, start=65535, step=1, count=1):
        super(PatternFlowEthernetEtherTypeCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowEthernetPfcQueue(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 7,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowEthernetPfcQueueCounter"},
        "decrement": {"type": "PatternFlowEthernetPfcQueueCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowEthernetPfcQueue, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetPfcQueueCounter
        """Factory property that returns an instance of the PatternFlowEthernetPfcQueueCounter class

        integer counter pattern

        Returns: PatternFlowEthernetPfcQueueCounter
        """
        return self._get_property(
            "increment", PatternFlowEthernetPfcQueueCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetPfcQueueCounter
        """Factory property that returns an instance of the PatternFlowEthernetPfcQueueCounter class

        integer counter pattern

        Returns: PatternFlowEthernetPfcQueueCounter
        """
        return self._get_property(
            "decrement", PatternFlowEthernetPfcQueueCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowEthernetPfcQueueCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowEthernetPfcQueueCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class FlowVlan(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "priority": {"type": "PatternFlowVlanPriority"},
        "cfi": {"type": "PatternFlowVlanCfi"},
        "id": {"type": "PatternFlowVlanId"},
        "tpid": {"type": "PatternFlowVlanTpid"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(FlowVlan, self).__init__()
        self._parent = parent

    @property
    def priority(self):
        # type: () -> PatternFlowVlanPriority
        """priority getter

        Priority code pointPriority code pointPriority code point

        Returns: PatternFlowVlanPriority
        """
        return self._get_property("priority", PatternFlowVlanPriority)

    @property
    def cfi(self):
        # type: () -> PatternFlowVlanCfi
        """cfi getter

        Canonical format indicator or drop elegible indicatorCanonical format indicator or drop elegible indicatorCanonical format indicator or drop elegible indicator

        Returns: PatternFlowVlanCfi
        """
        return self._get_property("cfi", PatternFlowVlanCfi)

    @property
    def id(self):
        # type: () -> PatternFlowVlanId
        """id getter

        Vlan identifierVlan identifierVlan identifier

        Returns: PatternFlowVlanId
        """
        return self._get_property("id", PatternFlowVlanId)

    @property
    def tpid(self):
        # type: () -> PatternFlowVlanTpid
        """tpid getter

        Protocol identifierProtocol identifierProtocol identifier

        Returns: PatternFlowVlanTpid
        """
        return self._get_property("tpid", PatternFlowVlanTpid)


class PatternFlowVlanPriority(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 7,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowVlanPriorityCounter"},
        "decrement": {"type": "PatternFlowVlanPriorityCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowVlanPriority, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowVlanPriorityCounter
        """Factory property that returns an instance of the PatternFlowVlanPriorityCounter class

        integer counter pattern

        Returns: PatternFlowVlanPriorityCounter
        """
        return self._get_property(
            "increment", PatternFlowVlanPriorityCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowVlanPriorityCounter
        """Factory property that returns an instance of the PatternFlowVlanPriorityCounter class

        integer counter pattern

        Returns: PatternFlowVlanPriorityCounter
        """
        return self._get_property(
            "decrement", PatternFlowVlanPriorityCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowVlanPriorityCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowVlanPriorityCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowVlanCfi(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 1,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowVlanCfiCounter"},
        "decrement": {"type": "PatternFlowVlanCfiCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowVlanCfi, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowVlanCfiCounter
        """Factory property that returns an instance of the PatternFlowVlanCfiCounter class

        integer counter pattern

        Returns: PatternFlowVlanCfiCounter
        """
        return self._get_property(
            "increment", PatternFlowVlanCfiCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowVlanCfiCounter
        """Factory property that returns an instance of the PatternFlowVlanCfiCounter class

        integer counter pattern

        Returns: PatternFlowVlanCfiCounter
        """
        return self._get_property(
            "decrement", PatternFlowVlanCfiCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowVlanCfiCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowVlanCfiCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowVlanId(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 4095,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 4095,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowVlanIdCounter"},
        "decrement": {"type": "PatternFlowVlanIdCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowVlanId, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowVlanIdCounter
        """Factory property that returns an instance of the PatternFlowVlanIdCounter class

        integer counter pattern

        Returns: PatternFlowVlanIdCounter
        """
        return self._get_property(
            "increment", PatternFlowVlanIdCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowVlanIdCounter
        """Factory property that returns an instance of the PatternFlowVlanIdCounter class

        integer counter pattern

        Returns: PatternFlowVlanIdCounter
        """
        return self._get_property(
            "decrement", PatternFlowVlanIdCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowVlanIdCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 4095,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 4095,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowVlanIdCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowVlanTpid(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowVlanTpidCounter"},
        "decrement": {"type": "PatternFlowVlanTpidCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 33024,
        "values": [33024],
    }  # type: Dict[str, Union(type)]

    X8100 = 33024  #
    X88A8 = 34984  #
    X9100 = 37120  #
    X9200 = 37376  #
    X9300 = 37632  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=33024, values=[33024], metric_group=None
    ):
        super(PatternFlowVlanTpid, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowVlanTpidCounter
        """Factory property that returns an instance of the PatternFlowVlanTpidCounter class

        integer counter pattern

        Returns: PatternFlowVlanTpidCounter
        """
        return self._get_property(
            "increment", PatternFlowVlanTpidCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowVlanTpidCounter
        """Factory property that returns an instance of the PatternFlowVlanTpidCounter class

        integer counter pattern

        Returns: PatternFlowVlanTpidCounter
        """
        return self._get_property(
            "decrement", PatternFlowVlanTpidCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowVlanTpidCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 33024,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    X8100 = 33024  #
    X88A8 = 34984  #
    X9100 = 37120  #
    X9200 = 37376  #
    X9300 = 37632  #

    def __init__(self, parent=None, start=33024, step=1, count=1):
        super(PatternFlowVlanTpidCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class FlowVxlan(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {"type": "PatternFlowVxlanFlags"},
        "reserved0": {"type": "PatternFlowVxlanReserved0"},
        "vni": {"type": "PatternFlowVxlanVni"},
        "reserved1": {"type": "PatternFlowVxlanReserved1"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(FlowVxlan, self).__init__()
        self._parent = parent

    @property
    def flags(self):
        # type: () -> PatternFlowVxlanFlags
        """flags getter

        Flags field with a bit format of RRRRIRRR. The I flag MUST be set to 1 for a valid vxlan network id (VNI). The other 7 bits (designated "R") are reserved fields and MUST be set to zero on transmission and ignored on receipt.Flags field with a bit format of RRRRIRRR. The I flag MUST be set to 1 for a valid vxlan network id (VNI). The other 7 bits (designated "R") are reserved fields and MUST be set to zero on transmission and ignored on receipt.Flags field with a bit format of RRRRIRRR. The I flag MUST be set to 1 for a valid vxlan network id (VNI). The other 7 bits (designated "R") are reserved fields and MUST be set to zero on transmission and ignored on receipt.

        Returns: PatternFlowVxlanFlags
        """
        return self._get_property("flags", PatternFlowVxlanFlags)

    @property
    def reserved0(self):
        # type: () -> PatternFlowVxlanReserved0
        """reserved0 getter

        Reserved fieldReserved fieldReserved field

        Returns: PatternFlowVxlanReserved0
        """
        return self._get_property("reserved0", PatternFlowVxlanReserved0)

    @property
    def vni(self):
        # type: () -> PatternFlowVxlanVni
        """vni getter

        VXLAN network idVXLAN network idVXLAN network id

        Returns: PatternFlowVxlanVni
        """
        return self._get_property("vni", PatternFlowVxlanVni)

    @property
    def reserved1(self):
        # type: () -> PatternFlowVxlanReserved1
        """reserved1 getter

        Reserved fieldReserved fieldReserved field

        Returns: PatternFlowVxlanReserved1
        """
        return self._get_property("reserved1", PatternFlowVxlanReserved1)


class PatternFlowVxlanFlags(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 255,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowVxlanFlagsCounter"},
        "decrement": {"type": "PatternFlowVxlanFlagsCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 8,
        "values": [8],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=8, values=[8], metric_group=None
    ):
        super(PatternFlowVxlanFlags, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowVxlanFlagsCounter
        """Factory property that returns an instance of the PatternFlowVxlanFlagsCounter class

        integer counter pattern

        Returns: PatternFlowVxlanFlagsCounter
        """
        return self._get_property(
            "increment", PatternFlowVxlanFlagsCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowVxlanFlagsCounter
        """Factory property that returns an instance of the PatternFlowVxlanFlagsCounter class

        integer counter pattern

        Returns: PatternFlowVxlanFlagsCounter
        """
        return self._get_property(
            "decrement", PatternFlowVxlanFlagsCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowVxlanFlagsCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 8,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=8, step=1, count=1):
        super(PatternFlowVxlanFlagsCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowVxlanReserved0(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowVxlanReserved0Counter"},
        "decrement": {"type": "PatternFlowVxlanReserved0Counter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowVxlanReserved0, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowVxlanReserved0Counter
        """Factory property that returns an instance of the PatternFlowVxlanReserved0Counter class

        integer counter pattern

        Returns: PatternFlowVxlanReserved0Counter
        """
        return self._get_property(
            "increment", PatternFlowVxlanReserved0Counter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowVxlanReserved0Counter
        """Factory property that returns an instance of the PatternFlowVxlanReserved0Counter class

        integer counter pattern

        Returns: PatternFlowVxlanReserved0Counter
        """
        return self._get_property(
            "decrement", PatternFlowVxlanReserved0Counter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowVxlanReserved0Counter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowVxlanReserved0Counter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowVxlanVni(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "auto",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "auto": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowVxlanVniCounter"},
        "decrement": {"type": "PatternFlowVxlanVniCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "auto",
        "value": 0,
        "values": [0],
        "auto": 0,
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    AUTO = "auto"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], auto=0, metric_group=None
    ):
        super(PatternFlowVxlanVni, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("auto", auto)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowVxlanVniCounter
        """Factory property that returns an instance of the PatternFlowVxlanVniCounter class

        integer counter pattern

        Returns: PatternFlowVxlanVniCounter
        """
        return self._get_property(
            "increment", PatternFlowVxlanVniCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowVxlanVniCounter
        """Factory property that returns an instance of the PatternFlowVxlanVniCounter class

        integer counter pattern

        Returns: PatternFlowVxlanVniCounter
        """
        return self._get_property(
            "decrement", PatternFlowVxlanVniCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def auto(self):
        # type: () -> int
        """auto getter

        The OTG implementation can provide a system generated. value for this property. If the OTG is unable to generate a value. the default value must be used.

        Returns: int
        """
        return self._get_property("auto")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowVxlanVniCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 16777215,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowVxlanVniCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowVxlanReserved1(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 255,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowVxlanReserved1Counter"},
        "decrement": {"type": "PatternFlowVxlanReserved1Counter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowVxlanReserved1, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowVxlanReserved1Counter
        """Factory property that returns an instance of the PatternFlowVxlanReserved1Counter class

        integer counter pattern

        Returns: PatternFlowVxlanReserved1Counter
        """
        return self._get_property(
            "increment", PatternFlowVxlanReserved1Counter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowVxlanReserved1Counter
        """Factory property that returns an instance of the PatternFlowVxlanReserved1Counter class

        integer counter pattern

        Returns: PatternFlowVxlanReserved1Counter
        """
        return self._get_property(
            "decrement", PatternFlowVxlanReserved1Counter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowVxlanReserved1Counter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowVxlanReserved1Counter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class FlowIpv4(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "version": {"type": "PatternFlowIpv4Version"},
        "header_length": {"type": "PatternFlowIpv4HeaderLength"},
        "priority": {"type": "FlowIpv4Priority"},
        "total_length": {"type": "PatternFlowIpv4TotalLength"},
        "identification": {"type": "PatternFlowIpv4Identification"},
        "reserved": {"type": "PatternFlowIpv4Reserved"},
        "dont_fragment": {"type": "PatternFlowIpv4DontFragment"},
        "more_fragments": {"type": "PatternFlowIpv4MoreFragments"},
        "fragment_offset": {"type": "PatternFlowIpv4FragmentOffset"},
        "time_to_live": {"type": "PatternFlowIpv4TimeToLive"},
        "protocol": {"type": "PatternFlowIpv4Protocol"},
        "header_checksum": {"type": "PatternFlowIpv4HeaderChecksum"},
        "src": {"type": "PatternFlowIpv4Src"},
        "dst": {"type": "PatternFlowIpv4Dst"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(FlowIpv4, self).__init__()
        self._parent = parent

    @property
    def version(self):
        # type: () -> PatternFlowIpv4Version
        """version getter

        VersionVersionVersion

        Returns: PatternFlowIpv4Version
        """
        return self._get_property("version", PatternFlowIpv4Version)

    @property
    def header_length(self):
        # type: () -> PatternFlowIpv4HeaderLength
        """header_length getter

        Header lengthHeader lengthHeader length

        Returns: PatternFlowIpv4HeaderLength
        """
        return self._get_property("header_length", PatternFlowIpv4HeaderLength)

    @property
    def priority(self):
        # type: () -> FlowIpv4Priority
        """priority getter

        A container for ipv4 raw, tos, dscp ip priorities.A container for ipv4 raw, tos, dscp ip priorities.A container for ipv4 raw, tos, dscp ip priorities.

        Returns: FlowIpv4Priority
        """
        return self._get_property("priority", FlowIpv4Priority)

    @property
    def total_length(self):
        # type: () -> PatternFlowIpv4TotalLength
        """total_length getter

        Total lengthTotal lengthTotal length

        Returns: PatternFlowIpv4TotalLength
        """
        return self._get_property("total_length", PatternFlowIpv4TotalLength)

    @property
    def identification(self):
        # type: () -> PatternFlowIpv4Identification
        """identification getter

        IdentificationIdentificationIdentification

        Returns: PatternFlowIpv4Identification
        """
        return self._get_property("identification", PatternFlowIpv4Identification)

    @property
    def reserved(self):
        # type: () -> PatternFlowIpv4Reserved
        """reserved getter

        Reserved flag.Reserved flag.Reserved flag.

        Returns: PatternFlowIpv4Reserved
        """
        return self._get_property("reserved", PatternFlowIpv4Reserved)

    @property
    def dont_fragment(self):
        # type: () -> PatternFlowIpv4DontFragment
        """dont_fragment getter

        Dont fragment flag If the dont_fragment flag is set and fragmentation is required to route the packet then the packet is dropped.Dont fragment flag If the dont_fragment flag is set and fragmentation is required to route the packet then the packet is dropped.Dont fragment flag If the dont_fragment flag is set and fragmentation is required to route the packet then the packet is dropped.

        Returns: PatternFlowIpv4DontFragment
        """
        return self._get_property("dont_fragment", PatternFlowIpv4DontFragment)

    @property
    def more_fragments(self):
        # type: () -> PatternFlowIpv4MoreFragments
        """more_fragments getter

        More fragments flagMore fragments flagMore fragments flag

        Returns: PatternFlowIpv4MoreFragments
        """
        return self._get_property("more_fragments", PatternFlowIpv4MoreFragments)

    @property
    def fragment_offset(self):
        # type: () -> PatternFlowIpv4FragmentOffset
        """fragment_offset getter

        Fragment offsetFragment offsetFragment offset

        Returns: PatternFlowIpv4FragmentOffset
        """
        return self._get_property("fragment_offset", PatternFlowIpv4FragmentOffset)

    @property
    def time_to_live(self):
        # type: () -> PatternFlowIpv4TimeToLive
        """time_to_live getter

        Time to liveTime to liveTime to live

        Returns: PatternFlowIpv4TimeToLive
        """
        return self._get_property("time_to_live", PatternFlowIpv4TimeToLive)

    @property
    def protocol(self):
        # type: () -> PatternFlowIpv4Protocol
        """protocol getter

        Protocol, default is 61 any host internal protocolProtocol, default is 61 any host internal protocolProtocol, default is 61 any host internal protocol

        Returns: PatternFlowIpv4Protocol
        """
        return self._get_property("protocol", PatternFlowIpv4Protocol)

    @property
    def header_checksum(self):
        # type: () -> PatternFlowIpv4HeaderChecksum
        """header_checksum getter

        Header checksumHeader checksumHeader checksum

        Returns: PatternFlowIpv4HeaderChecksum
        """
        return self._get_property("header_checksum", PatternFlowIpv4HeaderChecksum)

    @property
    def src(self):
        # type: () -> PatternFlowIpv4Src
        """src getter

        Source addressSource addressSource address

        Returns: PatternFlowIpv4Src
        """
        return self._get_property("src", PatternFlowIpv4Src)

    @property
    def dst(self):
        # type: () -> PatternFlowIpv4Dst
        """dst getter

        Destination addressDestination addressDestination address

        Returns: PatternFlowIpv4Dst
        """
        return self._get_property("dst", PatternFlowIpv4Dst)


class PatternFlowIpv4Version(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 15,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 15,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4VersionCounter"},
        "decrement": {"type": "PatternFlowIpv4VersionCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 4,
        "values": [4],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=4, values=[4], metric_group=None
    ):
        super(PatternFlowIpv4Version, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4VersionCounter
        """Factory property that returns an instance of the PatternFlowIpv4VersionCounter class

        integer counter pattern

        Returns: PatternFlowIpv4VersionCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4VersionCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4VersionCounter
        """Factory property that returns an instance of the PatternFlowIpv4VersionCounter class

        integer counter pattern

        Returns: PatternFlowIpv4VersionCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4VersionCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4VersionCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 15,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 15,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 4,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=4, step=1, count=1):
        super(PatternFlowIpv4VersionCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4HeaderLength(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "auto",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 15,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 15,
        },
        "auto": {
            "type": int,
            "minimum": 0,
            "maximum": 15,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4HeaderLengthCounter"},
        "decrement": {"type": "PatternFlowIpv4HeaderLengthCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "auto",
        "value": 5,
        "values": [5],
        "auto": 5,
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    AUTO = "auto"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=5, values=[5], auto=5, metric_group=None
    ):
        super(PatternFlowIpv4HeaderLength, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("auto", auto)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4HeaderLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv4HeaderLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv4HeaderLengthCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4HeaderLengthCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4HeaderLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv4HeaderLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv4HeaderLengthCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4HeaderLengthCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def auto(self):
        # type: () -> int
        """auto getter

        The OTG implementation can provide a system generated. value for this property. If the OTG is unable to generate a value. the default value must be used.

        Returns: int
        """
        return self._get_property("auto")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4HeaderLengthCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 15,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 15,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 5,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=5, step=1, count=1):
        super(PatternFlowIpv4HeaderLengthCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class FlowIpv4Priority(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "raw",
                "tos",
                "dscp",
            ],
        },
        "raw": {"type": "PatternFlowIpv4PriorityRaw"},
        "tos": {"type": "FlowIpv4Tos"},
        "dscp": {"type": "FlowIpv4Dscp"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "dscp",
    }  # type: Dict[str, Union(type)]

    RAW = "raw"  # type: str
    TOS = "tos"  # type: str
    DSCP = "dscp"  # type: str

    def __init__(self, parent=None, choice=None):
        super(FlowIpv4Priority, self).__init__()
        self._parent = parent
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def raw(self):
        # type: () -> PatternFlowIpv4PriorityRaw
        """Factory property that returns an instance of the PatternFlowIpv4PriorityRaw class

        Raw priority

        Returns: PatternFlowIpv4PriorityRaw
        """
        return self._get_property("raw", PatternFlowIpv4PriorityRaw, self, "raw")

    @property
    def tos(self):
        # type: () -> FlowIpv4Tos
        """Factory property that returns an instance of the FlowIpv4Tos class

        Type of service (TOS) packet field.

        Returns: FlowIpv4Tos
        """
        return self._get_property("tos", FlowIpv4Tos, self, "tos")

    @property
    def dscp(self):
        # type: () -> FlowIpv4Dscp
        """Factory property that returns an instance of the FlowIpv4Dscp class

        Differentiated services code point (DSCP) packet field.

        Returns: FlowIpv4Dscp
        """
        return self._get_property("dscp", FlowIpv4Dscp, self, "dscp")

    @property
    def choice(self):
        # type: () -> Union[Literal["dscp"], Literal["raw"], Literal["tos"]]
        """choice getter

        TBD

        Returns: Union[Literal["dscp"], Literal["raw"], Literal["tos"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["dscp"], Literal["raw"], Literal["tos"]]
        """
        self._set_property("choice", value)


class PatternFlowIpv4PriorityRaw(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 255,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4PriorityRawCounter"},
        "decrement": {"type": "PatternFlowIpv4PriorityRawCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4PriorityRaw, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4PriorityRawCounter
        """Factory property that returns an instance of the PatternFlowIpv4PriorityRawCounter class

        integer counter pattern

        Returns: PatternFlowIpv4PriorityRawCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4PriorityRawCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4PriorityRawCounter
        """Factory property that returns an instance of the PatternFlowIpv4PriorityRawCounter class

        integer counter pattern

        Returns: PatternFlowIpv4PriorityRawCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4PriorityRawCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4PriorityRawCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4PriorityRawCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class FlowIpv4Tos(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "precedence": {"type": "PatternFlowIpv4TosPrecedence"},
        "delay": {"type": "PatternFlowIpv4TosDelay"},
        "throughput": {"type": "PatternFlowIpv4TosThroughput"},
        "reliability": {"type": "PatternFlowIpv4TosReliability"},
        "monetary": {"type": "PatternFlowIpv4TosMonetary"},
        "unused": {"type": "PatternFlowIpv4TosUnused"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(FlowIpv4Tos, self).__init__()
        self._parent = parent

    @property
    def precedence(self):
        # type: () -> PatternFlowIpv4TosPrecedence
        """precedence getter

        PrecedencePrecedencePrecedence

        Returns: PatternFlowIpv4TosPrecedence
        """
        return self._get_property("precedence", PatternFlowIpv4TosPrecedence)

    @property
    def delay(self):
        # type: () -> PatternFlowIpv4TosDelay
        """delay getter

        DelayDelayDelay

        Returns: PatternFlowIpv4TosDelay
        """
        return self._get_property("delay", PatternFlowIpv4TosDelay)

    @property
    def throughput(self):
        # type: () -> PatternFlowIpv4TosThroughput
        """throughput getter

        ThroughputThroughputThroughput

        Returns: PatternFlowIpv4TosThroughput
        """
        return self._get_property("throughput", PatternFlowIpv4TosThroughput)

    @property
    def reliability(self):
        # type: () -> PatternFlowIpv4TosReliability
        """reliability getter

        ReliabilityReliabilityReliability

        Returns: PatternFlowIpv4TosReliability
        """
        return self._get_property("reliability", PatternFlowIpv4TosReliability)

    @property
    def monetary(self):
        # type: () -> PatternFlowIpv4TosMonetary
        """monetary getter

        MonetaryMonetaryMonetary

        Returns: PatternFlowIpv4TosMonetary
        """
        return self._get_property("monetary", PatternFlowIpv4TosMonetary)

    @property
    def unused(self):
        # type: () -> PatternFlowIpv4TosUnused
        """unused getter

        UnusedUnusedUnused

        Returns: PatternFlowIpv4TosUnused
        """
        return self._get_property("unused", PatternFlowIpv4TosUnused)


class PatternFlowIpv4TosPrecedence(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 7,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4TosPrecedenceCounter"},
        "decrement": {"type": "PatternFlowIpv4TosPrecedenceCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    ROUTINE = 0  #
    PRIORITY = 1  #
    IMMEDIATE = 2  #
    FLASH = 3  #
    FLASH_OVERRIDE = 4  #
    CRITIC_ECP = 5  #
    INTERNETWORK_CONTROL = 6  #
    NETWORK_CONTROL = 7  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4TosPrecedence, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosPrecedenceCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosPrecedenceCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosPrecedenceCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4TosPrecedenceCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosPrecedenceCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosPrecedenceCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosPrecedenceCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4TosPrecedenceCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4TosPrecedenceCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 7,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    ROUTINE = 0  #
    PRIORITY = 1  #
    IMMEDIATE = 2  #
    FLASH = 3  #
    FLASH_OVERRIDE = 4  #
    CRITIC_ECP = 5  #
    INTERNETWORK_CONTROL = 6  #
    NETWORK_CONTROL = 7  #

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4TosPrecedenceCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4TosDelay(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 1,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4TosDelayCounter"},
        "decrement": {"type": "PatternFlowIpv4TosDelayCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    NORMAL = 0  #
    LOW = 1  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4TosDelay, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosDelayCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosDelayCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosDelayCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4TosDelayCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosDelayCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosDelayCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosDelayCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4TosDelayCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4TosDelayCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    NORMAL = 0  #
    LOW = 1  #

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4TosDelayCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4TosThroughput(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 1,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4TosThroughputCounter"},
        "decrement": {"type": "PatternFlowIpv4TosThroughputCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    NORMAL = 0  #
    LOW = 1  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4TosThroughput, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosThroughputCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosThroughputCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosThroughputCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4TosThroughputCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosThroughputCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosThroughputCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosThroughputCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4TosThroughputCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4TosThroughputCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    NORMAL = 0  #
    LOW = 1  #

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4TosThroughputCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4TosReliability(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 1,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4TosReliabilityCounter"},
        "decrement": {"type": "PatternFlowIpv4TosReliabilityCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    NORMAL = 0  #
    LOW = 1  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4TosReliability, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosReliabilityCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosReliabilityCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosReliabilityCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4TosReliabilityCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosReliabilityCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosReliabilityCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosReliabilityCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4TosReliabilityCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4TosReliabilityCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    NORMAL = 0  #
    LOW = 1  #

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4TosReliabilityCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4TosMonetary(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 1,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4TosMonetaryCounter"},
        "decrement": {"type": "PatternFlowIpv4TosMonetaryCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    NORMAL = 0  #
    LOW = 1  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4TosMonetary, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosMonetaryCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosMonetaryCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosMonetaryCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4TosMonetaryCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosMonetaryCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosMonetaryCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosMonetaryCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4TosMonetaryCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4TosMonetaryCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    NORMAL = 0  #
    LOW = 1  #

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4TosMonetaryCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4TosUnused(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 1,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4TosUnusedCounter"},
        "decrement": {"type": "PatternFlowIpv4TosUnusedCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4TosUnused, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TosUnusedCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosUnusedCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosUnusedCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4TosUnusedCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TosUnusedCounter
        """Factory property that returns an instance of the PatternFlowIpv4TosUnusedCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TosUnusedCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4TosUnusedCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4TosUnusedCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4TosUnusedCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class FlowIpv4Dscp(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "phb": {"type": "PatternFlowIpv4DscpPhb"},
        "ecn": {"type": "PatternFlowIpv4DscpEcn"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(FlowIpv4Dscp, self).__init__()
        self._parent = parent

    @property
    def phb(self):
        # type: () -> PatternFlowIpv4DscpPhb
        """phb getter

        Per hop behaviorPer hop behaviorPer hop behavior

        Returns: PatternFlowIpv4DscpPhb
        """
        return self._get_property("phb", PatternFlowIpv4DscpPhb)

    @property
    def ecn(self):
        # type: () -> PatternFlowIpv4DscpEcn
        """ecn getter

        Explicit congestion notificationExplicit congestion notificationExplicit congestion notification

        Returns: PatternFlowIpv4DscpEcn
        """
        return self._get_property("ecn", PatternFlowIpv4DscpEcn)


class PatternFlowIpv4DscpPhb(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 63,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 63,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4DscpPhbCounter"},
        "decrement": {"type": "PatternFlowIpv4DscpPhbCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    DEFAULT = 0  #
    CS1 = 8  #
    CS2 = 16  #
    CS3 = 24  #
    CS4 = 32  #
    CS5 = 40  #
    CS6 = 48  #
    CS7 = 56  #
    AF11 = 10  #
    AF12 = 12  #
    AF13 = 14  #
    AF21 = 18  #
    AF22 = 20  #
    AF23 = 22  #
    AF31 = 26  #
    AF32 = 28  #
    AF33 = 30  #
    AF41 = 34  #
    AF42 = 36  #
    AF43 = 38  #
    EF46 = 46  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4DscpPhb, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4DscpPhbCounter
        """Factory property that returns an instance of the PatternFlowIpv4DscpPhbCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DscpPhbCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4DscpPhbCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4DscpPhbCounter
        """Factory property that returns an instance of the PatternFlowIpv4DscpPhbCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DscpPhbCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4DscpPhbCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4DscpPhbCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 63,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 63,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    DEFAULT = 0  #
    CS1 = 8  #
    CS2 = 16  #
    CS3 = 24  #
    CS4 = 32  #
    CS5 = 40  #
    CS6 = 48  #
    CS7 = 56  #
    AF11 = 10  #
    AF12 = 12  #
    AF13 = 14  #
    AF21 = 18  #
    AF22 = 20  #
    AF23 = 22  #
    AF31 = 26  #
    AF32 = 28  #
    AF33 = 30  #
    AF41 = 34  #
    AF42 = 36  #
    AF43 = 38  #
    EF46 = 46  #

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4DscpPhbCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4DscpEcn(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 3,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 3,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4DscpEcnCounter"},
        "decrement": {"type": "PatternFlowIpv4DscpEcnCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    NON_CAPABLE = 0  #
    CAPABLE_TRANSPORT_0 = 1  #
    CAPABLE_TRANSPORT_1 = 2  #
    CONGESTION_ENCOUNTERED = 3  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4DscpEcn, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4DscpEcnCounter
        """Factory property that returns an instance of the PatternFlowIpv4DscpEcnCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DscpEcnCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4DscpEcnCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4DscpEcnCounter
        """Factory property that returns an instance of the PatternFlowIpv4DscpEcnCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DscpEcnCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4DscpEcnCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4DscpEcnCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 3,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 3,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    NON_CAPABLE = 0  #
    CAPABLE_TRANSPORT_0 = 1  #
    CAPABLE_TRANSPORT_1 = 2  #
    CONGESTION_ENCOUNTERED = 3  #

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4DscpEcnCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4TotalLength(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "auto",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "auto": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4TotalLengthCounter"},
        "decrement": {"type": "PatternFlowIpv4TotalLengthCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "auto",
        "value": 46,
        "values": [46],
        "auto": 46,
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    AUTO = "auto"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value=46,
        values=[46],
        auto=46,
        metric_group=None,
    ):
        super(PatternFlowIpv4TotalLength, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("auto", auto)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TotalLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv4TotalLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TotalLengthCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4TotalLengthCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TotalLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv4TotalLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TotalLengthCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4TotalLengthCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def auto(self):
        # type: () -> int
        """auto getter

        The OTG implementation can provide a system generated. value for this property. If the OTG is unable to generate a value. the default value must be used.

        Returns: int
        """
        return self._get_property("auto")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4TotalLengthCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 46,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=46, step=1, count=1):
        super(PatternFlowIpv4TotalLengthCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4Identification(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4IdentificationCounter"},
        "decrement": {"type": "PatternFlowIpv4IdentificationCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4Identification, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4IdentificationCounter
        """Factory property that returns an instance of the PatternFlowIpv4IdentificationCounter class

        integer counter pattern

        Returns: PatternFlowIpv4IdentificationCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4IdentificationCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4IdentificationCounter
        """Factory property that returns an instance of the PatternFlowIpv4IdentificationCounter class

        integer counter pattern

        Returns: PatternFlowIpv4IdentificationCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4IdentificationCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4IdentificationCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4IdentificationCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4Reserved(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 1,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4ReservedCounter"},
        "decrement": {"type": "PatternFlowIpv4ReservedCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4Reserved, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4ReservedCounter
        """Factory property that returns an instance of the PatternFlowIpv4ReservedCounter class

        integer counter pattern

        Returns: PatternFlowIpv4ReservedCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4ReservedCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4ReservedCounter
        """Factory property that returns an instance of the PatternFlowIpv4ReservedCounter class

        integer counter pattern

        Returns: PatternFlowIpv4ReservedCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4ReservedCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4ReservedCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4ReservedCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4DontFragment(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 1,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4DontFragmentCounter"},
        "decrement": {"type": "PatternFlowIpv4DontFragmentCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4DontFragment, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4DontFragmentCounter
        """Factory property that returns an instance of the PatternFlowIpv4DontFragmentCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DontFragmentCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4DontFragmentCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4DontFragmentCounter
        """Factory property that returns an instance of the PatternFlowIpv4DontFragmentCounter class

        integer counter pattern

        Returns: PatternFlowIpv4DontFragmentCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4DontFragmentCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4DontFragmentCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4DontFragmentCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4MoreFragments(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 1,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4MoreFragmentsCounter"},
        "decrement": {"type": "PatternFlowIpv4MoreFragmentsCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4MoreFragments, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4MoreFragmentsCounter
        """Factory property that returns an instance of the PatternFlowIpv4MoreFragmentsCounter class

        integer counter pattern

        Returns: PatternFlowIpv4MoreFragmentsCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4MoreFragmentsCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4MoreFragmentsCounter
        """Factory property that returns an instance of the PatternFlowIpv4MoreFragmentsCounter class

        integer counter pattern

        Returns: PatternFlowIpv4MoreFragmentsCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4MoreFragmentsCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4MoreFragmentsCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 1,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4MoreFragmentsCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4FragmentOffset(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 31,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 31,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4FragmentOffsetCounter"},
        "decrement": {"type": "PatternFlowIpv4FragmentOffsetCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv4FragmentOffset, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4FragmentOffsetCounter
        """Factory property that returns an instance of the PatternFlowIpv4FragmentOffsetCounter class

        integer counter pattern

        Returns: PatternFlowIpv4FragmentOffsetCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4FragmentOffsetCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4FragmentOffsetCounter
        """Factory property that returns an instance of the PatternFlowIpv4FragmentOffsetCounter class

        integer counter pattern

        Returns: PatternFlowIpv4FragmentOffsetCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4FragmentOffsetCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4FragmentOffsetCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 31,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 31,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv4FragmentOffsetCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4TimeToLive(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 255,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4TimeToLiveCounter"},
        "decrement": {"type": "PatternFlowIpv4TimeToLiveCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 64,
        "values": [64],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=64, values=[64], metric_group=None
    ):
        super(PatternFlowIpv4TimeToLive, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4TimeToLiveCounter
        """Factory property that returns an instance of the PatternFlowIpv4TimeToLiveCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TimeToLiveCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4TimeToLiveCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4TimeToLiveCounter
        """Factory property that returns an instance of the PatternFlowIpv4TimeToLiveCounter class

        integer counter pattern

        Returns: PatternFlowIpv4TimeToLiveCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4TimeToLiveCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4TimeToLiveCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 64,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=64, step=1, count=1):
        super(PatternFlowIpv4TimeToLiveCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4Protocol(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "auto",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 255,
        },
        "auto": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4ProtocolCounter"},
        "decrement": {"type": "PatternFlowIpv4ProtocolCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "auto",
        "value": 61,
        "values": [61],
        "auto": 61,
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    AUTO = "auto"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value=61,
        values=[61],
        auto=61,
        metric_group=None,
    ):
        super(PatternFlowIpv4Protocol, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("auto", auto)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4ProtocolCounter
        """Factory property that returns an instance of the PatternFlowIpv4ProtocolCounter class

        integer counter pattern

        Returns: PatternFlowIpv4ProtocolCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4ProtocolCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4ProtocolCounter
        """Factory property that returns an instance of the PatternFlowIpv4ProtocolCounter class

        integer counter pattern

        Returns: PatternFlowIpv4ProtocolCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4ProtocolCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def auto(self):
        # type: () -> int
        """auto getter

        The OTG implementation can provide a system generated. value for this property. If the OTG is unable to generate a value. the default value must be used.

        Returns: int
        """
        return self._get_property("auto")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4ProtocolCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 61,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=61, step=1, count=1):
        super(PatternFlowIpv4ProtocolCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4HeaderChecksum(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "generated",
                "custom",
            ],
        },
        "generated": {
            "type": str,
            "enum": [
                "good",
                "bad",
            ],
        },
        "custom": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "generated",
        "generated": "good",
    }  # type: Dict[str, Union(type)]

    GENERATED = "generated"  # type: str
    CUSTOM = "custom"  # type: str

    GOOD = "good"  # type: str
    BAD = "bad"  # type: str

    def __init__(self, parent=None, choice=None, generated="good", custom=None):
        super(PatternFlowIpv4HeaderChecksum, self).__init__()
        self._parent = parent
        self._set_property("generated", generated)
        self._set_property("custom", custom)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def choice(self):
        # type: () -> Union[Literal["custom"], Literal["generated"]]
        """choice getter

        The type of checksum

        Returns: Union[Literal["custom"], Literal["generated"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of checksum

        value: Union[Literal["custom"], Literal["generated"]]
        """
        self._set_property("choice", value)

    @property
    def generated(self):
        # type: () -> Union[Literal["bad"], Literal["good"]]
        """generated getter

        A system generated checksum value

        Returns: Union[Literal["bad"], Literal["good"]]
        """
        return self._get_property("generated")

    @generated.setter
    def generated(self, value):
        """generated setter

        A system generated checksum value

        value: Union[Literal["bad"], Literal["good"]]
        """
        self._set_property("generated", value, "generated")

    @property
    def custom(self):
        # type: () -> int
        """custom getter

        A custom checksum value

        Returns: int
        """
        return self._get_property("custom")

    @custom.setter
    def custom(self, value):
        """custom setter

        A custom checksum value

        value: int
        """
        self._set_property("custom", value, "custom")


class PatternFlowIpv4Src(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": str,
            "format": "ipv4",
        },
        "values": {
            "type": list,
            "format": "ipv4",
            "itemtype": str,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4SrcCounter"},
        "decrement": {"type": "PatternFlowIpv4SrcCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": "0.0.0.0",
        "values": ["0.0.0.0"],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value="0.0.0.0",
        values=["0.0.0.0"],
        metric_group=None,
    ):
        super(PatternFlowIpv4Src, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4SrcCounter
        """Factory property that returns an instance of the PatternFlowIpv4SrcCounter class

        ipv4 counter pattern

        Returns: PatternFlowIpv4SrcCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4SrcCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4SrcCounter
        """Factory property that returns an instance of the PatternFlowIpv4SrcCounter class

        ipv4 counter pattern

        Returns: PatternFlowIpv4SrcCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4SrcCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4SrcCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": str,
            "format": "ipv4",
        },
        "step": {
            "type": str,
            "format": "ipv4",
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": "0.0.0.0",
        "step": "0.0.0.1",
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start="0.0.0.0", step="0.0.0.1", count=1):
        super(PatternFlowIpv4SrcCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv4Dst(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": str,
            "format": "ipv4",
        },
        "values": {
            "type": list,
            "format": "ipv4",
            "itemtype": str,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv4DstCounter"},
        "decrement": {"type": "PatternFlowIpv4DstCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": "0.0.0.0",
        "values": ["0.0.0.0"],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value="0.0.0.0",
        values=["0.0.0.0"],
        metric_group=None,
    ):
        super(PatternFlowIpv4Dst, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv4DstCounter
        """Factory property that returns an instance of the PatternFlowIpv4DstCounter class

        ipv4 counter pattern

        Returns: PatternFlowIpv4DstCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv4DstCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv4DstCounter
        """Factory property that returns an instance of the PatternFlowIpv4DstCounter class

        ipv4 counter pattern

        Returns: PatternFlowIpv4DstCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv4DstCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv4DstCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": str,
            "format": "ipv4",
        },
        "step": {
            "type": str,
            "format": "ipv4",
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": "0.0.0.0",
        "step": "0.0.0.1",
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start="0.0.0.0", step="0.0.0.1", count=1):
        super(PatternFlowIpv4DstCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class FlowIpv6(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "version": {"type": "PatternFlowIpv6Version"},
        "traffic_class": {"type": "PatternFlowIpv6TrafficClass"},
        "flow_label": {"type": "PatternFlowIpv6FlowLabel"},
        "payload_length": {"type": "PatternFlowIpv6PayloadLength"},
        "next_header": {"type": "PatternFlowIpv6NextHeader"},
        "hop_limit": {"type": "PatternFlowIpv6HopLimit"},
        "src": {"type": "PatternFlowIpv6Src"},
        "dst": {"type": "PatternFlowIpv6Dst"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(FlowIpv6, self).__init__()
        self._parent = parent

    @property
    def version(self):
        # type: () -> PatternFlowIpv6Version
        """version getter

        Version numberVersion numberVersion number

        Returns: PatternFlowIpv6Version
        """
        return self._get_property("version", PatternFlowIpv6Version)

    @property
    def traffic_class(self):
        # type: () -> PatternFlowIpv6TrafficClass
        """traffic_class getter

        Traffic classTraffic classTraffic class

        Returns: PatternFlowIpv6TrafficClass
        """
        return self._get_property("traffic_class", PatternFlowIpv6TrafficClass)

    @property
    def flow_label(self):
        # type: () -> PatternFlowIpv6FlowLabel
        """flow_label getter

        Flow labelFlow labelFlow label

        Returns: PatternFlowIpv6FlowLabel
        """
        return self._get_property("flow_label", PatternFlowIpv6FlowLabel)

    @property
    def payload_length(self):
        # type: () -> PatternFlowIpv6PayloadLength
        """payload_length getter

        Payload lengthPayload lengthPayload length

        Returns: PatternFlowIpv6PayloadLength
        """
        return self._get_property("payload_length", PatternFlowIpv6PayloadLength)

    @property
    def next_header(self):
        # type: () -> PatternFlowIpv6NextHeader
        """next_header getter

        Next headerNext headerNext header

        Returns: PatternFlowIpv6NextHeader
        """
        return self._get_property("next_header", PatternFlowIpv6NextHeader)

    @property
    def hop_limit(self):
        # type: () -> PatternFlowIpv6HopLimit
        """hop_limit getter

        Hop limitHop limitHop limit

        Returns: PatternFlowIpv6HopLimit
        """
        return self._get_property("hop_limit", PatternFlowIpv6HopLimit)

    @property
    def src(self):
        # type: () -> PatternFlowIpv6Src
        """src getter

        Source addressSource addressSource address

        Returns: PatternFlowIpv6Src
        """
        return self._get_property("src", PatternFlowIpv6Src)

    @property
    def dst(self):
        # type: () -> PatternFlowIpv6Dst
        """dst getter

        Destination addressDestination addressDestination address

        Returns: PatternFlowIpv6Dst
        """
        return self._get_property("dst", PatternFlowIpv6Dst)


class PatternFlowIpv6Version(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 15,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 15,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv6VersionCounter"},
        "decrement": {"type": "PatternFlowIpv6VersionCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 6,
        "values": [6],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=6, values=[6], metric_group=None
    ):
        super(PatternFlowIpv6Version, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6VersionCounter
        """Factory property that returns an instance of the PatternFlowIpv6VersionCounter class

        integer counter pattern

        Returns: PatternFlowIpv6VersionCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv6VersionCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6VersionCounter
        """Factory property that returns an instance of the PatternFlowIpv6VersionCounter class

        integer counter pattern

        Returns: PatternFlowIpv6VersionCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv6VersionCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv6VersionCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 15,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 15,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 6,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=6, step=1, count=1):
        super(PatternFlowIpv6VersionCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv6TrafficClass(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 255,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv6TrafficClassCounter"},
        "decrement": {"type": "PatternFlowIpv6TrafficClassCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv6TrafficClass, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6TrafficClassCounter
        """Factory property that returns an instance of the PatternFlowIpv6TrafficClassCounter class

        integer counter pattern

        Returns: PatternFlowIpv6TrafficClassCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv6TrafficClassCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6TrafficClassCounter
        """Factory property that returns an instance of the PatternFlowIpv6TrafficClassCounter class

        integer counter pattern

        Returns: PatternFlowIpv6TrafficClassCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv6TrafficClassCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv6TrafficClassCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv6TrafficClassCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv6FlowLabel(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 1048575,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 1048575,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv6FlowLabelCounter"},
        "decrement": {"type": "PatternFlowIpv6FlowLabelCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowIpv6FlowLabel, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6FlowLabelCounter
        """Factory property that returns an instance of the PatternFlowIpv6FlowLabelCounter class

        integer counter pattern

        Returns: PatternFlowIpv6FlowLabelCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv6FlowLabelCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6FlowLabelCounter
        """Factory property that returns an instance of the PatternFlowIpv6FlowLabelCounter class

        integer counter pattern

        Returns: PatternFlowIpv6FlowLabelCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv6FlowLabelCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv6FlowLabelCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 1048575,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 1048575,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv6FlowLabelCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv6PayloadLength(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "auto",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "auto": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv6PayloadLengthCounter"},
        "decrement": {"type": "PatternFlowIpv6PayloadLengthCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "auto",
        "value": 0,
        "values": [0],
        "auto": 0,
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    AUTO = "auto"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], auto=0, metric_group=None
    ):
        super(PatternFlowIpv6PayloadLength, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("auto", auto)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6PayloadLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv6PayloadLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv6PayloadLengthCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv6PayloadLengthCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6PayloadLengthCounter
        """Factory property that returns an instance of the PatternFlowIpv6PayloadLengthCounter class

        integer counter pattern

        Returns: PatternFlowIpv6PayloadLengthCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv6PayloadLengthCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def auto(self):
        # type: () -> int
        """auto getter

        The OTG implementation can provide a system generated. value for this property. If the OTG is unable to generate a value. the default value must be used.

        Returns: int
        """
        return self._get_property("auto")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv6PayloadLengthCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowIpv6PayloadLengthCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv6NextHeader(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "auto",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 255,
        },
        "auto": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv6NextHeaderCounter"},
        "decrement": {"type": "PatternFlowIpv6NextHeaderCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "auto",
        "value": 59,
        "values": [59],
        "auto": 59,
    }  # type: Dict[str, Union(type)]

    HOPOPT = 0  #
    ICMP = 1  #
    IGMP = 2  #
    GGP = 3  #
    IP_IN_IP = 4  #
    ST = 5  #
    TCP = 6  #
    CPT = 7  #
    EGP = 8  #
    IGP = 9  #
    NO_NEXT_HEADER = 59  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    AUTO = "auto"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value=59,
        values=[59],
        auto=59,
        metric_group=None,
    ):
        super(PatternFlowIpv6NextHeader, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("auto", auto)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6NextHeaderCounter
        """Factory property that returns an instance of the PatternFlowIpv6NextHeaderCounter class

        integer counter pattern

        Returns: PatternFlowIpv6NextHeaderCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv6NextHeaderCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6NextHeaderCounter
        """Factory property that returns an instance of the PatternFlowIpv6NextHeaderCounter class

        integer counter pattern

        Returns: PatternFlowIpv6NextHeaderCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv6NextHeaderCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["auto"], Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def auto(self):
        # type: () -> int
        """auto getter

        The OTG implementation can provide a system generated. value for this property. If the OTG is unable to generate a value. the default value must be used.

        Returns: int
        """
        return self._get_property("auto")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv6NextHeaderCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 59,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    HOPOPT = 0  #
    ICMP = 1  #
    IGMP = 2  #
    GGP = 3  #
    IP_IN_IP = 4  #
    ST = 5  #
    TCP = 6  #
    CPT = 7  #
    EGP = 8  #
    IGP = 9  #
    NO_NEXT_HEADER = 59  #

    def __init__(self, parent=None, start=59, step=1, count=1):
        super(PatternFlowIpv6NextHeaderCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv6HopLimit(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 255,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv6HopLimitCounter"},
        "decrement": {"type": "PatternFlowIpv6HopLimitCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 64,
        "values": [64],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=64, values=[64], metric_group=None
    ):
        super(PatternFlowIpv6HopLimit, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6HopLimitCounter
        """Factory property that returns an instance of the PatternFlowIpv6HopLimitCounter class

        integer counter pattern

        Returns: PatternFlowIpv6HopLimitCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv6HopLimitCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6HopLimitCounter
        """Factory property that returns an instance of the PatternFlowIpv6HopLimitCounter class

        integer counter pattern

        Returns: PatternFlowIpv6HopLimitCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv6HopLimitCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv6HopLimitCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 255,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 64,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=64, step=1, count=1):
        super(PatternFlowIpv6HopLimitCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv6Src(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": str,
            "format": "ipv6",
        },
        "values": {
            "type": list,
            "format": "ipv6",
            "itemtype": str,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv6SrcCounter"},
        "decrement": {"type": "PatternFlowIpv6SrcCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": "::0",
        "values": ["::0"],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value="::0", values=["::0"], metric_group=None
    ):
        super(PatternFlowIpv6Src, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6SrcCounter
        """Factory property that returns an instance of the PatternFlowIpv6SrcCounter class

        ipv6 counter pattern

        Returns: PatternFlowIpv6SrcCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv6SrcCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6SrcCounter
        """Factory property that returns an instance of the PatternFlowIpv6SrcCounter class

        ipv6 counter pattern

        Returns: PatternFlowIpv6SrcCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv6SrcCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv6SrcCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": str,
            "format": "ipv6",
        },
        "step": {
            "type": str,
            "format": "ipv6",
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": "::0",
        "step": "::1",
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start="::0", step="::1", count=1):
        super(PatternFlowIpv6SrcCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowIpv6Dst(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": str,
            "format": "ipv6",
        },
        "values": {
            "type": list,
            "format": "ipv6",
            "itemtype": str,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowIpv6DstCounter"},
        "decrement": {"type": "PatternFlowIpv6DstCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": "::0",
        "values": ["::0"],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value="::0", values=["::0"], metric_group=None
    ):
        super(PatternFlowIpv6Dst, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowIpv6DstCounter
        """Factory property that returns an instance of the PatternFlowIpv6DstCounter class

        ipv6 counter pattern

        Returns: PatternFlowIpv6DstCounter
        """
        return self._get_property(
            "increment", PatternFlowIpv6DstCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowIpv6DstCounter
        """Factory property that returns an instance of the PatternFlowIpv6DstCounter class

        ipv6 counter pattern

        Returns: PatternFlowIpv6DstCounter
        """
        return self._get_property(
            "decrement", PatternFlowIpv6DstCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowIpv6DstCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": str,
            "format": "ipv6",
        },
        "step": {
            "type": str,
            "format": "ipv6",
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": "::0",
        "step": "::1",
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start="::0", step="::1", count=1):
        super(PatternFlowIpv6DstCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class FlowPfcPause(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "dst": {"type": "PatternFlowPfcPauseDst"},
        "src": {"type": "PatternFlowPfcPauseSrc"},
        "ether_type": {"type": "PatternFlowPfcPauseEtherType"},
        "control_op_code": {"type": "PatternFlowPfcPauseControlOpCode"},
        "class_enable_vector": {"type": "PatternFlowPfcPauseClassEnableVector"},
        "pause_class_0": {"type": "PatternFlowPfcPausePauseClass0"},
        "pause_class_1": {"type": "PatternFlowPfcPausePauseClass1"},
        "pause_class_2": {"type": "PatternFlowPfcPausePauseClass2"},
        "pause_class_3": {"type": "PatternFlowPfcPausePauseClass3"},
        "pause_class_4": {"type": "PatternFlowPfcPausePauseClass4"},
        "pause_class_5": {"type": "PatternFlowPfcPausePauseClass5"},
        "pause_class_6": {"type": "PatternFlowPfcPausePauseClass6"},
        "pause_class_7": {"type": "PatternFlowPfcPausePauseClass7"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(FlowPfcPause, self).__init__()
        self._parent = parent

    @property
    def dst(self):
        # type: () -> PatternFlowPfcPauseDst
        """dst getter

        Destination MAC addressDestination MAC addressDestination MAC address

        Returns: PatternFlowPfcPauseDst
        """
        return self._get_property("dst", PatternFlowPfcPauseDst)

    @property
    def src(self):
        # type: () -> PatternFlowPfcPauseSrc
        """src getter

        Source MAC addressSource MAC addressSource MAC address

        Returns: PatternFlowPfcPauseSrc
        """
        return self._get_property("src", PatternFlowPfcPauseSrc)

    @property
    def ether_type(self):
        # type: () -> PatternFlowPfcPauseEtherType
        """ether_type getter

        Ethernet typeEthernet typeEthernet type

        Returns: PatternFlowPfcPauseEtherType
        """
        return self._get_property("ether_type", PatternFlowPfcPauseEtherType)

    @property
    def control_op_code(self):
        # type: () -> PatternFlowPfcPauseControlOpCode
        """control_op_code getter

        Control operation codeControl operation codeControl operation code

        Returns: PatternFlowPfcPauseControlOpCode
        """
        return self._get_property("control_op_code", PatternFlowPfcPauseControlOpCode)

    @property
    def class_enable_vector(self):
        # type: () -> PatternFlowPfcPauseClassEnableVector
        """class_enable_vector getter

        DestinationDestinationDestination

        Returns: PatternFlowPfcPauseClassEnableVector
        """
        return self._get_property(
            "class_enable_vector", PatternFlowPfcPauseClassEnableVector
        )

    @property
    def pause_class_0(self):
        # type: () -> PatternFlowPfcPausePauseClass0
        """pause_class_0 getter

        Pause class 0Pause class 0Pause class 0

        Returns: PatternFlowPfcPausePauseClass0
        """
        return self._get_property("pause_class_0", PatternFlowPfcPausePauseClass0)

    @property
    def pause_class_1(self):
        # type: () -> PatternFlowPfcPausePauseClass1
        """pause_class_1 getter

        Pause class 1Pause class 1Pause class 1

        Returns: PatternFlowPfcPausePauseClass1
        """
        return self._get_property("pause_class_1", PatternFlowPfcPausePauseClass1)

    @property
    def pause_class_2(self):
        # type: () -> PatternFlowPfcPausePauseClass2
        """pause_class_2 getter

        Pause class 2Pause class 2Pause class 2

        Returns: PatternFlowPfcPausePauseClass2
        """
        return self._get_property("pause_class_2", PatternFlowPfcPausePauseClass2)

    @property
    def pause_class_3(self):
        # type: () -> PatternFlowPfcPausePauseClass3
        """pause_class_3 getter

        Pause class 3Pause class 3Pause class 3

        Returns: PatternFlowPfcPausePauseClass3
        """
        return self._get_property("pause_class_3", PatternFlowPfcPausePauseClass3)

    @property
    def pause_class_4(self):
        # type: () -> PatternFlowPfcPausePauseClass4
        """pause_class_4 getter

        Pause class 4Pause class 4Pause class 4

        Returns: PatternFlowPfcPausePauseClass4
        """
        return self._get_property("pause_class_4", PatternFlowPfcPausePauseClass4)

    @property
    def pause_class_5(self):
        # type: () -> PatternFlowPfcPausePauseClass5
        """pause_class_5 getter

        Pause class 5Pause class 5Pause class 5

        Returns: PatternFlowPfcPausePauseClass5
        """
        return self._get_property("pause_class_5", PatternFlowPfcPausePauseClass5)

    @property
    def pause_class_6(self):
        # type: () -> PatternFlowPfcPausePauseClass6
        """pause_class_6 getter

        Pause class 6Pause class 6Pause class 6

        Returns: PatternFlowPfcPausePauseClass6
        """
        return self._get_property("pause_class_6", PatternFlowPfcPausePauseClass6)

    @property
    def pause_class_7(self):
        # type: () -> PatternFlowPfcPausePauseClass7
        """pause_class_7 getter

        Pause class 7Pause class 7Pause class 7

        Returns: PatternFlowPfcPausePauseClass7
        """
        return self._get_property("pause_class_7", PatternFlowPfcPausePauseClass7)


class PatternFlowPfcPauseDst(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": str,
            "format": "mac",
        },
        "values": {
            "type": list,
            "format": "mac",
            "itemtype": str,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPauseDstCounter"},
        "decrement": {"type": "PatternFlowPfcPauseDstCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": "01:80:c2:00:00:01",
        "values": ["01:80:c2:00:00:01"],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value="01:80:c2:00:00:01",
        values=["01:80:c2:00:00:01"],
        metric_group=None,
    ):
        super(PatternFlowPfcPauseDst, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPauseDstCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseDstCounter class

        mac counter pattern

        Returns: PatternFlowPfcPauseDstCounter
        """
        return self._get_property(
            "increment", PatternFlowPfcPauseDstCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPauseDstCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseDstCounter class

        mac counter pattern

        Returns: PatternFlowPfcPauseDstCounter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPauseDstCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPauseDstCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": str,
            "format": "mac",
        },
        "step": {
            "type": str,
            "format": "mac",
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": "01:80:c2:00:00:01",
        "step": "00:00:00:00:00:01",
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, start="01:80:c2:00:00:01", step="00:00:00:00:00:01", count=1
    ):
        super(PatternFlowPfcPauseDstCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPauseSrc(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": str,
            "format": "mac",
        },
        "values": {
            "type": list,
            "format": "mac",
            "itemtype": str,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPauseSrcCounter"},
        "decrement": {"type": "PatternFlowPfcPauseSrcCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": "00:00:00:00:00:00",
        "values": ["00:00:00:00:00:00"],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value="00:00:00:00:00:00",
        values=["00:00:00:00:00:00"],
        metric_group=None,
    ):
        super(PatternFlowPfcPauseSrc, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPauseSrcCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseSrcCounter class

        mac counter pattern

        Returns: PatternFlowPfcPauseSrcCounter
        """
        return self._get_property(
            "increment", PatternFlowPfcPauseSrcCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPauseSrcCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseSrcCounter class

        mac counter pattern

        Returns: PatternFlowPfcPauseSrcCounter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPauseSrcCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[str]
        """values getter

        TBD

        Returns: List[str]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[str]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPauseSrcCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": str,
            "format": "mac",
        },
        "step": {
            "type": str,
            "format": "mac",
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": "00:00:00:00:00:00",
        "step": "00:00:00:00:00:01",
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, start="00:00:00:00:00:00", step="00:00:00:00:00:01", count=1
    ):
        super(PatternFlowPfcPauseSrcCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> str
        """start getter

        TBD

        Returns: str
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: str
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> str
        """step getter

        TBD

        Returns: str
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: str
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPauseEtherType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPauseEtherTypeCounter"},
        "decrement": {"type": "PatternFlowPfcPauseEtherTypeCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 34824,
        "values": [34824],
    }  # type: Dict[str, Union(type)]

    X8808 = 34824  #

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=34824, values=[34824], metric_group=None
    ):
        super(PatternFlowPfcPauseEtherType, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPauseEtherTypeCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseEtherTypeCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseEtherTypeCounter
        """
        return self._get_property(
            "increment", PatternFlowPfcPauseEtherTypeCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPauseEtherTypeCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseEtherTypeCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseEtherTypeCounter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPauseEtherTypeCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPauseEtherTypeCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 34824,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    X8808 = 34824  #

    def __init__(self, parent=None, start=34824, step=1, count=1):
        super(PatternFlowPfcPauseEtherTypeCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPauseControlOpCode(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPauseControlOpCodeCounter"},
        "decrement": {"type": "PatternFlowPfcPauseControlOpCodeCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 257,
        "values": [257],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=257, values=[257], metric_group=None
    ):
        super(PatternFlowPfcPauseControlOpCode, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPauseControlOpCodeCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseControlOpCodeCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseControlOpCodeCounter
        """
        return self._get_property(
            "increment", PatternFlowPfcPauseControlOpCodeCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPauseControlOpCodeCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseControlOpCodeCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseControlOpCodeCounter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPauseControlOpCodeCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPauseControlOpCodeCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 257,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=257, step=1, count=1):
        super(PatternFlowPfcPauseControlOpCodeCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPauseClassEnableVector(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPauseClassEnableVectorCounter"},
        "decrement": {"type": "PatternFlowPfcPauseClassEnableVectorCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowPfcPauseClassEnableVector, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPauseClassEnableVectorCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseClassEnableVectorCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseClassEnableVectorCounter
        """
        return self._get_property(
            "increment", PatternFlowPfcPauseClassEnableVectorCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPauseClassEnableVectorCounter
        """Factory property that returns an instance of the PatternFlowPfcPauseClassEnableVectorCounter class

        integer counter pattern

        Returns: PatternFlowPfcPauseClassEnableVectorCounter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPauseClassEnableVectorCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPauseClassEnableVectorCounter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowPfcPauseClassEnableVectorCounter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPausePauseClass0(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPausePauseClass0Counter"},
        "decrement": {"type": "PatternFlowPfcPausePauseClass0Counter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowPfcPausePauseClass0, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass0Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass0Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass0Counter
        """
        return self._get_property(
            "increment", PatternFlowPfcPausePauseClass0Counter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass0Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass0Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass0Counter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPausePauseClass0Counter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPausePauseClass0Counter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowPfcPausePauseClass0Counter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPausePauseClass1(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPausePauseClass1Counter"},
        "decrement": {"type": "PatternFlowPfcPausePauseClass1Counter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowPfcPausePauseClass1, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass1Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass1Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass1Counter
        """
        return self._get_property(
            "increment", PatternFlowPfcPausePauseClass1Counter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass1Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass1Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass1Counter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPausePauseClass1Counter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPausePauseClass1Counter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowPfcPausePauseClass1Counter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPausePauseClass2(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPausePauseClass2Counter"},
        "decrement": {"type": "PatternFlowPfcPausePauseClass2Counter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowPfcPausePauseClass2, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass2Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass2Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass2Counter
        """
        return self._get_property(
            "increment", PatternFlowPfcPausePauseClass2Counter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass2Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass2Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass2Counter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPausePauseClass2Counter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPausePauseClass2Counter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowPfcPausePauseClass2Counter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPausePauseClass3(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPausePauseClass3Counter"},
        "decrement": {"type": "PatternFlowPfcPausePauseClass3Counter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowPfcPausePauseClass3, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass3Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass3Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass3Counter
        """
        return self._get_property(
            "increment", PatternFlowPfcPausePauseClass3Counter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass3Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass3Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass3Counter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPausePauseClass3Counter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPausePauseClass3Counter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowPfcPausePauseClass3Counter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPausePauseClass4(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPausePauseClass4Counter"},
        "decrement": {"type": "PatternFlowPfcPausePauseClass4Counter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowPfcPausePauseClass4, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass4Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass4Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass4Counter
        """
        return self._get_property(
            "increment", PatternFlowPfcPausePauseClass4Counter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass4Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass4Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass4Counter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPausePauseClass4Counter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPausePauseClass4Counter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowPfcPausePauseClass4Counter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPausePauseClass5(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPausePauseClass5Counter"},
        "decrement": {"type": "PatternFlowPfcPausePauseClass5Counter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowPfcPausePauseClass5, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass5Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass5Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass5Counter
        """
        return self._get_property(
            "increment", PatternFlowPfcPausePauseClass5Counter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass5Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass5Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass5Counter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPausePauseClass5Counter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPausePauseClass5Counter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowPfcPausePauseClass5Counter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPausePauseClass6(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPausePauseClass6Counter"},
        "decrement": {"type": "PatternFlowPfcPausePauseClass6Counter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowPfcPausePauseClass6, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass6Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass6Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass6Counter
        """
        return self._get_property(
            "increment", PatternFlowPfcPausePauseClass6Counter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass6Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass6Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass6Counter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPausePauseClass6Counter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPausePauseClass6Counter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowPfcPausePauseClass6Counter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class PatternFlowPfcPausePauseClass7(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "values": {
            "type": list,
            "itemtype": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowPfcPausePauseClass7Counter"},
        "decrement": {"type": "PatternFlowPfcPausePauseClass7Counter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": 0,
        "values": [0],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self, parent=None, choice=None, value=0, values=[0], metric_group=None
    ):
        super(PatternFlowPfcPausePauseClass7, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowPfcPausePauseClass7Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass7Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass7Counter
        """
        return self._get_property(
            "increment", PatternFlowPfcPausePauseClass7Counter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowPfcPausePauseClass7Counter
        """Factory property that returns an instance of the PatternFlowPfcPausePauseClass7Counter class

        integer counter pattern

        Returns: PatternFlowPfcPausePauseClass7Counter
        """
        return self._get_property(
            "decrement", PatternFlowPfcPausePauseClass7Counter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        self._set_property("choice", value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        TBD

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: int
        """
        self._set_property("value", value, "value")

    @property
    def values(self):
        # type: () -> List[int]
        """values getter

        TBD

        Returns: List[int]
        """
        return self._get_property("values")

    @values.setter
    def values(self, value):
        """values setter

        TBD

        value: List[int]
        """
        self._set_property("values", value, "values")

    @property
    def metric_group(self):
        # type: () -> str
        """metric_group getter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        Returns: str
        """
        return self._get_property("metric_group")

    @metric_group.setter
    def metric_group(self, value):
        """metric_group setter

        A unique name is used to indicate to the system that the field may extend the metric row key and create an aggregate metric row for every unique value. To have metric group columns appear in the flow metric rows the flow metric request allows for the metric_group value to be specified as part of the request.

        value: str
        """
        self._set_property("metric_group", value)


class PatternFlowPfcPausePauseClass7Counter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "start": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "step": {
            "type": int,
            "minimum": 0,
            "maximum": 65535,
        },
        "count": {"type": int},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "start": 0,
        "step": 1,
        "count": 1,
    }  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, start=0, step=1, count=1):
        super(PatternFlowPfcPausePauseClass7Counter, self).__init__()
        self._parent = parent
        self._set_property("start", start)
        self._set_property("step", step)
        self._set_property("count", count)

    @property
    def start(self):
        # type: () -> int
        """start getter

        TBD

        Returns: int
        """
        return self._get_property("start")

    @start.setter
    def start(self, value):
        """start setter

        TBD

        value: int
        """
        self._set_property("start", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        TBD

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        TBD

        value: int
        """
        self._set_property("step", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        TBD

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        TBD

        value: int
        """
        self._set_property("count", value)


class FlowEthernetPause(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "dst": {"type": "PatternFlowEthernetPauseDst"},
        "src": {"type": "PatternFlowEthernetPauseSrc"},
        "ether_type": {"type": "PatternFlowEthernetPauseEtherType"},
        "control_op_code": {"type": "PatternFlowEthernetPauseControlOpCode"},
        "time": {"type": "PatternFlowEthernetPauseTime"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(FlowEthernetPause, self).__init__()
        self._parent = parent

    @property
    def dst(self):
        # type: () -> PatternFlowEthernetPauseDst
        """dst getter

        Destination MAC addressDestination MAC addressDestination MAC address

        Returns: PatternFlowEthernetPauseDst
        """
        return self._get_property("dst", PatternFlowEthernetPauseDst)

    @property
    def src(self):
        # type: () -> PatternFlowEthernetPauseSrc
        """src getter

        Source MAC addressSource MAC addressSource MAC address

        Returns: PatternFlowEthernetPauseSrc
        """
        return self._get_property("src", PatternFlowEthernetPauseSrc)

    @property
    def ether_type(self):
        # type: () -> PatternFlowEthernetPauseEtherType
        """ether_type getter

        Ethernet typeEthernet typeEthernet type

        Returns: PatternFlowEthernetPauseEtherType
        """
        return self._get_property("ether_type", PatternFlowEthernetPauseEtherType)

    @property
    def control_op_code(self):
        # type: () -> PatternFlowEthernetPauseControlOpCode
        """control_op_code getter

        Control operation codeControl operation codeControl operation code

        Returns: PatternFlowEthernetPauseControlOpCode
        """
        return self._get_property(
            "control_op_code", PatternFlowEthernetPauseControlOpCode
        )

    @property
    def time(self):
        # type: () -> PatternFlowEthernetPauseTime
        """time getter

        TimeTimeTime

        Returns: PatternFlowEthernetPauseTime
        """
        return self._get_property("time", PatternFlowEthernetPauseTime)


class PatternFlowEthernetPauseDst(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "value",
                "values",
                "increment",
                "decrement",
            ],
        },
        "value": {
            "type": str,
            "format": "mac",
        },
        "values": {
            "type": list,
            "format": "mac",
            "itemtype": str,
        },
        "metric_group": {"type": str},
        "increment": {"type": "PatternFlowEthernetPauseDstCounter"},
        "decrement": {"type": "PatternFlowEthernetPauseDstCounter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "value",
        "value": "01:80:c2:00:00:01",
        "values": ["01:80:c2:00:00:01"],
    }  # type: Dict[str, Union(type)]

    VALUE = "value"  # type: str
    VALUES = "values"  # type: str
    INCREMENT = "increment"  # type: str
    DECREMENT = "decrement"  # type: str

    def __init__(
        self,
        parent=None,
        choice=None,
        value="01:80:c2:00:00:01",
        values=["01:80:c2:00:00:01"],
        metric_group=None,
    ):
        super(PatternFlowEthernetPauseDst, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("values", values)
        self._set_property("metric_group", metric_group)
        if "choice" in self._DEFAULTS and choice is None:
            getattr(self, self._DEFAULTS["choice"])
        else:
            self.choice = choice

    @property
    def increment(self):
        # type: () -> PatternFlowEthernetPauseDstCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseDstCounter class

        mac counter pattern

        Returns: PatternFlowEthernetPauseDstCounter
        """
        return self._get_property(
            "increment", PatternFlowEthernetPauseDstCounter, self, "increment"
        )

    @property
    def decrement(self):
        # type: () -> PatternFlowEthernetPauseDstCounter
        """Factory property that returns an instance of the PatternFlowEthernetPauseDstCounter class

        mac counter pattern

        Returns: PatternFlowEthernetPauseDstCounter
        """
        return self._get_property(
            "decrement", PatternFlowEthernetPauseDstCounter, self, "decrement"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """choice getter

        TBD

        Returns: Union[Literal["decrement"], Literal["increment"], Literal["value"], Literal["values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
   