from . import *
from testinlabel.request import Request
import os
import json
from PIL import Image, ImageFont, ImageDraw, ExifTags


class TLA():
    def __init__(self, accessKey, host="", debug=False, savePath=""):
        if host == "":
            self.host = "http://label-export.testin.cn/"
        else:
            if "http://" in host or "https://" in host:
                self.host = host.rstrip("/") + "/"
            else:
                self.host = "http://" + host.strip("/") + "/"
        self.__debug = debug
        self.AK = accessKey
        # self.SK = secretKey
        self.savePath = savePath
        # self.req = Request(host=self.host, AK=self.AK, SK=self.SK, debug=self.__debug)
        self.req = None
        self.data = None
        self.taskList = None
        self.taskKey = None
        self.jsonDir = None
        self.jsonPath = None
        self.readmePath = None

    def SetKey(self, taskKey):
        '''
        Set the dataset to be processed
        :param datasetId: the id of the dataset
        :return:
        '''
        self.taskKey = taskKey
        self.req = Request(host=self.host, AK=self.AK, task_key=self.taskKey, debug=self.__debug)

        if self.__debug:
            print(f"[SET_TASK_KEY] set task key:{taskKey}")

        self.jsonDir = os.path.join(self.savePath, "testinlabel-data", self.taskKey)
        if not os.path.exists(self.jsonDir):
            os.makedirs(self.jsonDir)

        self.jsonPath = os.path.join(self.jsonDir, taskKey + ".json")
        self.readmePath = os.path.join(self.jsonDir, "readme.txt")

        if os.path.exists(self.jsonPath):
            with open(self.jsonPath, "r", encoding="utf-8") as jf:
                self.data = json.load(jf)
                self.taskList = self.data["data"]["task_list"]

    def GetLabelData(self, hasUnable=False, status=[STATUS_PASS], overlay=False):
        """
        get labeled data by task_key, more about task_key, see: http://label.testin.cn/v/pm-task-list
        :param taskKey:
        :param hasUnable:
        :param status:
        :param save:
        :return:
        """
        if self.data is not None and not overlay:
            if self.__debug:
                print(
                    f"[GET_LABEL_DATA] label data already download to locah path: {self.jsonPath}, abort this request！if you need re-download anyway, you should set overlay to Ture"
                )
            self.taskList = self.data["data"]["task_list"]
            return self.data
        if self.req is None:
            print('请设置任务key')
            return None
        self.data = self.req.getTaskData(key=self.taskKey, hasUnable=hasUnable, status=status)

        with open(self.jsonPath, "w", encoding="utf-8") as f:
            json.dump(self.data, f)
            if self.__debug:
                print(f"[SAVE_LABEL_DATA] save data to:{self.jsonPath}")

        with open(self.readmePath, "w", encoding="utf-8") as rdm:
            rdmContent = """注意：
    当前目录为 testinlabel SDK 自动生成的目录，同级目录的json文件为数据缓存文件，在导出数据之后您可以按需删除。
    更多信息：http://ai.testin.cn/
    登入系统：https://label.testin.cn/
    
Notice:
     The current directory is automatically generated by the testinlabel SDK, the json file in the same level is cache data, you can delete it as your need after exporting the data. :)
     more about: http://ai.testin.cn/
     login: https://label.testin.cn/
            """
            rdm.write(rdmContent)

        self.taskList = self.data["data"]["task_list"]
        return self.data

    def draw_img(self, overlay, hasUnable, status):
        if self.data is not None and not overlay:
            if self.__debug:
                print(
                    f"[GET_LABEL_DATA] label data already download to locah path: {self.jsonPath}, abort this request！if you need re-download anyway, you should set overlay to Ture"
                )
            self.taskList = self.data["data"]["task_list"]
            # return self.data
        if self.req is None:
            print('请设置任务key')
            return None
        self.data = self.req.getTaskData(key=self.taskKey, hasUnable=hasUnable, status=status)

class DrawPic:
    """画图"""
    def __init__(self, im):
        self._im = im
        self._draw = ImageDraw.Draw(self._im)

    def save_img(self, out_path, file_name):
        """保存图片"""
        if not os.path.exists(out_path):
            os.makedirs(out_path)
        self._im.save(os.path.join(out_path, file_name))

    def get_im(self):
        """获取内存中的图片"""
        return self._im

    def rotate(self):
        """旋转图片"""
        if self._im.mode == 'P':
            self._im = self._im.convert('RGB')
        try:
            for orientation in ExifTags.TAGS.keys():
                if ExifTags.TAGS[orientation] == 'Orientation': break
            exif = dict(self._im._getexif().items())
            if exif[orientation] == 3:
                self._im = self._im.rotate(180, expand=True)
            elif exif[orientation] == 6:
                self._im = self._im.rotate(270, expand=True)
            elif exif[orientation] == 8:
                self._im = self._im.rotate(90, expand=True)
        except:
            pass

    def img_cuboid(self, mark, color=None, width=None):
        """日形框标注框"""
        point1 = mark['finshPoints'][0]
        point2 = mark['finshPoints'][1]
        x1 = [point1['x'], point2['x']]
        y1 = [point1['y'], point2['y']]
        point3 = mark['finshPoints'][2]
        point4 = mark['finshPoints'][3]
        x2 = [point3['x'], point4['x']]
        # y2 = [point3['y'], point4['y']]
        p1 = [min(x1), min(y1)]
        p2 = [min(x1), max(y1)]
        p3 = [max(x1), min(y1)]
        p4 = [max(x1), max(y1)]
        p5 = [point3['x'], point3['y']]
        p6 = [point4['x'], point4['y']]
        # p5 = [min(x2), min(y2)]
        # p6 = [min(x2), max(y2)]
        self._draw.rectangle(p1 + p4, outline=color, width=width)
        if min(x1) < min(x2):
            if p5[1] > p6[1]:
                self._draw.line(p4 + p5, fill=color, width=width)
                self._draw.line(p5 + p6, fill=color, width=width)
                self._draw.line(p3 + p6, fill=color, width=width)
            else:
                self._draw.line(p3 + p5, fill=color, width=width)
                self._draw.line(p5 + p6, fill=color, width=width)
                self._draw.line(p4 + p6, fill=color, width=width)

        else:
            if p5[1] > p6[1]:
                self._draw.line(p2 + p5, fill=color, width=width)
                self._draw.line(p6 + p5, fill=color, width=width)
                self._draw.line(p6 + p1, fill=color, width=width)
            else:
                self._draw.line(p1 + p5, fill=color, width=width)
                self._draw.line(p6 + p5, fill=color, width=width)
                self._draw.line(p6 + p2, fill=color, width=width)

    def img_rect(self, mark, fill=None, outline=None, width=1):
        """矩形标注框"""
        self._draw.rectangle([mark['point']['left'], mark['point']['top'], mark['point']['right'], mark['point']['bottom']],
                             outline=outline,
                             width=width,
                             fill=fill)

    def img_curve(self, mark, color=None, width=None):
        """曲线标注框"""
        for index, po in enumerate(mark['point']):
            p1 = [po['x'], po['y']]
            if index + 1 != len(mark['point']):
                p2 = [mark['point'][index + 1]['x'], mark['point'][index + 1]['y']]
                self._draw.line(p1 + p2, fill=color, width=width)

    def img_polygon(self, mark, fill=None, outline=None):
        """多边形标注框"""
        p = []
        for po in mark['point']:
            if po['x'] is None or po['y'] is None:
                continue
            p.append(po['x'])
            p.append(po['y'])
        self._draw.polygon(p, fill=fill, outline=outline)

    def img_point(self, mark, fill=None, outline=None, width=1, radius=None):
        """点标注"""
        self._draw.ellipse(
            (float(mark['point']['x']) - radius, float(mark['point']['y']) - radius, float(mark['point']['x']) + radius,
             float(mark['point']['y']) + radius),
            fill=fill,
            outline=outline,
            width=width)

    def img_ellipse(self, mark, fill=None, outline=None, width=1):
        """圆形标注"""
        self._draw.ellipse((mark['point']['left'], mark['point']['top'], mark['point']['right'], mark['point']['bottom']),
                           fill=fill,
                           outline=outline)

    def d3rect(self, mark, fill=None, outline=None, width=1):
        '''伪3D'''
        x_list = [point['x'] for point in mark['back']]
        print(min(x_list))
        y_list = [point['y'] for point in mark['back']]
        self._draw.rectangle([min(x_list), min(y_list), max(x_list), max(y_list)], outline=outline, width=width)
        x_list = [point['x'] for point in mark['front']]
        y_list = [point['y'] for point in mark['front']]
        self._draw.rectangle([min(x_list), min(y_list), max(x_list), max(y_list)], outline=outline, width=width)
        for index, point in enumerate(mark['back']):
            self._draw.line([point['x'], point['y'], mark['front'][index]['x'], mark['front'][index]['y']],
                            fill=fill,
                            width=width)

    def do_mosaic(self, points, neighbor=5):
        """
        马赛克的实现原理是把图像上某个像素点一定范围邻域内的所有点用邻域内左上像素点的颜色代替，这样可以模糊细节，但是可以保留大体的轮廓。
        :param frame: opencv frame
        :param int x :  马赛克左顶点x值
        :param int y:  马赛克左顶点y值
        :param int w:  马赛克宽
        :param int h:  马赛克高
        :param int neighbor:  马赛克每一块的宽
        """
        for point in points:
            x, y, w, h = point
            frame = cv2.cvtColor(np.asarray(self._im), cv2.COLOR_RGB2BGR)
            neighbor = w // 10
            fh, fw = frame.shape[0], frame.shape[1]
            if (y + h > fh) or (x + w > fw):
                return
            for i in range(0, h - neighbor, neighbor):  # 关键点0 减去neightbour 防止溢出
                for j in range(0, w - neighbor, neighbor):
                    rect = [j + x, i + y, neighbor, neighbor]
                    color = frame[i + y][j + x].tolist()  # 关键点1 tolist
                    left_up = (rect[0], rect[1])
                    right_down = (rect[0] + neighbor - 1, rect[1] + neighbor - 1)  # 关键点2 减去一个像素
                    cv2.rectangle(frame, left_up, right_down, color, -1)
            self._im = Image.fromarray(cv2.cvtColor(frame, cv2.COLOR_BGR2RGB))

    def text_font(self, x, y, text, color, fnt):
        self._draw.text((x, y), text, color, fnt)
