"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiPipelineLib = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const imagebuilder = require("@aws-cdk/aws-imagebuilder");
const kms = require("@aws-cdk/aws-kms");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const cleaner_1 = require("./cleaner");
const component_builder_1 = require("./component_builder");
const image_recipe_version_bump_1 = require("./image_recipe_version_bump");
const slack_notification_1 = require("./slack_notification");
const source_action_builder_1 = require("./source_action_builder");
const ssmupdate = require("./ssm_update");
const utils = require("./utils");
/**
 * Construct for creating a Codepipeline, EC2 Image builder pipeline from 1 pipeline configuration.
 *
 * @stability stable
 */
class AmiPipelineLib extends cdk.Construct {
    /**
     * Constructor.
     *
     * @stability stable
     */
    constructor(scope, id, pipelineConfig, componentDepsConfig, componentBuilder, optionalParams) {
        super(scope, id);
        this.id = id;
        this.pipelineConfig = pipelineConfig;
        this.pipelineConfig.name = this.id;
        this.componentDepsConfig = componentDepsConfig;
        this.slackConfig = {
            channel: optionalParams.channel,
            slackWebhookUrl: optionalParams.slackWebhookUrl,
            username: optionalParams.username,
        };
        this.componentBuilder = componentBuilder !== null && componentBuilder !== void 0 ? componentBuilder : new component_builder_1.ComponentBuilder(this, componentDepsConfig);
        this.sourceActionBuilder = new source_action_builder_1.SourceActionBuilder(this, pipelineConfig.sources, this.id);
        this.extraParameters = optionalParams.extraParams;
        this.createImagebuilderPipeline();
        this.createCodepipelineProject();
        this.createScheduledTask();
        this.createCleanerTask();
        if (this.topic) {
            new ssmupdate.SsmUpdateConstruct(this, 'SSMUpdate', this.topic, this.pipelineConfig);
            if (this.slackConfig && this.slackConfig.channel && this.slackConfig.slackWebhookUrl && this.slackConfig.username) {
                new slack_notification_1.SlackNotification(this, 'SlackNotification', this.topic, this.slackConfig, `${this.id}Recipe`);
            }
        }
    }
    /**
     * @stability stable
     */
    createCleanerTask() {
        if (this.pipelineConfig.cleaning_schedule) {
            new cleaner_1.Cleaner(this, 'Cleaner', this.pipelineConfig.cleaning_schedule, this.id);
        }
    }
    /**
     * @stability stable
     */
    createScheduledTask() {
        if (this.codepipeline && this.pipelineConfig.schedule) {
            const pipelineTarget = new event_targets.CodePipeline(this.codepipeline);
            new events.Rule(this, 'ScheduleRule', {
                schedule: events.Schedule.expression(this.pipelineConfig.schedule),
                targets: [pipelineTarget],
            });
        }
    }
    /**
     * @stability stable
     */
    createImagebuilderPipeline() {
        let image_id;
        let parentImage;
        if ('disk_size' in this.pipelineConfig && this.pipelineConfig.disk_size) {
            this.diskSize = this.pipelineConfig.disk_size;
        }
        if ('image_id' in this.pipelineConfig && this.pipelineConfig.image_id) {
            image_id = this.pipelineConfig.image_id;
        }
        else {
            switch (this.pipelineConfig.parent_image) {
                case 'AmazonLinux2':
                    parentImage = ec2.MachineImage.latestAmazonLinux({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        edition: ec2.AmazonLinuxEdition.STANDARD,
                        virtualization: ec2.AmazonLinuxVirt.HVM,
                        storage: ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
                        cpuType: ec2.AmazonLinuxCpuType.X86_64,
                    });
                    break;
                default:
                    parentImage = ec2.MachineImage.lookup(this.getLookupCriteria(this.pipelineConfig.parent_image));
                    break;
            }
        }
        // Use a custom block device mapping if encryption is required.
        let blockDeviceMappings = this.createBlockDeviceMapping();
        this.recipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
            name: `${this.id}Recipe`,
            parentImage: image_id ? image_id : parentImage ? parentImage.getImage(this).imageId : null,
            version: this.getNextRecipeVersion(`${this.id}Recipe`),
            components: this.getComponents(this.pipelineConfig.recipe.components),
            blockDeviceMappings: blockDeviceMappings,
            tags: {
                ShortName: `${this.id}Recipe`,
            },
        });
        const builderRole = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
            ],
            inlinePolicies: this.getInlinePolicies(this.pipelineConfig),
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [builderRole.roleName],
        });
        this.topic = new sns.Topic(this, 'AmiPipelineTopic');
        this.infrastructure = new imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: `${this.id} - EC2 IB Infrastructure`,
            instanceProfileName: instanceProfile.ref,
            snsTopicArn: this.topic.topicArn,
            instanceTypes: this.pipelineConfig.instance_type ? [this.pipelineConfig.instance_type] : undefined,
            terminateInstanceOnFailure: !('terminate_on_failure' in this.pipelineConfig) || this.pipelineConfig.terminate_on_failure,
            subnetId: this.pipelineConfig.subnet_id,
            securityGroupIds: this.pipelineConfig.security_group_ids ? this.pipelineConfig.security_group_ids.split(',') : undefined,
        });
        this.distributionConfig = this.createDistributionConfig();
        this.imagePipeline = new imagebuilder.CfnImagePipeline(this, 'AmiPipeline', {
            name: `${this.id} - AMI Pipeline`,
            imageRecipeArn: this.recipe.ref,
            infrastructureConfigurationArn: this.infrastructure.ref,
            distributionConfigurationArn: this.distributionConfig ? this.distributionConfig.ref : undefined,
        });
    }
    getComponents(components) {
        let compMap = this.componentBuilder.componentDependenciesMap;
        if (this.extraParameters) {
            for (let param of this.extraParameters) {
                let component = components.find((c) => c.name === param.componentName);
                if (component) {
                    component.parameters = component.parameters || {};
                    component.parameters = {
                        ...component.parameters,
                        ...param.parameters,
                    };
                }
            }
        }
        const compList = components.map((c) => ({
            componentArn: compMap[c.name].ref,
            parameters: 'parameters' in c ? Object.keys(c.parameters).map((k) => ({
                name: k,
                value: [c.parameters[k]],
            })) : [],
        }));
        return compList;
    }
    createDistributionConfig() {
        // Get current region
        const region = cdk.Stack.of(this).region;
        const accountId = cdk.Stack.of(this).account;
        const amiTags = {
            Name: this.id,
            CreatedBy: 'EC2 Image Builder',
            Ec2ImageBuilderArn: '{{imagebuilder:buildVersion}}',
            AutoClean: 'True',
        };
        let sharedWithDistributions = [];
        if (this.pipelineConfig.shared_with) {
            sharedWithDistributions = this.pipelineConfig.shared_with.map((s) => ({
                region: s.region,
                amiDistributionConfiguration: {
                    Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                    Description: `${this.id} Shared image`,
                    AmiTags: amiTags,
                    TargetAccountIds: s.account_ids,
                },
            }));
            // current region might already be included
            if (sharedWithDistributions.some(s => s.region === region)) {
                const distribution = sharedWithDistributions.find(s => s.region === region);
                //Push current account ID as distribute
                distribution === null || distribution === void 0 ? void 0 : distribution.amiDistributionConfiguration.TargetAccountIds.push(accountId);
            }
        }
        if (sharedWithDistributions.length == 0) {
            sharedWithDistributions = [{
                    region: region,
                    amiDistributionConfiguration: {
                        Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                        Description: `${this.id} Shared image`,
                        AmiTags: amiTags,
                        TargetAccountIds: [accountId],
                    },
                }];
        }
        return new imagebuilder.CfnDistributionConfiguration(this, 'DistributionConfig', {
            name: `${this.id} - Distribution`,
            distributions: sharedWithDistributions,
        });
    }
    getInlinePolicies(pipelineConfig) {
        let result = {};
        const statements = pipelineConfig.policy_statements;
        if (statements) {
            for (let statement in statements) {
                result[statement] = new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            sid: statement,
                            actions: statements[statement].actions,
                            effect: iam.Effect.ALLOW,
                            resources: statements[statement].resources,
                        }),
                    ],
                });
            }
        }
        return result;
    }
    createBlockDeviceMapping() {
        const key = this.pipelineConfig.encrypted ? new kms.Key(this, 'PipelineEncryption', {
            description: `Image encryption for ${this.pipelineConfig.name}`,
            alias: `${utils.pascalize(this.pipelineConfig.name)}_key`,
        }) : undefined;
        this.ebsEncryptionKey = key;
        let blockDeviceMappings = [
            {
                deviceName: this.pipelineConfig.parent_image.includes('AmazonLinux') ? '/dev/xvda' : '/dev/sda1',
                ebs: {
                    encrypted: key ? true : false,
                    volumeSize: this.pipelineConfig.disk_size ? this.pipelineConfig.disk_size : 8,
                    volumeType: 'gp2',
                    kmsKeyId: key === null || key === void 0 ? void 0 : key.keyArn,
                },
            },
        ];
        return blockDeviceMappings;
    }
    /**
     * @stability stable
     */
    getNextRecipeVersion(recipeName) {
        const recipeBumpCustomResource = new image_recipe_version_bump_1.ImageRecipeVersionBump(this, 'ImageRecipeVersionBump', recipeName);
        return recipeBumpCustomResource.nextVersion;
    }
    /**
     * @stability stable
     */
    createCodepipelineProject() {
        var _b;
        if (this.imagePipeline) {
            // Create CodePipeline project
            let envVariables = {
                IMAGE_PIPELINE_ARN: { value: this.imagePipeline.ref },
                PIPELINE_NAME: { value: this.imagePipeline.name },
            };
            if (this.slackConfig && this.slackConfig.slackWebhookUrl && this.slackConfig.channel && this.slackConfig.username) {
                envVariables.SLACK_WEBHOOK_URL = { value: this.slackConfig.slackWebhookUrl };
                envVariables.SLACK_CHANNEL = { value: this.slackConfig.channel };
                envVariables.SLACK_USERNAME = { value: this.slackConfig.username };
            }
            let imagebuilderBuild = new codebuild.PipelineProject(this, 'AmiPipelineBuild', {
                buildSpec: codebuild.BuildSpec.fromObject({
                    version: '0.2',
                    phases: {
                        install: {
                            commands: [
                                'apt-get update',
                                'apt-get install awscli -y',
                            ],
                        },
                        pre_build: {
                            commands: [
                                'if [ -n "$SLACK_WEBHOOK_URL" ]; then curl -X POST -H \'Content-type: application/json\' --data "{\\"text\\":\\"$PIPELINE_NAME build has started.\\"}" $SLACK_WEBHOOK_URL; fi',
                            ],
                        },
                        build: {
                            commands: 'aws imagebuilder start-image-pipeline-execution --image-pipeline-arn $IMAGE_PIPELINE_ARN',
                        },
                    },
                }),
                environment: {
                    buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                    environmentVariables: envVariables,
                },
            });
            imagebuilderBuild.addToRolePolicy(iam.PolicyStatement.fromJson({
                Sid: 'imagebuilderstart',
                Effect: 'Allow',
                Action: ['imagebuilder:StartImagePipelineExecution'],
                Resource: '*',
            }));
            const imageBuilderOutput = new codepipeline.Artifact('CdkBuildOutput');
            let sourceActions = this.sourceActionBuilder.createPipelineSources();
            const buildAction = new codepipeline_actions.CodeBuildAction({
                actionName: 'StartImageBuilder',
                project: imagebuilderBuild,
                input: (_b = sourceActions.find(a => a && a.sourceOutput)) === null || _b === void 0 ? void 0 : _b.sourceOutput,
                extraInputs: sourceActions.length > 1 ? sourceActions.slice(1).map(a => a.sourceOutput) : [],
                outputs: [imageBuilderOutput],
            });
            // Complete Pipeline Project
            this.codepipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: this.pipelineConfig.name,
                restartExecutionOnUpdate: true,
                stages: [
                    {
                        stageName: 'Source',
                        actions: sourceActions.map(c => c.action).filter((c) => c != null),
                    },
                    {
                        stageName: 'Build',
                        actions: [buildAction],
                    },
                ],
            });
        }
    }
    /**
     * @stability stable
     */
    getLookupCriteria(parentImage) {
        switch (parentImage) {
            case 'Ubuntu1804':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'Ubuntu2004':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'CentOS7':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['cvugziknvmxgqna9noibqnnsy'],
                    },
                };
            case 'CentOS8':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['47k9ia2igxpcce2bzo8u3kj03'],
                    },
                };
            default:
                return {
                    name: 'amzn-linux',
                };
        }
    }
}
exports.AmiPipelineLib = AmiPipelineLib;
_a = JSII_RTTI_SYMBOL_1;
AmiPipelineLib[_a] = { fqn: "halloumi-ami-pipelines.AmiPipelineLib", version: "0.0.27" };
//# sourceMappingURL=data:application/json;base64,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