"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("@aws-cdk/aws-codepipeline");
const action_1 = require("../action");
const jenkins_provider_1 = require("./jenkins-provider");
/**
 * The type of the Jenkins Action that determines its CodePipeline Category -
 * Build, or Test.
 * Note that a Jenkins provider, even if it has the same name,
 * must be separately registered for each type.
 */
var JenkinsActionType;
(function (JenkinsActionType) {
    /**
     * The Action will have the Build Category.
     */
    JenkinsActionType[JenkinsActionType["BUILD"] = 0] = "BUILD";
    /**
     * The Action will have the Test Category.
     */
    JenkinsActionType[JenkinsActionType["TEST"] = 1] = "TEST";
})(JenkinsActionType = exports.JenkinsActionType || (exports.JenkinsActionType = {}));
/**
 * Jenkins build CodePipeline Action.
 *
 * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/tutorials-four-stage-pipeline.html
 */
class JenkinsAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: props.type === JenkinsActionType.BUILD
                ? codepipeline.ActionCategory.BUILD
                : codepipeline.ActionCategory.TEST,
            provider: props.jenkinsProvider.providerName,
            owner: 'Custom',
            artifactBounds: jenkins_provider_1.jenkinsArtifactsBounds,
            version: props.jenkinsProvider.version,
        });
        this.props = props;
    }
    bound(_scope, _stage, _options) {
        if (this.actionProperties.category === codepipeline.ActionCategory.BUILD) {
            this.props.jenkinsProvider._registerBuildProvider();
        }
        else {
            this.props.jenkinsProvider._registerTestProvider();
        }
        return {
            configuration: {
                ProjectName: this.props.projectName,
            },
        };
    }
}
exports.JenkinsAction = JenkinsAction;
//# sourceMappingURL=data:application/json;base64,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