# !/usr/bin/env python
"""Define the unit tests for the :mod:`colour_datasets.loaders.kuopio` module."""

import numpy as np
import os
import unittest

from colour import SpectralShape

from colour_datasets.loaders.kuopio import (
    MatFileMetadata_KuopioUniversity,
    read_sds_from_mat_file_KuopioUniversity,
)
from colour_datasets.loaders.kuopio import (  # type: ignore[attr-defined]
    DatasetLoader_MunsellColorsMattSpectrofotometerMeasured,
    DatasetLoader_MunsellColorsMattAOTFMeasured,
    DatasetLoader_MunsellColorsGlossySpectrofotometerMeasured,
    DatasetLoader_MunsellColorsGlossyAllSpectrofotometerMeasured,
    DatasetLoader_ForestColors,
    DatasetLoader_PaperSpectra,
    DatasetLoader_LumberSpectra,
    DatasetLoader_AgfaIT872Set,
)

__author__ = "Colour Developers"
__copyright__ = "Copyright 2019 Colour Developers"
__license__ = "New BSD License - https://opensource.org/licenses/BSD-3-Clause"
__maintainer__ = "Colour Developers"
__email__ = "colour-developers@colour-science.org"
__status__ = "Production"

__all__ = [
    "TestReadSdsFromMatFileKuopioUniversity",
    "TestDatasetLoader_KuopioUniversity",
]


class TestReadSdsFromMatFileKuopioUniversity(unittest.TestCase):
    """
    Define :func:`colour_datasets.loaders.kuopio.\
read_sds_from_mat_file_KuopioUniversity` definition unit tests methods.
    """

    def test_read_sds_from_mat_file_KuopioUniversity(self):
        """
        Test :func:`colour_datasets.loaders.kuopio.\
read_sds_from_mat_file_KuopioUniversity` definition.
        """

        mat_file = os.path.join(
            os.path.dirname(__file__), "resources", "Kuopio_Typical.mat"
        )
        metadata = MatFileMetadata_KuopioUniversity(
            "munsell", SpectralShape(400, 700, 10), True, "S"
        )

        sds = read_sds_from_mat_file_KuopioUniversity(mat_file, metadata)
        self.assertEqual(len(sds), 32)

        np.testing.assert_array_almost_equal(
            sds["5R 6/8 (3)"].values,
            np.array(
                [
                    17.870000000000000,
                    19.840000000000000,
                    19.530000000000000,
                    18.730000000000000,
                    18.230000000000000,
                    17.630000000000000,
                    17.040000000000000,
                    16.660000000000000,
                    15.700000000000000,
                    14.080000000000000,
                    12.810000000000000,
                    12.650000000000000,
                    12.110000000000000,
                    10.840000000000000,
                    10.890000000000000,
                    14.260000000000000,
                    23.080000000000000,
                    33.950000000000000,
                    43.500000000000000,
                    49.790000000000000,
                    52.340000000000000,
                    53.170000000000000,
                    53.300000000000000,
                    53.100000000000000,
                    52.960000000000000,
                    52.840000000000000,
                    52.700000000000000,
                    52.480000000000000,
                    52.120000000000000,
                    51.940000000000000,
                    51.890000000000000,
                ]
            ),
            decimal=7,
        )


class TestDatasetLoader_KuopioUniversity(unittest.TestCase):
    """
    Define :class:`colour_datasets.loaders.kuopio.\
DatasetLoader_KuopioUniversity` class unit tests methods.
    """

    def test_required_attributes(self):
        """Test the presence of required attributes."""

        required_attributes = ("ID", "METADATA")

        for dataset_loader in (
            DatasetLoader_MunsellColorsMattSpectrofotometerMeasured,
            DatasetLoader_MunsellColorsMattAOTFMeasured,
            DatasetLoader_MunsellColorsGlossySpectrofotometerMeasured,
            DatasetLoader_MunsellColorsGlossyAllSpectrofotometerMeasured,
            DatasetLoader_ForestColors,
            DatasetLoader_PaperSpectra,
            DatasetLoader_LumberSpectra,
            DatasetLoader_AgfaIT872Set,
        ):
            for attribute in required_attributes:
                self.assertIn(attribute, dir(dataset_loader))

    def test_required_methods(self):
        """Test the presence of required methods."""

        required_methods = ("__init__", "load")

        for dataset_loader in (
            DatasetLoader_MunsellColorsMattSpectrofotometerMeasured,
            DatasetLoader_MunsellColorsMattAOTFMeasured,
            DatasetLoader_MunsellColorsGlossySpectrofotometerMeasured,
            DatasetLoader_MunsellColorsGlossyAllSpectrofotometerMeasured,
            DatasetLoader_ForestColors,
            DatasetLoader_PaperSpectra,
            DatasetLoader_LumberSpectra,
            DatasetLoader_AgfaIT872Set,
        ):
            for method in required_methods:
                self.assertIn(method, dir(dataset_loader))

    def test_load(self):
        """
        Test :func:`colour_datasets.loaders.kuopio.\
DatasetLoader_KuopioUniversity.load` method.
        """

        dataset_loaders = {
            DatasetLoader_MunsellColorsMattSpectrofotometerMeasured(): (
                "munsell",
                1269,
                "2.5R 5/2",
                np.array(
                    [
                        0.105600000000000,
                        0.114100000000000,
                        0.117000000000000,
                        0.119000000000000,
                        0.121900000000000,
                        0.124600000000000,
                        0.127100000000000,
                        0.135200000000000,
                        0.137100000000000,
                        0.141900000000000,
                        0.151300000000000,
                        0.155800000000000,
                        0.154900000000000,
                        0.162200000000000,
                        0.166000000000000,
                        0.168700000000000,
                        0.173300000000000,
                        0.174400000000000,
                        0.173900000000000,
                        0.179700000000000,
                        0.182200000000000,
                        0.183900000000000,
                        0.185500000000000,
                        0.184600000000000,
                        0.184800000000000,
                        0.187700000000000,
                        0.184600000000000,
                        0.183700000000000,
                        0.187600000000000,
                        0.186500000000000,
                        0.183200000000000,
                        0.185800000000000,
                        0.184400000000000,
                        0.181300000000000,
                        0.183800000000000,
                        0.181800000000000,
                        0.179500000000000,
                        0.181900000000000,
                        0.180200000000000,
                        0.178600000000000,
                        0.179600000000000,
                        0.176300000000000,
                        0.176300000000000,
                        0.178100000000000,
                        0.175100000000000,
                        0.174000000000000,
                        0.176300000000000,
                        0.173400000000000,
                        0.173800000000000,
                        0.175500000000000,
                        0.171100000000000,
                        0.170600000000000,
                        0.174300000000000,
                        0.171800000000000,
                        0.170600000000000,
                        0.172800000000000,
                        0.172300000000000,
                        0.171200000000000,
                        0.170900000000000,
                        0.169300000000000,
                        0.172000000000000,
                        0.175300000000000,
                        0.171500000000000,
                        0.169500000000000,
                        0.170800000000000,
                        0.167100000000000,
                        0.167700000000000,
                        0.170100000000000,
                        0.165600000000000,
                        0.164800000000000,
                        0.167500000000000,
                        0.164600000000000,
                        0.162500000000000,
                        0.165500000000000,
                        0.162500000000000,
                        0.163400000000000,
                        0.163600000000000,
                        0.161500000000000,
                        0.162400000000000,
                        0.164500000000000,
                        0.162200000000000,
                        0.163900000000000,
                        0.164900000000000,
                        0.162200000000000,
                        0.163600000000000,
                        0.166100000000000,
                        0.162900000000000,
                        0.164700000000000,
                        0.167400000000000,
                        0.164800000000000,
                        0.165200000000000,
                        0.165400000000000,
                        0.164400000000000,
                        0.166300000000000,
                        0.167600000000000,
                        0.163800000000000,
                        0.164400000000000,
                        0.163600000000000,
                        0.162100000000000,
                        0.163800000000000,
                        0.162900000000000,
                        0.158600000000000,
                        0.160300000000000,
                        0.159100000000000,
                        0.155300000000000,
                        0.157200000000000,
                        0.159000000000000,
                        0.154100000000000,
                        0.154500000000000,
                        0.154600000000000,
                        0.151100000000000,
                        0.153700000000000,
                        0.155000000000000,
                        0.151200000000000,
                        0.151100000000000,
                        0.151300000000000,
                        0.149300000000000,
                        0.152000000000000,
                        0.150700000000000,
                        0.148400000000000,
                        0.151300000000000,
                        0.152600000000000,
                        0.149000000000000,
                        0.152200000000000,
                        0.152700000000000,
                        0.149700000000000,
                        0.150800000000000,
                        0.151100000000000,
                        0.148300000000000,
                        0.152100000000000,
                        0.153100000000000,
                        0.150900000000000,
                        0.151100000000000,
                        0.149300000000000,
                        0.148500000000000,
                        0.150900000000000,
                        0.148300000000000,
                        0.145300000000000,
                        0.148100000000000,
                        0.147000000000000,
                        0.144900000000000,
                        0.146600000000000,
                        0.145300000000000,
                        0.143100000000000,
                        0.146300000000000,
                        0.144400000000000,
                        0.141000000000000,
                        0.141200000000000,
                        0.143700000000000,
                        0.142000000000000,
                        0.142900000000000,
                        0.142400000000000,
                        0.142700000000000,
                        0.145400000000000,
                        0.142900000000000,
                        0.141300000000000,
                        0.144500000000000,
                        0.143700000000000,
                        0.143400000000000,
                        0.148600000000000,
                        0.147500000000000,
                        0.147000000000000,
                        0.151300000000000,
                        0.153100000000000,
                        0.150800000000000,
                        0.155800000000000,
                        0.156400000000000,
                        0.158100000000000,
                        0.162700000000000,
                        0.164600000000000,
                        0.165200000000000,
                        0.171900000000000,
                        0.171900000000000,
                        0.173600000000000,
                        0.176300000000000,
                        0.176800000000000,
                        0.178200000000000,
                        0.183400000000000,
                        0.179800000000000,
                        0.177500000000000,
                        0.180400000000000,
                        0.184800000000000,
                        0.191700000000000,
                        0.194900000000000,
                        0.195100000000000,
                        0.193100000000000,
                        0.196400000000000,
                        0.196700000000000,
                        0.195900000000000,
                        0.201000000000000,
                        0.200100000000000,
                        0.197400000000000,
                        0.201200000000000,
                        0.200500000000000,
                        0.198200000000000,
                        0.203900000000000,
                        0.204700000000000,
                        0.201900000000000,
                        0.204000000000000,
                        0.203800000000000,
                        0.202000000000000,
                        0.206200000000000,
                        0.206600000000000,
                        0.204200000000000,
                        0.205500000000000,
                        0.204700000000000,
                        0.205100000000000,
                        0.208300000000000,
                        0.206300000000000,
                        0.204100000000000,
                        0.208100000000000,
                        0.206600000000000,
                        0.203300000000000,
                        0.206400000000000,
                        0.206400000000000,
                        0.203900000000000,
                        0.206100000000000,
                        0.205400000000000,
                        0.202700000000000,
                        0.204600000000000,
                        0.205300000000000,
                        0.204700000000000,
                        0.205800000000000,
                        0.204600000000000,
                        0.204000000000000,
                        0.206900000000000,
                        0.203500000000000,
                        0.202900000000000,
                        0.207000000000000,
                        0.206200000000000,
                        0.203400000000000,
                        0.206500000000000,
                        0.205500000000000,
                        0.204300000000000,
                        0.207100000000000,
                        0.205100000000000,
                        0.202300000000000,
                        0.205000000000000,
                        0.204100000000000,
                        0.203900000000000,
                        0.205800000000000,
                        0.203500000000000,
                        0.203800000000000,
                        0.205800000000000,
                        0.203200000000000,
                        0.201200000000000,
                        0.203700000000000,
                        0.200600000000000,
                        0.201400000000000,
                        0.206300000000000,
                        0.203100000000000,
                        0.201000000000000,
                        0.203600000000000,
                        0.200500000000000,
                        0.199300000000000,
                        0.202700000000000,
                        0.202300000000000,
                        0.201700000000000,
                        0.202900000000000,
                        0.199900000000000,
                        0.199200000000000,
                        0.202500000000000,
                        0.200400000000000,
                        0.199500000000000,
                        0.199800000000000,
                        0.196400000000000,
                        0.197200000000000,
                        0.200500000000000,
                        0.196800000000000,
                        0.196700000000000,
                        0.200100000000000,
                        0.197800000000000,
                        0.195700000000000,
                        0.198500000000000,
                        0.196500000000000,
                        0.197100000000000,
                        0.198400000000000,
                        0.195900000000000,
                        0.196400000000000,
                        0.200200000000000,
                        0.198300000000000,
                        0.199400000000000,
                        0.198500000000000,
                        0.193700000000000,
                        0.195300000000000,
                        0.198400000000000,
                        0.193300000000000,
                        0.193800000000000,
                        0.197200000000000,
                        0.193900000000000,
                        0.194700000000000,
                        0.197300000000000,
                        0.195000000000000,
                        0.197700000000000,
                        0.200700000000000,
                        0.197100000000000,
                        0.196300000000000,
                        0.197700000000000,
                        0.196800000000000,
                        0.197300000000000,
                        0.198000000000000,
                        0.193900000000000,
                        0.195300000000000,
                        0.196100000000000,
                        0.192900000000000,
                        0.195100000000000,
                        0.196100000000000,
                        0.188600000000000,
                        0.187100000000000,
                        0.192700000000000,
                        0.195800000000000,
                        0.194800000000000,
                        0.196100000000000,
                        0.192200000000000,
                        0.195100000000000,
                        0.196500000000000,
                        0.192600000000000,
                        0.195000000000000,
                        0.195500000000000,
                        0.193100000000000,
                        0.193200000000000,
                        0.194200000000000,
                        0.191800000000000,
                        0.194500000000000,
                        0.193600000000000,
                        0.191100000000000,
                        0.193100000000000,
                        0.192800000000000,
                        0.190600000000000,
                        0.193600000000000,
                        0.192400000000000,
                        0.189900000000000,
                        0.192900000000000,
                        0.193900000000000,
                        0.191700000000000,
                        0.194700000000000,
                        0.195900000000000,
                        0.192400000000000,
                        0.192900000000000,
                        0.193100000000000,
                        0.192200000000000,
                        0.195700000000000,
                        0.195500000000000,
                        0.192200000000000,
                        0.195100000000000,
                        0.193600000000000,
                        0.192300000000000,
                        0.196200000000000,
                        0.196100000000000,
                        0.193800000000000,
                        0.198400000000000,
                        0.196900000000000,
                        0.192900000000000,
                        0.196300000000000,
                        0.196800000000000,
                        0.194800000000000,
                        0.195600000000000,
                        0.194600000000000,
                        0.193700000000000,
                        0.196900000000000,
                        0.196100000000000,
                        0.194800000000000,
                        0.196700000000000,
                        0.193900000000000,
                        0.193400000000000,
                        0.197000000000000,
                        0.193000000000000,
                        0.192200000000000,
                        0.196000000000000,
                        0.193800000000000,
                        0.191800000000000,
                        0.194000000000000,
                        0.193500000000000,
                        0.191000000000000,
                        0.192300000000000,
                        0.190700000000000,
                        0.189500000000000,
                        0.191700000000000,
                        0.189900000000000,
                        0.191200000000000,
                        0.192400000000000,
                        0.189600000000000,
                        0.189500000000000,
                        0.193200000000000,
                        0.191200000000000,
                        0.188600000000000,
                        0.192800000000000,
                        0.191700000000000,
                        0.190800000000000,
                        0.194300000000000,
                        0.191600000000000,
                        0.190400000000000,
                        0.192700000000000,
                        0.191000000000000,
                        0.189300000000000,
                        0.192200000000000,
                        0.190200000000000,
                        0.190100000000000,
                        0.192900000000000,
                        0.191300000000000,
                        0.192900000000000,
                        0.196700000000000,
                        0.191600000000000,
                        0.192400000000000,
                        0.195900000000000,
                        0.189600000000000,
                        0.188700000000000,
                        0.192400000000000,
                        0.193200000000000,
                        0.193000000000000,
                        0.193400000000000,
                        0.190200000000000,
                        0.192200000000000,
                        0.193100000000000,
                        0.190000000000000,
                        0.191000000000000,
                        0.192500000000000,
                        0.190200000000000,
                        0.191300000000000,
                        0.192700000000000,
                        0.190000000000000,
                    ]
                ),
            ),
            DatasetLoader_MunsellColorsMattAOTFMeasured(): (
                "munsell",
                1250,
                "10bV50C01.NM5",
                np.array(
                    [
                        0.363525390625000,
                        0.486328125000000,
                        0.262451171875000,
                        0.270263671875000,
                        0.278076171875000,
                        0.293945312500000,
                        0.272705078125000,
                        0.253417968750000,
                        0.272216796875000,
                        0.255859375000000,
                        0.260498046875000,
                        0.253906250000000,
                        0.256591796875000,
                        0.248535156250000,
                        0.245849609375000,
                        0.243408203125000,
                        0.247802734375000,
                        0.240234375000000,
                        0.247314453125000,
                        0.243164062500000,
                        0.237548828125000,
                        0.238525390625000,
                        0.230957031250000,
                        0.227050781250000,
                        0.231689453125000,
                        0.232421875000000,
                        0.228027343750000,
                        0.223876953125000,
                        0.224853515625000,
                        0.219726562500000,
                        0.220703125000000,
                        0.218994140625000,
                        0.216552734375000,
                        0.217529296875000,
                        0.217041015625000,
                        0.213134765625000,
                        0.212402343750000,
                        0.204833984375000,
                        0.210205078125000,
                        0.205810546875000,
                        0.201416015625000,
                        0.202392578125000,
                        0.200439453125000,
                        0.198730468750000,
                        0.197998046875000,
                        0.193359375000000,
                        0.192871093750000,
                        0.193115234375000,
                        0.192626953125000,
                        0.188476562500000,
                        0.189208984375000,
                        0.185058593750000,
                        0.185546875000000,
                        0.186035156250000,
                        0.183349609375000,
                        0.183105468750000,
                        0.181884765625000,
                        0.178222656250000,
                        0.175292968750000,
                        0.169921875000000,
                        0.175048828125000,
                    ]
                ),
            ),
            DatasetLoader_MunsellColorsGlossySpectrofotometerMeasured(): (
                "munsell",
                32,
                "5R 5/6",
                np.array(
                    [
                        12.660000000000000,
                        13.540000000000000,
                        12.990000000000000,
                        12.260000000000000,
                        11.910000000000000,
                        11.580000000000000,
                        11.360000000000000,
                        11.430000000000000,
                        10.910000000000000,
                        9.8000000000000000,
                        9.1100000000000000,
                        9.1400000000000000,
                        8.5200000000000000,
                        7.4800000000000000,
                        8.1600000000000000,
                        11.190000000000000,
                        15.190000000000000,
                        18.460000000000000,
                        23.560000000000000,
                        29.770000000000000,
                        33.250000000000000,
                        34.400000000000000,
                        34.540000000000000,
                        34.350000000000000,
                        34.200000000000000,
                        34.050000000000000,
                        33.800000000000000,
                        33.560000000000000,
                        33.290000000000000,
                        33.080000000000000,
                        32.910000000000000,
                    ]
                ),
            ),
            DatasetLoader_MunsellColorsGlossyAllSpectrofotometerMeasured(): (
                "X",
                1600,
                "5",
                np.array(
                    [
                        0.0832583349355893,
                        0.0841964216140708,
                        0.0854254747054747,
                        0.0864870564212114,
                        0.0885143682165685,
                        0.0905455902475432,
                        0.0915811880405238,
                        0.0935670213290593,
                        0.0953374607500153,
                        0.0969212265220306,
                        0.0988861173336562,
                        0.1011019151764140,
                        0.1027070137118110,
                        0.1045144157706090,
                        0.1066298320094840,
                        0.1078871227364190,
                        0.1097310323760100,
                        0.1114069239380190,
                        0.1121451511457540,
                        0.1134318032825190,
                        0.1141553695955370,
                        0.1148042526315790,
                        0.1151973800818870,
                        0.1163717178232080,
                        0.1153836989247310,
                        0.1163973344056990,
                        0.1164192531233960,
                        0.1176007052049480,
                        0.1185813542341110,
                        0.1188167084135430,
                        0.1188947903717930,
                        0.1194576529747440,
                        0.1206333985004790,
                        0.1203924436437340,
                        0.1212710711071110,
                        0.1208673887423540,
                        0.1215377256924970,
                        0.1218716508912110,
                        0.1213794497567520,
                        0.1217316822846940,
                        0.1216057200200700,
                        0.1220691362725450,
                        0.1223934228755990,
                        0.1226491662040630,
                        0.1222738901730910,
                        0.1235775559991130,
                        0.1240115273049840,
                        0.1245753981184280,
                        0.1249519072803720,
                        0.1251793875497570,
                        0.1253437823548850,
                        0.1259486272019440,
                        0.1259670591996470,
                        0.1261504072273180,
                        0.1270547857142860,
                        0.1275530353200880,
                        0.1278131387343720,
                        0.1280998512642540,
                        0.1287212301001870,
                        0.1289580095830810,
                        0.1290085828891700,
                        0.1304132516826660,
                        0.1309290648193960,
                        0.1315601250826540,
                        0.1320659696068720,
                        0.1328932240677590,
                        0.1336453489265910,
                        0.1340303717553890,
                        0.1347657294298580,
                        0.1352923279986800,
                        0.1360370366290280,
                        0.1365566273001920,
                        0.1375466104152930,
                        0.1380393871162610,
                        0.1391758261775510,
                        0.1393372198783630,
                        0.1403947401936650,
                        0.1410517545489320,
                        0.1420981132075470,
                        0.1424267063197030,
                        0.1431591745373150,
                        0.1439438302804960,
                        0.1449724509333040,
                        0.1457406097108570,
                        0.1466319866826770,
                        0.1477144227624550,
                        0.1491561375701750,
                        0.1499657283479590,
                        0.1508730084930310,
                        0.1524472420812020,
                        0.1538901500326160,
                        0.1551999854276550,
                        0.1564189116238570,
                        0.1575284381833020,
                        0.1588692308277620,
                        0.1593696495517520,
                        0.1605326245110820,
                        0.1618569582133350,
                        0.1624176661422450,
                        0.1634395257586450,
                        0.1635596262494570,
                        0.1647163760720880,
                        0.1653961094581390,
                        0.1659311061379690,
                        0.1668263889643190,
                        0.1664016268098260,
                        0.1663602603460430,
                        0.1672364293227780,
                        0.1676109344315600,
                        0.1680388326738580,
                        0.1677260481471460,
                        0.1674615913396480,
                        0.1674423665261110,
                        0.1669457804244260,
                        0.1667212939521800,
                        0.1666681862479700,
                        0.1661996093893670,
                        0.1660631997190860,
                        0.1650462213562810,
                        0.1644598642563330,
                        0.1639480785837650,
                        0.1629394804605160,
                        0.1618968264677260,
                        0.1607553251918300,
                        0.1599774502784840,
                        0.1592006389084410,
                        0.1577751116168180,
                        0.1567381133546260,
                        0.1558041359727410,
                        0.1546063862270590,
                        0.1532839006439740,
                        0.1522304826541110,
                        0.1510174361195320,
                        0.1495370270065490,
                        0.1482986794128800,
                        0.1471751082251080,
                        0.1459533303020460,
                        0.1448406104887160,
                        0.1432260271395360,
                        0.1420294881655200,
                        0.1407796123863140,
                        0.1394713345247770,
                        0.1383847867252320,
                        0.1367663760554230,
                        0.1353930054621170,
                        0.1340665548764000,
                        0.1326094541324100,
                        0.1314476955556760,
                        0.1300619568392020,
                        0.1286112691620170,
                        0.1270600768689440,
                        0.1256763453237410,
                        0.1247108740387740,
                        0.1233902828348500,
                        0.1219225162024490,
                        0.1203756671729230,
                        0.1193858886718750,
                        0.1187244485879990,
                        0.1172117915401300,
                        0.1163088532870850,
                        0.1148534423920700,
                        0.1134792034486500,
                        0.1125721330001090,
                        0.1113368023192800,
                        0.1101989148244470,
                        0.1091195956961200,
                        0.1083813403562120,
                        0.1071390462089160,
                        0.1061137185040440,
                        0.1049129130387580,
                        0.1043954382535030,
                        0.1031281954323000,
                        0.1021073306429620,
                        0.1010716444082520,
                        0.1004949793702500,
                        0.0995802646626368,
                        0.0984846824799607,
                        0.0976298555319497,
                        0.0964366697093181,
                        0.0959713445121951,
                        0.0946097380316976,
                        0.0940169040274674,
                        0.0931408770974068,
                        0.0925075464007411,
                        0.0919924512854102,
                        0.0911384338532010,
                        0.0904112434318108,
                        0.0898916765781003,
                        0.0889631941324027,
                        0.0886735681284474,
                        0.0881560421558456,
                        0.0874990131233596,
                        0.0870141730990311,
                        0.0865602858079677,
                        0.0866091052286152,
                        0.0860980602739726,
                        0.0854415269900361,
                        0.0852274163424125,
                        0.0846683259332347,
                        0.0846999037966362,
                        0.0846302515481997,
                        0.0837975875576037,
                        0.0838024112149533,
                        0.0835321230735480,
                        0.0829661160327131,
                        0.0827144267149202,
                        0.0827143225629190,
                        0.0820904100032906,
                        0.0820583758300862,
                        0.0819189005552196,
                        0.0810632600471517,
                        0.0810455174001206,
                        0.0807908284431793,
                        0.0804156337410190,
                        0.0805326402629417,
                        0.0800952396585686,
                        0.0796956921896410,
                        0.0793305183644425,
                        0.0789345770872087,
                        0.0784959303128253,
                        0.0783585716629300,
                        0.0780296335618316,
                        0.0776355686360401,
                        0.0777728303000492,
                        0.0771084880319877,
                        0.0769203308138898,
                        0.0765511326039387,
                        0.0762573573616277,
                        0.0762127566381391,
                        0.0760990485894276,
                        0.0759584208223972,
                        0.0755359285636025,
                        0.0756633663670248,
                        0.0752572122010094,
                        0.0758166600909639,
                        0.0750690017513135,
                        0.0752405613919895,
                        0.0750479940841367,
                        0.0752528940517383,
                        0.0749732792022792,
                        0.0751002570131788,
                        0.0750104604924056,
                        0.0749880663745893,
                        0.0752553795596451,
                        0.0753496369021501,
                        0.0753240486895493,
                        0.0749273240054870,
                        0.0755281749629548,
                        0.0757077530932087,
                        0.0758634115061267,
                        0.0756506801228609,
                        0.0760071605101143,
                        0.0762060860026327,
                        0.0759151579640193,
                        0.0760791654510557,
                        0.0761815485996705,
                        0.0765150256522692,
                        0.0762693840004381,
                        0.0764163189645717,
                        0.0764907408057002,
                        0.0768342669584245,
                        0.0771621960440524,
                        0.0770743948220065,
                        0.0770292538916904,
                        0.0771631784423267,
                        0.0774133684557129,
                        0.0772509793050447,
                        0.0776359754048861,
                        0.0776684550740538,
                        0.0775999245903436,
                        0.0775543019880607,
                        0.0775452066523959,
                        0.0779931107448912,
                        0.0779379115287394,
                        0.0777371116127967,
                        0.0777113861657265,
                        0.0783069040254470,
                        0.0777791275336913,
                        0.0778322734546252,
                        0.0782278086575343,
                        0.0781885667306111,
                        0.0779885797133166,
                        0.0778922203584937,
                        0.0777887903693571,
                        0.0781322884794139,
                        0.0778500300990532,
                        0.0783473231527094,
                        0.0781106787355065,
                        0.0774791683038638,
                        0.0774638428430621,
                        0.0776397440804944,
                        0.0778363414820891,
                        0.0773739737159128,
                        0.0771565329105620,
                        0.0774208283325135,
                        0.0773433725061492,
                        0.0769061458287716,
                        0.0768768537704918,
                        0.0767942762841530,
                        0.0766405641193834,
                        0.0768223210852969,
                        0.0756511902310809,
                        0.0760848653489134,
                        0.0758909124746839,
                        0.0757557372797899,
                        0.0755393640350877,
                        0.0755921310541311,
                        0.0759533260309984,
                        0.0755523312534209,
                        0.0758025853417513,
                        0.0754538890712176,
                        0.0759966492343413,
                        0.0756392191463549,
                        0.0760002427745665,
                        0.0759172330727733,
                        0.0760517874821252,
                        0.0761247379087473,
                        0.0767259722054381,
                        0.0763790106863501,
                        0.0764716400109619,
                        0.0764261489525063,
                        0.0764849258345667,
                        0.0770762127916552,
                        0.0770786163439449,
                        0.0777177075901432,
                        0.0779242324199406,
                        0.0779871221093106,
                        0.0782395180299033,
                        0.0780202550409318,
                        0.0784945261194030,
                        0.0789988898659046,
                        0.0787182916666667,
                        0.0795837732500822,
                        0.0803447880449685,
                        0.0798549101363562,
                        0.0801640755957272,
                        0.0806020982436883,
                        0.0807538561632564,
                        0.0815723849317322,
                        0.0814840643355108,
                        0.0818510493352379,
                        0.0819726217696014,
                        0.0822825937877291,
                        0.0826006385614824,
                        0.0832230251162791,
                        0.0832401884518462,
                        0.0837584412217095,
                        0.0840583776960650,
                        0.0838307027945206,
                        0.0846559244351832,
                        0.0854320944276695,
                        0.0859695935852373,
                        0.0860562020024205,
                        0.0868489965268207,
                        0.0869247383567663,
                        0.0877802062760588,
                        0.0889851523971662,
                        0.0886742533164529,
                        0.0894202225519288,
                        0.0903602252401458,
                        0.0913718090645038,
                        0.0926356862097440,
                        0.0927020975529644,
                        0.0934591620557682,
                        0.0942531088738516,
                        0.0957034433521885,
                        0.0966463331682351,
                        0.0970120648886445,
                        0.0982979563203177,
                        0.0993772702256467,
                        0.1001024339091560,
                        0.1006514627853130,
                        0.1021924514103130,
                        0.1032385466651990,
                        0.1042875362287090,
                        0.1054265632733870,
                        0.1065878370941110,
                        0.1078802324765710,
                        0.1085841372602890,
                        0.1096687124910860,
                        0.1103224411182040,
                        0.1116595158900050,
                        0.1135477486645740,
                        0.1144331781621860,
                        0.1143250851485150,
                        0.1156502670851920,
                        0.1175013129411760,
                        0.1179270310695630,
                        0.1182087558274100,
                        0.1191784615553600,
                        0.1209157444943570,
                        0.1216799742574260,
                        0.1230600100148570,
                        0.1251525243466300,
                        0.1264191929573590,
                        0.1278286560939470,
                        0.1295155392232370,
                        0.1325001371944510,
                        0.1325402033842440,
                        0.1334973586771410,
                        0.1362069264544460,
                    ]
                ),
            ),
            DatasetLoader_ForestColors(): (
                "pine",
                370,
                "5",
                np.array(
                    [
                        0.010262410000000,
                        0.009839101400000,
                        0.012529907000000,
                        0.011030105000000,
                        0.010073634000000,
                        0.011320871000000,
                        0.011616203000000,
                        0.013212691000000,
                        0.012491421000000,
                        0.011912613000000,
                        0.013115942000000,
                        0.013417573000000,
                        0.013631902000000,
                        0.013967374000000,
                        0.014361868000000,
                        0.014427279000000,
                        0.014636329000000,
                        0.014908329000000,
                        0.014993297000000,
                        0.015136227000000,
                        0.015386547000000,
                        0.015711171000000,
                        0.015828966000000,
                        0.016981529000000,
                        0.018321589000000,
                        0.019439448000000,
                        0.021571993000000,
                        0.023876195000000,
                        0.025659029000000,
                        0.026894433000000,
                        0.028889134000000,
                        0.030469200000000,
                        0.030692223000000,
                        0.031212534000000,
                        0.030800426000000,
                        0.029837495000000,
                        0.029041031000000,
                        0.027807930000000,
                        0.027085866000000,
                        0.026870222000000,
                        0.026034403000000,
                        0.025490563000000,
                        0.025915747000000,
                        0.025255465000000,
                        0.024883133000000,
                        0.024609150000000,
                        0.023686946000000,
                        0.023991298000000,
                        0.023958765000000,
                        0.023967050000000,
                        0.023539582000000,
                        0.022725872000000,
                        0.022347244000000,
                        0.022138569000000,
                        0.021979660000000,
                        0.020823906000000,
                        0.021076211000000,
                        0.021165034000000,
                        0.022165784000000,
                        0.025146573000000,
                        0.029714434000000,
                        0.039837663000000,
                        0.052246223000000,
                        0.067425578000000,
                        0.083176671000000,
                        0.097080232000000,
                        0.111191460000000,
                        0.122961630000000,
                        0.134962030000000,
                        0.143059710000000,
                        0.149133660000000,
                        0.155173970000000,
                        0.155457870000000,
                        0.159591120000000,
                        0.164270350000000,
                        0.165211360000000,
                        0.167401470000000,
                        0.167736380000000,
                        0.169301000000000,
                        0.170914620000000,
                        0.171809910000000,
                        0.172325160000000,
                        0.174672460000000,
                        0.176431510000000,
                        0.174736990000000,
                        0.177491730000000,
                        0.176703620000000,
                        0.177523560000000,
                        0.182620180000000,
                        0.182529490000000,
                        0.183265810000000,
                        0.183518600000000,
                        0.186661620000000,
                    ]
                ),
            ),
            DatasetLoader_PaperSpectra(): (
                "newsprintsce",
                36,
                "5",
                np.array(
                    [
                        28.430000000000000,
                        37.390000000000000,
                        44.860000000000000,
                        48.860000000000000,
                        51.120000000000000,
                        52.330000000000000,
                        53.140000000000000,
                        53.930000000000000,
                        54.620000000000000,
                        55.090000000000000,
                        54.890000000000000,
                        53.670000000000000,
                        51.830000000000000,
                        50.610000000000000,
                        48.660000000000000,
                        45.180000000000000,
                        43.640000000000000,
                        48.450000000000000,
                        58.400000000000000,
                        67.180000000000000,
                        69.940000000000000,
                        69.630000000000000,
                        69.300000000000000,
                        69.340000000000000,
                        69.370000000000000,
                        69.190000000000000,
                        68.880000000000000,
                        68.610000000000000,
                        68.290000000000000,
                        68.250000000000000,
                        68.230000000000000,
                    ]
                ),
            ),
            DatasetLoader_LumberSpectra(): (
                "birchWp",
                12,
                "5",
                np.array(
                    [
                        0.044233333000000,
                        0.045133333000000,
                        0.045233333000000,
                        0.046333333000000,
                        0.046833333000000,
                        0.047633333000000,
                        0.048733333000000,
                        0.049633333000000,
                        0.049933333000000,
                        0.051733333000000,
                        0.052733333000000,
                        0.053133333000000,
                        0.053833333000000,
                        0.054633333000000,
                        0.055433333000000,
                        0.056333333000000,
                        0.056833333000000,
                        0.058033333000000,
                        0.058433333000000,
                        0.059633333000000,
                        0.059933333000000,
                        0.060433333000000,
                        0.061033333000000,
                        0.063233333000000,
                        0.063833333000000,
                        0.064133333000000,
                        0.064133333000000,
                        0.065533333000000,
                        0.066533333000000,
                        0.067033333000000,
                        0.067833333000000,
                        0.068233333000000,
                        0.068633333000000,
                        0.069933333000000,
                        0.070033333000000,
                        0.071533333000000,
                        0.071933333000000,
                        0.072433333000000,
                        0.072933333000000,
                        0.073833333000000,
                        0.074433333000000,
                        0.074933333000000,
                        0.075833333000000,
                        0.076233333000000,
                        0.076833333000000,
                        0.077233333000000,
                        0.077933333000000,
                        0.078133333000000,
                        0.078133333000000,
                        0.079933333000000,
                        0.080333333000000,
                        0.080833333000000,
                        0.081333333000000,
                        0.081633333000000,
                        0.082433333000000,
                        0.083733333000000,
                        0.083833333000000,
                        0.084233333000000,
                        0.085033333000000,
                        0.085733333000000,
                        0.085733333000000,
                        0.086333333000000,
                        0.086733333000000,
                        0.087433333000000,
                        0.088133333000000,
                        0.089033333000000,
                        0.089433333000000,
                        0.089733333000000,
                        0.090033333000000,
                        0.090333333000000,
                        0.090833333000000,
                        0.091533333000000,
                        0.092233333000000,
                        0.092633333000000,
                        0.092833333000000,
                        0.093333333000000,
                        0.094133333000000,
                        0.094833333000000,
                        0.095133333000000,
                        0.095833333000000,
                        0.096233333000000,
                        0.097133333000000,
                        0.096833333000000,
                        0.097733333000000,
                        0.098133333000000,
                        0.098933333000000,
                        0.099233333000000,
                        0.099633333000000,
                        0.100333330000000,
                        0.101433330000000,
                        0.101933330000000,
                        0.102533330000000,
                        0.102933330000000,
                        0.103633330000000,
                        0.103533330000000,
                        0.104533330000000,
                        0.104833330000000,
                        0.105833330000000,
                        0.106133330000000,
                        0.106933330000000,
                        0.106733330000000,
                        0.107733330000000,
                        0.108033330000000,
                        0.108133330000000,
                        0.108533330000000,
                        0.109633330000000,
                        0.109833330000000,
                        0.110533330000000,
                        0.111133330000000,
                        0.111633330000000,
                        0.111533330000000,
                        0.111833330000000,
                        0.113033330000000,
                        0.112833330000000,
                        0.113333330000000,
                        0.114033330000000,
                        0.114333330000000,
                        0.115233330000000,
                        0.116033330000000,
                        0.116433330000000,
                        0.116933330000000,
                        0.117333330000000,
                        0.117733330000000,
                        0.118633330000000,
                        0.118933330000000,
                        0.119633330000000,
                        0.119833330000000,
                        0.120733330000000,
                        0.121233330000000,
                        0.121833330000000,
                        0.122333330000000,
                        0.123133330000000,
                        0.123633330000000,
                        0.124133330000000,
                        0.124433330000000,
                        0.125233330000000,
                        0.125533330000000,
                        0.126033330000000,
                        0.126633330000000,
                        0.127033330000000,
                        0.127533330000000,
                        0.128033330000000,
                        0.128033330000000,
                        0.128833330000000,
                        0.129233330000000,
                        0.129433330000000,
                        0.130233330000000,
                        0.130833330000000,
                        0.130933330000000,
                        0.131833330000000,
                        0.132033330000000,
                        0.132433330000000,
                        0.133233330000000,
                        0.134233330000000,
                        0.134133330000000,
                        0.134533330000000,
                        0.135033330000000,
                        0.135433330000000,
                        0.136133330000000,
                        0.136033330000000,
                        0.136933330000000,
                        0.137733330000000,
                        0.138333330000000,
                        0.138533330000000,
                        0.139133330000000,
                        0.139633330000000,
                        0.139933330000000,
                        0.140133330000000,
                        0.140633330000000,
                        0.141433330000000,
                        0.141633330000000,
                        0.142433330000000,
                        0.142733330000000,
                        0.143933330000000,
                        0.143633330000000,
                        0.144233330000000,
                        0.144533330000000,
                        0.145333330000000,
                        0.145233330000000,
                        0.145933330000000,
                        0.146233330000000,
                        0.147133330000000,
                        0.147233330000000,
                        0.147533330000000,
                        0.148133330000000,
                        0.148733330000000,
                        0.148933330000000,
                        0.149533330000000,
                        0.149933330000000,
                        0.150733330000000,
                        0.151333330000000,
                        0.151633330000000,
                        0.152133330000000,
                        0.152033330000000,
                        0.152233330000000,
                        0.152333330000000,
                        0.153233330000000,
                        0.153833330000000,
                        0.154433330000000,
                        0.154333330000000,
                        0.154633330000000,
                        0.155433330000000,
                        0.155433330000000,
                        0.155333330000000,
                        0.155833330000000,
                        0.156833330000000,
                        0.157433330000000,
                        0.158033330000000,
                        0.158533330000000,
                        0.158933330000000,
                        0.158833330000000,
                        0.158533330000000,
                        0.158533330000000,
                        0.160633330000000,
                        0.161133330000000,
                        0.160933330000000,
                        0.161633330000000,
                        0.162033330000000,
                        0.162333330000000,
                        0.163033330000000,
                        0.163333330000000,
                        0.163433330000000,
                        0.163833330000000,
                        0.163933330000000,
                        0.164333330000000,
                        0.165433330000000,
                        0.165733330000000,
                        0.166033330000000,
                        0.166333330000000,
                        0.166433330000000,
                        0.166533330000000,
                        0.167833330000000,
                        0.167933330000000,
                        0.167733330000000,
                        0.168233330000000,
                        0.168333330000000,
                        0.168533330000000,
                        0.169333330000000,
                        0.169533330000000,
                        0.170333330000000,
                        0.170033330000000,
                        0.171033330000000,
                        0.170433330000000,
                        0.171233330000000,
                        0.171533330000000,
                        0.172233330000000,
                        0.172133330000000,
                        0.172233330000000,
                        0.172733330000000,
                        0.173533330000000,
                        0.174033330000000,
                        0.174133330000000,
                        0.175033330000000,
                        0.175433330000000,
                        0.175733330000000,
                        0.176133330000000,
                        0.175733330000000,
                        0.175833330000000,
                        0.175733330000000,
                        0.176833330000000,
                        0.176733330000000,
                        0.177033330000000,
                        0.176933330000000,
                        0.177233330000000,
                        0.178233330000000,
                        0.178933330000000,
                        0.178533330000000,
                        0.180033330000000,
                        0.180233330000000,
                        0.180633330000000,
                        0.180633330000000,
                        0.181433330000000,
                        0.180433330000000,
                        0.180833330000000,
                        0.181233330000000,
                        0.181033330000000,
                        0.181233330000000,
                        0.182333330000000,
                        0.181833330000000,
                        0.182133330000000,
                        0.183333330000000,
                        0.182333330000000,
                        0.182633330000000,
                        0.183533330000000,
                        0.183833330000000,
                        0.183933330000000,
                        0.183433330000000,
                        0.184733330000000,
                        0.184633330000000,
                        0.185033330000000,
                        0.185433330000000,
                        0.186033330000000,
                        0.185833330000000,
                        0.186833330000000,
                        0.185733330000000,
                        0.186433330000000,
                        0.187033330000000,
                        0.187333330000000,
                        0.187433330000000,
                        0.187833330000000,
                        0.187433330000000,
                        0.186333330000000,
                        0.186933330000000,
                        0.188433330000000,
                        0.188433330000000,
                        0.188833330000000,
                        0.189333330000000,
                        0.190133330000000,
                        0.189633330000000,
                        0.190433330000000,
                        0.190133330000000,
                        0.190733330000000,
                        0.190033330000000,
                        0.189933330000000,
                        0.190433330000000,
                        0.190433330000000,
                        0.190933330000000,
                        0.191633330000000,
                        0.191833330000000,
                        0.191933330000000,
                        0.191733330000000,
                        0.191233330000000,
                        0.192333330000000,
                        0.192833330000000,
                        0.193233330000000,
                        0.193633330000000,
                        0.193633330000000,
                        0.193033330000000,
                        0.192933330000000,
                        0.192833330000000,
                        0.193533330000000,
                        0.193433330000000,
                        0.193733330000000,
                        0.193833330000000,
                        0.194333330000000,
                        0.194033330000000,
                        0.195133330000000,
                        0.195033330000000,
                        0.194933330000000,
                        0.196233330000000,
                        0.197033330000000,
                        0.196833330000000,
                        0.197333330000000,
                        0.195533330000000,
                        0.195733330000000,
                        0.197233330000000,
                        0.198333330000000,
                        0.196433330000000,
                        0.197233330000000,
                        0.196833330000000,
                        0.197433330000000,
                        0.197033330000000,
                        0.196833330000000,
                        0.198433330000000,
                        0.198233330000000,
                        0.198233330000000,
                        0.198533330000000,
                        0.198233330000000,
                        0.197833330000000,
                        0.199133330000000,
                        0.199233330000000,
                        0.199333330000000,
                        0.199433330000000,
                        0.200133330000000,
                        0.200133330000000,
                        0.200533330000000,
                        0.199433330000000,
                        0.200633330000000,
                        0.200633330000000,
                        0.200233330000000,
                        0.199833330000000,
                        0.200133330000000,
                        0.201433330000000,
                        0.202233330000000,
                        0.201333330000000,
                        0.201233330000000,
                        0.201433330000000,
                        0.201833330000000,
                        0.201533330000000,
                        0.203233330000000,
                        0.202333330000000,
                        0.201433330000000,
                        0.203333330000000,
                        0.202733330000000,
                        0.202533330000000,
                        0.202633330000000,
                        0.203533330000000,
                        0.203433330000000,
                        0.202633330000000,
                        0.203133330000000,
                        0.203233330000000,
                        0.204533330000000,
                        0.204533330000000,
                        0.203533330000000,
                        0.203133330000000,
                        0.202633330000000,
                        0.203133330000000,
                        0.204433330000000,
                        0.205033330000000,
                        0.205533330000000,
                        0.204733330000000,
                        0.206333330000000,
                        0.205633330000000,
                        0.207733330000000,
                        0.207133330000000,
                        0.207233330000000,
                        0.206933330000000,
                        0.206833330000000,
                        0.209133330000000,
                        0.207533330000000,
                        0.207733330000000,
                        0.208333330000000,
                        0.208333330000000,
                        0.206133330000000,
                        0.207433330000000,
                        0.209033330000000,
                        0.209233330000000,
                        0.208633330000000,
                        0.207733330000000,
                        0.210233330000000,
                        0.209633330000000,
                        0.208833330000000,
                        0.210233330000000,
                        0.209633330000000,
                        0.210133330000000,
                        0.211033330000000,
                        0.210733330000000,
                        0.210133330000000,
                        0.210533330000000,
                        0.208633330000000,
                        0.209033330000000,
                        0.209733330000000,
                        0.210533330000000,
                        0.210033330000000,
                        0.208433330000000,
                        0.210433330000000,
                        0.210933330000000,
                        0.209633330000000,
                        0.210233330000000,
                        0.212233330000000,
                        0.212433330000000,
                        0.211433330000000,
                        0.212133330000000,
                        0.212733330000000,
                        0.211533330000000,
                        0.212033330000000,
                        0.211333330000000,
                        0.209733330000000,
                        0.210433330000000,
                        0.211233330000000,
                        0.212533330000000,
                        0.211533330000000,
                        0.211733330000000,
                        0.210133330000000,
                        0.210033330000000,
                        0.210833330000000,
                        0.211333330000000,
                        0.211233330000000,
                        0.213733330000000,
                        0.211133330000000,
                        0.211533330000000,
                        0.214833330000000,
                        0.211433330000000,
                        0.214633330000000,
                        0.214433330000000,
                        0.214833330000000,
                        0.216733330000000,
                        0.215833330000000,
                        0.214833330000000,
                        0.219333330000000,
                        0.216833330000000,
                        0.215333330000000,
                        0.215433330000000,
                        0.217633330000000,
                        0.216033330000000,
                        0.215233330000000,
                        0.217533330000000,
                        0.216933330000000,
                        0.215733330000000,
                        0.209633330000000,
                        0.209633330000000,
                        0.216766670000000,
                        0.217466670000000,
                        0.215466670000000,
                        0.215566670000000,
                        0.214766670000000,
                        0.213066670000000,
                        0.212366670000000,
                        0.212866670000000,
                        0.213166670000000,
                        0.211066670000000,
                        0.212366670000000,
                        0.213066670000000,
                        0.211666670000000,
                        0.209966670000000,
                        0.209366670000000,
                        0.210766670000000,
                        0.210066670000000,
                        0.210666670000000,
                        0.211766670000000,
                        0.208966670000000,
                        0.208266670000000,
                        0.210366670000000,
                        0.210866670000000,
                        0.209366670000000,
                        0.208966670000000,
                        0.209966670000000,
                        0.208166670000000,
                        0.207166670000000,
                        0.208766670000000,
                        0.208566670000000,
                        0.207566670000000,
                        0.205666670000000,
                        0.206166670000000,
                        0.206366670000000,
                        0.206166670000000,
                        0.206166670000000,
                        0.205766670000000,
                        0.204866670000000,
                        0.206066670000000,
                        0.205466670000000,
                        0.205066670000000,
                        0.204566670000000,
                        0.204266670000000,
                        0.204366670000000,
                        0.203666670000000,
                        0.203366670000000,
                        0.202066670000000,
                        0.202266670000000,
                        0.203866670000000,
                        0.203166670000000,
                        0.202866670000000,
                        0.201966670000000,
                        0.201166670000000,
                        0.201266670000000,
                        0.201266670000000,
                        0.200966670000000,
                        0.200766670000000,
                        0.200766670000000,
                        0.201266670000000,
                        0.200766670000000,
                        0.200066670000000,
                        0.199766670000000,
                        0.199366670000000,
                        0.199366670000000,
                        0.199466670000000,
                        0.199066670000000,
                        0.198466670000000,
                        0.198366670000000,
                        0.198466670000000,
                        0.198266670000000,
                        0.197966670000000,
                        0.198066670000000,
                        0.197266670000000,
                        0.196866670000000,
                        0.196566670000000,
                        0.196666670000000,
                        0.196266670000000,
                        0.195366670000000,
                        0.195366670000000,
                        0.195166670000000,
                        0.194066670000000,
                        0.193666670000000,
                        0.193266670000000,
                        0.193066670000000,
                        0.192266670000000,
                        0.192066670000000,
                        0.191766670000000,
                        0.190966670000000,
                        0.190666670000000,
                        0.190066670000000,
                        0.190066670000000,
                        0.190266670000000,
                        0.190366670000000,
                        0.190766670000000,
                        0.190866670000000,
                        0.190866670000000,
                        0.190966670000000,
                        0.190866670000000,
                        0.191166670000000,
                        0.191266670000000,
                        0.191366670000000,
                        0.191566670000000,
                        0.191766670000000,
                        0.191466670000000,
                        0.191766670000000,
                        0.191966670000000,
                        0.192166670000000,
                        0.191766670000000,
                        0.192366670000000,
                        0.192166670000000,
                        0.192266670000000,
                        0.192266670000000,
                        0.191966670000000,
                        0.191666670000000,
                        0.191966670000000,
                        0.191666670000000,
                        0.191466670000000,
                        0.191766670000000,
                        0.192266670000000,
                        0.191866670000000,
                        0.191866670000000,
                        0.191866670000000,
                        0.191966670000000,
                        0.191666670000000,
                        0.191266670000000,
                        0.191466670000000,
                        0.191566670000000,
                        0.191866670000000,
                        0.192566670000000,
                        0.192366670000000,
                        0.191966670000000,
                        0.192066670000000,
                        0.192366670000000,
                        0.192166670000000,
                        0.192266670000000,
                        0.192566670000000,
                        0.192866670000000,
                        0.192466670000000,
                        0.192966670000000,
                        0.192966670000000,
                        0.192966670000000,
                        0.192766670000000,
                        0.193066670000000,
                        0.193266670000000,
                        0.193066670000000,
                        0.193066670000000,
                        0.193366670000000,
                        0.192866670000000,
                        0.193366670000000,
                        0.193666670000000,
                        0.193966670000000,
                        0.193866670000000,
                        0.193566670000000,
                        0.193866670000000,
                        0.193566670000000,
                        0.193666670000000,
                        0.193966670000000,
                        0.194166670000000,
                        0.194366670000000,
                        0.194266670000000,
                        0.194066670000000,
                        0.194166670000000,
                        0.194266670000000,
                        0.194466670000000,
                        0.194466670000000,
                        0.194566670000000,
                        0.194866670000000,
                        0.194966670000000,
                        0.194866670000000,
                        0.194566670000000,
                        0.194466670000000,
                        0.194866670000000,
                        0.195166670000000,
                        0.195166670000000,
                        0.195066670000000,
                        0.195366670000000,
                        0.195566670000000,
                        0.195466670000000,
                        0.195766670000000,
                        0.195466670000000,
                        0.195466670000000,
                        0.195766670000000,
                        0.195466670000000,
                        0.195266670000000,
                        0.195566670000000,
                        0.195666670000000,
                        0.195666670000000,
                        0.195666670000000,
                        0.196366670000000,
                        0.196066670000000,
                        0.195766670000000,
                        0.195666670000000,
                        0.195966670000000,
                        0.195866670000000,
                        0.195866670000000,
                        0.196066670000000,
                        0.196566670000000,
                        0.196166670000000,
                        0.196666670000000,
                        0.196366670000000,
                        0.196466670000000,
                        0.196266670000000,
                        0.196066670000000,
                        0.196066670000000,
                        0.196366670000000,
                        0.196466670000000,
                        0.196466670000000,
                        0.196766670000000,
                        0.196866670000000,
                        0.196466670000000,
                        0.196866670000000,
                        0.196666670000000,
                        0.196066670000000,
                        0.196166670000000,
                        0.196666670000000,
                        0.196666670000000,
                        0.196666670000000,
                        0.197066670000000,
                        0.197366670000000,
                        0.197066670000000,
                        0.197166670000000,
                        0.197166670000000,
                        0.197366670000000,
                        0.197166670000000,
                        0.197066670000000,
                        0.197066670000000,
                        0.196766670000000,
                        0.197166670000000,
                        0.197266670000000,
                        0.196966670000000,
                        0.196966670000000,
                        0.197466670000000,
                        0.197066670000000,
                        0.196766670000000,
                        0.196966670000000,
                        0.197666670000000,
                        0.197066670000000,
                        0.196866670000000,
                        0.197166670000000,
                        0.197166670000000,
                        0.197366670000000,
                        0.197566670000000,
                        0.197466670000000,
                        0.197366670000000,
                        0.197366670000000,
                        0.197366670000000,
                        0.197266670000000,
                        0.196566670000000,
                        0.197266670000000,
                        0.197466670000000,
                        0.197066670000000,
                        0.196866670000000,
                        0.197066670000000,
                        0.196766670000000,
                        0.196966670000000,
                        0.197166670000000,
                        0.197366670000000,
                        0.196866670000000,
                        0.196966670000000,
                        0.196766670000000,
                        0.196466670000000,
                        0.195966670000000,
                        0.195666670000000,
                        0.195966670000000,
                        0.196066670000000,
                        0.195666670000000,
                        0.195366670000000,
                        0.195066670000000,
                        0.194966670000000,
                        0.194666670000000,
                        0.194566670000000,
                        0.194766670000000,
                        0.194466670000000,
                        0.194166670000000,
                        0.193866670000000,
                        0.193566670000000,
                        0.193366670000000,
                        0.193466670000000,
                        0.193866670000000,
                        0.193066670000000,
                        0.192866670000000,
                        0.192666670000000,
                        0.192366670000000,
                        0.192066670000000,
                        0.191966670000000,
                        0.191566670000000,
                        0.190966670000000,
                        0.190666670000000,
                        0.190666670000000,
                        0.190366670000000,
                        0.190266670000000,
                        0.190266670000000,
                        0.189866670000000,
                        0.189366670000000,
                        0.189066670000000,
                        0.189066670000000,
                        0.188466670000000,
                        0.188066670000000,
                        0.188166670000000,
                        0.187966670000000,
                        0.187466670000000,
                        0.187266670000000,
                        0.187266670000000,
                        0.187066670000000,
                        0.186766670000000,
                        0.186666670000000,
                        0.186666670000000,
                        0.186166670000000,
                        0.186466670000000,
                        0.186266670000000,
                        0.185966670000000,
                        0.185766670000000,
                        0.185766670000000,
                        0.185566670000000,
                        0.185166670000000,
                        0.184866670000000,
                        0.184966670000000,
                        0.185066670000000,
                        0.185166670000000,
                        0.184966670000000,
                        0.184466670000000,
                        0.184366670000000,
                        0.183866670000000,
                        0.183666670000000,
                        0.183666670000000,
                        0.183366670000000,
                        0.183066670000000,
                        0.183066670000000,
                        0.182166670000000,
                        0.180366670000000,
                        0.180166670000000,
                        0.180066670000000,
                        0.179766670000000,
                        0.179966670000000,
                        0.180066670000000,
                        0.179766670000000,
                        0.179566670000000,
                        0.179466670000000,
                        0.179766670000000,
                        0.179566670000000,
                        0.179466670000000,
                        0.179466670000000,
                        0.179466670000000,
                        0.179666670000000,
                        0.179566670000000,
                        0.179566670000000,
                        0.179366670000000,
                        0.179766670000000,
                        0.180166670000000,
                        0.179466670000000,
                        0.179466670000000,
                        0.179566670000000,
                        0.179466670000000,
                        0.179266670000000,
                        0.179466670000000,
                        0.179466670000000,
                        0.179766670000000,
                        0.179966670000000,
                        0.180266670000000,
                        0.180466670000000,
                        0.179766670000000,
                        0.180066670000000,
                        0.180266670000000,
                        0.179966670000000,
                        0.180166670000000,
                        0.180766670000000,
                        0.180666670000000,
                        0.180766670000000,
                        0.181066670000000,
                        0.180766670000000,
                        0.180766670000000,
                        0.181066670000000,
                        0.181366670000000,
                        0.181066670000000,
                        0.181266670000000,
                        0.181566670000000,
                        0.181566670000000,
                        0.181566670000000,
                        0.182066670000000,
                        0.182166670000000,
                        0.182066670000000,
                        0.182066670000000,
                        0.182066670000000,
                        0.182366670000000,
                        0.182266670000000,
                        0.182566670000000,
                        0.182566670000000,
                        0.182466670000000,
                        0.182966670000000,
                        0.182966670000000,
                        0.183166670000000,
                        0.182966670000000,
                        0.182366670000000,
                        0.182566670000000,
                        0.182966670000000,
                        0.183366670000000,
                        0.183366670000000,
                        0.183266670000000,
                        0.183166670000000,
                        0.183166670000000,
                        0.183566670000000,
                        0.183666670000000,
                        0.183466670000000,
                        0.183566670000000,
                        0.183566670000000,
                        0.183266670000000,
                        0.183466670000000,
                        0.184166670000000,
                        0.184366670000000,
                        0.183966670000000,
                        0.184066670000000,
                        0.184266670000000,
                        0.183866670000000,
                        0.183466670000000,
                        0.183666670000000,
                        0.183766670000000,
                        0.183866670000000,
                        0.183966670000000,
                        0.184266670000000,
                        0.184066670000000,
                        0.184166670000000,
                        0.184466670000000,
                        0.184366670000000,
                        0.184366670000000,
                        0.184866670000000,
                        0.185066670000000,
                        0.184866670000000,
                        0.184666670000000,
                        0.185166670000000,
                        0.185266670000000,
                        0.185566670000000,
                        0.185466670000000,
                        0.185266670000000,
                        0.185166670000000,
                        0.184966670000000,
                        0.185066670000000,
                        0.185366670000000,
                        0.185166670000000,
                        0.185366670000000,
                        0.185766670000000,
                        0.185666670000000,
                        0.185666670000000,
                        0.185366670000000,
                        0.185466670000000,
                        0.185066670000000,
                        0.184666670000000,
                        0.184666670000000,
                        0.184766670000000,
                        0.185366670000000,
                        0.185166670000000,
                        0.185366670000000,
                        0.185166670000000,
                        0.184866670000000,
                        0.184866670000000,
                        0.184566670000000,
                        0.184466670000000,
                        0.184566670000000,
                        0.184866670000000,
                        0.184666670000000,
                        0.184466670000000,
                        0.184366670000000,
                        0.184166670000000,
                        0.183466670000000,
                        0.183666670000000,
                        0.183866670000000,
                        0.183366670000000,
                        0.182766670000000,
                        0.182866670000000,
                        0.183266670000000,
                        0.182866670000000,
                        0.182966670000000,
                        0.182766670000000,
                        0.181966670000000,
                        0.181666670000000,
                        0.181266670000000,
                        0.180866670000000,
                        0.180466670000000,
                        0.180366670000000,
                        0.180666670000000,
                        0.180266670000000,
                        0.179366670000000,
                        0.179266670000000,
                        0.179066670000000,
                        0.178666670000000,
                        0.178466670000000,
                        0.178366670000000,
                        0.177966670000000,
                        0.177566670000000,
                        0.177766670000000,
                        0.177166670000000,
                        0.176866670000000,
                        0.176266670000000,
                        0.175666670000000,
                        0.175466670000000,
                        0.174866670000000,
                        0.174466670000000,
                        0.174166670000000,
                        0.173966670000000,
                        0.174366670000000,
                        0.174266670000000,
                        0.173766670000000,
                        0.173466670000000,
                        0.173166670000000,
                        0.173266670000000,
                        0.172266670000000,
                        0.171866670000000,
                        0.171566670000000,
                        0.171266670000000,
                        0.170766670000000,
                        0.170366670000000,
                        0.169566670000000,
                        0.169466670000000,
                        0.169166670000000,
                        0.169666670000000,
                        0.169666670000000,
                        0.169366670000000,
                        0.169366670000000,
                        0.169566670000000,
                        0.169766670000000,
                        0.169566670000000,
                        0.169466670000000,
                        0.169366670000000,
                        0.168166670000000,
                        0.167566670000000,
                        0.166866670000000,
                        0.167066670000000,
                        0.166666670000000,
                        0.166066670000000,
                        0.166266670000000,
                        0.165766670000000,
                        0.165566670000000,
                        0.165566670000000,
                        0.165166670000000,
                        0.164566670000000,
                        0.164166670000000,
                        0.163566670000000,
                        0.162466670000000,
                        0.161766670000000,
                        0.161866670000000,
                        0.160966670000000,
                        0.160266670000000,
                        0.159866670000000,
                        0.159566670000000,
                        0.159166670000000,
                        0.158166670000000,
                        0.157666670000000,
                        0.157066670000000,
                        0.156266670000000,
                        0.155466670000000,
                        0.154566670000000,
                        0.153766670000000,
                        0.153066670000000,
                        0.152066670000000,
                        0.151666670000000,
                        0.150666670000000,
                        0.150066670000000,
                        0.149966670000000,
                        0.149566670000000,
                        0.148566670000000,
                        0.148066670000000,
                        0.147766670000000,
                        0.147266670000000,
                        0.146266670000000,
                        0.146266670000000,
                        0.145466670000000,
                        0.144966670000000,
                        0.144466670000000,
                        0.144366670000000,
                        0.144366670000000,
                        0.143666670000000,
                        0.143466670000000,
                        0.143366670000000,
                        0.142966670000000,
                        0.142866670000000,
                        0.142166670000000,
                        0.142066670000000,
                        0.142266670000000,
                        0.142066670000000,
                        0.141966670000000,
                        0.141666670000000,
                        0.141366670000000,
                        0.141466670000000,
                        0.141366670000000,
                        0.140866670000000,
                        0.140966670000000,
                        0.141366670000000,
                        0.141166670000000,
                        0.141166670000000,
                        0.141366670000000,
                        0.141266670000000,
                        0.140966670000000,
                        0.140866670000000,
                        0.141066670000000,
                        0.141066670000000,
                        0.140866670000000,
                        0.141166670000000,
                        0.140866670000000,
                        0.140766670000000,
                        0.141366670000000,
                        0.141266670000000,
                        0.140866670000000,
                        0.140866670000000,
                        0.140966670000000,
                        0.140766670000000,
                        0.140466670000000,
                        0.140466670000000,
                        0.140566670000000,
                        0.140566670000000,
                        0.140966670000000,
                        0.140666670000000,
                        0.140466670000000,
                        0.140266670000000,
                        0.140166670000000,
                        0.140366670000000,
                        0.140266670000000,
                        0.140466670000000,
                        0.140566670000000,
                        0.140966670000000,
                        0.141466670000000,
                        0.141066670000000,
                        0.141366670000000,
                        0.141166670000000,
                        0.141366670000000,
                        0.141766670000000,
                        0.141666670000000,
                        0.141466670000000,
                        0.141666670000000,
                        0.141966670000000,
                        0.142266670000000,
                        0.141866670000000,
                        0.141666670000000,
                        0.142066670000000,
                        0.142266670000000,
                        0.142266670000000,
                        0.142566670000000,
                        0.142666670000000,
                        0.142766670000000,
                        0.143166670000000,
                        0.143266670000000,
                        0.143266670000000,
                        0.143066670000000,
                        0.143366670000000,
                        0.143566670000000,
                        0.143666670000000,
                        0.143866670000000,
                        0.144066670000000,
                        0.144166670000000,
                        0.143866670000000,
                        0.144666670000000,
                        0.144666670000000,
                        0.144666670000000,
                        0.144666670000000,
                        0.144866670000000,
                        0.145066670000000,
                        0.145166670000000,
                        0.145266670000000,
                        0.145566670000000,
                        0.145666670000000,
                        0.146166670000000,
                        0.146266670000000,
                        0.145666670000000,
                        0.145866670000000,
                        0.146366670000000,
                        0.146366670000000,
                        0.146066670000000,
                        0.145966670000000,
                        0.145866670000000,
                        0.146066670000000,
                        0.146866670000000,
                        0.146966670000000,
                        0.146666670000000,
                        0.146666670000000,
                        0.146766670000000,
                        0.146966670000000,
                        0.146766670000000,
                        0.146666670000000,
                        0.146766670000000,
                        0.146666670000000,
                        0.147166670000000,
                        0.147166670000000,
                        0.147066670000000,
                        0.147166670000000,
                        0.146966670000000,
                        0.146866670000000,
                        0.147166670000000,
                        0.147166670000000,
                        0.147066670000000,
                        0.147266670000000,
                        0.147866670000000,
                        0.147666670000000,
                        0.147066670000000,
                        0.147566670000000,
                        0.147366670000000,
                        0.147766670000000,
                        0.147566670000000,
                        0.147466670000000,
                        0.147766670000000,
                        0.147966670000000,
                        0.147966670000000,
                        0.147666670000000,
                        0.147966670000000,
                        0.148366670000000,
                        0.148166670000000,
                        0.148166670000000,
                        0.148366670000000,
                        0.148866670000000,
                        0.148566670000000,
                        0.148666670000000,
                        0.148666670000000,
                        0.148766670000000,
                        0.149066670000000,
                        0.148866670000000,
                        0.148866670000000,
                        0.148966670000000,
                        0.148866670000000,
                        0.148866670000000,
                        0.149066670000000,
                        0.148966670000000,
                        0.149066670000000,
                        0.149366670000000,
                        0.149966670000000,
                        0.149966670000000,
                        0.149766670000000,
                        0.149966670000000,
                        0.149966670000000,
                        0.149866670000000,
                        0.149966670000000,
                        0.150166670000000,
                        0.150666670000000,
                        0.150266670000000,
                        0.150666670000000,
                        0.150866670000000,
                        0.151066670000000,
                        0.151166670000000,
                        0.150866670000000,
                        0.150866670000000,
                        0.151166670000000,
                        0.151666670000000,
                        0.152266670000000,
                        0.152066670000000,
                        0.151966670000000,
                        0.152266670000000,
                        0.152366670000000,
                        0.152666670000000,
                        0.152866670000000,
                        0.153266670000000,
                        0.153166670000000,
                        0.153166670000000,
                        0.153666670000000,
                        0.153266670000000,
                        0.153866670000000,
                        0.154266670000000,
                        0.154666670000000,
                        0.154566670000000,
                        0.154566670000000,
                        0.154766670000000,
                        0.154866670000000,
                        0.154266670000000,
                        0.154966670000000,
                        0.155266670000000,
                        0.155866670000000,
                        0.155766670000000,
                        0.156166670000000,
                        0.156266670000000,
                        0.156066670000000,
                        0.156266670000000,
                        0.156266670000000,
                        0.156266670000000,
                        0.156466670000000,
                        0.156566670000000,
                        0.156466670000000,
                        0.156166670000000,
                        0.156466670000000,
                        0.156966670000000,
                        0.156966670000000,
                        0.156966670000000,
                        0.157066670000000,
                        0.157266670000000,
                        0.157366670000000,
                        0.157366670000000,
                        0.157566670000000,
                        0.157366670000000,
                        0.157466670000000,
                        0.157766670000000,
                        0.157366670000000,
                        0.157166670000000,
                        0.157666670000000,
                        0.157366670000000,
                        0.157366670000000,
                        0.157266670000000,
                        0.157466670000000,
                        0.157166670000000,
                        0.156966670000000,
                        0.157066670000000,
                        0.156866670000000,
                        0.156766670000000,
                        0.156766670000000,
                        0.156966670000000,
                        0.156866670000000,
                        0.156766670000000,
                        0.156566670000000,
                        0.156466670000000,
                        0.156666670000000,
                        0.155966670000000,
                        0.155666670000000,
                        0.155966670000000,
                        0.155866670000000,
                        0.155566670000000,
                        0.155966670000000,
                        0.156866670000000,
                        0.156566670000000,
                        0.156466670000000,
                        0.156366670000000,
                        0.155766670000000,
                        0.155766670000000,
                        0.155666670000000,
                        0.155266670000000,
                        0.154866670000000,
                        0.155466670000000,
                        0.154866670000000,
                        0.154966670000000,
                        0.154966670000000,
                        0.154566670000000,
                        0.154566670000000,
                        0.153966670000000,
                        0.154066670000000,
                        0.154066670000000,
                        0.153966670000000,
                        0.154166670000000,
                        0.154066670000000,
                        0.153666670000000,
                        0.153666670000000,
                        0.153866670000000,
                        0.153566670000000,
                        0.153066670000000,
                        0.153066670000000,
                        0.153066670000000,
                        0.152666670000000,
                        0.152866670000000,
                        0.153066670000000,
                        0.153066670000000,
                        0.152766670000000,
                        0.152566670000000,
                        0.152466670000000,
                        0.152466670000000,
                        0.152666670000000,
                        0.152466670000000,
                        0.152266670000000,
                        0.152066670000000,
                        0.152366670000000,
                        0.152266670000000,
                        0.152166670000000,
                        0.151766670000000,
                        0.151666670000000,
                        0.151866670000000,
                        0.151966670000000,
                        0.151666670000000,
                        0.151566670000000,
                        0.151866670000000,
                        0.151366670000000,
                        0.151366670000000,
                        0.151466670000000,
                        0.151466670000000,
                        0.151466670000000,
                        0.151566670000000,
                        0.151466670000000,
                        0.151566670000000,
                        0.151266670000000,
                        0.151466670000000,
                        0.151166670000000,
                        0.151066670000000,
                        0.151566670000000,
                        0.151566670000000,
                        0.151766670000000,
                        0.152066670000000,
                        0.151866670000000,
                        0.151666670000000,
                        0.151766670000000,
                        0.151966670000000,
                        0.151766670000000,
                        0.151966670000000,
                        0.152366670000000,
                        0.152666670000000,
                        0.152566670000000,
                        0.152466670000000,
                        0.152566670000000,
                        0.152166670000000,
                        0.151766670000000,
                        0.152266670000000,
                        0.152266670000000,
                        0.151866670000000,
                        0.152066670000000,
                        0.152166670000000,
                        0.152266670000000,
                        0.152466670000000,
                        0.152166670000000,
                        0.152066670000000,
                        0.152066670000000,
                        0.152666670000000,
                        0.152666670000000,
                        0.152166670000000,
                        0.152066670000000,
                        0.151666670000000,
                        0.151566670000000,
                        0.150966670000000,
                        0.150366670000000,
                        0.150566670000000,
                        0.150366670000000,
                        0.150866670000000,
                        0.150766670000000,
                        0.150966670000000,
                        0.151266670000000,
                        0.150966670000000,
                        0.150966670000000,
                        0.150966670000000,
                        0.150766670000000,
                        0.151066670000000,
                        0.151266670000000,
                        0.151966670000000,
                        0.151966670000000,
                        0.151566670000000,
                        0.151666670000000,
                        0.151466670000000,
                        0.151966670000000,
                        0.152166670000000,
                        0.152066670000000,
                        0.152166670000000,
                        0.152266670000000,
                        0.152666670000000,
                        0.152266670000000,
                        0.151766670000000,
                        0.152166670000000,
                        0.152166670000000,
                        0.151866670000000,
                        0.152066670000000,
                        0.152166670000000,
                        0.152366670000000,
                        0.152666670000000,
                        0.153066670000000,
                        0.152766670000000,
                        0.152566670000000,
                        0.152466670000000,
                        0.152266670000000,
                        0.152366670000000,
                        0.152166670000000,
                        0.152466670000000,
                        0.152266670000000,
                        0.152066670000000,
                        0.153366670000000,
                        0.153166670000000,
                        0.153066670000000,
                        0.153166670000000,
                        0.152866670000000,
                        0.153066670000000,
                        0.153266670000000,
                        0.153166670000000,
                        0.153266670000000,
                        0.153266670000000,
                        0.153666670000000,
                        0.153566670000000,
                        0.154166670000000,
                        0.153366670000000,
                        0.152766670000000,
                        0.153166670000000,
                        0.153866670000000,
                        0.153566670000000,
                        0.153866670000000,
                        0.154166670000000,
                        0.154766670000000,
                        0.154666670000000,
                        0.154966670000000,
                        0.155166670000000,
                        0.155166670000000,
                        0.155366670000000,
                        0.155366670000000,
                        0.155466670000000,
                        0.155466670000000,
                        0.156166670000000,
                        0.156166670000000,
                        0.155866670000000,
                        0.155566670000000,
                        0.155466670000000,
                        0.155366670000000,
                        0.154966670000000,
                        0.154966670000000,
                        0.154866670000000,
                        0.154066670000000,
                        0.154366670000000,
                        0.155366670000000,
                        0.154466670000000,
                        0.153866670000000,
                        0.153866670000000,
                        0.153766670000000,
                        0.153566670000000,
                        0.153766670000000,
                        0.154266670000000,
                        0.154366670000000,
                        0.154366670000000,
                        0.154766670000000,
                        0.154966670000000,
                        0.154966670000000,
                        0.154666670000000,
                        0.155466670000000,
                        0.155666670000000,
                        0.156166670000000,
                        0.156466670000000,
                        0.156366670000000,
                        0.156166670000000,
                        0.156966670000000,
                        0.155966670000000,
                        0.154966670000000,
                        0.154466670000000,
                        0.152766670000000,
                        0.151866670000000,
                        0.151066670000000,
                        0.150066670000000,
                        0.148566670000000,
                        0.148066670000000,
                        0.147366670000000,
                        0.146166670000000,
                        0.145466670000000,
                        0.144266670000000,
                        0.143666670000000,
                        0.143766670000000,
                        0.143066670000000,
                        0.142366670000000,
                        0.141466670000000,
                        0.141666670000000,
                        0.141166670000000,
                        0.140166670000000,
                        0.139566670000000,
                        0.139266670000000,
                        0.138166670000000,
                        0.137666670000000,
                        0.136666670000000,
                        0.136166670000000,
                        0.134766670000000,
                        0.134066670000000,
                        0.132966670000000,
                        0.132166670000000,
                        0.131066670000000,
                        0.130366670000000,
                        0.129366670000000,
                        0.128366670000000,
                        0.127166670000000,
                        0.126666670000000,
                        0.124966670000000,
                        0.124066670000000,
                        0.123866670000000,
                        0.123266670000000,
                        0.121466670000000,
                        0.121966670000000,
                        0.121266670000000,
                        0.120666670000000,
                        0.120066670000000,
                        0.119766670000000,
                        0.118866670000000,
                        0.118466670000000,
                        0.118566670000000,
                        0.118966670000000,
                        0.118266670000000,
                        0.117466670000000,
                        0.118066670000000,
                        0.117666670000000,
                        0.117266670000000,
                        0.117966670000000,
                        0.118166670000000,
                        0.117666670000000,
                        0.117766670000000,
                        0.117766670000000,
                        0.117666670000000,
                        0.117466670000000,
                        0.117866670000000,
                        0.118366670000000,
                        0.118766670000000,
                        0.118366670000000,
                        0.118766670000000,
                        0.119166670000000,
                        0.119766670000000,
                        0.118866670000000,
                        0.118766670000000,
                        0.119166670000000,
                        0.119266670000000,
                        0.119366670000000,
                        0.119866670000000,
                        0.119966670000000,
                        0.120066670000000,
                        0.120566670000000,
                        0.120966670000000,
                        0.120666670000000,
                        0.120566670000000,
                        0.120566670000000,
                        0.120766670000000,
                        0.120766670000000,
                        0.121066670000000,
                        0.121066670000000,
                        0.120866670000000,
                        0.121166670000000,
                        0.121766670000000,
                        0.121466670000000,
                        0.121166670000000,
                        0.121466670000000,
                        0.121366670000000,
                        0.121566670000000,
                        0.121466670000000,
                        0.121466670000000,
                        0.121666670000000,
                        0.121766670000000,
                        0.122566670000000,
                        0.122566670000000,
                        0.122566670000000,
                        0.122966670000000,
                        0.123666670000000,
                        0.124266670000000,
                        0.124466670000000,
                        0.124866670000000,
                        0.125966670000000,
                        0.125966670000000,
                        0.127266670000000,
                        0.127666670000000,
                        0.128466670000000,
                        0.128366670000000,
                        0.128866670000000,
                        0.129066670000000,
                        0.129366670000000,
                        0.129366670000000,
                        0.129466670000000,
                        0.129766670000000,
                        0.130466670000000,
                        0.130466670000000,
                        0.130866670000000,
                        0.131066670000000,
                        0.131466670000000,
                        0.131866670000000,
                        0.132366670000000,
                        0.132266670000000,
                        0.132666670000000,
                        0.133166670000000,
                        0.133366670000000,
                        0.133166670000000,
                        0.133566670000000,
                        0.133866670000000,
                        0.133966670000000,
                        0.134166670000000,
                        0.134366670000000,
                        0.134266670000000,
                        0.134166670000000,
                        0.134266670000000,
                        0.135066670000000,
                        0.134766670000000,
                        0.134566670000000,
                        0.134466670000000,
                        0.134066670000000,
                        0.134066670000000,
                        0.133566670000000,
                        0.133266670000000,
                        0.133466670000000,
                        0.133266670000000,
                        0.133966670000000,
                        0.133666670000000,
                        0.133066670000000,
                        0.133466670000000,
                        0.133366670000000,
                        0.133266670000000,
                        0.133466670000000,
                        0.133466670000000,
                        0.133066670000000,
                        0.132866670000000,
                        0.132766670000000,
                        0.132366670000000,
                        0.132166670000000,
                        0.131966670000000,
                        0.131566670000000,
                        0.131866670000000,
                        0.131266670000000,
                        0.131066670000000,
                        0.130866670000000,
                        0.130766670000000,
                        0.130866670000000,
                        0.130466670000000,
                        0.129966670000000,
                        0.129866670000000,
                        0.129566670000000,
                        0.129666670000000,
                        0.129366670000000,
                        0.128866670000000,
                        0.128266670000000,
                        0.128366670000000,
                        0.128366670000000,
                        0.127766670000000,
                        0.127466670000000,
                        0.127166670000000,
                        0.126766670000000,
                        0.126666670000000,
                        0.126466670000000,
                        0.126466670000000,
                        0.126066670000000,
                        0.125866670000000,
                        0.125766670000000,
                        0.125366670000000,
                        0.125366670000000,
                        0.124766670000000,
                        0.124266670000000,
                        0.123866670000000,
                        0.123266670000000,
                        0.123566670000000,
                        0.123066670000000,
                        0.122766670000000,
                        0.122866670000000,
                        0.122666670000000,
                        0.122466670000000,
                        0.122366670000000,
                        0.122066670000000,
                        0.121866670000000,
                        0.121466670000000,
                        0.121566670000000,
                        0.121266670000000,
                        0.120766670000000,
                        0.121366670000000,
                        0.120966670000000,
                        0.120266670000000,
                        0.120266670000000,
                        0.120066670000000,
                        0.119766670000000,
                        0.120066670000000,
                        0.120266670000000,
                        0.119766670000000,
                        0.119366670000000,
                        0.119666670000000,
                        0.119366670000000,
                        0.119566670000000,
                        0.119266670000000,
                        0.118566670000000,
                        0.118466670000000,
                        0.119066670000000,
                        0.118766670000000,
                        0.118666670000000,
                        0.118666670000000,
                        0.119366670000000,
                        0.119166670000000,
                        0.119666670000000,
                        0.118866670000000,
                        0.118266670000000,
                        0.118666670000000,
                        0.119166670000000,
                        0.118866670000000,
                        0.118466670000000,
                        0.118566670000000,
                        0.119066670000000,
                        0.118166670000000,
                        0.119066670000000,
                        0.118866670000000,
                        0.118766670000000,
                        0.118666670000000,
                        0.118766670000000,
                        0.119466670000000,
                        0.118666670000000,
                        0.118766670000000,
                        0.119266670000000,
                        0.118566670000000,
                        0.118866670000000,
                        0.119166670000000,
                        0.118766670000000,
                        0.118866670000000,
                        0.118666670000000,
                        0.119366670000000,
                        0.119266670000000,
                        0.119166670000000,
                        0.119866670000000,
                        0.120166670000000,
                        0.119566670000000,
                        0.120166670000000,
                        0.120466670000000,
                        0.119966670000000,
                        0.120166670000000,
                        0.120166670000000,
                        0.120066670000000,
                        0.119166670000000,
                        0.120666670000000,
                        0.120466670000000,
                        0.120166670000000,
                        0.120266670000000,
                        0.119966670000000,
                        0.119866670000000,
                        0.120866670000000,
                        0.120566670000000,
                        0.120866670000000,
                        0.121366670000000,
                        0.121566670000000,
                        0.121466670000000,
                        0.121566670000000,
                        0.122166670000000,
                        0.123066670000000,
                        0.124166670000000,
                        0.123766670000000,
                        0.122766670000000,
                        0.123466670000000,
                        0.124066670000000,
                        0.125466670000000,
                        0.124666670000000,
                        0.124366670000000,
                        0.124266670000000,
                        0.124066670000000,
                        0.124366670000000,
                        0.124866670000000,
                        0.124266670000000,
                        0.124966670000000,
                        0.125366670000000,
                        0.125466670000000,
                        0.124766670000000,
                        0.124166670000000,
                        0.124366670000000,
                        0.124566670000000,
                        0.123966670000000,
                        0.124366670000000,
                        0.124166670000000,
                        0.124766670000000,
                        0.124866670000000,
                        0.125766670000000,
                        0.126066670000000,
                        0.125166670000000,
                        0.126466670000000,
                        0.126466670000000,
                        0.126266670000000,
                        0.127066670000000,
                        0.127766670000000,
                        0.127366670000000,
                        0.126366670000000,
                        0.128266670000000,
                        0.127966670000000,
                        0.127366670000000,
                        0.127666670000000,
                        0.128366670000000,
                        0.127566670000000,
                        0.126866670000000,
                        0.127266670000000,
                        0.128766670000000,
                        0.127966670000000,
                        0.129466670000000,
                        0.130066670000000,
                        0.129866670000000,
                        0.128666670000000,
                        0.128166670000000,
                        0.129366670000000,
                        0.128266670000000,
                        0.127366670000000,
                        0.129166670000000,
                        0.128166670000000,
                        0.130766670000000,
                        0.130766670000000,
                        0.130566670000000,
                        0.129566670000000,
                        0.128366670000000,
                        0.128366670000000,
                        0.128766670000000,
                        0.127366670000000,
                        0.127966670000000,
                        0.128066670000000,
                        0.129066670000000,
                        0.127766670000000,
                        0.127266670000000,
                        0.127966670000000,
                        0.129366670000000,
                        0.129166670000000,
                        0.128266670000000,
                        0.127666670000000,
                        0.125066670000000,
                        0.124566670000000,
                        0.126166670000000,
                        0.124966670000000,
                        0.125866670000000,
                        0.127566670000000,
                        0.125566670000000,
                        0.125466670000000,
                        0.122366670000000,
                        0.123766670000000,
                        0.121066670000000,
                        0.119666670000000,
                        0.122366670000000,
                        0.120966670000000,
                        0.119566670000000,
                        0.120766670000000,
                        0.119966670000000,
                        0.119666670000000,
                        0.118066670000000,
                        0.119066670000000,
                        0.118666670000000,
                        0.116166670000000,
                        0.117266670000000,
                        0.119666670000000,
                        0.118566670000000,
                        0.115766670000000,
                        0.115266670000000,
                        0.116666670000000,
                        0.116466670000000,
                        0.116066670000000,
                        0.112066670000000,
                        0.111066670000000,
                        0.112866670000000,
                        0.113366670000000,
                        0.114266670000000,
                        0.112766670000000,
                        0.112166670000000,
                        0.113766670000000,
                        0.110966670000000,
                        0.111066670000000,
                        0.111466670000000,
                        0.112766670000000,
                        0.112866670000000,
                        0.111966670000000,
                        0.110666670000000,
                        0.111066670000000,
                        0.113266670000000,
                        0.112366670000000,
                        0.110966670000000,
                        0.110166670000000,
                        0.110566670000000,
                        0.111666670000000,
                        0.113066670000000,
                        0.111166670000000,
                        0.112366670000000,
                        0.114466670000000,
                        0.112266670000000,
                        0.111066670000000,
                        0.111966670000000,
                        0.111466670000000,
                        0.110366670000000,
                        0.109466670000000,
                        0.114066670000000,
                        0.113466670000000,
                        0.113366670000000,
                        0.114566670000000,
                        0.113966670000000,
                        0.115766670000000,
                        0.113366670000000,
                        0.113366670000000,
                        0.111766670000000,
                        0.107366670000000,
                        0.111066670000000,
                        0.112666670000000,
                        0.110066670000000,
                        0.112066670000000,
                        0.113466670000000,
                        0.114266670000000,
                        0.113066670000000,
                        0.114066670000000,
                        0.107566670000000,
                        0.108066670000000,
                        0.116366670000000,
                        0.116666670000000,
                        0.115266670000000,
                        0.112266670000000,
                        0.114466670000000,
                        0.114066670000000,
                        0.113166670000000,
                        0.111466670000000,
                        0.109266670000000,
                        0.109466670000000,
                        0.111466670000000,
                        0.110066670000000,
                        0.111266670000000,
                        0.111166670000000,
                        0.111166670000000,
                        0.109866670000000,
                        0.110066670000000,
                        0.109966670000000,
                        0.106266670000000,
                        0.107566670000000,
                        0.111766670000000,
                        0.112066670000000,
                        0.111866670000000,
                        0.110366670000000,
                        0.107466670000000,
                        0.107366670000000,
                        0.111966670000000,
                        0.108066670000000,
                        0.108666670000000,
                        0.109066670000000,
                        0.111466670000000,
                        0.107166670000000,
                        0.104366670000000,
                        0.107766670000000,
                        0.110766670000000,
                        0.110666670000000,
                        0.110366670000000,
                        0.110566670000000,
                        0.111266670000000,
                        0.111266670000000,
                        0.113866670000000,
                        0.111566670000000,
                        0.109466670000000,
                        0.108666670000000,
                        0.110466670000000,
                        0.109866670000000,
                        0.105266670000000,
                        0.109966670000000,
                        0.108666670000000,
                        0.107466670000000,
                        0.112766670000000,
                        0.112366670000000,
                        0.111966670000000,
                        0.107366670000000,
                        0.110266670000000,
                        0.110666670000000,
                        0.109566670000000,
                        0.110466670000000,
                        0.110866670000000,
                        0.111566670000000,
                        0.109166670000000,
                        0.108766670000000,
                        0.104266670000000,
                        0.106766670000000,
                        0.107866670000000,
                        0.107566670000000,
                        0.109466670000000,
                        0.109366670000000,
                        0.106666670000000,
                        0.107566670000000,
                        0.116166670000000,
                        0.114266670000000,
                        0.114466670000000,
                        0.112966670000000,
                        0.109466670000000,
                        0.109566670000000,
                        0.107366670000000,
                        0.105566670000000,
                        0.109866670000000,
                        0.115766670000000,
                        0.117766670000000,
                        0.113166670000000,
                        0.111566670000000,
                        0.113766670000000,
                        0.112966670000000,
                        0.111766670000000,
                        0.114266670000000,
                        0.113666670000000,
                        0.108866670000000,
                        0.108766670000000,
                        0.113166670000000,
                        0.112966670000000,
                        0.111966670000000,
                        0.111366670000000,
                        0.111566670000000,
                        0.111466670000000,
                        0.110066670000000,
                        0.111066670000000,
                        0.113266670000000,
                        0.107466670000000,
                        0.114166670000000,
                        0.113266670000000,
                        0.111666670000000,
                        0.108766670000000,
                        0.105666670000000,
                        0.106766670000000,
                        0.106666670000000,
                        0.111266670000000,
                        0.109266670000000,
                        0.107466670000000,
                        0.112366670000000,
                        0.113366670000000,
                        0.110066670000000,
                        0.106366670000000,
                        0.109166670000000,
                        0.111166670000000,
                        0.105466670000000,
                        0.102966670000000,
                        0.105966670000000,
                        0.106266670000000,
                        0.112866670000000,
                        0.111366670000000,
                        0.107766670000000,
                        0.106366670000000,
                        0.104766670000000,
                        0.108966670000000,
                        0.109366670000000,
                        0.107966670000000,
                        0.106066670000000,
                        0.106666670000000,
                        0.105966670000000,
                        0.103066670000000,
                        0.102766670000000,
                        0.103266670000000,
                        0.099166667000000,
                        0.105166670000000,
                        0.105066670000000,
                        0.101866670000000,
                        0.104666670000000,
                        0.106366670000000,
                        0.105966670000000,
                        0.100866670000000,
                        0.101566670000000,
                        0.107166670000000,
                        0.105966670000000,
                        0.104966670000000,
                        0.105466670000000,
                        0.112866670000000,
                        0.106266670000000,
                        0.104466670000000,
                        0.106666670000000,
                        0.103566670000000,
                        0.103066670000000,
                        0.097566667000000,
                        0.108366670000000,
                        0.103966670000000,
                        0.102266670000000,
                        0.100266670000000,
                        0.102866670000000,
                        0.094066667000000,
                        0.104766670000000,
                        0.104166670000000,
                        0.091766667000000,
                        0.090566667000000,
                        0.094666667000000,
                        0.098866667000000,
                        0.095666667000000,
                        0.096666667000000,
                        0.094366667000000,
                        0.091066667000000,
                        0.097966667000000,
                        0.095066667000000,
                        0.099266667000000,
                        0.091966667000000,
                        0.094966667000000,
                        0.099266667000000,
                        0.094466667000000,
                        0.088366667000000,
                        0.092566667000000,
                        0.096466667000000,
                        0.094366667000000,
                        0.092866667000000,
                        0.102266670000000,
                        0.095266667000000,
                        0.089366667000000,
                        0.098566667000000,
                        0.099466667000000,
                        0.095866667000000,
                        0.085666667000000,
                        0.091066667000000,
                        0.103866670000000,
                        0.097166667000000,
                        0.102766670000000,
                        0.101766670000000,
                        0.099366667000000,
                        0.094266667000000,
                        0.091166667000000,
                        0.091466667000000,
                        0.084366667000000,
                        0.085066667000000,
                        0.100666670000000,
                        0.101466670000000,
                        0.098766667000000,
                        0.097666667000000,
                        0.097466667000000,
                        0.091866667000000,
                        0.084666667000000,
                        0.094666667000000,
                        0.096566667000000,
                        0.087066667000000,
                        0.107666670000000,
                        0.099666667000000,
                        0.093566667000000,
                        0.093566667000000,
                        0.094666667000000,
                        0.093066667000000,
                        0.086266667000000,
                        0.085966667000000,
                        0.092266667000000,
                        0.097966667000000,
                        0.099166667000000,
                        0.097866667000000,
                        0.088466667000000,
                        0.092166667000000,
                        0.096066667000000,
                        0.097566667000000,
                        0.107766670000000,
                        0.098166667000000,
                        0.092066667000000,
                        0.097566667000000,
                        0.107966670000000,
                        0.093366667000000,
                        0.102966670000000,
                        0.106766670000000,
                        0.100166670000000,
                        0.104166670000000,
                        0.099166667000000,
                        0.098266667000000,
                        0.095166667000000,
                        0.104766670000000,
                        0.098166667000000,
                        0.101566670000000,
                        0.097566667000000,
                        0.099966667000000,
                        0.085066667000000,
                        0.084866667000000,
                        0.094266667000000,
                        0.087966667000000,
                        0.094566667000000,
                        0.104766670000000,
                        0.104866670000000,
                        0.106666670000000,
                        0.104166670000000,
                        0.115366670000000,
                        0.110066670000000,
                        0.103766670000000,
                        0.104066670000000,
                        0.100766670000000,
                        0.112366670000000,
                        0.106266670000000,
                        0.116066670000000,
                        0.122966670000000,
                        0.106366670000000,
                        0.104566670000000,
                        0.114966670000000,
                        0.122566670000000,
                        0.115766670000000,
                        0.122266670000000,
                        0.112866670000000,
                        0.106066670000000,
                        0.128666670000000,
                        0.128066670000000,
                        0.120866670000000,
                        0.101866670000000,
                        0.108366670000000,
                        0.114366670000000,
                        0.114466670000000,
                        0.113466670000000,
                        0.110566670000000,
                        0.096666667000000,
                        0.118666670000000,
                        0.115566670000000,
                        0.107166670000000,
                        0.111266670000000,
                        0.117166670000000,
                        0.120366670000000,
                        0.123066670000000,
                        0.102666670000000,
                        0.098766667000000,
                        0.117266670000000,
                        0.145466670000000,
                        0.123366670000000,
                        0.123666670000000,
                        0.134666670000000,
                        0.129566670000000,
                        0.135366670000000,
                        0.120466670000000,
                        0.108766670000000,
                        0.112166670000000,
                        0.100266670000000,
                        0.128266670000000,
                        0.129966670000000,
                        0.118766670000000,
                        0.133766670000000,
                        0.129966670000000,
                        0.125766670000000,
                        0.127166670000000,
                        0.119066670000000,
                        0.116466670000000,
                        0.115366670000000,
                        0.124166670000000,
                        0.116166670000000,
                        0.109866670000000,
                        0.110566670000000,
                        0.116766670000000,
                        0.110366670000000,
                        0.111666670000000,
                        0.113966670000000,
                        0.107866670000000,
                        0.107066670000000,
                        0.118166670000000,
                        0.110466670000000,
                        0.109166670000000,
                        0.105866670000000,
                        0.095566667000000,
                        0.095566667000000,
                        0.097366667000000,
                        0.096366667000000,
                        0.092966667000000,
                        0.088466667000000,
                        0.092366667000000,
                        0.093266667000000,
                        0.095566667000000,
                        0.096666667000000,
                        0.102666670000000,
                        0.100966670000000,
                        0.092066667000000,
                        0.090066667000000,
                        0.093066667000000,
                        0.089966667000000,
                        0.095766667000000,
                        0.097966667000000,
                        0.099966667000000,
                        0.094166667000000,
                        0.092366667000000,
                        0.097866667000000,
                        0.094966667000000,
                        0.093866667000000,
                        0.094066667000000,
                        0.097466667000000,
                        0.106466670000000,
                        0.099966667000000,
                        0.102966670000000,
                        0.098166667000000,
                        0.103566670000000,
                        0.106166670000000,
                        0.103366670000000,
                        0.103466670000000,
                        0.092766667000000,
                        0.095466667000000,
                        0.114066670000000,
                        0.099866667000000,
                        0.094766667000000,
                        0.105166670000000,
                        0.092566667000000,
                        0.093666667000000,
                        0.080566667000000,
                        0.081866667000000,
                        0.080866667000000,
                        0.075166667000000,
                        0.101966670000000,
                        0.093266667000000,
                        0.074666667000000,
                        0.078366667000000,
                        0.085066667000000,
                        0.089066667000000,
                        0.087566667000000,
                        0.091166667000000,
                        0.098666667000000,
                        0.092466667000000,
                        0.139666670000000,
                        0.083266667000000,
                        0.064766667000000,
                        0.087166667000000,
                        0.156066670000000,
                        0.181266670000000,
                        0.122966670000000,
                        0.173566670000000,
                        0.207666670000000,
                        0.213466670000000,
                        0.178966670000000,
                        0.277466670000000,
                    ]
                ),
            ),
            DatasetLoader_AgfaIT872Set(): (
                "agfa",
                289,
                "5",
                np.array(
                    [
                        8.6300000000000000,
                        12.090000000000000,
                        14.140000000000000,
                        14.020000000000000,
                        14.160000000000000,
                        14.190000000000000,
                        14.250000000000000,
                        14.530000000000000,
                        14.810000000000000,
                        14.880000000000000,
                        14.480000000000000,
                        13.610000000000000,
                        12.600000000000000,
                        11.720000000000000,
                        11.600000000000000,
                        12.690000000000000,
                        14.160000000000000,
                        16.350000000000000,
                        18.900000000000000,
                        20.870000000000000,
                        21.630000000000000,
                        21.230000000000000,
                        20.360000000000000,
                        19.520000000000000,
                        18.790000000000000,
                        18.400000000000000,
                        18.530000000000000,
                        19.180000000000000,
                        20.460000000000000,
                        22.580000000000000,
                        25.470000000000000,
                    ]
                ),
            ),
        }
        for dataset_loader, values in dataset_loaders.items():
            self.assertEqual(len(dataset_loader.load()[values[0]]), values[1])
            np.testing.assert_array_almost_equal(
                dataset_loader.content[values[0]][values[2]].values,
                values[3],
                decimal=7,
            )


if __name__ == "__main__":
    unittest.main()
