from typing import Any

from pyconnectwise.endpoints.base.connectwise_endpoint import ConnectWiseEndpoint
from pyconnectwise.endpoints.manage.ProjectTicketsIdNotesCountEndpoint import ProjectTicketsIdNotesCountEndpoint
from pyconnectwise.endpoints.manage.ProjectTicketsIdNotesIdEndpoint import ProjectTicketsIdNotesIdEndpoint
from pyconnectwise.interfaces import IDeleteable, IGettable, IPaginateable, IPatchable, IPostable, IPuttable
from pyconnectwise.models.manage import TicketNote
from pyconnectwise.responses.paginated_response import PaginatedResponse
from pyconnectwise.types import JSON, ConnectWiseAutomateRequestParams, ConnectWiseManageRequestParams, PatchRequestData


class ProjectTicketsIdNotesEndpoint(
    ConnectWiseEndpoint,
    IGettable[list[TicketNote], ConnectWiseManageRequestParams],
    IPostable[TicketNote, ConnectWiseManageRequestParams],
    IPaginateable[TicketNote, ConnectWiseManageRequestParams],
):
    def __init__(self, client, parent_endpoint=None):
        ConnectWiseEndpoint.__init__(self, client, "notes", parent_endpoint=parent_endpoint)
        IGettable.__init__(self, list[TicketNote])
        IPostable.__init__(self, TicketNote)
        IPaginateable.__init__(self, TicketNote)

        self.count = self._register_child_endpoint(ProjectTicketsIdNotesCountEndpoint(client, parent_endpoint=self))

    def id(self, id: int) -> ProjectTicketsIdNotesIdEndpoint:
        """
        Sets the ID for this endpoint and returns an initialized ProjectTicketsIdNotesIdEndpoint object to move down the chain.

        Parameters:
            id (int): The ID to set.
        Returns:
            ProjectTicketsIdNotesIdEndpoint: The initialized ProjectTicketsIdNotesIdEndpoint object.
        """
        child = ProjectTicketsIdNotesIdEndpoint(self.client, parent_endpoint=self)
        child._id = id
        return child

    def paginated(
        self, page: int, page_size: int, params: ConnectWiseManageRequestParams | None = None
    ) -> PaginatedResponse[TicketNote]:
        """
        Performs a GET request against the /project/tickets/{id}/notes endpoint and returns an initialized PaginatedResponse object.

        Parameters:
            page (int): The page number to request.
            page_size (int): The number of results to return per page.
            params (dict[str, int | str]): The parameters to send in the request query string.
        Returns:
            PaginatedResponse[TicketNote]: The initialized PaginatedResponse object.
        """
        if params:
            params["page"] = page
            params["pageSize"] = page_size
        else:
            params = {"page": page, "pageSize": page_size}
        return PaginatedResponse(super()._make_request("GET", params=params), TicketNote, self, page, page_size, params)

    def get(self, data: JSON | None = None, params: ConnectWiseManageRequestParams | None = None) -> list[TicketNote]:
        """
        Performs a GET request against the /project/tickets/{id}/notes endpoint.

        Parameters:
            data (dict[str, Any]): The data to send in the request body.
            params (dict[str, int | str]): The parameters to send in the request query string.
        Returns:
            list[TicketNote]: The parsed response data.
        """
        return self._parse_many(TicketNote, super()._make_request("GET", data=data, params=params).json())

    def post(self, data: JSON | None = None, params: ConnectWiseManageRequestParams | None = None) -> TicketNote:
        """
        Performs a POST request against the /project/tickets/{id}/notes endpoint.

        Parameters:
            data (dict[str, Any]): The data to send in the request body.
            params (dict[str, int | str]): The parameters to send in the request query string.
        Returns:
            TicketNote: The parsed response data.
        """
        return self._parse_one(TicketNote, super()._make_request("POST", data=data, params=params).json())
