# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mtfl']

package_data = \
{'': ['*']}

install_requires = \
['attrs>=19.3.0,<20.0.0',
 'discrete-signals>=0.7.3,<0.8.0',
 'lenses>=0.5.0,<0.6.0',
 'parsimonious>=0.8.1,<0.9.0',
 'pylint>=2.6.0,<3.0.0']

setup_kwargs = {
    'name': 'metric-temporal-fuzzy-logic',
    'version': '0.0.6',
    'description': 'A library for manipulating and evaluating metric temporal (fuzzy) logic.',
    'long_description': '# py-metric-temporal-fuzzy-logic\n\n<!-- [![Build Status](https://cloud.drone.io/api/badges/mvcisback/py-metric-temporal-logic/status.svg)](https://cloud.drone.io/mvcisback/py-metric-temporal-logic) -->\n<!-- [![codecov](https://codecov.io/gh/mvcisback/py-metric-temporal-logic/branch/master/graph/badge.svg)](https://codecov.io/gh/mvcisback/py-metric-temporal-logic) -->\n<!-- [![PyPI version](https://badge.fury.io/py/metric-temporal-logic.svg)](https://badge.fury.io/py/metric-temporal-logic) -->\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)\n[![DOI](https://zenodo.org/badge/72686883.svg)](https://zenodo.org/badge/latestdoi/72686883)\n\n> **WARNING**: This is an experimental fork of the `metric-temporal-logic` library, \n> available at [mvcisback/py-metric-temporal-logic](https://github.com/mvcisback/py-metric-temporal-logic). \n> THis fork adds support for different fuzzy connectives to the evaluation logic; the \n> changes support different semantics for the core logical operations: norm (and), \n> conorm (or), implication, and negation (not). Unless those changes are a strong \n> requirement, we recommend the use of the original library.\n\n<!-- markdown-toc start - Don\'t edit this section. Run M-x markdown-toc-generate-toc again -->\n**Table of Contents**\n\n- [About](#about)\n- [Installation](#installation)\n- [Usage](#usage)\n    - [Python Operator API](#python-operator-api)\n        - [Propositional logic (using python syntax)](#propositional-logic-using-python-syntax)\n        - [Modal Logic (using python syntax)](#modal-logic-using-python-syntax)\n    - [String based API](#string-based-api)\n        - [Propositional logic (parse api)](#propositional-logic-parse-api)\n        - [Modal Logic (parser api)](#modal-logic-parser-api)\n    - [Boolean Evaluation](#boolean-evaluation)\n    - [Fuzzy Evaluation](#fuzzy-evaluation)\n    - [Quantitative Evaluate (Signal Temporal Logic)](#quantitative-evaluate-signal-temporal-logic)\n    - [Utilities](#utilities)\n- [Similar Projects](#similar-projects)\n- [Citing](#citing)\n\n<!-- markdown-toc end -->\n\n# About\n\nPython library for working with Metric Temporal Logic (MTL) with fuzzy logic. Metric\nTemporal Logic is an extension of Linear Temporal Logic (LTL) for\nspecifying properties over time series (See [Alur][1]). Fuzzy Logic introduces fuzzy \nvaluation of the properties, as an example in the interval `[0;1]` as opposed to crisp \n`True`/`False` values (See [FT-LTL][2]). The library does not introduce Fuzzy Time \noperators such as *almost always*. Some practical examples are given in the usage.\n\n# Installation\n\nThere is currently no release of the library. It needs to be built locally using the\n[poetry](https://poetry.eustace.io/) python package/dependency\nmanagement tool:\n`$ poetry build`\n\nThe package can be installed locally using your reference package manager.\n\n<!--\nIf you just need to use `metric-temporal-logic`, you can just run:\n\n`$ pip install metric-temporal-logic`\n\nFor developers, note that this project uses the\n[poetry](https://poetry.eustace.io/) python package/dependency\nmanagement tool. Please familarize yourself with it and then\nrun:\n\n`$ poetry install`\n-->\n\n# Usage\n\nTo begin, we import `mtfl`.\n\n```python\nimport mtfl\n```\n\nThere are **two** APIs for interacting with the `mtfl` module. Namely, one can specify the MTL expression using:\n1. [Python Operators](#python-operator-api).\n2. [Strings + The parse API](#string-based-api).\n\nWe begin with the Python Operator API:\n\n## Python Operator API\n\n### Propositional logic (using python syntax)\n```python\na, b = mtfl.parse(\'a\'), mtfl.parse(\'b\')\nphi0 = ~a\nphi1 = a & b\nphi2 = a | b\nphi3 = a ^ b\nphi4 = a.iff(b)\nphi5 = a.implies(b)\n```\n\n\n### Modal Logic (using python syntax)\n\n```python\na, b = mtfl.parse(\'a\'), mtfl.parse(\'b\')\n\n# Eventually `a` will hold.\nphi1 = a.eventually()\n\n# `a & b` will always hold.\nphi2 = (a & b).always()\n\n# `a` until `b`\nphi3 = a.until(b)\n\n# `a` weak until `b`\nphi4 = a.weak_until(b)\n\n# Whenever `a` holds, then `b` holds in the next two time units.\nphi5 = (a.implies(b.eventually(lo=0, hi=2))).always()\n\n# We also support timed until.\nphi6 = a.timed_until(b, lo=0, hi=2)\n\n# `a` holds in two time steps.\nphi7 = a >> 2\n```\n\n\n## String based API\n\n### Propositional logic (parse api)\n```python\n# - Lowercase strings denote atomic predicates.\nphi0 = mtfl.parse(\'atomicpred\')\n\n# - infix operators need to be surrounded by parens.\nphi1 = mtfl.parse(\'((a & b & c) | d | e)\')\nphi2 = mtfl.parse(\'(a -> b) & (~a -> c)\')\nphi3 = mtfl.parse(\'(a -> b -> c)\')\nphi4 = mtfl.parse(\'(a <-> b <-> c)\')\nphi5 = mtfl.parse(\'(x ^ y ^ z)\')\n\n# - Unary operators (negation)\nphi6 = mtfl.parse(\'~a\')\nphi7 = mtfl.parse(\'~(a)\')\n```\n\n### Modal Logic (parser api)\n\n```python\n# Eventually `x` will hold.\nphi1 = mtfl.parse(\'F x\')\n\n# `x & y` will always hold.\nphi2 = mtfl.parse(\'G(x & y)\')\n\n# `x` holds until `y` holds. \n# Note that since `U` is binary, it requires parens.\nphi3 = mtfl.parse(\'(x U y)\')\n\n# Weak until (`y` never has to hold).\nphi4 = mtfl.parse(\'(x W y)\')\n\n# Whenever `x` holds, then `y` holds in the next two time units.\nphi5 = mtfl.parse(\'G(x -> F[0, 2] y)\')\n\n# We also support timed until.\nphi6 = mtfl.parse(\'(a U[0, 2] b)\')\n\n# Finally, if time is discretized, we also support the next operator.\n# Thus, LTL can also be modeled.\n# `a` holds in two time steps.\nphi7 = mtfl.parse(\'XX a\')\n```\n\n## Quantitative Evaluate (Signal Temporal Logic)\n\nGiven a property `phi`, one can evaluate is a timeseries satisifies\n`phi`. Time Series can either be defined using a dictionary mapping\natomic predicate names to lists of (`time`, `val`) pairs **or** using\nthe [DiscreteSignals](https://github.com/mvcisback/DiscreteSignals)\nAPI (used internally).\n\nThere are two types of evaluation. One uses the boolean semantics of\nMTL and the other uses Signal Temporal Logic like semantics.\n\n\n```python\n# Assumes piece wise constant interpolation.\ndata = {\n    \'a\': [(0, 100), (1, -1), (3, -2)],\n    \'b\': [(0, 20), (0.2, 2), (4, -10)]\n}\n\nphi = mtfl.parse(\'F(a | b)\')\nprint(phi(data))\n# output: 100\n\n# Evaluate at t=3\nprint(phi(data, time=3))\n# output: 2\n\n# Evaluate with discrete time\nphi = mtfl.parse(\'X b\')\nprint(phi(data, dt=0.2))\n# output: 2\n```\n\n## Boolean Evaluation\n\nTo Boolean semantics can be thought of as a special case of the\nquantitative semantics where `True ↦ 1` and `False ↦ -1`.  This\nconversion happens automatically using the `quantitative=False`\nflag.\n\n\n```python\n# Assumes piece wise constant interpolation.\ndata = {\n    \'a\': [(0, True), (1, False), (3, False)],\n    \'b\': [(0, False), (0.2, True), (4, False)]\n}\n\nphi = mtfl.parse(\'F(a | b)\')\nprint(phi(data, quantitative=False))\n# output: True\n\nphi = mtfl.parse(\'F(a | b)\')\nprint(phi(data))\n# output: True\n\n# Note, quantitative parameter defaults to False\n\n# Evaluate at t=3. \nprint(phi(data, time=3, quantitative=False))\n# output: False\n\n# Compute sliding satisifaction.\nprint(phi(data, time=None, quantitative=False))\n# output: [(0, True), (0.2, True), (4, False)]\n\n# Evaluate with discrete time\nphi = mtfl.parse(\'X b\')\nprint(phi(data, dt=0.2, quantitative=False))\n# output: True\n```\n\n\n## Fuzzy Evaluation\n\nFuzzy evaluation considers the signals as fuzzy values or values to be\ncompared through fuzzy operators. The connectives used for evaluation,\nthat is the implementation of the basic logic operations such as `and`\nor `or`, can be specified through the `logic` parameter. The connectives \navailable in `mtfl.connective` are `default`, `zadeh`, `godel`, \n`lukasiewicz`, and `product` (See [FT-LTL][2]).\n\n```python\na = mtfl.parse("a")\nb = mtfl.parse("b")\n\nd = {\n    "a": [(0,  5.), (1, 10.),           (3,  0.), (4, 10.), (5, 11.)],\n    "b": [(0, 15.),           (2,  5.),           (4, 10.),        ],\n}\n\n# Crisp comparison between the value of a and a constant\ni = a < 6\nprint(i, i(d, time=None, logic=mtfl.connective.zadeh, quantitative=True))\n# output: (a < 6) [(0, 1.0), (1, 0.0), (3, 1.0), (4, 0.0), (5, 0.0)]\n\n# Crisp comparison between the value of a and b\ni = a < b\nprint(i, i(d, time=None, logic=mtfl.connective.zadeh, quantitative=True))\n# output: (a < b) [(0, 1.0), (1, 1.0), (2, 0.0), (3, 1.0), (4, 0.0), (5, 0.0)]\n\n# Fuzzy comparison between the value of a and b, increasing as a decreases in the interval (b;b+10]\ni = a.lt(b, 10)\nprint(i, i(d, time=None, logic=mtfl.connective.zadeh, quantitative=True))\n# output: (a <[~10] b) [(0, 1.0), (1, 1.0), (2, 0.5), (3, 1.0), (4, 1.0), (5, 0.9)]\n\n# Temporal fuzzy operation\ni = a.lt(b, 10).always()\nprint(i, i(d, logic=mtfl.connective.zadeh, quantitative=True))\n# output: G(a <[~10] b) 0.5\n```\n\n### Fuzzy caveats\n\nConsider using `quantitvative=True` when specifying a fuzzy logic. \nCombining fuzzy and boolean evaluation might result in unexpected \nbehaviours due to the underlying conversion of signal values to \nboolean ones. \n\nNote that the library only evaluates signals and operations at \npivot points, where their value changes. This might result in \ninvalid values for some connectives (`product`). Consider the \nfollowing example:\n```\ndata = {\n    \'a\': [(0, 0.5), (1, 0.1)],\n}\n\nphi = mtfl.parse(\'G(a)\')\nprint(phi(data, quantitative=False))\n# output: 0.05\n```\n\nUnder the `product` connective (See [FT-LTL][2]), the norm operator\nis defined as the product of two values. `phi` thus evaluates to \n`0.05 = 0.5 * 0.1` across the whole signal when it should actually \ntend towards `0` due to the repeated valuation of `a < 1` at all \ntime instants.\n\n\n## Utilities\n```python\nimport mtfl\nfrom mtfl import utils\n\nprint(utils.scope(mtfl.parse(\'XX a\'), dt=0.1))\n# output: 0.2\n\nprint(utils.discretize(mtfl.parse(\'F[0, 0.2] a\'), dt=0.1))\n# output: (a | X a | XX a)\n```\n\n# Similar Projects\nFeel free to open up a pull-request to add other similar projects. This library was written to meet some of my unique needs, for example I wanted the AST to be immutable and wanted the library to **just** handle manipulating MTL. Many other similar projects exist with different goals.\n\n1. https://github.com/doganulus/python-monitors\n1. https://github.com/STLInspector/STLInspector\n\n# Citing\n\n    @misc{pyMTL,\n      author       = {Marcell Vazquez-Chanlatte},\n      title        = {mvcisback/py-metric-temporal-logic: v0.1.1},\n      month        = jan,\n      year         = 2019,\n      doi          = {10.5281/zenodo.2548862},\n      url          = {https://doi.org/10.5281/zenodo.2548862}\n    }\n\n[1]: https://link.springer.com/chapter/10.1007/BFb0031988\n[2]: https://dl.acm.org/doi/10.1145/2629606\n',
    'author': 'Gaudeval',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Gaudeval/py-metric-temporal-fuzzy-logic',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
