"""
Utility functions for handling creation, writing, reading and parsing of BIDS files

MIT License

Copyright (c) 2017-2021 Mike Tyszka

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
"""

import os
import sys
import shutil
import json
import pydicom
import numpy as np


def read_json(fname):
    """
    Safely read JSON sidecar file into a dictionary
    :param fname: string
        JSON filename
    :return: dictionary structure
    """

    try:
        fd = open(fname, 'r')
        json_dict = json.load(fd)
        fd.close()
    except IOError:
        print('*** {}'.format(fname))
        print('*** JSON sidecar not found - returning empty dictionary')
        json_dict = dict()
    except json.decoder.JSONDecodeError:
        print('*** {}'.format(fname))
        print('*** JSON sidecar decoding error - returning empty dictionary')
        json_dict = dict()

    return json_dict


def write_json(fname, meta_dict, overwrite=False):
    """
    Write a dictionary to a JSON file. Account for overwrite flag
    :param fname: string
        JSON filename
    :param meta_dict: dictionary
        Dictionary
    :param overwrite: bool
        Overwrite flag
    :return:
    """

    bname = os.path.basename(fname)

    if os.path.isfile(fname):
        if overwrite:
            print('    Overwriting previous %s' % bname)
            create_file = True
        else:
            print('    Preserving previous %s' % bname)
            create_file = False
    else:
        print('    Creating new %s' % bname)
        create_file = True

    if create_file:
        with open(fname, 'w') as fd:
            json.dump(meta_dict, fd, indent=4, separators=(',', ':'))


def dcm_info(dcm_dir):
    """
    Extract relevant subject information from DICOM header
    - Assumes only one subject present within dcm_dir

    :param dcm_dir: directory containing all DICOM files or DICOM subfolders
    :return info_dict: DICOM header information dictionary
    """

    # Init the DICOM structure
    ds = []

    # Init the subject info dictionary
    info_dict = dict()

    # Walk through dcm_dir looking for valid DICOM files
    for subdir, dirs, files in os.walk(dcm_dir):

        for file in files:

            try:
                ds = pydicom.read_file(os.path.join(subdir, file))
            except Exception as err:
                # Silently skip problem files in DICOM directory
                continue

            # Break out if valid DICOM read
            if ds:
                break

    if ds:

        # Fill dictionary
        # Note that DICOM anonymization tools sometimes clear these fields
        if hasattr(ds, 'PatientSex'):
            info_dict['Sex'] = ds.PatientSex
        else:
            info_dict['Sex'] = 'Unknown'

        if hasattr(ds, 'PatientAge'):
            info_dict['Age'] = ds.PatientAge
        else:
            info_dict['Age'] = 0

    else:

        print('* No DICOM header information found in %s' % dcm_dir)
        print('* Confirm that DICOM images in this folder are uncompressed')
        print('* Exiting')
        sys.exit(1)

    return info_dict


def parse_dcm2niix_fname(fname):
    """
    Parse dcm2niix filename into values
    Current dcm2niix output filename format is '%n--%d--s%s--e%e'
    which translates to '<PatientName>--<SeriesDescription>--s<SeriesNo>--e<EchoNo>'
    Check dcm2niix call in __main__.py to confirm current format string

    NOTES:
    - Multiple series generated by single acquisition (eg multiecho fieldmaps, localizers, etc) are
      handled through the dcm2niix extensions (_e1, e2_ph, _i00001, etc).
    - Series number resets following subject re-landmarking make the SeriesNo useful only for
      determining series uniqueness and not for ordering or run numbering.
    - See also bidskit.translate.auto_run_no()

    Current dcm2niix version : v1.0.20220720

    :param fname: str, BIDS-style image or sidecar filename
    :return info: dict
    """

    # Create info dictionary
    info = dict()

    # Parent directory
    info['DirName'] = os.path.dirname(fname)

    # Strip parent directory and extension(s)
    fname, fext = strip_extensions(os.path.basename(fname))

    # Split filename at '--'s
    vals = fname.split('--')

    # Filename format: <PatientName>--<SeriesDescription>--s<SeriesNo>--e<EchoNo>[_ph].<ext>
    info['SubjName'] = vals[0]
    info['SerDesc'] = vals[1]
    info['SerNo'] = np.int(vals[2].replace('s', ''))

    # Check for phase image suffix after echo number (eg "e2_ph")
    bits = vals[3].split('_', 1)
    info['EchoNo'] = np.int(bits[0].replace('e', ''))

    # Record any suffix after the echo number key (typically "ph" if anything)
    if len(bits) > 1:
        info['Suffix'] = '_' + bits[1]
    else:
        info['Suffix'] = ''

    return info


def parse_bids_fname_keyvals(fname):
    """
    BIDS-like filename parser

    Parse BIDS-like filename into key-value pairs
    Supports both BIDS-like and ReproIn format filenames
    eg "task-rest_run-1_bold" or "anat-T1w_run-2"

    :param fname: str, pathlike
        Raw BIDS-like filename with possible file extension(s)
    :return: dict,
        Dictionary of key-value pairs parsed from BIDS-like or ReproIn string
    :return: str
        Containing directory
    """

    # Split filename into containing directory and base name
    dname = os.path.dirname(fname)
    bids_stub = os.path.basename(fname)

    # Init return dictionary with BIDS 1.7.0 and ReproIn key names
    bids_keys = {
        'func': "",  # ReproIn
        'fmap': "",  # ReproIn
        'anat': "",  # ReproIn
        'dwi': "",  # ReproIn
        'sub': "",
        'ses': "",
        'task': "",
        'run': "",
        'acq': "",
        'dir': "",
        'ce': "",
        'rec': "",
        'mod': "",
        'echo': "",
        'proc': "",
        'part': "",
        'suffix': "unknown",  # Should always return a non-empty string
    }

    # Extract base filename and strip up to two extensions
    # Accounts for both '.nii' and '.nii.gz' variants
    bids_stub, ext1 = os.path.splitext(bids_stub)
    bids_stub, ext2 = os.path.splitext(bids_stub)

    # Remember full extension
    bids_keys['extension'] = ext2 + ext1

    # Check for recon variants keys at end of BIDS stub string
    # These may have a leading '_' or ' ' eg 'acq-mez_T1w RMS' and 'task-rest_bold_SBRef'

    recon_list = ['SBRef', 'RMS']
    recon_key = ''

    for recon_str in recon_list:
        if bids_stub.endswith(recon_str):
            recon_start = bids_stub.rfind(recon_str)
            recon_key = recon_str
            bids_stub = bids_stub[:(recon_start-1)]

    # Find position of last underscore in basename
    last_underscore = bids_stub.rfind('_')

    if last_underscore < 0:

        # No underscores found in bids_stub. Interpret as isolated suffix (eg "T2w")
        bids_keys['suffix'] = bids_stub

    else:

        suffix_start = bids_stub.rfind('_') + 1
        draft_suffix = bids_stub[suffix_start:]

        # Handle special case of no suffix, only key-value pairs
        if '-' in draft_suffix or len(draft_suffix) < 1:

            # Leave suffix empty in dict
            bids_keys['suffix'] = ''

        else:

            # Split basename into prefix and suffix
            bids_keys['suffix'] = bids_stub[suffix_start:]
            bids_stub = bids_stub[:(suffix_start-1)]

    # Divide filename into keys and values
    # Value segments are delimited by '<key>-' strings

    key_name = []
    val_start = []
    key_start = []

    # Search for any valid keys in filename
    # Record key and value start indices within string
    for key in bids_keys:

        key_str = key + '-'

        i0 = bids_stub.find(key_str)
        if i0 > -1:
            i1 = i0 + len(key_str)
            key_name.append(key)
            key_start.append(i0)
            val_start.append(i1)

    # Convert lists to numpy arrays
    key_name = np.array(key_name)
    key_start = np.array(key_start)
    val_start = np.array(val_start)

    # Sort keys by position in filename
    key_order = np.argsort(key_start)
    key_name_sorted = key_name[key_order]
    val_start_sorted = val_start[key_order]
    val_end_sorted = key_start[np.roll(key_order, -1)] - 1

    # Fill BIDS key-value dictionary
    for kc in range(len(key_name_sorted)):

        kname = key_name_sorted[kc]
        vstart = val_start_sorted[kc]
        vend = val_end_sorted[kc]

        # Catch negative vend (only happens for final key-value without suffix)
        if vend < 0:
            bids_keys[kname] = bids_stub[vstart:]
        else:
            bids_keys[kname] = bids_stub[vstart:vend]

    # Tidy up Siemens recon extensions
    if 'SBRef' in recon_key:
        # Replace entire double suffix with 'sbref'
        bids_keys['suffix'] = 'sbref'

    if 'RMS' in recon_key:
        # Add 'rms' to acq key
        bids_keys['acq'] = bids_keys['acq'] + 'rms'

    # Finally purge any empty keys
    bids_keys = {k: v for k, v in bids_keys.items() if v}

    return bids_keys, dname


def safe_mkdir(dname):
    """
    Safely create a directory path
    :param dname: string
    :return:
    """

    if not os.path.isdir(dname):
        os.makedirs(dname, exist_ok=True)


def safe_copy(fname1, fname2, overwrite=False):
    """
    Copy file accounting for overwrite flag
    :param fname1: str
    :param fname2: str
    :param overwrite: bool
    :return:
    """

    bname1, bname2 = os.path.basename(fname1), os.path.basename(fname2)

    if os.path.isfile(fname2):
        if overwrite:
            print('    Copying %s to %s (overwrite)' % (bname1, bname2))
            create_file = True
        else:
            print('    Preserving previous %s' % bname2)
            create_file = False
    else:
        print('    Copying %s to %s' % (bname1, bname2))
        create_file = True

    if create_file:
        shutil.copy(fname1, fname2)


def create_file_if_missing(filename, content):
    """
    Create file if missing, so we do not override any possibly introduced changes

    :param filename:
    :param content:
    :return:
    """

    if os.path.lexists(filename):
        return False

    dirname = os.path.dirname(filename)

    if not os.path.exists(dirname):
        os.makedirs(dirname)

    with open(filename, 'w') as f:
        f.write(content)

    return True


def strip_extensions(fname):
    """
    Remove one or more extensions from a filename
    :param fname:
    :return:
    """

    fstub, fext = os.path.splitext(fname)
    if fext == '.gz':
        fstub, fext2 = os.path.splitext(fstub)
        fext = fext2 + fext
    return fstub, fext


def nii_to_json(nii_fname, nii_ext):
    """
    Replace Nifti extension ('.nii.gz' or '.nii') with '.json'

    :param nii_fname:
    :return: json_fname
    """
    if nii_ext in nii_fname:
        json_fname = nii_fname.replace(nii_ext, '.json')
    else:
        print('* Unknown extension for %s' % nii_fname)
        json_fname = nii_fname + '.json'

    return json_fname