"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Port = exports.Protocol = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * Protocol for use in Connection Rules
 *
 * https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml
 */
var Protocol;
(function (Protocol) {
    Protocol["ALL"] = "-1";
    Protocol["HOPOPT"] = "0";
    Protocol["ICMP"] = "icmp";
    Protocol["IGMP"] = "2";
    Protocol["GGP"] = "3";
    Protocol["IPV4"] = "4";
    Protocol["ST"] = "5";
    Protocol["TCP"] = "tcp";
    Protocol["CBT"] = "7";
    Protocol["EGP"] = "8";
    Protocol["IGP"] = "9";
    Protocol["BBN_RCC_MON"] = "10";
    Protocol["NVP_II"] = "11";
    Protocol["PUP"] = "12";
    Protocol["EMCON"] = "14";
    Protocol["XNET"] = "15";
    Protocol["CHAOS"] = "16";
    Protocol["UDP"] = "udp";
    Protocol["MUX"] = "18";
    Protocol["DCN_MEAS"] = "19";
    Protocol["HMP"] = "20";
    Protocol["PRM"] = "21";
    Protocol["XNS_IDP"] = "22";
    Protocol["TRUNK_1"] = "23";
    Protocol["TRUNK_2"] = "24";
    Protocol["LEAF_1"] = "25";
    Protocol["LEAF_2"] = "26";
    Protocol["RDP"] = "27";
    Protocol["IRTP"] = "28";
    Protocol["ISO_TP4"] = "29";
    Protocol["NETBLT"] = "30";
    Protocol["MFE_NSP"] = "31";
    Protocol["MERIT_INP"] = "32";
    Protocol["DCCP"] = "33";
    Protocol["THREEPC"] = "34";
    Protocol["IDPR"] = "35";
    Protocol["XTP"] = "36";
    Protocol["DDP"] = "37";
    Protocol["IDPR_CMTP"] = "38";
    Protocol["TPPLUSPLUS"] = "39";
    Protocol["IL"] = "40";
    Protocol["IPV6"] = "41";
    Protocol["SDRP"] = "42";
    Protocol["IPV6_ROUTE"] = "43";
    Protocol["IPV6_FRAG"] = "44";
    Protocol["IDRP"] = "45";
    Protocol["RSVP"] = "46";
    Protocol["GRE"] = "47";
    Protocol["DSR"] = "48";
    Protocol["BNA"] = "49";
    Protocol["ESP"] = "50";
    Protocol["AH"] = "51";
    Protocol["I_NLSP"] = "52";
    Protocol["SWIPE"] = "53";
    Protocol["NARP"] = "54";
    Protocol["MOBILE"] = "55";
    Protocol["TLSP"] = "56";
    Protocol["SKIP"] = "57";
    Protocol["ICMPV6"] = "icmpv6";
    Protocol["IPV6_NONXT"] = "59";
    Protocol["IPV6_OPTS"] = "60";
    Protocol["CFTP"] = "62";
    Protocol["ANY_LOCAL"] = "63";
    Protocol["SAT_EXPAK"] = "64";
    Protocol["KRYPTOLAN"] = "65";
    Protocol["RVD"] = "66";
    Protocol["IPPC"] = "67";
    Protocol["ANY_DFS"] = "68";
    Protocol["SAT_MON"] = "69";
    Protocol["VISA"] = "70";
    Protocol["IPCV"] = "71";
    Protocol["CPNX"] = "72";
    Protocol["CPHB"] = "73";
    Protocol["WSN"] = "74";
    Protocol["PVP"] = "75";
    Protocol["BR_SAT_MON"] = "76";
    Protocol["SUN_ND"] = "77";
    Protocol["WB_MON"] = "78";
    Protocol["WB_EXPAK"] = "79";
    Protocol["ISO_IP"] = "80";
    Protocol["VMTP"] = "81";
    Protocol["SECURE_VMTP"] = "82";
    Protocol["VINES"] = "83";
    Protocol["TTP"] = "84";
    Protocol["IPTM"] = "84_";
    Protocol["NSFNET_IGP"] = "85";
    Protocol["DGP"] = "86";
    Protocol["TCF"] = "87";
    Protocol["EIGRP"] = "88";
    Protocol["OSPFIGP"] = "89";
    Protocol["SPRITE_RPC"] = "90";
    Protocol["LARP"] = "91";
    Protocol["MTP"] = "92";
    Protocol["AX_25"] = "93";
    Protocol["IPIP"] = "94";
    Protocol["MICP"] = "95";
    Protocol["SCC_SP"] = "96";
    Protocol["ETHERIP"] = "97";
    Protocol["ENCAP"] = "98";
    Protocol["ANY_ENC"] = "99";
    Protocol["GMTP"] = "100";
    Protocol["IFMP"] = "101";
    Protocol["PNNI"] = "102";
    Protocol["PIM"] = "103";
    Protocol["ARIS"] = "104";
    Protocol["SCPS"] = "105";
    Protocol["QNX"] = "106";
    Protocol["A_N"] = "107";
    Protocol["IPCOMP"] = "108";
    Protocol["SNP"] = "109";
    Protocol["COMPAQ_PEER"] = "110";
    Protocol["IPX_IN_IP"] = "111";
    Protocol["VRRP"] = "112";
    Protocol["PGM"] = "113";
    Protocol["ANY_0_HOP"] = "114";
    Protocol["L2_T_P"] = "115";
    Protocol["DDX"] = "116";
    Protocol["IATP"] = "117";
    Protocol["STP"] = "118";
    Protocol["SRP"] = "119";
    Protocol["UTI"] = "120";
    Protocol["SMP"] = "121";
    Protocol["SM"] = "122";
    Protocol["PTP"] = "123";
    Protocol["ISIS_IPV4"] = "124";
    Protocol["FIRE"] = "125";
    Protocol["CRTP"] = "126";
    Protocol["CRUDP"] = "127";
    Protocol["SSCOPMCE"] = "128";
    Protocol["IPLT"] = "129";
    Protocol["SPS"] = "130";
    Protocol["PIPE"] = "131";
    Protocol["SCTP"] = "132";
    Protocol["FC"] = "133";
    Protocol["RSVP_E2E_IGNORE"] = "134";
    Protocol["MOBILITY_HEADER"] = "135";
    Protocol["UDPLITE"] = "136";
    Protocol["MPLS_IN_IP"] = "137";
    Protocol["MANET"] = "138";
    Protocol["HIP"] = "139";
    Protocol["SHIM6"] = "140";
    Protocol["WESP"] = "141";
    Protocol["ROHC"] = "142";
    Protocol["ETHERNET"] = "143";
    Protocol["EXPERIMENT_1"] = "253";
    Protocol["EXPERIMENT_2"] = "254";
    Protocol["RESERVED"] = "255";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Interface for classes that provide the connection-specification parts of a security group rule
 */
class Port {
    constructor(props) {
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_PortProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Port);
            }
            throw error;
        }
        this.canInlineRule =
            !core_1.Token.isUnresolved(props.fromPort) && !core_1.Token.isUnresolved(props.toPort);
    }
    /**
     * A single TCP port
     */
    static tcp(port) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: port,
            toPort: port,
            stringRepresentation: renderPort(port),
        });
    }
    /**
     * A TCP port range
     */
    static tcpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any TCP traffic
     */
    static allTcp() {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'ALL PORTS',
        });
    }
    /**
     * A single UDP port
     */
    static udp(port) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: port,
            toPort: port,
            stringRepresentation: `UDP ${renderPort(port)}`,
        });
    }
    /**
     * A UDP port range
     */
    static udpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `UDP ${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any UDP traffic
     */
    static allUdp() {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'UDP ALL PORTS',
        });
    }
    /**
     * A specific combination of ICMP type and code
     *
     * @see https://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml
     */
    static icmpTypeAndCode(type, code) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: code,
            stringRepresentation: `ICMP Type ${type} Code ${code}`,
        });
    }
    /**
     * All codes for a single ICMP type
     */
    static icmpType(type) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: -1,
            stringRepresentation: `ICMP Type ${type}`,
        });
    }
    /**
     * ICMP ping (echo) traffic
     */
    static icmpPing() {
        return Port.icmpType(8);
    }
    /**
     * All ICMP traffic
     */
    static allIcmp() {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: -1,
            toPort: -1,
            stringRepresentation: 'ALL ICMP',
        });
    }
    /**
     * All traffic
     */
    static allTraffic() {
        return new Port({
            protocol: Protocol.ALL,
            stringRepresentation: 'ALL TRAFFIC',
        });
    }
    /**
     * A single ESP port
     */
    static esp() {
        return new Port({
            protocol: Protocol.ESP,
            fromPort: 50,
            toPort: 50,
            stringRepresentation: 'ESP 50',
        });
    }
    /**
     * A single AH port
     */
    static ah() {
        return new Port({
            protocol: Protocol.AH,
            fromPort: 51,
            toPort: 51,
            stringRepresentation: 'AH 51',
        });
    }
    /**
     * Produce the ingress/egress rule JSON for the given connection
     */
    toRuleJson() {
        // JSII does not allow enum types to have same value. So to support the enum, the enum with same value has to be mapped later.
        const PROTOCOL_MAP = {
            [Protocol.IPTM]: '84',
        };
        return {
            ipProtocol: PROTOCOL_MAP[this.props.protocol] ?? this.props.protocol,
            fromPort: this.props.fromPort,
            toPort: this.props.toPort,
        };
    }
    toString() {
        return this.props.stringRepresentation;
    }
}
exports.Port = Port;
_a = JSII_RTTI_SYMBOL_1;
Port[_a] = { fqn: "@aws-cdk/aws-ec2.Port", version: "1.185.0" };
function renderPort(port) {
    return core_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
}
//# sourceMappingURL=data:application/json;base64,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