"""
Module contains user defined objects directly implemented to map to the Element
API.
"""
from __future__ import unicode_literals
from __future__ import absolute_import
from solidfire.common import model as data_model

AUTO_GENERATE_SECRET = "AUTO-GENERATE-CHAP-SECRET"


class CHAPSecret(data_model.DataObject):
    """
    A representation of the CHAP Secret

    :param secret: [required] A 12 - 16 character string.
    :type start: str

    """

    secret = data_model.property(
        "secret", str,
        array=False, optional=False,
        documentation="A 12 - 16 character string."
    )

    def __init__(self, **kwargs):#changed parameters number @subansu
        secretInput = kwargs.pop('secretInput',None)#@subansu
        data_model.DataObject.__init__(self, **kwargs)
        if (secretInput != None):
            self.secret = secretInput

    @staticmethod
    def auto_generate():
        """
        Used in 'AddAccount' and 'ModifyAccount' to cause a CHAP Secret to be
        auto generated by the cluster.
        """
        chap = CHAPSecret()
        chap.secret = AUTO_GENERATE_SECRET
        return chap

    @classmethod
    def custom_extract(cls, data):
        """
        Deserialize the property from json.

        :param data: the data to be converted.

        :return: the extracted data.
        """
        chap = CHAPSecret()
        chap.secret = data
        return chap

    def custom_to_json(self):
        """
        Custom json serialize logic
        :return:
        """
        if self.secret == AUTO_GENERATE_SECRET:
            return None
        else:
            return self.secret


class Weekday:
    """
    A better representation of the Weekday. Used in the DaysOfWeekFrequency
    class.
    """

    Sunday = ('Sunday', 0)
    Monday = ('Monday', 1)
    Tuesday = ('Tuesday', 2)
    Wednesday = ('Wednesday', 3)
    Thursday = ('Thursday', 4)
    Friday = ('Friday', 5)
    Saturday = ('Saturday', 6)

    all_weekdays = [Sunday, Monday, Tuesday, Wednesday, Thursday, Friday,
                    Saturday]

    @staticmethod
    def from_id(weekday_id):
        for weekday in Weekday.all_weekdays:
            if weekday[1] == weekday_id:
                return weekday
        raise AttributeError

    @staticmethod
    def from_name(weekday_name):
        for weekday in Weekday.all_weekdays:
            if weekday[0] == weekday_name:
                return weekday
        raise AttributeError


class Frequency(data_model.DataObject):
    """
    Abstract class used to determine the frequency of a Schedule.
    """

    def __init__(self, **kwargs):
        # Set up these guys to default to 0.
        if "minutes" not in kwargs.keys() or kwargs["minutes"] is None:
            kwargs["minutes"] = 0
        if "hours" not in kwargs.keys() or kwargs["hours"] is None:
            kwargs["hours"] = 0
        data_model.DataObject.__init__(self, **kwargs)


class TimeIntervalFrequency(Frequency):
    """
    Use this class when setting the frequency of a Schedule to Time Interval.
    Time Interval schedules trigger after the specified interval of time has
    passed.

    :param minutes:  The minutes that will elapse before the next snapshot is
    created.
    Valid values are: 0 - 59. Default is 0.
    :type minutes: int

    :param hours: The hours that will elapse before the next snapshot is
    created.
    Valid values  0 - 24. Default is 0.
    :type hours: int

    :param days: The days that will elapse before the next snapshot is created.
    Default is 0.
    :type days: int
    """
    minutes = data_model.property(
        "minutes", int,
        array=False, optional=True,
        documentation="Number of minutes between snapshots. "
                      "Valid values are: 0 - 59"
    )

    hours = data_model.property(
        "hours", int,
        array=False, optional=True,
        documentation="Number of hours between snapshots. "
                      "Valid values are: 0 - 24"
    )
    days = data_model.property(
        "days", int,
        array=False, optional=False,
        documentation="Number of days between snapshots."
    )


class DaysOfWeekFrequency(Frequency):
    """
    Use this class when setting the frequency of a Schedule to Days Of Week.
    Days of week schedules trigger on the weekdays and time specified.

    :param minutes:  The minutes within the hour the snapshot will be created
    on the set weekdays.
    Valid values are: 0 - 59. Default is 0.
    :type minutes: int

    :param The hour of the day the next snapshot will be created on the set
    weekdays.
    Valid values are: 0 - 24. Default is 0
    :type hours: int

    :param weekdays: The days of the week a snapshot will be created on.
    :type weekdays: int
    """
    minutes = data_model.property(
        "minutes", int,
        array=False, optional=True,
        documentation="Minute of hour the snapshot is to be created. "
                      "Valid values are: 0 - 59"
    )

    hours = data_model.property(
        "hours", int,
        array=False, optional=True,
        documentation="Hour of day the snapshot is to be created. "
                      "Valid values are: 0 - 24"
    )
    weekdays = data_model.property(
        "weekdays", Weekday,
        array=True, optional=False,
        documentation="Days of the week the snapshot is to be created. "
                      "Valid values are: "
                      "Weekday - 1 - 7 (Sunday - Saturday)."
    )


class DaysOfMonthFrequency(Frequency):
    """
    Use this class when setting the frequency of a Schedule to Days Of Month.
    Days of month schedules trigger on the monthdays and time specified.

    :param minutes:  The minutes within the hour the snapshot will be created
    on the set weekdays.
    Valid values are: 0 - 59. Default is 0.
    :type minutes: int

    :param The hour of the day the next snapshot will be created on the set
    weekdays.
    Valid values are: 0 - 24. Default is 0
    :type hours: int

    :param monthdays: An array of the days of the month a snapshot will be
    created on.
    Valid values are: 1 - 31
    :type monthdays: int
    """
    minutes = data_model.property(
        "minutes", int,
        array=False, optional=True,
        documentation="Minute of hour the snapshot is to be created. "
                      "Valid values are: 0 - 59"
    )

    hours = data_model.property(
        "hours", int,
        array=False, optional=True,
        documentation="Hour of day the snapshot is to be created. "
                      "Valid values are: 0 - 24"
    )
    monthdays = data_model.property(
        "monthdays", int,
        array=True, optional=False,
        documentation="Days of the month the snapshot is to be created. "
                      "Valid values are: 1 - 31"
    )
