/* *********************************************************************
 * This Original Work is copyright of 51 Degrees Mobile Experts Limited.
 * Copyright 2023 51 Degrees Mobile Experts Limited, Davidson House,
 * Forbury Square, Reading, Berkshire, United Kingdom RG1 3EU.
 *
 * This Original Work is licensed under the European Union Public Licence
 * (EUPL) v.1.2 and is subject to its terms as set out below.
 *
 * If a copy of the EUPL was not distributed with this file, You can obtain
 * one at https://opensource.org/licenses/EUPL-1.2.
 *
 * The 'Compatible Licences' set out in the Appendix to the EUPL (as may be
 * amended by the European Commission) shall be deemed incompatible for
 * the purposes of the Work and the provisions of the compatibility
 * clause in Article 5 of the EUPL shall not apply.
 *
 * If using the Work as, or as part of, a network application, by
 * including the attribution notice(s) required under Article 5 of the EUPL
 * in the end user terms of the application under an appropriate heading,
 * such notice(s) shall fulfill the requirements of that article.
 * ********************************************************************* */

#ifndef FIFTYONE_DEGREES_ARRAY_H_INCLUDED
#define FIFTYONE_DEGREES_ARRAY_H_INCLUDED

/**
 * @ingroup FiftyOneDegreesCommon
 * @defgroup FiftyOneDegreesArray Arrays
 *
 * Macros used to quickly define array structures.
 *
 * ## Introduction
 *
 * Array structures contain an array of fixed size items, along with the
 * number of items and the capacity of the array. This makes passing it around
 * much simpler as the extra information can be passed with it.
 *
 * The macros also allow additional members to the added to the structure.
 *
 * @{
 */

#include <stdint.h>
#include "memory.h"

/**
 * Simple array structure definition macro used for generic types.
 * @param t type of item
 * @param m additional members
 */
#define FIFTYONE_DEGREES_ARRAY_TYPE(t, m) \
/** Array of items of type t used to easily access and track the
    size of the array. */ \
typedef struct fiftyone_degrees_array_##t##_t { \
	uint32_t count; /**< Number of used items */ \
	uint32_t capacity; /**< Number of available items */ \
	t* items; /**< Pointer to the first item */ \
	m /**< Add any members provided by the caller */ \
} t##Array;

/**
 * Works out the size of the array with a given capacity for the type.
 */
#define FIFTYONE_DEGREES_ARRAY_SIZE(t, c) (sizeof(t##Array) + (sizeof(t) * (c)))

/**
 * Initialises the array.
 */
#define FIFTYONE_DEGREES_ARRAY_CREATE(t, i, c) \
i = (t##Array*)fiftyoneDegreesMalloc(FIFTYONE_DEGREES_ARRAY_SIZE(t,c)); \
if (i != NULL) { \
i->items = c ? (t*)(i + 1) : NULL; \
i->count = 0; \
i->capacity = c; \
}

/**
 * @}
 */ 
#endif
