"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FromCloudFormation = void 0;
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const token_1 = require("./token");
/**
 * This class contains functions for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class FromCloudFormation {
    static parseValue(cfnValue) {
        return parseCfnValueToCdkValue(cfnValue);
    }
    // nothing to for any but return it
    static getAny(value) { return value; }
    // nothing to do - if 'value' is not a boolean or a Token,
    // a validator should report that at runtime
    static getBoolean(value) { return value; }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(value, this.getString);
    }
    static getArray(value, mapper) {
        if (!Array.isArray(value)) {
            // break the type system, and just return the given value,
            // which hopefully will be reported as invalid by the validator
            // of the property we're transforming
            // (unless it's a deploy-time value,
            // which we can't map over at build time anyway)
            return value;
        }
        return value.map(mapper);
    }
    static getMap(value, mapper) {
        if (typeof value !== 'object') {
            // if the input is not a map (= object in JS land),
            // just return it, and let the validator of this property handle it
            // (unless it's a deploy-time value,
            // which we can't map over at build time anyway)
            return value;
        }
        const ret = {};
        for (const [key, val] of Object.entries(value)) {
            ret[key] = mapper(val);
        }
        return ret;
    }
    static parseCreationPolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = FromCloudFormation.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingCreationPolicy: parseAutoScalingCreationPolicy(policy.AutoScalingCreationPolicy),
            resourceSignal: parseResourceSignal(policy.ResourceSignal),
        });
        function parseAutoScalingCreationPolicy(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
            });
        }
        function parseResourceSignal(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                count: FromCloudFormation.getNumber(p.Count),
                timeout: FromCloudFormation.getString(p.Timeout),
            });
        }
    }
    static parseUpdatePolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = FromCloudFormation.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingReplacingUpdate: parseAutoScalingReplacingUpdate(policy.AutoScalingReplacingUpdate),
            autoScalingRollingUpdate: parseAutoScalingRollingUpdate(policy.AutoScalingRollingUpdate),
            autoScalingScheduledAction: parseAutoScalingScheduledAction(policy.AutoScalingScheduledAction),
            codeDeployLambdaAliasUpdate: parseCodeDeployLambdaAliasUpdate(policy.CodeDeployLambdaAliasUpdate),
            enableVersionUpgrade: policy.EnableVersionUpgrade,
            useOnlineResharding: policy.UseOnlineResharding,
        });
        function parseAutoScalingReplacingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                willReplace: p.WillReplace,
            });
        }
        function parseAutoScalingRollingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                maxBatchSize: FromCloudFormation.getNumber(p.MaxBatchSize),
                minInstancesInService: FromCloudFormation.getNumber(p.MinInstancesInService),
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
                pauseTime: FromCloudFormation.getString(p.PauseTime),
                suspendProcesses: FromCloudFormation.getStringArray(p.SuspendProcesses),
                waitOnResourceSignals: p.WaitOnResourceSignals,
            });
        }
        function parseCodeDeployLambdaAliasUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return {
                beforeAllowTrafficHook: FromCloudFormation.getString(p.BeforeAllowTrafficHook),
                afterAllowTrafficHook: FromCloudFormation.getString(p.AfterAllowTrafficHook),
                applicationName: FromCloudFormation.getString(p.ApplicationName),
                deploymentGroupName: FromCloudFormation.getString(p.DeploymentGroupName),
            };
        }
        function parseAutoScalingScheduledAction(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                ignoreUnmodifiedGroupSizeProperties: p.IgnoreUnmodifiedGroupSizeProperties,
            });
        }
    }
    static parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
}
exports.FromCloudFormation = FromCloudFormation;
function parseCfnValueToCdkValue(cfnValue) {
    // == null captures undefined as well
    if (cfnValue == null) {
        return undefined;
    }
    // if we have any late-bound values,
    // just return them
    if (token_1.isResolvableObject(cfnValue)) {
        return cfnValue;
    }
    if (Array.isArray(cfnValue)) {
        return cfnValue.map(el => parseCfnValueToCdkValue(el));
    }
    if (typeof cfnValue === 'object') {
        // an object can be either a CFN intrinsic, or an actual object
        const cfnIntrinsic = parseIfCfnIntrinsic(cfnValue);
        if (cfnIntrinsic) {
            return cfnIntrinsic;
        }
        const ret = {};
        for (const [key, val] of Object.entries(cfnValue)) {
            ret[key] = parseCfnValueToCdkValue(val);
        }
        return ret;
    }
    // in all other cases, just return the input
    return cfnValue;
}
function parseIfCfnIntrinsic(object) {
    var _a;
    const key = looksLikeCfnIntrinsic(object);
    switch (key) {
        case undefined:
            return undefined;
        case 'Ref': {
            // ToDo handle translating logical IDs
            return (_a = specialCaseRefs(object[key])) !== null && _a !== void 0 ? _a : cfn_fn_1.Fn._ref(object[key]);
        }
        case 'Fn::GetAtt': {
            // Fn::GetAtt takes a 2-element list as its argument
            const value = object[key];
            // ToDo same comment here as in Ref above
            return cfn_fn_1.Fn.getAtt((value[0]), value[1]);
        }
        case 'Fn::Join': {
            // Fn::Join takes a 2-element list as its argument,
            // where the first element is the delimiter,
            // and the second is the list of elements to join
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.join(value[0], value[1]);
        }
        case 'Fn::Cidr': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.cidr(value[0], value[1], value[2]);
        }
        case 'Fn::FindInMap': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.findInMap(value[0], value[1], value[2]);
        }
        case 'Fn::Select': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.select(value[0], value[1]);
        }
        case 'Fn::GetAZs': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.getAzs(value);
        }
        case 'Fn::ImportValue': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.importValue(value);
        }
        case 'Fn::Split': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.split(value[0], value[1]);
        }
        case 'Fn::Transform': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.transform(value.Name, value.Parameters);
        }
        case 'Fn::Base64': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.base64(value);
        }
        case 'Fn::If': {
            // Fn::If takes a 3-element list as its argument
            // ToDo the first argument is the name of the condition,
            // so we will need to retrieve the actual object from the template
            // when we handle preserveLogicalIds=false
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionIf(value[0], value[1], value[2]);
        }
        case 'Fn::Equals': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
        }
        case 'Fn::And': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionAnd(...value);
        }
        case 'Fn::Not': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionNot(value[0]);
        }
        case 'Fn::Or': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionOr(...value);
        }
        default:
            throw new Error(`Unsupported CloudFormation function '${key}'`);
    }
}
function looksLikeCfnIntrinsic(object) {
    const objectKeys = Object.keys(object);
    // a CFN intrinsic is always an object with a single key
    if (objectKeys.length !== 1) {
        return undefined;
    }
    const key = objectKeys[0];
    return key === 'Ref' || key.startsWith('Fn::') ? key : undefined;
}
function specialCaseRefs(value) {
    switch (value) {
        case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
        case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
        case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
        case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
        case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
        case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
        case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
        case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
        default: return undefined;
    }
}
function undefinedIfAllValuesAreEmpty(object) {
    return Object.values(object).some(v => v !== undefined) ? object : undefined;
}
//# sourceMappingURL=data:application/json;base64,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