"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = void 0;
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const tree_metadata_1 = require("./private/tree-metadata");
const stage_1 = require("./stage");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps.html
 */
class App extends stage_1.Stage {
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        var _a;
        super(undefined, '', {
            outdir: (_a = props.outdir) !== null && _a !== void 0 ? _a : process.env[cxapi.OUTDIR_ENV],
        });
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        if (props.runtimeInfo === false) {
            this.node.setContext(cxapi.DISABLE_VERSION_REPORTING, true);
        }
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
        if (props.treeMetadata === undefined || props.treeMetadata) {
            new tree_metadata_1.TreeMetadata(this);
        }
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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