"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_parse = require("../../core/lib/cfn-parse"); // Automatically re-written from '@aws-cdk/core/lib/cfn-parse'
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        // read the template into a JS object
        this.template = futils.readJsonSync(props.templateFile);
        // ToDo implement preserveLogicalIds=false
        this.preserveLogicalIds = true;
        // first, instantiate the conditions
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.createCondition(conditionName);
        }
        // then, instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the give name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            // render all sections of the template unchanged,
            // except Conditions and Resources, which will be taken care of by the created L1s
            if (section !== 'Conditions' && section !== 'Resources') {
                ret[section] = this.template[section];
            }
        }
        return ret;
    }
    createCondition(conditionName) {
        // ToDo condition expressions can refer to other conditions -
        // will be important when implementing preserveLogicalIds=false
        const expression = cfn_parse.FromCloudFormation.parseValue(this.template.Conditions[conditionName]);
        const cfnCondition = new core.CfnCondition(this, conditionName, {
            expression,
        });
        // ToDo handle renaming of the logical IDs of the conditions
        cfnCondition.overrideLogicalId(conditionName);
        this.conditions[conditionName] = cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
        if (!l1ClassFqn) {
            // currently, we only handle types we know the L1 for -
            // in the future, we might construct an instance of CfnResource instead
            throw new Error(`Unrecognized CloudFormation resource type: '${resourceAttributes.Type}'`);
        }
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Type', 'Properties', 'Condition', 'DependsOn', 'Metadata',
            'CreationPolicy', 'UpdatePolicy', 'DeletionPolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The ${attribute} resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const [moduleName, ...className] = l1ClassFqn.split('.');
        const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
        const jsClassFromModule = module[className.join('.')];
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
        };
        const options = {
            finder,
        };
        const l1Instance = jsClassFromModule.fromCloudFormation(this, logicalId, resourceAttributes, options);
        if (this.preserveLogicalIds) {
            // override the logical ID to match the original template
            l1Instance.overrideLogicalId(logicalId);
        }
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,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