"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generate = void 0;
const fs = require("fs");
const path = require("path");
const semver = require("semver");
// eslint-disable-next-line import/no-extraneous-dependencies
const tjs = require("typescript-json-schema");
function log(message) {
    // tslint:disable-next-line:no-console
    console.log(message);
}
function bump() {
    const metadataPath = '../schema/cloud-assembly.version.json';
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const metadata = require(metadataPath);
    const oldVersion = metadata.version;
    const newVersion = semver.inc(oldVersion, 'major');
    log(`Updating schema version: ${oldVersion} -> ${newVersion}`);
    const out = path.join(__dirname, metadataPath);
    fs.writeFileSync(out, JSON.stringify({ version: newVersion }));
}
function generate(out, shouldBump) {
    const settings = {
        required: true,
        ref: true,
        topRef: true,
        noExtraProps: false,
        out,
    };
    const compilerOptions = {
        strictNullChecks: true,
    };
    const program = tjs.getProgramFromFiles([path.join(__dirname, '../lib/schema.d.ts')], compilerOptions);
    const schema = tjs.generateSchema(program, 'AssemblyManifest', settings);
    augmentDescription(schema);
    addAnyMetadataEntry(schema);
    if (shouldBump) {
        bump();
    }
    if (out) {
        log(`Generating schema to ${out}`);
        fs.writeFileSync(out, JSON.stringify(schema, null, 4));
    }
    return schema;
}
exports.generate = generate;
/**
 * Remove 'default' from the schema since its generated
 * from the tsdocs, which are not necessarily actual values,
 * but rather descriptive behavior.
 *
 * To keep this inforamtion in the schema, we append it to the
 * 'description' of the property.
 */
function augmentDescription(schema) {
    function _recurse(o) {
        for (const prop in o) {
            if (prop === 'description' && typeof o[prop] === 'string') {
                const description = o[prop];
                const defaultValue = o.default;
                if (!defaultValue) {
                    // property doesn't have a default value
                    // skip
                    continue;
                }
                const descriptionWithDefault = `${description} (Default ${defaultValue})`;
                delete o.default;
                o[prop] = descriptionWithDefault;
            }
            else if (typeof o[prop] === 'object') {
                _recurse(o[prop]);
            }
        }
    }
    _recurse(schema);
}
/**
 * Patch the properties of MetadataEntry to allow
 * specifying any free form data. This is needed since source
 * code doesn't allow this in order to enforce stricter jsii
 * compatibility checks.
 */
function addAnyMetadataEntry(schema) {
    schema.definitions.MetadataEntry.properties.data.anyOf.push({ description: 'Free form data.' });
}
//# sourceMappingURL=data:application/json;base64,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