"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Manifest = exports.VERSION_MISMATCH = void 0;
const fs = require("fs");
const jsonschema = require("jsonschema");
const semver = require("semver");
const metadata_schema_1 = require("./metadata-schema");
const assembly = require("./schema");
// this prefix is used by the CLI to identify this specific error.
// in which case we want to instruct the user to upgrade his CLI.
// see exec.ts#createAssembly
exports.VERSION_MISMATCH = 'Cloud assembly schema version mismatch';
/**
 * Protocol utility class.
 */
class Manifest {
    constructor() { }
    /**
     * Save manifest to file.
     *
     * @param manifest - manifest.
     */
    static save(manifest, filePath) {
        fs.writeFileSync(filePath, JSON.stringify(manifest, undefined, 2));
    }
    /**
     * Load manifest from file.
     *
     * @param filePath - path to the manifest file.
     */
    static load(filePath) {
        const raw = JSON.parse(fs.readFileSync(filePath, { encoding: 'utf-8' }));
        Manifest.patchStackTags(raw);
        Manifest.validate(raw);
        return raw;
    }
    /**
     * Fetch the current schema version number.
     */
    static version() {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require('../schema/cloud-assembly.version.json').version;
    }
    static validate(manifest) {
        function parseVersion(version) {
            const ver = semver.valid(version);
            if (!ver) {
                throw new Error(`Invalid semver string: "${version}"`);
            }
            return ver;
        }
        const maxSupported = parseVersion(Manifest.version());
        const actual = parseVersion(manifest.version);
        // first validate the version should be accepted.
        if (semver.gt(actual, maxSupported)) {
            // we use a well known error prefix so that the CLI can identify this specific error
            // and print some more context to the user.
            throw new Error(`${exports.VERSION_MISMATCH}: Maximum schema version supported is ${maxSupported}, but found ${actual}`);
        }
        // now validate the format is good.
        const validator = new jsonschema.Validator();
        const result = validator.validate(manifest, Manifest.schema, {
            // does exist but is not in the TypeScript definitions
            nestedErrors: true,
            allowUnknownAttributes: false,
        });
        if (!result.valid) {
            throw new Error(`Invalid assembly manifest:\n${result}`);
        }
    }
    /**
     * This requires some explaining...
     *
     * We previously used `{ Key, Value }` for the object that represents a stack tag. (Notice the casing)
     * @link https://github.com/aws/aws-cdk/blob/v1.27.0/packages/aws-cdk/lib/api/cxapp/stacks.ts#L427.
     *
     * When that object moved to this package, it had to be JSII compliant, which meant the property
     * names must be `camelCased`, and not `PascalCased`. This meant it no longer matches the structure in the `manifest.json` file.
     * In order to support current manifest files, we have to translate the `PascalCased` representation to the new `camelCased` one.
     *
     * Note that the serialization itself still writes `PascalCased` because it relates to how CloudFormation expects it.
     *
     * Ideally, we would start writing the `camelCased` and translate to how CloudFormation expects it when needed. But this requires nasty
     * backwards-compatibility code and it just doesn't seem to be worth the effort.
     */
    static patchStackTags(manifest) {
        for (const artifact of Object.values(manifest.artifacts || [])) {
            if (artifact.type === assembly.ArtifactType.AWS_CLOUDFORMATION_STACK) {
                for (const metadataEntries of Object.values(artifact.metadata || [])) {
                    for (const metadataEntry of metadataEntries) {
                        if (metadataEntry.type === metadata_schema_1.ArtifactMetadataEntryType.STACK_TAGS && metadataEntry.data) {
                            const metadataAny = metadataEntry;
                            metadataAny.data = metadataAny.data.map((t) => {
                                return { key: t.Key, value: t.Value };
                            });
                        }
                    }
                }
            }
        }
    }
}
exports.Manifest = Manifest;
// eslint-disable-next-line @typescript-eslint/no-require-imports
Manifest.schema = require('../schema/cloud-assembly.schema.json');
//# sourceMappingURL=data:application/json;base64,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