"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetManifestSchema = void 0;
const semver = require("semver");
const my_package_json_1 = require("./private/my-package-json");
const schema_helpers_1 = require("./private/schema-helpers");
const PACKAGE_VERSION = my_package_json_1.loadMyPackageJson().version;
/**
 * Static class with loader routines
 *
 * This class mostly exists to put the schema structs into input position
 * (taken into a function), so that the jsii-diff checker will make sure all
 * structs are only allowed to be weakened in future updates. For example,
 * it is now allowed to add new required fields, since old CDK frameworks
 * would not be emitting those fields yet.
 *
 * At the same time, we might as well validate the structure so code doesn't
 * barf on invalid disk input.
 */
class AssetManifestSchema {
    /**
     * Validate the given structured object as a valid ManifestFile schema
     */
    static validate(file) {
        const obj = file;
        if (typeof obj !== 'object' || obj === null) {
            throw new Error(`Expected object, got '${obj}`);
        }
        schema_helpers_1.expectKey(obj, 'version', schema_helpers_1.isString);
        // Current tool must be >= the version used to write the manifest
        // (disregarding MVs) which we can verify by ^-prefixing the file version.
        if (!semver.satisfies(AssetManifestSchema.currentVersion(), `^${obj.version}`)) {
            throw new Error(`Need CDK Tools >= '${obj.version}' to read this file (current is '${AssetManifestSchema.currentVersion()}')`);
        }
        schema_helpers_1.expectKey(obj, 'files', schema_helpers_1.isMapOf(schema_helpers_1.isObjectAnd(isFileAsset)), true);
        schema_helpers_1.expectKey(obj, 'dockerImages', schema_helpers_1.isMapOf(schema_helpers_1.isObjectAnd(isDockerImageAsset)), true);
    }
    /**
     * Take a ManifestFile as input
     *
     * The presence of this method makes sure the struct is only ever weakened
     * in future releases.
     */
    static input(file) {
        this.validate(file);
    }
    /**
     * Return the version of the schema module
     */
    static currentVersion() {
        return PACKAGE_VERSION;
    }
}
exports.AssetManifestSchema = AssetManifestSchema;
function isFileAsset(entry) {
    schema_helpers_1.expectKey(entry, 'source', source => {
        schema_helpers_1.assertIsObject(source);
        schema_helpers_1.expectKey(source, 'path', schema_helpers_1.isString);
        schema_helpers_1.expectKey(source, 'packaging', schema_helpers_1.isFileAssetPackaging, true);
        return source;
    });
    schema_helpers_1.expectKey(entry, 'destinations', schema_helpers_1.isMapOf(destination => {
        schema_helpers_1.assertIsObject(destination);
        schema_helpers_1.expectKey(destination, 'region', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleArn', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleExternalId', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'bucketName', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'objectKey', schema_helpers_1.isString);
        return destination;
    }));
    return entry;
}
function isDockerImageAsset(entry) {
    schema_helpers_1.expectKey(entry, 'source', source => {
        schema_helpers_1.assertIsObject(source);
        schema_helpers_1.expectKey(source, 'directory', schema_helpers_1.isString);
        schema_helpers_1.expectKey(source, 'dockerFile', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(source, 'dockerBuildTarget', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(source, 'dockerBuildArgs', schema_helpers_1.isMapOf(schema_helpers_1.isString), true);
        return source;
    });
    schema_helpers_1.expectKey(entry, 'destinations', schema_helpers_1.isMapOf(destination => {
        schema_helpers_1.assertIsObject(destination);
        schema_helpers_1.expectKey(destination, 'region', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleArn', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleExternalId', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'repositoryName', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'imageTag', schema_helpers_1.isString);
        return destination;
    }));
    return entry;
}
//# sourceMappingURL=data:application/json;base64,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