"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Activity = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const stepfunctions_generated_1 = require("./stepfunctions.generated");
/**
 * Define a new Step Functions Activity
 */
class Activity extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.activityName ||
                core_1.Lazy.stringValue({ produce: () => this.generateName() }),
        });
        const resource = new stepfunctions_generated_1.CfnActivity(this, 'Resource', {
            name: this.physicalName,
        });
        this.activityArn = this.getResourceArnAttribute(resource.ref, {
            service: 'states',
            resource: 'activity',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.activityName = this.getResourceNameAttribute(resource.attrName);
    }
    /**
     * Construct an Activity from an existing Activity ARN
     */
    static fromActivityArn(scope, id, activityArn) {
        class Imported extends core_1.Resource {
            get activityArn() { return activityArn; }
            get activityName() {
                return core_1.Stack.of(this).parseArn(activityArn, ':').resourceName || '';
            }
        }
        return new Imported(scope, id);
    }
    /**
     * Construct an Activity from an existing Activity Name
     */
    static fromActivityName(scope, id, activityName) {
        return Activity.fromActivityArn(scope, id, core_1.Stack.of(scope).formatArn({
            service: 'states',
            resource: 'activity',
            resourceName: activityName,
            sep: ':',
        }));
    }
    /**
     * Return the given named metric for this Activity
     *
     * @default sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: { ActivityArn: this.activityArn },
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * The interval, in milliseconds, between the time the activity starts and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricRunTime(props) {
        return this.metric('ActivityRunTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, for which the activity stays in the schedule state.
     *
     * @default average over 5 minutes
     */
    metricScheduleTime(props) {
        return this.metric('ActivityScheduleTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, between the time the activity is scheduled and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricTime(props) {
        return this.metric('ActivityTime', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the number of times this activity is scheduled
     *
     * @default sum over 5 minutes
     */
    metricScheduled(props) {
        return this.metric('ActivitiesScheduled', props);
    }
    /**
     * Metric for the number of times this activity times out
     *
     * @default sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.metric('ActivitiesTimedOut', props);
    }
    /**
     * Metric for the number of times this activity is started
     *
     * @default sum over 5 minutes
     */
    metricStarted(props) {
        return this.metric('ActivitiesStarted', props);
    }
    /**
     * Metric for the number of times this activity succeeds
     *
     * @default sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.metric('ActivitiesSucceeded', props);
    }
    /**
     * Metric for the number of times this activity fails
     *
     * @default sum over 5 minutes
     */
    metricFailed(props) {
        return this.metric('ActivitiesFailed', props);
    }
    /**
     * Metric for the number of times the heartbeat times out for this activity
     *
     * @default sum over 5 minutes
     */
    metricHeartbeatTimedOut(props) {
        return this.metric('ActivitiesHeartbeatTimedOut', props);
    }
    generateName() {
        const name = this.node.uniqueId;
        if (name.length > 80) {
            return name.substring(0, 40) + name.substring(name.length - 40);
        }
        return name;
    }
}
exports.Activity = Activity;
//# sourceMappingURL=data:application/json;base64,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