"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InvocationType = exports.RunLambdaTask = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Invoke a Lambda function as a Task
 *
 * OUTPUT: the output of this task is either the return value of Lambda's
 * Invoke call, or whatever the Lambda Function posted back using
 * `SendTaskSuccess/SendTaskFailure` in `waitForTaskToken` mode.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 * @deprecated Use `LambdaInvoke`
 */
class RunLambdaTask {
    constructor(lambdaFunction, props = {}) {
        this.lambdaFunction = lambdaFunction;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Lambda.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is missing in payload (pass Context.taskToken somewhere in payload)');
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('lambda', 'invoke', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [this.lambdaFunction.functionArn],
                    actions: ['lambda:InvokeFunction'],
                })],
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: { LambdaFunctionArn: this.lambdaFunction.functionArn },
            parameters: {
                FunctionName: this.lambdaFunction.functionName,
                Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier,
            },
        };
    }
}
exports.RunLambdaTask = RunLambdaTask;
/**
 * Invocation type of a Lambda
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke synchronously
     *
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke asynchronously
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * TValidate parameter values and verify that the user or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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