"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlueStartJobRun = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * Starts an AWS Glue job in a Task state
 *
 * OUTPUT: the output of this task is a JobRun structure, for details consult
 * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-JobRun
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html
 */
class GlueStartJobRun extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, GlueStartJobRun.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.getPolicies();
        this.taskMetrics = {
            metricPrefixSingular: 'GlueJob',
            metricPrefixPlural: 'GlueJobs',
            metricDimensions: { GlueJobName: this.props.glueJobName },
        };
    }
    renderTask() {
        var _a, _b;
        const notificationProperty = this.props.notifyDelayAfter ? { NotifyDelayAfter: this.props.notifyDelayAfter.toMinutes() } : null;
        return {
            Resource: task_utils_1.integrationResourceArn('glue', 'startJobRun', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobName: this.props.glueJobName,
                Arguments: (_a = this.props.arguments) === null || _a === void 0 ? void 0 : _a.value,
                Timeout: (_b = this.props.timeout) === null || _b === void 0 ? void 0 : _b.toMinutes(),
                SecurityConfiguration: this.props.securityConfiguration,
                NotificationProperty: notificationProperty,
            }),
            TimeoutSeconds: undefined,
        };
    }
    getPolicies() {
        let iamActions;
        if (this.integrationPattern === sfn.IntegrationPattern.REQUEST_RESPONSE) {
            iamActions = ['glue:StartJobRun'];
        }
        else if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            iamActions = [
                'glue:StartJobRun',
                'glue:GetJobRun',
                'glue:GetJobRuns',
                'glue:BatchStopJobRun',
            ];
        }
        return [new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'glue',
                        resource: 'job',
                        resourceName: this.props.glueJobName,
                    }),
                ],
                actions: iamActions,
            })];
    }
}
exports.GlueStartJobRun = GlueStartJobRun;
GlueStartJobRun.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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