"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrAddStep = exports.ActionOnFailure = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * The action to take when the cluster step fails.
 * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html
 *
 * Here, they are named as TERMINATE_JOB_FLOW, TERMINATE_CLUSTER, CANCEL_AND_WAIT, and CONTINUE respectively.
 *
 * @default CONTINUE
 *
 * @experimental
 */
var ActionOnFailure;
(function (ActionOnFailure) {
    /**
     * Terminate the Cluster on Step Failure
     */
    ActionOnFailure["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    /**
     * Cancel Step execution and enter WAITING state
     */
    ActionOnFailure["CANCEL_AND_WAIT"] = "CANCEL_AND_WAIT";
    /**
     * Continue to the next Step
     */
    ActionOnFailure["CONTINUE"] = "CONTINUE";
})(ActionOnFailure = exports.ActionOnFailure || (exports.ActionOnFailure = {}));
/**
 * A Step Functions Task to add a Step to an EMR Cluster
 *
 * The StepConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the StepId
 *
 * @experimental
 */
class EmrAddStep extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.actionOnFailure = (_a = props.actionOnFailure) !== null && _a !== void 0 ? _a : ActionOnFailure.CONTINUE;
        this.integrationPattern = (_b = props.integrationPattern) !== null && _b !== void 0 ? _b : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.createPolicyStatements();
    }
    renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'addStep', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ClusterId: this.props.clusterId,
                Step: {
                    Name: this.props.name,
                    ActionOnFailure: this.actionOnFailure.valueOf(),
                    HadoopJarStep: {
                        Jar: this.props.jar,
                        MainClass: this.props.mainClass,
                        Args: this.props.args,
                        Properties: (this.props.properties === undefined) ?
                            undefined :
                            Object.entries(this.props.properties).map(kv => ({
                                Key: kv[0],
                                Value: kv[1],
                            })),
                    },
                },
            }),
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call AddStep.
     */
    createPolicyStatements() {
        const stack = core_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:AddJobFlowSteps',
                    'elasticmapreduce:DescribeStep',
                    'elasticmapreduce:CancelSteps',
                ],
                resources: [`arn:aws:elasticmapreduce:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:cluster/*`],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRAddJobFlowStepsRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.EmrAddStep = EmrAddStep;
EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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