"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CallDynamoDB = exports.DynamoUpdateItem = exports.DynamoDeleteItem = exports.DynamoPutItem = exports.DynamoGetItem = exports.DynamoProjectionExpression = exports.DynamoAttributeValue = exports.DynamoReturnValues = exports.DynamoItemCollectionMetrics = exports.DynamoConsumedCapacity = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 */
var DynamoConsumedCapacity;
(function (DynamoConsumedCapacity) {
    /**
     * The response includes the aggregate ConsumedCapacity for the operation,
     * together with ConsumedCapacity for each table and secondary index that was accessed
     */
    DynamoConsumedCapacity["INDEXES"] = "INDEXES";
    /**
     * The response includes only the aggregate ConsumedCapacity for the operation.
     */
    DynamoConsumedCapacity["TOTAL"] = "TOTAL";
    /**
     * No ConsumedCapacity details are included in the response.
     */
    DynamoConsumedCapacity["NONE"] = "NONE";
})(DynamoConsumedCapacity = exports.DynamoConsumedCapacity || (exports.DynamoConsumedCapacity = {}));
/**
 * Determines whether item collection metrics are returned.
 */
var DynamoItemCollectionMetrics;
(function (DynamoItemCollectionMetrics) {
    /**
     * If set to SIZE, the response includes statistics about item collections,
     * if any, that were modified during the operation.
     */
    DynamoItemCollectionMetrics["SIZE"] = "SIZE";
    /**
     * If set to NONE, no statistics are returned.
     */
    DynamoItemCollectionMetrics["NONE"] = "NONE";
})(DynamoItemCollectionMetrics = exports.DynamoItemCollectionMetrics || (exports.DynamoItemCollectionMetrics = {}));
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed
 */
var DynamoReturnValues;
(function (DynamoReturnValues) {
    /**
     * Nothing is returned
     */
    DynamoReturnValues["NONE"] = "NONE";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_OLD"] = "ALL_OLD";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_OLD"] = "UPDATED_OLD";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_NEW"] = "ALL_NEW";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_NEW"] = "UPDATED_NEW";
})(DynamoReturnValues = exports.DynamoReturnValues || (exports.DynamoReturnValues = {}));
/**
 * Class to generate AttributeValue
 */
class DynamoAttributeValue {
    constructor() {
        this.attributeValue = {};
    }
    /**
     * Sets an attribute of type String. For example:  "S": "Hello"
     */
    withS(value) {
        this.attributeValue.S = value;
        return this;
    }
    /**
     * Sets an attribute of type Number. For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    withN(value) {
        this.attributeValue.N = value;
        return this;
    }
    /**
     * Sets an attribute of type Binary. For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     */
    withB(value) {
        this.attributeValue.B = value;
        return this;
    }
    /**
     * Sets an attribute of type String Set. For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     */
    withSS(value) {
        this.attributeValue.SS = value;
        return this;
    }
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    withNS(value) {
        this.attributeValue.NS = value;
        return this;
    }
    /**
     * Sets an attribute of type Binary Set. For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     */
    withBS(value) {
        this.attributeValue.BS = value;
        return this;
    }
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     */
    withM(value) {
        this.attributeValue.M = transformAttributeValueMap(value);
        return this;
    }
    /**
     * Sets an attribute of type List. For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     */
    withL(value) {
        this.attributeValue.L = value.map(val => val.toObject());
        return this;
    }
    /**
     * Sets an attribute of type Null. For example:  "NULL": true
     */
    withNULL(value) {
        this.attributeValue.NULL = value;
        return this;
    }
    /**
     * Sets an attribute of type Boolean. For example:  "BOOL": true
     */
    withBOOL(value) {
        this.attributeValue.BOOL = value;
        return this;
    }
    /**
     * Return the attributeValue object
     */
    toObject() {
        return this.attributeValue;
    }
}
exports.DynamoAttributeValue = DynamoAttributeValue;
/**
 * Class to generate projection expression
 */
class DynamoProjectionExpression {
    constructor() {
        this.expression = [];
    }
    /**
     * Adds the passed attribute to the chain
     *
     * @param attr Attribute name
     */
    withAttribute(attr) {
        if (this.expression.length) {
            this.expression.push(`.${attr}`);
        }
        else {
            this.expression.push(attr);
        }
        return this;
    }
    /**
     * Adds the array literal access for passed index
     *
     * @param index array index
     */
    atIndex(index) {
        if (!this.expression.length) {
            throw new Error('Expression must start with an attribute');
        }
        this.expression.push(`[${index}]`);
        return this;
    }
    /**
     * converts and return the string expression
     */
    toString() {
        return this.expression.join('');
    }
}
exports.DynamoProjectionExpression = DynamoProjectionExpression;
/**
 * A StepFunctions task to call DynamoGetItem
 */
class DynamoGetItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        var _a;
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.GET),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.GET),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConsistentRead: (_a = this.props.consistentRead) !== null && _a !== void 0 ? _a : false,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ProjectionExpression: this.configureProjectionExpression(this.props.projectionExpression),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
            },
        };
    }
    configureProjectionExpression(expressions) {
        return expressions
            ? expressions.map(expression => expression.toString()).join(',')
            : undefined;
    }
}
exports.DynamoGetItem = DynamoGetItem;
/**
 * A StepFunctions task to call DynamoPutItem
 */
class DynamoPutItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.PUT),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.PUT),
            parameters: {
                Item: transformAttributeValueMap(this.props.item),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues,
            },
        };
    }
}
exports.DynamoPutItem = DynamoPutItem;
/**
 * A StepFunctions task to call DynamoDeleteItem
 */
class DynamoDeleteItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.DELETE),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.DELETE),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues,
            },
        };
    }
}
exports.DynamoDeleteItem = DynamoDeleteItem;
/**
 * A StepFunctions task to call DynamoUpdateItem
 */
class DynamoUpdateItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.UPDATE),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.UPDATE),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues,
                UpdateExpression: this.props.updateExpression,
            },
        };
    }
}
exports.DynamoUpdateItem = DynamoUpdateItem;
/**
 * A helper wrapper class to call all DynamoDB APIs
 */
class CallDynamoDB {
    /**
     * Method to get DynamoGetItem task
     *
     * @param props DynamoGetItemProps
     */
    static getItem(props) {
        return new DynamoGetItem(props);
    }
    /**
     * Method to get DynamoPutItem task
     *
     * @param props DynamoPutItemProps
     */
    static putItem(props) {
        return new DynamoPutItem(props);
    }
    /**
     * Method to get DynamoDeleteItem task
     *
     * @param props DynamoDeleteItemProps
     */
    static deleteItem(props) {
        return new DynamoDeleteItem(props);
    }
    /**
     * Method to get DynamoUpdateItem task
     *
     * @param props DynamoUpdateItemProps
     */
    static updateItem(props) {
        return new DynamoUpdateItem(props);
    }
}
exports.CallDynamoDB = CallDynamoDB;
var DynamoMethod;
(function (DynamoMethod) {
    DynamoMethod["GET"] = "Get";
    DynamoMethod["PUT"] = "Put";
    DynamoMethod["DELETE"] = "Delete";
    DynamoMethod["UPDATE"] = "Update";
})(DynamoMethod || (DynamoMethod = {}));
function validateTableName(tableName) {
    if (tableName.length < 3 ||
        tableName.length > 255 ||
        !new RegExp(/[a-zA-Z0-9_.-]+$/).test(tableName)) {
        throw new Error(`TableName should not contain alphanumeric characters and should be between 3-255 characters long. Received: ${tableName}`);
    }
}
function getDynamoResourceArn(method) {
    return resource_arn_suffix_1.getResourceArn('dynamodb', `${method.toLowerCase()}Item`, sfn.ServiceIntegrationPattern.FIRE_AND_FORGET);
}
function getDynamoPolicyStatements(task, tableName, method) {
    return [
        new iam.PolicyStatement({
            resources: [
                core_1.Stack.of(task).formatArn({
                    service: 'dynamodb',
                    resource: 'table',
                    resourceName: tableName,
                }),
            ],
            actions: [`dynamodb:${method}Item`],
        }),
    ];
}
function configurePrimaryKey(partitionKey, sortKey) {
    const key = {
        [partitionKey.name]: partitionKey.value.toObject(),
    };
    if (sortKey) {
        key[sortKey.name] = sortKey.value.toObject();
    }
    return key;
}
function transformAttributeValueMap(attrMap) {
    const transformedValue = {};
    for (const key in attrMap) {
        if (key) {
            transformedValue[key] = attrMap[key].toObject();
        }
    }
    return attrMap ? transformedValue : undefined;
}
//# sourceMappingURL=data:application/json;base64,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