import * as s3 from "../../aws-s3";
import * as s3_assets from "../../aws-s3-assets";
import * as cdk from "../../core";
export declare abstract class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static fromBucket(bucket: s3.IBucket, key: string, objectVersion?: string): S3Code;
    /**
     * @deprecated use `fromBucket`
     */
    static bucket(bucket: s3.IBucket, key: string, objectVersion?: string): S3Code;
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static fromInline(code: string): InlineCode;
    /**
     * @deprecated use `fromInline`
     */
    static inline(code: string): InlineCode;
    /**
     * Loads the function code from a local disk path.
     *
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static fromAsset(path: string, options?: s3_assets.AssetOptions): AssetCode;
    /**
     * @deprecated use `fromAsset`
     */
    static asset(path: string): AssetCode;
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @returns a new instance of `CfnParametersCode`
     * @param props optional construction properties of {@link CfnParametersCode}
     */
    static fromCfnParameters(props?: CfnParametersCodeProps): CfnParametersCode;
    /**
     * @deprecated use `fromCfnParameters`
     */
    static cfnParameters(props?: CfnParametersCodeProps): CfnParametersCode;
    /**
     * Determines whether this Code is inline code or not.
     *
     * @deprecated this value is ignored since inline is now determined based on the
     * the `inlineCode` field of `CodeConfig` returned from `bind()`.
     */
    abstract readonly isInline: boolean;
    /**
     * Called when the lambda or layer is initialized to allow this object to bind
     * to the stack, add resources and have fun.
     *
     * @param scope The binding scope. Don't be smart about trying to down-cast or
     * assume it's initialized. You may just use it as a construct scope.
     */
    abstract bind(scope: cdk.Construct): CodeConfig;
    /**
     * Called after the CFN function resource has been created to allow the code
     * class to bind to it. Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindToResource(_resource: cdk.CfnResource, _options?: ResourceBindOptions): void;
}
export interface CodeConfig {
    /**
     * The location of the code in S3 (mutually exclusive with `inlineCode`).
     */
    readonly s3Location?: s3.Location;
    /**
     * Inline code (mutually exclusive with `s3Location`).
     */
    readonly inlineCode?: string;
}
/**
 * Lambda code from an S3 archive.
 */
export declare class S3Code extends Code {
    private key;
    private objectVersion?;
    readonly isInline = false;
    private bucketName;
    constructor(bucket: s3.IBucket, key: string, objectVersion?: string | undefined);
    bind(_scope: cdk.Construct): CodeConfig;
}
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
export declare class InlineCode extends Code {
    private code;
    readonly isInline = true;
    constructor(code: string);
    bind(_scope: cdk.Construct): CodeConfig;
}
/**
 * Lambda code from a local directory.
 */
export declare class AssetCode extends Code {
    readonly path: string;
    private readonly options;
    readonly isInline = false;
    private asset?;
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path: string, options?: s3_assets.AssetOptions);
    bind(scope: cdk.Construct): CodeConfig;
    bindToResource(resource: cdk.CfnResource, options?: ResourceBindOptions): void;
}
export interface ResourceBindOptions {
    /**
     * The name of the CloudFormation property to annotate with asset metadata.
     * @see https://github.com/aws/aws-cdk/issues/1432
     * @default Code
     */
    readonly resourceProperty?: string;
}
/**
 * Construction properties for {@link CfnParametersCode}.
 */
export interface CfnParametersCodeProps {
    /**
     * The CloudFormation parameter that represents the name of the S3 Bucket
     * where the Lambda code will be located in.
     * Must be of type 'String'.
     *
     * @default a new parameter will be created
     */
    readonly bucketNameParam?: cdk.CfnParameter;
    /**
     * The CloudFormation parameter that represents the path inside the S3 Bucket
     * where the Lambda code will be located at.
     * Must be of type 'String'.
     *
     * @default a new parameter will be created
     */
    readonly objectKeyParam?: cdk.CfnParameter;
}
/**
 * Lambda code defined using 2 CloudFormation parameters.
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
export declare class CfnParametersCode extends Code {
    readonly isInline = false;
    private _bucketNameParam?;
    private _objectKeyParam?;
    constructor(props?: CfnParametersCodeProps);
    bind(scope: cdk.Construct): CodeConfig;
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code
     */
    assign(location: s3.Location): {
        [name: string]: any;
    };
    get bucketNameParam(): string;
    get objectKeyParam(): string;
}
