"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const fs = require("fs");
const path = require("path");
const bundling_1 = require("./bundling");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        const entry = findEntry(id, props.entry);
        const handler = (_a = props.handler) !== null && _a !== void 0 ? _a : 'handler';
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = (_b = props.runtime) !== null && _b !== void 0 ? _b : defaultRunTime;
        const projectRoot = (_c = props.projectRoot) !== null && _c !== void 0 ? _c : util_1.findGitPath();
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        const nodeDockerTag = (_d = props.nodeDockerTag) !== null && _d !== void 0 ? _d : `${process.versions.node}-alpine`;
        super(scope, id, {
            ...props,
            runtime,
            code: bundling_1.Bundling.parcel({
                entry,
                global: handler,
                minify: props.minify,
                sourceMaps: props.sourceMaps,
                cacheDir: props.cacheDir,
                nodeVersion: extractVersion(runtime),
                nodeDockerTag,
                projectRoot: path.resolve(projectRoot),
                environment: props.containerEnvironment,
            }),
            handler: `index.${handler}`,
        });
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(js|ts)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
/**
 * Extracts the version from the runtime
 */
function extractVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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